#!/usr/bin/perl
use strict;
use warnings;

# Convert between Excel serial dates and Time::Moment
use Time::Moment 0.26;

use constant EXCEL_EPOCH => Time::Moment->from_string('1899-12-30T00Z')->rd;

sub moment_from_excel {
    @_ == 1 or die q/Usage: moment_from_excel(date)/;
    my ($date) = @_;
    return Time::Moment->from_jd($date + ($date < 61), epoch => EXCEL_EPOCH);
}

sub moment_to_excel {
    @_ == 1 or die q/Usage: moment_to_excel(moment)/;
    my ($moment) = @_;
    my $date = $moment->rd - EXCEL_EPOCH;
    return $date - ($date < 61);
}

# Test cases recycled from John McNamara's libxlsxwriter
# http://libxlsxwriter.github.io/
my @tests = (
    [ '1899-12-31T00:00:00Z',     0                  ],
    [ '1900-01-01T00:00:00Z',     1                  ],
    [ '1900-02-27T00:00:00Z',     58                 ],
    [ '1900-02-28T00:00:00Z',     59                 ],
    [ '1900-03-01T00:00:00Z',     61                 ],
    [ '1900-03-02T00:00:00Z',     62                 ],
    [ '1982-08-25T00:15:20.213Z', 30188.010650613425 ],
    [ '2065-04-19T00:16:48.290Z', 60376.011670023145 ],
    [ '2147-12-15T00:55:25.446Z', 90565.038488958337 ],
    [ '2230-08-10T01:02:46.891Z', 120753.04359827546 ],
    [ '2313-04-06T01:04:15.597Z', 150942.04462496529 ],
    [ '2395-11-30T01:09:40.889Z', 181130.04838991899 ],
    [ '2478-07-25T01:11:32.560Z', 211318.04968240741 ],
    [ '2561-03-21T01:30:19.169Z', 241507.06272186342 ],
    [ '2643-11-15T01:48:25.580Z', 271695.07529606484 ],
    [ '2726-07-12T02:03:31.919Z', 301884.08578609955 ],
    [ '2809-03-06T02:11:11.986Z', 332072.09111094906 ],
    [ '2891-10-31T02:24:37.095Z', 362261.10042934027 ],
    [ '2974-06-26T02:35:07.220Z', 392449.10772245371 ],
    [ '3057-02-19T02:45:12.109Z', 422637.1147234838  ],
    [ '3139-10-17T03:06:39.990Z', 452826.12962951389 ],
    [ '3222-06-11T03:08:08.251Z', 483014.13065105322 ],
    [ '3305-02-05T03:19:12.576Z', 513203.13834       ],
    [ '3387-10-01T03:29:42.574Z', 543391.14563164348 ],
    [ '3470-05-27T03:37:30.813Z', 573579.15105107636 ],
    [ '3553-01-21T04:14:38.231Z', 603768.17683137732 ],
    [ '3635-09-16T04:16:28.559Z', 633956.17810832174 ],
    [ '3718-05-13T04:17:58.222Z', 664145.17914608796 ],
    [ '3801-01-06T04:21:41.794Z', 694333.18173372687 ],
    [ '3883-09-02T04:56:35.792Z', 724522.20596981479 ],
    [ '3966-04-28T05:25:14.885Z', 754710.2258667245  ],
    [ '4048-12-21T05:26:05.724Z', 784898.22645513888 ],
    [ '4131-08-18T05:46:44.068Z', 815087.24078782403 ],
    [ '4214-04-13T05:48:01.141Z', 845275.24167987274 ],
    [ '4296-12-07T05:53:52.315Z', 875464.24574438657 ],
    [ '4379-08-03T06:14:48.580Z', 905652.26028449077 ],
    [ '4462-03-28T06:46:15.738Z', 935840.28212659725 ],
    [ '4544-11-22T07:31:20.407Z', 966029.31343063654 ],
    [ '4627-07-19T07:58:33.754Z', 996217.33233511576 ],
    [ '4710-03-15T08:07:43.130Z', 1026406.3386936343 ],
    [ '4792-11-07T08:29:11.091Z', 1056594.3536005903 ],
    [ '4875-07-04T09:08:15.328Z', 1086783.3807329629 ],
    [ '4958-02-27T09:30:41.781Z', 1116971.3963169097 ],
    [ '5040-10-23T09:34:04.462Z', 1147159.3986627546 ],
    [ '5123-06-20T09:37:23.945Z', 1177348.4009715857 ],
    [ '5206-02-12T09:37:56.655Z', 1207536.4013501736 ],
    [ '5288-10-08T09:45:12.230Z', 1237725.406391551  ],
    [ '5371-06-04T09:54:14.782Z', 1267913.412671088  ],
    [ '5454-01-28T09:54:22.108Z', 1298101.4127558796 ],
    [ '5536-09-24T10:01:36.151Z', 1328290.4177795255 ],
    [ '5619-05-20T12:09:48.602Z', 1358478.5068125231 ],
    [ '5702-01-14T12:34:08.549Z', 1388667.5237100578 ],
    [ '5784-09-08T12:56:06.495Z', 1418855.5389640625 ],
    [ '5867-05-06T12:58:58.217Z', 1449044.5409515856 ],
    [ '5949-12-30T12:59:54.263Z', 1479232.5416002662 ],
    [ '6032-08-24T13:34:41.331Z', 1509420.5657561459 ],
    [ '6115-04-21T13:58:28.601Z', 1539609.5822754744 ],
    [ '6197-12-14T14:02:16.899Z', 1569797.5849178126 ],
    [ '6280-08-10T14:36:17.444Z', 1599986.6085352316 ],
    [ '6363-04-06T14:37:57.451Z', 1630174.60969272   ],
    [ '6445-11-30T14:57:42.757Z', 1660363.6234115392 ],
    [ '6528-07-26T15:10:48.307Z', 1690551.6325035533 ],
    [ '6611-03-22T15:14:39.890Z', 1720739.635183912  ],
    [ '6693-11-15T15:19:47.988Z', 1750928.6387498612 ],
    [ '6776-07-11T16:04:24.344Z', 1781116.6697262037 ],
    [ '6859-03-07T16:22:23.952Z', 1811305.6822216667 ],
    [ '6941-10-31T16:29:55.999Z', 1841493.6874536921 ],
    [ '7024-06-26T16:58:20.259Z', 1871681.7071789235 ],
    [ '7107-02-21T17:04:02.415Z', 1901870.7111390624 ],
    [ '7189-10-16T17:18:29.630Z', 1932058.7211762732 ],
    [ '7272-06-11T17:47:21.323Z', 1962247.7412190163 ],
    [ '7355-02-05T17:53:29.866Z', 1992435.7454845603 ],
    [ '7437-10-02T17:53:41.076Z', 2022624.7456143056 ],
    [ '7520-05-28T17:55:06.044Z', 2052812.7465977315 ],
    [ '7603-01-21T18:14:49.151Z', 2083000.7602910995 ],
    [ '7685-09-16T18:17:45.738Z', 2113189.7623349307 ],
    [ '7768-05-12T18:29:59.700Z', 2143377.7708298611 ],
    [ '7851-01-07T18:33:21.233Z', 2173566.773162419  ],
    [ '7933-09-02T19:14:24.673Z', 2203754.8016744559 ],
    [ '8016-04-27T19:17:12.816Z', 2233942.8036205554 ],
    [ '8098-12-22T19:23:36.418Z', 2264131.8080603937 ],
    [ '8181-08-17T19:46:25.908Z', 2294319.8239109721 ],
    [ '8264-04-13T20:07:47.314Z', 2324508.8387420601 ],
    [ '8346-12-08T20:31:37.603Z', 2354696.855296331  ],
    [ '8429-08-03T20:39:57.770Z', 2384885.8610853008 ],
    [ '8512-03-29T20:50:17.067Z', 2415073.8682530904 ],
    [ '8594-11-22T21:02:57.827Z', 2445261.8770581828 ],
    [ '8677-07-19T21:23:05.519Z', 2475450.8910360998 ],
    [ '8760-03-14T21:34:49.572Z', 2505638.8991848612 ],
    [ '8842-11-08T21:39:05.944Z', 2535827.9021521294 ],
    [ '8925-07-04T21:39:18.426Z', 2566015.9022965971 ],
    [ '9008-02-28T21:46:07.769Z', 2596203.9070343636 ],
    [ '9090-10-24T21:57:55.662Z', 2626392.9152275696 ],
    [ '9173-06-19T22:19:11.732Z', 2656580.9299968979 ],
    [ '9256-02-13T22:23:51.376Z', 2686769.9332335186 ],
    [ '9338-10-09T22:27:58.771Z', 2716957.9360968866 ],
    [ '9421-06-05T22:43:30.392Z', 2747146.9468795368 ],
    [ '9504-01-30T22:48:25.834Z', 2777334.9502990046 ],
    [ '9586-09-24T22:53:51.727Z', 2807522.9540709145 ],
    [ '9669-05-20T23:12:56.536Z', 2837711.9673210187 ],
    [ '9752-01-14T23:15:54.109Z', 2867899.9693762613 ],
    [ '9834-09-10T23:17:12.632Z', 2898088.9702850925 ],
    [ '9999-12-31T23:59:59Z',     2958465.999988426  ],
);

use Time::Moment        0.26;
use Test::More          0.88;
use Test::Number::Delta relative => 1E-10;

foreach my $test (@tests) {
    my ($string, $date) = @$test;
    my $tm = moment_from_excel($date);
    is($tm->to_string, $string, "moment_from_excel($date)");
    delta_ok(moment_to_excel($tm), $date, "moment_to_excel($tm)");
}

done_testing();

eval {
    require DateTime::Format::Excel;
    require Benchmark;
    {
        print "\nComparing DateTime::Format::Excel and Time::Moment\n";
        my $date = 30188.010650613425;
        Benchmark::cmpthese( -10, {
            'DateTime' => sub {
                my $dt = DateTime::Format::Excel->parse_datetime($date);
            },
            'Time::Moment' => sub {
                my $tm = moment_from_excel($date);
            },
        });
    }  
};
