// Copyright (C) 2017-2023 Internet Systems Consortium, Inc. ("ISC")
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include <config.h>

#include <cc/command_interpreter.h>
#include <cc/data.h>
#include <cc/simple_parser.h>
#include <cc/cfg_to_element.h>
#include <testutils/user_context_utils.h>
#include <dhcp/testutils/iface_mgr_test_config.h>
#include <dhcpsrv/cfgmgr.h>
#include <dhcpsrv/parsers/simple_parser4.h>
#include <dhcp4/dhcp4_srv.h>
#include <dhcp4/ctrl_dhcp4_srv.h>
#include <dhcp4/json_config_parser.h>
#include <dhcp4/tests/dhcp4_test_utils.h>
#include <dhcp4/tests/get_config_unittest.h>
#include <testutils/gtest_utils.h>

#include <boost/algorithm/string.hpp>
#include <gtest/gtest.h>

#include <iostream>
#include <string>
#include <sstream>
#include <list>

using namespace isc::config;
using namespace isc::data;
using namespace isc::dhcp;
using namespace isc::dhcp::test;
using namespace isc::test;

namespace {

/// @name How to fill configurations
///
/// Copy get_config_unittest.cc.skel into get_config_unittest.cc
///
/// For the extracted configurations define the EXTRACT_CONFIG and
/// recompile this file. Run dhcp4_unittests on Dhcp4ParserTest
/// redirecting the standard error to a temporary file, e.g. by
/// @code
///    ./dhcp4_unittests --gtest_filter="Dhcp4Parser*" > /dev/null 2> x
/// @endcode
///
/// Update EXTRACTED_CONFIGS with the file content
///
/// When configurations have been extracted the corresponding unparsed
/// configurations must be generated. To do that define GENERATE_ACTION
/// and recompile this file. Run dhcp4_unittests on Dhcp4GetConfigTest
/// redirecting the standard error to a temporary file, e.g. by
/// @code
///    ./dhcp4_unittests --gtest_filter="Dhcp4GetConfig*" > /dev/null 2> u
/// @endcode
///
/// Update UNPARSED_CONFIGS with the file content, recompile this file
/// without EXTRACT_CONFIG and GENERATE_ACTION.
///
/// @note Check for failures at each step!
/// @note The tests of this file do not check if configs returned
/// by @ref isc::dhcp::CfgToElement::ToElement() are complete.
/// This has to be done manually.
///
///@{
/// @brief extracted configurations
const char* EXTRACTED_CONFIGS[] = {
/// put this after const char* EXTRACTED_CONFIGS[] = {
    // CONFIGURATION 0
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [ ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 1
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 2
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 3
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"max-valid-lifetime\": 5000,\n"
"        \"min-valid-lifetime\": 3000,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 4
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1024,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 100,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 34,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101 - 192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 5
"{\n"
"        \"boot-file-name\": \"bar\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"next-server\": \"1.2.3.4\",\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"server-hostname\": \"foo\",\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 6
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"boot-file-name\": \"bar\",\n"
"                \"id\": 1,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"server-hostname\": \"foo\",\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 7
"{\n"
"        \"boot-file-name\": \"nofile\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"next-server\": \"192.0.0.1\",\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"server-hostname\": \"nohost\",\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"boot-file-name\": \"bootfile.efi\",\n"
"                \"id\": 1,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"server-hostname\": \"some-name.example.org\",\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 8
"{\n"
"        \"echo-client-id\": false,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 9
"{\n"
"        \"echo-client-id\": true,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 10
"{\n"
"        \"compatibility\": {\n"
"            \"exclude-first-last-24\": true,\n"
"            \"ignore-dhcp-server-identifier\": true,\n"
"            \"ignore-rai-link-selection\": true,\n"
"            \"lenient-option-parsing\": true\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 11
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"match-client-id\": true,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"match-client-id\": false,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 12
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"match-client-id\": false,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 13
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"authoritative\": true,\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"authoritative\": false,\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 14
"{\n"
"        \"authoritative\": true,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"authoritative\": false,\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 15
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"max-valid-lifetime\": 5000,\n"
"        \"min-valid-lifetime\": 3000,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 5,\n"
"                \"min-valid-lifetime\": 3,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"renew-timer\": 1,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 16
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.2.200-192.0.2.255\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/25\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.3.128/25\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 17
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.128/28\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 18
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 19
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"uint16, ipv4-address, ipv6-address, string\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"record\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 20
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 101,\n"
"                \"name\": \"foo-2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 21
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": true,\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 22
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"sub-opts-space\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 23
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 109,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 24
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 170,\n"
"                \"name\": \"unassigned-option-170\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 25
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\"\n"
"            },\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"01\",\n"
"                \"name\": \"default-ip-ttl\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 26
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    },\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 27
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 56,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 28
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 29
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"11\",\n"
"                \"name\": \"base-option\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 222,\n"
"                \"encapsulate\": \"isc\",\n"
"                \"name\": \"base-option\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"uint8\"\n"
"            },\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 30
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"AB\",\n"
"                \"name\": \"dhcp-message\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    },\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 31
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"0102030405060708090A\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"FF\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 32
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"ABCDEF0105\",\n"
"                                \"name\": \"dhcp-message\"\n"
"                            },\n"
"                            {\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"01\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 33
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"ABCDEF0105\",\n"
"                                \"name\": \"dhcp-message\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    },\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"01\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.200 - 192.0.2.250\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 34
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"true, 10.0.0.3, 127.0.0.1\",\n"
"                \"name\": \"slp-directory-agent\"\n"
"            },\n"
"            {\n"
"                \"data\": \"false, \",\n"
"                \"name\": \"slp-service-scope\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 35
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 36
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"name\": \"vendor-encapsulated-options\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 37
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"option-one\",\n"
"                \"space\": \"vendor-4491\"\n"
"            },\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"option-two\",\n"
"                \"space\": \"vendor-1234\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 38
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"data\": \"this is a string vendor-opt\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-4491\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 39
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"eth1\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 40
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"*\", \"eth1\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 41
"{\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 42
"{\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"generated-prefix\": \"test.prefix\",\n"
"            \"hostname-char-replacement\": \"x\",\n"
"            \"hostname-char-set\": \"[^A-Z]\",\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": true,\n"
"            \"override-no-update\": true,\n"
"            \"qualifying-suffix\": \"test.suffix.\",\n"
"            \"replace-client-name\": \"when-present\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 43
"{\n"
"        \"ddns-generated-prefix\": \"global.prefix\",\n"
"        \"ddns-override-client-update\": true,\n"
"        \"ddns-override-no-update\": true,\n"
"        \"ddns-qualifying-suffix\": \"global.suffix.\",\n"
"        \"ddns-replace-client-name\": \"always\",\n"
"        \"ddns-send-updates\": false,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"generated-prefix\": \"d2.prefix\",\n"
"            \"hostname-char-replacement\": \"z\",\n"
"            \"hostname-char-set\": \"[^0-9]\",\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"d2.suffix.\",\n"
"            \"replace-client-name\": \"when-present\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"hostname-char-replacement\": \"x\",\n"
"        \"hostname-char-set\": \"[^A-Z]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 44
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"192.0.2.123\"\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 45
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ \"192.0.3.123\", \"192.0.3.124\" ]\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 46
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"client-class\": \"alpha\",\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"client-class\": \"beta\",\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"client-class\": \"gamma\",\n"
"                \"id\": 3,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 4,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101 - 192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 47
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"client-class\": \"alpha\",\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    },\n"
"                    {\n"
"                        \"client-class\": \"beta\",\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    },\n"
"                    {\n"
"                        \"client-class\": \"gamma\",\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101 - 192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.0.0/16\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 48
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 123,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 234,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0A\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.3.15\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"32\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    },\n"
"                    {\n"
"                        \"hostname\": \"\",\n"
"                        \"hw-address\": \"01:02:03:04:05:06\",\n"
"                        \"ip-address\": \"192.0.3.120\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.3.95\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"11\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 542,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"0A:09:08:07:06:05:04:03:02:01\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.101\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.4.11\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"95\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    },\n"
"                    {\n"
"                        \"circuit-id\": \"060504030201\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.102\"\n"
"                    },\n"
"                    {\n"
"                        \"client-id\": \"05:01:02:03:04:05:06\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.103\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 49
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 234,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0A\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"123\",\n"
"                                \"name\": \"foo\",\n"
"                                \"space\": \"isc\"\n"
"                            }\n"
"                        ]\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 50
"{\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.1.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"subnet\": \"192.0.1.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": true,\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 3,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": false,\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 4,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": false,\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 5,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 6,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.6.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"subnet\": \"192.0.6.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 7,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.7.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": true,\n"
"                \"subnet\": \"192.0.7.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 51
"{\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-out-of-pool\": true,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 52
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 53
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 54
"{\n"
"        \"decline-probation-period\": 12345,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 55
"{\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 35,\n"
"            \"hold-reclaimed-time\": 1800,\n"
"            \"max-reclaim-leases\": 50,\n"
"            \"max-reclaim-time\": 100,\n"
"            \"reclaim-timer-wait-time\": 20,\n"
"            \"unwarned-reclaim-cycles\": 10\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 56
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 57
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-subnet\": \"2001:db8::123/45\",\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 58
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 59
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-subnet\": \"2001:db8::543/21\",\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 60
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface-id\": \"vlan123\",\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 61
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"name\": \"one\"\n"
"            },\n"
"            {\n"
"                \"name\": \"two\"\n"
"            },\n"
"            {\n"
"                \"name\": \"three\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 62
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 1000,\n"
"                \"name\": \"one\",\n"
"                \"valid-lifetime\": 2000\n"
"            },\n"
"            {\n"
"                \"name\": \"two\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 63
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 1000,\n"
"                \"name\": \"one\",\n"
"                \"template-test\": \"''\",\n"
"                \"valid-lifetime\": 2000\n"
"            },\n"
"            {\n"
"                \"name\": \"two\",\n"
"                \"template-test\": \"''\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 64
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 65
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": { }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 66
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 67
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0 - 192.0.2.15\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 68
"{\n"
"        \"hosts-databases\": [\n"
"            {\n"
"                \"name\": \"keatest1\",\n"
"                \"password\": \"keatest\",\n"
"                \"type\": \"mysql\",\n"
"                \"user\": \"keatest\"\n"
"            },\n"
"            {\n"
"                \"name\": \"keatest2\",\n"
"                \"password\": \"keatest\",\n"
"                \"type\": \"mysql\",\n"
"                \"user\": \"keatest\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 69
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"name\": \"all\",\n"
"                \"test\": \"'' == ''\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"match all\"\n"
"                }\n"
"            },\n"
"            {\n"
"                \"name\": \"none\"\n"
"            },\n"
"            {\n"
"                \"name\": \"both\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"a comment\",\n"
"                    \"version\": 1\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"control-socket\": {\n"
"            \"socket-name\": \"/tmp/kea4-ctrl-socket\",\n"
"            \"socket-type\": \"unix\",\n"
"            \"user-context\": {\n"
"                \"comment\": \"Indirect comment\"\n"
"            }\n"
"        },\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"user-context\": {\n"
"                \"comment\": \"No dynamic DNS\"\n"
"            }\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false,\n"
"            \"user-context\": {\n"
"                \"comment\": \"Use wildcard\"\n"
"            }\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"Set option value\"\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"An option definition\"\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"shared-networks\": [\n"
"            {\n"
"                \"name\": \"foo\",\n"
"                \"subnet4\": [\n"
"                    {\n"
"                        \"id\": 100,\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"pool\": \"192.0.1.1-192.0.1.10\",\n"
"                                \"user-context\": {\n"
"                                    \"comment\": \"A pool\"\n"
"                                }\n"
"                            }\n"
"                        ],\n"
"                        \"reservations\": [\n"
"                            {\n"
"                                \"hostname\": \"foo.example.com\",\n"
"                                \"hw-address\": \"AA:BB:CC:DD:EE:FF\",\n"
"                                \"option-data\": [\n"
"                                    {\n"
"                                        \"data\": \"example.com\",\n"
"                                        \"name\": \"domain-name\",\n"
"                                        \"user-context\": {\n"
"                                            \"comment\": \"An option in a reservation\"\n"
"                                        }\n"
"                                    }\n"
"                                ],\n"
"                                \"user-context\": {\n"
"                                    \"comment\": \"A host reservation\"\n"
"                                }\n"
"                            }\n"
"                        ],\n"
"                        \"subnet\": \"192.0.1.0/24\",\n"
"                        \"user-context\": {\n"
"                            \"comment\": \"A subnet\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"user-context\": {\n"
"                    \"comment\": \"A shared network\"\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"user-context\": {\n"
"            \"comment\": \"A DHCPv4 server\"\n"
"        }\n"
"    }\n",
    // CONFIGURATION 70
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations\": [\n"
"            {\n"
"                \"duid\": \"01:02:03:04:05:06:07:08:09:0A\",\n"
"                \"hostname\": \"global1\",\n"
"                \"ip-address\": \"192.0.200.1\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"data\": \"192.0.3.15\",\n"
"                        \"name\": \"name-servers\"\n"
"                    },\n"
"                    {\n"
"                        \"data\": \"32\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ]\n"
"            },\n"
"            {\n"
"                \"hostname\": \"global2\",\n"
"                \"hw-address\": \"01:02:03:04:05:06\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"data\": \"192.0.3.95\",\n"
"                        \"name\": \"name-servers\"\n"
"                    },\n"
"                    {\n"
"                        \"data\": \"11\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ]\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 123,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 542,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 71
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"shared-networks\": [\n"
"            {\n"
"                \"calculate-tee-times\": true,\n"
"                \"name\": \"foo\",\n"
"                \"subnet4\": [\n"
"                    {\n"
"                        \"calculate-tee-times\": false,\n"
"                        \"id\": 100,\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"pool\": \"192.0.1.1-192.0.1.10\"\n"
"                            }\n"
"                        ],\n"
"                        \"subnet\": \"192.0.1.0/24\",\n"
"                        \"t1-percent\": 0.45,\n"
"                        \"t2-percent\": 0.65\n"
"                    },\n"
"                    {\n"
"                        \"id\": 200,\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                            }\n"
"                        ],\n"
"                        \"subnet\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"t1-percent\": 0.4,\n"
"                \"t2-percent\": 0.75\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 300,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0 - 192.0.3.15\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 72
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": true,\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 73
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"store-extended-info\": true,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 2,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 74
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"statistic-default-sample-age\": 5,\n"
"        \"statistic-default-sample-count\": 10,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 75
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 76
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 1024,\n"
"            \"thread-pool-size\": 48\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 77
"{\n"
"        \"ddns-conflict-resolution-mode\": \"no-check-with-dhcid\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"ddns-conflict-resolution-mode\": \"check-exists-with-dhcid\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"ddns-conflict-resolution-mode\": \"no-check-without-dhcid\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.1 - 192.0.4.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"ddns-conflict-resolution-mode\": \"no-check-with-dhcid\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.1 - 192.0.5.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.6.1 - 192.0.6.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.6.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n"
};

/// @brief unparsed configurations
const char* UNPARSED_CONFIGS[] = {
///put this after const char* UNPARSED_CONFIGS[] = {
    // CONFIGURATION 0
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 1
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 2
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 3
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"max-valid-lifetime\": 5000,\n"
"        \"min-valid-lifetime\": 3000,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 5000,\n"
"                \"min-valid-lifetime\": 3000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 4
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 34,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.5.101-192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 100,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1024,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 5
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"bar\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"1.2.3.4\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"foo\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 6
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"boot-file-name\": \"bar\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"server-hostname\": \"foo\",\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 7
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"nofile\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"192.0.0.1\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"nohost\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"boot-file-name\": \"bootfile.efi\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"server-hostname\": \"some-name.example.org\",\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 8
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": false,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 9
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 10
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"compatibility\": {\n"
"            \"exclude-first-last-24\": true,\n"
"            \"ignore-dhcp-server-identifier\": true,\n"
"            \"ignore-rai-link-selection\": true,\n"
"            \"lenient-option-parsing\": true\n"
"        },\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 11
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"match-client-id\": true,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"match-client-id\": false,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 12
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"match-client-id\": false,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 13
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"authoritative\": true,\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"authoritative\": false,\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 14
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": true,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"authoritative\": false,\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 15
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"max-valid-lifetime\": 5000,\n"
"        \"min-valid-lifetime\": 3000,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 5,\n"
"                \"min-valid-lifetime\": 3,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 16
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    },\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.200-192.0.2.255\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.0/25\"\n"
"                    },\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.128/25\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 17
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.128/28\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 18
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 19
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"uint16, ipv4-address, ipv6-address, string\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"record\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 20
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 101,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo-2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 21
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": true,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 22
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"sub-opts-space\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 23
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 109,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 24
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 170,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"unassigned-option-170\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 25
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 23,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"01\",\n"
"                \"name\": \"default-ip-ttl\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 26
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 27
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 56,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 56,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 28
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 29
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 222,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"11\",\n"
"                \"name\": \"base-option\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 222,\n"
"                \"encapsulate\": \"isc\",\n"
"                \"name\": \"base-option\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"uint8\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 3000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 3000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 30
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"AB\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 31
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"0102030405060708090A\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"FF\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 32
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 56,\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"ABCDEF0105\",\n"
"                                \"name\": \"dhcp-message\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"01\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 33
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 56,\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"ABCDEF0105\",\n"
"                                \"name\": \"dhcp-message\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    },\n"
"                    {\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": false,\n"
"                                \"data\": \"01\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"pool\": \"192.0.2.200-192.0.2.250\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 34
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 78,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"true, 10.0.0.3, 127.0.0.1\",\n"
"                \"name\": \"slp-directory-agent\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 79,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"false, \",\n"
"                \"name\": \"slp-service-scope\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 35
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 36
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 43,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"\",\n"
"                \"name\": \"vendor-encapsulated-options\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 3000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 3000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 37
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"1234\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-1234\"\n"
"            },\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 38
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 100,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"this is a string vendor-opt\",\n"
"                \"name\": \"foo\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-4491\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 39
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"eth1\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 40
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\", \"eth0\", \"eth1\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 41
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 42
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"test.prefix\",\n"
"        \"ddns-override-client-update\": true,\n"
"        \"ddns-override-no-update\": true,\n"
"        \"ddns-qualifying-suffix\": \"test.suffix.\",\n"
"        \"ddns-replace-client-name\": \"when-present\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"x\",\n"
"        \"hostname-char-set\": \"[^A-Z]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 43
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"global.prefix\",\n"
"        \"ddns-override-client-update\": true,\n"
"        \"ddns-override-no-update\": true,\n"
"        \"ddns-qualifying-suffix\": \"global.suffix.\",\n"
"        \"ddns-replace-client-name\": \"always\",\n"
"        \"ddns-send-updates\": false,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": true,\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"x\",\n"
"        \"hostname-char-set\": \"[^A-Z]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 44
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4,\n"
"                \"min-valid-lifetime\": 4,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ \"192.0.2.123\" ]\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 45
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4,\n"
"                \"min-valid-lifetime\": 4,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ \"192.0.3.123\", \"192.0.3.124\" ]\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 46
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"client-class\": \"alpha\",\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"client-class\": \"beta\",\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"client-class\": \"gamma\",\n"
"                \"id\": 3,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 4,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.5.101-192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 47
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"client-class\": \"alpha\",\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    },\n"
"                    {\n"
"                        \"client-class\": \"beta\",\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    },\n"
"                    {\n"
"                        \"client-class\": \"gamma\",\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    },\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.5.101-192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.0.0/16\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 48
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 123,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 234,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"hostname\": \"\",\n"
"                        \"hw-address\": \"01:02:03:04:05:06\",\n"
"                        \"ip-address\": \"192.0.3.120\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.3.95\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"11\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0a\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.3.15\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"32\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 542,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"client-id\": \"05010203040506\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.103\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [ ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"circuit-id\": \"060504030201\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.102\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [ ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"0a:09:08:07:06:05:04:03:02:01\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.101\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.4.11\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"95\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 49
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 234,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0a\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"always-send\": false,\n"
"                                \"code\": 100,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"123\",\n"
"                                \"name\": \"foo\",\n"
"                                \"never-send\": false,\n"
"                                \"space\": \"isc\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 50
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.1.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.1.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": true,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 3,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": false,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 4,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": false,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 5,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.5.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 6,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.6.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.6.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 7,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.7.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": true,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": true,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.7.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 51
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": true,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"reservations-global\": false,\n"
"                \"reservations-in-subnet\": true,\n"
"                \"reservations-out-of-pool\": false,\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 52
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 53
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 54
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 12345,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 55
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 35,\n"
"            \"hold-reclaimed-time\": 1800,\n"
"            \"max-reclaim-leases\": 50,\n"
"            \"max-reclaim-time\": 100,\n"
"            \"reclaim-timer-wait-time\": 20,\n"
"            \"unwarned-reclaim-cycles\": 10\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 56
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 57
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"2001:db8::123/45\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 58
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 59
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"2001:db8::543/21\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 60
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"vlan123\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 61
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"one\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"two\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"three\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            }\n"
"        ],\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 62
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 1000,\n"
"                \"name\": \"one\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\",\n"
"                \"valid-lifetime\": 2000\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"two\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            }\n"
"        ],\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 7200,\n"
"                \"min-valid-lifetime\": 7200,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 7200\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 63
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"max-valid-lifetime\": 3000,\n"
"                \"min-valid-lifetime\": 1000,\n"
"                \"name\": \"one\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\",\n"
"                \"template-test\": \"''\",\n"
"                \"valid-lifetime\": 2000\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"two\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\",\n"
"                \"template-test\": \"''\"\n"
"            }\n"
"        ],\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 7200,\n"
"                \"min-valid-lifetime\": 7200,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 7200\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 64
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 65
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": { }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 66
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 67
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 68
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"hosts-databases\": [\n"
"            {\n"
"                \"name\": \"keatest1\",\n"
"                \"password\": \"keatest\",\n"
"                \"type\": \"mysql\",\n"
"                \"user\": \"keatest\"\n"
"            },\n"
"            {\n"
"                \"name\": \"keatest2\",\n"
"                \"password\": \"keatest\",\n"
"                \"type\": \"mysql\",\n"
"                \"user\": \"keatest\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 69
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"all\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\",\n"
"                \"test\": \"'' == ''\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"match all\"\n"
"                }\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"none\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"both\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"option-def\": [ ],\n"
"                \"server-hostname\": \"\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"a comment\",\n"
"                    \"version\": 1\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"control-socket\": {\n"
"            \"socket-name\": \"/tmp/kea4-ctrl-socket\",\n"
"            \"socket-type\": \"unix\",\n"
"            \"user-context\": {\n"
"                \"comment\": \"Indirect comment\"\n"
"            }\n"
"        },\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001,\n"
"            \"user-context\": {\n"
"                \"comment\": \"No dynamic DNS\"\n"
"            }\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false,\n"
"            \"user-context\": {\n"
"                \"comment\": \"Use wildcard\"\n"
"            }\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [\n"
"            {\n"
"                \"always-send\": false,\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"never-send\": false,\n"
"                \"space\": \"dhcp4\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"Set option value\"\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\",\n"
"                \"user-context\": {\n"
"                    \"comment\": \"An option definition\"\n"
"                }\n"
"            }\n"
"        ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [\n"
"            {\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"max-valid-lifetime\": 7200,\n"
"                \"min-valid-lifetime\": 7200,\n"
"                \"name\": \"foo\",\n"
"                \"option-data\": [ ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"store-extended-info\": false,\n"
"                \"subnet4\": [\n"
"                    {\n"
"                        \"4o6-interface\": \"\",\n"
"                        \"4o6-interface-id\": \"\",\n"
"                        \"4o6-subnet\": \"\",\n"
"                        \"allocator\": \"iterative\",\n"
"                        \"calculate-tee-times\": false,\n"
"                        \"id\": 100,\n"
"                        \"max-valid-lifetime\": 7200,\n"
"                        \"min-valid-lifetime\": 7200,\n"
"                        \"option-data\": [ ],\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"option-data\": [ ],\n"
"                                \"pool\": \"192.0.1.1-192.0.1.10\",\n"
"                                \"user-context\": {\n"
"                                    \"comment\": \"A pool\"\n"
"                                }\n"
"                            }\n"
"                        ],\n"
"                        \"relay\": {\n"
"                            \"ip-addresses\": [ ]\n"
"                        },\n"
"                        \"reservations\": [\n"
"                            {\n"
"                                \"boot-file-name\": \"\",\n"
"                                \"client-classes\": [ ],\n"
"                                \"hostname\": \"foo.example.com\",\n"
"                                \"hw-address\": \"aa:bb:cc:dd:ee:ff\",\n"
"                                \"next-server\": \"0.0.0.0\",\n"
"                                \"option-data\": [\n"
"                                    {\n"
"                                        \"always-send\": false,\n"
"                                        \"code\": 15,\n"
"                                        \"csv-format\": true,\n"
"                                        \"data\": \"example.com\",\n"
"                                        \"name\": \"domain-name\",\n"
"                                        \"never-send\": false,\n"
"                                        \"space\": \"dhcp4\",\n"
"                                        \"user-context\": {\n"
"                                            \"comment\": \"An option in a reservation\"\n"
"                                        }\n"
"                                    }\n"
"                                ],\n"
"                                \"server-hostname\": \"\",\n"
"                                \"user-context\": {\n"
"                                    \"comment\": \"A host reservation\"\n"
"                                }\n"
"                            }\n"
"                        ],\n"
"                        \"store-extended-info\": false,\n"
"                        \"subnet\": \"192.0.1.0/24\",\n"
"                        \"t1-percent\": 0.5,\n"
"                        \"t2-percent\": 0.875,\n"
"                        \"user-context\": {\n"
"                            \"comment\": \"A subnet\"\n"
"                        },\n"
"                        \"valid-lifetime\": 7200\n"
"                    }\n"
"                ],\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"user-context\": {\n"
"                    \"comment\": \"A shared network\"\n"
"                },\n"
"                \"valid-lifetime\": 7200\n"
"            }\n"
"        ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"user-context\": {\n"
"            \"comment\": \"A DHCPv4 server\"\n"
"        },\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 70
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"client-classes\": [ ],\n"
"                \"hostname\": \"global2\",\n"
"                \"hw-address\": \"01:02:03:04:05:06\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 5,\n"
"                        \"csv-format\": true,\n"
"                        \"data\": \"192.0.3.95\",\n"
"                        \"name\": \"name-servers\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": true,\n"
"                        \"data\": \"11\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"client-classes\": [ ],\n"
"                \"duid\": \"01:02:03:04:05:06:07:08:09:0a\",\n"
"                \"hostname\": \"global1\",\n"
"                \"ip-address\": \"192.0.200.1\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 5,\n"
"                        \"csv-format\": true,\n"
"                        \"data\": \"192.0.3.15\",\n"
"                        \"name\": \"name-servers\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"always-send\": false,\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": true,\n"
"                        \"data\": \"32\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"never-send\": false,\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"server-hostname\": \"\"\n"
"            }\n"
"        ],\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 123,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 542,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 71
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [\n"
"            {\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": true,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"name\": \"foo\",\n"
"                \"option-data\": [ ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"store-extended-info\": false,\n"
"                \"subnet4\": [\n"
"                    {\n"
"                        \"4o6-interface\": \"\",\n"
"                        \"4o6-interface-id\": \"\",\n"
"                        \"4o6-subnet\": \"\",\n"
"                        \"allocator\": \"iterative\",\n"
"                        \"calculate-tee-times\": false,\n"
"                        \"id\": 100,\n"
"                        \"max-valid-lifetime\": 4000,\n"
"                        \"min-valid-lifetime\": 4000,\n"
"                        \"option-data\": [ ],\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"option-data\": [ ],\n"
"                                \"pool\": \"192.0.1.1-192.0.1.10\"\n"
"                            }\n"
"                        ],\n"
"                        \"relay\": {\n"
"                            \"ip-addresses\": [ ]\n"
"                        },\n"
"                        \"reservations\": [ ],\n"
"                        \"store-extended-info\": false,\n"
"                        \"subnet\": \"192.0.1.0/24\",\n"
"                        \"t1-percent\": 0.45,\n"
"                        \"t2-percent\": 0.65,\n"
"                        \"valid-lifetime\": 4000\n"
"                    },\n"
"                    {\n"
"                        \"4o6-interface\": \"\",\n"
"                        \"4o6-interface-id\": \"\",\n"
"                        \"4o6-subnet\": \"\",\n"
"                        \"allocator\": \"iterative\",\n"
"                        \"calculate-tee-times\": true,\n"
"                        \"id\": 200,\n"
"                        \"max-valid-lifetime\": 4000,\n"
"                        \"min-valid-lifetime\": 4000,\n"
"                        \"option-data\": [ ],\n"
"                        \"pools\": [\n"
"                            {\n"
"                                \"option-data\": [ ],\n"
"                                \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                            }\n"
"                        ],\n"
"                        \"relay\": {\n"
"                            \"ip-addresses\": [ ]\n"
"                        },\n"
"                        \"reservations\": [ ],\n"
"                        \"store-extended-info\": false,\n"
"                        \"subnet\": \"192.0.2.0/24\",\n"
"                        \"t1-percent\": 0.4,\n"
"                        \"t2-percent\": 0.75,\n"
"                        \"valid-lifetime\": 4000\n"
"                    }\n"
"                ],\n"
"                \"t1-percent\": 0.4,\n"
"                \"t2-percent\": 0.75,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 300,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.0/28\"\n"
"                    }\n"
"                ],\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 72
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": true,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 73
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": true,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": true,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 74
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 5,\n"
"        \"statistic-default-sample-count\": 10,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 75
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 76
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 1024,\n"
"            \"thread-pool-size\": 48\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [ ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 7200\n"
"    }\n",
    // CONFIGURATION 77
"{\n"
"        \"allocator\": \"iterative\",\n"
"        \"authoritative\": false,\n"
"        \"boot-file-name\": \"\",\n"
"        \"calculate-tee-times\": false,\n"
"        \"ddns-conflict-resolution-mode\": \"no-check-with-dhcid\",\n"
"        \"ddns-generated-prefix\": \"myhost\",\n"
"        \"ddns-override-client-update\": false,\n"
"        \"ddns-override-no-update\": false,\n"
"        \"ddns-qualifying-suffix\": \"\",\n"
"        \"ddns-replace-client-name\": \"never\",\n"
"        \"ddns-send-updates\": true,\n"
"        \"ddns-update-on-renew\": false,\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"enable-updates\": false,\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp-queue-control\": {\n"
"            \"capacity\": 64,\n"
"            \"enable-queue\": false,\n"
"            \"queue-type\": \"kea-ring4\"\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"early-global-reservations-lookup\": false,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\", \"client-id\" ],\n"
"        \"hostname-char-replacement\": \"\",\n"
"        \"hostname-char-set\": \"[^A-Za-z0-9.-]\",\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ],\n"
"            \"re-detect\": false\n"
"        },\n"
"        \"ip-reservations-unique\": true,\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"multi-threading\": {\n"
"            \"enable-multi-threading\": true,\n"
"            \"packet-queue-size\": 64,\n"
"            \"thread-pool-size\": 0\n"
"        },\n"
"        \"next-server\": \"0.0.0.0\",\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"parked-packet-limit\": 256,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"reservations-global\": false,\n"
"        \"reservations-in-subnet\": true,\n"
"        \"reservations-lookup-first\": false,\n"
"        \"reservations-out-of-pool\": false,\n"
"        \"sanity-checks\": {\n"
"            \"extended-info-checks\": \"fix\",\n"
"            \"lease-checks\": \"warn\"\n"
"        },\n"
"        \"server-hostname\": \"\",\n"
"        \"server-tag\": \"\",\n"
"        \"shared-networks\": [ ],\n"
"        \"statistic-default-sample-age\": 0,\n"
"        \"statistic-default-sample-count\": 20,\n"
"        \"store-extended-info\": false,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"ddns-conflict-resolution-mode\": \"check-with-dhcid\",\n"
"                \"id\": 1,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"ddns-conflict-resolution-mode\": \"check-exists-with-dhcid\",\n"
"                \"id\": 2,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"ddns-conflict-resolution-mode\": \"no-check-without-dhcid\",\n"
"                \"id\": 3,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.4.1-192.0.4.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"ddns-conflict-resolution-mode\": \"no-check-with-dhcid\",\n"
"                \"id\": 4,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.5.1-192.0.5.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"allocator\": \"iterative\",\n"
"                \"calculate-tee-times\": false,\n"
"                \"id\": 5,\n"
"                \"max-valid-lifetime\": 4000,\n"
"                \"min-valid-lifetime\": 4000,\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"option-data\": [ ],\n"
"                        \"pool\": \"192.0.6.1-192.0.6.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-addresses\": [ ]\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservations\": [ ],\n"
"                \"store-extended-info\": false,\n"
"                \"subnet\": \"192.0.6.0/24\",\n"
"                \"t1-percent\": 0.5,\n"
"                \"t2-percent\": 0.875,\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ],\n"
"        \"t1-percent\": 0.5,\n"
"        \"t2-percent\": 0.875,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n"
};

/// @brief the number of configurations
const size_t max_config_counter = sizeof(EXTRACTED_CONFIGS) / sizeof(char*);
///@}

/// @brief the extraction counter
///
/// < 0 means do not extract, >= 0 means extract on extractConfig() calls
/// and increment
#ifdef EXTRACT_CONFIG
int extract_count = 0;
#else
int extract_count = -1;
#endif

/// @brief the generate action
/// false means do nothing, true means unparse extracted configurations
#ifdef GENERATE_ACTION
const bool generate_action = true;
#else
const bool generate_action = false;
static_assert(max_config_counter == sizeof(UNPARSED_CONFIGS) / sizeof(char*),
              "unparsed configurations must be generated");
#endif

/// @brief format and output a configuration
void
outputFormatted(const std::string& config) {
    // pretty print it
    ConstElementPtr json = parseDHCP4(config);
    std::string prettier = prettyPrint(json, 4, 4);
    // get it as a line array
    std::list<std::string> lines;
    boost::split(lines, prettier, boost::is_any_of("\n"));
    // add escapes using again JSON
    std::list<std::string> escapes;
    while (!lines.empty()) {
        const std::string& line = lines.front();
        ConstElementPtr escaping = Element::create(line + "\n");
        escapes.push_back(escaping->str());
        lines.pop_front();
    }
    // output them on std::cerr
    while (!escapes.empty()) {
        std::cerr << "\n" << escapes.front();
        escapes.pop_front();
    }
}

}  // namespace

namespace isc {
namespace dhcp {
namespace test {

/// @ref isc::dhcp::test::extractConfig in the header
void
extractConfig(const std::string& config) {
    // skip when disable
    if (extract_count < 0) {
        return;
    }
    // mark beginning
    if (extract_count == 0) {
        // header (note there is no trailer)
        std::cerr << "/// put this after const char* EXTRACTED_CONFIGS[] = {\n";
    } else {
        // end of previous configuration
        std::cerr << ",\n";
    }
    std::cerr << "    // CONFIGURATION " << extract_count;
    try {
        outputFormatted(config);
    } catch (...) {
        // mark error
        std::cerr << "\n//// got an error\n";
    }
    ++extract_count;
}

}  // namespace test
}  // namespace dhcp
}  // namespace isc

namespace {

/// Test fixture class (code from Dhcp4ParserTest)
class Dhcp4GetConfigTest : public ::testing::TestWithParam<size_t> {
public:
    Dhcp4GetConfigTest()
    : rcode_(-1) {
        // Open port 0 means to not do anything at all. We don't want to
        // deal with sockets here, just check if configuration handling
        // is sane.
        srv_.reset(new ControlledDhcpv4Srv(0));
        // Create fresh context.
        resetConfiguration();
    }

    ~Dhcp4GetConfigTest() {
        resetConfiguration();
    };

    /// @brief Parse and Execute configuration
    ///
    /// Parses a configuration and executes a configuration of the server.
    /// If the operation fails, the current test will register a failure.
    ///
    /// @param config Configuration to parse
    /// @param operation Operation being performed.  In the case of an error,
    ///        the error text will include the string "unable to <operation>.".
    ///
    /// @return true if the configuration succeeded, false if not.
    bool
    executeConfiguration(const std::string& config, const char* operation) {
        // clear config manager
        CfgMgr::instance().clear();

        // enable fake network interfaces
        IfaceMgrTestConfig test_config(true);

        // try JSON parser
        ConstElementPtr json;
        try {
            json = parseJSON(config);
        } catch (const std::exception& ex) {
            ADD_FAILURE() << "invalid JSON for " << operation
                          << " failed with " << ex.what()
                          << " on\n" << config << "\n";
            return (false);
        }

        // try DHCP4 parser
        try {
            json = parseDHCP4(config, true);
        } catch (...) {
            ADD_FAILURE() << "parsing failed for " << operation
                          << " on\n" << prettyPrint(json) << "\n";
            return (false);
        }

        // try DHCP4 configure
        ConstElementPtr status;
        try {
            status = configureDhcp4Server(*srv_, json);
        } catch (const std::exception& ex) {
            ADD_FAILURE() << "configure for " << operation
                          << " failed with " << ex.what()
                          << " on\n" << prettyPrint(json) << "\n";
            return (false);
        }

        // The status object must not be NULL
        if (!status) {
            ADD_FAILURE() << "configure for " << operation
                          << " returned null on\n"
                          << prettyPrint(json) << "\n";
            return (false);
        }

        // Returned value should be 0 (configuration success)
        comment_ = parseAnswer(rcode_, status);
        if (rcode_ != 0) {
            string reason = "";
            if (comment_) {
                reason = string(" (") + comment_->stringValue() + string(")");
            }
            ADD_FAILURE() << "configure for " << operation
                          << " returned error code "
                          << rcode_ << reason << " on\n"
                          << prettyPrint(json) << "\n";
            return (false);
        }
        return (true);
    }

    /// @brief Reset configuration database.
    ///
    /// This function resets configuration data base by
    /// removing all subnets and option-data. Reset must
    /// be performed after each test to make sure that
    /// contents of the database do not affect result of
    /// subsequent tests.
    void resetConfiguration() {
        string config = "{"
            "\"interfaces-config\": { \"interfaces\": [ \"*\" ] },"
            "\"valid-lifetime\": 4000, "
            "\"subnet4\": [ ], "
            "\"dhcp-ddns\": { \"enable-updates\" : false }, "
            "\"option-def\": [ ], "
            "\"option-data\": [ ] }";
        EXPECT_TRUE(executeConfiguration(config, "reset configuration"));
        CfgMgr::instance().clear();
        CfgMgr::instance().setFamily(AF_INET);
    }

    boost::scoped_ptr<ControlledDhcpv4Srv> srv_; ///< DHCP4 server under test
    int rcode_;                         ///< Return code from element parsing
    ConstElementPtr comment_;           ///< Reason for parse fail
};

/// Test a configuration
TEST_P(Dhcp4GetConfigTest, run) {
    // configurations have not been extracted yet
    if (max_config_counter == 0) {
        return;
    }

    // get the index of configurations to test
    size_t config_counter = GetParam();

    // emit unparsed header if wanted
    if ((config_counter == 0) && generate_action) {
        std::cerr << "///put this after const char* UNPARSED_CONFIGS[] = {\n";
    }

    // get the extracted configuration
    std::string config = EXTRACTED_CONFIGS[config_counter];
    std::ostringstream ss;
    ss << "extracted config #" << config_counter;

    // execute the extracted configuration
    ASSERT_TRUE(executeConfiguration(config, ss.str().c_str()));

    // unparse it
    ConstSrvConfigPtr extracted = CfgMgr::instance().getStagingCfg();
    ConstElementPtr unparsed;
    ASSERT_NO_THROW(unparsed = extracted->toElement());
    ConstElementPtr dhcp;
    ASSERT_NO_THROW(dhcp = unparsed->get("Dhcp4"));
    ASSERT_TRUE(dhcp);

    // dump if wanted else check
    std::string expected;
    if (generate_action) {
        if (config_counter > 0) {
            std::cerr << ",\n";
        }
        std::cerr << "    // CONFIGURATION " << config_counter;
        ASSERT_NO_THROW_LOG(expected = prettyPrint(dhcp));
        ASSERT_NO_THROW_LOG(outputFormatted(dhcp->str()));
    } else {
        expected = UNPARSED_CONFIGS[config_counter];
        // get the expected config using the dhcpv4 syntax parser
        ElementPtr jsond;
        ASSERT_NO_THROW_LOG(jsond = parseDHCP4(expected, true));
        ElementPtr jsonj;
        // get the expected config using the generic JSON syntax parser
        ASSERT_NO_THROW_LOG(jsonj = parseJSON(expected));
        // the generic JSON parser does not handle comments
        EXPECT_TRUE(isEquivalent(jsond, moveComments(jsonj)));
        // check that unparsed and expected values match
        EXPECT_TRUE(isEquivalent(dhcp, jsonj));
        // check on pretty prints too
        std::string current = prettyPrint(dhcp, 4, 4) + "\n";
        EXPECT_EQ(expected, current);
        if (expected != current) {
            expected = current;
        }
    }

    // execute the dhcp configuration
    ss.str("");
    ss << "unparsed config #" << config_counter;
    EXPECT_TRUE(executeConfiguration(expected, ss.str().c_str()));

    // is it a fixed point?
    ConstSrvConfigPtr extracted2 = CfgMgr::instance().getStagingCfg();
    ConstElementPtr unparsed2;
    ASSERT_NO_THROW_LOG(unparsed2 = extracted2->toElement());
    ASSERT_TRUE(unparsed2);
    EXPECT_TRUE(isEquivalent(unparsed, unparsed2));
}

class IntToString {
public:
    std::string operator()(const testing::TestParamInfo<size_t>& n) {
        std::ostringstream ss;
        ss << static_cast<size_t>(n.param);
        return (ss.str());
    }
};

/// Define the parameterized test loop.
#ifdef INSTANTIATE_TEST_SUITE_P
INSTANTIATE_TEST_SUITE_P(Dhcp4GetConfigTest, Dhcp4GetConfigTest,
                         ::testing::Range(static_cast<size_t>(0),
                                          max_config_counter),
                         IntToString());
#else
INSTANTIATE_TEST_CASE_P(Dhcp4GetConfigTest, Dhcp4GetConfigTest,
                        ::testing::Range(static_cast<size_t>(0),
                                         max_config_counter),
                        IntToString());
#endif
}  // namespace
