/* Copyright (C) 1999-2013 Thomas S. Glascock */
/* www.soomka.com */

#include <pthread.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ncurses.h>
#include <math.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <time.h>

#define HOUR 1
#define MINUTE 2
#define SECOND 3
#define VERSION "0.3.1a"

#define ALARMLENGTH (45)
#define SNOOZELENGTH (5)

char key;
int getkeys;
int rewriteconf;
int ignoreconf;

void *keygetcher(void *)
{
	extern char key;

	while (1)
	{
		pthread_testcancel();
		key=getch();
	}
	return 0;
}

inline int is_even(int numma)
{
	return (!(numma & 1));
}

inline float htod(int hours, int minutes, int seconds)
{
	return	 	( ((float)(hours>12?hours-12:hours) + ((float)minutes / 60.0) + ((float)seconds / 3600.0)) * 30.0 );
}

inline float mtod(int minutes, int seconds, int microseconds)
{
	return	 	( ((float)minutes + ((float)seconds / 60.0)  + ((float)microseconds / 60000000.0)) * 6.0 );
}

inline float stod(int seconds, int microseconds)
{
	return	 	( ((float)seconds + ((float)microseconds  / 1000000.0)) * 6.0 );
}

class clockywock
{
	public:
		clockywock();
		~clockywock();
		void PrintFace();
		void PrintHands(int,int,int,int,int);
		void RunClock();
		void Info();
		void Slide();
		void RedoWindows();
		void SaveSettings();
		void LoadSettings();

	private:
		FILE *clockywockfile;
		char *clockywockfilename;
		struct stat *confstat, *oldconfstat;
		int wb1;
		int wb2;
		int wb3;
		int rolex;
		int bouncy;
		int snooze;
		int silenced;
		int tripped;
		int alarm_on;
		int alarm_hour;
		int alarm_minute;
		int alarm_ringing;
		int alarm_length;
		int snooze_length;
		int tmptimeout;
		int slider;
		int color;
		int bozo;
                int face_color;
                int hour_color;
                int minute_color;
                int second_color;
                int bg_color;
		int user_lines_toggle;
		int user_lines;
		int orig_lines;
		int main_lines;
		int main_cols;
		int prevcols;
		int prevlines;
		int slide_lines;
		int slide_cols;
		int slide_x;
		int slide_y;
		int slide_x_direction;
		int slide_y_direction;
		int slide_lock;
		long alarmend;
		long snoozeend;
		long tmpend;
		float pct[5];
		float pi;
		float aspect;
		float pi_six;
		float pi_180;
		float y_half;
		float x_half;
		float ymult;
		float prevhour;
		float prevminute;
		float prevsecond;
		char *junk;
		char *symbol[5];
		char *cnames[10];
		WINDOW *clock_win;
};

clockywock::clockywock()
{
	int i;

	clockywockfilename=(char *)calloc(1,256);
	strcat(clockywockfilename,getenv("HOME"));
	strcat(clockywockfilename,"/.clockywock");

	confstat = (struct stat *)calloc(1,sizeof(struct stat));
	oldconfstat = (struct stat *)calloc(1,sizeof(struct stat));
	stat(clockywockfilename,confstat);
	stat(clockywockfilename,oldconfstat);

	pi=3.14159265358979;
	aspect=1.5;

	junk=(char *)calloc(1,256);

	for (i=1;i<=3;++i) {
		symbol[i]=(char *)calloc(1,1);
	}

	for (i=0;i<=9;++i) {
		cnames[i]=(char *)calloc(1,10);
	}

	sprintf(cnames[1],"default");
	sprintf(cnames[2],"black");
	sprintf(cnames[3],"red");
	sprintf(cnames[4],"green");
	sprintf(cnames[5],"yellow");
	sprintf(cnames[6],"blue");
	sprintf(cnames[7],"magenta");
	sprintf(cnames[8],"cyan");
	sprintf(cnames[9],"white");

	*symbol[1]='h';
	*symbol[2]='m';
	*symbol[3]='s';

	pi_six=pi/6.0;
	pi_180=pi/180.0;

	pct[HOUR]=60.0/100.0;
	pct[MINUTE]=90.0/100.0;
	pct[SECOND]=95.0/100.0;

	alarm_on=0;
	alarm_hour=0;
	alarm_minute=0;
	alarm_ringing=0;

	wb1=wb2=wb3=0;
	rolex=1;
	bouncy=1;
	snooze=0;
	silenced=0;
	tripped=0;
	alarmend=0;
	snoozeend=0;
	tmpend=0;
	tmptimeout=0;
	slider=0;
        color=0;
        bozo=0;
	slide_x=0;
	slide_y=0;
	slide_x_direction=1;
	slide_y_direction=1;
	slide_lock=0;

	face_color=5;
	hour_color=3;
	minute_color=4;
	second_color=6;

	bg_color=COLOR_MAGENTA;

	alarm_length=ALARMLENGTH;
	snooze_length=SNOOZELENGTH;

	initscr();
	start_color();
        assume_default_colors(-1, -1);

	attrset(A_NORMAL);

	curs_set(0);
	halfdelay(0);
	noecho();
        user_lines_toggle=0;
	user_lines=LINES;
        orig_lines=LINES;
	main_lines=LINES;
	main_cols=COLS;
	prevcols=COLS;
	prevlines=LINES;

	clock_win=newwin(main_lines,main_cols,0,0);

	RedoWindows();

	prevhour=prevminute=prevsecond=-1;
}

clockywock::~clockywock()
{
	free(junk);
	curs_set(1);
	nocbreak();
	echo();
	nl();
	endwin();
}

void clockywock::SaveSettings()
{
	extern int rewriteconf;
	if (rewriteconf == 0) return;
	clockywockfile=fopen(clockywockfilename,"w");
        fprintf(clockywockfile,"# Warning: lazy programmer - file format is fixed.\n");
	fprintf(clockywockfile,"# do not rearrange, add or delete lines.\n");
	fprintf(clockywockfile,"# if you hose the file up, just delete and restart clockywock.\n");
	fprintf(clockywockfile,"%d alarm on\n",alarm_on);
	fprintf(clockywockfile,"%d alarm hour\n",alarm_hour);
	fprintf(clockywockfile,"%d alarm minute\n",alarm_minute);
	fprintf(clockywockfile,"%d alarm ringing\n",alarm_ringing);
	fprintf(clockywockfile,"%d alarm length\n",alarm_length);
	fprintf(clockywockfile,"%d snooze length\n",snooze_length);
	fprintf(clockywockfile,"%d slider mode\n",slider);
	fprintf(clockywockfile,"%d color mode\n",color);
	fprintf(clockywockfile,"%d hour color\n",hour_color);
	fprintf(clockywockfile,"%d minute color\n",minute_color);
	fprintf(clockywockfile,"%d second color\n",second_color);
	fprintf(clockywockfile,"%d face color\n",face_color);
	fprintf(clockywockfile,"%d bg color\n",bg_color);
	fprintf(clockywockfile,"%c hour char \n",*symbol[1]);
	fprintf(clockywockfile,"%c minute char \n",*symbol[2]);
	fprintf(clockywockfile,"%c second char \n",*symbol[3]);
	fprintf(clockywockfile,"%f aspect ratio\n",aspect);
	fprintf(clockywockfile,"%d num lines\n",user_lines);
	fprintf(clockywockfile,"%d user_lines toggle\n",user_lines_toggle);
	fprintf(clockywockfile,"%d rolex toggle\n",rolex);
	fprintf(clockywockfile,"%d bouncy toggle\n",bouncy);
	fprintf(clockywockfile,"%d bozo toggle\n",bozo);
	fclose(clockywockfile);
}

void clockywock::LoadSettings()
{
        char *waste1, *waste2;
	int dummy;
	extern int ignoreconf;
	if (ignoreconf == 1) return;
	waste1=(char *)calloc(1,256);
	waste2=(char *)calloc(1,256);
	clockywockfile=fopen(clockywockfilename,"r");
	if (clockywockfile!=NULL)
	{
		waste2=fgets(waste1,256,clockywockfile);
		waste2=fgets(waste1,256,clockywockfile);
		waste2=fgets(waste1,256,clockywockfile);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&alarm_on,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&alarm_hour,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&alarm_minute,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&alarm_ringing,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&alarm_length,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&snooze_length,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&slider,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&hour_color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&minute_color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&second_color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&face_color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&bg_color,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%c %s %s\n",symbol[1],waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%c %s %s\n",symbol[2],waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%c %s %s\n",symbol[3],waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%f %s %s\n",&aspect,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&user_lines,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&user_lines_toggle,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&rolex,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&bouncy,waste1,waste2);
		dummy=fscanf(clockywockfile,(const char *)"%d %s %s\n",&bozo,waste1,waste2);
		// waste call to get rid of compile warning
		sprintf(waste1,"%d",dummy);
		fclose(clockywockfile);
	}
}

void clockywock::Slide()
{
	if (slide_y_direction==1) {
		if (slide_y<slide_lines) slide_y=slide_y+1; else slide_y_direction=-1;
	} else {
		if (slide_y>0) slide_y=slide_y-1; else slide_y_direction=1;
	}

	if (slide_x_direction==1) {
		if (slide_x<slide_cols) slide_x=slide_x+1; else slide_x_direction=-1;
	} else {
		if (slide_x>0) slide_x=slide_x-1; else slide_x_direction=1;
	}

	mvwin(clock_win,slide_y,slide_x);

	werase(stdscr);

	if (color) {
		wbkgd(stdscr,COLOR_PAIR(face_color) | ' ');
	} else {
		wbkgd(stdscr,COLOR_PAIR(0) | ' ');
	}

}

void clockywock::Info()
{


// MACRO
#define infoprint(S,...) \
	sprintf(junk,S,##__VA_ARGS__);\
	if(iwincols<strlen(junk)) iwincols=strlen(junk);\
	wresize(iwin,iwinlines+2,iwincols+3);\
	mvwaddstr(iwin,iwinlines++,2,junk);\
	mvwin(iwin,int((LINES/2)-(iwinlines/2))-1,int((COLS/2)-(iwincols/2))-1);
// MACRO END

	WINDOW *iwin;
	int iwinlines;
	size_t iwincols;
	iwincols=0;
	iwinlines=1;

	iwin=newwin(0,0,2,2);

	if (color) {
		wbkgd(iwin,COLOR_PAIR(13) | ' ');
		wattrset(iwin,COLOR_PAIR(0));
        }

	infoprint("[1!]  hour char               %s",symbol[1]);
	infoprint("[2@]  minute char             %s   [a]     alarm toggle          %3s",	symbol[2],alarm_on==1?"on":"off");
	infoprint("[3#]  second char             %s   [HhMm]  alarm time          %02d:%02d",	symbol[3],alarm_hour,alarm_minute);
	infoprint("[4]   hour color        %7s   [Tt]    alarm length       %2d min",		cnames[hour_color],alarm_length);
	infoprint("[5]   minute color      %7s   [Zz]    snooze length      %2d min",		cnames[minute_color],snooze_length);
	infoprint("[6]   second color      %7s",						cnames[second_color]);
	infoprint("[7]   face color        %7s   [space] snooooze",				cnames[face_color]);
	infoprint("[8]   background color  %7s   [enter] kill alarm",				cnames[bg_color+2]);
	infoprint(" ");
	infoprint("[Ww]  aspect ratio         %3.2f   [s]     slider mode           %s ",	aspect,slider==1?" on":"off");
	infoprint("[Ll]  user # of lines        %2d   [c]     color mode          %s",		user_lines,color==1?"color":" mono");
	infoprint("[i]   user/term lines      %s   [x]     toggle motor/ticks  %s",		user_lines_toggle==1?"user":"term",rolex==1?"motor":"ticks");
	infoprint("[r]   reset aspect/lines          [y]     sloppy second hand  %s",		bouncy==1?"loose":"tight");
	infoprint("                                  [b]     bozo colors           %s",            bozo==1?" on":"off");
	infoprint("[q,(esc)) bye bye");

	box(iwin,0,0);

	mvwchgat(iwin,1,32,1,A_BOLD,color==1?hour_color+8:NULL,NULL);
	mvwchgat(iwin,2,32,1,A_BOLD,color==1?minute_color+8:NULL,NULL); mvwchgat(iwin,2,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,3,32,1,A_BOLD,color==1?second_color+8:NULL,NULL); mvwchgat(iwin,3,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,4,26,8,A_BOLD,color==1?13:NULL,NULL); mvwchgat(iwin,4,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,5,26,8,A_BOLD,color==1?13:NULL,NULL); mvwchgat(iwin,5,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,6,26,8,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,7,26,8,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,8,26,8,A_BOLD,color==1?13:NULL,NULL);

	mvwchgat(iwin,10,26,8,A_BOLD,color==1?13:NULL,NULL); mvwchgat(iwin,10,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,11,26,8,A_BOLD,color==1?13:NULL,NULL); mvwchgat(iwin,11,63,6,A_BOLD,color==1?13:NULL,NULL);
	mvwchgat(iwin,12,26,8,A_BOLD,color==1?13:NULL,NULL); mvwchgat(iwin,12,63,6,A_BOLD,color==1?13:NULL,NULL);
	                                                     mvwchgat(iwin,13,63,6,A_BOLD,color==1?13:NULL,NULL);
	                                                     mvwchgat(iwin,14,63,6,A_BOLD,color==1?13:NULL,NULL);

	overwrite(iwin,clock_win);

	delwin(iwin);
}

void clockywock::RedoWindows()
{

	init_pair(1, -1, bg_color);
	init_pair(2, COLOR_BLACK, bg_color);
	init_pair(3, COLOR_RED, bg_color);
	init_pair(4, COLOR_GREEN, bg_color);
	init_pair(5, COLOR_YELLOW, bg_color);
	init_pair(6, COLOR_BLUE, bg_color);
	init_pair(7, COLOR_MAGENTA, bg_color);
	init_pair(8, COLOR_CYAN, bg_color);
	init_pair(9, COLOR_WHITE, bg_color);
        init_pair(10, COLOR_BLACK, COLOR_BLUE);
        init_pair(11, COLOR_RED, COLOR_BLUE);
        init_pair(12, COLOR_GREEN, COLOR_BLUE);
        init_pair(13, COLOR_YELLOW, COLOR_BLUE);
        init_pair(14, COLOR_BLUE, COLOR_BLUE);
        init_pair(15, COLOR_MAGENTA, COLOR_BLUE);
        init_pair(16, COLOR_CYAN, COLOR_BLUE);
        init_pair(17, COLOR_WHITE, COLOR_BLUE);

	if (slider==1)
	{
		main_lines=(LINES-(LINES/8));
		main_cols=(COLS-(COLS/8));
		slide_lines=LINES-main_lines;
		slide_cols=COLS-main_cols;
	} else {
		if (user_lines_toggle != 0) {
			main_lines=user_lines;
 		} else {
			main_lines=LINES;
		}
		main_cols=COLS;
		// gotta move them back to 0 if exiting slider
		mvwin(clock_win,0,0);
	}

	x_half=(main_cols/2.0)+(is_even(main_cols)?-0.5:0.0);
	y_half=(main_lines/2.0)+(is_even(main_lines)?-0.5:0.0);

	ymult=(y_half/x_half);

	wresize(clock_win,main_lines,main_cols);

	if (color) {
                assume_default_colors(face_color, bg_color);

		wattrset(clock_win,COLOR_PAIR(face_color)|A_BOLD);

		bkgd(COLOR_PAIR(face_color) | ' ');
		wbkgd(clock_win,COLOR_PAIR(face_color) | ' ');

	} else {
                assume_default_colors(-1,-1);

		wattrset(clock_win,0);

		bkgd(COLOR_PAIR(0) | ' ');
		wbkgd(clock_win,COLOR_PAIR(0) | ' ');
	}

	werase(clock_win);
	erase();

	prevhour=-1;
	prevminute=-1;
	prevsecond=-1;
}

void clockywock::PrintFace()
{
	int m;

	if (color) {
		wattrset(clock_win,COLOR_PAIR(face_color)|A_REVERSE);
	} else {
		wattrset(clock_win,COLOR_PAIR(0)|A_REVERSE);
	}
       
	for (m=0;m<=11;++m)
	{
		mvwaddstr(clock_win,
			(int)(y_half + (  y_half * (-cos(pi_six*m)))),
			(int)(x_half + (( x_half * ( sin(pi_six*m)))) / aspect),
			" ");
	}

	if (color) {
		wattrset(clock_win,COLOR_PAIR(face_color)|A_BOLD);
	} else {
		wattrset(clock_win,COLOR_PAIR(0)|A_BOLD);
	}

	sprintf(junk,"clockywock v.%s", VERSION);
	mvwaddstr(clock_win,main_lines-1,0,junk);
	mvwaddstr(clock_win,(int)(y_half),(int)(x_half),"O");
}

void clockywock::PrintHands(int hour, int minute, int second, int usec, int granular)
{
	float x;
	float sinzh, coszh, sinzm, coszm, sinzs, coszs;
	float handlengthh, handlengthm, handlengths, maxhl;
	float unith, unitm, units;

        maxhl=1.0;

	werase(clock_win);

        if (color) {
		wbkgd(clock_win,COLOR_PAIR(hour_color) | ' ');
	} else {
		wbkgd(clock_win,COLOR_PAIR(0) | ' ');
	}

	unith=htod(hour,minute,second) * pi_180;
	if (granular == 1) {
		unitm=mtod(minute,second,usec) * pi_180;
	} else {
		unitm=mtod(minute,second,0) * pi_180;
	}
	units=stod(second,usec)        * pi_180;

	coszh=((cos(unith))*ymult); sinzh=((sin(unith))/aspect);
	coszm=((cos(unitm))*ymult); sinzm=((sin(unitm))/aspect);
	coszs=((cos(units))*ymult); sinzs=((sin(units))/aspect);

	handlengthh=(int)(x_half * pct[HOUR]);
	handlengthm=(int)(x_half * pct[MINUTE]);
	handlengths=(int)(x_half * pct[SECOND]);

        if (maxhl < handlengthh) maxhl = handlengthh;
        if (maxhl < handlengthm) maxhl = handlengthm;
        if (maxhl < handlengths) maxhl = handlengths;

	wattrset(clock_win,0);

	for (x=1;x!=maxhl;++x)
	{
		if (x <= handlengthh) {
			if (color) wattrset(clock_win, COLOR_PAIR(hour_color)|A_BOLD);
			if (color) if (bozo) wattrset(clock_win, COLOR_PAIR(random()%8)|A_BOLD);
			mvwaddstr(clock_win, (int)(y_half+(x*-coszh)), (int)(x_half+(x*sinzh)), symbol[HOUR]);
		}
		if (x <= handlengthm) {
			if (color) wattrset(clock_win, COLOR_PAIR(minute_color)|A_BOLD);
			if (color) if (bozo) wattrset(clock_win, COLOR_PAIR(random()%8)|A_BOLD);
			mvwaddstr(clock_win, (int)(y_half+(x*-coszm)), (int)(x_half+(x*sinzm)), symbol[MINUTE]);
		}
		if (x <= handlengths) {
			if (color) wattrset(clock_win, COLOR_PAIR(second_color)|A_BOLD);
			if (color) if (bozo) wattrset(clock_win, COLOR_PAIR(random()%8)|A_BOLD);
			mvwaddstr(clock_win, (int)(y_half+(x*-coszs)), (int)(x_half+(x*sinzs)), symbol[SECOND]);
		}
	}

	PrintFace();
	prevhour=hour;
	prevminute=minute;
	prevsecond=second;
}

void clockywock::RunClock()
{
	extern char key;
	extern int getkeys;

	struct timeval td;
	struct tm *ct;

	pthread_t keythread;

	ct=(struct tm*)NULL;

	LoadSettings();

	if (getkeys == 1) pthread_create(&keythread,NULL,keygetcher,NULL);

	RedoWindows();

	for (;;)
	{
		// If config file changes, reread it and apply changes.

		stat(clockywockfilename,confstat);
		if (oldconfstat->st_mtime != confstat->st_mtime) {
			LoadSettings();
			stat(clockywockfilename,oldconfstat);
			RedoWindows();
		}

		// If the user changes the window size, redo it.

		if ((COLS!=prevcols)||(LINES!=prevlines))
		{
			RedoWindows();
			prevcols=COLS;
			prevlines=LINES;
		}

		gettimeofday(&td,(struct timezone *)NULL);
		ct=localtime(&td.tv_sec);


		// PRINT HANDS
		if (rolex==0) {
			if (bouncy != 0) {
				if (td.tv_usec > 0	&& wb1==0) 	{wb1=1;PrintHands(ct->tm_hour, ct->tm_min, ct->tm_sec, 200000,0);	}
				if (td.tv_usec > 60000	&& wb2==0)	{wb2=1;PrintHands(ct->tm_hour, ct->tm_min, ct->tm_sec, -150000,0);	}
				if (td.tv_usec > 120000	&& wb3==0)	{wb3=1;PrintHands(ct->tm_hour, ct->tm_min, ct->tm_sec, 0,0);		}
				if (prevsecond != ct->tm_sec) wb1=wb2=wb3=0;
			} else {
				if (prevsecond != ct->tm_sec) PrintHands(ct->tm_hour, ct->tm_min, ct->tm_sec, td.tv_usec,1);
			}
		} else {
			PrintHands(ct->tm_hour, ct->tm_min, ct->tm_sec, td.tv_usec,1);
		}

		// DONE PRINTING HANDS

		if (alarm_on==1)
		{
			if (silenced==0)
			{
				if (alarmend > td.tv_sec)
				{
					if (snoozeend < td.tv_sec)
					{
						alarm_ringing=1;
						snooze=0;
					} else
						alarm_ringing=0;
				} else
					alarmend=snooze=alarm_ringing=tripped=0;
			} else
			{
				if (alarmend < td.tv_sec)
					silenced=tripped=alarm_ringing=0;
				else
				{
					alarm_ringing=0;
				}
			}
			if (((ct->tm_hour==alarm_hour)&&(ct->tm_min==alarm_minute)))
			{
				if (tripped==0)
				{
					alarmend=td.tv_sec+(alarm_length * 60) ;
					tripped=1;
				}
			}
			if (alarm_ringing==1) { beep(); flash(); sleep(1); }
		}

		if (tmpend<td.tv_sec) { tmpend=tmptimeout=0; }

		// key is fetched in a separate thread

		if ((key!=-1 && key!=255))
		{
			tmpend=td.tv_sec+5;
			tmptimeout=1;

			switch (key)
			{

				/* toggle alarm on/off */
				case 'a': alarm_on=(1-(alarm_on)); break;

				/* set alarm time */
	
				case 'h': ++alarm_hour ; if (alarm_hour > 23) alarm_hour=0; break;
				case 'H': --alarm_hour ; if (alarm_hour <  0) alarm_hour=23; break;
	
				case 'm': ++alarm_minute ; if (alarm_minute > 59) alarm_minute=0; break;
				case 'M': --alarm_minute ; if (alarm_minute <  0) alarm_minute=59; break;
	
				case 't': ++alarm_length ; if (alarm_length > 60) alarm_length=15; break;
				case 'T': --alarm_length ; if (alarm_length < 15) alarm_length=60; break;

				/* toggle rolex mode */

				case 'x': rolex = 1 - rolex ; break;
				case 'y': if (rolex==0) bouncy = 1 - bouncy ; break;

				/* set snooze length */
	
				case 'z': ++snooze_length ; if (snooze_length >30) snooze_length = 3; break;
				case 'Z': --snooze_length ; if (snooze_length <3) snooze_length = 30; break;
	
				/* set lines */
	
				case 'L':	user_lines=user_lines+1 ; 
						if (user_lines > 199) user_lines = 199; 
						RedoWindows() ;
						break;

				case 'l':	user_lines=user_lines-1 ; 
						if (user_lines < 2) user_lines = 2; 
						RedoWindows() ;
						break;

				case 'i': 	user_lines_toggle = 1 - user_lines_toggle ; 
						RedoWindows() ;
						break;

				/* set aspect */
	
				case 'w': aspect=aspect+0.1 ; if (aspect >3) aspect = 3; RedoWindows() ; break;
				case 'W': aspect=aspect-0.1 ; if (aspect <.3) aspect = .3; RedoWindows() ; break;

				/* set aspect */
	
				case 'r': aspect=1.333333333 ; LINES=orig_lines;  RedoWindows() ; break;

				/* hand char change */
				case '1': ++*symbol[1] ; if (*symbol[1] >126) *symbol[1] = 33; prevsecond=-1; break;
				case '!': --*symbol[1] ; if (*symbol[1] < 33) *symbol[1] = 126; prevsecond=-1; break;
	
				case '2': ++*symbol[2] ; if (*symbol[2] >126) *symbol[2] = 33; prevminute=-1; break;
				case '@': --*symbol[2] ; if (*symbol[2] < 33) *symbol[2] = 126; prevminute=-1; break;
	
				case '3': ++*symbol[3] ; if (*symbol[3] >126) *symbol[3] = 33; break;
				case '#': --*symbol[3] ; if (*symbol[3] < 33) *symbol[3] = 126; break;

				/* color change */
				/* hour, minute, second, and face cycle by COLOR PAIR number */
				/* background cycles by ACTUAL COLOR number */

				case '4': ++hour_color   ; if (hour_color > 9 || hour_color < 1 )   hour_color=1;
						RedoWindows();break;
				case '5': ++minute_color ; if (minute_color > 9 || minute_color < 1 ) minute_color=1;
						RedoWindows();break;
				case '6': ++second_color ; if (second_color > 9 || second_color < 1 ) second_color=1; 
						RedoWindows();break;
				case '7': ++face_color   ; if (face_color > 9 || face_color < 1 )   face_color=1;   
						RedoWindows();break;

				case '8': ++bg_color     ; if (bg_color > 7 || bg_color < -1)     bg_color=-1;     
						RedoWindows();break;
	
	
				/* say bye bye */

				case 27:
				case 'q':
				case 'Q':
				{
					clear();
					SaveSettings();
					return;
					break;
				}
	
				/* toggle color */

				case 'c':
					color=1-color;
                                        RedoWindows();
					break;

				case 'b':
					bozo=1-bozo;
                                        RedoWindows();
					break;

				/* toggle slider mode */

				case 's':
					slider=1-slider;
					RedoWindows();
					break;

				/* snooze */
	
				case ' ': 
					if ((alarm_on==1) && (alarm_ringing==1))
					{
						snooze=1;
						snoozeend=td.tv_sec+ (snooze_length * 60);
					}
					break;

				/* silence alarm */
	
				case 10:
					if (alarm_on==1) silenced=1; break;
	
				case 13:
					if (alarm_on==1) silenced=1; break;
	
			}
			key=-1;
		}

		// Always slide before redoing the Info window
		if ((slider==1)  && (slide_lock==0) && ((ct->tm_sec%5)==1))
		{
			Slide();
			slide_lock=1;
		} 

		// Put up info window while info window timeout is going on
		if (tmptimeout==1)
			Info();

		if (((ct->tm_sec - 1)%5)==1) slide_lock=0;

		wnoutrefresh(stdscr);
		wnoutrefresh(clock_win);
		doupdate();
		usleep(30000);
	} // for loop
}

int main(int argc, char **argv)
{
	extern char key;

	int q;

	extern int getkeys;
	extern int rewriteconf;
	extern int ignoreconf;

	getkeys=1;
	ignoreconf=0;
	rewriteconf=1;
	key=-1;

	while (1) {
		q = getopt (argc, argv, "hkri");

		if (q == -1)
			break;

		switch (q) {
			case 'h':
				printf ("help\n");
				printf ("\n");
				printf ("Args: -h = this helpful help screen\n");
				printf ("      -k = ignore key input (ctrl-c to exit)\n");
				printf ("      -i = ignore config but leave config file alone\n");
				printf ("      -r = ignore config and rewrite upon exit\n");
				printf ("\n");
				exit(0);
				break;
			case 'k':
				getkeys = 0;
				break;
			case 'r':
				ignoreconf = 1;
				rewriteconf = 1;
				break;
			case 'i':
				ignoreconf = 1;
				rewriteconf = 0;
				break;
		}
	}

	clockywock slimex;

	slimex.RunClock();
}
