/******************************************************************************
*******************************************************************************
**
**    Copyright 1997 Kenneth W. Preslan <kpreslan@lcse.umn.edu>
**              1998-1999 Grant M. Erickson <grant@lcse.umn.edu>
**
**    University of Minnesota
**    Department of Electrical and Computer Engineering
**
**    Brocade Communications Systems, Inc.
** 
**    Module name: fwdl-prisa.C
**    Date:        1998/09/30
**
**    Description:
**      This modules implements code which downloads firmware to a Seagate
**      drive attached to a Prisa Networks NetFX Fibre Channel adapter.
**
*******************************************************************************
******************************************************************************/

/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *  
 *    This program is distributed in the hope that it will be useful,  
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <nport.h>
#include <scsiInitiator.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/fc.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "fwdl.h"
#include "fwdl-prisa.h"


// Global Variables

static si_req      req;		// NetFX command block and data structure

// Function Prototypes

static int	 openDevice(const char *device, int flags);
static int	 closeDevice(int fd);
static int	 scsiTestUnitReady(int fd);
static int	 scsiInquiry(int fd, void *buffer);
static int	 scsiWriteFW(int fd, void *buffer, unsigned int size);
#ifdef DEBUG
static void	 dumpReqStruct(void);
#endif
static int	 checkStatus(void *data);

scsi_ops_t prisaOps = {
	openDevice,
	closeDevice,
	scsiTestUnitReady,
	scsiInquiry,
	scsiWriteFW,
};


/******************************************************************************
*******************************************************************************
**
** static int openDevice()
**
** Description:
**   This routine opens a Prisa Fibre Channel device.
**
** Input(s):
**  *device - Name of the target device to open.
**   flags  - Not used.
**
** Returns:
**   A Prisa "file descriptor" if OK, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
openDevice(const char *device, int flags)
{
  struct stat statbuf;
  int fd = -1;
  fc_NameIdentifier wwn;

  // Check to see if the device actually exists first
  
  if (stat(device, &statbuf) < 0)
    {
      fprintf(stderr, "Could not open \"%s\": %s\n", device, strerror(errno));
      exit(EXIT_FAILURE);
    }
  else if (!S_ISCHR(statbuf.st_mode) && !S_ISBLK(statbuf.st_mode))
    {
      fprintf(stderr, "The device file \"%s\" is not a character or block "
	      "device.\n", device);
      exit(EXIT_FAILURE);
    }

  if ((fd = open(device, O_RDONLY)) < 0)
    {
      fprintf(stderr, "%s: open: %s\n", device, strerror(errno));
      close(fd);
      exit(EXIT_FAILURE);
    }

  if (ioctl(fd, _IOR(202, 6, fc_NameIdentifier), &wwn) < 0)
    {
      fprintf(stderr, "%s: ioctl: %s\n", device, strerror(errno));
      exit(EXIT_FAILURE);
    }

  close(fd);

#ifdef DEBUG
  if (debug)
    fprintf(stderr, "Device WWN: 0x%llx\n", wwn);
#endif

  // We need to have exclusive access to the device in order to
  // download the firmware.

  if ((fd = siOpen(wwn, siOpen_Exclusive, 1)) < 0)
    {
      fprintf(stderr,
	      "Could not open \"%s\" for read/write/exclusive access.\n", 
	      device);
      perror("");
      exit(EXIT_FAILURE);
    }

  return(fd);
}


/******************************************************************************
*******************************************************************************
**
** static int closeDevice()
**
** Description:
**   This routine closes the device associated with the Prisa "file 
**   descriptor".
**
** Input(s):
**   fd - A Prisa "file descriptor" obtained via siOpen().
**
** Returns:
**   Always returns 0.
**
*******************************************************************************
******************************************************************************/

static int
closeDevice(int fd)
{
  siClose(fd);

  return (0);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiTestUnitReady()
**
** Description:
**   This routine closes the device associated with the Prisa "file
*    descriptor".
**
** Input(s):
**   fd - A Prisa "file descriptor" obtained via siOpen().
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
scsiTestUnitReady(int fd)
{
  int status;

  // Clear command block and data request structure
  bzero(&req, sizeof(si_req));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));

  req.lun = 0x00;
  req.cdb.w1 = 0x00000000;
  req.cdb.w2 = 0x00000000;
  req.cdb.w3 = 0x00000000;
  req.cdb.w4 = 0x00000000;
  req.task_attribute = siSimple;
  req.transtype = siNoTransfer;
  req.timeout = SCSI_CMD_TIMEOUT;
  req.sense = (u_char *)&global_sdata;        
  req.sense_bytes = sizeof(senseData_t);

  status = siRequest(fd, &req);

#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif

  checkStatus(&req);
  return(status);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiInquiry()
**
** Description:
**   This routine...
**
** Input(s):
**
**
** Output(s):
**
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
scsiInquiry(int fd, void *buffer)
{
  int   status;

  // Clear command block and data request structure area
  bzero(&req, sizeof(si_req));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));
      
  req.lun = 0x00;
  req.cdb.w1 = 0x12000000;
  req.cdb.w2 = 0x30000000;
  req.cdb.w3 = 0x00000000;
  req.cdb.w4 = 0x00000000;
  req.task_attribute = siSimple;
  req.transtype = siRead;
  req.timeout = SCSI_CMD_TIMEOUT;
  req.data = (caddr_t)buffer;
  req.data_bytes = sizeof(inquiryData_t);
  req.sense = (u_char *)&global_sdata;        
  req.sense_bytes = sizeof(senseData_t);
  
  status = siRequest(fd, &req);

#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif

  checkStatus(&req);
  return(status);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiWriteFW()
**
** Description:
**   This routine...
**
** Input(s):
**
**
** Output(s):
**
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
scsiWriteFW(int fd, void *buffer, unsigned int size)
{
  int   status;

  // Clear command block and data request structure area
  bzero(&req, sizeof(si_req));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));
      
  req.lun = 0x00;
  req.cdb.w1 = 0x3B050000;
  req.cdb.w2 = (size & 0x00FFFF00) >>  8;
  req.cdb.w3 = (size & 0x000000FF) << 24;
  req.cdb.w4 = 0x00000000;
  req.task_attribute = siSimple;
  req.transtype = siWrite;
  req.timeout = SCSI_CMD_TIMEOUT;
  req.data = (caddr_t)buffer;
  req.data_bytes = size;
  req.sense = (u_char *)&global_sdata;        
  req.sense_bytes = sizeof(senseData_t);
  
  status = siRequest(fd, &req);

#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif
  
  if (checkStatus(&req) == siCheckCondition)
    {
      if (global_sdata.asc == 0x26)
	{
	  if (global_sdata.ascq == 0x98)
	    {
	      fprintf(stderr, "Firmware checksum failed!\n");
	    }
	  else if (global_sdata.ascq == 0x99)
	    {
	      fprintf(stderr, "The firmware is incompatible with this product!\n");
	    }
	}
    }
  
  return(status);
}

#ifdef DEBUG
/******************************************************************************
*******************************************************************************
**
** void dumpReqStruct()
**
** Description:
**
** Input(s):
**
** Output(s):
**
** Returns:
**
*******************************************************************************
******************************************************************************/

void dumpReqStruct(void)
{
  fprintf(stderr, "\nSCSI initiator request structure:\n");
  fprintf(stderr, "  Request structure address:  %#08x\n", &req);
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  fprintf(stderr, "Logical Unit:      0x%03x\t", req.lun);
  fprintf(stderr, "Task Attribute:    0x%03x\n", req.task_attribute);
  fprintf(stderr, "Timeout (sec):        %01.1f\t", 
	  (float)(req.timeout)/100.0);
  fprintf(stderr, "Transfer Type:     %s\n", 
	  (req.transtype == 0 ? "NONE" : 
	   (req.transtype == 1 ? "READ" : "WRITE")));
  fprintf(stderr, "Data Xfer Length:  0x%03x\t", req.data_bytes);
  fprintf(stderr, "Sense Buffer Size: 0x%03x\n", req.sense_bytes);
  fprintf(stderr, "SCSI Command Block:\t");
  fprintf(stderr, "0x%08x\n", req.cdb.w1);
  fprintf(stderr, "\t\t\t0x%08x\n", req.cdb.w2);
  fprintf(stderr, "\t\t\t0x%08x\n", req.cdb.w3);
  fprintf(stderr, "\t\t\t0x%08x\n", req.cdb.w4);
  if (req.transtype != siNoTransfer)
    {
      fprintf(stderr, 
	      "--------------------------------------------------------\n");
      fprintf(stderr, "Data Buffer: (%d bytes)\n\t", req.data_bytes);
      for (int i = 0; i < req.data_bytes; i++)
	{
	  fprintf(stderr, "%02x ", req.data[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr,"\n");
    }
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  fprintf(stderr, "Result:       %02d\t", req.result);
  fprintf(stderr, "SCSI Status:  0x%02x\n", req.status);
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  if (req.status != 0)
    {
      fprintf(stderr, "Sense data: (%d bytes)\n\t", 
	      req.sense_received);
      for (int i = 0; i < sizeof(senseData_t); i++)
	{
	  fprintf(stderr, "%02x ", req.sense[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr, 
	      "\n--------------------------------------------------------\n");
    }
  fflush(stderr);    
}
#endif // DEBUG


/******************************************************************************
*******************************************************************************
**
** static int checkStatus()
**
** Description:
**   This routine...
**
** Input(s):
**
**
** Output(s):
**
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
checkStatus(void *data)
{
  struct si_req	*req = (struct si_req *)data;


  if (req->status == 0)
    return(0);
  else
    {
      fprintf(stderr, "Sense data: (%d bytes)\n\t", req->sense_received);
      for (int i = 0; i < sizeof(senseData_t); i++)
	{
	  fprintf(stderr, "%02x ", req->sense[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr, 
	      "\n--------------------------------------------------------\n");

      switch(req->status)
	{
	case siReservationConflict:
	  fprintf(stderr, "Reservation Conflict\n");
	  break;
	case siCheckCondition:
	  fprintf(stderr, "Check Condition\n");
	  break;
	case siConditionMet:
	case siBusy:
	case siIntermediate:
	case siIntermediateConditionMet:
	case siCommandTerminated:
	case siTaskSetFull:
	case siACAActive:
	  fprintf(stderr, "Bad Condition: 0x%x\n", req->status);
	  break;
	}
      return(req->status);
    }
}
