/***********************************************************************
* Name:
*        dapismp.c
*
* Function:
*        Sample IBM Storage ProtectAPI application.  Main entry 
*        point and top level functions.
*
* Status:
*        Version 2, Release 1
*
* Environment:
*        This is a PLATFORM-INDEPENDENT source file. As such it may
*        contain no dependencies on any specific operating system
*        environment or hardware platform.
*
* Description:
*        The file contains the functions that are invoked from the
*        menu processing code and tables.
***********************************************************************/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dsmapitd.h"      /* IBM Storage Protect API type definitions        */
#include "dsmapifp.h"      /* IBM Storage Protect API function prototypes.    */
#include "dapitype.h"      /* IBM Storage Protect Sample API type definitions.*/
#include "dapiproc.h"
#include "dapidata.h"      /* IBM Storage Protect Sample API values and menus.*/
#include "dapiutil.h"

#ifdef _MAC
#include <syslog.h>
#include <locale.h>
#define MAC_UTF8_LOCALE "en_US.UTF-8"
#endif

/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/
static void Display_Menu(menu *xmenu);
static void Process_Menu(menu *xmenu);
static void Set_Preferences(preferences *pref);
static int  ValidateParam(char *instr, char *compstr, int min, int max);
static int  ParseCommandLine(int argc, char *argv[]);

char **globalArgv;

#ifdef DYNALOAD_DSMAPI

/* Stub for dynamic loader */

int main(int argc, char **argv)

{

/*=== Set correct locale ===*/
#ifdef _MAC
   setlocale(LC_ALL, MAC_UTF8_LOCALE);
#endif

   printf("\n\nLoading IBM Storage Protect API ...\n");
   if (!LoadADSMApi())
   {
      printf("\nError Loading API, exiting.\n\n");
      return -1;
   }
   printf ("\n\n");

   sampmain(argc, argv);


   UnLoadADSMApi();

   return 0;
}

#endif


/*----------------------------------------------------------------------+
| Name:    main()
|
| Action:  Main entry point for the IBM Storage Protect sample API
|          application. Put out welcome banner, get API version, and 
|          start menu processing for the application.
|
| Input:   None
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/


#ifdef DYNALOAD_DSMAPI
int sampmain(int argc,char **argv)
#else
int main(int argc,char **argv)
#endif
{
   char          uniStr[20];
   dsUint32_t    apiVersion, applVersion;

/*=== Set correct locale ===*/
#ifdef _MAC
   setlocale(LC_ALL, MAC_UTF8_LOCALE);
#endif

   globalArgv = argv;

   Set_Preferences(&pref);  /* Set default values that user can modify */

   if (ParseCommandLine(argc, argv) != 0)
     printf("\nInvalid command line option ignored!\n");

   memset(&apiLibVer,0x00,sizeof(dsmApiVersionEx));
   apiLibVer.stVersion = apiVersionExVer;
   dsmQueryApiVersionEx(&apiLibVer);

   applVersion = (10000 * DSM_API_VERSION) + (1000 * DSM_API_RELEASE) + (100 * DSM_API_LEVEL)
                 + DSM_API_SUBLEVEL;

   apiVersion = (10000 * apiLibVer.version) + (1000 * apiLibVer.release) + (100 * apiLibVer.level)
                 + apiLibVer.subLevel;

   /* check for compatibility problems */
   if (apiVersion < applVersion)
   { 
      printf("The TIBM Storage Protect API library Version = %d.%d.%d.%d is at a lower version\n",
         apiLibVer.version,
         apiLibVer.release,
         apiLibVer.level,
         apiLibVer.subLevel);
     printf("than the application version = %d.%d.%d.%d.\n",
         DSM_API_VERSION,
         DSM_API_RELEASE,
         DSM_API_LEVEL,
         DSM_API_SUBLEVEL);
     printf("Please upgrade the API accordingly.\n");
     return -1;
   }
   if (100 * apiLibVer.version + 10 * apiLibVer.release + apiLibVer.level >= 412)
   {
      if(apiLibVer.unicode)
         strcpy(uniStr,"(unicode)");
      else
         strcpy(uniStr,"         ");
   }

   printf("\n*************************************************************************\n");
   printf(  "* Welcome to the sample application for IBM Storage Protect API.       *\n");
   printf(  "* API Library Version = %d.%d.%d.%d %s                               *\n",
        apiLibVer.version,
        apiLibVer.release,
        apiLibVer.level,
        apiLibVer.subLevel,
        uniStr);
   printf(  "*************************************************************************\n");

   Process_Menu(main_menu);   /* Go process the menu and functions. */
   return 0;
}

/*----------------------------------------------------------------------+
| Name:    Process_Menu()
|
| Action:  Take the passed menu data structure and disply it and accept
|          user input.  Perform proper function calls based on user
|          input as directed from menu structure.
|
| Input:   xmenu       - Pointer to the menu to process.
|
| Returns: RC_OK            - Successful
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   This is a recursive procedure to handle nested menus.
+----------------------------------------------------------------------*/
static void Process_Menu(menu *xmenu)

{
   char menu_input[INPUT_BUFFER_SIZE];
   int i;
   int sel;
   int rc;
   char valid_entry;
   char done;
   menu_entry *menu_item;

   valid_entry = FALSE;
   done        = FALSE;

   /*---------------------------------------------------------------------*/
   /* Loop through the menu processing good entries and flagging bad.     */
   /* Exit when user indicates by invoking the proper function.           */
   /*---------------------------------------------------------------------*/
   while (!valid_entry || !done)
   {
      memset(menu_input,0x00, sizeof(menu_input)); /* clear out buff.    */
      Display_Menu(xmenu);        /* Display the menu being processed   */
      ourGetS(menu_input);        /* Get user input.                    */

      if (menu_input[0] == 0x00) 
         strcpy(menu_input,"9999"); /* No input?*/

      sel = atoi(menu_input);     /* Convert to integer for compares.   */

      /* Process each item in the menu to see if it was chosen.         */
      for(i=0;;i++)             /*  Process till we find the end or hit */
      {
         menu_item = &xmenu->mentry[i];
         if (menu_item->selection == sel)  /* Got a match?  */
         {
            valid_entry = TRUE;             /* Yep..set to true.   */
            if (menu_item->sel_func)        /* Supposed to call a function? */
            {
               rc = (menu_item->sel_func)(menu_item->sel_dialog); /* Call function*/
               if (menu_item->exit_menu) done = TRUE;              /* Exit menu    */
            }
            else
            {
               if (menu_item->next_menu)        /* Nested menu?  */
               {
                  Process_Menu(menu_item->next_menu);      /* Recursive menu call */
                  if (menu_item->exit_menu) done = TRUE;   /* Exit menu  */
               }
               else
               {
                  printf("Code error on definition of menu item %d %s\n",
                     menu_item->selection,menu_item->sel_desc);
                  printf("Sample API exiting.\n");
                  exit(99);
               }
            }
            break;  /* Found & processed entry...get out. */
         }
         if (menu_item->selection == MENU_END)
         {
            valid_entry = FALSE;
            break;
         }
      }
      if (!valid_entry)
      {
         /* check for "no input", then print nothing */
         if (sel == 9999)
         {
            printf("\n");
         }
         else
         {
            printf("Invalid selection.  Try again.\n",menu_input);
         }
      }
   }
}
/*----------------------------------------------------------------------+
| Name:    display_menu()
|
| Action:  Take the passed menu data structure and disply it to the
|          screen.  Once displayed return to the caller.
|
| Input:   xmenu       - Pointer to the menu to process.
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static void Display_Menu(menu *xmenu)

{
   int i;
   menu_entry *menu_item;

   printf("%s\n",xmenu->instructions);

   for(i=0;;i++)
   {
      menu_item = &xmenu->mentry[i];
      if (menu_item->selection == MENU_END) 
         break;
      printf("%2d.  %s\n",menu_item->selection,
             menu_item->sel_desc);
   }

   printf("\nEnter selection ==>"); fflush(stdout);

}
/*----------------------------------------------------------------------+
| Name:    Set_Preferences(preference *pref)
|
| Action:  Set the preference structure passed to default values.
|
| Input:   pref        - Pointer to preference data struct.
|
| Returns: none
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static void Set_Preferences(preferences *pref)
{
   pref->sz_send_buff = 1024 * 1024;     /* Send buffer size */
   pref->sz_recv_buff = 1024 * 1024;     /* Receive buffer size */
   pref->verbose      = bFalse;    /* Show verbose output? */
   pref->qry_details  = bFalse;    /* Show details on queries? */
   pref->max_restore_items = 3000; /* Number of items in a restore loop */
   pref->use_est      = bTrue;     /* Use real or fake size estimate? */
   pref->echo_in      = bFalse;    /* Echo input lines? */
}
/*----------------------------------------------------------------------+
| Name:    ParseCommandLine()
|
| Action:  Parse the command line and return completion code.
|
| Input:   Command line arguments (argc, argv).
|
| Returns: 0 - OK
|          1 - invalid parameters found
|
| Side
| Effects: Global variable pref is being set.
|
| Notes:   None
+----------------------------------------------------------------------*/
static  int  ParseCommandLine(int argc, char *argv[])
{
   int i, rc;

   rc = 0;
   for (i = 1; i < argc; i++) 
   {
      /* keyword parameters */
      if ((argv[i][0] == '/') || (argv[i][0] == '-')) 
      {
         if (ValidateParam(argv[i]+1, "V", 1, 1)) 
         {
            pref.echo_in = bTrue;
            continue;
         }
      }

      /* wrong parameter */
      rc=1;
   }
   return rc;
}
/*----------------------------------------------------------------------+
| Name:    ValidateParam()
|
| Action:  See if parameter is valid abbreviation and format.
|
| Input:   instr   - parameter
|          compstr - to be compared with
|          min     - minimum chracters of abbreviation
|          max     - maximum chracters of token
|
| Returns: 0 - invalid parameter
|          1 - valid parameter
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static int  ValidateParam(char *instr, char *compstr, int min, int max)
{
   unsigned int l, i;
   char param[1024];

   /* make sure the param is not longer than max allows if max>0 */
   if ((max > 0) && (strlen(instr) > max))
      return 0;

   /* make sure the param is not too long to handle */
   if (strlen(instr) >= sizeof param)
      return 0;

   strcpy(param, instr);

   /* assign string len of instr */
   l = strlen(param);

   /* convert to upper case */
   for (i = 0; param[i] != 0; i++)
      param[i] = (char) toupper(param[i]);

   /* verify parameter is part of the keyword */
   if (strncmp(param, compstr, min) != 0)
      return 0;

   /* verify parameter is long enough */
   if (l >= (unsigned int) min)
      return 1;
   else
      return 0;
}
