/***********************************************************************
* Name:
*        dsmapipw.c
*
* Function:
*        Sample program to change passwords for IBM Storage Protect 
*        clients using the IBM Storage Protect API.
*
* This source code is the sample of how to change the password using 
* IBM Storage Protect API procedure calls.
* IBM Storage Protect provides this source code "AS IS" without 
* warranty of any kind.
*
* Status:
*        Version 1, Release 1
***********************************************************************/
#include <sys/ioctl.h>
#include <sys/types.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <termio.h>
#include <unistd.h>

#include "dsmapitd.h"      /* IBM Storage Protect API type definitions                */
#include "dsmapifp.h"      /* IBM Storage Protect API function prototypes.            */
#include "dsmrc.h"         /* IBM Storage Protect API return codes.                   */


/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static int16 getPwInput();
static int ReadPass();
static void Catch();
static int install_sig_handler();
static void rcApiOut();
#else
static int16 getPwInput(char *node,
                        char *current_pw,
                        char *new_pw1,
                        char *new_pw2);
static int ReadPass(char *buffer, int length);
static void Catch( int signo );
static int install_sig_handler(int signum, void(*sig_handler)(int) ) ;
static void rcApiOut(int16 rc) ;
#endif
/*----------------------------------------------------------------------+
| Globals
.----------------------------------------------------------------------*/
dsmApiVersion    apiVer;
uint32           dsmHandle;
ApiSessInfo      dsmSessInfo;
static int intrupt;

/*----------------------------------------------------------------------+
| Name:    main()
|
| Action:  Main entry point.
|
| Input:   None
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
main(argc, argv)
   int  argc;
   char **argv;
#else
main(int argc,char **argv)
#endif

{
int           rc;

char node[DSM_MAX_ID_LENGTH + 1];
char current_pw[DSM_MAX_VERIFIER_LENGTH + 1];
char new_pw1   [DSM_MAX_VERIFIER_LENGTH + 1];
char new_pw2   [DSM_MAX_VERIFIER_LENGTH + 1];

   printf("\n********************************************************************************\n");
   printf(  "* IBM Storage Protect                                                         *\n");
   memset(&apiVer,0x00,sizeof(dsmApiVersion));
   dsmQueryApiVersion(&apiVer);
   printf("* API Library Version = %d.%d.%d.%d %s                   *\n",
	   apiLibVer.version,
	   apiLibVer.release,
	   apiLibVer.level,
	   apiLibVer.subLevel,
	   uniStr);
   printf(  "********************************************************************************\n");

   rc = getPwInput(node,current_pw,new_pw1,new_pw2);
   if (rc) exit(1);  /* No since going any further........*/

   rc = dsmInit(&dsmHandle,    /* Will contain session handle on retrun. */
                &apiVer,       /* Version of the API we are using.       */
                node,          /* Node name of client.                   */
                NULL,          /* Use null owner name for this call.     */
                current_pw,    /* Node's password.                       */
                "AIX",         /* Name of our node type.                 */
                NULL,NULL);    /* Use default configs and options.       */

   if (rc)
   {
      printf("*** Server signon failed: ");
      rcApiOut(rc);
      exit(1);
   }

   rc = dsmChangePW(dsmHandle,current_pw,new_pw1);
   if (rc)
   {
      printf("*** Password change failed: ");
      rcApiOut(rc);
   }
   else
   {
      printf("\nYour new password has been accepted and updated.\n");
   }

   rc = dsmTerminate(dsmHandle);
   if (rc)
   {
      printf("*** Session logoff failed: ");
      rcApiOut(rc);
   }
}

/*----------------------------------------------------------------------+
| Name:    getPwInput()
|
| Action:  This function is invoked from the sample program main menu.
|          Prompt the end user for current password and new passwords,
|          and if those match send the new password to the server.
|
| Input:   sel_dialog  - Pointer to dialog table to process or NULL if
|                        no dialog.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static int16 getPwInput(node, current_pw, new_pw1, new_pw2)
   char *node;
   char *current_pw;
   char *new_pw1;
   char *new_pw2;
#else
static int16 getPwInput(char *node,
                        char *current_pw,
                        char *new_pw1,
                        char *new_pw2)
#endif
{

#define Input_Len  1025
   bool_t   done = bFalse;
   bool_t   pw_match = bFalse;
   uint16   pw_trys;
   uint16   rc;
   uint16   len;
   char input[Input_Len];

   pw_trys = 0;
   while (!done)
   {
      /*--------------------------------------------------------------
      | Node name input.  Null input is fine to use node name from
      | options file.
      .----------------------------------------------------------------*/
      printf("Enter your node name (or just ENTER to use default): ");
      
	  fgets(input, input_Len-1, stdin);
	
	  //removes the '\n' character that is not part of the input
	  int i;
	  for(i=0 ; i<input_Len-1 ; i++)
	  {
	     if(input[i] == '\n')
	     {
		   input[i]='\0';
		   break;
	     }
	  }

      len = strlen(input);
      if (len > DSM_MAX_ID_LENGTH)
      {
        printf("Node name too long.  Please try again.\n");
        continue;
      }
      else
      {
         strcpy(node,input);
      }

      /*--------------------------------------------------------------
      | Current password input.
      .----------------------------------------------------------------*/
      printf("Enter your current password:");
      rc = ReadPass(input,Input_Len);
      printf("\n");
      len = strlen(input);
      if ((len > DSM_MAX_VERIFIER_LENGTH) || !len || (rc < 0))
      {
        printf("Current password is invalid.  Please try again.\n");
        continue;
      }
      else
      {
         strcpy(current_pw,input);
      }

      /*--------------------------------------------------------------
      | New pw input, first copy
      .----------------------------------------------------------------*/
      printf("Enter your new password:");
      rc = ReadPass(input,Input_Len);
      printf("\n");
      len = strlen(input);
      if ((len > DSM_MAX_VERIFIER_LENGTH) || !len || (rc < 0))
      {
        printf("New password is invalid.  Please try again.\n");
        continue;
      }
      else
      {
         strcpy(new_pw1,input);
      }

      /*--------------------------------------------------------------
      | New pw input, second copy
      .----------------------------------------------------------------*/
      printf("Enter your new password again:");
      rc = ReadPass(input,Input_Len);
      printf("\n");
      len = strlen(input);
      if ((len > DSM_MAX_VERIFIER_LENGTH) || !len || (rc < 0))
      {
        printf("New password is invalid.  Please try again.\n");
        continue;
      }
      else
      {
         strcpy(new_pw2,input);
      }

      /*--------------------------------------------------------------
      | Compare new pw copies to make sure no typos from user.
      .----------------------------------------------------------------*/
      if ((strcmp(new_pw1,new_pw2)))
      {
         pw_trys++;
         if (pw_trys > 3)
         {
            printf("Password entry failed.  Program exiting.\n");
            done = bTrue;
         }
         else
         {
            printf("\nYour new passwords do not match.  Try again.\n");
         }
      }
      else
      {
         done = bTrue;
         pw_match = bTrue;
      }
   }

   if (!pw_match) return 1;   /* Get out now since pw's did not match */

return 0;
}



/*----------------------------------------------------------------------+
| int  ReadPass(char *buffer, int length);                              |
|                                                                       |
|      Function: Perform a system dependent terminal read (without      |
|                echo).                                                 |
|                                                                       |
|      Process:  Requires a pointer to a buffer of sufficient size to   |
|                hold the maximum length password and an integer        |
|                containing the length of the buffer to fill.           |
|                                                                       |
|      Returns: >0 - on successful completion, the length of password   |
|               -1 - for an error condition                             |
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static int
ReadPass(buffer, length)
   char *buffer;
   int  length;
#else
static int
ReadPass(char *buffer, int length)
#endif
{

  struct termio  ttyState, ttyStateSave;
  register char  *p;
  register int   c;
  int            rc;
  FILE           *ttyFH;

  struct sigaction  action;

  /* Let's flush any prompt to the terminal that may be pending. */
  fflush(stdout);

  /* Open the console input device */
  if((ttyFH = fopen("/dev/tty", "r")) == NULL)
    return(-1);
  else
    setbuf(ttyFH, (char *)NULL);

  /* Reset the interrupt flag */
  intrupt = 0;

  /* Trap the "BREAK" interrupt */
  (void) sigaction( SIGINT, NULL, &action );    /* save current hdlr */
  (void) install_sig_handler( SIGINT, Catch );  /* install new hdlr  */

  /* Get current state */
  rc = ioctl(fileno(ttyFH), TCGETA, &ttyStateSave);
  if (rc == -1)
    return(-1);

  /* Copy the saved tty state into a work field */
  ttyState = ttyStateSave;

  /* Turn off ECHO */
  ttyState.c_lflag &= ~ECHO;
  rc = ioctl(fileno(ttyFH), TCSETA, &ttyState);
  if (rc == -1)
    return(-1);

  /* Read the password (quietly) */
  for(p=buffer; !intrupt && (c = getc(ttyFH)) != '\n' && c != EOF; ) {
    if(p < buffer+length)
      *p++ = c;
  }
  *p = '\0';

  /* Restore the tty state settings */
  rc = ioctl(fileno(ttyFH), TCSETA, &ttyStateSave);
  if (rc == -1)
    return(-1);

  /* Reset the interrupt handler */
  (void) sigaction( SIGINT, &action, NULL );

  if(ttyFH != stdin)
    (void) fclose(ttyFH);
  if(intrupt)
    (void) kill(getpid(), SIGINT);

  /* Upper case the password. */
  /*StrUpper(buffer);*/

  return(strlen(buffer));


}  /* ReadPass */

/*----------------------------------------------------------------------+
| int  rcApiOut(int16 rc)                                               |
|                                                                       |
|      Function: Translate return codes to messages.                    |
|                echo).                                                 |
|                                                                       |
|                                                                       |
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static void
rcApiOut(rc)
   int16 rc;
#else
static void
rcApiOut(int16 rc)
#endif
{

printf("(%i) ",rc);
switch (rc)
{
   case DSM_RS_ABORT_SYSTEM_ERROR :
        printf("Abort: system error.\n"); break;
   case DSM_RS_ABORT_NO_MATCH :
        printf("Abort: no match.\n"); break;
   case DSM_RS_ABORT_BY_CLIENT :
        printf("Abort: client abort.\n"); break;
   case DSM_RS_ABORT_ACTIVE_NOT_FOUND :
        printf("Abort: active not found.\n"); break;
   case DSM_RS_ABORT_NO_DATA :
        printf("Abort: No data.\n"); break;
   case DSM_RS_ABORT_BAD_VERIFIER :
        printf("Abort: Bad password.\n"); break;
   case DSM_RS_ABORT_NODE_IN_USE :
        printf("Abort: Node in use.\n"); break;
   case DSM_RS_ABORT_EXPDATE_TOO_LOW :
        printf("Abort: Expiration date too low.\n"); break;
   case DSM_RS_ABORT_DATA_OFFLINE :
        printf("Abort: Data offline.\n"); break;
   case DSM_RS_ABORT_EXCLUDED_BY_SIZE :
        printf("Abort: Excluded by size.\n"); break;
   case DSM_RS_ABORT_NO_REPOSIT_SPACE :
        printf("Abort: No repository space.\n");break;
   case DSM_RS_ABORT_MOUNT_NOT_POSSIBLE :
        printf("Abort: Mount not possible.\n"); break;
   case DSM_RS_ABORT_SIZESTIMATE_EXCEED :
        printf("Abort : size estimate exceeded.\n"); break;
   case DSM_RS_ABORT_DATA_UNAVAILABLE :
        printf("Abort: Data unavailable.\n"); break;
   case DSM_RS_ABORT_RETRY :
        printf("Abort: Retry.\n"); break;
   case DSM_RS_ABORT_NO_LOG_SPACE :
        printf("Abort: No log space.\n"); break;
   case DSM_RS_ABORT_NO_DB_SPACE :
        printf("Abort: No DB Space.\n"); break;
   case DSM_RS_ABORT_FS_NOT_DEFINED :
        printf("Abort: Node not defined.\n"); break;
   case DSM_RS_ABORT_NODE_ALREADY_DEFED :
        printf("Abort: Node already defined.\n"); break;
   case DSM_RS_ABORT_NO_DEFAULT_DOMAIN :
        printf("Abort: No default domain.\n"); break;
   case DSM_RS_ABORT_INVALID_NODENAME :
        printf("Abort: Invalid nodename.\n"); break;
   case DSM_RS_ABORT_INVALID_POL_BIND :
        printf("Abort: Invalid policy binding.\n"); break;
   case DSM_RS_ABORT_DEST_NOT_DEFINED :
        printf("Abort: Storage destination not defined.\n"); break;
   case DSM_RS_ABORT_WAIT_FOR_SPACE :
        printf("Abort: Wait for space.\n"); break;
   case DSM_RS_ABORT_NOT_AUTHORIZED :
        printf("Abort: User not authorized.\n"); break;
   case DSM_RS_ABORT_RULE_ALREADY_DEFED :
        printf("Abort: Access rule already defined.\n"); break;

   case DSM_RC_REJECT_NO_RESOURCES :
        printf("No resources for signon.\n"); break;
   case DSM_RC_REJECT_VERIFIER_EXPIRED :
        printf("Password has expired.\n"); break;
   case DSM_RC_REJECT_ID_UNKNOWN :
        printf("Node name is unknown.\n"); break;
   case DSM_RC_REJECT_DUPLICATE_ID :
   case DSM_RC_REJECT_SERVER_DISABLED :
        printf("Server is in disabled state.\n"); break;
   case DSM_RC_REJECT_CLOSED_REGISTER :
        printf("Server registration set to closed.\n"); break;
   case DSM_RC_REJECT_CLIENT_DOWNLEVEL :
        printf("Client code is downlevel from supported server level.\n");
        break;
   case DSM_RC_REJECT_SERVER_DOWNLEVEL :
        printf("Server code is downlevel from client level.\n"); break;
   case DSM_RC_REJECT_ID_IN_USE :
        printf("Node id is already in use.\n"); break;
   case DSM_RC_REJECT_ID_LOCKED :
        printf("Node id is locked from access at the server.\n"); break;
   case DSM_RC_SIGNONREJECT_LICENSE_MAX :
        printf("Max user license reached at server.\n"); break;
   case DSM_RC_USER_ABORT :
        printf("Processing aborted by user.\n"); break;
   case DSM_RC_NO_MEMORY :
        printf("No memory left to complete request.\n"); break;
   case DSM_RC_TA_COMM_DOWN :
        printf("Communications link down\n"); break;
   case DSM_RC_FILE_NOT_FOUND :
        printf("Specified file not found.\n"); break;
   case DSM_RC_PATH_NOT_FOUND :
        printf("Specified path doesn't exist.\n"); break;
   case DSM_RC_ACCESS_DENIED :
        printf("Access denied due to improper permission.\n"); break;
   case DSM_RC_NO_HANDLES :
        printf("No more file handles available.\n"); break;
   case DSM_RC_FILE_EXISTS :
        printf("File already exists.\n"); break;
   case DSM_RC_INVALID_PARM :
        printf("Invalid parameter passed. CRITICAL.\n"); break;
   case DSM_RC_INVALID_HANDLE :
        printf("Invalid file handle passed.\n"); break;
   case DSM_RC_DISK_FULL :
        printf("Out of disk space.\n"); break;
   case DSM_RC_PROTOCOL_VIOLATION :
        printf("Internal protocol violation. CRITICAL.\n"); break;
   case DSM_RC_UNKNOWN_ERROR :
        printf("Unknown system error. CRITICAL.\n"); break;
   case DSM_RC_UNEXPECTED_ERROR :
        printf("Unexpected error. CRITICAL.\n");
   case DSM_RC_FILE_BEING_EXECUTED :
        printf("No write is allowed.\n"); break;
   case DSM_RC_DIR_NO_SPACE :
        printf("Directory can't be expanded.\n"); break;
   case DSM_RC_LOOPED_SYM_LINK :
        printf("Too many symbolic links were encountered.\n"); break;
   case DSM_RC_FILE_NAME_TOO_LONG :
        printf("File name too long.\n"); break;
   case DSM_RC_FILE_SPACE_LOCKED :
        printf("Filespace is locked by the system.\n"); break;
   case DSM_RC_FINISHED :
        printf("Finished processing.\n"); break;
   case DSM_RC_UNKNOWN_FORMAT :
        printf("Unknown format.\n"); break;
   case DSM_RC_NO_AUTHORIZATION :
        printf("Client has no auth to read data.\n"); break;
   case DSM_RC_FILE_SPACE_NOT_FOUND :
        printf("Specified file space not found.\n"); break;
   case DSM_RC_TXN_ABORTED :
        printf("Transaction aborted.\n"); break;
   case DSM_RC_SUBDIR_AS_FILE :
        printf("Subdirectory name exists as file.\n"); break;
   case DSM_RC_PROCESS_NO_SPACE :
        printf("Process has no more disk space.\n"); break;
   case DSM_RC_PATH_TOO_LONG :
        printf("A directory path being built became too long.\n"); break;
   case DSM_RC_NOT_COMPRESSED :
        printf("File thought to be compressed is actually not.\n"); break;
   case DSM_RC_TOO_MANY_BITS :
        printf("File was compressed using more bits than can be handled.\n");
        break;
   case DSM_RC_SYSTEM_ERROR :
        printf("Internal system error.\n"); break;
   case DSM_RC_NO_SERVER_RESOURCES :
        printf("Server out of resources.\n"); break;
   case DSM_RC_FS_NOT_KNOWN :
        printf("The file space is not known by the server.\n"); break;
   case DSM_RC_NO_LEADING_DIRSEP :
        printf("No leading directory separator.\n"); break;
   case DSM_RC_WILDCARD_DIR :
        printf("Wildcard character in directory path when not allowed.\n");
        break;
   case DSM_RC_COMM_PROTOCOL_ERROR :
        printf("Communications protocol error.\n"); break;
   case DSM_RC_AUTH_FAILURE :
        printf("Authentication failure!\n"); break;
   case DSM_RC_TA_NOT_VALID :
        printf("TA component not a root and/or SUID program.\n"); break;
   case DSM_RC_KILLED :
        printf("Process killed.\n");
   case DSM_RC_WOULD_BLOCK :
        printf("Op. would cause the system to block waiting for input\n");
   case DSM_RC_TOO_SMALL :
        printf("Area for compiled pattern small.\n"); break;
   case DSM_RC_UNCLOSED :
        printf("No closing bracket in pattern.\n"); break;
   case DSM_RC_NO_STARTING_DELIMITER :
        printf("Pattern has to start with directory delimiter.\n"); break;
   case DSM_RC_NEEDED_DIR_DELIMITER :
        printf("A directory delimiter is needed around the ... string.\n");
        break;
   case DSM_RC_UNKNOWN_FILE_DATA_TYPE :
        printf("Structured file data type is unknown.\n"); break;
   case DSM_RC_BUFFER_OVERFLOW :
        printf("Data buffer overflow.\n"); break;
   case DSM_RC_NO_COMPRESS_MEMORY :
        printf("Compress/Expand out of memory.\n"); break;
   case DSM_RC_COMPRESS_GREW :
        printf("Compressed object grew.\n"); break;
   case DSM_RC_INV_COMM_METHOD :
        printf("Invalid comm method specified.\n"); break;
   case DSM_RC_WILL_ABORT :
        printf("Transaction will be aborted.\n"); break;
   case DSM_RC_FS_WRITE_LOCKED :
        printf("File space is write locked.\n"); break;
   case DSM_RC_SKIPPED_BY_USER :
        printf("User wanted file skipped for case of ABORT_DATA_OFFLINE.\n");
        break;
   case DSM_RC_TA_NOT_FOUND :
        printf("TA not found in it's directory.\n"); break;
   case DSM_RC_TA_ACCESS_DENIED :
        printf("Access to TA is denied.\n"); break;
   case DSM_RC_FS_NOT_READY :
        printf("File space not ready.\n"); break;
   case DSM_RC_FS_IS_BAD :
        printf("File space is bad.\n"); break;
   case DSM_RC_FIO_ERROR :
        printf("File input/output error.\n"); break;
   case DSM_RC_WRITE_FAILURE :
        printf("Error writing to file.\n"); break;
   case DSM_RC_OVER_FILE_SIZE_LIMIT :
        printf("File over system/user limit.\n"); break;
   case DSM_RC_CANNOT_MAKE :
        printf("Could not create file/directory, could be bad name.\n"); break;
   case DSM_RC_NO_PASS_FILE :
        printf("Password file needed and user is not root.\n"); break;
   case DSM_RC_VERFILE_OLD :
        printf("Password stored locally doesn't match the one at server.\n");
        break;
   case DSM_RC_INPUT_ERROR :
        printf("Input error.\n"); break;
   case DSM_RC_REJECT_PLATFORM_MISMATCH :
        printf("Platform name doesn't match with registered platform.\n");
        break;
   case DSM_RC_TL_NOT_FILE_OWNER :
        printf("User trying to backup a file & is not the file's owner.\n");
        break;
   case DSM_RC_UNMATCHED_QUOTE :
        printf("Missing starting or ending quote.\n"); break;

/*---------------------------------------------------------------------------*/
/* Return codes 180-200 are reserved for Policy Set handling in common code  */
/*---------------------------------------------------------------------------*/
   case DSM_RC_PS_MULTBCG :
        printf("Multiple backup copy groups in 1 MC.\n"); break;
   case DSM_RC_PS_MULTACG :
        printf("Multiple arch. copy groups in 1 MC.\n"); break;
   case DSM_RC_PS_NODFLTMC :
        printf("Default MC name not in policy set.\n"); break;
   case DSM_RC_TL_NOBCG :
        printf("Backup req, no backup copy group.\n"); break;
   case DSM_RC_TL_EXCLUDED :
        printf("Backup req, excl. by in/ex filter.\n"); break;
   case DSM_RC_TL_NOACG :
        printf("Archive req, no archive copy group.\n"); break;
   case DSM_RC_PS_INVALID_ARCHMC :
        printf("Invalid MC name in archive override.\n"); break;
   case DSM_RC_NO_PS_DATA :
        printf("No policy set data on the server.\n"); break;
   case DSM_RC_PS_INVALID_DIRMC :
        printf("Invalid directory MC specified in the options file.\n"); break;
   case DSM_RC_PS_NO_CG_IN_DIR_MC :
        printf("No backup copy group in directory MC. \n"); break;
   case DSM_RC_DUP_LABEL :
        printf("Duplicate label on system.\n"); break;
   case DSM_RC_NO_LABEL :
        printf("Filespace has no label.\n"); break;
   case DSM_RC_LOG_CANT_BE_OPENED :
        printf("Error trying to open error log.\n"); break;
   case DSM_RC_LOG_ERROR_WRITING_TO_LOG :
        printf("Error occurred writing to the log file.\n"); break;
   case DSM_RC_LOG_NOT_SPECIFIED :
        printf("No error log file was specified.\n"); break;


/* TCP/IP error codes */
   case DSM_RC_TCPIP_FAILURE :
        printf("TCP/IP communications failure.\n"); break;
   case DSM_RC_CONN_TIMEDOUT :
        printf("TCP/IP connection attempt timed out.\n"); break;
   case DSM_RC_CONN_REFUSED :
        printf("TCP/IP connection refused by host.\n"); break;
   case DSM_RC_BAD_HOST_NAME :
        printf("TCP/IP invalid host name specified.\n"); break;
   case DSM_RC_NETWORK_UNREACHABLE :
        printf("TCP/IP host name unreachable.\n"); break;
   case DSM_RC_WINSOCK_MISSING :
        printf("TCP/IP WINSOCK.DLL missing.\n"); break;
   case DSM_RC_TCPIP_DLL_LOADFAILURE :
        printf("Error from LoadLibrary.\n"); break;
   case DSM_RC_TCPIP_LOADFAILURE :
        printf("Error from GetProcAddress.\n"); break;

   case DSM_RC_NULL_OBJNAME :
        printf("Null object name (filespace name missing?)\n"); break;
   case DSM_RC_NULL_DATABLKPTR :
        printf("Null DataBlk pointer.\n"); break;
   case DSM_RC_NULL_OBJATTRPTR :
        printf("Object Attr Pointer is NULL.\n"); break;
   case DSM_RC_NO_SESS_BLK :
        printf("No server session info.\n"); break;
   case DSM_RC_NO_POLICY_BLK :
        printf("No policy hdr info.\n"); break;
   case DSM_RC_INVALID_OBJTYPE :
        printf("Invalid object type.\n"); break;
   case DSM_RC_INVALID_VOTE :
        printf("Invalid vote.\n"); break;
   case DSM_RC_INVALID_OPT :
        printf("Error in options string.\n"); break;
   case DSM_RC_INVALID_DS_HANDLE :
        printf("Invalid IBM Storage Protect API handle.\n");
   case DSM_RC_INVALID_REPOS :
        printf("Invalid value for repository.\n"); break;
   case DSM_RC_INVALID_OBJNAME :
        printf("Invalid full path name.\n"); break;
   case DSM_RC_INVALID_OBJOWNER :
        printf("Invalid object owner name.\n"); break;
   case DSM_RC_INVALID_ACTYPE :
        printf("Invalid action type.\n"); break;
   case DSM_RC_INVALID_SENDTYPE :
        printf("Invalid send type.\n"); break;
   case DSM_RC_INVALID_PARAMETER :
        printf("Invalid parameter.\n");
   case DSM_RC_INVALID_OBJSTATE :
        printf("Invalid objstate: Acitve, inactive, or any match?\n"); break;
   case DSM_RC_INVALID_MCNAME :
        printf("Mgmt class name not found.\n"); break;
   case DSM_RC_NEWPW_REQD :
        printf("New password is required.\n"); break;
   case DSM_RC_OLDPW_REQD :
        printf("Old password is required.\n"); break;
   case DSM_RC_NO_OWNER_REQD :
        printf("Owner not allowed. Allow default.\n"); break;
   case DSM_RC_NO_NODE_REQD :
        printf("Node not allowed with pw=generate.\n"); break;
   case DSM_RC_BAD_CALL_SEQUENCE :
         printf("Sequence of DSM calls not allowed.\n"); break;
   case DSM_RC_WILDCHAR_NOTALLOWED :
        printf("Wild card not allowed for hl,ll.\n"); break;
   case DSM_RC_FSNAME_NOTFOUND :
        printf("Filespace name not found.\n"); break;
   case DSM_RC_FS_NOT_REGISTERED :
        printf("Filespace name not registered.\n"); break;
   case DSM_RC_OBJID_NOTFOUND :
        printf("No object id to restore.\n"); break;
   case DSM_RC_WRONG_VERSION :
        printf("Wrong level of code.\n");
   case DSM_RC_NEEDTO_ENDTXN :
        printf("Need to call dsEndTxn.\n"); break;
   case DSM_RC_OBJ_EXCLUDED :
        printf("Object is excluded by MC.\n"); break;
   case DSM_RC_OBJ_NOBCG :
        printf("Object has no backup copy group.\n"); break;
   case DSM_RC_OBJ_NOACG :
        printf("Object has no archive copy group.\n"); break;
   case DSM_RC_APISYSTEM_ERROR :
        printf("API internal error.\n"); break;
   case DSM_RC_DESC_TOOLONG :
        printf("Description is too long.\n"); break;
   case DSM_RC_OBJINFO_TOOLONG :
        printf("Object attr objinfo too long.\n"); break;
   case DSM_RC_HL_TOOLONG :
        printf("High level qualifier is too long.\n"); break;
   case DSM_RC_PASSWD_TOOLONG :
        printf("Password is too long.\n"); break;
   case DSM_RC_FILESPACE_TOOLONG :
        printf("Filespace name is too long.\n"); break;
   case DSM_RC_LL_TOOLONG :
        printf("Low level qualifier is too long.\n"); break;
   case DSM_RC_FSINFO_TOOLONG :
        printf("Filespace length is too big.\n"); break;
   case DSM_RC_MORE_DATA :
        printf("There is more data to restore.\n"); break;
   case DSM_RC_BUFF_TOO_SMALL :
        printf("Buffer is too small.\n"); break;
   case DSM_RC_NO_OPT_FILE :
        printf("No default user configuration file.\n"); break;
   case DSM_RC_INVALID_KEYWORD :
        printf("Invalid option keyword.\n"); break;
   case DSM_RC_PATTERN_TOO_COMPLEX :
        printf("Can't match Include/Exclude entry.\n"); break;
   case DSM_RC_NO_CLOSING_BRACKET :
        printf("Missing closing bracket inc/excl.\n"); break;
   case DSM_RC_INVALID_SERVER :
        printf("Invalid server name from client.\n"); break;
   case DSM_RC_NO_HOST_ADDR :
        printf("Not enough info to connect server.\n"); break;
   case DSM_RC_MACHINE_SAME :
        printf("-NODENAME same as real name.\n"); break;
   case DSM_RC_NEED_ROOT :
        printf("API caller must be root.\n"); break;
   case DSM_RC_NEEDTO_CALL_BINDMC :
        printf("dsmBindMC must be called first.\n"); break;
   case DSM_RC_CHECK_REASON_CODE :
        printf("Txn failed, check reason code from dsmEndTxn.\n"); break;


   default :
        printf("Unknown rc.\n");
}
return;
}

/*----------------------------------------------------------------------+
| int install_sig_handler(int signum, void(*sig_handler)(int) )         |
|                                                                       |
| Function: Uses sigaction to establish signal handler.                 |
|                                                                       |
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static int
install_sig_handler(signum, sig_handler)
   int  signum;
   void (*sig_handler)();
#else
static int
install_sig_handler(
                     int signum,
                     void (*sig_handler)(int) )
#endif
{

  struct sigaction   action;
  int                rc;

  action.sa_handler = sig_handler;      /* signal handler function  */
  sigemptyset( &action.sa_mask );       /* mask of signals to block */

  action.sa_flags = SA_NOCLDSTOP;

  rc = sigaction(
                   signum,  /*  I */ /* signal identifier            */
                  &action,  /* *I */ /* new action for signal        */
                   NULL );  /* *O */ /* previous action - not needed */

  return (rc);

}  /* install_sig_handler() */

/*----------------------------------------------------------------------+
| void Catch( int signo )                                               |
|                                                                       |
| Function: Signal handler.                                             |
|                                                                       |
+----------------------------------------------------------------------*/
#ifdef _NO_PROTO_
static void Catch( signo )
   int signo;
#else
static void Catch( int signo )
#endif
{
  signo;        /* so compiler won't complain about unused arguments */


  intrupt = 1;
}

