/*---------------------------------------------------------------------+
| Module-name:                                                         |
|        dsmgrp.c (Password prompt version)                            |
|                                                                      |
| Function:                                                            |
|        Invokes the series of IBM Storage Protect APIs necessary   |
|        to create logical group and perform the following             |
|        operations on the groups in this order:                       |
|                                                                      |
|        o Send a Group Leader Object                                  |
|        o Add 4 Group Members to the Group                            |
|        o Perform a Normal Query, which should return 5               |
|          different objects. The Leader object and 4 member objects   |
|        o Perform a Query Group Members, which should only return     |
|          the 4 different members                                     |
|        o Query Group Leaders. Should return 0 objects because no     |
|          group has been closed yet                                   |
|        o Query Open Group Leaders. Should return the 1 group leader  |
|        o Close the group created                                     |
|        o Query Group Leaders Again. Shold return 1 object because    |
|          the group has been closed.                                  |
|        o Query Open Group Leaders Again. This time it should return  |
|          nothing, since there are no open groups anymore             |
|        o Remove 2 members from the Group                             |
|        o Query Group Members Again. This time it should return       |
|          only 2 objects, since 2 have been removed                   |
|        o Perform an Assign of 2 Members to the group                 |
|        o Query Group Members One final time. This time it should     |
|          return 4 objects again                                      |
|                                                                      |
| Setup:                                                               |
|        You should first setup the environment variables to point to  |
|        the correct directories and dsm.opt file.  To do this, make   |
|        sure that DSMI_CONFIG points to your dsm.opt file, and        |
|        DSMI_DIR and DSMI_LOG point to your message text repository   |
|        system file dsm.sys  and error log.                           |
|                                                                      |
|           set DSMI_CONFIG=\yourDirectoryPath\dsm.opt                 |
|           set DSMI_DIR=\InstallationPath                             |
|           set DSMI_LOG=\AnyDesiredDirectory                          |
|                                                                      |
|        The next thing that should be done is the creation of the     |
|        dsm.opt file.  With the exception of the server address --    |
|        which will be your server address, the line in your dsm.opt   |
|        should be...                                                  |
|                                                                      |
|           COMMMETHOD       TCPip                                     |
|           TCPSERVERADDRESS host.company.com                          |
|                                                                      |
|        Next, in the server specified in your dsm.opt file, you will  |
|        have to register 1 node -- unix1.                             |
|        To do this, use 'dsmadmc' to logon to                         |
|        your IBM Storage Protect server, and type 'reg node unix1  |
|        unix1'.                                                       |
|        All that is left is to compile the source code by             |
|        using the supplied make file, and run the executable.         |
|                                                                      |
|           For AIX:  make -f makegrp.aix                              |
|           For Sun:  make -f makegrp.sol                              |
|           for HP:   make -f makegrp.hp                               |
|                                                                      |
+---------------------------------------------------------------------*/

#include <pthread.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "dsmapitd.h"
#include "dsmapifp.h"
#include "dsmrc.h"

#define NODE1 "unix1"


#define MTAPI     316

void *Thread1(void *string);

int NumberOfThreads;

int main(int argc, char *argv[])
{

   pthread_t       thrd1, thrd2, thrd3, thrd4;
   pthread_attr_t  attr;

   dsInt16_t          rc= 0, apiver=0;
   dsUint32_t         apiVersion, applVersion;
   dsmApiVersionEx    apiLibVer;
   envSetUp           dsmEnvSetUp;
   char               *rcMsg;
   char               uniStr[20];
   int                detached;

   /* set attr to detached state so resources will be released
      when thread exists */
   pthread_attr_init(&attr);

#if defined(_HPUX)
   pthread_attr_setstacksize (&attr, 1024 * 1024);
#endif


#if defined(_OPEN_THREADS)
   detached = __DETACHED;
   pthread_attr_setdetachstate(&attr, &detached);
#else
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
#endif

   NumberOfThreads = 1;

   memset(&apiLibVer, 0x00, sizeof(dsmApiVersionEx));
   apiLibVer.stVersion = apiVersionExVer;
   dsmQueryApiVersionEx(&apiLibVer);

   applVersion = (10000 * DSM_API_VERSION) + (1000 * DSM_API_RELEASE) + (100 * DSM_API_LEVEL)
                 + DSM_API_SUBLEVEL;

   apiVersion = (10000 * apiLibVer.version) + (1000 * apiLibVer.release) + (100 * apiLibVer.level)
                 + apiLibVer.subLevel;

   /* check for compatibility problems */
   if (apiVersion < applVersion)
   { printf("The IBM Storage Protect API library Version = %d.%d.%d.%d is at a lower version\n",
        apiLibVer.version,
        apiLibVer.release,
        apiLibVer.level,
        apiLibVer.subLevel);
     printf("than the application version = %d.%d.%d.%d.\n",
        DSM_API_VERSION,
        DSM_API_RELEASE,
        DSM_API_LEVEL,
        DSM_API_SUBLEVEL);
     printf("Please upgrade the API accordingly.\n");
     return -1;
   }

   apiver = (apiLibVer.version * 100) + (apiLibVer.release * 10) + apiLibVer.level;
   if (apiver < MTAPI)
   {
      printf("\n***********************************************************\n");
      printf(  "The IBM Storage Protect API library is lower than 3.1.6 \n");
      printf(  "(multithreaded API)                                        \n");
      printf(  "Install the current level.                                 \n");
      printf(  "***********************************************************\n");
      return 0;
   }

   if (apiver >= 412)
   {
      if(apiLibVer.unicode)
         strcpy(uniStr,"(unicode)");
      else
         strcpy(uniStr,"         ");
   }

   printf("\n*************************************************************\n");
   printf(  "* Welcome to the sample application for                     *\n");
   printf(  "* IBM Storage Protect API.                                 *\n");
   printf(  "* API Library Version = %d.%d.%d.%d %s                   *\n",
   apiLibVer.version,
   apiLibVer.release,
   apiLibVer.level,
   apiLibVer.subLevel,
   uniStr);
   printf(  "*************************************************************\n");

   memset(&dsmEnvSetUp, 0x00, sizeof(envSetUp));

   dsmEnvSetUp.stVersion = envSetUpVersion;
   strcpy(dsmEnvSetUp.dsmiDir, "");
   strcpy(dsmEnvSetUp.dsmiConfig, "");
   strcpy(dsmEnvSetUp.dsmiLog, "");
   strcpy(dsmEnvSetUp.logName, "");
   dsmEnvSetUp.argv = argv;

   if ((rc = dsmSetUp(DSM_MULTITHREAD, &dsmEnvSetUp)) != DSM_RC_OK)
   {
          rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);
      dsmRCMsg(0,rc,rcMsg);
      printf("Error in dsmSetUp rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmCleanUp(DSM_MULTITHREAD);
      return 0;
   }

   NumberOfThreads++;
   rc = pthread_create(&thrd1, &attr, Thread1, NULL);
   if(rc) {
     printf("Thread1 rc is %d\n",rc);
     return 0;
   }



   while(NumberOfThreads > 1)
      sleep(1);

   printf("NumberofThreads is >%d<\n",NumberOfThreads);

   dsmCleanUp(DSM_MULTITHREAD);
   sleep(1);

   return 0;
}

void *Thread1(void *string)
{

   dsInt16_t          rc;
   int                i, counter=0;
   char               options[100];
   char               configfile[10];
   mcBindKey          bindKey;
   ObjAttr            objAttrArea;
   DataBlk            dataBlkArea, qDataBlkArea, getBlk;
   qryBackupData      queryBuffer;
   qryRespBackupData  qbDataArea;
   qryBackupGroups    qbBackGroups;
   dsUint16_t         reason;
   dsmObjName         objName;
   dsmQueryType       queryType;
   dsmGetList         dsmObjList;
   regFSData          regFilespace;
   dsmApiVersion      dsmApiVer;
   char               *rcMsg;
   dsUint32_t         handle = 0;
   char               unique[100];

   dsmInitExIn_t      initIn;
   dsmInitExOut_t     initOut;
   dsmApiVersionEx    apiApplVer;
   dsmGroupHandlerIn_t    dsmGroupHandlerIn;
   dsmGroupHandlerOut_t   dsmGroupHandlerOut;
   dsmEndTxnExIn_t        dsmEndTxnExIn;
   dsmEndTxnExOut_t       dsmEndTxnExOut;
   dsmEndSendObjExIn_t    dsmEndSendObjExIn;
   dsmEndSendObjExOut_t   dsmEndSendObjExOut;

   dsmDelInfo             delInfo;
   dsmGetList             objList;
   dsStruct64_t           memberToRemoveID;
   dsBool_t               idSet = bFalse;

   memset(&dsmGroupHandlerIn, 0x00, sizeof(dsmGroupHandlerIn_t));
   memset(&dsmGroupHandlerOut, 0x00, sizeof(dsmGroupHandlerOut_t));
   memset(&dsmEndTxnExIn,  0x00, sizeof(dsmEndTxnExIn_t));
   memset(&dsmEndTxnExOut, 0x00, sizeof(dsmEndTxnExOut_t));
   memset(&dsmEndSendObjExIn,  0x00, sizeof(dsmEndSendObjExIn_t));
   memset(&dsmEndSendObjExOut, 0x00, sizeof(dsmEndSendObjExOut_t));
   memset(&qbBackGroups,0x00, sizeof(qryBackupGroups));

   memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

   printf("Thread1 started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /************** Initialize the API session ******************************/
   rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   /************ The following 2 can be updated optionally *****************/
   configfile[0]= '\0';    /* API config file for dsmInit */
   options[0] = '\0';

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NODE1;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NODE1;
   initIn.applicationTypeP = "API Test1";
   initIn.configfile       = configfile;
   initIn.options          = options;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("thread1 error in dsmInit. rcMsg=%s\n",rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      NumberOfThreads--;
      dsmTerminate(handle);
      return NULL;
   }

   printf("Thread1 after dsmInit, session handle is %d\n",handle);

   strcpy(objName.fs,"/grp");
   strcpy(objName.hl,"/GroupLeader");
   strcpy(objName.ll,"/");
   objName.objType = DSM_OBJ_FILE;



   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,"/grp");
   strcpy(regFilespace.fsType,"CALLER1_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");

   /******************** Register the file space ************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   bindKey.stVersion = mcBindKeyVersion;

   if ((rc = dsmBindMC(handle, &objName, stBackup, &bindKey)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBindMC. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

    /******************  send group handler **************************/
   printf("\n***************************************\n");
   printf("*** Sending the Group Leader Object ***\n");
   printf("***************************************\n");

   dsmGroupHandlerIn.stVersion  = dsmGroupHandlerInVersion;
   dsmGroupHandlerIn.dsmHandle  = handle;
   dsmGroupHandlerIn.groupType  = DSM_GROUPTYPE_PEER;
   dsmGroupHandlerIn.actionType = DSM_GROUP_ACTION_OPEN;
   dsmGroupHandlerIn.memberType = DSM_MEMBERTYPE_LEADER;
   dsmGroupHandlerIn.objNameP   = &objName;
   dsmGroupHandlerIn.uniqueGroupTagP = unique;
   strcpy(dsmGroupHandlerIn.uniqueGroupTagP, "TESTGROUP");

   if ((rc = dsmGroupHandler(&dsmGroupHandlerIn, &dsmGroupHandlerOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmGroupHandler. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }
   /****************** Backup the data ******************************/
   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;           /* size estimate of object */
   objAttrArea.sizeEstimate.lo = 420;         /* size estimate of object */
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 10;            /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"abcdefghij");  /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea,
        NULL)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBackup. rcMsg=%s\n",rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   dataBlkArea.bufferPtr = (char *)malloc(20);
   dataBlkArea.stVersion = DataBlkVersion;
   dataBlkArea.bufferLen = 20;              /* buffer length in bytes */
   /* datastream to backup */
   memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1T1T1", 20);

   for (i=1; i<=21; i++)
   {
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK)
          {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread1 error in dsmSendData. rcMsg=%s\n",rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;
                 return NULL;
      }

      if(i == 20)
         memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1TlT\0", 20);
   }

   dsmEndSendObj(handle);
   dsmEndTxnExIn.stVersion = dsmEndTxnExInVersion;
   dsmEndTxnExIn.dsmHandle = handle;
   dsmEndTxnExIn.vote      = DSM_VOTE_COMMIT;

   dsmEndTxnExOut.stVersion = dsmEndTxnExOutVersion;
   rc = dsmEndTxnEx(&dsmEndTxnExIn, &dsmEndTxnExOut);
   if (rc != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }
   reason = dsmEndTxnExOut.reason;
   printf(dsTEXT("After endTxn leaderObjId is %u, %u\n"),
          dsmEndTxnExOut.groupLeaderObjId.hi,
          dsmEndTxnExOut.groupLeaderObjId.lo );


   /***************** add members ************************************/

   printf("\n***************************************\n");
   printf("***** Adding Group Member Objects *****\n");
   printf("***************************************\n");
   objName.objType = DSM_OBJ_FILE;
   strcpy(objName.ll,dsTEXT("/member"));
   for (i = 1; i < 5; i++)
   {
      char temp[4];

      if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmBeginTxn. rcMsg=%s\n"),rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }

      dsmGroupHandlerIn.groupType   = DSM_GROUPTYPE_PEER;
      dsmGroupHandlerIn.actionType  = DSM_GROUP_ACTION_ADD;
      dsmGroupHandlerIn.memberType  = DSM_MEMBERTYPE_MEMBER;
      dsmGroupHandlerIn.objNameP    = &objName;
      dsmGroupHandlerIn.leaderObjId = dsmEndTxnExOut.groupLeaderObjId;

      if ((rc = dsmGroupHandler(&dsmGroupHandlerIn, &dsmGroupHandlerOut)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmGroupHandler. rcMsg=%s\n"),rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }

      printf(dsTEXT("returned from dsmGroupHandler ADD rc = 0\n"));

      sprintf(temp,dsTEXT("%d"), i);
      strcat(objName.ll,temp);
      if ((rc = dsmBindMC(handle, &objName, stBackup, &bindKey)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmBindMC. rcMsg=%s\n"), rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
      if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea,
        NULL)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmBackup. rcMsg=%s\n"),rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmSendData. rcMsg=%s\n"),rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
      dsmEndSendObjExIn.stVersion  = dsmEndSendObjExInVersion;
      dsmEndSendObjExIn.dsmHandle  = handle;
      dsmEndSendObjExOut.stVersion = dsmEndSendObjExOutVersion;
      if ((rc = dsmEndSendObjEx(&dsmEndSendObjExIn, &dsmEndSendObjExOut)) != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmSendData. rcMsg=%s\n"),rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
      rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason);
      if (rc != DSM_RC_OK)
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf(dsTEXT("Thread1 error in dsmEndTxn. rcMsg=%s\n"), rcMsg);
         printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
      printf(dsTEXT("finished sending object %s\n"), objName.ll);

   }/* end of for loop*/
   printf("\n\n****************************************************************\n");
   printf("******************* Performing a Normal Query ******************\n");
   printf("****************************************************************");
   queryType = qtBackup;

   qbDataArea.stVersion = qryRespBackupDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespBackupData);

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);

   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, "/*");
   strcpy(queryBuffer.objName->ll, "/*");
   queryBuffer.objName->objType = DSM_OBJ_WILDCARD;
   strcpy(queryBuffer.owner, "");
   queryBuffer.objState = DSM_ACTIVE;
   queryBuffer.stVersion = qryBackupDataVersion;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
       counter = counter + 1;
       printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
              dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
              dsTEXT("  *** groupLeaderId is %u %u\n"),
              qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
              qbDataArea.isGroupLeader,
              qbDataArea.objName.objType, qbDataArea.compressType,
              qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
              qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }
   if (counter == 5)
       printf("+++++++++++++++++++ Normal Query Successful +++++++++++++++++++\n");
   else
       printf("+++++++++++++++++++ Normal Query Failed!!! ++++++++++++++++++++\n"); 

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

  /*=== query all the group members ===*/
   qbBackGroups.owner = (char *)malloc(10); 
   qbBackGroups.stVersion        = qryBackupGroupsVersion;
   qbBackGroups.groupType        = DSM_GROUPTYPE_PEER;
   qbBackGroups.fsName           = objName.fs;
   strcpy(qbBackGroups.owner, "");
   qbBackGroups.groupLeaderObjId = dsmEndTxnExOut.groupLeaderObjId;
   qbBackGroups.objType          = DSM_OBJ_WILDCARD;



   printf("\n\n****************************************************************\n");
   printf("********************** Query Group Members *********************\n");
   printf("****************************************************************");
   if ((rc=dsmBeginQuery(handle, qtBackupGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if( counter == 4)
      printf("+++++++++++++++ Query Group Members Succesfull ++++++++++++++++\n");
   else
      printf("+++++++++++++++ Query Group Members Failed!!! ++++++++++++++++\n");

   if (!idSet)
   {
           memberToRemoveID = qbDataArea.objId;
           idSet = bTrue;
   }
   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }            /* end query group members *====================================*/

   /*=== query all the group leaders ===========================================*/
   qbBackGroups.stVersion        = qryBackupGroupsVersion;
   qbBackGroups.groupType        = DSM_GROUPTYPE_PEER;
   qbBackGroups.fsName = objName.fs;
   qbBackGroups.groupLeaderObjId.hi = 0;
   qbBackGroups.groupLeaderObjId.lo = 0;
   qbBackGroups.objType          = DSM_OBJ_WILDCARD;


   printf("\n\n****************************************************************\n");
   printf("********************** Query Group Leaders *********************\n");
   printf("****************************************************************\n");
   if ((rc=dsmBeginQuery(handle, qtBackupGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if ( counter == 0)
      printf("+++++++++++++++++ Query Group Leaders Successfull ++++++++++++++\n");
   else
      printf("+++++++++++++++++ Query Group Leaders Failed!!! +++++++++++++++\n");

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   } /* end query group leaders *============================================*/

   printf("\n\n****************************************************************\n");
   printf("******************* Query Open Group Leaders *******************\n");
   printf("****************************************************************");
   if ((rc=dsmBeginQuery(handle, qtOpenGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }
   
   if ( counter == 1)
      printf("+++++++++++++ Query Open Group Leaders Successfull +++++++++++++\n");
   else
      printf("+++++++++++++ Query Open Group Leaders Failed!!! ++++++++++++++\n");

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   } /* end query group leaders *===============================================*/

   /* Calling Close of group *==================================================*/
   printf("\n***************************************\n");
   printf("******* Calling Close of Group ********\n");
   printf("***************************************\n");
   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginTxn. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }
   dsmGroupHandlerIn.groupType  = DSM_GROUPTYPE_PEER;
   dsmGroupHandlerIn.actionType = DSM_GROUP_ACTION_CLOSE;
   dsmGroupHandlerIn.memberType = DSM_MEMBERTYPE_LEADER;
   strcpy(objName.ll,dsTEXT("/"));
   dsmGroupHandlerIn.objNameP = &objName;
   dsmGroupHandlerIn.objNameP->objType = DSM_OBJ_FILE;
   dsmGroupHandlerIn.leaderObjId = dsmEndTxnExOut.groupLeaderObjId;

   if ((rc = dsmGroupHandler(&dsmGroupHandlerIn, &dsmGroupHandlerOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmGroupHandler. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn. rcMsg=%s\n"), rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn.reason %d\n"), reason);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf(">>>>>>>>Successfully returned from dsmGroupHandler CLOSE rc = 0\n");

   /*=== query all the group leaders ===*/
   printf("\n\n****************************************************************\n");
   printf("********************** Query Group Leaders *********************\n");
   printf("****************************************************************");   
   qbBackGroups.stVersion        = qryBackupGroupsVersion;
   qbBackGroups.groupType        = DSM_GROUPTYPE_PEER;
   qbBackGroups.fsName = objName.fs;
   qbBackGroups.groupLeaderObjId.hi = 0;
   qbBackGroups.groupLeaderObjId.lo = 0;
   qbBackGroups.objType          = DSM_OBJ_WILDCARD;

   if ((rc=dsmBeginQuery(handle, qtBackupGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }
   
   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if ( counter == 1)
      printf("+++++++++++++++++ Query Group Leaders Successfull ++++++++++++++\n");
   else
      printf("+++++++++++++++++ Query Group Leaders Failed!!! +++++++++++++++\n");

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }  /* end query group leaders *=================================================*/

   printf("\n\n****************************************************************\n");
   printf("******************* Query Open Group Leaders *******************\n");
   printf("****************************************************************\n");
   if ((rc=dsmBeginQuery(handle, qtOpenGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if ( counter == 0)
      printf("+++++++++++++ Query Open Group Leaders Successfull +++++++++++++\n");
   else
      printf("+++++++++++++ Query Open Group Leaders Failed!!! ++++++++++++++\n"); 

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }  /* end query open group leaders *=============================================*/

   /* calling remove of a member ==================================================*/
   printf("\n***************************************\n");
   printf("****** Calling Remove of a Member *****\n");
   printf("***************************************\n");

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginTxn. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }
   dsmGroupHandlerIn.groupType  = DSM_GROUPTYPE_PEER;
   dsmGroupHandlerIn.actionType = DSM_GROUP_ACTION_REMOVE;
   dsmGroupHandlerIn.memberType = DSM_MEMBERTYPE_MEMBER;
   strcpy(objName.ll,dsTEXT("/member1"));
   dsmGroupHandlerIn.objNameP = &objName;
   dsmGroupHandlerIn.objNameP->objType = DSM_OBJ_FILE;
   dsmGroupHandlerIn.leaderObjId = dsmEndTxnExOut.groupLeaderObjId;
   objList.stVersion = dsmGetListVersion;
   objList.numObjId = 2;
   objList.objId = (ObjID *)malloc(sizeof(ObjID) * objList.numObjId);

   objList.objId[0] = memberToRemoveID;
   memberToRemoveID.lo = memberToRemoveID.lo - 1;
   objList.objId[1] = memberToRemoveID;
   dsmGroupHandlerIn.memberObjList = objList;

   if ((rc = dsmGroupHandler(&dsmGroupHandlerIn, &dsmGroupHandlerOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmGroupHandler. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }


   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn. rcMsg=%s\n"), rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn.reason %d\n"), reason);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }
   printf(dsTEXT(">>>>>>>>>>>>>>>Successful removal of members:\n"), rcMsg);
   printf(dsTEXT("Member ID %u %u\n"), memberToRemoveID.hi, memberToRemoveID.lo);
   printf(dsTEXT("Member ID %u %u\n"), memberToRemoveID.hi, memberToRemoveID.lo + 1);
   /* end remove of a member *======================================================*/

   /*=== query all the group members ===*/
   printf("\n\n****************************************************************\n");
   printf("********************** Query Group Members *********************\n");
   printf("****************************************************************"); 

   qbBackGroups.stVersion        = qryBackupGroupsVersion;
   qbBackGroups.groupType        = DSM_GROUPTYPE_PEER;
   qbBackGroups.fsName           = objName.fs;
   strcpy(qbBackGroups.owner, "");
   qbBackGroups.groupLeaderObjId = dsmEndTxnExOut.groupLeaderObjId;
   qbBackGroups.objType          = DSM_OBJ_WILDCARD;

    if ((rc=dsmBeginQuery(handle, qtBackupGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if (counter == 2)
      printf("+++++++++++++++ Query Group Members Succesfull ++++++++++++++++\n");
   else
      printf("+++++++++++++++ Query Group Members Failed!!! ++++++++++++++++\n");

   if (!idSet)
   {
           memberToRemoveID = qbDataArea.objId;
           idSet = bTrue;
   }
   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   } /* end  query group members *===========================================*/

   /* calling assignto of a member *============================================*/
   printf("\n***************************************\n");
   printf("******* Calling Assign A Member *******\n");
   printf("***************************************\n"); 

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginTxn. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }
   dsmGroupHandlerIn.groupType  = DSM_GROUPTYPE_PEER;
   dsmGroupHandlerIn.actionType = DSM_GROUP_ACTION_ASSIGNTO;
   dsmGroupHandlerIn.memberType = DSM_MEMBERTYPE_MEMBER;
   strcpy(objName.ll,dsTEXT("/member1"));
   dsmGroupHandlerIn.objNameP = &objName;
   dsmGroupHandlerIn.objNameP->objType = DSM_OBJ_FILE;
   dsmGroupHandlerIn.leaderObjId = dsmEndTxnExOut.groupLeaderObjId;
   objList.stVersion = dsmGetListVersion;
   objList.numObjId = 2;
   objList.objId = (ObjID *)malloc(sizeof(ObjID) * objList.numObjId);

   objList.objId[0] = memberToRemoveID;
   memberToRemoveID.lo  += 1;
   objList.objId[1] = memberToRemoveID;
   dsmGroupHandlerIn.memberObjList = objList;

   if ((rc = dsmGroupHandler(&dsmGroupHandlerIn, &dsmGroupHandlerOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmGroupHandler. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }


   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn. rcMsg=%s\n"), rcMsg);
      printf(dsTEXT("Thread1 error in dsmEndTxn.reason %d\n"), reason);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }
   printf(dsTEXT(">>>>>>>>>>>>>>>Successful assignment of members:\n"), rcMsg);
   printf(dsTEXT("Member ID %u %u\n"), memberToRemoveID.hi, memberToRemoveID.lo - 1);
   printf(dsTEXT("Member ID %u %u\n"), memberToRemoveID.hi, memberToRemoveID.lo);

   /*=== query all the group members ===*/
   printf("\n\n****************************************************************\n");
   printf("********************** Query Group Members *********************\n");
   printf("****************************************************************");

   qbBackGroups.stVersion        = qryBackupGroupsVersion;
   qbBackGroups.groupType        = DSM_GROUPTYPE_PEER;
   qbBackGroups.fsName           = objName.fs;
   strcpy(qbBackGroups.owner, "");
   qbBackGroups.groupLeaderObjId = dsmEndTxnExOut.groupLeaderObjId;
   qbBackGroups.objType          = DSM_OBJ_WILDCARD;

   if ((rc=dsmBeginQuery(handle, qtBackupGroups, (void *)&qbBackGroups ))
        != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error in dsmBeginQuery. rcMsg=%s\n"),rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   counter = 0;
   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
   {
      counter = counter + 1;
      printf(dsTEXT("\n*** Thread1 query response %s%s%s isGroupleader %d***\n")
             dsTEXT("  *** objType is %d compressed is %d objState is %d id is %u %u\n")
             dsTEXT("  *** groupLeaderId is %u %u\n"),
             qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
             qbDataArea.isGroupLeader,
             qbDataArea.objName.objType, qbDataArea.compressType,
             qbDataArea.objState, qbDataArea.objId.hi, qbDataArea.objId.lo,
             qbDataArea.baseObjId.hi, qbDataArea.baseObjId.lo);
   }

   if( counter == 4)
      printf("+++++++++++++++ Query Group Members Succesfull ++++++++++++++++\n\n");
   else
      printf("+++++++++++++++ Query Group Members Failed!!! ++++++++++++++++\n\n");

   if (!idSet)
   {
           memberToRemoveID = qbDataArea.objId;
           idSet = bTrue;
   }
   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA))
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmGetNextQObj. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf(dsTEXT("Thread1 error from dsmEndQuery. rcMsg=%s\n"), rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      dsmTerminate(handle);
      NumberOfThreads--;
          return NULL;
   }  /* end query group members *==========================================*/

   if ((rc = dsmTerminate(handle)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmTerminate. rcMsg=%s\n", rcMsg);
      printf(" ++++++++++++++++ DSMGRP FAILED +++++++++++++++\n\n");
      free(rcMsg);
      NumberOfThreads--;
      return NULL;
   }

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dataBlkArea.bufferPtr);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   free(qbBackGroups.owner);
   free(rcMsg);

   printf("Thread 1 called dsmTerminate\n");
   printf("Number of threads = >%d< \n",NumberOfThreads);
   NumberOfThreads--;
   return NULL;
}



