#include "pkcs11types.h"

#ifndef _AES_H_
#define _AES_H_
#endif
#if !defined(TRUE)
#define TRUE 1
#endif

#if !defined(FALSE)
#define FALSE 0
#endif

#define MAX_KEY_SIZE  64
#define MAX_TEXT_SIZE 4096
#define MAX_IV_SIZE  64
#define MAX_COUNTER_SIZE 16
#define MAX_AAD_SIZE 512
#define MAX_TAG_SIZE 16
#define AES_BLOCK_SIZE	16
#define AES_COUNTER_VALUE "0123456789012345"
#define AES_IV_VALUE "1234567890123456"
#define AES_KEY_LEN 32

#define AES_IV_SIZE 16
#define AES_COUNTER_SIZE 16
#define MAX_CHUNKS 8


struct aes_test_vector {
	char key[MAX_KEY_SIZE];
	unsigned char klen;
	char plaintext[MAX_TEXT_SIZE];
	unsigned char plen;
	char ciphertext[MAX_TEXT_SIZE];
	unsigned char clen;
	char iv[MAX_IV_SIZE];           // aes cbc mode
	unsigned char ivlen;            // aes cbc mode
	char counter[MAX_COUNTER_SIZE]; // aes ctr mode
	unsigned char counterlen;       // aes ctr mode
	unsigned int counterbits;       // aes ctr mode
	unsigned char aad[MAX_AAD_SIZE];// aes gcm mode
	unsigned int aadlen;			// aes gcm mode
	unsigned int tag[MAX_TAG_SIZE];	// aes gcm mode
	unsigned int taglen;			// aes gcm mode
	int chunks_plain[MAX_CHUNKS];
	int num_chunks_plain;
	int chunks_ciph[MAX_CHUNKS];
	int num_chunks_ciph;
};

struct published_test_suite_info {
	const char *name;
	unsigned int tvcount;
	struct aes_test_vector *tv;
	unsigned int size;
	CK_MECHANISM mech;
};

struct generated_test_suite_info {
        const char *name;
        CK_MECHANISM mech;
	unsigned int size;
};

char aes_iv[] = { 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
                  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f };

struct CK_AES_CTR_PARAMS aesctr = {
        .ulCounterBits = AES_COUNTER_SIZE,
        .cb = { 0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
                0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff },
};

struct CK_GCM_PARAMS aesgcm = {
	.ulIvLen = AES_BLOCK_SIZE,
	.ulAADLen = 0,
	.ulTagBits = 16,
};

struct CK_MECHANISM aes_keygen = {
        .mechanism      = CKM_AES_KEY_GEN,
        .ulParameterLen = 0,
        .pParameter     = NULL,
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ecb_tv[] = {

	{	// #0
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.ciphertext =	{ 0x3a,0xd7,0x7b,0xb4,0x0d,0x7a,0x36,0x60,
				  0xa8,0x9e,0xca,0xf3,0x24,0x66,0xef,0x97,
				  0xf5,0xd3,0xd5,0x85,0x03,0xb9,0x69,0x9d,
				  0xe7,0x85,0x89,0x5a,0x96,0xfd,0xba,0xaf,
				  0x43,0xb1,0xcd,0x7f,0x59,0x8e,0xce,0x23,
				  0x88,0x1b,0x00,0xe3,0xed,0x03,0x06,0x88,
				  0x7b,0x0c,0x78,0x5e,0x27,0xe8,0xad,0x3f,
				  0x82,0x23,0x20,0x71,0x04,0x72,0x5d,0xd4 },
		.clen = 64,
	}, {	// #1
		.key =		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 4,
		.chunks_plain = 	{ 16, 16, 16, 16 },
		.ciphertext =	{ 0xbd,0x33,0x4f,0x1d,0x6e,0x45,0xf2,0x5f,
				  0xf7,0x12,0xa2,0x14,0x57,0x1f,0xa5,0xcc,
				  0x97,0x41,0x04,0x84,0x6d,0x0a,0xd3,0xad,
				  0x77,0x34,0xec,0xb3,0xec,0xee,0x4e,0xef,
				  0xef,0x7a,0xfd,0x22,0x70,0xe2,0xe6,0x0a,
				  0xdc,0xe0,0xba,0x2f,0xac,0xe6,0x44,0x4e,
				  0x9a,0x4b,0x41,0xba,0x73,0x8d,0x6c,0x72,
				  0xfb,0x16,0x69,0x16,0x03,0xc1,0x8e,0x0e },
		.clen = 64,
		.num_chunks_ciph = 4,
		.chunks_ciph =	{ 16, 16, 16, 16 },
	}, {	// #2
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain = 	{ 25, 0, 8, -1, 31 },
		.ciphertext =	{ 0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
				  0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8,
				  0x59,0x1c,0xcb,0x10,0xd4,0x10,0xed,0x26,
				  0xdc,0x5b,0xa7,0x4a,0x31,0x36,0x28,0x70,
				  0xb6,0xed,0x21,0xb9,0x9c,0xa6,0xf4,0xf9,
				  0xf1,0x53,0xe7,0xb1,0xbe,0xaf,0xed,0x1d,
				  0x23,0x30,0x4b,0x7a,0x39,0xf9,0xf3,0xff,
				  0x06,0x7d,0x8d,0x8f,0x9e,0x24,0xec,0xc7 },
		.clen = 64,
		.num_chunks_ciph = 5,
		.chunks_ciph = 	{ 25, 0, 8, -1, 31 },
	},
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cbc_tv[] = {
	{	// #0
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.ciphertext =	{ 0x76,0x49,0xab,0xac,0x81,0x19,0xb2,0x46,
				  0xce,0xe9,0x8e,0x9b,0x12,0xe9,0x19,0x7d,
				  0x50,0x86,0xcb,0x9b,0x50,0x72,0x19,0xee,
				  0x95,0xdb,0x11,0x3a,0x91,0x76,0x78,0xb2,
				  0x73,0xbe,0xd6,0xb8,0xe3,0xc1,0x74,0x3b,
				  0x71,0x16,0xe6,0x9e,0x22,0x22,0x95,0x16,
				  0x3f,0xf1,0xca,0xa1,0x68,0x1f,0xac,0x09,
				  0x12,0x0e,0xca,0x30,0x75,0x86,0xe1,0xa7 },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
	}, {	// #1
		.key = 		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 4,
		.chunks_plain = 	{ 16, 16, 16, 16 },
		.ciphertext =	{ 0x4f,0x02,0x1d,0xb2,0x43,0xbc,0x63,0x3d,
				  0x71,0x78,0x18,0x3a,0x9f,0xa0,0x71,0xe8,
				  0xb4,0xd9,0xad,0xa9,0xad,0x7d,0xed,0xf4,
				  0xe5,0xe7,0x38,0x76,0x3f,0x69,0x14,0x5a,
				  0x57,0x1b,0x24,0x20,0x12,0xfb,0x7a,0xe0,
				  0x7f,0xa9,0xba,0xac,0x3d,0xf1,0x02,0xe0,
				  0x08,0xb0,0xe2,0x79,0x88,0x59,0x88,0x81,
				  0xd9,0x20,0xa9,0xe6,0x4f,0x56,0x15,0xcd },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 4,
		.chunks_ciph = 	{ 16, 16, 16, 16 },
	}, {	// #2
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain = 	{ 25, 0, 8, -1, 31 },
		.ciphertext =	{ 0xf5,0x8c,0x4c,0x04,0xd6,0xe5,0xf1,0xba,
				  0x77,0x9e,0xab,0xfb,0x5f,0x7b,0xfb,0xd6,
				  0x9c,0xfc,0x4e,0x96,0x7e,0xdb,0x80,0x8d,
				  0x67,0x9f,0x77,0x7b,0xc6,0x70,0x2c,0x7d,
				  0x39,0xf2,0x33,0x69,0xa9,0xd9,0xba,0xcf,
				  0xa5,0x30,0xe2,0x63,0x04,0x23,0x14,0x61,
				  0xb2,0xeb,0x05,0xe2,0xc3,0x9b,0xe9,0xfc,
				  0xda,0x6c,0x19,0x07,0x8c,0x6a,0x9d,0x1b },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 5,
		.chunks_ciph = 	{ 25, 0, 8, -1, 31 },
	},
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ctr_tv[] = {
	{	// #0
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.counter = 	{ 0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
				  0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff },
		.counterlen = 16,
		.counterbits = 128,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.ciphertext =	{ 0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
				  0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
				  0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
				  0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
				  0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
				  0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
				  0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
				  0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee },
		.clen = 64,

	}, {	// #1
		.key =		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.counter =	{ 0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
				  0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff },
		.counterlen = 16,
		.counterbits = 128,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen =  64,
		.num_chunks_plain = 4,
		.chunks_plain = 	{ 16, 16, 16, 16 },
		.ciphertext =	{ 0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
				  0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b,
				  0x09,0x03,0x39,0xec,0x0a,0xa6,0xfa,0xef,
				  0xd5,0xcc,0xc2,0xc6,0xf4,0xce,0x8e,0x94,
				  0x1e,0x36,0xb2,0x6b,0xd1,0xeb,0xc6,0x70,
				  0xd1,0xbd,0x1d,0x66,0x56,0x20,0xab,0xf7,
				  0x4f,0x78,0xa7,0xf6,0xd2,0x98,0x09,0x58,
				  0x5a,0x97,0xda,0xec,0x58,0xc6,0xb0,0x50 },
		.clen = 64,
		.num_chunks_ciph = 4,
		.chunks_ciph = 	{ 16, 16, 16, 16 },
	}, {	// #2
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.counter =	{ 0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
				  0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff },
		.counterlen = 16,
		.counterbits = 128,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain	  = { 25, 0, 8, -1, 31 },
		.ciphertext =	{ 0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
				  0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28,
				  0xf4,0x43,0xe3,0xca,0x4d,0x62,0xb5,0x9a,
				  0xca,0x84,0xe9,0x90,0xca,0xca,0xf5,0xc5,
				  0x2b,0x09,0x30,0xda,0xa2,0x3d,0xe9,0x4c,
				  0xe8,0x70,0x17,0xba,0x2d,0x84,0x98,0x8d,
				  0xdf,0xc9,0xc5,0x8d,0xb6,0x7a,0xad,0xa6,
				  0x13,0xc2,0xdd,0x08,0x45,0x79,0x41,0xa6 },
		.clen = 64,
		.num_chunks_ciph = 5,
		.chunks_ciph = 	{ 25, 0, 8, -1, 31 },
	},
};

static struct aes_test_vector aes_gcm_tv[] = {
	{	/* #0 test with chunks */
		.key =		{ 0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,
				  0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08 },
		.klen = 16,
		.iv =		{ 0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,
				  0xde,0xca,0xf8,0x88, },
		.ivlen = 12,
		.aad =		{},
		.aadlen = 0,
		.tag =		{ 0x4d,0x5c,0x2a,0xf3,0x27,0xcd,0x64,0xa6,
				  0x2c,0xf3,0x5a,0xbd,0x2b,0xa6,0xfa,0xb4 },
		.taglen = 128,
		.plaintext =	{ 0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,
				  0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
				  0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,
				  0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
				  0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,
				  0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
				  0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,
				  0xba,0x63,0x7b,0x39,0x1a,0xaf,0xd2,0x55 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain =	{ 20, 5, 7, 1, 31 },
		.ciphertext =	{ 0x42,0x83,0x1e,0xc2,0x21,0x77,0x74,0x24,
				  0x4b,0x72,0x21,0xb7,0x84,0xd0,0xd4,0x9c,
				  0xe3,0xaa,0x21,0x2f,0x2c,0x02,0xa4,0xe0,
				  0x35,0xc1,0x7e,0x23,0x29,0xac,0xa1,0x2e,
				  0x21,0xd5,0x14,0xb2,0x54,0x66,0x93,0x1c,
				  0x7d,0x8f,0x6a,0x5a,0xac,0x84,0xaa,0x05,
				  0x1b,0xa3,0x0b,0x39,0x6a,0x0a,0xac,0x97,
				  0x3d,0x58,0xe0,0x91,0x47,0x3f,0x59,0x85,
				  0x4d,0x5c,0x2a,0xf3,0x27,0xcd,0x64,0xa6,
				  0x2c,0xf3,0x5a,0xbd,0x2b,0xa6,0xfa,0xb4 },
		.clen = 80,
		.num_chunks_ciph = 7,
		.chunks_ciph =	{ 20, 5, 7, 1, 31, 10, 6 },
	}, {

	/* #1 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 0]
	 * [AADlen = 0]
	 * [Taglen = 128]
	 */
		.key =		{ 0x11,0x75,0x4c,0xd7,0x2a,0xec,0x30,0x9b,
				  0xf5,0x2f,0x76,0x87,0x21,0x2e,0x89,0x57 },
		.klen = 16,
		.iv =		{ 0x3c,0x81,0x9d,0x9a,0x9b,0xed,0x08,0x76,
				  0x15,0x03,0x0b,0x65 },
		.ivlen = 12,
		.aad =		{},
		.aadlen = 0,
		.tag =		{ 0x25,0x03,0x27,0xc6,0x74,0xaa,0xf4,0x77,
				  0xae,0xf2,0x67,0x57,0x48,0xcf,0x69,0x71 },
		.taglen = 128,
		.plaintext =	{ },
		.plen = 0,
		.ciphertext =	{ 0x25,0x03,0x27,0xc6,0x74,0xaa,0xf4,0x77,
				  0xae,0xf2,0x67,0x57,0x48,0xcf,0x69,0x71 },
		.clen = 16,
	}, {

	/* #2 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 128]
	 * [AADlen = 0]
	 * [Taglen = 128]
	 */
		.key =		{ 0x99,0x71,0x07,0x10,0x59,0xab,0xc0,0x09,
				  0xe4,0xf2,0xbd,0x69,0x86,0x9d,0xb3,0x38 },
		.klen = 16,
		.iv =		{ 0x07,0xa9,0xa9,0x5e,0xa3,0x82,0x1e,0x9c,
				  0x13,0xc6,0x32,0x51},
		.ivlen = 12,
		.aad =		{ },
		.aadlen = 0,
		.tag =		{ 0x78,0x70,0xd9,0x11,0x7f,0x54,0x81,0x1a,
				  0x34,0x69,0x70,0xf1,0xde,0x09,0x0c,0x41 },
		.taglen = 128,
		.plaintext =	{ 0xf5,0x4b,0xc3,0x50,0x1f,0xed,0x4f,0x6f,
				  0x6d,0xfb,0x5e,0xa8,0x01,0x06,0xdf,0x0b,
				  0xd8,0x36,0xe6,0x82,0x62,0x25,0xb7,0x5c,
				  0x02,0x22,0xf6,0xe8,0x59,0xb3,0x59,0x83 },
		.plen = 32,
		.ciphertext =	{ 0x05,0x56,0xc1,0x59,0xf8,0x4e,0xf3,0x6c,
				  0xb1,0x60,0x2b,0x45,0x26,0xb1,0x20,0x09,
				  0xc7,0x75,0x61,0x1b,0xff,0xb6,0x4d,0xc0,
				  0xd9,0xca,0x92,0x97,0xcd,0x2c,0x6a,0x01,
				  0x78,0x70,0xd9,0x11,0x7f,0x54,0x81,0x1a,
				  0x34,0x69,0x70,0xf1,0xde,0x09,0x0c,0x41},
		.clen = 48,
	}, {

	/* #3 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 104]
	 * [AADlen = 0]
	 * [Taglen = 128]
	 */
		.key =		{ 0xfe,0x9b,0xb4,0x7d,0xeb,0x3a,0x61,0xe4,
				  0x23,0xc2,0x23,0x18,0x41,0xcf,0xd1,0xfb },
		.klen = 16,
		.iv =		{ 0x4d,0x32,0x8e,0xb7,0x76,0xf5,0x00,0xa2,
				  0xf7,0xfb,0x47,0xaa },
		.ivlen = 12,
		.aad =		{ },
		.aadlen = 0,
		.tag =		{ 0x43,0xfd,0x47,0x27,0xfe,0x5c,0xdb,0x4b,
				  0x5b,0x42,0x81,0x8d,0xea,0x7e,0xf8,0xc9 },
		.taglen = 128,
		.plaintext =	{ 0xf1,0xcc,0x38,0x18,0xe4,0x21,0x87,0x6b,
				  0xb6,0xb8,0xbb,0xd6,0xc9 },
		.plen = 13,
		.ciphertext =	{ 0xb8,0x8c,0x5c,0x19,0x77,0xb3,0x5b,0x51,
				  0x7b,0x0a,0xea,0xe9,0x67,0x43,0xfd,0x47,
				  0x27,0xfe,0x5c,0xdb,0x4b,0x5b,0x42,0x81,
				  0x8d,0xea,0x7e,0xf8,0xc9 },
		.clen = 29,
	}, {

	/* #4 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 128]
	 * [AADlen = 0]
	 * [Taglen = 96]
	 */
		.key = 		{ 0xf0,0x0f,0xdd,0x01,0x8c,0x02,0xe0,0x35,
				  0x76,0x00,0x8b,0x51,0x6e,0xa9,0x71,0xad },
		.klen = 16,
		.iv = 		{ 0x3b,0x3e,0x27,0x6f,0x9e,0x98,0xb1,0xec,
				  0xb7,0xce,0x6d,0x28 },
		.ivlen = 12,
		.aad =		{ },
		.aadlen = 0,
		.tag =		{ 0xcb,0xa0,0x6b,0xb4,0xf6,0xe0,0x97,0x19,
				  0x92,0x50,0xb0,0xd1 },
		.taglen = 96,
		.plaintext =	{ 0x28,0x53,0xe6,0x6b,0x7b,0x1b,0x3e,0x1f,
				  0xa3,0xd1,0xf3,0x72,0x79,0xac,0x82,0xbe },
		.plen = 16,
		.ciphertext =	{ 0x55,0xd2,0xda,0x7a,0x3f,0xb7,0x73,0xb8,
				  0xa0,0x73,0xdb,0x49,0x9e,0x24,0xbf,0x62,
				  0xcb,0xa0,0x6b,0xb4,0xf6,0xe0,0x97,0x19,
				  0x92,0x50,0xb0,0xd1 },
		.clen = 28,
	}, {

	/* #5 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 128]
	 * [AADlen = 128]
	 * [Taglen = 128]
	 */
		.key =		{ 0xc9,0x39,0xcc,0x13,0x39,0x7c,0x1d,0x37,
				  0xde,0x6a,0xe0,0xe1,0xcb,0x7c,0x42,0x3c },
		.klen = 16,
		.iv = 		{ 0xb3,0xd8,0xcc,0x01,0x7c,0xbb,0x89,0xb3,
				  0x9e,0x0f,0x67,0xe2 },
		.ivlen = 12,
		.aad =		{ 0x24,0x82,0x56,0x02,0xbd,0x12,0xa9,0x84,
				  0xe0,0x09,0x2d,0x3e,0x44,0x8e,0xda,0x5f},
		.aadlen = 16,
		.tag = 		{ 0x00,0x32,0xa1,0xdc,0x85,0xf1,0xc9,0x78,
				  0x69,0x25,0xa2,0xe7,0x1d,0x82,0x72,0xdd },
		.taglen = 128,
		.plaintext =	{ 0xc3,0xb3,0xc4,0x1f,0x11,0x3a,0x31,0xb7,
				  0x3d,0x9a,0x5c,0xd4,0x32,0x10,0x30,0x69 },
		.plen = 16,
		.ciphertext =	{ 0x93,0xfe,0x7d,0x9e,0x9b,0xfd,0x10,0x34,
				  0x8a,0x56,0x06,0xe5,0xca,0xfa,0x73,0x54,
				  0x00,0x32,0xa1,0xdc,0x85,0xf1,0xc9,0x78,
				  0x69,0x25,0xa2,0xe7,0x1d,0x82,0x72,0xdd },
		.clen = 32,
	}, {

	/* #6 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 128]
	 * [AADlen = 128]
	 * [Taglen = 96]
	 */
		.key =		{ 0x56,0x2a,0xe8,0xaa,0xdb,0x8d,0x23,0xe0,
				  0xf2,0x71,0xa9,0x9a,0x7d,0x1b,0xd4,0xd1 },
		.klen = 16,
		.iv =		{ 0xf7,0xa5,0xe2,0x39,0x94,0x13,0xb8,0x9b,
				  0x6a,0xd3,0x1a,0xff },
		.ivlen = 12,
		.aad =		{ 0x2b,0x96,0x80,0xb8,0x86,0xb3,0xef,0xb7,
				  0xc6,0x35,0x4b,0x38,0xc6,0x3b,0x53,0x73},
		.aadlen = 16,
		.tag =		{ 0xcb,0xf2,0x01,0x51,0x84,0xff,0xfb,0x82,
				  0xf2,0x65,0x1c,0x36 },
		.taglen = 96,
		.plaintext =	{ 0xbb,0xdc,0x35,0x04,0xd8,0x03,0x68,0x2a,
				  0xa0,0x8a,0x77,0x3c,0xde,0x5f,0x23,0x1a },
		.plen = 16,
		.ciphertext =	{ 0xe2,0xb7,0xe5,0xed,0x5f,0xf2,0x7f,0xc8,
				  0x66,0x41,0x48,0xf5,0xa6,0x28,0xa4,0x6d,
				  0xcb,0xf2,0x01,0x51,0x84,0xff,0xfb,0x82,
				  0xf2,0x65,0x1c,0x36 },
		.clen = 28,
	}, {

	/* #7 - NIST -
	 * [Keylen = 128]
	 * [IVlen  = 96]
	 * [PTlen  = 408]
	 * [AADlen = 160]
	 * [Taglen = 32]
	 */
		.key =		{ 0xe6,0x49,0x03,0xa7,0x7d,0x2c,0x8f,0x54,
				  0xe5,0x74,0x13,0x54,0x89,0x5f,0x9f,0x25 },
		.klen = 16,
		.iv =		{ 0x75,0xbf,0xc0,0xf3,0xc6,0xac,0x07,0x1a,
				  0xf0,0x43,0x43,0x18 },
		.ivlen = 12,
		.aad =		{ 0x41,0x6b,0x40,0xf1,0x4b,0xdb,0x9f,0x0a,
				  0xce,0xf9,0x96,0xc9,0x63,0xd2,0x3b,0xcf,
				  0x10,0xb7,0x25,0x18},
		.aadlen = 20,
		.tag =		{ 0x5c,0x52,0x6f,0x9d },
		.taglen = 32,
		.plaintext =	{ 0x19,0x56,0x1f,0x57,0xd5,0x7d,0x9a,0x96,
				  0x1b,0xbc,0x6a,0xc5,0x63,0x45,0x56,0xd0,
				  0x05,0xfa,0x60,0x10,0xd9,0x0b,0xd2,0x18,
			          0xc6,0x27,0x75,0x37,0xa4,0x3f,0x8d,0x3f,
				  0xa8,0xf2,0x9a,0x16,0xe4,0xcc,0x49,0x5b,
				  0x49,0xb8,0xaf,0x19,0x5d,0x91,0x7c,0xb7,
				  0x60,0xc3,0x4f },
		.plen = 51,
		.ciphertext =	{ 0x89,0x8a,0xbb,0x3d,0x70,0x69,0xc0,0x59,
				  0x19,0x04,0x6f,0xe4,0x8c,0xa9,0xa4,0x43,
				  0xa5,0xd2,0xbd,0x2d,0x28,0x50,0x3f,0xd0,
				  0xa2,0x71,0x6b,0x2e,0xf5,0xa1,0x75,0xf7,
				  0x48,0x68,0xf7,0x91,0x7f,0x55,0x42,0x14,
				  0x4b,0x67,0x04,0xdf,0x8a,0x42,0xcc,0x11,
				  0xc9,0x65,0xc3,0x5c,0x52,0x6f,0x9d },
		.clen = 55,
	}, {

	/* #8 - NIST -
	 * [Keylen = 192]
	 * [IVlen = 96]
	 * [PTlen = 128]
	 * [AADlen = 128]
	 * [Taglen = 128]
	 */
		.key =		{ 0x6f,0x44,0xf5,0x2c,0x2f,0x62,0xda,0xe4,
				  0xe8,0x68,0x4b,0xd2,0xbc,0x7d,0x16,0xee,
				  0x7c,0x55,0x73,0x30,0x30,0x5a,0x79,0x0d},
		.klen = 24,
		.iv =		{ 0x9a,0xe3,0x58,0x25,0xd7,0xc7,0xed,0xc9,
				  0xa3,0x9a,0x07,0x32},
		.ivlen = 12,
		.aad =		{ 0x1b,0x42,0x36,0xb8,0x46,0xfc,0x2a,0x0f,
				  0x78,0x28,0x81,0xba,0x48,0xa0,0x67,0xe9},
		.aadlen = 16,
		.tag =		{ 0x1c,0x19,0x80,0x86,0x45,0x0a,0xe1,0x83,
				  0x4d,0xd6,0xc2,0x63,0x67,0x96,0xbc,0xe2},
		.taglen = 128,
		.plaintext =	{ 0x37,0x22,0x2d,0x30,0x89,0x5e,0xb9,0x58,
				  0x84,0xbb,0xbb,0xae,0xe4,0xd9,0xca,0xe1 },
		.plen = 16,
		.ciphertext =	{ 0xa5,0x4b,0x5d,0xa3,0x3f,0xc1,0x19,0x6a,
				  0x8e,0xf3,0x1a,0x53,0x21,0xbf,0xca,0xeb,
				  0x1c,0x19,0x80,0x86,0x45,0x0a,0xe1,0x83,
				  0x4d,0xd6,0xc2,0x63,0x67,0x96,0xbc,0xe2},
		.clen = 32,
	}, {

	/* #9 - NIST -
	 * [Keylen = 256]
	 * [IVlen = 96]
	 * [PTlen = 128]
	 * [AADlen = 128]
	 * [Taglen = 128]
	 */
		.key =		{ 0x92,0xe1,0x1d,0xcd,0xaa,0x86,0x6f,0x5c,
				  0xe7,0x90,0xfd,0x24,0x50,0x1f,0x92,0x50,
				  0x9a,0xac,0xf4,0xcb,0x8b,0x13,0x39,0xd5,
				  0x0c,0x9c,0x12,0x40,0x93,0x5d,0xd0,0x8b},
		.klen = 32,
		.iv =		{ 0xac,0x93,0xa1,0xa6,0x14,0x52,0x99,0xbd,
				  0xe9,0x02,0xf2,0x1a},
		.ivlen = 12,
		.aad =		{ 0x1e,0x08,0x89,0x01,0x6f,0x67,0x60,0x1c,
				  0x8e,0xbe,0xa4,0x94,0x3b,0xc2,0x3a,0xd6},
		.aadlen = 16,
		.tag =		{ 0xec,0xa5,0xaa,0x77,0xd5,0x1d,0x4a,0x0a,
				  0x14,0xd9,0xc5,0x1e,0x1d,0xa4,0x74,0xab},
		.taglen = 128,
		.plaintext =	{ 0x2d,0x71,0xbc,0xfa,0x91,0x4e,0x4a,0xc0,
				  0x45,0xb2,0xaa,0x60,0x95,0x5f,0xad,0x24},
		.plen = 16,
		.ciphertext =	{ 0x89,0x95,0xae,0x2e,0x6d,0xf3,0xdb,0xf9,
				  0x6f,0xac,0x7b,0x71,0x37,0xba,0xe6,0x7f,
				  0xec,0xa5,0xaa,0x77,0xd5,0x1d,0x4a,0x0a,
				  0x14,0xd9,0xc5,0x1e,0x1d,0xa4,0x74,0xab},
		.clen = 32,
	},
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cfb8_tv[] = {
	{
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d },
		.plen = 18,
		.ciphertext =	{ 0x3b,0x79,0x42,0x4c,0x9c,0x0d,0xd4,0x36,
				  0xba,0xce,0x9e,0x0e,0xd4,0x58,0x6a,0x4f,
				  0x32,0xb9 },
		.clen = 18,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
	},{
		.key =		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d },
		.plen = 18,
		.num_chunks_plain =	2,
		.chunks_plain =	{ 8, 10 },
		.ciphertext =	{ 0xcd,0xa2,0x52,0x1e,0xf0,0xa9,0x05,0xca,
				  0x44,0xcd,0x05,0x7c,0xbf,0x0d,0x47,0xa0,
				  0x67,0x8a },
		.clen = 18,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph =	2,
		.chunks_ciph =	{ 8, 10 },
	},{
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d },
		.plen = 18,
		.num_chunks_plain =	5,
		.chunks_plain = 	{ 5, 0, 9, -1, 4 },
		.ciphertext =	{ 0xdc,0x1f,0x1a,0x85,0x20,0xa6,0x4d,0xb5,
				  0x5f,0xcc,0x8a,0xc5,0x54,0x84,0x4e,0x88,
				  0x97,0x00 },
		.clen = 18,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph =	5,
		.chunks_ciph = 	{ 5, 0, 9, -1, 4 },
	}
};
		
/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cfb128_tv[] = {
	{	// #0
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.ciphertext =	{ 0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
				  0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
				  0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
				  0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b,
				  0x26,0x75,0x1f,0x67,0xa3,0xcb,0xb1,0x40,
				  0xb1,0x80,0x8c,0xf1,0x87,0xa4,0xf4,0xdf,
				  0xc0,0x4b,0x05,0x35,0x7c,0x5d,0x1c,0x0e,
				  0xea,0xc4,0xc6,0x6f,0x9f,0xf7,0xf2,0xe6 },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
	},{	// #2
		.key =		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 4,
		.chunks_plain = 	{ 16, 16, 16, 16 },
		.ciphertext =	{ 0xcd,0xc8,0x0d,0x6f,0xdd,0xf1,0x8c,0xab,
				  0x34,0xc2,0x59,0x09,0xc9,0x9a,0x41,0x74,
				  0x67,0xce,0x7f,0x7f,0x81,0x17,0x36,0x21,
				  0x96,0x1a,0x2b,0x70,0x17,0x1d,0x3d,0x7a,
				  0x2e,0x1e,0x8a,0x1d,0xd5,0x9b,0x88,0xb1,
				  0xc8,0xe6,0x0f,0xed,0x1e,0xfa,0xc4,0xc9,
				  0xc0,0x5f,0x9f,0x9c,0xa9,0x83,0x4f,0xa0,
				  0x42,0xae,0x8f,0xba,0x58,0x4b,0x09,0xff },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 4,
		.chunks_ciph = 	{ 16, 16, 16, 16 },
	},{	// #3
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain = 	{ 25, 0, 8, -1, 31 },
		.ciphertext =	{ 0xdc,0x7e,0x84,0xbf,0xda,0x79,0x16,0x4b,
				  0x7e,0xcd,0x84,0x86,0x98,0x5d,0x38,0x60,
				  0x39,0xff,0xed,0x14,0x3b,0x28,0xb1,0xc8,
				  0x32,0x11,0x3c,0x63,0x31,0xe5,0x40,0x7b,
				  0xdf,0x10,0x13,0x24,0x15,0xe5,0x4b,0x92,
				  0xa1,0x3e,0xd0,0xa8,0x26,0x7a,0xe2,0xf9,
				  0x75,0xa3,0x85,0x74,0x1a,0xb9,0xce,0xf8,
				  0x20,0x31,0x62,0x3d,0x55,0xb1,0xe4,0x71 },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 5,
		.chunks_ciph = 	{ 25, 0, 8, -1, 31 },
	}
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ofb_tv[] = {
	{	// #0
		.key =		{ 0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
				  0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c },
		.klen = 16,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.ciphertext =	{ 0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
				  0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
				  0x77,0x89,0x50,0x8d,0x16,0x91,0x8f,0x03,
				  0xf5,0x3c,0x52,0xda,0xc5,0x4e,0xd8,0x25,
				  0x97,0x40,0x05,0x1e,0x9c,0x5f,0xec,0xf6,
				  0x43,0x44,0xf7,0xa8,0x22,0x60,0xed,0xcc,
				  0x30,0x4c,0x65,0x28,0xf6,0x59,0xc7,0x78,
				  0x66,0xa5,0x10,0xd9,0xc1,0xd6,0xae,0x5e },
		.clen = 64,	
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
	},{	// #1
		.key =		{ 0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
				  0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
				  0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b },
		.klen = 24,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 4,
		.chunks_plain = 	{ 16, 16, 16, 16 },
		.ciphertext =	{ 0xcd,0xc8,0x0d,0x6f,0xdd,0xf1,0x8c,0xab,
				  0x34,0xc2,0x59,0x09,0xc9,0x9a,0x41,0x74,
				  0xfc,0xc2,0x8b,0x8d,0x4c,0x63,0x83,0x7c,
				  0x09,0xe8,0x17,0x00,0xc1,0x10,0x04,0x01,	
				  0x8d,0x9a,0x9a,0xea,0xc0,0xf6,0x59,0x6f,
				  0x55,0x9c,0x6d,0x4d,0xaf,0x59,0xa5,0xf2,
				  0x6d,0x9f,0x20,0x08,0x57,0xca,0x6c,0x3e,
				  0x9c,0xac,0x52,0x4b,0xd9,0xac,0xc9,0x2a },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 4,
		.chunks_ciph = 	{ 16, 16, 16, 16 },
	},{	//  #2
		.key =		{ 0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
				  0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
				  0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
				  0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4 },
		.klen = 32,
		.plaintext =	{ 0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
				  0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
				  0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
				  0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
				  0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
				  0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
				  0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
				  0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10 },
		.plen = 64,
		.num_chunks_plain = 5,
		.chunks_plain = 	{ 25, 0, 8, -1, 31 },
		.ciphertext =	{ 0xdc,0x7e,0x84,0xbf,0xda,0x79,0x16,0x4b,
				  0x7e,0xcd,0x84,0x86,0x98,0x5d,0x38,0x60,
				  0x4f,0xeb,0xdc,0x67,0x40,0xd2,0x0b,0x3a,
				  0xc8,0x8f,0x6a,0xd8,0x2a,0x4f,0xb0,0x8d,
				  0x71,0xab,0x47,0xa0,0x86,0xe8,0x6e,0xed,
				  0xf3,0x9d,0x1c,0x5b,0xba,0x97,0xc4,0x08,
				  0x01,0x26,0x14,0x1d,0x67,0xf3,0x7b,0xe8,
				  0x53,0x8f,0x5a,0x8b,0xe7,0x40,0xe4,0x84 },
		.clen = 64,
		.iv =		{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
				  0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f },
		.ivlen = 16,
		.num_chunks_ciph = 5,
		.chunks_ciph = 	{ 25, 0, 8, -1, 31 },
	}
};
			
#define NUM_OF_PUBLISHED_TESTSUITES	7

struct published_test_suite_info published_test_suites[] = {
	{
		.name="AES_ECB",
		.tvcount = 3,
		.tv = aes_ecb_tv,
		.size = AES_BLOCK_SIZE, 
		.mech = {CKM_AES_ECB, 0, 0},
	}, {
		.name = "AES_CBC",
		.tvcount = 3,
		.tv = aes_cbc_tv,
		.size = AES_BLOCK_SIZE,
		.mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
	}, {
		.name = "AES_CTR",
		.tvcount = 3,
		.tv =  aes_ctr_tv,
		.size = AES_BLOCK_SIZE,
		.mech = {CKM_AES_CTR, &aesctr, sizeof(aesctr)},
	}, {
		.name = "AES_GCM",
		.tvcount = 10,
		.tv =  aes_gcm_tv,
		.size = AES_BLOCK_SIZE,
		.mech = {CKM_AES_GCM, &aesgcm, sizeof(aesgcm)},
	}, {
		.name = "AES_CFB8",
		.tvcount = 3,
		.tv =  aes_cfb8_tv,
		.size = 1,
		.mech = {CKM_AES_CFB8, &aes_iv, AES_IV_SIZE},
	}, {
		.name = "AES_CFB128",
		.tvcount = 3,
		.tv =  aes_cfb128_tv,
		.size = AES_BLOCK_SIZE,
		.mech = {CKM_AES_CFB128, &aes_iv, AES_IV_SIZE},
	}, {
		.name = "AES_OFB",
		.tvcount = 3,
		.tv =  aes_ofb_tv,
		.size = AES_BLOCK_SIZE,
		.mech = {CKM_AES_OFB, &aes_iv, AES_IV_SIZE},
	}
};

#define NUM_OF_GENERATED_TESTSUITES 4

struct generated_test_suite_info generated_test_suites[] = {
        {
                .name = "AES_ECB",
                .mech = {CKM_AES_ECB, 0, 0},
        }, {
                .name = "AES_CBC",
                .mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
        }, {
                .name = "AES_CBC_PAD",
                .mech = {CKM_AES_CBC_PAD, &aes_iv, AES_IV_SIZE},
        }, {
                .name = "AES_CTR",
                .mech = {CKM_AES_CTR, &aesctr, sizeof(aesctr)},
        }
};

#define NUM_OF_GENERATED_ERR_TESTSUITES 2

struct generated_test_suite_info generated_err_test_suites[] = {
        {
                .name = "AES_ECB",
                .mech = {CKM_AES_ECB, 0, 0},
        }, {
                .name = "AES_CBC",
                .mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
        }
};
