
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_editbox.c 2377 2007-06-28 18:30:21Z mschwerin $
 */
#include "config.h"

#include <assert.h>

#ifdef HAVE_LIBX11
#include <X11/X.h>
#include <X11/keysym.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "odk.h"
#include "odk_xk.h"
#include "otk.h"

typedef struct {
    otk_widget_t widget;

    char *text;
    char *display_text;

    int max_length;
    int cur_length;

    int cur_position;

    int first_visible;
    int num_visible;

    int carret_x;

    bool is_password;

    /**
     * This is the callback that is called whenever the content of the editbox
     * is changed.
     */
    void *change_cb_data;
    otk_editbox_cb_t change_cb;

    /**
     * This is the callback that is called when the editbox receives an
     * ACTIVATE event.
     */
    void *activate_cb_data;
    otk_editbox_cb_t activate_cb;
} otk_editbox_t;


/*
 * Returns the width of the current display text.
 */
static int
editbox_display_text_width (otk_editbox_t * editbox)
{
    int tw;
    int th;
    odk_osd_set_font (editbox->widget.odk, editbox->widget.font,
                      editbox->widget.fontsize);
    odk_get_text_size (editbox->widget.odk, editbox->display_text, &tw, &th);

    return tw;
}


static void
editbox_display_text_create (otk_editbox_t * editbox)
{
    ho_free (editbox->display_text);
    assert (editbox->first_visible + editbox->num_visible <=
            strlen (editbox->text));

    editbox->display_text = ho_malloc (editbox->num_visible + 1);

    if (editbox->is_password) {
        memset (editbox->display_text, '*', editbox->num_visible);
    }
    else {
        memcpy (editbox->display_text,
                editbox->text + editbox->first_visible, editbox->num_visible);
    }
    editbox->display_text[editbox->num_visible] = 0;
}


/* 
 * This does NOT change the first visible character. It only adjusts the
 * number of visible characters.
 */
static void
editbox_display_text_correct_length (otk_editbox_t * editbox)
{
    int mw = editbox->widget.w - 10;
    editbox->num_visible = strlen (editbox->text) - editbox->first_visible;
    editbox_display_text_create (editbox);
    while (editbox_display_text_width (editbox) > mw) {
        editbox->num_visible--;
        editbox_display_text_create (editbox);
    }
}


/*
 * This moves the visible area to the right by 5 characters if the current
 * position reaches the last visible character.
 */
static void
editbox_display_text_jump_right (otk_editbox_t * editbox)
{
    int mw = editbox->widget.w - 10;
    int last_visible = editbox->first_visible + editbox->num_visible;
    if (editbox->cur_position >= last_visible) {
        last_visible += 5;
        if (last_visible > editbox->cur_length)
            last_visible = editbox->cur_length;

        editbox->first_visible = last_visible - editbox->num_visible;
        editbox_display_text_create (editbox);
        while (editbox_display_text_width (editbox) > mw) {
            editbox->num_visible--;
            editbox->first_visible = last_visible - editbox->num_visible;
            editbox_display_text_create (editbox);
        }
    }
}


/*
 * This moves the visible area to the left by 5 characters if the current
 * position reaches the first visible character.
 */
static void
editbox_display_text_jump_left (otk_editbox_t * editbox)
{
    if (editbox->cur_position <= editbox->first_visible) {
        editbox->first_visible -= 5;
        if (editbox->first_visible < 0)
            editbox->first_visible = 0;
        editbox_display_text_correct_length (editbox);
    }
}


static void
editbox_calculate_carret_position (otk_editbox_t * editbox)
{
    /* Calculate the x position of the caret. */
    int len = editbox->cur_position - editbox->first_visible;
    char txt[len + 1];
    strncpy (txt, editbox->display_text, len);
    txt[len] = 0;
    int tw;
    int th;
    odk_osd_set_font (editbox->widget.odk, editbox->widget.font,
                      editbox->widget.fontsize);
    odk_get_text_size (editbox->widget.odk, txt, &tw, &th);
    editbox->carret_x = editbox->widget.x + 5 + tw;
    editbox->widget.need_repaint = true;
}


void
otk_editbox_is_password (otk_widget_t * this, bool is_password)
{

    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    editbox->is_password = is_password;
}


void
otk_editbox_set_text (otk_widget_t * this, const char *text)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    strncpy (editbox->text, text, editbox->max_length);

    if (strlen (text) < editbox->max_length) {
        editbox->cur_length = strlen (text);
    }
    else {
        editbox->cur_length = editbox->max_length;
    }

    editbox->cur_position = 0;
    editbox->carret_x = editbox->widget.x + 5;
    editbox->first_visible = 0;
    editbox_display_text_correct_length (editbox);

    this->need_repaint = true;
}


char *
otk_editbox_get_text (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return NULL;

    return ho_strdup (editbox->text);
}


static void
editbox_draw (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    int palette = 0;
    if (!this->is_enabled) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_DISABLED);
    }
    else if (this->is_focused) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_FOCUSED);
    }
    else {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
    }
    int background_color = palette + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = palette + OSD_TEXT_PALETTE_FOREGROUND;

    if (this->is_focused) {
        odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                       background_color, true);
    }
    {
        odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                       foreground_color, false);
    }

    odk_osd_set_font (this->odk, this->font, this->fontsize);
    odk_draw_text (this->odk, this->x + 5, this->y + (this->h / 2),
                   editbox->display_text, this->alignment, palette);

    if (this->is_focused) {
        odk_draw_line (this->odk, editbox->carret_x, this->y + 5,
                       editbox->carret_x, this->y + this->h - 5, 1,
                       foreground_color);
    }

    this->need_repaint = false;
}


static void
editbox_destroy (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    ho_free (editbox->text);
    ho_free (editbox->display_text);

    ho_free (editbox);
}


static void
editbox_key_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    switch (ev->source.key) {
    case OXINE_KEY_INPUT:
        /* Input events are handled right here. */
        break;
    case OXINE_KEY_UP:
    case OXINE_KEY_DOWN:
    case OXINE_KEY_PREV_WIDGET:
    case OXINE_KEY_NEXT_WIDGET:
    case OXINE_KEY_MENU_HELP:
    case OXINE_KEY_MENU_MAIN:
    case OXINE_KEY_MENU_MUSIC:
    case OXINE_KEY_MENU_VIDEO:
    case OXINE_KEY_MENU_IMAGE:
    case OXINE_KEY_MENU_PLAYBACK:
    case OXINE_KEY_MENU_PLAYLIST:
    case OXINE_KEY_MENU_SETTINGS:
    case OXINE_KEY_MENU_WEATHER:
    case OXINE_KEY_TELEVISION:
    case OXINE_KEY_SAVE:
    case OXINE_KEY_QUIT:
        /* These events are passed along unchanged. */
        return;
    case OXINE_KEY_ACTIVATE:
        if (editbox->activate_cb) {
            editbox->activate_cb (editbox->activate_cb_data,
                                  ho_strdup (editbox->text));
        }
    default:
        /* All other events are swallowed. */
        ev->source.key = OXINE_KEY_NULL;
        return;
    }

    int mw = editbox->widget.w - 10;
    int key = ev->data.keyboard.key;
    int modifier = ev->data.keyboard.modifier;
    modifier &= (ControlMask | Mod1Mask);

    switch (key) {
    case XK_Escape:
        ev->source.key = OXINE_KEY_BACK;
    case XK_Tab:
        return;
    }

    if ((key == XK_Delete)
        || ((modifier & ControlMask)
            && ((key == XK_d) || (key == XK_D)))) {
        if (editbox->cur_position < editbox->cur_length) {
            int i = editbox->cur_position;
            for (; i < editbox->cur_length; i++)
                editbox->text[i] = editbox->text[i + 1];
            editbox->cur_length--;
            editbox_display_text_correct_length (editbox);

            if (editbox->change_cb) {
                editbox->change_cb (editbox->change_cb_data,
                                    ho_strdup (editbox->text));
            }
        }
    }
    else if (key == XK_BackSpace) {
        if (editbox->cur_position > 0) {

            if (editbox->cur_position < editbox->cur_length) {
                int len = strlen (editbox->text);
                int i = editbox->cur_position - 1;
                for (; i < editbox->cur_length; i++)
                    editbox->text[i] = editbox->text[i + 1];
                assert (len - 1 == strlen (editbox->text));
            }
            else {
                editbox->text[editbox->cur_position - 1] = 0;
            }
            editbox->cur_position--;
            editbox->cur_length--;

            editbox->num_visible--;
            editbox_display_text_create (editbox);
            editbox_display_text_jump_left (editbox);

            if (editbox->change_cb) {
                editbox->change_cb (editbox->change_cb_data,
                                    ho_strdup (editbox->text));
            }
        }
    }
    else if ((modifier & ControlMask)
             && ((key == XK_k) || (key == XK_K))) {
        /* This deletes everything from the current 
         * position to the end of the line. */
        editbox->cur_length = editbox->cur_position;
        editbox->text[editbox->cur_length] = 0;

        editbox->num_visible = editbox->cur_position - editbox->first_visible;
        editbox_display_text_correct_length (editbox);

        if (editbox->change_cb) {
            editbox->change_cb (editbox->change_cb_data,
                                ho_strdup (editbox->text));
        }
    }
    else if ((modifier & ControlMask)
             && ((key == XK_u) || (key == XK_U))) {
        /* This deletes everything from the current 
         * position backwards to beginning of the line. */
        if (editbox->cur_position > 0) {
            int i = editbox->cur_position;
            for (; i <= editbox->cur_length; i++) {
                editbox->text[i - editbox->cur_position] = editbox->text[i];
            }
            editbox->cur_length -= editbox->cur_position;
            editbox->cur_position = 0;

            editbox->first_visible = 0;
            editbox_display_text_correct_length (editbox);

            if (editbox->change_cb) {
                editbox->change_cb (editbox->change_cb_data,
                                    ho_strdup (editbox->text));
            }
        }
    }
    else if ((modifier & ControlMask)
             && ((key == XK_w) || (key == XK_W))) {
        /* This deletes everything from the current 
         * position backwards to the next space. */
        if (editbox->cur_position > 0) {
            int s = editbox->cur_position - 1;
            while ((s >= 0) && (editbox->text[s] == ' ')) {
                s--;
            }
            while ((s >= 0) && (editbox->text[s] != ' ')) {
                s--;
            }
            s++;

            int i = editbox->cur_position;
            int j = s;
            for (; i <= editbox->cur_length; i++, j++) {
                editbox->text[j] = editbox->text[i];
            }
            editbox->cur_length -= editbox->cur_position - s;
            editbox->cur_position = s;

            if (editbox->cur_position < editbox->first_visible) {
                editbox->first_visible = editbox->cur_position;
            }
            editbox_display_text_correct_length (editbox);
            editbox_display_text_jump_left (editbox);

            if (editbox->change_cb) {
                editbox->change_cb (editbox->change_cb_data,
                                    ho_strdup (editbox->text));
            }
        }
    }
    else if ((key == XK_Left)
             || ((modifier & ControlMask)
                 && ((key == XK_b) || (key == XK_B)))) {
        if (editbox->cur_position > 0) {
            editbox->cur_position--;
            editbox_display_text_jump_left (editbox);
        }
    }
    else if ((key == XK_Right)
             || ((modifier & ControlMask)
                 && ((key == XK_f) || (key == XK_F)))) {
        if (editbox->cur_position < editbox->cur_length) {
            editbox->cur_position++;
            editbox_display_text_jump_right (editbox);
        }
    }
    else if ((key == XK_Home)
             || ((modifier & ControlMask)
                 && ((key == XK_a) || (key == XK_A)))) {
        editbox->cur_position = 0;

        editbox->first_visible = 0;
        editbox_display_text_correct_length (editbox);
    }
    else if ((key == XK_End)
             || ((modifier & ControlMask)
                 && ((key == XK_e) || (key == XK_E)))) {
        editbox->cur_position = editbox->cur_length;

        editbox->first_visible = 0;
        editbox->num_visible = strlen (editbox->text);
        editbox_display_text_create (editbox);
        while (editbox_display_text_width (editbox) > mw) {
            editbox->num_visible--;
            editbox->first_visible =
                editbox->cur_length - editbox->num_visible;
            editbox_display_text_create (editbox);
        }
    }
    else if (modifier == 0) {
        if (editbox->cur_length < editbox->max_length) {

            int append = (editbox->cur_position == editbox->cur_length);
            if (append) {
                editbox->text[editbox->cur_length] = key;
            }
            else {
                int i = editbox->cur_length;
                for (; i > editbox->cur_position; i--) {
                    editbox->text[i] = editbox->text[i - 1];
                }
                editbox->text[editbox->cur_position] = key;
            }
            editbox->cur_length++;
            editbox->cur_position++;
            editbox->text[editbox->cur_length] = 0;
            if (append) {
                editbox->num_visible++;
                editbox_display_text_create (editbox);
                while (editbox_display_text_width (editbox) > mw) {
                    editbox->first_visible++;
                    editbox->num_visible--;
                    editbox_display_text_create (editbox);
                }
            }
            else {
                editbox_display_text_correct_length (editbox);
                editbox_display_text_jump_right (editbox);
            }

            if (editbox->change_cb) {
                editbox->change_cb (editbox->change_cb_data,
                                    ho_strdup (editbox->text));
            }
        }
    }

#ifdef DEBUG
    assert (editbox->cur_position >= editbox->first_visible);
    assert (editbox->cur_position <=
            editbox->first_visible + editbox->num_visible);
    assert (editbox->num_visible == strlen (editbox->display_text));

    assert (editbox->cur_position >= 0);
    assert (editbox->cur_position <= editbox->cur_length);
    assert (editbox->cur_length == strlen (editbox->text));
#endif

    editbox_calculate_carret_position (editbox);

    otk_draw (editbox->widget.otk);
}


static void
editbox_button_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_EDITBOX))
        return;

    oxine_button_id_t saved_button = ev->source.button;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;

    switch (saved_button) {
    case OXINE_MOUSE_BUTTON_LEFT:
        odk_osd_set_font (editbox->widget.odk, editbox->widget.font,
                          editbox->widget.fontsize);

        if (editbox->carret_x > ev->data.mouse.pos.x) {
            while ((editbox->cur_position > 0)
                   && (editbox->carret_x > ev->data.mouse.pos.x)) {
                editbox->cur_position--;
                editbox_calculate_carret_position (editbox);
            }
        }

        else {
            while ((editbox->cur_position <
                    editbox->first_visible + editbox->num_visible)
                   && (editbox->carret_x < ev->data.mouse.pos.x)) {
                editbox->cur_position++;
                editbox_calculate_carret_position (editbox);
            }
        }
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.button = saved_button;
        break;
    }
}


otk_widget_t *
otk_editbox_new (otk_t * otk, int x, int y, int w, int h,
                 int max_length,
                 otk_editbox_cb_t change_cb, void *change_cb_data,
                 otk_editbox_cb_t activate_cb, void *activate_cb_data)
{
    otk_editbox_t *editbox = ho_new (otk_editbox_t);

    otk_widget_constructor ((otk_widget_t *) editbox, otk,
                            OTK_WIDGET_EDITBOX, x, y, w, h);

    editbox->widget.draw = editbox_draw;
    editbox->widget.destroy = editbox_destroy;

    editbox->widget.key_handler = editbox_key_handler;
    editbox->widget.button_handler = editbox_button_handler;

    editbox->text = ho_malloc (max_length + 1);
    editbox->display_text = ho_strdup ("");

    editbox->max_length = max_length;
    editbox->cur_length = 0;

    editbox->cur_position = 0;

    editbox->first_visible = 0;
    editbox->num_visible = 0;
    editbox->carret_x = editbox->widget.x + 5;

    editbox->change_cb = change_cb;
    editbox->change_cb_data = change_cb_data;
    editbox->activate_cb = activate_cb;
    editbox->activate_cb_data = activate_cb_data;

    return (otk_widget_t *) editbox;
}
