// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package repos

import (
	"context"
	"encoding/json"
	"io/ioutil"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/core"
	"github.com/sirupsen/logrus"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func init() {
	logrus.SetOutput(ioutil.Discard)
}

var (
	mockRepo = &core.Repository{
		ID:        1,
		Namespace: "octocat",
		Name:      "hello-world",
		Slug:      "octocat/hello-world",
		Counter:   42,
		Branch:    "master",
	}

	mockRepos = []*core.Repository{
		{
			ID:        1,
			Namespace: "octocat",
			Name:      "hello-world",
			Slug:      "octocat/hello-world",
		},
		{
			ID:        1,
			Namespace: "octocat",
			Name:      "spoon-knife",
			Slug:      "octocat/spoon-knife",
		},
	}
)

func TestFind(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/api/repos/octocat/hello-world", nil)
	r = r.WithContext(request.WithRepo(
		context.Background(), mockRepo,
	))

	router := chi.NewRouter()
	router.Get("/api/repos/{owner}/{name}", HandleFind())
	router.ServeHTTP(w, r)

	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(core.Repository), mockRepo
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
