// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package config

import (
	"context"
	"errors"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
)

var noContext = context.TODO()

var mockFile = []byte(`
kind: pipeline
name: default

steps: []
`)

func TestRepository(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConfigArgs{
		User:   &core.User{Login: "octocat"},
		Repo:   &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build:  &core.Build{After: "6d144de7"},
		Config: nil,
	}

	resp := &core.File{Data: mockFile}

	files := mock.NewMockFileService(controller)
	files.EXPECT().Find(noContext, args.User, args.Repo.Slug, args.Build.After, args.Build.Ref, args.Repo.Config).Return(resp, nil)

	service := Repository(files)
	result, err := service.Find(noContext, args)
	if err != nil {
		t.Error(err)
	}

	if result.Data != string(resp.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestRepositoryErr(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConfigArgs{
		User:   &core.User{Login: "octocat"},
		Repo:   &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build:  &core.Build{After: "6d144de7"},
		Config: nil,
	}

	resp := errors.New("")

	files := mock.NewMockFileService(controller)
	files.EXPECT().Find(noContext, args.User, args.Repo.Slug, args.Build.After, args.Build.Ref, args.Repo.Config).Return(nil, resp)

	service := Repository(files)
	_, err := service.Find(noContext, args)
	if err != resp {
		t.Errorf("expect error returned from file service")
	}
}
