// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package converter

import (
	"context"
	"errors"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
)

var noContext = context.Background()

var mockFile = `
kind: pipeline
type: docker
name: testing
`

func TestCombine(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		User:   &core.User{Login: "octocat"},
		Repo:   &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build:  &core.Build{After: "6d144de7"},
		Config: &core.Config{},
	}

	resp := &core.Config{Data: string(mockFile)}

	service := mock.NewMockConvertService(controller)
	service.EXPECT().Convert(noContext, args).Return(resp, nil)

	result, err := Combine(false, service).Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result.Data != string(resp.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestCombineErr(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	resp := errors.New("")
	service := mock.NewMockConvertService(controller)
	service.EXPECT().Convert(noContext, nil).Return(nil, resp)

	_, err := Combine(false, service).Convert(noContext, nil)
	if err != resp {
		t.Errorf("expected convert service error")
	}
}

func TestCombineNoConfig(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		User:  &core.User{Login: "octocat"},
		Repo:  &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build: &core.Build{After: "6d144de7"},
	}

	resp := &core.Config{Data: string(mockFile)}

	service1 := mock.NewMockConvertService(controller)
	service1.EXPECT().Convert(noContext, args).Return(nil, nil)

	service2 := mock.NewMockConvertService(controller)
	service2.EXPECT().Convert(noContext, args).Return(&core.Config{}, nil)

	service3 := mock.NewMockConvertService(controller)
	service3.EXPECT().Convert(noContext, args).Return(resp, nil)

	result, err := Combine(false, service1, service2, service3).Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result.Data != string(resp.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestCombineEmptyConfig(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		User:   &core.User{Login: "octocat"},
		Repo:   &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build:  &core.Build{After: "6d144de7"},
		Config: &core.Config{Data: string(mockFile)},
	}

	service1 := mock.NewMockConvertService(controller)
	service1.EXPECT().Convert(noContext, args).Return(nil, nil)

	result, err := Combine(false, service1).Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result != args.Config {
		t.Errorf("unexpected file contents")
	}
}
