// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package status

import (
	"context"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"
	"github.com/drone/drone/mock/mockscm"
	"github.com/drone/go-scm/scm"

	"github.com/golang/mock/gomock"
)

var noContext = context.Background()

func TestStatus(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{}

	mockRenewer := mock.NewMockRenewer(controller)
	mockRenewer.EXPECT().Renew(gomock.Any(), mockUser, false).Return(nil)

	statusInput := &scm.StatusInput{
		Title:  "Build #1",
		State:  scm.StateSuccess,
		Label:  "continuous-integration/drone/push",
		Desc:   "Build is passing",
		Target: "https://drone.company.com/octocat/hello-world/1",
	}

	mockRepos := mockscm.NewMockRepositoryService(controller)
	mockRepos.EXPECT().CreateStatus(gomock.Any(), "octocat/hello-world", "a6586b3db244fb6b1198f2b25c213ded5b44f9fa", statusInput).Return(nil, nil, nil)

	client := new(scm.Client)
	client.Repositories = mockRepos

	service := New(client, mockRenewer, Config{Base: "https://drone.company.com"})
	err := service.Send(noContext, mockUser, &core.StatusInput{
		Repo: &core.Repository{Slug: "octocat/hello-world"},
		Build: &core.Build{
			Number: 1,
			Event:  core.EventPush,
			Status: core.StatusPassing,
			After:  "a6586b3db244fb6b1198f2b25c213ded5b44f9fa",
		},
	})
	if err != nil {
		t.Error(err)
	}
}

func TestStatus_ErrNotSupported(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{}

	mockRenewer := mock.NewMockRenewer(controller)
	mockRenewer.EXPECT().Renew(gomock.Any(), mockUser, false).Return(nil)

	statusInput := &scm.StatusInput{
		Title:  "Build #1",
		State:  scm.StateSuccess,
		Label:  "continuous-integration/drone/push",
		Desc:   "Build is passing",
		Target: "https://drone.company.com/octocat/hello-world/1",
	}

	mockRepos := mockscm.NewMockRepositoryService(controller)
	mockRepos.EXPECT().CreateStatus(gomock.Any(), "octocat/hello-world", "a6586b3db244fb6b1198f2b25c213ded5b44f9fa", statusInput).Return(nil, nil, scm.ErrNotSupported)

	client := new(scm.Client)
	client.Repositories = mockRepos

	service := New(client, mockRenewer, Config{Base: "https://drone.company.com"})
	err := service.Send(noContext, mockUser, &core.StatusInput{
		Repo: &core.Repository{Slug: "octocat/hello-world"},
		Build: &core.Build{
			Number: 1,
			Event:  core.EventPush,
			Status: core.StatusPassing,
			After:  "a6586b3db244fb6b1198f2b25c213ded5b44f9fa",
		},
	})
	if err != nil {
		t.Error(err)
	}
}

func TestStatus_RenewalError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{}

	mockRenewer := mock.NewMockRenewer(controller)
	mockRenewer.EXPECT().Renew(gomock.Any(), mockUser, false).Return(scm.ErrNotAuthorized)

	service := New(nil, mockRenewer, Config{Base: "https://drone.company.com"})
	err := service.Send(noContext, mockUser, &core.StatusInput{Build: &core.Build{}})
	if err == nil {
		t.Errorf("Expect error refreshing token")
	}
}

func TestStatus_Disabled(t *testing.T) {
	service := New(nil, nil, Config{Disabled: true})
	err := service.Send(noContext, nil, nil)
	if err != nil {
		t.Error(err)
	}
}
