//
// jja: swiss army knife for chess file formats
// src/main.rs: Error handling
//
// Copyright (c) 2023 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0-or-later

use crate::tr;

/// Error returned when Zobrist hash collisions are detected during PGN restore to JJA database.
#[derive(Debug)]
pub struct HashCollision {
    text: String,
}

impl HashCollision {
    /// Creates a new `HashCollision` error with the given number of detected collisions.
    pub fn new(count: usize) -> HashCollision {
        HashCollision {
            text: tr!("{} hash collisions detected in JJA database!", count),
        }
    }

    fn description(&self) -> &str {
        &self.text
    }
}

impl std::fmt::Display for HashCollision {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(f, "{}", self.description())
    }
}

impl std::error::Error for HashCollision {
    fn description(&self) -> &str {
        self.description()
    }

    fn cause(&self) -> Option<&dyn std::error::Error> {
        // Generic error, underlying cause isn't tracked.
        None
    }
}
