/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef INTERFACE_H
#define INTERFACE_H

#include <iostream>
#include <fstream>

#include "model.h"
#include "ali.h"
#include "rate.h"
#include "outstream.h"
#include "ptnls.h"
#include "model.h"
#include "urtree.h"

#ifdef PARALLEL
#	include <mpi.h>
#endif // PARALLEL

const int PROG_ALL            = 0;
const int PROG_COMPUTE_MATRIX = 1;
const int PROG_PARM_ESTIMATE  = 2;

extern int program_mode;
extern bool load_check_point;
extern bool major_step_test;
extern bool nni_test;
extern int MAX_IT_ALL_PAM_OPT;
extern int omp_threads;

const MODEL DEFAULT_SUBST_MODEL = HKY85;

const char separator_str[] = "******************************************************************************";

/**
	parameters which will be recorded in the checkpoint file, to recover after interruption.
*/
struct CommonParameters {

	/**
		number of iterations
	*/
    int        nIter;


	/**
		maximum number of iterations
	*/
    int        max_nIter;

	/**
		current iterations
	*/
	int cur_nIter;

	/**
		probability of deleting sequences
	*/
    double     probDel;

	/**
		number of representatives
	*/
    int        nRep;

	/**
		stopping rule: YES, YES_MIN_ITER, YES_MAX_ITER, NO
	*/
    STOPPING_TYPE     stoppingRule;

	/**
		type of data: NUCLEOTIDE, AMINO_ACID, CODON
	*/
    DATA_TYPE  dataType;

	/**
		type of model: HKY85, TN93, GTR, WAG, JTT,...
	*/
    MODEL      model;

	/**
		frequencies of bases
	*/
    double      baseA, baseC, baseG, baseT;

	/**
		type of estimating frequencies: ESTIMATE, USER_DEFINED, EQUAL,...
	*/
    PAM_TYPE    baseFrqType;

	/**
		transition/transvesion ratio
	*/
    double      tsTvRatio;

	/**
	*/
    PAM_TYPE    tsTvRatioType;

	/**
		pyridimine/purine ratio
	*/
    double      pyPuRatio;

	/**
		pyridimine/purine ratio type
	*/
    PAM_TYPE    pyPuRatioType;


	/**
		transition frequency A<->G

	*/
    double      tsAG;
	/**
		transition frequency C<->T
	*/
    double      tsCT;
	/**
		transversion frequency A<->C
	*/
    double      tvAC;
	/**
		transversion frequency A<->T
	*/
    double      tvAT;
	/**
		transversion frequency C<->G
	*/
    double      tvCG;
	/**
		transversion frequency G<->T
	*/
    double      tvGT;

	/**
		type of estimating parameters for GTR
	*/
    PAM_TYPE    genPamType;

	/**
		rate type: UNIFORM, SITE_SPECIFIC, GAMMA
	*/
    RATE_TYPE   rateType;

	/**
		number of rates
	*/
    int         nRate;

	/**
		gamme shape 
	*/
    double      gammaShape;

	/**
		type of estimating gamma shape: ESTIMATE, USER_DEFINED
	*/
    PAM_TYPE    gammaShapeType;

	/**
		proportion of invariable sites
	*/
    double 		prob_invar;

	/**
		type of estimating proportion of invariable sites: ESTIMATE, USER_DEFINED
	*/
    PAM_TYPE   	prob_invar_type;

	/**
		true if using a number of invariable sites
	*/
    //bool 		invariable_site;
	
	/**
		codon model
	*/
	MODEL 		codon_model;

	/**
		type of estimating codon frequencies: ESTIMATE, USER_DEFINED, EQUAL, F1x4, F3x4, CODON_TABLE
	*/
    PAM_TYPE    codonFrqType;

	/**
		transition/transvesion ratio for codon model
	*/
    double      codon_tsTvRatio;

	/**
		type of estimating transition/transvesion ratio for codon model
	*/
    PAM_TYPE    codon_tsTvRatioType;

	/**
		type of non-synonymous/synonymous ratio
	*/
    NSSITE_TYPE nsSy_ratio_type;

	/**
		number of Ns/Sy classes
	*/
    int        	nsSy_classes;

	/**
		vector of non-synonymous/synonymous ratios for all classes
	*/
	double 		nsSy_ratio_val[MAX_NUM_CLASS];

	/**
		vector of non-synonymous/synonymous proportion for all classes
	*/
	double 		nsSy_prob_val[MAX_NUM_CLASS];

	/**
		index of outgroup sequence
	*/
    int         outGrpSeqNo;

	/**
		2 for BRENT method to optimize parameters
		1 for Simulated Annealing + BFGS
		0 for BFGS algorithm only
	*/
	int 		pam_brent; 

	/**
		rate file name if load model from file
	*/
	string rate_file;

	/**
		true if all pair (AP) method for site-specific rate
	*/
	bool ap_sitespec;


	/**
		the number of bootstrap samples to create from the original alignment
	*/
	int gen_bootstrap;

	/**
		the current number of bootstrap samples 
	*/
	int cur_bootstrap;


	/**
		true if wanting to output the pattern frequencies and log-likelihood to a file
	*/
	bool print_pat_logl;

	/**
		true if wanting to output the pattern frequencies and log-likelihood to a file
	*/
	bool print_site_logl;

	/**
		print all splits of the bootstrap analysis
	*/
	bool print_splits;

	/**
		true if going to build a consensus tree from all intermediate trees
	*/
	bool build_consensus;

	/**
		current best log likelihood
	*/
	double bestLogLi;

	/**
		elapsed time of the run
	*/
	double progTime;

};

/**
	other parameters which will not be recorded in checkpoint file.
*/
struct InputParameters {
	/**
		the output prefix
	*/
	char *out_prefix;

	/**
		param file, in case of not wanting to commands from standard input
	*/
	char *param_file;

	/**
		seed for random number generator
	*/
	int random_seed;
	
	/**
		TRUE if inputing parameters by  the text menu (default).
		FALSE otherwise
	*/
	bool text_menu;

	/**
		alignment file name
	*/
	char ali_file[FILENAME_LEN];

	/**
		user tree file name
	*/
	char tree_file[FILENAME_LEN];

	/**
		TRUE to record the log-likelihood improvement of fastNNI for all IQPNNI iterations
	*/
	bool nni_lh_record;
};

/**
	initialize Parameters which will recorded in checkpoint file
	@param params Parameters
*/
void initializeParams(CommonParameters &params);

/**
	initialize input Parameters which are not recorded in checkpoint file
	@param in_pam Parameters
*/
void initInputParams(InputParameters &in_pam);

/**
	menu-driven interface
*/
class Interface {

public :

	/**
		estimate the number of required iterations at the beginning
	*/
	int cmpnIter();

	/**
		estimate the probability of deleting leaves
	*/
	double cmpProbSel (PAM_TYPE sourceType);

	/**
	*/
	void getIntNumber (istream &fpin, const char *msg, double lowerBound, double upperBound, int &value);

	/**
	*/
	void getDoubleNumber (istream &fpin, const char *msg, double lowerBound, double upperBound, double &value);

	/**
	*/
	//=====================================================================================
	//=====================================================================================
	int getOutGrpSeq (istream &fpin);

	/*=========================================================================================
	  Get the parameter by interactive interface
	//=========================================================================================*/
	/**
	*/
	int getParameter (istream &fpin, CommonParameters &params, InputParameters &in_pam);

	/**
	*/
	int getCommandLine( int argc, char *argv[], CommonParameters &params, InputParameters &in_pam);

	//====================================================================================
	//ab:cccc:dd: X
	/**
	*/
	int getIntNum (const char  *aLine);

	//====================================================================================
	//ab:cccc:dd: X
	/**
	*/
	double getDoubleNum (const char  *aLine);

	//====================================================================================
	//====================================================================================
	//get all parameters from the last stopped point
	/**
	*/
	int getPamCheckPoint (const char *checkPointFileName, CommonParameters &params);

	//====================================================================================
	/**
	*/
	//====================================================================================
	int isExistedFile (const char *fileName);

	/*====================================================================================
	 
	Interactive interface to get the user paramemters
	 
	====================================================================================*/
	/**
	*/
	int getUserPam ( int argc, char *argv[], CommonParameters &params, InputParameters &in_pam);

	/*****************************************************************
	Print the usage dialog  
	****************************************************************/
	/**
	*/
	void usage();

	//=================================================================
	//=================================================================
	/**
	*/
	void writeSpecificRate (const char *rateFileName);

	//=====================================================================================================
	//=====================================================================================================
	//=====================================================================================================
	/**
	*/
	void writeOut (CommonParameters params, InputParameters in_pam, UrTree &tree,
	               Vec<char> &conTree, const char *startedDate);


private:
	/**
	*/
	int nSeq_;
};

void writePamCheckPoint (CommonParameters &params, const char  *checkPointFileName,                               int         isFinished);


#endif

