// Copyright (C) 2018-2019 Robin Krahl <robin.krahl@ireas.org>
// SPDX-License-Identifier: MIT

use crate::device::{Device, DeviceWrapper, Librem, Pro, Storage};
use crate::error::{CommandError, Error, LibraryError};
use crate::util::{get_command_result, get_cstring, get_last_error, result_from_string};

const SLOT_COUNT: u8 = 16;

/// A password safe on a Nitrokey device.
///
/// The password safe stores a tuple consisting of a name, a login and a password on a slot.  Use
/// [`get_slot_count`][] to access the number of available slots.  The slots are addressed starting
/// with zero.  To retrieve a password safe from a Nitrokey device, use the [`get_password_safe`][]
/// method from the [`GetPasswordSafe`][] trait.  Note that the device must live at least as long
/// as the password safe.
///
/// Once the password safe has been unlocked, it can be accessed without a password.  Therefore it
/// is mandatory to call [`lock`][] on the corresponding device after the password store is used.
/// As this command may have side effects on the Nitrokey Storage, it cannot be called
/// automatically once the password safe is destroyed.
///
/// # Examples
///
/// Open a password safe and access a password:
///
/// ```no_run
/// use nitrokey::{Device, GetPasswordSafe, PasswordSafe};
/// # use nitrokey::Error;
///
/// fn use_password_safe(pws: &PasswordSafe) -> Result<(), Error> {
///     let slot = pws.get_slot(0)?;
///     println!("Credentials for {}:", slot.get_name()?);
///     println!("login:    {}", slot.get_login()?);
///     println!("password: {}", slot.get_password()?);
///     Ok(())
/// }
///
/// # fn try_main() -> Result<(), Error> {
/// let mut manager = nitrokey::take()?;
/// let mut device = manager.connect()?;
/// let pws = device.get_password_safe("123456")?;
/// use_password_safe(&pws);
/// drop(pws);
/// device.lock()?;
/// #     Ok(())
/// # }
/// ```
///
/// [`get_slot_count`]: #method.get_slot_count
/// [`get_password_safe`]: trait.GetPasswordSafe.html#method.get_password_safe
/// [`lock`]: trait.Device.html#method.lock
/// [`GetPasswordSafe`]: trait.GetPasswordSafe.html
#[derive(Debug)]
pub struct PasswordSafe<'a, 'b> {
    _device: &'a dyn Device<'b>,
}

/// A slot of a [`PasswordSafe`][].
///
/// [`PasswordSafe`]: struct.PasswordSafe.html
#[derive(Clone, Copy, Debug)]
pub struct PasswordSlot<'p, 'a, 'b> {
    slot: u8,
    _pws: &'p PasswordSafe<'a, 'b>,
}

/// Provides access to a [`PasswordSafe`][].
///
/// The device that implements this trait must always live at least as long as a password safe
/// retrieved from it.
///
/// [`PasswordSafe`]: struct.PasswordSafe.html
pub trait GetPasswordSafe<'a> {
    /// Enables and returns the password safe.
    ///
    /// The underlying device must always live at least as long as a password safe retrieved from
    /// it.  It is mandatory to lock the underlying device using [`lock`][] after the password safe
    /// has been used.  Otherwise, other applications can access the password store without
    /// authentication.
    ///
    /// If this method returns an `AesDecryptionFailed` (Nitrokey Pro) or `Unknown` (Nitrokey
    /// Storage) error, the AES data object on the smart card could not be accessed.  This problem
    /// occurs after a factory reset using `gpg --card-edit` and can be fixed using the
    /// [`Device::build_aes_key`][] command.
    ///
    /// # Errors
    ///
    /// - [`AesDecryptionFailed`][] if the secret for the password safe could not be decrypted
    ///   (Nitrokey Pro only)
    /// - [`InvalidString`][] if one of the provided passwords contains a null byte
    /// - [`Unknown`][] if the secret for the password safe could not be decrypted (Nitrokey
    ///   Storage only)
    /// - [`WrongPassword`][] if the current user password is wrong
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::{Device, GetPasswordSafe, PasswordSafe};
    /// # use nitrokey::Error;
    ///
    /// fn use_password_safe(pws: &PasswordSafe) {}
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// match device.get_password_safe("123456") {
    ///     Ok(pws) => {
    ///         use_password_safe(&pws);
    ///     },
    ///     Err(err) => eprintln!("Could not open the password safe: {}", err),
    /// };
    /// device.lock()?;
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`device`]: struct.PasswordSafe.html#method.device
    /// [`lock`]: trait.Device.html#method.lock
    /// [`AesDecryptionFailed`]: enum.CommandError.html#variant.AesDecryptionFailed
    /// [`Device::build_aes_key`]: trait.Device.html#method.build_aes_key
    /// [`InvalidString`]: enum.LibraryError.html#variant.InvalidString
    /// [`Unknown`]: enum.CommandError.html#variant.Unknown
    /// [`WrongPassword`]: enum.CommandError.html#variant.WrongPassword
    fn get_password_safe(&mut self, user_pin: &str) -> Result<PasswordSafe<'_, 'a>, Error>;
}

fn get_password_safe<'a, 'b>(
    device: &'a dyn Device<'b>,
    user_pin: &str,
) -> Result<PasswordSafe<'a, 'b>, Error> {
    let user_pin_string = get_cstring(user_pin)?;
    get_command_result(unsafe { nitrokey_sys::NK_enable_password_safe(user_pin_string.as_ptr()) })
        .map(|_| PasswordSafe { _device: device })
}

fn get_pws_result(s: String) -> Result<String, Error> {
    if s.is_empty() {
        Err(CommandError::SlotNotProgrammed.into())
    } else {
        Ok(s)
    }
}

impl<'a, 'b> PasswordSafe<'a, 'b> {
    /// Returns the number of slots in this password safe.
    pub fn get_slot_count(&self) -> u8 {
        SLOT_COUNT
    }

    /// Returns the status of all password slots.
    ///
    /// The status indicates whether a slot is programmed or not.
    ///
    /// This method is deprecated.  Use [`get_slots`][] instead.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// pws.get_slot_status()?.iter().enumerate().for_each(|(slot, programmed)| {
    ///     let status = match *programmed {
    ///         true => "programmed",
    ///         false => "not programmed",
    ///     };
    ///     println!("Slot {}: {}", slot, status);
    /// });
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`get_slots`]: #method.get_slots
    #[deprecated(since = "0.9.0", note = "Use get_slots() instead")]
    pub fn get_slot_status(&self) -> Result<[bool; 16], Error> {
        let status_ptr = unsafe { nitrokey_sys::NK_get_password_safe_slot_status() };
        if status_ptr.is_null() {
            return Err(get_last_error());
        }
        let status_array_ptr = status_ptr as *const [u8; SLOT_COUNT as usize];
        let status_array = unsafe { *status_array_ptr };
        let mut result = [false; SLOT_COUNT as usize];
        for i in 0..SLOT_COUNT {
            result[i as usize] = status_array[i as usize] == 1;
        }
        unsafe {
            nitrokey_sys::NK_free_password_safe_slot_status(status_ptr);
        }
        Ok(result)
    }

    /// Returns all password slots.
    ///
    /// The returned vector contains one element per password slot.  If the slot is not programmed,
    /// the element is `None`.  Otherwise it is a [`PasswordSlot`][] instance for the slot.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// for (idx, slot) in pws.get_slots()?.iter().enumerate() {
    ///     let status = match *slot {
    ///         Some(_) => "programmed",
    ///         None => "not programmed",
    ///     };
    ///     println!("Slot {}: {}", idx, status);
    /// }
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`PasswordSlot`]: struct.PasswordSlot.html
    pub fn get_slots(&self) -> Result<Vec<Option<PasswordSlot<'_, 'a, 'b>>>, Error> {
        let mut slots = Vec::new();
        let status_ptr = unsafe { nitrokey_sys::NK_get_password_safe_slot_status() };
        if status_ptr.is_null() {
            return Err(get_last_error());
        }
        let status_array_ptr = status_ptr as *const [u8; SLOT_COUNT as usize];
        let status_array = unsafe { *status_array_ptr };
        for slot in 0..SLOT_COUNT {
            if status_array[usize::from(slot)] == 1 {
                slots.push(Some(PasswordSlot { slot, _pws: self }));
            } else {
                slots.push(None);
            }
        }
        unsafe {
            nitrokey_sys::NK_free_password_safe_slot_status(status_ptr);
        }
        Ok(slots)
    }

    /// Returns the slot with the given index if the slot is programmed.
    ///
    /// This method uses [`get_slots`][] to check whether the slot is programmed.  To access the
    /// slot without checking whether it is programmed, use [`get_slot_unchecked`][].
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    /// - [`SlotNotProgrammed`][] if the slot is not programmed
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// let slot = pws.get_slot(0)?;
    /// let name = slot.get_name()?;
    /// let login = slot.get_login()?;
    /// let password = slot.get_password()?;
    /// println!("Credentials for {}: login {}, password {}", name, login, password);
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`SlotNotProgrammed`]: enum.CommandError.html#variant.SlotNotProgrammed
    /// [`get_slot_unchecked`]: #method.get_slot_unchecked
    /// [`get_slots`]: #method.get_slots
    pub fn get_slot(&self, slot: u8) -> Result<PasswordSlot<'_, 'a, 'b>, Error> {
        let slot = usize::from(slot);
        let slots = self.get_slots()?;
        if slot < slots.len() {
            slots[slot].ok_or_else(|| CommandError::SlotNotProgrammed.into())
        } else {
            Err(LibraryError::InvalidSlot.into())
        }
    }

    /// Returns the slot with the given index without checking whether it is programmed.
    ///
    /// To check whether a slot is programmed, use [`get_slots`][] or [`get_slot`][].
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// let slot = pws.get_slot_unchecked(0)?;
    /// let name = slot.get_name()?;
    /// let login = slot.get_login()?;
    /// let password = slot.get_password()?;
    /// println!("Credentials for {}: login {}, password {}", name, login, password);
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`get_slot`]: #method.get_slot
    /// [`get_slots`]: #method.get_slots
    pub fn get_slot_unchecked(&self, slot: u8) -> Result<PasswordSlot<'_, 'a, 'b>, Error> {
        if slot < self.get_slot_count() {
            Ok(PasswordSlot { slot, _pws: self })
        } else {
            Err(LibraryError::InvalidSlot.into())
        }
    }

    /// Returns the name of the given slot (if it is programmed).
    ///
    /// This method also returns a `SlotNotProgrammed` error if the name is empty.
    ///
    /// This method is deprecated.  Instead, get a [`PasswordSlot`][] instance by calling
    /// [`get_slot`][], [`get_slot_unchecked`][] or [`get_slots`][], and then use
    /// [`PasswordSlot::get_name`][].
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    /// - [`SlotNotProgrammed`][] if the slot is not programmed
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// match device.get_password_safe("123456") {
    ///     Ok(pws) => {
    ///         let name = pws.get_slot_name(0)?;
    ///         let login = pws.get_slot_login(0)?;
    ///         let password = pws.get_slot_password(0)?;
    ///         println!("Credentials for {}: login {}, password {}", name, login, password);
    ///     },
    ///     Err(err) => eprintln!("Could not open the password safe: {}", err),
    /// };
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`PasswordSlot`]: struct.PasswordSlot.html
    /// [`PasswordSlot::get_name`]: struct.PasswordSlot.html#method.get_name
    /// [`SlotNotProgrammed`]: enum.CommandError.html#variant.SlotNotProgrammed
    /// [`get_slot`]: #method.get_slot
    /// [`get_slot_unchecked`]: #method.get_slot_unchecked
    /// [`get_slots`]: #method.get_slots
    #[deprecated(since = "0.9.0", note = "Use get_slot(slot)?.get_name() instead")]
    pub fn get_slot_name(&self, slot: u8) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_name(slot) })
            .and_then(get_pws_result)
    }

    /// Returns the login for the given slot (if it is programmed).
    ///
    /// This method also returns a `SlotNotProgrammed` error if the login is empty.
    ///
    /// This method is deprecated.  Instead, get a [`PasswordSlot`][] instance by calling
    /// [`get_slot`][], [`get_slot_unchecked`][] or [`get_slots`][], and then use
    /// [`PasswordSlot::get_login`][].
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    /// - [`SlotNotProgrammed`][] if the slot is not programmed
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// let name = pws.get_slot_name(0)?;
    /// let login = pws.get_slot_login(0)?;
    /// let password = pws.get_slot_login(0)?;
    /// println!("Credentials for {}: login {}, password {}", name, login, password);
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`PasswordSlot`]: struct.PasswordSlot.html
    /// [`PasswordSlot::get_login`]: struct.PasswordSlot.html#method.get_login
    /// [`SlotNotProgrammed`]: enum.CommandError.html#variant.SlotNotProgrammed
    /// [`get_slot`]: #method.get_slot
    /// [`get_slot_unchecked`]: #method.get_slot_unchecked
    /// [`get_slots`]: #method.get_slots
    #[deprecated(since = "0.9.0", note = "Use get_slot(slot)?.get_login() instead")]
    pub fn get_slot_login(&self, slot: u8) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_login(slot) })
            .and_then(get_pws_result)
    }

    /// Returns the password for the given slot (if it is programmed).
    ///
    /// This method also returns a `SlotNotProgrammed` error if the password is empty.
    ///
    /// This method is deprecated.  Instead, get a [`PasswordSlot`][] instance by calling
    /// [`get_slot`][], [`get_slot_unchecked`][] or [`get_slots`][], and then use
    /// [`PasswordSlot::get_password`][].
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    /// - [`SlotNotProgrammed`][] if the slot is not programmed
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let pws = device.get_password_safe("123456")?;
    /// let name = pws.get_slot_name(0)?;
    /// let login = pws.get_slot_login(0)?;
    /// let password = pws.get_slot_login(0)?;
    /// println!("Credentials for {}: login {}, password {}", name, login, password);
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`PasswordSlot`]: struct.PasswordSlot.html
    /// [`PasswordSlot::get_password`]: struct.PasswordSlot.html#method.get_password
    /// [`SlotNotProgrammed`]: enum.CommandError.html#variant.SlotNotProgrammed
    /// [`get_slot`]: #method.get_slot
    /// [`get_slot_unchecked`]: #method.get_slot_unchecked
    /// [`get_slots`]: #method.get_slots
    #[deprecated(since = "0.9.0", note = "Use get_slot(slot)?.get_password() instead")]
    pub fn get_slot_password(&self, slot: u8) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_password(slot) })
            .and_then(get_pws_result)
    }

    /// Writes the given slot with the given name, login and password.
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    /// - [`InvalidString`][] if the provided token ID contains a null byte
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let mut pws = device.get_password_safe("123456")?;
    /// pws.write_slot(0, "rust-lang.org", "admin", "passw0rd")?;
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    /// [`InvalidString`]: enum.LibraryError.html#variant.InvalidString
    pub fn write_slot(
        &mut self,
        slot: u8,
        name: &str,
        login: &str,
        password: &str,
    ) -> Result<(), Error> {
        let name_string = get_cstring(name)?;
        let login_string = get_cstring(login)?;
        let password_string = get_cstring(password)?;
        get_command_result(unsafe {
            nitrokey_sys::NK_write_password_safe_slot(
                slot,
                name_string.as_ptr(),
                login_string.as_ptr(),
                password_string.as_ptr(),
            )
        })
    }

    /// Erases the given slot.  Erasing clears the stored name, login and password (if the slot was
    /// programmed).
    ///
    /// # Errors
    ///
    /// - [`InvalidSlot`][] if the given slot is out of range
    ///
    /// # Example
    ///
    /// ```no_run
    /// use nitrokey::GetPasswordSafe;
    /// # use nitrokey::Error;
    ///
    /// # fn try_main() -> Result<(), Error> {
    /// let mut manager = nitrokey::take()?;
    /// let mut device = manager.connect()?;
    /// let mut pws = device.get_password_safe("123456")?;
    /// match pws.erase_slot(0) {
    ///     Ok(()) => println!("Erased slot 0."),
    ///     Err(err) => eprintln!("Could not erase slot 0: {}", err),
    /// };
    /// #     Ok(())
    /// # }
    /// ```
    ///
    /// [`InvalidSlot`]: enum.LibraryError.html#variant.InvalidSlot
    pub fn erase_slot(&mut self, slot: u8) -> Result<(), Error> {
        get_command_result(unsafe { nitrokey_sys::NK_erase_password_safe_slot(slot) })
    }
}

impl<'a, 'b> Drop for PasswordSafe<'a, 'b> {
    fn drop(&mut self) {
        // TODO: disable the password safe -- NK_lock_device has side effects on the Nitrokey
        // Storage, see https://github.com/Nitrokey/nitrokey-storage-firmware/issues/65
    }
}

impl<'p, 'a, 'b> PasswordSlot<'p, 'a, 'b> {
    /// Returns the index of this PWS slot.
    pub fn index(&self) -> u8 {
        self.slot
    }

    /// Returns the name stored in this PWS slot.
    pub fn get_name(&self) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_name(self.slot) })
    }

    /// Returns the login stored in this PWS slot.
    pub fn get_login(&self) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_login(self.slot) })
    }

    /// Returns the password stored in this PWS slot.
    pub fn get_password(&self) -> Result<String, Error> {
        result_from_string(unsafe { nitrokey_sys::NK_get_password_safe_slot_password(self.slot) })
    }
}

impl<'a> GetPasswordSafe<'a> for Librem<'a> {
    fn get_password_safe(&mut self, user_pin: &str) -> Result<PasswordSafe<'_, 'a>, Error> {
        get_password_safe(self, user_pin)
    }
}

impl<'a> GetPasswordSafe<'a> for Pro<'a> {
    fn get_password_safe(&mut self, user_pin: &str) -> Result<PasswordSafe<'_, 'a>, Error> {
        get_password_safe(self, user_pin)
    }
}

impl<'a> GetPasswordSafe<'a> for Storage<'a> {
    fn get_password_safe(&mut self, user_pin: &str) -> Result<PasswordSafe<'_, 'a>, Error> {
        get_password_safe(self, user_pin)
    }
}

impl<'a> GetPasswordSafe<'a> for DeviceWrapper<'a> {
    fn get_password_safe(&mut self, user_pin: &str) -> Result<PasswordSafe<'_, 'a>, Error> {
        get_password_safe(self, user_pin)
    }
}
