/* $Id: barnyard.c,v 1.19 2004/03/27 16:42:59 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
** Portions Copyright (C) 2003-2004 Sourcefire, Inc.
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>

#include "util.h"
#include "barnyard.h"
#include "output-plugins/op_plugbase.h"
#include "input-plugins/dp_plugbase.h"
#include "spool.h"
#include "sid.h"
#include "classification.h"
#include "CommandLineArgs.h"
#include "ProgVars.h"

ProgVars pv;

void ShowBanner();
void ShowUsage();
void LoadDataProcessors();
int PluginsExit();
int PluginsRestart();

void HandleSigTerm(int signal);
void HandleSigInt(int signal);
void HandleSigQuit(int signal);
void HandleSigHup(int signal);
void HandleSigPipe(int signal);

extern DpFunctionalNode *DpFuncList;

int main(int argc, char *argv[])
{
    CommandLineArgs *clargs = NULL;

    memset(&pv, 0, sizeof(ProgVars));

    ShowBanner();

    /* Process the command line */
    if(CommandLineArgs_Parse(argc, argv, &clargs) != 0)
        FatalError("Failed to parse command line\n");

    if(clargs->version_flag)
    {
        exit(0);
    }

    if(clargs->usage_flag)
    {
        CommandLineArgs_PrintUsage(stdout);
        exit(0);
    }

    if(clargs->verbosity >= 3)
    {
        CommandLineArgs_Fprintf(clargs, stdout);
    }

    if(CommandLineArgs_Validate(clargs) != 0)
    {
        fprintf(stderr, "Incorrect command line usage\n");
        CommandLineArgs_PrintUsage(stderr);
        exit(1);
    }

    /* Initialize subsystems */
    LoadDataProcessors();
    LoadOutputPlugins();
    InitProtoNames();
    SetupDataProcessors();

    /* Process the configuration */
    if(ProgVars_Populate(clargs, &pv) != 0)
    {
        FatalError("Invalid configuration\n");
        exit(1);
    }

    if(clargs->dry_run_flag)
    {
        ProgVars_Fprintf(&pv, stdout);
        DpFuncs_LogOutputPluginConfigs(DpFuncList);
        exit(0);
    }
    
    if(pv.verbose >= 2)
        ProgVars_Fprintf(&pv, stdout);

    InitSidMap(pv.sid_msg_file);
    InitGenData(pv.gen_msg_file);
    InitClassData(pv.class_file);

    if(pv.run_mode == MODE_BATCH)
    {
        /* Batch processing mode */
        int idx = 0;
        if(!pv.filelist)
        {
            FatalError("No files to process\n");
        }
        while(pv.filelist[idx])
        {
            if(pv.verbose >= 1)
                LogMessage("Processing: %s\n", pv.filelist[idx]);
            DoOneShot(pv.spool_dir, pv.filelist[idx]);
            idx++;
        }
        CleanExit(0);
    }

    /* Daemon mode */
    signal(SIGTERM, HandleSigTerm);
    signal(SIGINT, HandleSigInt);
    signal(SIGQUIT, HandleSigQuit);
    signal(SIGHUP, HandleSigHup);
    signal(SIGPIPE, HandleSigPipe);

    openlog("barnyard", LOG_PID, LOG_DAEMON);

    if(pv.daemon_flag)
    {
        GoDaemon();
        if(CreatePidFile(pv.pid_file))
        {
            free(pv.pid_file);
            pv.pid_file = NULL;
            CleanExit(-1);
        }
    }
    /* Continual processing mode */
    while(1)
    {
        while(!(pv.exit || pv.reload))
        {
            ProcessSpool(pv.spool_dir, pv.spool_file, pv.record_number, 
                    pv.timet);
            pv.stop = 0;
        }
        if(pv.exit)
            CleanExit(0);
        pv.reload = 0;
        LogMessage("Reloading configuration\n");
        FreeDpFuncList(DpFuncList);
        DpFuncList = NULL;
        SetupDataProcessors();
        if(ProgVars_Populate(clargs, &pv) != 0)
        {
            FatalError("Invalid configuration\n");
            exit(1);
        }
        InitSidMap(pv.sid_msg_file);
        InitGenData(pv.gen_msg_file);
        InitClassData(pv.class_file);
    }
    return 0;
}

void ShowBanner()
{
    printf("Barnyard Version %s (Build %s)\n", VERSION, BUILD);
    return;
}

void HandleSigTerm(int signal)
{
    pv.stop = 1;
    pv.exit = 1;
}

void HandleSigInt(int signal)
{
    pv.stop = 1;
    pv.exit = 1;
}

void HandleSigQuit(int signal)
{
    pv.stop = 1;
    pv.exit = 1;
}

void HandleSigHup(int signal)
{
    pv.stop = 1;
    pv.reload = 1;
}

void HandleSigPipe(int signal)
{
    ;
}

int BarnyardSleep(unsigned int seconds)
{
    /* check for signals that need to be handled */
    if(pv.stop)
        return 1;
    /* sleep */
    sleep(seconds);
    return 0;
}

void CleanExit(int exit_val)
{
    LogMessage("Exiting\n");
    PluginsExit();
    if(pv.pid_file)
        unlink(pv.pid_file);
    exit(exit_val);
}

int PluginsExit()
{
    /* Free all of the output plugins */
    FreeOutputPluginList(outputPlugins, 1);
    outputPlugins = NULL;
    return 0;
}

int PluginsRestart()
{
    return 0;
}

