/* $Id: dp_plugbase.c,v 1.8 2004/03/16 04:18:20 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


/*  I N C L U D E S  *****************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include "util.h"
#include "dp_plugbase.h"
#include "output-plugins/op_plugbase.h"
#include "mstring.h"

#include "dp_alert.h"
#include "dp_log.h"
#include "dp_stream_stat.h"
#include "barnyard.h"

/*  G L O B A L S  ************************************************/
DpRegistrationNode *DpRegistrationList;  /* list of available dataprocessors */

DpFunctionalNode *DpFuncList; /* dataprocessor functional (run-time) list */

extern char *file_name;
extern int file_line;

/*  P R O T O T Y P E S  ************************************************/
char *StripParens(char *);
DpRegistrationNode *NewDpRegistrationNode();
DpFunctionalNode *NewDpFunctionalNode();


void LoadDataProcessors()
{
    if(pv.verbose)
        LogMessage("Loading Data Processors...\n");
    AlertDpInit();
	LogDpInit();
    StreamStatDpInit();

    return;
}

void SetupDataProcessors(void)
{
    DataProcessorSetup("dp_alert", NULL);
    DataProcessorSetup("dp_log", NULL);
    DataProcessorSetup("dp_stream_stat", NULL);
}
    

int DataProcessorSetup(char *name, char *args)
{
    DpRegistrationNode *rn;
    DpFunctionalNode *idx;

    idx = NewDpFunctionalNode();

    for(rn = DpRegistrationList ; rn != NULL ; rn=rn->next)
    {
        if(!strcasecmp(rn->name, name))
        {
            rn->initFunc(args, idx);
            return 0;
        }
    }
    
    FatalError("Unable to load data processor \"%s\", exiting!\n", name);
    return 0;   /* keep compiler from complaining */
}

int DataProcessorStart(DpFunctionalNode *dpfn)
{
    OutputPluginListNode *opln;

    if(dpfn == NULL)
        return 1;

    if(dpfn->started == 1)
        return 0;

    /* start all of the associated output plugins */
    opln = dpfn->oList;
    while(opln != NULL)
    {
        OutputPluginStart(opln->outputPlugin, dpfn->context.file_header);
        opln = opln->next;
    }
    dpfn->started = 1;
    return 0;
}

int DataProcessorStop(DpFunctionalNode *dpfn)
{
    OutputPluginListNode *opln;
   
    if(dpfn == NULL)
        return 1;

    if(dpfn->started == 0)
        return 0;

    /* stop all of the associated plugins */
    opln = dpfn->oList;
    while(opln != NULL)
    {
        OutputPluginStop(opln->outputPlugin);
        opln = opln->next;
    }

    dpfn->started = 0;
    return 0;
}

int DataProcessorRestart(DpFunctionalNode *dpfn)
{
    if(dpfn == NULL)
        return 1;
    if(dpfn->started == 0)
        return 1;
    DataProcessorStop(dpfn);
    DataProcessorStart(dpfn);
    return 0;
}

int DataProcessorExit(DpFunctionalNode *dpfn)
{
    OutputPluginListNode *opln;
    
    if(dpfn == NULL)
        return 1;
    
    /* Stop the data processor */
    if(dpfn->started == 1)
        DataProcessorStop(dpfn);
 
    /* Free the output plugin list */
    FreeOutputPluginList(dpfn->oList, 0);
    opln = NULL;

    if(dpfn->context.file_header != NULL)
        free(dpfn->context.file_header);
    return 0;
}


void DpFuncs_LogOutputPluginConfigs(DpFunctionalNode *func_node)
{
    int plugin_configured = 0;
    OutputPluginListNode *oList = NULL;
    OutputPlugin *op = NULL;
    while(func_node)
    {
        plugin_configured = 0;
        LogMessage("Output plugins enabled for '%s' records\n", 
                func_node->type);
        oList = func_node->oList;
        LogMessage("-------------------------------------------------------\n");
        while(oList)
        {
            op = oList->outputPlugin;
            if(op)
            {
                if(op->logConfigFunc)
                {
                    op->logConfigFunc(op);
                }
                else
                {
                    LogMessage("Output plug-in '%s' configured\n", op->name);
                }
                plugin_configured = 1;
            }
            oList = oList->next;
        }
        if(!plugin_configured)
            LogMessage("None configured\n");
        LogMessage("=======================================================\n");
        func_node = func_node->next;
    }
}

void RegisterDp(char *name, void (*Setup)(char *, DpFunctionalNode *), 
        char *type, PluginInfo *pi)
{
    DpRegistrationNode *idx;

    idx = NewDpRegistrationNode();

    idx->name = strdup(name);
    idx->initFunc = Setup;
    idx->type = strdup(type);
    idx->pi.author =  strdup(pi->author);
    idx->pi.version = strdup(pi->version);
    idx->pi.type = strdup(pi->type);
    idx->pi.copyright = strdup(pi->copyright);
    idx->pi.description = strdup(pi->description);
    idx->pi.usage = strdup(pi->usage);
}

void RegisterDpReadRecord(int (*ReadRecord)(SpoolFileHandle *), 
        DpFunctionalNode *dpfn)
{
    dpfn->readRecordFunc = ReadRecord;
    return;
}

void RegisterDpReadFileHeader(int (*func)(DpFunctionalNode *, 
            SpoolFileHandle *), DpFunctionalNode *dpfn)
        
{
    dpfn->readFileHeaderFunc = func;
    return;
}

void RegisterDpHeaderCompare(int (*func)(DpFunctionalNode *, void *, void *), 
        DpFunctionalNode *dpfn)
{
    dpfn->headerCompareFunc = func;
    return;
}

void RegisterDpProcessRecord(int (*func)(void *, DpFunctionalNode *), DpFunctionalNode *dpfn)
{
    dpfn->processRecordFunc = func;
    return;
}

DpFunctionalNode *GetDpType(char *type)
{
    DpFunctionalNode *idx;

    idx = DpFuncList;

    if(idx == NULL)
    {
        LogMessage("GetDpType: No Data processors registered!\n");
        return NULL;
    }

#ifdef DEBUG
    printf("GetDpType for \"%s\"\n", type);
#endif
    
    while(idx != NULL)
    {
#ifdef DEBUG
        printf("compare: %s => %s\n", type, idx->type);
#endif

        if(!strncasecmp(type, idx->type, strlen(type)))
        {
            return idx;
        }

        idx = idx->next;
    }

    return NULL;
}


void FreeDpRegList(DpRegistrationNode *idx)
{
    if(idx == NULL)
    {
        return;
    }

    if(idx->next != NULL)
    {
        FreeDpRegList(idx->next);
    }

    free(idx->name);
    free(idx->type);
    free(idx);    
}

void FreeDpFuncList(DpFunctionalNode *idx)
{
    if(idx == NULL)
    {
        return;
    }

    if(idx->next != NULL)
    {
        FreeDpFuncList(idx->next);
    }

    DataProcessorExit(idx);
    
    free(idx);

    return;
}


DpFunctionalNode *NewDpFunctionalNode()
{
    DpFunctionalNode *tmp;

    if(DpFuncList == NULL)
    {
        DpFuncList = (DpFunctionalNode *) SafeAlloc(sizeof(DpFunctionalNode));
		
        return DpFuncList;
    }
    else
    {
        tmp = DpFuncList;
        
        while(tmp->next != NULL) 
            tmp = tmp->next;

        tmp->next = (DpFunctionalNode *) SafeAlloc(sizeof(DpFunctionalNode));
        
        return tmp->next;
    }
}


DpRegistrationNode *NewDpRegistrationNode()
{
    DpRegistrationNode *idx;
    
    if(DpRegistrationList == NULL)
    {
        DpRegistrationList = (DpRegistrationNode *)
            SafeAlloc(sizeof(DpRegistrationNode));
        
        return DpRegistrationList;
    }
    else
    {
        idx = DpRegistrationList;
        
        while(idx->next != NULL) { idx = idx->next; }

        idx->next = (DpRegistrationNode *)
            SafeAlloc(sizeof(DpRegistrationNode));

        idx = idx->next;

        return idx;
    }
}


DpFunctionalNode *LookupFunctions(u_int32_t magic)
{
    DpFunctionalNode *idx;

#ifdef DEBUG
	printf("Looking for magic: %.8x\n", magic);
#endif
	
    idx = DpFuncList;

    if(idx == NULL)
    {
		LogMessage("DpFuncList is NULL!!!!\n");
        return NULL;
    }
    
    while(idx != NULL)
    {
#ifdef DEBUG
		printf("magic ?= %.8x\n", idx->magic);
#endif
        if(idx->magic == magic)
        {
            return idx;
        }

        idx = idx->next;
    }

    return NULL;
}
