/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.logging;

import java.io.Closeable;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class Throttler
implements Closeable {
    private static final Logger classLogger = LogManager.getLogger(Throttler.class);
    private final TimeValue resetInterval;
    private Duration durationToWait;
    private final Clock clock;
    private final ConcurrentMap<String, LogExecutor> logExecutors;
    private final AtomicReference<Scheduler.Cancellable> cancellableTask = new AtomicReference();
    private final AtomicBoolean isRunning = new AtomicBoolean(true);

    public Throttler(TimeValue resetInterval, TimeValue durationToWait, ThreadPool threadPool) {
        this(resetInterval, durationToWait, Clock.systemUTC(), threadPool, new ConcurrentHashMap<String, LogExecutor>());
    }

    Throttler(TimeValue resetInterval, TimeValue durationToWait, Clock clock, ThreadPool threadPool, ConcurrentMap<String, LogExecutor> logExecutors) {
        Objects.requireNonNull(durationToWait);
        Objects.requireNonNull(threadPool);
        this.resetInterval = Objects.requireNonNull(resetInterval);
        this.durationToWait = Duration.ofMillis(durationToWait.millis());
        this.clock = Objects.requireNonNull(clock);
        this.logExecutors = Objects.requireNonNull(logExecutors);
        this.cancellableTask.set(this.startResetTask(threadPool));
    }

    private Scheduler.Cancellable startResetTask(ThreadPool threadPool) {
        classLogger.debug(() -> Strings.format((String)"Reset task scheduled with interval [%s]", (Object[])new Object[]{this.resetInterval}));
        return threadPool.scheduleWithFixedDelay(this.logExecutors::clear, this.resetInterval, (Executor)threadPool.executor("inference_utility"));
    }

    public void setDurationToWait(TimeValue durationToWait) {
        this.durationToWait = Duration.ofMillis(durationToWait.millis());
    }

    public void execute(String message, Consumer<String> consumer) {
        if (!this.isRunning.get()) {
            return;
        }
        LogExecutor logExecutor = this.logExecutors.compute(message, (key, value) -> {
            if (value == null) {
                return new LogExecutor(this.clock, consumer);
            }
            return value.compute(consumer, this.durationToWait);
        });
        logExecutor.log(message);
    }

    @Override
    public void close() {
        this.isRunning.set(false);
        this.cancellableTask.get().cancel();
        this.logExecutors.clear();
    }

    private static class LogExecutor {
        private final long skippedLogCalls;
        private final Instant timeOfLastLogCall;
        private final Clock clock;
        private final Consumer<String> consumer;

        LogExecutor(Clock clock, Consumer<String> throttledConsumer) {
            this(clock, 0L, throttledConsumer);
        }

        LogExecutor(Clock clock, long skippedLogCalls, Consumer<String> consumer) {
            this.skippedLogCalls = skippedLogCalls;
            this.clock = Objects.requireNonNull(clock);
            this.timeOfLastLogCall = Instant.now(this.clock);
            this.consumer = Objects.requireNonNull(consumer);
        }

        void log(String message) {
            this.consumer.accept(message);
        }

        LogExecutor compute(Consumer<String> executor, Duration durationToWait) {
            if (this.hasDurationExpired(durationToWait)) {
                String messageToAppend = "";
                if (this.skippedLogCalls == 1L) {
                    messageToAppend = ", repeated 1 time";
                } else if (this.skippedLogCalls > 1L) {
                    messageToAppend = Strings.format((String)", repeated %s times", (Object[])new Object[]{this.skippedLogCalls});
                }
                String stringToAppend = messageToAppend;
                return new LogExecutor(this.clock, 0L, message -> executor.accept(message.concat(stringToAppend)));
            }
            return new LogExecutor(this.clock, this.skippedLogCalls + 1L, message -> {});
        }

        private boolean hasDurationExpired(Duration durationToWait) {
            Instant now = Instant.now(this.clock);
            return now.isAfter(this.timeOfLastLogCall.plus(durationToWait));
        }
    }
}

