/*
 * dvi2gr - convert DVI into Grace text string
 * 
 * Copyright (c) 2000 Evgeny Stambulchik
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <t1lib.h>

#include "dvilib.h"

#define DVI_SCALE 655360

/* A very rough estimate for a char width */
#define FAKE_CHAR_WIDTH    300000

/* Use same font for all resolutions */
#define OPTIMIZE_FONTS

/* Don't advance text (though accumulate the value) if less than this */
#define DH_THRESH   60000
/* Exit if vertical movement exceeds the value (usually, page number) */
#define DV_THRESH   3000000

static int c_h = 0, c_v = 0, c_fid = -1;
static float c_z = 1.0;

#ifdef OPTIMIZE_FONTS
static char c_fname[64] = "none";
#endif

static void errmsg(const char *msg)
{
    fprintf(stderr, msg);
}

static void moveto(DVI_S32 h, DVI_S32 v)
{
    int dh, dv;

    dv = v - c_v;
    if (DV_THRESH && c_v != 0 && abs(dv) > DV_THRESH) {
        exit(0);
    }
    if (dv != 0) {
        if (c_v != 0) {
            printf("\\v{%.2f}", - (float) dv/DVI_SCALE/c_z);
        }
        c_v = v;
    }
    
    dh = h - c_h;
    if (abs(dh) > DH_THRESH) {
        if (c_h != 0) {
            printf("\\h{%.2f}", (float) dh/DVI_SCALE/c_z);
        }
        c_h = h;
    }
}

static void setfont(int fid, char *fname)
{
#ifdef OPTIMIZE_FONTS
    char *p;
    
    c_fid = fid;
    
    if (strstr(fname, c_fname)) {
        return;
    }
    
    p = c_fname;
    strcpy(p, fname);
    while(*p && !isdigit(*p)) {
        p++;
    }
    *p = '\0';
#else
#  define c_fname fname
#endif
    
    if (strcmp(c_fname, "Symbol") == 0) {
        printf("\\x");
    } else {
        printf("\\f{%s}", c_fname);
    }
}

static int write_char(DVI *dvip, int c, int advance)
{
    DVI_finfo *finfo;
    float width;
    int i, t1_font_id;
    
    if (c < 0 || c > 255) {
        /* Can't handle multibyte chars */
        return 0;
    }
    
    moveto(dvip->state->h, dvip->state->v);
    
    finfo = DVI_get_finfo(dvip, dvip->font_id);
    
    if (dvip->font_id != c_fid) {
        float zoom;
        
        setfont(dvip->font_id, finfo->name);
        
        zoom = (float) finfo->magsize/DVI_SCALE;
        if (zoom != c_z) {
            printf("\\Z{%.2f}", zoom);
            c_z = zoom;
        }
    }

    if (!advance) {
        printf("\\m{9}");
    }
    if (c == '\\') {
        printf("\\\\");
    } else
    if (c > 32 && c < 127) {
        printf("%c", (char) c);
    } else {
        printf("\\#{%02x}", c);
    }
    if (!advance) {
        printf("\\M{9}");
    }

    t1_font_id = -1;
    for (i = 0; i < T1_GetNoFonts(); i++) {
        char *fname = T1_GetFontFileName(i);
        if (fname && strstr(fname, finfo->name)) {
            t1_font_id = i;
            break;
        }
    }
    if (t1_font_id < 0) {
        char buf[128];
        sprintf(buf, "%s.pfb", finfo->name);
        t1_font_id = T1_AddFont(buf);
        if (t1_font_id >= 0) {
            T1_LoadFont(t1_font_id);
        }
    }
    if (t1_font_id >= 0) {
        int charwidth = T1_GetCharWidth(t1_font_id, c);
        width = charwidth*DVI_SCALE/1000.0;
    } else {
        errmsg("Font not found, using fake char width\n");
        width = FAKE_CHAR_WIDTH;
    }

    c_h += c_z*width;
    
    return c_z*width;
}

static void draw_rule(DVI *dvip, int a, int b)
{
    int i, n;
    float f = (float) abs(b)*2/DVI_SCALE;
    moveto(dvip->state->h, dvip->state->v);
    
    n = (int) (f + 0.5);
    if (n == 0) {
        n = 1;
    }
    
    setfont(-1, "Symbol");
    printf("\\z{.5}\\v{-.22}\\#{");
    for (i = 0; i < n; i++) {
        printf("be");
    }
    printf("}\\v{.22}");
    
    c_z *= 0.5;
    c_h += c_z*n*DVI_SCALE;
}

static FILE *fp;

static void end_processing(void)
{
    printf("\n");

    fclose(fp);
}

int init_t1(char *fpath)
{
    /* Initialize t1-library */
    if (T1_InitLib(IGNORE_CONFIGFILE) == NULL) {
        return 1;
    }

    T1_SetFileSearchPath(T1_PFAB_PATH, fpath);
    T1_SetFileSearchPath(T1_AFM_PATH, fpath);
    
    /* Rasterization parameters */
    T1_SetDeviceResolutions(72.0, 72.0);
    
    return 0;
}

int main(int argc, char **argv)
{
    DVI *dvip;
    
    if (argc != 3) {
        errmsg("usage: dvi2gr <font_path> <dvi_file>\n");
        exit(1);
    }

    fp = fopen(argv[2], "rb");

    if (fp == NULL) {
        errmsg("Fatal error: cannot open input file\n");
        exit(1);
    }
    
    init_t1(argv[1]);
    
    atexit(end_processing);
    
    dvip = DVI_new(fp);

    DVI_add_hooks(dvip, write_char, draw_rule);

    DVI_process(dvip);
    
    DVI_free(dvip);    
    exit(0);
}
