/*
 * sturmbahnfahrer
 * (c) 2006 by Bram Stolk
 * bram at gmail.com
 * LICENSED ACCORDING TO THE GPL
 */

#include "cartobject.h"

#define LENGTH 1.6      // chassis length
#define WIDTH  1.12     // chassis width
#define HEIGHT 0.20     // chassis height
#define WHEELW 0.2      // wheel width
#define WHEELRAD 0.2    // wheel radius



CartObject::CartObject
(
  ssgEntity *bodymodel,
  ssgEntity *wheelmodel,
  ssgEntity *licplatemodel,
  dWorldID world, 
  dSpaceID bigspace,
  sgVec3 initialpos
) :
WorldObject(bodymodel),
desired_wheel_speed(0),
reaction_time(0),
state("accelerating")
{
  name = "cart";

  // load visible model as plib entity
  ssgBranch *branch = new ssgBranch();
  chassis_trf = new ssgTransform();
  chassis_trf->setName("chassis_trf");
  chassis_trf->addKid(bodymodel);
  if (licplatemodel) chassis_trf->addKid(licplatemodel);

  branch->addKid(chassis_trf);
  int i;
  for (i=0; i<4; i++)
  {
    wheel_trfs[i] = new ssgTransform();
    branch->addKid(wheel_trfs[i]);
    wheel_trfs[i]->addKid(wheelmodel);
  }
  entity = branch;

  // chassis
  chassis_body = dBodyCreate(world);
  dBodySetAutoDisableFlag(chassis_body, false);
  dBodySetPosition (chassis_body, initialpos[0],initialpos[1],initialpos[2]);

  chassis_geom = dCreateBox (0,LENGTH,WIDTH,HEIGHT);
  dGeomSetBody(chassis_geom, chassis_body);
  dGeomSetData (chassis_geom, this);

  backrest_geom = dCreateBox (0, 0.1, 0.7, 0.4);
  dGeomSetData (backrest_geom, this);
  dGeomSetBody(backrest_geom, chassis_body);

  // wheel bodies
  for (i=0; i<4; i++)
  {
    wheel_bodies[i] = dBodyCreate(world);
    dBodySetAutoDisableFlag(wheel_bodies[i], false);
    dQuaternion q;
    dQFromAxisAndAngle (q,1,0,0,M_PI*0.5);
    dBodySetQuaternion (wheel_bodies[i],q);
    if (i<=2) dBodySetFiniteRotationMode(wheel_bodies[i], true); // Added, BSt
    wheel_geoms[i] = dCreateCylinder(0, WHEELRAD, WHEELW);
    dGeomSetBody (wheel_geoms[i],wheel_bodies[i]);
    dGeomSetData (wheel_geoms[i],this);
  }
  dBodySetPosition (wheel_bodies[0],initialpos[0]+0.5,initialpos[1]+0.50,initialpos[2]-HEIGHT*0.2);
  dBodySetPosition (wheel_bodies[1],initialpos[0]+0.5,initialpos[1]-0.50,initialpos[2]-HEIGHT*0.2);
  dBodySetPosition (wheel_bodies[2],initialpos[0]-0.5,initialpos[1]+0.50,initialpos[2]-HEIGHT*0.2);
  dBodySetPosition (wheel_bodies[3],initialpos[0]-0.5,initialpos[1]-0.50,initialpos[2]-HEIGHT*0.2);

  SetDensity(2.5);

  // front and back wheel hinges
  for (i=0; i<4; i++)
  {
    joint[i] = dJointCreateHinge2 (world,0);
    dJointAttach (joint[i],chassis_body,wheel_bodies[i]);
    const dReal *a = dBodyGetPosition (wheel_bodies[i]);
    dJointSetHinge2Anchor (joint[i],a[0],a[1],a[2]);
    dJointSetHinge2Axis1 (joint[i],0,0,1);
    dJointSetHinge2Axis2 (joint[i],0,1,0);
  }

  // set joint suspension
  for (i=0; i<4; i++)
  {
    dJointSetHinge2Param (joint[i],dParamSuspensionERP,0.35);
    dJointSetHinge2Param (joint[i],dParamSuspensionCFM,0.02);
  }

  // lock wheels along the steering axis
  for (i=0; i<=3; i++)
  {
    // set stops to make sure wheels always stay in alignment
    dJointSetHinge2Param (joint[i],dParamLoStop,0);
    dJointSetHinge2Param (joint[i],dParamHiStop,0);
    // Do this twice... weird ode bug! see mailing list and Jon's carworld
    dJointSetHinge2Param (joint[i],dParamLoStop,0);
    dJointSetHinge2Param (joint[i],dParamHiStop,0);
    // Set error reduction parameter, and contraint force mix-in.
    dJointSetHinge2Param (joint[i],dParamStopERP,0.99);
    dJointSetHinge2Param (joint[i],dParamStopCFM,0.01);
  }

  space = dSimpleSpaceCreate(bigspace);
  dSpaceSetCleanup(space,0);

  dSpaceAdd (space,backrest_geom);
  dSpaceAdd (space,chassis_geom);

  for (i=0; i<4; i++)
    dSpaceAdd (space,wheel_geoms[i]);
}


CartObject::~CartObject()
{
  int i;
  for (i=0; i<4; i++)
  {
    dGeomDestroy(wheel_geoms[i]);
    dJointDestroy(joint[i]);
  }
  dGeomDestroy(chassis_geom);
  dGeomDestroy(backrest_geom);
  dSpaceDestroy(space);
}


void CartObject::Simulate(float dt)
{
  dReal speedv = 0.0;
  dReal fmax   = 0.0;

  if (state=="accelerating")
  {
    speedv = desired_wheel_speed;
    fmax = 1.5;
  }

  if (state=="freewheeling")
  {
    speedv = 0;
    fmax = 0.01;
  }

  if (state=="braking")
  {
    speedv = 0;
    fmax = 2.0;
  }

  // motor: drive rear wheels.
  for (int jnr=2; jnr<=3; jnr++)
  {
    dJointSetHinge2Param (joint[jnr],dParamVel2,-speedv);
    dJointSetHinge2Param (joint[jnr],dParamFMax2,fmax);
  }
}


void CartObject::Sustain(float dt)
{
  static float delay;

  if (state == "accelerating")
  {
    const dReal *pos = dBodyGetPosition(chassis_body);
    if (pos[0] > -0.5)
    {
      // front wheels cross the middle-line
      state = "reacting";
      delay = reaction_time;
    }
  }

  if (state == "reacting")
  {
    if (delay > 0.0)
    {
      delay -= dt;
    }
    else
    {
      state = "braking";
    }
  }

  Simulate(dt);

  SetTransformFromBody(chassis_trf, chassis_body);
  for (int i=0; i<4; i++)
    SetTransformFromBody(wheel_trfs[i], wheel_bodies[i]);
}


void CartObject::SetDensity(float d)
{
  dMass m_total, m_chassis, m_backrest, m_wheel;

  dMassSetZero(&m_total);
  dMassSetBox(&m_chassis,d,LENGTH,WIDTH,HEIGHT);
  dMassSetBox(&m_backrest,2*d,0.1,0.7,0.4);
  dMassTranslate(&m_backrest,-0.7,0,0.3);
  dMassAdd(&m_total, &m_chassis);
//  dMassAdd(&m_total, &m_backrest);

  dReal *o = m_total.c;
  dGeomSetOffsetPosition(chassis_geom,  -o[0], -o[1], -o[2]);
  dGeomSetOffsetPosition(backrest_geom, -0.7-o[0], 0.0-o[1], 0.3-o[2]);

  dMassTranslate(&m_total, -o[0], -o[1], -o[2]);
  dBodySetMass(chassis_body, &m_total);

  dMassSetCylinder (&m_wheel, 2.5, 2, WHEELRAD, WHEELW);
  for (int i=0; i<4; i++)
    dBodySetMass (wheel_bodies[i],&m_wheel);
}


void CartObject::SetDesiredLinearSpeed(float l) 
{
  desired_wheel_speed = l / WHEELRAD;
}

