/* Xorshift 1.2
 *
 * Namespace: RNG_ / rng_ 
 */

/* Copyright (c) 2009, The Static Void Project
 *
 * Permission to use, copy, modify, and/or distribute this software for 
 * any purpose with or without fee is hereby granted, provided that the 
 * above copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */


/* Xorshift is a very fast and simple yet high quality RNG.
 *
 * It uses the xorshift technique developed by George Marsaglia
 * hence the name. 
 *
 * Portability:
 *
 * Works on ILP32, LP64, and LLP64 systems. The code assumes "int" 
 * is 32 bit wide which is true on all of these systems.
 *
 * Quality:
 *
 * Period about 2^64, passes all Diehard (http://i.cs.hku.hk/~diehard/)
 * tests for randomness.
 */

#include "xorshift.h"


/* Xorshift RNG
 */
#define RNG_INT  (X=69069U*X+123U,Y^=Y<<13U,Y^=Y>>17U,Y^=Y<<5U,X+Y)
#define RNG_REAL (RNG_INT*2.328306437e-10F)


/* Xorshift RNG state data
 */
static unsigned int X = RNG_X_SEED;
static unsigned int Y = RNG_Y_SEED;



/* Returns a random integer x such that 0 <= x < n
 * (Uniform distribution)
 */
int rng_(int n)
{
        return (int)(RNG_REAL * n);
}



/* Returns a random integer x such that lb <= x <= ub
 * (Uniform distribution)
 */
int rng_r(int lb, int ub)
{
        return (int)(lb + RNG_REAL * ((ub - lb) + 1));
}



/* Rolls dice and returns the result
 */
int rng_dice(int n_dice, int n_sides)
{
        int result = 0;

        while (n_dice-- > 0) 
                result += (int)(1 + RNG_REAL * n_sides);

        return result;
}



/* Returns a random unsigned integer
 */
unsigned int rng_number(void)
{
        return RNG_INT;
}



/* Sets the state of the RNG (x and y may NOT be 0)
 */
void rng_set_state(unsigned int x, unsigned int y)
{
        X = x; Y = y;
}



/* Gets the state of the RNG 
 */
void rng_get_state(unsigned int *x, unsigned int *y)
{
        *x = X; *y = Y;
}

