/*
 * globalsymbols.h
 *
 *  Created on: Feb 21, 2012
 *      Author: cedric
 */

#include "yamlconfigurable.h"
#include "ginac/ginac.h"

#ifndef GLOBALSYMBOLS_H_
#define GLOBALSYMBOLS_H_

namespace Reduze {

/// global symbols
class GlobalSymbols: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("global_symbols");
		s.set_short_description("Declaration of global symbols.");
		s.set_long_description(s.short_description());
		s.add_option("coupling_constants", false, "sequence of strings", "" //
						"Definition of the coupling constant symbols.");
		s.add_option("su_n_dimensions", false, "2-element map", "" //
						"Dimension symbols for the fundamental and adjoint"
						" representation of su(N).");
		s.add_option("space_time_dimension", false, "string", "" //
						"The space time dimension symbol.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	GlobalSymbols();
	virtual ~GlobalSymbols() {
	}

	/// read in the symbols from a YAML Node
	virtual void read(const YAML::Node& node);
	virtual void print(YAML::Emitter& ye) const;

	// to be backward compatible: overwrite symbols by symbols defined in
	// Kinematics and FeynmanRules. This function must be called after a
	// GlobalSymbols object has been constructed without using read(...)
	void set_external_symbols();

	/// adjoint dimension of su(N)
	const GiNaC::realsymbol& Na() const;
	/// fundamental dimension of su(N)
	const GiNaC::realsymbol& Nc() const;
	/// coupling constants
	const GiNaC::lst& coupling_constants() const;
	/// space time dimension
	const GiNaC::realsymbol& d() const;
	/// get all symbols
	const GiNaC::lst& all() const;

private:
	/// coupling constants
	GiNaC::lst coupling_constants_;
	/// adjoint dimension of su(N)
	GiNaC::realsymbol Na_; // Nc^2 - 1
	/// fundamental dimension of su(N)
	GiNaC::realsymbol Nc_;
	/// space time dimension
	GiNaC::realsymbol d_;
	/// all symbols
	GiNaC::lst all_;

private:
	void set_all();
};

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const GlobalSymbols& r) {
	r.print(ye);
	return ye;
}

//
//
//

/// paths to external programs
class Paths: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("paths");
		s.set_short_description("Declaration of paths.");
		s.set_long_description(s.short_description());

		s.add_option("fermat", false, "string", "" //
						"The path to the Fermat executable. Currently,"
						" Fermat support is limited to handling reduction equations"
						" and does not cover advanced linear combinations with"
						" functions.");

		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	Paths() {
	}
	virtual ~Paths() {
	}
	virtual void read(const YAML::Node& node);
	virtual void print(YAML::Emitter& os) const;

	// getters
	const std::string& fermat() const;

private:
	std::string fermat_;
};

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Paths& r) {
	r.print(ye);
	return ye;
}

class GlobalOptions: public YAMLAutoConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("global_options");
		s.set_short_description("Global options.");
		s.set_long_description("Configuration of global Reduze features which"
				" are not specific to any particular job.");
		s.add_option("integral_ordering", false, "string", ""//
				"The ordering to be used for comparisons of integrals."
				" This option affects the type of master integrals."
				" Change it between runs only if you know what you are doing,"
				" see also the job normalize."
				" Possible values are: "
				" \"rs\" (sorts first by r, then by s),"
				" \"sr\" (sorts first by s, then by r),"
				" \"degrlex_r\" and \"degrlex_s\" (total degree orderings).");
/*
		s.add_option("use_laurent_polynomials", false, "boolean", "" //
						"Fermat only. Whether to use Laurent polynomials, e.g."
						" t^(-1)/(t-1) instead of 1/(t^2-t).");
*/
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	GlobalOptions() : integral_ordering("rs") /* use_laurent_polynomials(false) */ {
		add_auto_options();
	}
	virtual ~GlobalOptions() {
	}
protected:
	virtual void add_auto_options() {
		add_auto_io("integral_ordering", integral_ordering);
//		add_auto_io("use_laurent_polynomials", use_laurent_polynomials);
	}
public:
	std::string integral_ordering;
//	bool use_laurent_polynomials;
};

} // namespac Reduze

#endif /* GLOBALSYMBOLS_H_ */
