/*  jobqueue.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOBQUEUE_H_
#define JOBQUEUE_H_

#include <string>
#include <list>
#include <map>
#include <set>
#include "../yaml/include/yaml.h"
#include "job.h"
#ifdef HAVE_MPI
#include "mpi.h"
#endif

namespace Reduze {

class Job;

/// Queue for instances of Job with dynamic optimization of the execution order
class JobQueue {
public:

	JobQueue();
	virtual ~JobQueue();

	static void print_jobs(std::ostream&);

	/// returns a runnable job if possible (creates auxiliary jobs if necessary)
	/** if there are pending jobs without open dependencies, one of them is returned
	 ** else if a newly created auxiliary job resolves dependencies, it is returned
	 ** else zero is returned */
	Job* provide_runnable_job();

	void set_job_status(int jobid, Job::Status);
	/// add elapsed times for a job
	/** walltime: absolute wall time,
	 ** num_proc: number of processes running this job for given wall time,
	 ** tot_cputime: sum of CPU times for all involved processes **/
	//void add_job_times(int jobid, double walltime, int num_proc,
	//		double tot_cputime);
	Job::Status job_status(int id) const;

	/// a string description of a job
    std::string job_string(const Job*) const;

	Job* job_by_id(int id) {
		return job_by_id_[id];
	}
	size_t num_jobs() const;
	size_t num_running_jobs() const;
	bool has_pending_jobs() const;
	bool has_running_jobs() const;
	void run();
	bool is_completed() const;
	/// print in DOT output format
	/** run "dot -Tps filename.dot -o filename.ps" to generate PostScript **/
	void print_dot(const std::string& filename) const;
	/// JobQueue becomes owner of inserted Job
	void insert(Job* job);
	void print_job(const Job* job, std::ostream& os, bool verbose = false) const;
	void print(std::ostream& os, bool verbose = false) const;
    void read(const std::string& jobfile);
	friend std::ostream& operator<<(std::ostream& os, const JobQueue& q);
	friend YAML::Emitter& operator<<(YAML::Emitter&, const JobQueue&);

	/// returns the maximal process duration in seconds
	double timeout() const;

	/// returns the maximal number of parallel jobs
	size_t max_parallel_jobs() const;

	/// returns time between collection and analysis of performance measurements
	double time_interval_analysis() const;

	/// returns maximum number of workers required to be released by one employer
	size_t max_workers_release() const;

	bool resolve_dependencies(int timeout = 0);

	const std::list<Job*>& jobs() const;

	// public since needed by jobcenter
	bool insert_next_joblist();

private:
	void run_inserted_jobs();
	void insert(const std::string& keyword, const YAML::Node& job_node);
	std::list<Job*>::iterator insert(Job* job, std::list<Job*>::iterator pos);
	Job* find_most_referenced_runnable_job();
	void waitforfile(const std::string& file, const Job* j, int timeout);

	/// list of all jobs in the queue
	std::list<Job*> jobs_;
	/// map job id -> job for jobs in the queue
	std::map<int, Job*> job_by_id_;
	/// map job id -> job ids on which job depends
	/** (job id > 0 for real jobs, = 0 for unknown dep., < 0 for unresolved file)*/
	std::map<int, std::set<int> > dependencies_;
	/// input files of jobs
	/** note: is set if and only if dependency lookup for job was complete */
	std::map<int, std::list<std::string> > input_of_id_;
	/// output files of jobs
	std::map<int, std::list<std::string> > output_of_id_;
	/// jobs (value) depending on pending job with given id (key)
	std::map<int, std::set<int> > references_;
	/// status of the jobs indexed by job id
	std::map<int, Job::Status> job_status_;
	/// job which is responsible for providing a certain file
	std::map<std::string, int> job_for_file_;
	/// maximal process duration in seconds
	double timeout_;
	/// maximal number of parallel jobs
	size_t max_parallel_jobs_;
	/// time interval between rebalancing analysis in seconds
	double time_interval_analysis_;
	/// maximum number of workers required to be released by one employer
	size_t max_workers_release_;
	/// highest job id assigned
	unsigned max_job_id_;

	/** sorted lists of unparsed jobs (YAML node by keyword) which can be filled
	 ** into the JobQueue at once. The jobs of (n+1)-th list will only be performed
	 ** after all jobs of the n-th list are completed. The jobs in the sublist
	 ** are performed according to their dependencies by the JobQueue/JobCenter
	 **/
	std::list<std::list<std::pair<std::string, YAML::Node*> > > sorted_job_lists_;
};

inline const std::list<Job*>& JobQueue::jobs() const {
	return jobs_;
}

} // namespace Reduze

#endif /* JOBQUEUE_H_ */
