/*  reduzermpi.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifdef HAVE_MPI

#ifndef REDUZERMPI_H_
#define REDUZERMPI_H_

#include <map>
#include <queue>
#include <string>
#include <mpi.h>

#include "reduzer.h"

namespace Reduze {

class Communicator;
class IntegralFamily;

/// worker for a reduction of a system of equations
class ReduzerMPILeave {
public:
	ReduzerMPILeave(MPI::Intracomm* comm, int root_rank);
	/// repeat waiting for jobs from root and processing them
	void run();
	/// print timing measurements
	void print_times() const;
	/// commands to be sent to root
	enum Cmd {
		CmdTransferResultsBegin, CmdTransferResultsEnd
	};
	enum Times {
		TimeSend, TimeRecv, TimeWork, TimeTotal, NTimes
	};

private:
	ReduzerMPILeave();
	MPI::Intracomm* comm;
	/// sends results to root
	void send_jobresult(const SectorJobResult& result);
	/// receives a new job from root
	SectorJob recv_job();

	double time[NTimes];
	int root_rank;
};

/// manager for a parallelized reduction of a system of equations
template<class reduzer_type>
class ReduzerMPIRootBase: public reduzer_type {
public:
	ReduzerMPIRootBase(const ReduzerOptions& opts, MPI::Intracomm* comm,
			int jobcenter_rank);
	virtual ~ReduzerMPIRootBase();

	/// perform the reduction
	virtual void run();

	/// print timing measurements
	void print_times() const;

	/// commands to control leaves
	enum Cmd {
		CmdReduce, CmdFinish
	};

	/// a buffer for sending jobs and receiving results from leaves
	struct Buffer {
		Buffer();
		~Buffer();
		/// reallocate buf according to msg_len if necessary
		void realloc();
		size_t num_integrals, maxnum_integrals;
		size_t num_coefficients, maxnum_coefficients;
		INTIndex* integrals;
		char* coefficients;
		double time[ReduzerMPILeave::NTimes];
		MPI::Request requests[3];
	};

protected:
	ReduzerMPIRootBase();
	/// cleanup previous timings
	void clear_times();

private:
	/// communicator
	MPI::Intracomm* comm;
	/// rank of JobCenter
	int jobcenter_rank;

	std::queue<int> idle_leaves;
	std::set<int> all_leaves;

	bool want_more_workers;
	int num_workers_optimal;

	/// times in seconds, all are accumulative
	enum Times {
		TimeInsert, TimeFindJob, TimeWait, TimeCompute, NTimes
	};
	double time[NTimes];
	/// leave times gathered since the last JobCenter analysis
	double curr_leave_time[ReduzerMPILeave::NTimes];
	/// total leave times
	double leave_time[ReduzerMPILeave::NTimes];

	/// sends a job to a specific leave
	void send_job(const SectorJobLight& job, int leave, Buffer& b);
	/// receives results from a specific leave
	//void receive_results(int leave);
	/// starts receiving of results
	void recv_jobresult_begin(int from, Buffer& buf);
	/// inserts received results results
	SectorJobLightResult recv_jobresult_end(int from, Buffer& buf);

	void execute_jobcenter_command(int cmd);
};

class ReduzerMemMPIRoot: public ReduzerMPIRootBase<ReduzerMem> {
public:
	ReduzerMemMPIRoot(const ReduzerOptions& opts, MPI::Intracomm* comm,
			int jobcenter_rank) :
			ReduzerMPIRootBase<ReduzerMem>(opts, comm, jobcenter_rank) {
	}
	virtual ~ReduzerMemMPIRoot() {
	}

private:
	ReduzerMemMPIRoot() :
			ReduzerMPIRootBase<ReduzerMem>() {
	}
};

#ifdef HAVE_DATABASE
class ReduzerDbMPIRoot: public ReduzerMPIRootBase<ReduzerDb> {
public:
	ReduzerDbMPIRoot(const ReduzerOptions& opts, MPI::Intracomm* comm,
			int jobcenter_rank) :
	ReduzerMPIRootBase<ReduzerDb> (opts, comm, jobcenter_rank) {
	}
	virtual ~ReduzerDbMPIRoot() {
	}

private:
	ReduzerDbMPIRoot() :
	ReduzerMPIRootBase<ReduzerDb> () {
	}
};
#endif // HAVE_DATABASE
} // namespace Reduze

#endif /* REDUZERMPI_H_ */

#endif // HAVE_MPI
