/*  topology.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "topology.h"
#include <fstream>
#include <sstream>
#include <cstdlib>
#include "functions.h"
#include "files.h"
#include "yamlutils.h"
#include "combinatorics.h"
#include "undirectedgraph.h"
#include "integralfamily.h"
#include "triconnected.h"

using namespace std;
using GiNaC::ex;
using GiNaC::symbol;
using GiNaC::lst;

namespace Reduze {

// construction and destruction

Topology::Topology() {
}

Topology::~Topology() {
}

// general

void Topology::set_name(const string& name) {
	name_ = name;
}

std::string Topology::name() const {
	return name_;
}

void Topology::swap(Topology& other, bool call_back) {
	name_.swap(other.name_);
	edges_.swap(other.edges_);
	nodes_.swap(other.nodes_);
	edges_of_node_.swap(other.edges_of_node_);
	if (call_back) {
		swapped();
		other.swapped();
	}
}

// edge and node basics

int Topology::next_edge_id() const {
	return (edges_.empty() ? 1 : (--edges_.end())->first + 1);
}

int Topology::insert_edge(const Edge& e) {
	if (edges_.find(e.id) != edges_.end())
		ABORT("edge " << e.id << " exists already");
	edges_.insert(make_pair(e.id, e));
	if (nodes_.insert(e.from).second)
		inserted_node(e.from);
	if (nodes_.insert(e.to).second)
		inserted_node(e.to);
	edges_of_node_[e.from].push_back(e);
	if (e.to != e.from)
		edges_of_node_[e.to].push_back(e);
	inserted_edge(e.id);
	return e.id;
}

int Topology::insert_edge(int from, int to) {
	return insert_edge(Edge(from, to, next_edge_id()));
}

void Topology::remove_edge(int edge_id) {
	map<int, Edge>::iterator eit = edges_.find(edge_id);
	if (eit == edges_.end())
		ABORT("no such edge with id " << edge_id << " to remove");
	Edge edge = eit->second;
	int node = edge.from;
	while (true) {
		list<Edge>& edges = edges_of_node_[node];
		for (list<Edge>::iterator e = edges.begin(); e != edges.end();)
			if (e->id == edge_id)
				edges.erase(e++);
			else
				++e;
		if (node == edge.to)
			break;
		node = edge.to;
	}
	edges_.erase(edge_id);
	removed_edge(edge_id);
}

void Topology::contract_edge(int edge_id, bool remove_greater_node) {
	map<int, Edge>::iterator eit = edges_.find(edge_id);
	if (eit == edges_.end())
		ABORT("no such edge with id " << edge_id << " to contract");
	const Edge& e = eit->second;
	if (remove_greater_node)
		identify_nodes(std::max(e.from, e.to), std::min(e.from, e.to));
	else
		identify_nodes(std::min(e.from, e.to), std::max(e.from, e.to));
	remove_edge(edge_id);
}

void Topology::contract_bridges() {
	list<Topology> bcc = biconnected_components();
	for (list<Topology>::const_iterator c = bcc.begin(); c != bcc.end(); ++c)
		if (c->num_edges() == 1 && //
				!is_external_edge(c->edges().begin()->first) && //
				!c->edges().begin()->second.is_self_loop())
			contract_edge(c->edges().begin()->first);
}

void Topology::remove_node(int node) {
	VERIFY(!nodes().empty());
	int last_id = *nodes().rbegin();
	map<int, int> newnodes; // new node id to use for a given edge
	map<int, int>::const_iterator nn;

	// update edges for 'node', edges by id
	list<Edge>& es = edges_of_node_[node];
	for (list<Edge>::const_iterator e = es.begin(); e != es.end(); ++e) {
		Edge& ee = edges_[e->id];
		int newnode = ++last_id;
		nodes_.insert(newnode);
		ee.from = (ee.from == node ? newnode : ee.from);
		ee.to = (ee.to == node ? newnode : ee.to);
		newnodes[ee.id] = newnode;
		edges_of_node_[newnode].push_back(ee);
	}

	// update edges of neighbor nodes
	set<int> nbnodes = neighbors(node);
	for (set<int>::const_iterator n = nbnodes.begin(); n != nbnodes.end(); ++n) {
		list<Edge>& es = edges_of_node_[*n];
		for (list<Edge>::iterator e = es.begin(); e != es.end(); ++e) {
			if ((nn = newnodes.find(e->id)) != newnodes.end()) {
				e->from = (e->from == node ? nn->second : e->from);
				e->to = (e->to == node ? nn->second : e->to);
			}
		}
	}

	// signal insertion of new nodes
	for (nn = newnodes.begin(); nn != newnodes.end(); ++nn)
		inserted_node(nn->second);

	nodes_.erase(node);
	removed_node(node);
}

void Topology::remove_isolated_node(int node) {
	const list<Edge>& es = edges_of_node(node);
	if (!es.empty())
		ABORT("Node " << node << " is not isolated.");
	nodes_.erase(node);
	edges_of_node_.erase(node);
}

void Topology::identify_nodes(int from, int to) {
	if (from == to)
		return;

	// update edges for 'to', edges by id
	list<Edge>& es = edges_of_node_[from];
	for (list<Edge>::const_iterator e = es.begin(); e != es.end(); ++e) {
		Edge& ee = edges_[e->id];
		bool new_adj = !(ee.from == to || ee.to == to);
		ee.from = (ee.from == from ? to : ee.from);
		ee.to = (ee.to == from ? to : ee.to);
		if (new_adj)
			edges_of_node_[to].push_back(ee);
	}

	// update edges of neighbor nodes
	set<int> nnodes = neighbors(from);
	for (set<int>::const_iterator n = nnodes.begin(); n != nnodes.end(); ++n) {
		list<Edge>& es = edges_of_node_[*n];
		for (list<Edge>::iterator e = es.begin(); e != es.end(); ++e) {
			e->from = (e->from == from ? to : e->from);
			e->to = (e->to == from ? to : e->to);
		}
	}

	edges_of_node_.erase(from);
	nodes_.erase(from);

	removed_node(from);
}

void Topology::reverse_edge(int edge_id) {
	Edge& e = edges_[edge_id];
	e.reverse();
	list<Edge>& f = edges_of_node_[e.from];
	for (list<Edge>::iterator e = f.begin(); e != f.end(); ++e)
		if (e->id == edge_id) {
			e->reverse();
			break;
		}
	list<Edge>& t = edges_of_node_[e.to];
	for (list<Edge>::iterator e = t.begin(); e != t.end(); ++e)
		if (e->id == edge_id) {
			e->reverse();
			break;
		}
}

void Topology::remove_external_legs() {
	const list<Edge> ext = external_edges();
	for (list<Edge>::const_iterator e = ext.begin(); e != ext.end(); ++e) {
		bool from_is_external = is_external_node(e->from);
		bool to_is_external = is_external_node(e->to);
		remove_edge(e->id);
		if (from_is_external)
			remove_isolated_node(e->from);
		if (to_is_external)
			remove_isolated_node(e->to);
	}
}

void Topology::join_external_nodes() {
	const list<Edge> ext = external_edges();
	if (ext.empty())
		return;
	ASSERT(!nodes().empty());
	int n = std::max(1, *nodes().rbegin() + 1);
	for (list<Edge>::const_iterator e = ext.begin(); e != ext.end(); ++e) {
		Edge edge = *e; // copy
		if (is_external_node(e->from))
			edge.from = n;
		if (is_external_node(e->to))
			edge.to = n;
		remove_edge(e->id);
		if (edge.from != e->from)
			remove_isolated_node(e->from);
		if (edge.to != e->to)
			remove_isolated_node(e->to);
		insert_edge(edge);
	}
}

void Topology::insert_node(int node_id) {
	nodes_.insert(node_id);
	edges_of_node_[node_id];
	inserted_node(node_id);
}

void Topology::inserted_node(int) {
}

void Topology::inserted_edge(int) {
}

void Topology::removed_node(int) {
}

void Topology::removed_edge(int) {
}

void Topology::swapped() {
}

std::pair<unsigned int, unsigned int> Topology::cleave(int node,
		const std::set<Edge>& edges, int sign, int new_edge) {
	if (!contains_node(node))
		ABORT("Topology " << name_ << " doesn't contain node " << node);
	//int new_node = max(1, *nodes_.rbegin() + 1);
	// gives a nicer numbering
	int new_node = 1;
	while (nodes_.find(new_node) != nodes_.end())
		++new_node;

	list<Edge>& es = edges_of_node_[node];
	unsigned int count = 0;
	for (list<Edge>::iterator e = es.begin(); e != es.end();)
		if (edges.find(*e) != edges.end())
			es.erase(e++), ++count;
		else
			++e;

	if (count != edges.size())
		ABORT("Mismatch in given edges and edges connecting node " << node);

	if (sign > 0)
		insert_edge(Edge(new_node, node, new_edge));
	else
		insert_edge(Edge(node, new_node, new_edge));

	// update edges
	map<int, Edge>::iterator ma;
	for (set<Edge>::const_iterator m = edges.begin(); m != edges.end(); ++m) {
		if ((ma = edges_.find(m->id)) != edges_.end()) {
			if (ma->second.from == node)
				ma->second.from = new_node;
			else if (ma->second.to == node)
				ma->second.to = new_node;
			else
				ABORT("edge " << ma->first << " doesn't connect to node " << node);
		} else {
			ABORT("edge " << *m << " not contained in edges of topology " << name_);
		}
		edges_of_node_[new_node].push_back(*m);
	}

	// update edges of node
	for (map<int, list<Edge> >::iterator it = edges_of_node_.begin();
			it != edges_of_node_.end(); ++it) {
		for (list<Edge>::iterator e = it->second.begin(); e != it->second.end(); ++e) {
			if (edges.find(*e) != edges.end()) {
				if (e->from == node)
					e->from = new_node;
				if (e->to == node)
					e->to = new_node;
			}
		}
	}

	return make_pair(new_node, new_edge);
}

void Topology::remove_degree_2_nodes() {
	set<int> removed_edges;
	set<int> nodes = nodes_; // copy
	// reverse iteration to delete nodes with larger id first
	for (set<int>::const_reverse_iterator n = nodes.rbegin(); n != nodes.rend();
			++n) {
		if (!contains_node(*n))
			continue;
		const list<Edge>& edges = edges_of_node(*n);
		if (edges.size() != 2)
			continue;
		// e1 --- n --- e2  :  delete node n and edge with greater id
		Edge e1 = *edges.begin();
		Edge e2 = *edges.rbegin();
		if (e1.is_self_loop() || e2.is_self_loop())
			continue;
		if (is_external_edge(e1.id) && is_external_edge(e2.id)) {
			WARNING("Node " << *n << " with two external edges attached not removed.");
			continue;
		}
		Edge del = (e1.id < e2.id ? e2 : e1);
		ASSERT(*n == del.from || *n == del.to);
		bool delete_larger_node = (*n == std::max(del.from, del.to));
		contract_edge(del.id, delete_larger_node);
	}
}

Edge Topology::insert_degree_2_node(int edge_id) {
	Edge e = edge(edge_id);
	int node = std::max(1, *nodes_.rbegin() + 1);
	bool is_outgoing = is_external_node(e.to);
	remove_edge(edge_id);
	// keep original edge id on external legs
	if (is_outgoing) {
		int from = e.from;
		e.from = node;
		insert_edge(e);
		int id = insert_edge(from, node);
		return Edge(from, node, id);
	}
	int to = e.to;
	e.to = node;
	insert_edge(e);
	int id = insert_edge(node, to);
	return Edge(node, to, id);
}

// getters

unsigned int Topology::num_edges() const {
	return edges_.size();
}

unsigned int Topology::num_nodes() const {
	return nodes_.size();
}

unsigned int Topology::num_external_nodes() const {
	unsigned int count = 0;
	for (set<int>::const_iterator n = nodes_.begin(); n != nodes_.end(); ++n)
		if (is_external_node(*n))
			++count;
	return count;
}

unsigned int Topology::num_internal_edges() const {
	unsigned int count = 0;
	map<int, Edge>::const_iterator e;
	for (e = edges_.begin(); e != edges_.end(); ++e)
		if (!is_external_edge(e->first))
			++count;
	return count;
}

bool Topology::contains_node(int node) const {
	return nodes_.find(node) != nodes_.end();
}

bool Topology::contains_edge(int edge) const {
	return edges_.find(edge) != edges_.end();
}

bool Topology::is_external_edge(int edge_id) const {
	map<int, Edge>::const_iterator it = edges_.find(edge_id);
	if (it == edges_.end())
		ABORT("Edge " << edge_id << " does not exist");
	return (is_external_node(it->second.from)
					|| is_external_node(it->second.to));
}

bool Topology::is_external_node(int node) const {
	const list<Edge>& es = edges_of_node(node);
	ASSERT(nodes_.find(node) != nodes_.end());
	if (es.size() != 1)
		return false;
	return !es.begin()->is_self_loop();
}

const std::set<int>& Topology::nodes() const {
	return nodes_;
}

const std::map<int, Edge>& Topology::edges() const {
	return edges_;
}

const list<Edge>& Topology::edges_of_node(int node) const {
	map<int, list<Edge> >::const_iterator i = edges_of_node_.find(node);
	if (i == edges_of_node_.end())
		ABORT("no such node " << node);
	return i->second;
}

const map<int, list<Edge> >& Topology::edges_of_nodes() const {
	return edges_of_node_;
}

const Edge& Topology::edge(int edge_id) const {
	map<int, Edge>::const_iterator it = edges_.find(edge_id);
	if (it == edges_.end())
		ABORT("Edge " << edge_id << " does not exist");
	return it->second;
}

std::list<Edge> Topology::find_edges_between(int node1, int node2) const {
	list<Edge> res;
	for (map<int, Edge>::const_iterator e = edges_.begin(); e != edges_.end(); ++e)
		if (e->second.is_between(node1, node2))
			res.push_back(e->second);
	return res;
}

std::set<int> Topology::neighbors(int node) const {
	set<int> neighbors;
	list<Edge> es = edges_of_node(node);
	for (list<Edge>::const_iterator e = es.begin(); e != es.end(); ++e) {
		neighbors.insert(e->from);
		neighbors.insert(e->to);
	}
	neighbors.erase(node);
	return neighbors;
}

std::set<int> Topology::internal_nodes() const {
	set<int> res;
	set<int>::iterator pos = res.begin();
	for (set<int>::const_iterator n = nodes_.begin(); n != nodes_.end(); ++n)
		if (!is_external_node(*n))
			pos = res.insert(pos, *n);
	return res;
}

std::set<int> Topology::external_nodes() const {
	set<int> res;
	set<int>::iterator pos = res.begin();
	for (set<int>::const_iterator n = nodes_.begin(); n != nodes_.end(); ++n)
		if (is_external_node(*n))
			pos = res.insert(pos, *n);
	return res;
}

std::list<Edge> Topology::internal_edges() const {
	list<Edge> result;
	map<int, Edge>::const_iterator e;
	for (e = edges_.begin(); e != edges_.end(); ++e)
		if (!is_external_edge(e->first))
			result.push_back(e->second);
	return result;
}

std::list<Edge> Topology::external_edges() const {
	list<Edge> result;
	map<int, Edge>::const_iterator e;
	for (e = edges_.begin(); e != edges_.end(); ++e)
		if (is_external_edge(e->first))
			result.push_back(e->second);
	return result;
}

std::set<int> Topology::find_loop_edges() const {
	set<int> inter;
	one_pi_components(inter);
	set<int> result;
	map<int, Edge>::const_iterator e;
	for (e = edges_.begin(); e != edges_.end(); ++e)
		if (!is_external_edge(e->first) && inter.find(e->first) == inter.end())
			result.insert(e->first);
	return result;
}

std::pair<std::set<int>, std::set<int> > Topology::spanning_tree_and_cycles(
		int node) const {
	set<int> tree, cycles; // edges: spanning tree, cycle
	set<int> done, todo; // nodes: visited, yet to visit

	for (todo.insert(node); !todo.empty();) {
		int n = *todo.begin();
		list<Edge> edges = edges_of_node(n);
		for (list<Edge>::const_iterator e = edges.begin(); e != edges.end();
				++e) {
			if (tree.find(e->id) != tree.end()
					|| cycles.find(e->id) != cycles.end())
				continue;
			int next = e->opposite(n);
			if (done.find(next) == done.end()
					&& todo.find(next) == todo.end()) {
				todo.insert(next);
				tree.insert(e->id);
			} else {
				cycles.insert(e->id);
			}
		}
		todo.erase(n);
		done.insert(n);
	}
	return make_pair(tree, cycles);
}

std::set<int> Topology::find_cycles() const {
	set<int> cycles;
	const list<Topology> comp = connected_components();
	for (list<Topology>::const_iterator t = comp.begin(); t != comp.end();
			++t) {
		ASSERT(!t->nodes_.empty());
		int node = *t->nodes_.begin();
		pair<set<int>, set<int> > tmp = t->spanning_tree_and_cycles(node);
		cycles.insert(tmp.second.begin(), tmp.second.end());
	}
	// compare size of cycles with alternative calculation of first Betti number
	unsigned int nc = num_edges() - num_nodes() + comp.size();

	/*cout << "topo   " << name_ << endl;
	 cout << "cycles " << cycles.size() << "\n";
	 cout << "edges  " << num_edges() << "\n";
	 cout << "nodes  " << num_nodes() << "\n";
	 cout << "cc     " << connected_components().size() << "\n";
	 cout << "nc     " << nc << "\n";
	 */
	ASSERT(cycles.size() == nc);
	return cycles;
}

// topological analysis

bool Topology::is_connected() const {
	return connected_components().size() == 1;
}

bool Topology::is_one_particle_irreducible() const {
	set<int> edges;
	return one_pi_components(edges).size() == 1;
}

Topology Topology::connected_component(int node, int disabled_edge) const {
	Topology t;
	if (!contains_node(node))
		return t;
	t.insert_node(node); // for special case of node without edges
	set<int> todo_nodes, visited_nodes;
	for (todo_nodes.insert(node); !todo_nodes.empty();) {
		int n = *todo_nodes.begin();
		list<Edge> edges = edges_of_node(n);
		for (list<Edge>::const_iterator e = edges.begin(); e != edges.end();
				++e) {
			if (e->id == disabled_edge)
				continue;
			int next = e->opposite(n);
			if (visited_nodes.find(next) == visited_nodes.end())
				todo_nodes.insert(next);
			if (!t.contains_edge(e->id))
				t.insert_edge(*e);
		}
		todo_nodes.erase(n);
		visited_nodes.insert(n);
	}
	return t;
}

list<Topology> Topology::connected_components() const {
	list<Topology> comps;
	set<int> nodes = nodes_;
	while (!nodes.empty()) {
		int node = *nodes.begin();
		nodes.erase(nodes.begin());
		Topology t = connected_component(node);
		set<int>::const_iterator it;
		for (it = t.nodes_.begin(); it != t.nodes_.end(); ++it)
			nodes.erase(*it);
		comps.push_back(t);
	}
	return comps;
}

/*
 string indent(int n) {
 string s;
 for (int i = 0; i < n; ++i)
 s += ' ';
 return s;
 }
 */

// Implementation might be included in DFSPalmTree some day. For now, we keep
// it separate, since DFSPalmTree works only for restricted edge/node numbers.
// NOTE: unlike in the original algorithm, we use original node ids in lowpt
// (and not the positions in DFS !)
void find_biconnected(const Topology& topo, //
		int v, // current node
		set<int>& todo, // unvisited nodes (separate variable for disconn. comps)
		map<int, int>& pos, // position wrt DFS search
		map<int, int>& lowpt, // node with lowest position reachable by second path
		map<int, bool>& is_arc, // true if edge is tree arc, false if frond
		stack<Edge>& edges, list<Topology>& comps, // return biconnected components
		set<int>* cuts // return cut nodes if non-zero
		) {
	todo.erase(v);
	int num_numbers = pos.size();
	pos[v] = num_numbers + 1;
	lowpt[v] = v;

	//cout << indent(pos[v]) << "|" << v << "| visit node\n";
	int num_chops = 0; // number of components chopped off current node
	const list<Edge>& el = topo.edges_of_node(v);
	for (list<Edge>::const_iterator e = el.begin(); e != el.end(); ++e) {
		if (is_arc.find(e->id) != is_arc.end()) // need one visit only
			continue;
		//cout << indent(pos[v]) << "visiting edge " << *e << "\n";
		edges.push(*e);
		int w = e->opposite(v); // son of v
		if (pos.find(w) == pos.end()) { // w is new node (edge: tree arc)
			is_arc[e->id] = true; // TreeArc
			find_biconnected(topo, w, todo, pos, lowpt, is_arc, edges, comps,
					cuts);
			lowpt[v] = (pos[lowpt[w]] < pos[lowpt[v]] ? lowpt[w] : lowpt[v]);
		} else { // w is old node (edge: palm frond)
			is_arc[e->id] = false; // PalmFrond
			lowpt[v] = (pos[w] < pos[lowpt[v]] ? w : lowpt[v]);
		}
		if ((w == v) // self-loop
		|| (is_arc[e->id] && lowpt[w] == w) // single edge
				|| (is_arc[e->id] && lowpt[w] == v) // loop bcc
				) {
			// found cut node or root_node with last component
			bool is_root_node = (pos[v] == 1);
			if (cuts != 0 && !(is_root_node && num_chops == 0))
				cuts->insert(v); // is really a cut node
			// cout << indent(pos[v]) << "CHOP OFF COMPONENT ! at " << v << endl;
			// chop off a component
			Topology comp;
			while (true) {
				ASSERT(!edges.empty());
				Edge e = edges.top();
				edges.pop();
				//cout << indent(pos[v]) << "inserting " << e << endl;
				comp.insert_edge(e);
				if (e.from == v && e.to == v) // self-loop (frond)
					break;
				if ((e.from == v || e.to == v) && is_arc[e.id]) // first arc
					break;
			}
			++num_chops;
			comps.push_back(comp);
		}
	}
	//cout << indent(pos[v]) << "done with " << v << "\n";
}

list<Topology> Topology::biconnected_components(set<int>* cut_nodes) const {
	list<Topology> bcc;
	set<int> todo = nodes_;
	while (!todo.empty()) {
		map<int, int> pos, lowpt;
		map<int, bool> edge_type;
		stack<Edge> edges;
		find_biconnected(*this, *todo.begin(), todo, pos, lowpt, edge_type,
				edges, bcc, cut_nodes);
	}
	/*
	 LOGXX("found " << bcc.size() << " biconnected components");
	 int count = 0;
	 for (list<Topology>::iterator b = bcc.begin(); b != bcc.end(); ++b, ++count)
	 b->print_postscript("bcc" + to_string(count) + ".ps");
	 */
	return bcc;
}

list<Topology> Topology::biconnected_vacuum_components(
		list<Topology>& other) const {
	set<int> cutset;
	list<Topology> bcclist = biconnected_components(&cutset);
	LOGXX("found " << bcclist.size() << " biconnected components");
	vector<Topology> bcc(bcclist.begin(), bcclist.end()); // biconnect. comps
	vector<set<int> > cuts(bcc.size()); // connecting nodes for each comp
	map<int, set<int> > comps; // comps for each cut node
	list<int> vtodo; // components identified as vacuum but not processed yet
	// find cut and external nodes of biconnected components
	for (size_t c = 0; c < bcc.size(); ++c) {
		const set<int>& ns = bcc[c].nodes();
		for (set<int>::const_iterator n = ns.begin(); n != ns.end(); ++n)
			if (is_external_node(*n)) {
				cuts[c].insert(*n); // like cut node (never removed)
			} else if (cutset.find(*n) != cutset.end()) {
				cuts[c].insert(*n);
				comps[*n].insert(c);
			}
		if (cuts[c].size() <= 1)
			vtodo.push_back(c);
	}

	const map<int, set<int> > comps_backup(comps); // copy

	// ignore vacuum component in other components
	while (!vtodo.empty()) {
		size_t v = vtodo.front();
		vtodo.pop_front();
		if (cuts[v].size() == 1) { // connected comp., update rest
			int n = *cuts[v].begin(); // cut node
			comps[n].erase(v); // ignore v for cut node
			if (comps[n].size() == 1) { // one comp. left at n
				int next = *comps[n].begin();
				cuts[next].erase(n); // ignore n
				if (cuts[next].size() <= 1) // next is vacuum
					vtodo.push_back(next);
			}
		}
	}

	// assume: vacuum components have 0 or 1 cut node

	int lauf = std::max(1, *nodes_.rbegin() + 1);
	map<int, map<int, int> > rename; // nodes to be renamed for each vacuum component
	for (map<int, set<int> >::const_iterator c = comps_backup.begin();
			c != comps_backup.end(); ++c) {
		for (set<int>::const_iterator s = c->second.begin();
				s != c->second.end(); ++s)
			if (cuts[*s].size() <= 1)
				rename[*s].insert(make_pair(c->first, lauf++));
	}

	list<Topology> vacuum;
	for (size_t c = 0; c < bcc.size(); ++c) {
		if (cuts[c].size() > 1) {
			other.push_back(bcc[c]);
		} else {
			vacuum.push_back(Topology());
			Topology& vac = vacuum.back();
			const map<int, Edge>& es = bcc[c].edges();
			const map<int, int>& re = rename[c];
			for (map<int, Edge>::const_iterator e = es.begin(); e != es.end();
					++e) {
				Edge ed = e->second;
				map<int, int>::const_iterator r1 = re.find(ed.from);
				map<int, int>::const_iterator r2 = re.find(ed.to);
				if (r1 != re.end())
					ed.from = r1->second;
				if (r2 != re.end())
					ed.to = r2->second;
				vac.insert_edge(ed);
			}
		}
	}

	LOGXX("found " << vacuum.size() << " biconnected vacuum components");
	return vacuum;
}

list<Topology> Topology::one_pi_components(set<int>& inter_edges) const {
	list<Topology> result;
	if (!is_connected()) { // disconnected
		list<Topology> c = connected_components();
		for (list<Topology>::const_iterator t = c.begin(); t != c.end(); ++t) {
			list<Topology> l = t->one_pi_components(inter_edges);
			result.splice(result.end(), l);
		}
	} else { // connected
		Topology t1;
		map<int, Edge>::const_iterator eit;
		for (eit = edges_.begin(); eit != edges_.end(); ++eit) {
			if (is_external_edge(eit->first))
				continue;
			t1 = connected_component(eit->second.from, eit->first);
			if (!t1.contains_node(eit->second.to)) // found reduction
				break;
		}
		if (eit != edges_.end()) { // one particle reducible
			Topology t2 = connected_component(eit->second.to, eit->first);
			list<Topology> l1 = t1.one_pi_components(inter_edges);
			list<Topology> l2 = t2.one_pi_components(inter_edges);
			inter_edges.insert(eit->first);
			result.splice(result.end(), l1);
			result.splice(result.end(), l2);
		} else { // 1pi
			result.push_back(*this);
		}
	}
	return result;
}

int Topology::num_loop_edges() const {
	set<int> dummy;
	std::list<Topology> comp = one_pi_components(dummy);
	int num = 0;
	for (list<Topology>::const_iterator c = comp.begin(); c != comp.end(); ++c)
		num += c->num_internal_edges();
	return num;
}

void reverse_all_edges(Topology& t) {
	map<int, Edge> es = t.edges();
	for (map<int, Edge>::const_iterator e = es.begin(); e != es.end(); ++e)
		t.reverse_edge(e->first);
}

/// merges external nodes into one, returns new node id
int identify_external_nodes(Topology& t) {
	VERIFY(!t.nodes().empty());
	int new_node = std::max(1, *t.nodes().rbegin() + 1);
	t.insert_node(new_node);
	set<int> ns = t.nodes();
	for (set<int>::const_iterator n = ns.begin(); n != ns.end(); ++n)
		if (t.is_external_node(*n))
			t.identify_nodes(*n, new_node);
	return new_node;
}

std::pair<std::list<std::map<int, int> >, edge_attributes> Topology::get_edge_coloring() const {
	list<map<int, int> > edge_coloring;
	edge_attributes eas;
	return make_pair(edge_coloring, eas);
}

std::list<std::map<int, int> > Topology::get_node_coloring(
		bool permute_external_nodes) const {

	list<map<int, int> > result;
	const int num_ext_nodes = num_external_nodes();
	if (permute_external_nodes || num_ext_nodes == 0)
		return result;

	result.push_back(map<int, int>());
	int lauf = -num_ext_nodes;
	set<int>::const_iterator n;
	for (n = nodes_.begin(); n != nodes_.end(); ++n) {
		if (is_external_node(*n))
			result.back()[*n] = lauf++;
		else
			result.back()[*n] = 0;
	}
	return result;;
}

// output

void Topology::print_dot(const string& filename) const {
	ofstream of(filename.c_str());
	of << "graph " << (name_ != "graph" ? name_ : "graph0") << " {\n";
	map<int, Edge>::const_iterator it;
	for (it = edges_.begin(); it != edges_.end(); ++it) {
		const Edge& e = it->second;
		of << e.from << " -- " << e.to << " [label=\"" << it->first
				<< "\",fontsize=10];\n";
	}
	// nodes: just the label
	for (set<int>::const_iterator n = nodes_.begin(); n != nodes_.end(); ++n)
		of << *n << " [fontcolor=\"blue\",color=blue];\n";

	// problem with labels for multiple edges using neato:
	//		of << *n
	//				<< " [fontcolor=\"blue\",shape=circle,style=filled,color=blue,fixedsize=true,width=0,height=0];\n";
	//,shape=circle,width=0.15,height=0.15,color=\"blue\""
	of << "}\n";
	of.close();
}

void Topology::print_postscript(const string& filename) const {
	string dotfilename = filename + ".dot";
	print_dot(dotfilename);
	stringstream ss;
	ss << "neato -Tps " << dotfilename << " -o " << filename;
	int ret = system(ss.str().c_str());
	VERIFY(ret == 0);
}

YAML::Emitter& operator<<(YAML::Emitter& ye, const Topology& t) {
	using namespace YAML;
	ye << BeginMap;
	ye << Key << "name" << Value << t.name();
	ye << Key << "edges" << Value << Flow << BeginSeq;
	const map<int, Edge>& es = t.edges();
	for (map<int, Edge>::const_iterator e = es.begin(); e != es.end(); ++e) {
		ye << BeginSeq;
		ye << e->second.from << e->second.to << e->second.id;
		ye << EndSeq;
	}
	ye << EndSeq;
	ye << Key << "nodes" << Value << Flow << BeginSeq;
	const set<int>& node = t.nodes();
	for (set<int>::const_iterator n = node.begin(); n != node.end(); ++n)
		ye << *n;
	ye << EndSeq;

	ye << EndMap;

	return ye;
}

std::ostream& operator<<(std::ostream& os, const Topology& t) {
	YAML::Emitter ye;
	ye << t;
	os << ye.c_str();
	return os;
}

} // namespace Reduze

