class ExpressionResolver {
    hooks;
    // Per-file symbol table for statically analyzable variables.
    // Maps variableName -> either:
    //  - string[] (possible string values)
    //  - Record<string, string> (object of static string properties)
    variableTable = new Map();
    // Shared (cross-file) table for enums / exported object maps that should persist
    sharedEnumTable = new Map();
    constructor(hooks) {
        this.hooks = hooks;
    }
    /**
     * Clear per-file captured variables. Enums / shared maps are kept.
     */
    resetFileSymbols() {
        this.variableTable.clear();
    }
    /**
     * Capture a VariableDeclarator node to record simple statically analyzable
     * initializers (string literals, object expressions of string literals,
     * template literals and simple concatenations).
     *
     * This is called during AST traversal before deeper walking so later
     * identifier/member-expression usage can be resolved.
     *
     * @param node - VariableDeclarator-like node (has .id and .init)
     */
    captureVariableDeclarator(node) {
        try {
            if (!node || !node.id || !node.init)
                return;
            // only handle simple identifier bindings like `const x = ...`
            if (node.id.type !== 'Identifier')
                return;
            const name = node.id.value;
            const init = node.init;
            // ObjectExpression -> map of string props
            if (init.type === 'ObjectExpression' && Array.isArray(init.properties)) {
                const map = {};
                for (const p of init.properties) {
                    if (!p || p.type !== 'KeyValueProperty')
                        continue;
                    const keyNode = p.key;
                    const keyName = keyNode?.type === 'Identifier' ? keyNode.value : keyNode?.type === 'StringLiteral' ? keyNode.value : undefined;
                    if (!keyName)
                        continue;
                    const valExpr = p.value;
                    const vals = this.resolvePossibleStringValuesFromExpression(valExpr);
                    // Only capture properties that we can statically resolve to a single string.
                    if (vals.length === 1) {
                        map[keyName] = vals[0];
                    }
                }
                // If at least one property was resolvable, record the partial map.
                if (Object.keys(map).length > 0) {
                    this.variableTable.set(name, map);
                    return;
                }
            }
            // For other initializers, try to resolve to one-or-more strings
            const vals = this.resolvePossibleStringValuesFromExpression(init);
            if (vals.length > 0) {
                this.variableTable.set(name, vals);
            }
        }
        catch {
            // be silent - conservative only
        }
    }
    /**
     * Capture a TypeScript enum declaration so members can be resolved later.
     * Accepts SWC node shapes like `TsEnumDeclaration` / `TSEnumDeclaration`.
     *
     * Enums are stored in the shared table so they are available across files.
     */
    captureEnumDeclaration(node) {
        try {
            if (!node || !node.id || !Array.isArray(node.members))
                return;
            const name = node.id.type === 'Identifier' ? node.id.value : undefined;
            if (!name)
                return;
            const map = {};
            for (const m of node.members) {
                if (!m || !m.id)
                    continue;
                const keyNode = m.id;
                const memberName = keyNode.type === 'Identifier' ? keyNode.value : keyNode.type === 'StringLiteral' ? keyNode.value : undefined;
                if (!memberName)
                    continue;
                const init = m.init ?? m.initializer;
                if (init && init.type === 'StringLiteral') {
                    map[memberName] = init.value;
                }
            }
            if (Object.keys(map).length > 0) {
                this.sharedEnumTable.set(name, map);
            }
        }
        catch {
            // noop
        }
    }
    /**
     * Resolves an expression to one or more possible context string values that can be
     * determined statically from the AST. This is a wrapper around the plugin hook
     * `extractContextFromExpression` and {@link resolvePossibleStringValuesFromExpression}.
     *
     * @param expression - The SWC AST expression node to resolve
     * @returns An array of possible context string values that the expression may produce.
     */
    resolvePossibleContextStringValues(expression) {
        const strings = this.hooks.resolvePossibleContextStringValues?.(expression) ?? [];
        return [...strings, ...this.resolvePossibleStringValuesFromExpression(expression)];
    }
    /**
     * Resolves an expression to one or more possible key string values that can be
     * determined statically from the AST. This is a wrapper around the plugin hook
     * `extractKeysFromExpression` and {@link resolvePossibleStringValuesFromExpression}.
     *
     * @param expression - The SWC AST expression node to resolve
     * @returns An array of possible key string values that the expression may produce.
     */
    resolvePossibleKeyStringValues(expression) {
        const strings = this.hooks.resolvePossibleKeyStringValues?.(expression) ?? [];
        return [...strings, ...this.resolvePossibleStringValuesFromExpression(expression)];
    }
    /**
     * Resolves an expression to one or more possible string values that can be
     * determined statically from the AST.
     *
     * Supports:
     * - StringLiteral -> single value (filtered to exclude empty strings for context)
     * - NumericLiteral -> single value
     * - BooleanLiteral -> single value
     * - ConditionalExpression (ternary) -> union of consequent and alternate resolved values
     * - TemplateLiteral -> union of all possible string values
     * - The identifier `undefined` -> empty array
     *
     * For any other expression types (identifiers, function calls, member expressions,
     * etc.) the value cannot be determined statically and an empty array is returned.
     *
     * @param expression - The SWC AST expression node to resolve
     * @param returnEmptyStrings - Whether to include empty strings in the result
     * @returns An array of possible string values that the expression may produce.
     */
    resolvePossibleStringValuesFromExpression(expression, returnEmptyStrings = false) {
        // Support selector-style arrow functions used by the selector API:
        // e.g. ($) => $.path.to.key  ->  'path.to.key'
        if (expression.type === 'ArrowFunctionExpression') {
            try {
                let body = expression.body;
                // Handle block body with return statement
                if (body.type === 'BlockStatement') {
                    const returnStmt = body.stmts.find((s) => s.type === 'ReturnStatement');
                    if (returnStmt?.type === 'ReturnStatement' && returnStmt.argument) {
                        body = returnStmt.argument;
                    }
                    else {
                        return [];
                    }
                }
                let current = body;
                const parts = [];
                while (current && current.type === 'MemberExpression') {
                    const prop = current.property;
                    if (prop.type === 'Identifier') {
                        parts.unshift(prop.value);
                    }
                    else if (prop.type === 'Computed' && prop.expression && prop.expression.type === 'StringLiteral') {
                        parts.unshift(prop.expression.value);
                    }
                    else {
                        return [];
                    }
                    current = current.object;
                }
                if (parts.length > 0) {
                    return [parts.join('.')];
                }
            }
            catch {
                return [];
            }
        }
        if (expression.type === 'StringLiteral') {
            // Filter out empty strings as they should be treated as "no context" like i18next does
            return expression.value || returnEmptyStrings ? [expression.value] : [];
        }
        if (expression.type === 'ConditionalExpression') { // This is a ternary operator
            const consequentValues = this.resolvePossibleStringValuesFromExpression(expression.consequent, returnEmptyStrings);
            const alternateValues = this.resolvePossibleStringValuesFromExpression(expression.alternate, returnEmptyStrings);
            return [...consequentValues, ...alternateValues];
        }
        if (expression.type === 'Identifier' && expression.value === 'undefined') {
            return []; // Handle the `undefined` case
        }
        if (expression.type === 'TemplateLiteral') {
            return this.resolvePossibleStringValuesFromTemplateString(expression);
        }
        // MemberExpression: try to resolve object identifier to an object map in the symbol table
        if (expression.type === 'MemberExpression') {
            try {
                const obj = expression.object;
                const prop = expression.property;
                // only handle simple identifier base + simple property (Identifier or computed StringLiteral)
                if (obj.type === 'Identifier') {
                    const baseVar = this.variableTable.get(obj.value);
                    const baseShared = this.sharedEnumTable.get(obj.value);
                    const base = baseVar ?? baseShared;
                    if (base && typeof base !== 'string' && !Array.isArray(base)) {
                        let propName;
                        if (prop.type === 'Identifier')
                            propName = prop.value;
                        else if (prop.type === 'Computed' && prop.expression?.type === 'StringLiteral')
                            propName = prop.expression.value;
                        if (propName && base[propName] !== undefined) {
                            return [base[propName]];
                        }
                    }
                }
            }
            catch { }
        }
        // Binary concatenation support (e.g., a + '_' + b)
        // SWC binary expr can be represented as `BinExpr` with left/right; be permissive:
        if (expression.left && expression.right) {
            try {
                const exprAny = expression;
                const leftNode = exprAny.left;
                const rightNode = exprAny.right;
                // Detect explicit binary concatenation (plus) nodes and only then produce concatenated combos.
                const isBinaryConcat = 
                // SWC older shape: BinExpr with op === '+'
                (exprAny.type === 'BinExpr' && exprAny.op === '+') ||
                    // Standard AST: BinaryExpression with operator === '+'
                    (exprAny.type === 'BinaryExpression' && exprAny.operator === '+') ||
                    // Fallbacks
                    exprAny.operator === '+' || exprAny.op === '+';
                if (isBinaryConcat) {
                    const leftVals = this.resolvePossibleStringValuesFromExpression(leftNode, returnEmptyStrings);
                    const rightVals = this.resolvePossibleStringValuesFromExpression(rightNode, returnEmptyStrings);
                    if (leftVals.length > 0 && rightVals.length > 0) {
                        const combos = [];
                        for (const L of leftVals) {
                            for (const R of rightVals) {
                                combos.push(`${L}${R}`);
                            }
                        }
                        return combos;
                    }
                }
                // Handle logical nullish coalescing (a ?? b): result is either left (when not null/undefined) OR right.
                // Represent this conservatively as the union of possible left and right values.
                const isNullishCoalesce = 
                // SWC may emit as BinaryExpression with operator '??'
                (exprAny.type === 'BinaryExpression' && exprAny.operator === '??') ||
                    (exprAny.type === 'LogicalExpression' && exprAny.operator === '??') ||
                    exprAny.operator === '??' || exprAny.op === '??';
                if (isNullishCoalesce) {
                    const leftVals = this.resolvePossibleStringValuesFromExpression(leftNode, returnEmptyStrings);
                    const rightVals = this.resolvePossibleStringValuesFromExpression(rightNode, returnEmptyStrings);
                    if (leftVals.length > 0 || rightVals.length > 0) {
                        return Array.from(new Set([...leftVals, ...rightVals]));
                    }
                }
            }
            catch { }
        }
        if (expression.type === 'NumericLiteral' || expression.type === 'BooleanLiteral') {
            return [`${expression.value}`]; // Handle literals like 5 or true
        }
        // Support building translation keys for
        // `variable satisfies 'coaching' | 'therapy'`
        if (expression.type === 'TsSatisfiesExpression' || expression.type === 'TsAsExpression') {
            const annotation = expression.typeAnnotation;
            return this.resolvePossibleStringValuesFromType(annotation, returnEmptyStrings);
        }
        // Identifier resolution via captured per-file variable table only
        if (expression.type === 'Identifier') {
            const v = this.variableTable.get(expression.value);
            if (!v)
                return [];
            if (Array.isArray(v))
                return v;
            // object map - cannot be used directly as key, so return empty
            return [];
        }
        // We can't statically determine the value of other expressions (e.g., variables, function calls)
        return [];
    }
    resolvePossibleStringValuesFromType(type, returnEmptyStrings = false) {
        if (type.type === 'TsUnionType') {
            return type.types.flatMap((t) => this.resolvePossibleStringValuesFromType(t, returnEmptyStrings));
        }
        if (type.type === 'TsLiteralType') {
            if (type.literal.type === 'StringLiteral') {
                // Filter out empty strings as they should be treated as "no context" like i18next does
                return type.literal.value || returnEmptyStrings ? [type.literal.value] : [];
            }
            if (type.literal.type === 'TemplateLiteral') {
                return this.resolvePossibleStringValuesFromTemplateLiteralType(type.literal);
            }
            if (type.literal.type === 'NumericLiteral' || type.literal.type === 'BooleanLiteral') {
                return [`${type.literal.value}`]; // Handle literals like 5 or true
            }
        }
        // We can't statically determine the value of other expressions (e.g., variables, function calls)
        return [];
    }
    /**
     * Resolves a template literal string to one or more possible strings that can be
     * determined statically from the AST.
     *
     * @param templateString - The SWC AST template literal string to resolve
     * @returns An array of possible string values that the template may produce.
     */
    resolvePossibleStringValuesFromTemplateString(templateString) {
        // If there are no expressions, we can just return the cooked value
        if (templateString.quasis.length === 1 && templateString.expressions.length === 0) {
            // Ex. `translation.key.no.substitution`
            return [templateString.quasis[0].cooked || ''];
        }
        // Ex. `translation.key.with.expression.${x ? 'title' : 'description'}`
        const [firstQuasis, ...tails] = templateString.quasis;
        const stringValues = templateString.expressions.reduce((heads, expression, i) => {
            return heads.flatMap((head) => {
                const tail = tails[i]?.cooked ?? '';
                return this.resolvePossibleStringValuesFromExpression(expression, true).map((expressionValue) => `${head}${expressionValue}${tail}`);
            });
        }, [firstQuasis.cooked ?? '']);
        return stringValues;
    }
    /**
     * Resolves a template literal type to one or more possible strings that can be
     * determined statically from the AST.
     *
     * @param templateLiteralType - The SWC AST template literal type to resolve
     * @returns An array of possible string values that the template may produce.
     */
    resolvePossibleStringValuesFromTemplateLiteralType(templateLiteralType) {
        // If there are no types, we can just return the cooked value
        if (templateLiteralType.quasis.length === 1 && templateLiteralType.types.length === 0) {
            // Ex. `translation.key.no.substitution`
            return [templateLiteralType.quasis[0].cooked || ''];
        }
        // Ex. `translation.key.with.expression.${'title' | 'description'}`
        const [firstQuasis, ...tails] = templateLiteralType.quasis;
        const stringValues = templateLiteralType.types.reduce((heads, type, i) => {
            return heads.flatMap((head) => {
                const tail = tails[i]?.cooked ?? '';
                return this.resolvePossibleStringValuesFromType(type, true).map((expressionValue) => `${head}${expressionValue}${tail}`);
            });
        }, [firstQuasis.cooked ?? '']);
        return stringValues;
    }
}

export { ExpressionResolver };
