/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <stdlib.h>

#include <glib.h>

#include <libjconfig/jiterator.h>
#include <libjconfig/jconfig.h>

#include <librbc/rbcdisk.h>

#include "app.h"


#define MAX_LOGINS_MIN              1
#define MAX_LOGINS_MAX              64
#define MAX_LOGINS_DEFAULT          12

#define MAX_LOGINS_PER_PORT_MIN     1
#define MAX_LOGINS_PER_PORT_MAX     64
#define MAX_LOGINS_PER_PORT_DEFAULT 4

#define DISK_NUMBER_MIN             0
#define DISK_NUMBER_MAX             15

#define ORDER_MIN                   0
#define ORDER_MAX                   1048576


static gboolean load_global_options   (EndpointApp   *app,
				       JConfig       *cf);
static gboolean load_disks            (EndpointApp   *app,
				       JConfig       *cf);
static gboolean load_partitions       (EndpointApp   *app,
				       JConfig       *cf);

static gboolean get_int_attribute     (JConfigStanza *cfs,
				       const gchar   *name,
				       gint          *value,
				       gint           min,
				       gint           max);
static gboolean get_boolean_attribute (JConfigStanza *cfs,
                                       const gchar   *name,
				       gboolean      *value);


EndpointApp *
endpoint_app_new (const gchar *configfile)
{
  EndpointApp *app;
  JConfig     *cf;

  g_return_val_if_fail (configfile != NULL, NULL);

  cf = j_config_parse_file (configfile);

  if (!cf)
    return NULL;
 
  app = g_new0 (EndpointApp, 1);
 
  app->max_logins = MAX_LOGINS_DEFAULT;
  app->max_logins_per_port = MAX_LOGINS_PER_PORT_DEFAULT;
  app->honor_exclusive = FALSE;

  app->disks = g_hash_table_new_full (g_direct_hash, g_direct_equal,
				      NULL, (GDestroyNotify) rbc_disk_destroy);

  if (!load_global_options (app, cf) || !load_disks (app, cf))
    {
      endpoint_app_destroy (app);
      app = NULL;
    }

  j_config_free (cf);

  /* TODO: multiple LUN */
  if (app)
    {
      RBCDisk *disk;

      disk = g_hash_table_lookup (app->disks, GINT_TO_POINTER (0));

      if (disk == NULL || rbc_disk_get_type (disk) == RBC_DISK_VIRTUAL)
	{
	  endpoint_app_destroy (app);
	  app = NULL;
      }
    }

  return app;
}

void
endpoint_app_destroy (EndpointApp *app)
{
  g_return_if_fail (app != NULL);

  g_hash_table_destroy (app->disks);
  g_free (app);

  return;
}

#define CHECK(cond) G_STMT_START {	\
  if (!(cond))				\
    {					\
      ret = FALSE;			\
      break;				\
    }					\
} G_STMT_END

static gboolean
load_global_options (EndpointApp *app,
		     JConfig     *cf)
{
  JIterator     *iter;
  JConfigStanza *cfs;
  gboolean       ret = TRUE;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (cf != NULL, FALSE);

  iter = j_config_get_stanzas (cf, "global", NULL, 0);

  /* FIXME: hmm, allow only one global stanza? */
  while (j_iterator_has_more (iter))
    {
      cfs = j_iterator_get_next (iter);

      CHECK (get_int_attribute (cfs, "max_logins",
				&app->max_logins,
	                        MAX_LOGINS_MIN,
				MAX_LOGINS_MAX));
      CHECK (get_int_attribute (cfs, "max_logins_per_port",
				&app->max_logins_per_port,
	                        MAX_LOGINS_PER_PORT_MIN,
				MAX_LOGINS_PER_PORT_MAX));

      CHECK (get_boolean_attribute (cfs, "honor_exclusive_login",
	                            &app->honor_exclusive));
    }

  j_iterator_free (iter);

  if (ret && app->max_logins < app->max_logins_per_port)
    {
      /* ERROR MESSAGE */
      return FALSE;
    }

  return ret;
}

static gboolean
load_disks (EndpointApp *app,
            JConfig     *cf)
{
  JIterator     *iter;
  JConfigStanza *cfs;
  gchar         *type_string;
  RBCDiskType    type;
  gint           number;
  gchar         *filename;
  RBCDisk       *disk = NULL;
  gboolean       ret = TRUE;

  iter = j_config_get_stanzas (cf, "disk", NULL, 0);

  while (j_iterator_has_more (iter))
    {
      cfs = j_iterator_get_next (iter);

      disk = NULL;

      CHECK (type_string = j_config_get_attribute (cfs, "type"));

      if (g_ascii_strcasecmp (type_string, "physical") == 0)
	{
	  type = RBC_DISK_PHYSICAL;
	}
      else if (g_ascii_strcasecmp (type_string, "virtual") == 0)
	{
	  type = RBC_DISK_VIRTUAL;
	}
      else
	{
	  ret = FALSE;
	  break;
	}

      CHECK (get_int_attribute (cfs, "number",
	                        &number,
				DISK_NUMBER_MIN,
				DISK_NUMBER_MAX));

      disk = rbc_disk_new (type);

      if (type == RBC_DISK_PHYSICAL)
	{
	  CHECK (filename = j_config_get_attribute (cfs, "filename"));
	  CHECK (rbc_disk_set_filename (disk, filename));
	}

      /* FIXME: hmm, check for dupes? */
      g_hash_table_insert (app->disks, GINT_TO_POINTER (number), disk);
    }

  if (!ret)
    {
      if (disk)
	rbc_disk_destroy (disk);

      return FALSE;
    }
  else
    return load_partitions (app, cf);
}

static gboolean
load_partitions (EndpointApp *app,
                 JConfig     *cf)
{
  JIterator     *iter;
  JConfigStanza *cfs;
  gint           number;
  gchar         *filename;
  gint           order;
  RBCDisk       *disk;
  gboolean       ret = TRUE;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (cf != NULL, FALSE);

  iter = j_config_get_stanzas (cf, "partition", NULL, 0);

  while (j_iterator_has_more (iter))
    {
      cfs = j_iterator_get_next (iter);

      CHECK (get_int_attribute (cfs, "disk",
				&number,
	                        DISK_NUMBER_MIN,
				DISK_NUMBER_MAX));

      CHECK (filename = j_config_get_attribute (cfs, "filename"));

      CHECK (order = get_int_attribute (cfs, "order",
	                                &order,
					ORDER_MIN,
					ORDER_MAX));

      CHECK (disk = g_hash_table_lookup (app->disks, GINT_TO_POINTER (number)));

      CHECK (rbc_disk_get_type (disk) == RBC_DISK_VIRTUAL);

      rbc_disk_add_partition (disk, filename, order);
    }

  j_iterator_free (iter);

  return ret;
}

static gboolean
get_int_attribute (JConfigStanza *cfs,
		   const gchar   *name,
		   gint          *value,
		   gint           min,
		   gint           max)
{
  gchar    *buf, *end;
  glong     number;
  gboolean  ret = TRUE;

  g_return_val_if_fail (cfs != NULL, FALSE);
  g_return_val_if_fail (name != NULL, FALSE);
  g_return_val_if_fail (value != NULL, FALSE);

  buf = j_config_get_attribute (cfs, name);

  if (buf)
    {
      number = strtol (buf, &end, 0);

      if (*buf && !*end && number >= min && number <= max)
	*value = number;
      else
	ret = FALSE;
    }

  g_free (buf);

  return ret;
}

static gboolean
get_boolean_attribute (JConfigStanza *cfs,
		       const gchar   *name,
		       gboolean      *value)
{
  gchar    *buf;
  gboolean  ret = TRUE;

  g_return_val_if_fail (cfs != NULL, FALSE);
  g_return_val_if_fail (name != NULL, FALSE);
  g_return_val_if_fail (value != NULL, FALSE);

  buf = j_config_get_attribute (cfs, name);

  if (buf)
    {
      if ((g_ascii_strcasecmp (buf, "on") == 0) ||
	  (g_ascii_strcasecmp (buf, "true") == 0) ||
	  (g_ascii_strcasecmp (buf, "yes") == 0) ||
	  (g_ascii_strcasecmp (buf, "1") == 0))
	{
	  *value = TRUE;
	}
      else if ((g_ascii_strcasecmp (buf, "off") == 0) ||
	       (g_ascii_strcasecmp (buf, "false") == 0) ||
	       (g_ascii_strcasecmp (buf, "no") == 0) ||
	       (g_ascii_strcasecmp (buf, "0") == 0))
	{
	  *value = FALSE;
	}
      else
	ret = FALSE;
    }

  g_free (buf);

  return ret;
}
