#!/usr/bin/env bash
#
# Copyright 2016 The Bazel Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This script handles interface library generation for dynamic library
# link action.
#
# Bazel can be configured to generate external interface library script
# to generate interface libraries in CppLinkAction for dynamic libraries.
# This is not needed on Windows (as the "interface" libraries are
# generated by default). This script therefore handles the cases when
# external script is provided, or when no script should be used.

set -eu

E_LINKER_COMMAND_NOT_FOUND=12
E_INTERFACE_BUILDER_NOT_FOUND=13


SUFFIX=".rewritten"

other_args=""

if [[ "$#" -eq 1 ]]; then
  if [[ "$1" != @* ]]; then
    echo "Parameter file must start with @" 1>&2;
    exit "$E_LINKER_COMMAND_NOT_FOUND"
  fi

  filename=$(echo "$1" | cut -c2-)
  first_five_lines=$(head -n 5 $filename)

  # Should generate interface library switch (<yes|no>); if the value is "no",
  # following 3 args are ignored (but must be present)
  GENERATE_INTERFACE_LIBRARY=$(echo "$first_five_lines" | head -n1 | tail -n1)
  # Tool which can generate interface library from dynamic library file
  INTERFACE_LIBRARY_BUILDER=$(echo "$first_five_lines" | head -n2 | tail -n1)
  # Dynamic library from which we want to generate interface library
  DYNAMIC_LIBRARY=$(echo "$first_five_lines" | head -n3 | tail -n1)
  # Resulting interface library
  INTERFACE_LIBRARY=$(echo "$first_five_lines" | head -n4 | tail -n1)
  # The command used to generate the dynamic library
  LINKER_COMMAND=$(echo "$first_five_lines" | head -n5 | tail -n1)

  rest_of_lines=$(tail -n +6 $filename)
  new_param_file="${filename}${SUFFIX}"
  echo "$rest_of_lines" > $new_param_file
  other_args="@$new_param_file"

  if [[ ! -e "$LINKER_COMMAND" ]]; then
    echo "Linker command ($LINKER_COMMAND) not found." 1>&2;
    exit "$E_LINKER_COMMAND_NOT_FOUND"
  fi

  if [[ "no" == "$GENERATE_INTERFACE_LIBRARY" ]]; then
      INTERFACE_GENERATION=:
  else
      if [[ ! -e "$INTERFACE_LIBRARY_BUILDER" ]]; then
        echo "Interface library builder ($INTERFACE_LIBRARY_BUILDER)
        not found." 1>&2;
        exit "$E_INTERFACE_BUILDER_NOT_FOUND"
      fi
      INTERFACE_GENERATION="${INTERFACE_LIBRARY_BUILDER} ${DYNAMIC_LIBRARY}
      ${INTERFACE_LIBRARY}"
  fi

  ${LINKER_COMMAND} "$other_args" && ${INTERFACE_GENERATION}
else
  # TODO(b/113358321): Remove this branch once projects are migrated to not
  #  splitting the linking command line.
  # Should generate interface library switch (<yes|no>); if the value is "no",
  # following 3 args are ignored (but must be present)
  GENERATE_INTERFACE_LIBRARY="$1"
  # Tool which can generate interface library from dynamic library file
  INTERFACE_LIBRARY_BUILDER="$2"
  # Dynamic library from which we want to generate interface library
  DYNAMIC_LIBRARY="$3"
  # Resulting interface library
  INTERFACE_LIBRARY="$4"
  # The command used to generate the dynamic library
  LINKER_COMMAND="$5"
  shift 5
  if [[ ! -e "$LINKER_COMMAND" ]]; then
    echo "Linker command ($LINKER_COMMAND) not found." 1>&2;
    exit "$E_LINKER_COMMAND_NOT_FOUND"
  fi

  if [[ "no" == "$GENERATE_INTERFACE_LIBRARY" ]]; then
      INTERFACE_GENERATION=:
  else
      if [[ ! -e "$INTERFACE_LIBRARY_BUILDER" ]]; then
        echo "Interface library builder ($INTERFACE_LIBRARY_BUILDER)
        not found." 1>&2;
        exit "$E_INTERFACE_BUILDER_NOT_FOUND"
      fi
      INTERFACE_GENERATION="${INTERFACE_LIBRARY_BUILDER} ${DYNAMIC_LIBRARY}
      ${INTERFACE_LIBRARY}"
  fi

  ${LINKER_COMMAND} "$@" && ${INTERFACE_GENERATION}
fi
