/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

enum CutOff {
    MAX = 10000,
    MIN = -10000
}

struct BrainSpec {
    int  thinking_time;         // 1 .. MAX_TIME (seconds)
    int  thinking_depth;        // 1 .. MAX_MOVE
    int  discernment;           // 1 .. MaxDiscernment. Higher => can discern "winning" moves better
    public BrainSpec () {
        thinking_time = 1;
        thinking_depth = 2;
        discernment = 2;
    }
}

class Brain : Object {
    /**
     * After the non-blocking request_think(game, vp, out ticket), the client will be 
     * notified using this signal. The client should ensure that the ticket sent by the
     * signal matches the ticket issued by request_think.
     */
    public signal void got_move_signal (long ticket, Move move);

    /**
     * Sequence number issued by request_think and stamped on got_move_signal.
     */
    private long ticket = 0;

    /**
     * Used to enforce maximum thinking time.
     */
    private Timer timer = new Timer ();

    /**
     * Set by request_think
     */
    private BrainSpec brain_spec;

    /**
     * Set by request_think
     */
    private Game root;

    // Thread controls:
    private Thread<void*> thinking_thread;
    private Mutex mutex = Mutex ();
    private Cond cond = Cond ();

    /**
     * Set true to kill the brain.
     */
    private bool exiting;

    /**
     * Set true to abort current thinking.
     */
    private bool stopping;

    /**
     * Set true to start thinking.
     */
    private bool thinking;

    /**
     * The brain thinks using its own thread so that the GUI remains responsive.
     */
    public Brain () {
        exiting = false;
        stopping = false;
        thinking = false;
        thinking_thread = new Thread<void*> ("thinking", run);
    }

    ~Brain () {
        request_exit();
    }

    /**
     * Request the thinking thread to exit. Wait for the request to be complied.
     */
    public void request_exit () {
        exiting = true;
        stopping = true;
        mutex.@lock (); {
            thinking = false;
            cond.@signal ();
        } mutex.unlock ();
        thinking_thread.join ();
    }

    /**
     * Request the thinking thread to abandon on-going thinking. 
     * Wait for the request to be complied.
     */
    public void request_stop () {
        stopping = true;
        mutex.@lock (); {
            stopping = false;
        } mutex.unlock ();
    }

    /**
     * Request thinking on a new game configuration, killing any
     * on-going thinking prior to that.
     * This function returns its result via an out argument because the
     * result has to be set within the critical block. This alternative won't work:
     *    issued_ticket = request_think (game, vp)  <-- error
     * because the assignment is done outside the critical region.
     */
    public void request_think (Game game, BrainSpec spec, out long issued_ticked)
        requires (game.stage != Stage.GAME_OVER) {
        stopping = true;
        mutex.@lock (); {
            stopping = false;
            thinking = true;
            root = new Game.clone (game);
            issued_ticked = ++ticket;
            brain_spec = spec;
            cond.@signal ();
        } mutex.unlock ();
    }

    /**
     * The task for the thinking thread. To stop this thread, call request_exit.
     */
    private void* run () {
        while (! exiting) {
            mutex.@lock (); {
              if (thinking) {
                  Move response = think ();
                  if (!stopping) got_move_signal (ticket, response);
                  thinking = false;
              }
              if (!exiting) cond.wait (mutex);
            } mutex.unlock ();
        }//endwhile
        return null;
    }
          
    /**
     * Perform iterative deepening, and then inject some randomness to 
     * the move selection.
     */
    private Move think () {
        // Initialise according to player's characteristics
        var max_time = brain_spec.thinking_time;
        var max_ply  = brain_spec.thinking_depth * 2;
              
        // Initialise the root node
        root.generate_moves ();
        if (root.numMoves() == 1) return root.getMove (0);  // The only move
        foreach (Move move in root) {
            move.@value = CutOff.MIN;
        }
              
        // Perform iterative-deepening
        timer.reset ();
        for (int depth=1; depth < max_ply; depth++) {
            // Interrupted or out of time?
            if (stopping || timer.elapsed () > max_time) break;
            bool depth_reached = false;
            // The positions at ply 1 are FULLY evaluated. This is done so that
            // the random selection of moves will be based on the
            // actual value of each move (rather than the "at-most" value when
            // cut-off is applied).
            foreach (Move move in root) {
                Game game = new Game.clone(root);
                game.perform(move, true);
                int val = - search(1, depth, max_time, ref depth_reached,
                    game, CutOff.MIN, CutOff.MAX);
                // Out of time?
                if (stopping || timer.elapsed () > max_time) {
                    // move.value = the value from the previous shallow search
                    break;
                }
                else {
                    // Moves of equal values are ranked randomly.
                    move.@value = val * 10 + Random.int_range (0, 10);
                }
            }
            root.sort_moves ();
            // If the previous search iteration did not need 
            // that many plies, we are done.
            if (!depth_reached) break;
        }
        return select_move_randomly ();
    }
        
    /**
     * Search to the given depth. Return evaluation.
     * Side effect: depth_reached set to true if ply == depth.
     */
    private int search (int ply, int depth, int max_time, ref bool depth_reached,
                    Game game, int alpha, int beta) {
        // If stopping prematurely, return a dummy value
        if (stopping || timer.elapsed () > max_time) return 0;
        // If game over or max depth is reached, backtrack
        if (game.stage == Stage.GAME_OVER  ||  ply == depth) {
            if (ply == depth) depth_reached = true;
            return game.score[game.player] - game.score[1 - game.player];
        }

        foreach (Move move in game) {
            Game game2 = new Game.clone (game);
            game2.perform (move, true);  // Generate moves
            int val = - search (ply + 1, depth, max_time, ref depth_reached,
                                game2, -beta, -alpha);

            if (val > alpha) alpha = val;
            if (alpha > beta) break;  // prune
        }//endforeach
        return alpha;
    }

        
    /**
     * Provide some variability in the line of play.
     * Assume root's branches are sorted in descending values.
     */
    private Move select_move_randomly () {
        if (root.stage == Stage.OPENING) {
            // Reduce contrast between good and bad moves
            foreach (Move move in root) {
                move.@value /= 10;
            }
        }
        else {
            // Add further discernment to contrast good and bad moves
            int discernment = (brain_spec.discernment - 1) * 5;
            foreach (Move move in root) {
                if (move.@value > 0) move.@value += discernment;
            }
        }
        // "Translate" the values so that the smallest one is 1.
        // Calculate the total of the translated values.
        int smallest_value = root.getMove (root.numMoves () - 1).value;
        int total = 0;
        foreach (Move move in root) {
            total += move.@value - smallest_value + 1;
        }
        // Roll the dice
        int num_rolls = brain_spec.discernment;
        // In the early stage of the game, prefer variety rather than consistency
        if (root.stage == Stage.OPENING) num_rolls = 1;
        else if (root.stage == Stage.SELECT) num_rolls /= 2;
        int roll = Random.int_range (0, total);
        for (int i=1; i < num_rolls; i++) {
            int roll2 = Random.int_range (0, total);
            if (roll > roll2) roll = roll2;
        }
        // Look at the move selected by the best roll
        foreach (Move move in root) {
            roll -= move.@value - smallest_value + 1;
            if (roll < 0) return move;
        }
        return root.getMove (0);
    }

}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
