/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

struct Rgb {
    public double red;
    public double green;
    public double blue;
    public double alpha;
    public Rgb (double r, double g, double b, double a = 1) {
        red = r;
        green = g;
        blue = b;
        alpha = a;
    }

    public void set_source_for (Cairo.Context cr) {
        cr.set_source_rgba (red, green, blue, alpha);
    }

    public Rgb darken (double factor) {
        factor = 1 - factor;
        return Rgb (red * factor, green * factor, blue * factor, alpha);
    }

    public Rgb lighten (double factor) {
        if (factor < 0) return darken (-factor);
        return Rgb (red + (1 - red) * factor,
                     green + (1 - green) * factor,
                     blue + (1 - blue) * factor,
                     alpha);
    }
}

class Theme2DCairo : Theme2D {
    // Signal to indicate changes due to incremental async theme construction
    public signal void changed ();

    private string path = null;
    private int num_frames = 40;

    /**
     * Create an image for the table. The origin (0,0) for cr is the top-left
     * corner of the table (i.e. the left and top fillers have negative coordinates)
     */
    delegate void ForgeTableFunction (Cairo.Context cr);
    private ForgeTableFunction forge_table_function;

    /**
     * Create an image for frame f of the given piece.
     */
    delegate void ForgePieceFunction (Cairo.Context cr, Piece piece, int size, int f);
    private ForgePieceFunction forge_piece_function;

    /**
     * Principal colors for the themes
     */
    private Rgb spin_color = Rgb (0.2, 0.5, 0.1);
    private Rgb dizzy_color = Rgb (0.2, 0.5, 0.7);
    private Rgb tv_color = Rgb (0.7, 0.5, 0.2);
    private Rgb shiny_color = Rgb (0.6, 0.4, 0.3);
    private Rgb flower_color = Rgb (0.35, 0.3, 0.2);

    public Theme2DCairo () {
        add (_("Spin"), "0");
        add (_("Dizzy"), "dizzy");
        add (_("Nothing Good On TV"), "tv");
        add (_("Smooth and Shiny"), "shiny");
        add (_("Enchanted Flowers"), "flower");
        add (_("Galaxy"), "galaxy");
    }

    private void add (string name, string path) {
        var item = ThemeItem () {name=name, path=path, engine=this};
        select (path, 0, 0);

        // Set item.pixbuf
        var size = 48;
        var p = Piece.KAS_1;
        var image = new Cairo.ImageSurface (Cairo.Format.ARGB32, size, size);
        var cr = new Cairo.Context (image);
        cr.translate (size / 2, size / 2);
        forge_piece_function (cr, p, size, layout.kas_frame[p]);
        item.pixbuf = Gdk.pixbuf_get_from_surface (image, 0, 0, size, size);
        items += item;
    }

    /**
     * If width == 0, then we are actually only peeking for the purpose
     * of populating the theme menu.
     */
    public override Theme? select (string path, int width, int height) {
        time_span = 2.0;

        int kas_cycle = num_frames - 1;
        int piece_cycle = num_frames / 2 - 1;
        layout = default_layout;
        layout.filler_left = 150;
        layout.filler_right = 150;
        layout.filler_top = 100;
        layout.filler_bottom = 100;

        switch (path) {
        case "0" :
            cycles = {kas_cycle, kas_cycle, piece_cycle, piece_cycle};
            forge_table_function = forge_table_spin;
            forge_piece_function = forge_piece_spin;
            layout.num_wins_y = {100, 900};
            layout.kas_x = {1150, 1050};
            layout.kas_y = {367, 500};
            layout.score_x = {1050, 1150};
            layout.score_y = layout.kas_y;
            layout.score_font_size = 30;
            layout.num_rounds_y = 667;
            layout.num_rounds_font_size = 25;
            layout.num_wins_color = spin_color.darken (0.3);
            layout.score_color = {spin_color.darken (0.5), spin_color.darken (0.5)};
            layout.num_rounds_color = spin_color.darken (0.5);
            break;
        case "dizzy" :
            cycles = {kas_cycle, kas_cycle, piece_cycle, piece_cycle};
            forge_table_function = forge_table_dizzy;
            forge_piece_function = forge_piece_dizzy;
            layout.num_wins_x = {1120, 1120};
            layout.kas_x = {1120, 1120};
            layout.score_x = {1120, 1120};
            layout.score_y = {300, 700};
            layout.num_rounds_x = 1120;
            layout.board_x = -30;
            layout.board_y = -30;
            layout.board_size = 1060;
            layout.num_wins_color = dizzy_color.darken (0.2);
            layout.score_color = {dizzy_color, dizzy_color};
            layout.num_rounds_color = dizzy_color;
            break;
        case "tv" :
            cycles = {- kas_cycle, - kas_cycle, piece_cycle, piece_cycle};
            forge_table_function = forge_table_tv;
            forge_piece_function = forge_piece_tv;
            layout.num_wins_x = {1070, 1070};
            layout.num_wins_y = {100, 900};
            layout.num_wins_color = Rgb (1, 1, 1);
            layout.num_wins_font_size = 40;
            layout.kas_x = {1150, 1150};
            layout.kas_y = {500, 580};
            layout.score_x = {1070, 1070};
            layout.score_y = {500, 580};
            layout.score_color = {Rgb (1.0, 0.2, 0.3), Rgb (0.3, 0.5, 1.0)};
            layout.score_font_size = 25;
            layout.num_rounds_x = 1070;
            layout.num_rounds_y = 420;
            layout.num_rounds_color = Rgb (0.2, 0.8, 0.3);
            layout.num_rounds_font_size = 25;
            layout.num_rounds_format = 2;
            break;
        case "shiny" :
            cycles = {0, 0, 0, 0};
            forge_table_function = forge_table_shiny;
            forge_piece_function = forge_piece_shiny;
            layout.kas_x = {1140, 1140};
            layout.kas_y = {460, 540};
            layout.score_x = {1070, 1070};
            layout.score_y = {460, 540};
            layout.score_font_size = 35;
            layout.num_rounds_x = 500;
            layout.num_rounds_y = 970;
            layout.num_rounds_font_size = 25;
            layout.num_rounds_format = 1;
            layout.num_rounds_color = shiny_color.lighten (-0.5);
            layout.score_color = {shiny_color.lighten (-0.5), shiny_color.lighten (-0.5)};
            layout.num_wins_color = shiny_color.lighten (0.5);
            break;
        case "flower" :
            cycles = {piece_cycle, piece_cycle, 0, 0};
            forge_table_function = forge_table_flower;
            forge_piece_function = forge_piece_flower;
            layout.board_x = 20;
            layout.width = 1150;
            layout.num_wins_x = {1050, 1050};
            layout.num_wins_font_size = 40;
            layout.score_x = {1090, 1020};
            layout.score_y = {400, 650};
            layout.score_font_size = 25;
            layout.kas_x = layout.score_x;
            layout.kas_y = layout.score_y;
            layout.kas_frame = {num_frames / 3, num_frames / 3};
            layout.num_rounds_x = 200;
            layout.num_rounds_y = 960;
            layout.num_rounds_format = 1;
            layout.num_wins_color = flower_color.lighten (0.7);
            layout.score_color = {flower_color.darken (0.2), flower_color.darken (0.2)};
            layout.num_rounds_color = flower_color.lighten (0.5);
            break;
        case "galaxy" :
            kas_cycle = - num_frames * 3 / 4;
            cycles = {kas_cycle, kas_cycle, piece_cycle, piece_cycle};
            forge_table_function = forge_table_galaxy;
            forge_piece_function = forge_piece_galaxy;
            layout.kas_frame = {num_frames - 1, num_frames - 1};
            layout.num_wins_color = Rgb (0.4, 0.4, 0.4);
            layout.score_color = {Rgb (0.4, 0.1, 0.1), Rgb (0.1, 0.1, 0.4)};
            layout.num_rounds_color = Rgb (0.1, 0.4, 0.1);
            break;
        default :
            assert_not_reached ();
        }

        layout_changed = true;
        this.path = path;
        if (width != 0) {
            resize (width, height);
        }
        return this;
    }

    public override void resize_images () {
        resize_images_async.begin ();
    }

    /**
     * Construct the theme images incrementally
     */
    int ticket = 0;

    private async void resize_images_async () {
        var my_ticket = ++ticket;

        images[Piece.BOARD] = new Cairo.ImageSurface (Cairo.Format.ARGB32, table.image_width, table.image_height);
        var cr = new Cairo.Context (images[Piece.BOARD]);
        cr.translate (table.filler_left, table.filler_top);
        forge_table_function (cr);

        int size = 66 * table.board_size / 572;
        for (int i=0; i < Piece.COUNT; i++) {
            images[i] = new Cairo.ImageSurface (Cairo.Format.ARGB32, size * num_frames, size);
        }

        GLib.Idle.add (resize_images_async.callback);
        yield;
        if (my_ticket != ticket) return;
        changed ();

        for (int i=0; i < Piece.COUNT; i++) {
            var piece = (Piece) i;
            cr = new Cairo.Context (images[i]);
            for (int f=0; f < num_frames; f++) {
                cr.save ();
                cr.translate (size * f + size / 2, size / 2);
                forge_piece_function (cr, piece, size, f);
                cr.restore ();
                // yield often
                GLib.Idle.add (resize_images_async.callback);
                yield;
                if (my_ticket != ticket) return;
            }//for f
            // but don't signal often
            changed ();
        }//for i
    }

    /**
     * A call to free indicates that the theme has been changed. So invalidate
     * ticket so that any ongoing async theme construction is properly aborted.
     */
    public override void free () {
        ++ticket;
        base.free ();
    }

    private void forge_table_spin (Cairo.Context cr) {
        spin_color.set_source_for (cr);
        cr.paint ();
        
        // Create score area
        double x = table.board_size;
        double width = table.board_size / 15.0;
        cr.move_to (x, -2 * width);
        for (int i= -2; i < 17; i++) {
            double y = i * width;
            // Wavy lines
            double dx = (i % 2 == 0 ? 0.4 : -0.4) * width;
            cr.curve_to (x + dx, y + 0.3 * width,
                         x + dx, y + 0.7 * width,
                         x, y + width);
        }
        cr.line_to (table.image_width, table.image_height);
        cr.line_to (table.image_width, -table.filler_top);
        spin_color.lighten (0.2).set_source_for (cr);
        cr.fill ();

        draw_grids (cr, spin_color.lighten (0.2), 0.1);
        paint_texture (cr, 2, {2, 0.05, 0, 0, 1, 1, 2, 0.05, 0, 0, 1, 0});
    }

    private void forge_table_dizzy (Cairo.Context cr) {
        dizzy_color.set_source_for (cr);
        cr.paint ();

        // Create score area
        double x = table.board_x + table.board_size - table.x;
        double width = table.board_size / 31.0;
        for (int i = -4; i < 35; i++) {
            double y = table.board_y + i * width - table.y;
            double dx = (i % 2 == 0 ? 0.5 : -0.5) * width;
            cr.line_to (x + dx, y);
        }
        cr.line_to (table.image_width, table.image_height);
        cr.line_to (table.image_width, -table.filler_top);
        dizzy_color.lighten (0.5).set_source_for (cr);
        cr.fill ();

        paint_texture (cr, 3, {1.5, 0.05, 0, 0, 1, 1,
                               1.5, 0.05, 0, 0, 0, 1,
                               1.5, 0.1,  0, 0, 1, 0});
        draw_grids (cr, dizzy_color.lighten (0.5), 0.08);
    }

    private void forge_table_tv (Cairo.Context cr) {
        tv_color.set_source_for (cr);
        cr.paint ();

        paint_texture (cr, 3, {2, 0.05, 0, 0, 1, 1,
                               2, 0.05, 0, 0, 0, 1,
                               2, 0.1,  0, 0, 1, 0});
        draw_grids (cr, tv_color.darken (0.2), 0.1);

        cr.translate (-table.x, -table.y);
        for (int side = 0; side < 2; side++) {
            cr.save ();
            cr.translate (table.score_x[side], table.score_y[side]);
            tv_box (cr, Piece.NULL, (int)(0.15 * table.width), 0);
            cr.restore ();
        }
        for (int side = 0; side < 2; side++) {
            cr.save ();
            cr.translate (table.num_wins_x[side], table.num_wins_y[side]);
            tv_box (cr, Piece.NULL, (int)(0.20 * table.width), 0);
            cr.restore ();
        }
        cr.translate (table.num_rounds_x, table.num_rounds_y);
        tv_box (cr, Piece.NULL, (int)(0.15 * table.width), 0);
    }

    private void forge_table_shiny (Cairo.Context cr) {
        shiny_color.set_source_for (cr);
        cr.paint ();

        cr.rectangle (table.board_size, -table.filler_top, table.image_width, table.image_height);
        shiny_color.darken (0.1).set_source_for (cr);
        cr.fill ();

        draw_grids (cr, shiny_color.lighten (0.5), 0.05);
    }

    private void forge_table_flower (Cairo.Context cr) {
        flower_color.set_source_for (cr);
        cr.paint ();
        draw_grids (cr, flower_color.lighten (0.5), 0.1);
        paint_texture (cr, 2, {2, 0.05, 0, 0, 1, 1, 2, 0.05, 0, 0, 1, 0});

        // Flowers:       petals     xy         radii     light    grain   shift
        double[] flower = {10,  1.05, 0.10,  0.05, 0.25,  0.0,     1.5,     0,        // num win
                            10,  1.05, 0.10,  0.02, 0.15,  0.1,     0.75,    1.0/20,
                             7,  1.05, 0.90,  0.05, 0.25,  0.1,     0.75,    0,        // num win
                             7,  1.05, 0.90,  0.01, 0.15,  0.05,    1.0,     1.0/14,
                             5,  1.09, 0.40,  0.01, 0.20,  -0.01,   1.0,     0,        // kas + score
                            10,  1.02, 0.65,  0.05, 0.15,  -0.02,   0.75,    0,        // kas + score
                            20,  0.20, 0.96,  0.05, 0.40,  0.0,     2.0,     0}; 
        int n = 0;
        for (int i = 0; i < flower.length / 8; i++) {
            cr.save ();
            int num_petals = (int)(flower[n++]);
            var x = flower[n++] * table.height;
            var y = flower[n++] * table.height;
            var r1 = flower[n++] * table.height;
            var r2 = flower[n++] * table.height;
            var light = flower[n++];
            var grain = flower[n++];
            var shift = flower[n++] * 2 * Math.PI;
            trace_flower (cr, num_petals, x, y, r1, r2, shift);
            cr.clip ();
            flower_color.lighten (light).set_source_for (cr);
            cr.paint ();
            draw_grids (cr, flower_color.lighten (0.5), 0.1);
            paint_texture (cr, 2, {grain, 0.1, 0, 0, 1, 1, grain, 0.1, 0, 0, 1, 0});
            cr.restore ();
        }
    }

    private void forge_table_galaxy (Cairo.Context cr) {
        var gap = cell_width ();
        var shade = new Cairo.Pattern.linear (0, 0, table.width, 0);
        shade.add_color_stop_rgb (0.8, 0, 0, 0);
        shade.add_color_stop_rgb (0.9, 0.07, 0.01, 0.01);
        cr.set_source (shade);
        cr.paint ();

        Cairo.ImageSurface[] brush = create_spark_brush (
            Rgb (1, 0.9, 0.64), Rgb (0.9, 0.95, 0.23), Rgb (1, 0.91, 0.5),
            Rgb (0.99, 0.79, 0), Rgb (0.99, 0.55, 0), Rgb (0.99, 0.19, 0));

        var line_width = 0.2 * gap;
        var brush_width = brush[0].get_width ();
        for (var y = 1.5 * gap; y <= 11.5 * gap; y += gap) {
            for (var x = 1.5 * gap; x <= 11.5 * gap; x += 0.05 * gap) {
                if (y > 5.5 * gap && y < 7.5 * gap && x > 5.5 * gap && x < 7.5 * gap) continue;
                cr.save ();
                cr.translate (x - line_width / 2, y - line_width / 2);
                cr.scale (line_width / brush_width, line_width / brush_width);
                cr.rectangle (0, 0, brush_width, brush_width);
                cr.set_source_surface (brush[Random.int_range (0, brush.length)], 0, 0);
                cr.fill ();
                cr.restore ();

                cr.save ();
                cr.translate (y - line_width / 2, x - line_width / 2);
                cr.scale (line_width / brush_width, line_width / brush_width);
                cr.rectangle (0, 0, brush_width, brush_width);
                cr.set_source_surface (brush[Random.int_range (0, brush.length)], 0, 0);
                cr.fill ();
                cr.restore ();
            }
        }
    }

    private void shine_light (Cairo.Context cr, double brightness, double radius, double tilt) {
        var spot_angle = 0.25 * Math.PI - 0.75 * Math.PI * tilt;
        var spot_radius = radius * (0.5 + 0.3 * tilt);
        var spot_x = spot_radius * Math.cos (spot_angle);
        var spot_y = spot_radius * Math.sin (spot_angle);
        var shade = new Cairo.Pattern.radial (-spot_x, - spot_y, 0, 0, 0, radius);
        shade.add_color_stop_rgba (0.0, 1, 1, 1, 0.95 * brightness);
        shade.add_color_stop_rgba (0.3, 1, 1, 1, 0.5 * brightness);
        shade.add_color_stop_rgba (1.0, 0.2, 0.2, 0.2, 0.2 * brightness);
        cr.set_source (shade);
        cr.arc (0, 0, radius, 0, 2 * Math.PI);
        cr.fill ();
    }

    private void forge_piece_spin (Cairo.Context cr, Piece piece, int size, int f) {
        cr.save ();
        var n_rotation = piece.is_kas () ? 1 : 2;
        cr.rotate (n_rotation * 2 * Math.PI * f / (num_frames - 1));
        Rgb[] rgb1 = {Rgb (1.0, 0.1, 0.3), Rgb (0.1, 0.3, 1.0), Rgb (0.0, 0.0, 0.0), Rgb (0.8, 0.8, 0.8)};
        Rgb[] rgb2 = {Rgb (1.0, 0.4, 0.6), Rgb (0.4, 0.6, 1.0), Rgb (0.2, 0.2, 0.2), Rgb (0.9, 0.9, 0.9)};
        Rgb dark = rgb1[piece];
        Rgb light = rgb2[piece];
        var radius = size * (piece.is_kas () || f < num_frames / 2 ? 0.3 : 0.6 - 0.6 * f / (num_frames - 1));
        var color1 = new Cairo.Pattern.radial (0, 0, 0, 0, 0, radius);
        color1.add_color_stop_rgba (0.0, dark.red, dark.green, dark.blue, 0.9);
        color1.add_color_stop_rgba (0.8, dark.red, dark.green, dark.blue, 0.95);
        color1.add_color_stop_rgba (1.0, dark.red, dark.green, dark.blue, 1.0);
        cr.set_source (color1);
        cr.arc (0, 0, radius, 0, Math.PI);
        cr.arc (-radius / 2, 0, radius / 2, Math.PI, 2 * Math.PI);
        cr.arc_negative (radius / 2, 0, radius / 2 - 0.5, Math.PI, 0);
        cr.fill ();
        var color2 = new Cairo.Pattern.radial (0, 0, 0, 0, 0, radius);
        color2.add_color_stop_rgba (0.0, light.red, light.green, light.blue, 0.9);
        color2.add_color_stop_rgba (0.8, light.red, light.green, light.blue, 0.95);
        color2.add_color_stop_rgba (1.0, light.red, light.green, light.blue, 1.0);
        cr.set_source (color2);
        cr.arc (0, 0, radius, Math.PI, 2 * Math.PI);
        cr.arc (radius / 2, 0, radius / 2, 0, Math.PI);
        cr.arc_negative (-radius / 2, 0, radius / 2 - 0.5, 2 * Math.PI, Math.PI);
        cr.fill ();
        cr.restore ();
        shine_light (cr, 1.0, radius, 0.0);
    }

    private void forge_piece_dizzy (Cairo.Context cr, Piece piece, int size, int f) {
        Rgb[] rgb = {Rgb (1.0, 0.0, 0.0), Rgb (0.0, 0.0, 1.0), Rgb (0.0, 0.0, 0.0), Rgb (1.0, 1.0, 1.0)};
        Rgb paint = rgb[piece];
        var phase = Math.remainder (4 * Math.PI * f / (num_frames - 1), 2 * Math.PI);
        var num_rings = 5;
        var band = 0.25;
        var radius = size * (piece.is_kas () ? 0.3 : 0.3 - 0.05 * Math.sin (phase));
        var alpha = 1.0;
        if (! piece.is_kas () && f > num_frames / 2) {
            // fade = 1 to 0
            var fade = (Math.cos (Math.PI * (f - num_frames / 2) / (num_frames / 2)) + 1) / 2;
            band += 0.9 * (1 - fade);
            alpha *= fade;
            radius *= (1 + (1 - fade) / 2);
        }
        var delta = phase / (2 * Math.PI);
        cr.arc (0, 0, radius, 0, 2 * Math.PI);
        for (int ring = num_rings; ring >= 0; ring--) {
            var r = (delta + ring) * radius / num_rings;
            var r2 = (delta + ring - band) * radius / num_rings;
            if (r > radius) continue;
            cr.arc_negative (0, 0, r, 2 * Math.PI, 0);
            cr.arc (0, 0, r2, 0, 2 * Math.PI);
        }
        var color = new Cairo.Pattern.radial (0, 0, 0, 0, 0, radius);
        color.add_color_stop_rgba (0.0, paint.red, paint.green, paint.blue, 1.0 * alpha);
        color.add_color_stop_rgba (0.8, paint.red, paint.green, paint.blue, 0.8 * alpha);
        color.add_color_stop_rgba (1.0, paint.red, paint.green, paint.blue, 0.0 * alpha);
        cr.set_source (color);
        cr.fill ();
        shine_light (cr, alpha, radius, 0.0);
    }

    private void forge_piece_tv (Cairo.Context cr, Piece piece, int size, int f) {
        if (piece.is_kas ())
            tv_control (cr, piece, size, f);
        else
            tv_box (cr, piece, size, f);
    }

    private void tv_control (Cairo.Context cr, Piece piece, int size, int f) {
        Rgb paint = piece == Piece.KAS_0 ? Rgb (1.0, 0.2, 0.3) : Rgb (0.3, 0.5, 1.0);
        var width = 0.3 * size;
        var height = 0.5 * size;
        var x = -0.15 * size;
        var y = -0.25 * size;
        var shadow_depth = 0.05 * width;
        trace_rounded_rectangle (cr, x + shadow_depth, y + shadow_depth, width, height, 0.15 * width);
        paint.darken (0.5).set_source_for (cr);
        cr.fill ();
        trace_rounded_rectangle (cr, x, y, width, height, 0.15 * width);
        paint.set_source_for (cr);
        cr.fill ();
        // Relative size of buttons, gaps and borders :
        //        __[ ]_[ ]_[ ]__ 
        // 0.3 + 1 + 0.1 + 1 + 0.1 + 1 + 0.3 = 3.8
        var proportion = 3.8;
        var button_width = 0.9 * width / proportion;
        var shadow_width = 0.1 * width / proportion;
        int pressed_button = f % 2 == 0 ? -1 : Random.int_range (0, 15);
        for (int i=0; i < 15; i++) {
            var col = i % 3;
            var row = i / 3;
            var x2 = x + (0.3 + col * 1.1) / proportion * width;
            var y2 = y + (0.4 + row * 1.1) / proportion * width;
            fill_rectangle (cr, paint.darken (0.5), x2 + shadow_width, y2 + shadow_width, button_width, button_width);
            if (i == pressed_button) {
                x2 += shadow_width;
                y2 += shadow_width;
            }
            fill_rectangle (cr, paint.lighten (0.5), x2, y2, button_width, button_width);
        }
    }

    private void tv_box (Cairo.Context cr, Piece piece, int size, int f) {
        Rgb paint = piece == Piece.NULL ? tv_color.darken (0.5) :
                    piece == Piece.BLACK ? Rgb (0.2, 0.2, 0.2) : Rgb (0.8, 0.8, 0.8);

        // Outer frame
        var x1 = -0.3 * size;
        var y1 = -0.2 * size;
        var x2 = -x1;
        var y2 = -y1;
        var width = 0.6 * size;
        var height = 0.4 * size;
        trace_rounded_rectangle (cr, x1, y1, width, height, 0.1 * width);
        cr.clip ();
        // Top and bottom
        var color = new Cairo.Pattern.linear (x1, y1, x1, y2);
        var lighted = paint.lighten (0.5);
        color.add_color_stop_rgb (0.4, lighted.red, lighted.green, lighted.blue);
        lighted = paint.lighten (-0.5);
        color.add_color_stop_rgb (0.6, lighted.red, lighted.green, lighted.blue);
        cr.set_source (color);
        cr.paint ();
        // Left and right
        fill_triangle (cr, paint.lighten (0.3), x1, y1, x1, y2, x1 + height / 2, 0);
        fill_triangle (cr, paint.lighten (0.0), x2, y1, x2, y2, x2 - height / 2, 0);

        // Inner frame
        var x = -0.25 * size;
        var y = -0.15 * size;
        var w = 0.5 * size;
        var h = 0.3 * size;
        trace_rounded_rectangle (cr, x, y, w, h, 0.05 * w);
        cr.clip ();
        // Top and bottom
        color = new Cairo.Pattern.linear (x, y, x, -y);
        lighted = paint.lighten (-0.5);
        color.add_color_stop_rgb (0.4, lighted.red, lighted.green, lighted.blue);
        lighted = paint.lighten (0.5);
        color.add_color_stop_rgb (0.6, lighted.red, lighted.green, lighted.blue);
        cr.set_source (color);
        cr.paint ();
        // Left and right
        fill_triangle (cr, paint.lighten (0.0), x1, y1, x1, y2, x1 + height / 2, 0);
        fill_triangle (cr, paint.lighten (0.3), x2, y2, x2, y1, x2 - height / 2, 0);

        // Screen
        x = -0.22 * size;
        y = -0.13 * size;
        w = 0.44 * size;
        h = 0.26 * size;
        cr.rectangle (x, y, w, h);
        cr.clip ();

        // Tv programmes
        if (f == 0) {
            paint.darken (piece == Piece.NULL ? 1.0 : 0.1).set_source_for (cr);
            cr.paint ();
        }
        else if (f < 10) {
            cr.set_source_rgb (0.2, 1, 0.2);
            cr.paint ();
            trace_flower (cr, 5, 0, 0, 0, 0.05 * (f % 10) * size);
            cr.set_source_rgb (1, 0.4, 0.8);
            cr.fill ();
        }
        else if (f < 20) {
            cr.set_source_rgb (0, 0.7, 1.0);
            cr.paint ();
            cr.set_source_rgb (1.0, 0.2, 0);
            cr.arc (0, -0.01 * (f % 10) * size, 0.1 * size, 0, 2 * Math.PI);
            cr.fill ();
            fill_triangle (cr, Rgb (0.2, 0.6, 0.9), -0.8 * size, y2, 0.5 * size, y2, -0.15 * size, 0);
            fill_triangle (cr, Rgb (0.3, 0.4, 0.8), -0.6 * size, y2, 0.5 * size, y2, 0.1 * size, 0.02 * size);
        }
        else {
            cr.set_source_rgb (0, 0, 0);
            cr.paint ();
            cr.set_source_rgb (1, 1, 0);
            show_text (cr, "Pasang\nEmas"[0:1 + int.min (10, f - 20) % 11], (int)(0.06 * size), 0, 0);
        }
    }

    private void forge_piece_shiny (Cairo.Context cr, Piece piece, int size, int f) {
        cr.save ();
        var radius = 0.30 * size;
        // Lift
        var n1 = int.min (num_frames - 2, piece.is_kas () ? f : f * 2);
        var n2 = num_frames;
        cr.translate (0, - n1 * size / 3.0 / n2);
        cr.scale (1.0, (double) (n2 - n1 / 2) / n2);
        // Color
        double[] reds =   {1.0, 0.3, 0.2, 1.0};
        double[] greens = {0.2, 0.2, 0.2, 1.0};
        double[] blues =  {0.3, 1.0, 0.2, 1.0};
        var red = reds[piece];
        var green = greens[piece];
        var blue = blues[piece];
        var color = new Cairo.Pattern.radial (0, 0, 0, 0, 0, radius);
        color.add_color_stop_rgba (0, red, green, blue, 0.98);
        color.add_color_stop_rgba (0.8, red, green, blue, 1.0);
        cr.set_source (color);
        cr.set_source_rgb (red, green, blue);
        cr.arc (0, 0, radius, 0, 2 * Math.PI);
        cr.fill ();
        shine_light (cr, 1.0, radius, 1.0 * n1 / n2);
        cr.restore ();
    }

    /**
     * For speed, the flower texture is created only once and used for all copies
     */
    private Cairo.ImageSurface flower_texture = null;

    private void forge_piece_flower (Cairo.Context cr, Piece piece, int size, int f) {
        if (flower_texture == null) {
            flower_texture = create_texture (3, {1.5, 0.05, 0, 0, 1, 1,
                                                 1.5, 0.05, 0, 0, 1, 0,
                                                 1.5, 0.05, 0, 0, 0, 1});
        }
        cr.save ();
        // Create flower-shaped clip area
        var radius = 0.33 * size;
        var frac = 1.0 * f / (num_frames / 2);
        var frac2 = double.min (1.0, frac);
        var r_out = (1.0 + frac2) * radius;
        var r_in  = (1.0 - frac2) * radius;
        var num_petals = piece.is_kas () ? 16 : 8;
        int num_removed = (!piece.is_kas () && f > num_frames / 2) ?
                            (int)((frac - 1) * (num_petals + 1)) + 1 : 0;
        var shift = piece.is_kas () && f > num_frames / 2 ? (frac - 1) * 2 * Math.PI / num_petals : 0.0;
        trace_flower (cr, num_petals, 0, 0, r_in, r_out, shift, num_removed);
        cr.clip ();
        // Paint with radial shade
        double[] reds =   {0.8, 0.3, 0.2, 1.0};
        double[] greens = {0.3, 0.5, 0.2, 1.0};
        double[] blues =  {0.4, 0.8, 0.2, 1.0};
        var red = reds[piece];
        var green = greens[piece];
        var blue = blues[piece];
        var color = new Cairo.Pattern.radial (0, 0, 0, 0, 0, (2.0 - frac2) * radius);
        color.add_color_stop_rgb (0, red, green, blue);
        color.add_color_stop_rgb (0.5, red, green, blue);
        color.add_color_stop_rgb (1.0, 1.0, 1.0, 0);
        cr.set_source (color);
        cr.paint ();
        // Paint with texture
        cr.set_source_surface (flower_texture, 0, 0);
        cr.get_source().set_extend (Cairo.Extend.REPEAT);
        cr.paint ();
        cr.restore ();
    }


    private double[,] stars = null;

    private void forge_piece_galaxy (Cairo.Context cr, Piece piece, int size, int f) {
        cr.save ();
        cr.arc (0, 0, size / 2, 0, 2 * Math.PI);
        cr.clip ();
        switch (piece) {
        case Piece.BLACK :
            galaxy_black (cr, size, f);
            break;
        case Piece.WHITE :
            galaxy_white (cr, size, f);
            break;
        default :
            galaxy_kas (cr, piece, size, f);
            break;
        }
        cr.restore ();
    }

    private void galaxy_kas (Cairo.Context cr, Piece piece, int size, int f) {
        int n_corners = 20;
        if (f == 0 || stars == null) {
            stars = new double[n_corners * 2, 2];
            // Phase randomizer
            for (int i=0; i < n_corners * 2; i++) {
                stars[i, 0] = Random.next_double () * 2 * Math.PI;   // Theta
                stars[i, 1] = Random.next_double () * 2 * Math.PI;   // Length of control tangents
            }
        }
        var phase = 4 * Math.PI * f / num_frames;
        // 0 <= distort <= 1, starting from 0 at frame 0
        var distort = f > num_frames / 2 ? 1.0 : Math.sin (Math.PI / 2 * f / (num_frames / 2));
        var radius = (0.2 + 0.3 * distort) * size;
        var corona = 0.7 * radius;
        for (int layer=0; layer < corona; layer++) {
            radius -= 1.0;
            // End points
            double[] fx = new double[n_corners];
            double[] fy = new double[n_corners];
            // Control points, 2 for each end points.
            double[] cx = new double[2 * n_corners];
            double[] cy = new double[2 * n_corners];
            for (int i=0; i < n_corners; i++) {
                // Fixed points around a cirle
                var theta = 2 * Math.PI * (i + 1) / n_corners;
                // Rotate the circle
                //theta += 4 * Math.PI * f / num_frames;
                fx[i] = -radius * Math.cos (theta);
                fy[i] = -radius * Math.sin (theta);
                // Make theta perpendicular to circle
                theta += Math.PI / 2;
                // theta += angle between -pi/3 and pi/3
                theta += Math.sin (phase + stars[i, 0]) * Math.PI / 3 * distort;
                var n = i * 2;
                var len = (0.2 + 0.2 * Math.sin (distort * stars[n, 1] + phase)) * radius;
                cx[n] = fx[i] + len * Math.cos (theta);
                cy[n] = fy[i] + len * Math.sin (theta);
                len = (0.2 + 0.2 *  Math.sin (distort * stars[n + 1, 1] + phase)) * radius;
                cx[n + 1] = fx[i] - len * Math.cos (theta);
                cy[n + 1] = fy[i] - len * Math.sin (theta);
            }
            cr.move_to (fx[0], fy[0]);
            for (int i=0; i < n_corners; i++) {
                var j = (i + 1) % n_corners;   // Index to next end point
                cr.curve_to (cx[2 * i + 1], cy[2 * i + 1],  cx[2 * j] , cy[2 * j],  fx[j], fy[j]);
            }
            var alpha = Math.pow (layer / corona, 3) * 0.6;
            if (piece == Piece.KAS_0)
                cr.set_source (new Cairo.Pattern.rgba (1, alpha, alpha, alpha));
            else
                cr.set_source (new Cairo.Pattern.rgba (alpha, alpha, 1, alpha));
            cr.fill ();
        }//for layer
    }

    private void galaxy_black (Cairo.Context cr, int size, int f) {
        if (f == num_frames - 1) return;
        int num_stars = 1000;
        if (f == 0) {
            stars = new double[num_stars, 5];
            for (int i=0; i < num_stars; i++) {
                // Spiral: r = theta
                // r. (Calculated using initial step of Box-Muller transform)
                stars[i, 0] = Math.sqrt (- 2 * Math.log (Random.next_double ()));
                // theta + random
                stars[i, 1] = stars[i, 0] * 2 * Math.PI - Math.PI + Random.double_range (-1, 1);
                
                // Random colors
                stars[i, 2] = Random.double_range (0.4, 1);
                stars[i, 3] = Random.double_range (0.1, 0.5);
                stars[i, 4] = Random.double_range (0.4, 1);
            }
        }

        double t = f <= num_frames / 2 ? 0 : (1.0 * f - num_frames / 2) / (num_frames / 2);
        double expand = 0.6 * t * t;
        for (int i=0; i < num_stars; i++) {
            var r = stars[i, 0] + expand;
            var theta = stars[i, 1] - 4 * Math.PI * f / num_frames;
            var x = 0.25 * size * r * Math.sin (theta);
            var y = 0.25 * size * r * Math.cos (theta);
            if (i >= num_stars / 2) {
                y = -y; x = -x;
            }
            // The stars are brightest at the centre
            var cf = (1.5 - double.min (1.3, Math.fabs (r)));
            var light = cf * size / 100;
            cr.set_source_rgb (stars[i, 2], stars[i, 3], stars[i,4]);
            cr.arc (x, y, light, 0, 2 * Math.PI);
            cr.fill ();
        }
    }

    Cairo.ImageSurface[] cluster_brush;

    private void galaxy_white (Cairo.Context cr, int size, int f) {
        if (f == num_frames - 1) return;
        int num_clusters = 200;
        if (f == 0) {
            cluster_brush = create_spark_brush (
                Rgb (1, 1, 1), Rgb (0.9, 0.95, 0.95), Rgb (0.9, 0.9, 0.8),
                Rgb (0.9, 0.8, 0.8), Rgb (1, 1, 0.9), Rgb (1, 1, 0));
            stars = new double[num_clusters, 4];
            for (int i=0; i < num_clusters; i++) {
                double r = 0, theta;
                do {
                    r = Math.sqrt (- 2 * Math.log (Random.next_double ()));
                } while (r > 1.0);
                theta = 2 * Math.PI * Random.next_double ();
                stars[i, 0] = 0.2 * size * r * Math.sin (theta);
                stars[i, 1] = 0.2 * size * r * Math.cos (theta);
                stars[i, 2] = Random.int_range (0, cluster_brush.length);
                stars[i, 3] = 2 * Math.PI * Random.next_double ();
            }
        }

        var brush_width = cluster_brush[0].get_width ();
        var cluster_width = 0.2 * size;
        var scale = cluster_width / brush_width;
        if (f >= num_frames / 2) {
            // arcsin -1/2.5 = 3.55. So, the animation ends with
            //    scale *= 1.0 + (2.5 - sin 3.55)  =  1.0 - 1.0  =  0
            scale *= 1.0 + 2.5 * Math.sin (3.55 * (f - num_frames / 2) / (num_frames / 2));
        }
        for (int i=0; i < num_clusters; i++) {
            cr.save ();
            cr.translate (stars[i, 0], stars[i, 1]);
            cr.scale (scale, scale);
            cr.rotate (stars[i, 3] + (i % 2 == 0 ? -1 : 1) * 4 * Math.PI * f / num_frames);
            cr.translate (-0.5 * brush_width, -0.5 * brush_width);
            cr.rectangle (0, 0, brush_width, brush_width);
            cr.set_source_surface (cluster_brush[(int)stars[i, 2]], 0, 0);
            cr.fill ();
            cr.restore ();
        }
    }

    /**
     * Emulate the Spark brush from Gimp (to create stars)
     */
    private Cairo.ImageSurface[] create_spark_brush (Rgb core, Rgb crust, Rgb surface,
        Rgb ray1, Rgb ray2, Rgb ray3)
    {
        int[] pt = {0, 35, 30, 25,
                     1, 20, 16, 6,  1, 41, 42, 5,
                     2, 29, 22, 13,
                     3, 35, 38, 14,
                     4, 29, 26, 10,
                     5, 33, 21, 7,  5, 18, 45, 3,
                     6, 27, 23, 11,  6, 48, 42, 5,
                     7, 32, 31, 12,
                     8, 28, 20, 3,  8, 50, 14, 3,  8, 38, 44, 3};
        Cairo.ImageSurface[] brush = new Cairo.ImageSurface[9];
        for (int i=0; i < brush.length; i++) {
            brush[i] = new Cairo.ImageSurface (Cairo.Format.ARGB32, 60, 60);
        }
        for (int i=0; i < pt.length;) {
            var n = pt[i++];
            var brush_cr = new Cairo.Context (brush[n]);
            var x = pt[i++];
            var y = pt[i++];
            var r = pt[i++];
            var pattern = new Cairo.Pattern.radial (x, y, 0, x, y, r);
            pattern.add_color_stop_rgba (0, core.red, core.green, core.blue, 1);
            pattern.add_color_stop_rgba (12.0 / 60, crust.red, crust.green, crust.blue, 0.9);
            pattern.add_color_stop_rgba (15.0 / 60, surface.red, surface.green, surface.blue, 0.66);
            pattern.add_color_stop_rgba (20.0 / 60, ray1.red, ray1.green, ray1.blue, 0.49);
            pattern.add_color_stop_rgba (30.0 / 60, ray2.red, ray2.green, ray2.blue, 0.43);
            pattern.add_color_stop_rgba (50.0 / 60, ray3.red, ray3.green, ray3.blue, 0.10);
            pattern.add_color_stop_rgba (1, 0, 0, 0, 0);
            brush_cr.set_source (pattern);
            brush_cr.paint ();
        }
        return brush;
    }

    /**
     * Trace a flower pattern. Don't draw, just trace.
     */
    private void trace_flower (Cairo.Context cr, int num_petals, double x, double y,
                                 double radius1, double radius2,
                                 double shift = 0, int num_removed = 0) {
        cr.move_to (x + radius1, y);
        for (int i = num_removed; i <= num_petals; i++) {
            var angle = Math.PI * 2 * i / num_petals + shift;
            var x1 = x + radius2 * Math.cos (angle);
            var y1 = y + radius2 * Math.sin (angle);
            angle = Math.PI * 2 * (i + 1) / num_petals + shift;
            var x2 = x + radius2 * Math.cos (angle);
            var y2 = y + radius2 * Math.sin (angle);
            var x3 = x + radius1 * Math.cos (angle);
            var y3 = y + radius1 * Math.sin (angle);
            cr.curve_to (x1, y1, x2, y2, x3, y3);
        }
    }

    private void trace_rounded_rectangle (Cairo.Context cr, double x, double y,
                                            double width, double height, double radius) {
        var x2 = x + width;
        var y2 = y + height;
        var r2 = radius / 2;
        cr.move_to (x + radius, y);
        cr.line_to (x2 - radius, y);
        cr.curve_to (x2 - r2, y,  x2, y + r2,  x2, y + radius);
        cr.line_to (x2, y2 - radius);
        cr.curve_to (x2, y2 - r2,  x2 - r2, y2,  x2 - radius, y2);
        cr.line_to (x + radius, y2);
        cr.curve_to (x + r2, y2,  x, y2 - r2,  x, y2 - radius);
        cr.line_to (x, y + radius);
        cr.curve_to (x, y + r2,  x + r2, y,  x + radius, y);
    }

    private void fill_triangle (Cairo.Context cr, Rgb color, double x1, double y1,
                                  double x2, double y2, double x3, double y3) {
        trace_triangle (cr, x1, y1, x2, y2, x3, y3);
        color.set_source_for (cr);
        cr.fill ();
    }

    private void trace_triangle (Cairo.Context cr, double x1, double y1,
                                  double x2, double y2, double x3, double y3) {
        cr.move_to (x1, y1);
        cr.line_to (x2, y2);
        cr.line_to (x3, y3);
        cr.line_to (x1, y1);
    }

    private void fill_rectangle (Cairo.Context cr, Rgb color, double x, double y, double w, double h) {
        cr.rectangle (x, y, w, h);
        color.set_source_for (cr);
        cr.fill ();
    }

    /**
     * Paint horizontal and vertical lines, forming a grid.
     */
    private void draw_grids (Cairo.Context cr, Rgb line_color, double line_width) {
        cr.save ();
        cr.translate (table.board_x - table.x, table.board_y - table.y);
        line_color.set_source_for (cr);
        int gap = cell_width ();
        for (int i=0; i < 11; i++) {
            var x = gap * 1.5 + gap * i - 0.5 * line_width * gap;
            var y = gap * 1.5 - 0.5 * line_width * gap;
            var w = line_width * gap;
            var h = 10 * gap;
            if (i != 5) {
                cr.rectangle (x, y, w, h + w);
                cr.rectangle (y, x, h, w);
            }
            else {
                cr.rectangle (x, y, w, 4 * gap);
                cr.rectangle (x, y + 6 * gap, w, 4 * gap);
                cr.rectangle (y, x, 4 * gap, w);
                cr.rectangle (y + 6 * gap, x, 4 * gap, w);
            }
        }
        cr.fill ();
        cr.restore ();
    }

    /**
     * Paint a texture using layers translucent black and white bands.
     * Each layer is specified using 5 numbers:
     *   grain size     : the spacing between the bands
     *   alpha          : 0 to 1
     *   x1, y1, x2, y2 : e.g., 0, 0, 1, 1 means the band goes this way ////
     *                          0, 0, 1, 0 means the band goes this way ||||
     */
    private void paint_texture (Cairo.Context cr, int num_layers, double[] layer_spec) {
        assert (num_layers * 6 == layer_spec.length);
        // For efficiency, the texture is first created on a small tile.
        // Then this tile is used as a paint.
        var tile = create_texture (num_layers, layer_spec);
        cr.set_source_surface (tile, 0, 0);
        cr.get_source().set_extend (Cairo.Extend.REPEAT);
        cr.paint ();
    }

    /**
     * See paint_texture for arguments and usage
     */
    private Cairo.ImageSurface create_texture (int num_layers, double[] layer_spec) {
        var tile_size = (int) (16 * layer_spec[0]);
        var tile = new Cairo.ImageSurface (Cairo.Format.ARGB32, tile_size, tile_size);
        var tile_cr = new Cairo.Context (tile);

        int n = 0;
        for (int layer=0; layer < num_layers; layer++) {
            var size = 1024;
            var grain_size = layer_spec[n++];
            var alpha = layer_spec[n++];
            var x1 = layer_spec[n++] * size;
            var y1 = layer_spec[n++] * size;
            var x2 = layer_spec[n++] * size;
            var y2 = layer_spec[n++] * size;
            var shade = new Cairo.Pattern.linear (x1, y1, x2, y2);
            int v = 0;
            for (double i=0; i <= size * 2; i += grain_size) {
                v++;
                double val = v % 2;
                shade.add_color_stop_rgba ((double) i / size, val, val, val, alpha);
            }
            tile_cr.set_source (shade);
            tile_cr.paint ();
        }
        return tile;
    }

}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
