/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

struct ThemeItem {
    public string name;
    public string path;
    public Gdk.Pixbuf pixbuf;
    public string? license;
    public string? artist;
    public string? comment;
    public Theme engine;
    public ThemeItem () {
        license = null;
        artist = null;
        comment = null;
    }
}

/**
 * Control the animation sequence of a token.
 */
struct Film {
    public int stop;    // Stop film when this frame number is reached.
    public int length;  // Number of frames in film
    public int viewed;  // Frame shown
    public int n;       // Logical frame number
    public Film () {
        stop = 0;
        length = 1;
        viewed = 0;
        n = 0;
    }
}

abstract class Theme : Object {
    /**
     * A Theme can actually be a meta-theme for several themes having similar
     * processing requirement. Each actual theme is listed here.
     */
    public ThemeItem[] items;

    /**
     * Used by ThemeSelector to select one of the actual themes listed in items[].
     */
    public abstract Theme? select (string path, int width, int height);

    /**
     * ThemeSelector invokes this so that screen images can be prepared.
     */
    public abstract void resize (int width, int height);

    /**
     * Maintain a small footprint. Free memory when this meta-theme
     * is not used.
     */
    public abstract void free ();

    /**
     * How many seconds it take to play an entire sequence of film frames.
     */
    public double time_span {public get; protected set;}

    protected Film[] films = new Film[BOARD_SIZE];

    construct {
        for (int i=0; i < films.length; i++) films[i] = Film ();
        time_span = 1.3;
    }

    protected int[] cycles = new int[Piece.COUNT];

    public abstract void draw (GameView game_view, Cairo.Context cr);
    public abstract void queue_draw_piece (Gtk.DrawingArea view, Piece piece, Point p);
    public abstract int count_frames (Piece piece);

    /**
     * Convert board position to screen coordinates.
     */
    public abstract Point to_point (int pos, bool rotated);

    /**
     * Convert screen coordinates p to board position.
     * Return -1 if p is not touching any pieces on the game board
     * (or a dragged kas is not touching any junction).
     */
    public abstract int to_position (Game game, Point p, bool rotated, bool precise);

    /**
     * Return 0 if p is within the TOP side of the board,
     * 1 if within the BOTTOM side,
     * -1 if outside
     */
    public abstract int to_side (Point p);

    /**
     * Start the visual effect of picking up pieces to capture.
     * full==false means pick half-way.
     */
    public void pick_booties (Game game, Move? move, bool full) {
        if (move == null) return;
        foreach (int pos in move.booties) {
            Piece piece = game.board[pos];
            // In the formula below, "+1" caters for odd number of frames.
            int num_frames = count_frames (piece);
            films[pos].length = num_frames;
            var half_way = (num_frames + 1) / 2 - 1;
            films[pos].stop = full ? num_frames - 1 : half_way;
            if (full) films[pos].n = films[pos].viewed;
        }
    }

    /**
     * Start the visual effect of picking up the kas that is currently being dragged.
     * full==false means pick half-way.
     */
    public void pick_kas (Game game) {
        int num_frames = count_frames (game.kas);
        films[game.kas].length = num_frames;
        films[game.kas].stop = num_frames - 1;
    }

    /**
     * Start the visual effect of dropping pieces.
     */
    public void drop_pieces () {
        for (int pos=2; pos < BOARD_SIZE; pos++) {
            films[pos].stop = 0;
        }
    }

    /**
     * Start the visual effect of dropping the kas.
     */
    public void drop_kas (Game game) {
        for (int i=0; i < 2; i++) {
            films[i].stop = 0;
            var pos = game.kas_position[i];
            if (pos == 0) continue;
            films[pos].stop = 0;
            films[pos].length = films[i].length;
            films[pos].n = films[i].n;
            films[pos].viewed = films[i].viewed;
        }
    }

    /**
     * Previous value of lapse when animate() was invoked.
     */
    private double prev_lapse = 0;

    /**
     * Previous kas position. Used to detect if the kas has been moved.
     */
    private Point prev_kas_point;

    /**
     * Advance animation frames.
     * lapse is in the range 0..1
     * Return true if still animating.
     *
     * If cycle is zero, then the film is played sequentially.
     * If positive, then a portion is played repeatedly.
     * If negative, then a portion is played repeatedly forward and backward.
     *
     * e.g., for film.stop == 6
     *   cycle = 0       n     0 1 2 3 4 5 6 stop
     *                  viewed 0 1 2 3 4 5 6 stop
     *
     *   cycle = 3       n     0 1 2*3 4 5 6*3 4 5 6*3 4 5 6 ...
     *                  viewed 0 1 2 3 4 5 6 3 4 5 6 3 4 5 6 ...   
     *
     *   cycle < -3      n     0 1 2*3 4 5 6 7 8*3 4 5 6 7 8 ...
     *                  viewed 0 1 2 3 4 5 6 5 4 3 4 5 6 5 4 ...
     */
    public bool animate (GameView game_view, double lapse) {
        // The kas may changed either in appearance or in pixel location
        bool kas_changed = (prev_kas_point != game_view.kas_point);
        bool animating = kas_changed;
        for (int pos=0; pos < BOARD_SIZE; pos++) {
            Film* film = &films[pos];
            Piece piece = pos <= 1 ? (Piece) pos : game_view.game.board[pos];
            int cycle = piece.is_real () ? cycles[piece] : 0;
            if (! piece.is_kas () && film->stop == film->length - 1) cycle = 0;
            // Skip if the last frame is reached for a dropped
            // piece (stop==0) or a non-cyclic piece (cycle==0)
            if (film->n == film->stop && (film->stop == 0 || cycle == 0)) continue;
            // At least one piece is still being animated
            animating = true;
            // Advance animation only when sufficient time has elapsed to show the next frame
            if ((int)(lapse * film->length) == (int)(prev_lapse * film->length))
                continue;
            int max = piece.is_kas () || film->stop == film->length - 1 ? film->length - 1 : (film->length + 1) / 2 - 1;
            if (cycle <= 0) {             
                film->n +=  (film->stop == 0) ? -1 : +1;
                if (film->n >= max - cycle) film->n = max + cycle;
            }
            else { // cycle > 0
                film->n += (film->stop == 0 && film->n <= max - cycle) ? -1 : +1;
                if (film->n > max) film->n = max - cycle;
            }
            if (film->n < 0) {
                debug ("Theme file is malformed");
                film->n = 0;
            }
            film->viewed = (cycle < 0 && film->n > max) ? max * 2 - film->n : film->n;
            if (film->viewed < 0) {
                debug ("Negative frame number");
                film->viewed = 0;
            }
            if (pos > 1) {
                game_view.queue_draw_piece (pos);
            }
            else {
                kas_changed = true;
            }
        }//endfor
        if (kas_changed) {
            queue_draw_piece (game_view, game_view.game.kas, prev_kas_point);      // Erase from old position
            queue_draw_piece (game_view, game_view.game.kas, game_view.kas_point); // Draw in new position
            prev_kas_point = game_view.kas_point;
        }
        prev_lapse = lapse;
        return animating;
    }

}//class Theme

class ThemeSwitch : Gtk.Box {
    public signal void changed ();
    public Theme theme {get; private set;}
    private Theme[] themes = {new Theme2DCairo (), new Theme2DFile ()};
    private ThemeItem[] theme_items = {};
    private Gtk.Allocation size = Gtk.Allocation (){width=100, height=100};

    private Gtk.TreeView tree_view;
    private Gtk.TreeSelection tree_selection;
    private Gtk.ListStore tree_model;
    
    public ThemeSwitch () {
        theme = themes[0].select ("0", size.width, size.height);
        // Propagate changes in Theme2DCairo due to incremental
        // async theme construction
        (themes[0] as Theme2DCairo).changed.connect (() => {
            changed ();
        });
        put_contents ();
    }

    private void put_contents () {
        get_style_context () .add_class ("box-top-level");
        orientation = Gtk.Orientation.VERTICAL;
        var list = create_theme_list ();
        list.set_size_request (300, 350);
        add (list);
        show_all ();
    }

    private Gtk.Widget create_theme_list () {
        // Construct model and view for selectable theme list
        tree_view  = new Gtk.TreeView ();
        tree_view.model = tree_model = new Gtk.ListStore (2, typeof (Gdk.Pixbuf), typeof (string));;
        tree_view.insert_column_with_attributes (-1, "Pixbuf", new Gtk.CellRendererPixbuf (), "pixbuf", 0);
        tree_view.insert_column_with_attributes (-1, "Theme Name", new Gtk.CellRendererText (), "text", 1);

        // List of all available themes
        foreach (var t in themes) {
            foreach (var item in t.items) {
                theme_items += item;
                Gtk.TreeIter iter;
                tree_model.append (out iter);
                tree_model.set_value (iter, 0, item.pixbuf);
                tree_model.set_value (iter, 1, item.name);
            }
        }

        // Register listener for change in pattern selection
        tree_selection = tree_view.get_selection ();
        tree_selection.changed.connect (on_selection);

        // Set up visual
        tree_view.columns_autosize ();
        tree_view.set_headers_visible (false);
        tree_view.set_cursor (new Gtk.TreePath.from_string ("0"), null, false);
        var scrolled = new Gtk.ScrolledWindow (null, null);
        scrolled.set_policy (Gtk.PolicyType.NEVER, Gtk.PolicyType.ALWAYS);
        scrolled.add (tree_view);
        
        return scrolled;
    }

    /**
     * Callback whenever a different theme is selected.
     * Side effect: requested_theme
     */
    private void on_selection () {
        if (tree_selection.count_selected_rows () == 1) {
            Gtk.TreeIter iter;
            Gtk.TreeModel model;
            tree_selection.get_selected (out model, out iter);
            int n = int.parse (model.get_path (iter).to_string ());
            var item = theme_items[n];
            theme.free ();
            theme = item.engine.select (item.path, size.width, size.height);
            if (theme == null) {
                theme = themes[0].select ("0", size.width, size.height);
                item = theme_items[0];
            }
            changed ();
        }
    }

    /**
     * Select a random theme.
     * Side effect: requested_theme, indirectly through on_selection.
     */
    public void random () {
        int rnd = Random.int_range (0, theme_items.length);
        var path = new Gtk.TreePath.from_string (rnd.to_string ());
        tree_selection.unselect_all ();
        tree_view.scroll_to_cell (path, null, false, 0, 0);
        tree_selection.select_path (path);
    }

    public void resize (Gtk.Allocation allocation) {
        size = allocation;
        theme.resize (size.width, size.height);
        changed ();
    }
}//class ThemeSwitch
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
