/* File:      cp_opt.P
** Author(s): Swift, Warren
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: cp_opt.P,v 1.27 2013-05-14 13:19:05 dwarren Exp $
** 
*/

:- import select/3 from basics.

:- mode cp_opt(?,?).
cp_opt(module(Symtab,DcList,ClList,QrList,Par),
       module(Symtab,DcList,ClList1,QrList,Par)) :-
	cp_opt0(ClList, ClList1, Symtab).

cp_opt0([], [],_Symtab).
cp_opt0([pred(Sym,Pscs,Oth)|CList0],Outlist,Symtab):-
	sym_type(Sym, tabled(_Tabind,_Ar)),!,
	cp_opt_block(pred(Sym,Pscs,Oth), pred(Sym1,Pscs1,Oth1), 
					 Symtab,Switch),
	(Switch =:= 0 -> Outlist = [pred(Sym,Pscs,Oth)|CList1]
	 ; Outlist = [pred(Sym1,Pscs,Oth),pred(Sym,Pscs1,Oth1)|CList1]
	),
	cp_opt0(CList0, CList1,Symtab).
cp_opt0([One|Rest],[One|Rest1],Symtab):-
	cp_opt0(Rest,Rest1,Symtab).

/*======================================================================*/
/* BigPicture =\= 0 if the code needs transformation, that is, if
   it cannot be optimized.  Conceptually, the default consists of
   transforming a tabled predicate

	tabpred:- body1.
	tabpred:- body2.

   into
	tabpred:- '_$tabpred'.
	
	'_$tabpred':- body1.
	'_$tabpred':- body2.

   where _$tabpred is NOT tabled.  Optimization in this context consists
   of forgoing the transformation.  

   Currently the transformation is made when
	1) One of the bodies has a hard cut.
	or 
	2) When the indexing will generate more than one try for any
	entry point (at least if I got my check right).

   As can be seen from the .H file, I have cannibalized other pieces
   of code to do this.  I have documented this code in their original 
   source files, where needed.						*/   
/*======================================================================*/

cp_opt_block(pred(Sym,Pscs,Oth),pred(Sym1,Pscs1,Oth1),Symtab,BigPicture):-
	have_hardcut(Pscs,Hc),
	multiple_index_groups(Sym,Pscs,Mult),
	uses_approx(Sym,Approx),
	BigPicture is Hc + Mult + Approx,
	cp_opt_block1(BigPicture,
		      pred(Sym,Pscs,Oth),pred(Sym1,Pscs1,Oth1),Symtab).

cp_opt_block1(0,_,_,_):-!.
cp_opt_block1(_,pred(Sym,_Pscs,_Oth),pred(Sym1,Pscs1,Oth1),Symtab):-
	sym_name(Sym,Pred,Arity),
	transform_pred(Pred,Pred1),
	functor(Head,Pred,Arity),      % p(...)
	functor(NewCall,Pred1,Arity),     % '_$p'(...)
	bind_args(Arity,Head,NewCall),
	parse_preddef_closed(NewCall,Head, ClList1, Symtab),
	ClList1 =[pred(Sym1,_Pscs1,_Oth1)|_T],
%	parse_preddef(Head, NewCall, ClList, Symtab),
	cp_opt_table_body(Sym,Head, NewCall,ClList, Symtab),
	ClList =[pred(_Sym1,Pscs1,Oth1)|_T].

%:- import console_writeln/1 from standard.
cp_opt_table_body(Sym,_,_,ClList1, Symtab):-
	sym_type(Sym,approximation(ApproxSpec)),!,
	sym_name(Sym,Pred,Arity),
	transform_pred(Pred,Pred1),
	functor(Head,Pred,Arity),      % p(...)
	functor(NewCall,Pred1,Arity),     % '_$p'(...)
	bind_args(Arity,Head,NewCall),
	get_calling_proc(ApproxSpec,Breg,ApproxCall),
	parse_preddef_closed(Head, (get_breg(Breg),
	               NewCall,
		       ApproxCall),ClList1, Symtab),
	ensure_imported_list([im(get_breg/1,machine),
			      im(early_complete_on_nth/2,tables)],Symtab).
cp_opt_table_body(_Sym,Head,NewCall, ClList1, Symtab):-
	parse_preddef_closed(Head, NewCall,ClList1, Symtab).

get_calling_proc(use_first(N),Breg,early_complete_on_nth(Breg,N)).

transform_pred(Pred,Pred1):-
	concat_atom([Pred,'_$'],Pred1).

bind_args(0,_,_):-!.
bind_args(N,Head,Body):-
	arg(N,Head,A),
	arg(N,Body,A),
	N1 is N - 1,
	bind_args(N1,Head,Body).	

multiple_index_groups(Sym, Pscs, Res):-
	sym_name(Sym,_Pred,Arity),
	( Arity =:= 0 -> Res = 0
	; sym_index(Sym, Narg),
	  cpo_index_argpats(Pscs,Narg,Argpat),
	  index_split(Argpat,Groups),
	  ( member(arglabel(g,_,_),Groups) -> Res = 1 ; Res = 0 )
	).

cpo_index_argpats([],_,[]).
cpo_index_argpats([clause(Arglist,_,_)|CRest], N, 
			[arglabel(Pattern,NewArg,_L)|APatternRest]) :-
        nthmember(Arg, Arglist, N),
	cpo_index_argpat(Arg,Pattern, NewArg),
	cpo_index_argpats(CRest,N,APatternRest).

cpo_index_argpat(constant(Sym), n, []) :- sym_name(Sym, [], 0).
cpo_index_argpat(integer(X), i, X).
cpo_index_argpat(real(X), i, X).
cpo_index_argpat(constant(C), c, C).
cpo_index_argpat(varocc(_), v, v).
cpo_index_argpat(string(_), l, l).
cpo_index_argpat(structure(Sym,_), l,l) :- sym_name(Sym, '.', 2).
cpo_index_argpat(structure(Sym,_), s,Sym).

uses_approx(Sym,Approx):- 
	sym_type(Sym,approximation(_ApproxSpec)),
	\+ sym_type(Sym,aggregation(_AggSpec))
	-> Approx = 1 ; Approx = 0.

/************************************************************************/
/* Given an aggregate declaration, now of the form, e.g.
:- table short_path(_,_,min(_,_,_)*99999999)

In general the spec is either 
    POTerm where POTerm is F/2 or POTerm = f(_,_)
    Jointerm*Id where Jointerm is f/3 or JoinTerm = f(_,_,_)
                      and Id = term or ID = i/1.

The user is responsible for having defined the binary function min/3.

transforms clauses of short_path adding:
     short_path(A,B,C) :- bagReduce(X,'_$short_path'(A,B,X),C,min(_,_),99999999).
and modifying short_path clauses as:
     '_$short_path'(..) :- short_path_body1.
     '_$short_path'(..) :- short_path_body2.

Legal aggregation specs are of the form:
   reduceFun(_,_,_)-Identity  which will be transformed as above.
   reduceFun/3-Identity  which will be transformed similarly
   ordRel(_,_)  which will cause a tranformation to (e.g.)
       short_path(A,B,C) :- bagPO(X,'_$short_path'(A,B,X),C,ordRel(_,_)).
   ordRel/2  which will be treated as ordRel(_,_).

									*/
/************************************************************************/

:- mode perform_tabling_transforms(?,?).
perform_tabling_transforms(Mod1,Mod2):-
	aggregation_transform(Mod1,Mod2).
%	approximation_transform(Mod2,Mod2).

aggregation_transform(module(Symtab,DcList,ClList0,QrList0,Par),
		      module(Symtab,DcList,ClList,QrList,Par)) :-
	extract_symlist([aggregation(_),(dynamic)],SymList,Symtab),
	(SymList == []
	 -> ClList1 = ClList0,
	    QrList = QrList0
	 ;  ensure_imported_list([im(set_rename_dyn_pred/2,xsb_hook)
				  ],Symtab),
	    syms_to_undef_preds(SymList,ClList0,ClList1),
	    fix_main_dynamics(SymList,Symtab,QrList0,QrList)
	),
	aggr_trans(ClList1,ClList,Symtab).

syms_to_undef_preds([],ClList,ClList).
syms_to_undef_preds([Sym|Syms],ClList0,ClList) :-
    syms_to_undef_preds(Syms,[pred(Sym,[],_)|ClList0],ClList).

fix_main_dynamics([],_Symtab,QrList,QrList).
/*fix_main_dynamics([Sym|SymList],Symtab,QrList0,QrList) :- !,
    Sym = sym(Pred,Arity,_Props,_Oth),
    QrList0 = [pred(MainSym,Clauses,Stuff)],
    (MainSym = sym('_$main',0,_,_) -> true ; abort(error)),
    QrList1 = [pred(MainSym,Clauses,Stuff)],
    fix_main_dynamics(SymList,Symtab,QrList1,QrList). */
fix_main_dynamics([Sym|SymList],Symtab,QrList0,QrList) :-
    Sym = sym(Pred,Arity,_Props,_Oth),
    QrList0 = [pred(MainSym,Clauses,Stuff)],
    (MainSym = sym('_$main',0,_,_)
     ->	true
     ;	abort('ERROR: wrong _$main in aggregation?')
    ),
    transform_pred(Pred,DPred),
    integer(Arity),
    sym_insert(DPred,Arity,[dynamic],Symtab,DSym),
    replace_dyn_sym(Clauses,Sym,DSym,DClauses0),
    functor(MHead,Pred,Arity),
    MHead =.. [Pred|Args],
    DHead =.. [DPred|Args],
    trans_goals((set_rename_dyn_pred(MHead,DHead)->fail),CmdCode,Symtab,0,VA),
    ensure_gensym_num_larger(VA),
    DClauses = [clause([],CmdCode,_)|DClauses0],
    QrList1 = [pred(MainSym,DClauses,Stuff)],
    fix_main_dynamics(SymList,Symtab,QrList1,QrList).

replace_dyn_sym([],_Sym,_DSym,[]).
replace_dyn_sym([Clause|Clauses],Sym,DSym,[DClause|DClauses]) :-
    DynSym = sym((dynamic),1,_,_),
    (Clause = clause([],if(goal(DynSym,[structure(Sym,Args)]),Then),More)
     ->	DClause = clause([],if(goal(DynSym,[structure(DSym,Args)]),Then),More)
     ;	DClause = Clause
    ),
    replace_dyn_sym(Clauses,Sym,DSym,DClauses).

aggr_trans([],[],_SymTab).
aggr_trans([pred(Sym,Pscs,Oth)|ClList0],ClList,SymTab) :-
	sym_type(Sym,aggregation(Spec)),
	!,
	(extract_aggr_spec(Spec,ArgNo,ExistArgList,AggSpec)
	 ->	aggr_trans_pred(Sym,Pscs,Oth,ArgNo,ExistArgList,AggSpec,ClList,SymTab)
	 ;	warning(('Illegal aggregation specification: ',Spec)),
		ClList = [pred(Sym,Pscs,Oth)|_]
	),
	tail_var(ClList,ClList1),
	aggr_trans(ClList0,ClList1,SymTab).
aggr_trans([pred(Sym,Pscs,Oth)|ClList0],ClList,SymTab) :-
	sym_type(Sym,subsumption_modes(Modes)),
	!,
	subsumption_modes_trans_pred(Sym,Pscs,Oth,Modes,ClList,SymTab),
	tail_var(ClList,ClList1),
	aggr_trans(ClList0,ClList1,SymTab).	
aggr_trans([pred(Sym,Pscs,Oth)|ClList0],ClList,SymTab) :-
	sym_type(Sym,table_index(IndexSpec)),
	!,
	(sym_type(Sym,tabled(_,_))
	 -> sym_name(Sym,Pred,Arity),
	    warning(('table_index-ed predicate ',Pred/Arity,' should *not* be explicitly tabled.'))
	 ;  true
	),
	table_index_trans_pred(Sym,Pscs,Oth,IndexSpec,ClList,SymTab),
	tail_var(ClList,ClList1),
	aggr_trans(ClList0,ClList1,SymTab).	
aggr_trans([Pred|ClList0],[Pred|ClList],SymTab) :-
	aggr_trans(ClList0,ClList,SymTab).

aggr_trans_pred(Sym,Pscs,Oth,ArgNo,ExistArgList,AggSpec,[pred(Sym1,Pscs,Oth)|ClList2],SymTab) :-
	sym_name(Sym,Pred,Arity),
	length(PArgs,Arity),
	length(P1Args,Arity),
	equate_but_argno(PArgs,P1Args,1,ExistArgList,ExistVarList,Var,Var1,ArgNo),
	transform_pred(Pred,Pred1),
	PredTerm =.. [Pred|PArgs], Pred1Term =.. [Pred1|P1Args],
	generate_agg_clauses(AggSpec,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab),
	length(Args,Arity),
	NewTerm =.. [Pred1|Args],
	!,
	parse_preddef_closed(NewTerm,true,[pred(Sym1,_,_)],SymTab).



%% only for debugging
:- export mywriteln/1.
%mywriteln(Term):- standard:writeln(Term).
mywriteln(_Term).

generate_agg_clauses(po(OpSpec),Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :- !,
	functor(PredTerm,Pred,Arity),
	(spec_skel(OpSpec,OpSpec1,2)
	 ->	true
	 ;	syntax_error(['Improper po specification in tabling declaration for ',Pred,'/',Arity])
	),
	sym_propin(tabled,Sym),
	copy_term(OpSpec1,OpSpec2),
	arg(1,OpSpec1,OVar), arg(2,OpSpec1,Var1),
	arg(1,OpSpec2,Var1), arg(2,OpSpec2,OVar),
	Body = ('_$savecp'(Breg),
		breg_retskel(Breg,Arity,Skel,Cs),
		excess_vars(Skel,ExistVarList,[],Vars),
		copy_term(t(Vars,Skel,Var),t(Vars,OSkel,OVar)),
		Pred1Term,
		\+ ('_$$get_returns'(Cs,OSkel,Leaf),
		    (OVar == Var1 ; OpSpec1)
		   ),
		findall(Leaf,('_$$get_returns'(Cs,OSkel,Leaf),once(OpSpec2)),Leaves),
		('_$member'(Leaf,Leaves),
		 delete_return(Cs,Leaf),
		 fail
		 ;	
		 Var = Var1
		)
	       ),
	parse_preddef_closed(PredTerm,Body,ClList2,SymTab),
	globalize('/',2,SymTab),
	globalize(var,0,SymTab),
	ensure_imported_list([im(once/1,standard),
			      im(breg_retskel/4,machine),
			      im(excess_vars/4,setof),
			      im(findall/3,setof),
			      im('_$member'/2,basics),
			      im(copy_term/2,basics),
			      im(instantiation_error/3,error_handler),
			      im('_$$get_returns'/3,tables),
			      im('delete_return'/2,tables)
			      ],SymTab).

generate_agg_clauses(AggrSpec,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :-
	(AggrSpec = fold(OpSpec,Id)
	 ;
	 AggrSpec = lattice(OpSpec), Id = no_identity
	),
	!,
	functor(PredTerm,Pred,Arity),
	(spec_skel(OpSpec,OpSpec1,3)
	 ->	true
	 ;	syntax_error(['Improper join specification in tabling declaration for ',Pred,'/',Arity])
	),
	(Id = F/1
	 ->	GetID =.. [F,OVar],
		Init = (GetID, OpSpec1)
	 ; Id == no_identity
	 ->	Init = (Var1 = Var)
	 ;	Init = (OVar = Id, OpSpec1)
	),
	sym_propin(tabled,Sym),
	arg(1,OpSpec1,OVar), arg(2,OpSpec1,Var1), arg(3,OpSpec1,Var), 
	get_actions_for_lattice_aggr_success(Sym,[Cs,Leaf,Var,OVar,Breg],Actions),
	Body = ('_$savecp'(Breg),
		breg_retskel(Breg,Arity,Skel,Cs),
		(nonvar(Var) -> instantiation_error(Pred/Arity,Arity,var) ; true),
		excess_vars(Skel,[Var|ExistVarList],[],Vars),
		copy_term(t(Vars,Skel,Var),t(Vars1,OSkel,OVar)),
		DefTerm = ifUndef(XX,YY), % if seen undefined
		XX = 0,			  % undef seen?
		YY = 0,			  % varibles seen in groupby values?
		Pred1Term,
		xwam_state(2,DelayReg),
		(XX =:= 0, DelayReg =:= 0 % opt for all defined
		 ->	(YY =:= 0, ground(Vars) % opt for all groupby vals ground
			 ->	(Vars = Vars1,
				 '_$$get_returns'(Cs,OSkel,Leaf)
				 ->	OpSpec1,
					Var \== OVar,
				    	%%\+ variant(Var,OVar),
				    	Actions
				 ;	Init
				)
			 ;	(YY =:= 0 -> term_set_arg(DefTerm,2,1,1) ; true),
				findall(t(OVar,Var,Leaf1,Sub,RSub),
					('_$$get_returns'(Cs,OSkel,Leaf1),
					 OpSpec1,
					 (subsumes_chk(Vars1,Vars) -> Sub=1 ; Sub=0),
					 (subsumes_chk(Vars,Vars1) -> RSub=1 ; RSub=0)
					),
					AllRets),
				('_$member'(t(OVar,Var,Leaf,1,_),AllRets),
				 Var == OVar
				 ->	fail % a subsuming answer as good or better
				 ;	('_$member'(t(OVar,Var,Leaf,_,1),AllRets),
					 Var \== OVar,
					 %%\+ variant(Var,OVar),
					 Actions, % delete answer this one is better than
					 fail
					 ;
					 Init % add this answer
					)
				)
			)
		 ;	term_set_arg(DefTerm,1,1,1), % undef, must handle from now on
			Vars = Vars1,
			findall(t(OSkel,Leaf1,TV1,Var),
				('_$$get_returns_and_tvs'(Cs,OSkel,Leaf1,TV1),
				 OpSpec1),
				RetList),
			(DelayReg =:= 0, % all deletion before returning any ans, otw bug
			 '_$member'(t(OSkel,Leaf,_,Var),RetList),
			 Var \== OVar,
			 Actions,
			 fail
			 ;	
			 \+ '_$member'(t(_,_,t,_),RetList),
			 Init
			 ;	
			 '_$member'(t(OSkel,Leaf,TV1,Var),RetList),
			 Var \== OVar,
			 (DelayReg =:= 0
			  ->	 (TV1 == t
				  ->	 true
				  ;	 undefined
				 )
			  ;	 true
			 )
			)
		)
	       ),
	parse_preddef_closed(PredTerm,Body,ClList2,SymTab),
	%%globalize('/',2,SymTab),
	%%globalize(var,0,SymTab),
	ensure_imported_list([im(nonvar/1,standard),
			      im((->)/2,standard),
			      im((';')/2,standard),
			      im((',')/2,standard),
			      im((\==)/2,standard),
			      im((==)/2,standard),
			      im((=:=)/2,standard),
			      im((=)/2,standard),
			      im(breg_retskel/4,machine),
			      im(excess_vars/4,setof),
			      im(findall/3,setof),
			      im(copy_term/2,basics),
			      im(instantiation_error/3,error_handler),
			      im('_$$get_returns_and_tvs'/4,tables),
			      im('_$$get_returns'/3,tables),
			      im(delete_return/2,tables),
			      im(undefined/0,tables),
			      im(xwam_state/2,machine),
			      im('_$member'/2,basics),
			      im(variant/2,subsumes),
			      im(subsumes_chk/2,subsumes),
			      im(early_complete_on_nth/2,tables),
			      im(term_set_arg/4,machine),
			      im(early_complete_on_delta/4,tables)
			      ],SymTab).


generate_agg_clauses(termset,_Sym,PredTerm,Var1,[],Pred1Term,Var,ClList2,SymTab) :- !,
	functor(Pred1Term,NPred,Arity),
	sym_insert(NPred,Arity,[tabled,use_subsumptive_tabling],SymTab,_),
	Body = Pred1Term,
	excess_vars(Pred1Term,[Var1],[],Vars),
	copy_term(t(Pred1Term,Var1,Vars),t(Pred1Temp,AbsVar,Vars)),
	Var = {AbsVar:Pred1Temp},
	parse_preddef_closed(PredTerm,Body,ClList2,SymTab).
generate_agg_clauses(OpSpec-Iden,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :- !,
	generate_agg_clauses(fold(OpSpec,Iden),Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab).
generate_agg_clauses(OpSpec,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :- !,
	generate_agg_clauses(po(OpSpec),Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab).
generate_agg_clauses(_OpSpec,_Sym,PredTerm,_Var1,_ExistVarList,_Pred1Term,_Var,_ClList2,_SymTab) :-
	functor(PredTerm,Pred,Arity),
	syntax_error(['Improper join specification in tabling declaration for ',Pred,'/',Arity]).

%%---

:- mode get_return_w_tv(+,?,?,?).
get_return_w_tv(Cs,OSkel,Leaf,TV):- 
	xwam_state(2,DelayReg_start),mywriteln(start(DelayReg_start)),
	'_$$get_returns'(Cs,OSkel,Leaf),
	xwam_state(2,DelayReg), mywriteln(end(DelayReg)),
	(DelayReg = DelayReg_start    ->     TV = true         ;      TV = undefined).

/* if new answer is undefined it should succeed if it is the same as
 * the old undefined answer -- it will later fail 

If new is undefined it needs only be compared to the previous greatest
true answer (if any).  If greater, then succeed

If new is true and greater than previous true, delete all undefined
that it is greater than.

So, in either case compare new to the prior true, then branch on its truth value.

*/
:- mode call_join_op(+,?,?,?,?,?).
call_join_op(New,Old,Join,Var,OVar,Actions):- 
	mywriteln(call_join_op(New,Old,Join,Var,OVar,Actions)),
	call(Join),
 	Var \== OVar,
	(New = t -> call(Actions) ; true).

%---
	
get_actions_for_lattice_aggr_success(Sym,[Cs,Leaf,Var,Ovar,Breg],Actions):-
	sym_type(Sym,approximation(ApproxSpec)),!,
%	console_writeln(using_approximation(ApproxSpec)),
	(ApproxSpec = use_first(N) -> 
	    Actions = (delete_return(Cs,Leaf),early_complete_on_nth(Breg,N))
         ; ApproxSpec = epsilon(E) -> 
	    Actions = (delete_return(Cs,Leaf),early_complete_on_delta(Breg,Var,Ovar,E)) ).
get_actions_for_lattice_aggr_success(_Sym,[Cs,Leaf|_],delete_return(Cs,Leaf)).

spec_skel(F/A,Term,Ar) :- integer(A), nonvar(F), !, Ar = A, functor(Term,F,A).
spec_skel(Term0,Term,Ar) :- term_psc(Term0,PSC), psc_arity(PSC,Ar0), Ar0 = Ar, term_new(PSC,Term).
  %%functor(Term0,_,Ar), copy_term(Term0,Term).

equate_but_argno([],[],_,[],[],_,_,_).
equate_but_argno([X|PArgs],[Y|P1Args],CurArg,ArgList,VarList,A,B,ArgNo) :-
	(ArgNo =:= CurArg
	 ->	A = X, Y = B,
		ArgList0 = ArgList, VarList0 = VarList
	 ;	(ArgList = [CurArg|ArgList0]
		 ->	VarList = [Y|VarList0]
		 ;	X = Y,
			ArgList0 = ArgList, VarList0 = VarList
		)
	),
	CurArg1 is CurArg + 1,
	equate_but_argno(PArgs,P1Args,CurArg1,ArgList0,VarList0,A,B,ArgNo).

:- mode extract_aggr_spec(?,?,?,?).
extract_aggr_spec(Spec,ArgNo,ExistArgList,AggSpec) :-
	functor(Spec,_,Arity),
	get_aggr_spec0(1,Spec,Arity,ExistArgList,ArgNo,AggSpec).

get_aggr_spec0(ArgNo,Spec,Arity,ExistArgList,OpArgNo,OpSpec) :-
	(ArgNo > Arity
	 ->	ExistArgList = []
	 ;	arg(ArgNo,Spec,ArgVal),
		ArgNo1 is ArgNo+1,
		(non_cpopt_var(ArgVal)
		 ->	(ArgVal == (^)
			 ->	ExistArgList = [ArgNo|EAL1],
				get_aggr_spec0(ArgNo1,Spec,Arity,EAL1,OpArgNo,OpSpec)
			 ;	var(OpArgNo),  % only set once
				OpSpec = ArgVal, OpArgNo = ArgNo,
				get_aggr_spec0(ArgNo1,Spec,Arity,ExistArgList,OpArgNo,OpSpec)
			)
		 ;	get_aggr_spec0(ArgNo1,Spec,Arity,ExistArgList,OpArgNo,OpSpec)
		)
	).

non_cpopt_var(Term):-
	nonvar(Term),
	\+ (Term = '$$var'(Id), integer(Id)).

ensure_imported_list([],_SymTab).
ensure_imported_list([im(PA,Mod)|Imps],SymTab) :-
    ensure_imported(PA,Mod,SymTab),
    ensure_imported_list(Imps,SymTab).

ensure_imported(Pred/Arity,Mod,SymTab) :-
	sym_insert(Mod,0,[module],SymTab,_),
	sym_insert(Pred,Arity,[],SymTab,Sym),
	(sym_prop(global,Sym)
	 ->	true
	 ;	sym_propin(im(Mod),Sym)
	).

globalize(Pred,Arity,SymTab) :-
	sym_insert(Pred,Arity,[],SymTab,Sym),
	sym_propin(global,Sym).

%% e.g. use_subsumptive_tabling p(+,-,-), p(-,-,-).
/* transform predicate for abstracting subsumptive calls. */
subsumption_modes_trans_pred(Sym,Pscs,Oth,Modes,[pred(Sym1,Pscs,Oth)|NewClList],SymTab) :-
	sym_name(Sym,Pred,Arity),
	sym_insert(Pred,Arity,[tabled,use_subsumptive_tabling],SymTab,_),
	length(PArgs,Arity),
	transform_pred(Pred,Pred1),
	build_subs_abs_clause_body(Modes,Pred,Pred1,PArgs,Body),
	PredCall =.. [Pred|PArgs],
	%%print_rule(abs_rule,PredCall,Body),
	parse_preddef_closed(PredCall,Body,NewClList,SymTab),
	NewTerm =.. [Pred1|PArgs],
	parse_preddef_closed(NewTerm,true,[pred(Sym1,_,_)],SymTab), % add to symtab
	ensure_imported(table_error/1,error_handler,SymTab),
	ensure_imported(term_to_atom/3,string,SymTab).

/*** for when debugging ***
print_rule(Label,Head,Body) :-
	(do_all
	 num_vars:numbervars(p(Head,Body)),
	 standard:write(userout,Label),
	 standard:writeln(userout,':'),
	 standard:writeq(userout,Head),standard:writeln(userout,' :-'),
	 standard:write(userout,'  '),standard:writeq(userout,Body),
	 standard:writeln(userout,'.')
	).
***/

%% Modes must be open tailed...
build_subs_abs_clause_body(Modes,Pred,Pred1,PArgs,Body) :-
	(var(Modes)
	 ->	length(PArgs,Arity),
		concat_atom(['Illegal Mode in call to ',Pred,'/',Arity],Message),
		Body = table_error(Message)
	 ;	Modes = [Mode|ModeTail],
		build_nonvar_plus_var_minus_cl(Mode,PArgs,NonVarTests,VarTests),
		build_plus_orig_minus_equates(Mode,PArgs,VListMNew,EquateList),
		list_to_commalist(EquateList,EquateVars),
		PredCall =.. [Pred|VListMNew],
		Pred1Call =.. [Pred1|PArgs],
		Body = (NonVarTests -> (VarTests -> Pred1Call ; (PredCall, EquateVars)) ; BodyRest),
		build_subs_abs_clause_body(ModeTail,Pred,Pred1,PArgs,BodyRest)
	).
		 
build_nonvar_plus_var_minus_cl(Mode,PArgs,NonVarTests,VarTests) :-
	build_nonvar_plus_var_minus(Mode,PArgs,NonVarTestsL,VarTestsL),
	list_to_commalist(NonVarTestsL,NonVarTests),
	list_to_commalist(VarTestsL,VarTests).

build_nonvar_plus_var_minus([],[],[],[]).
build_nonvar_plus_var_minus([+|Mode],[Var|PArgs],[nonvar(Var)|NVTests],VarTests) :-
	build_nonvar_plus_var_minus(Mode,PArgs,NVTests,VarTests).
build_nonvar_plus_var_minus([-|Mode],[Var|PArgs],NVTests,[var(Var)|VarTests]) :-
	build_nonvar_plus_var_minus(Mode,PArgs,NVTests,VarTests).

list_to_commalist([],true) :- !.
list_to_commalist([A],A) :- !.
list_to_commalist([A|B],(A,Bc)) :- list_to_commalist(B,Bc).

build_plus_orig_minus_equates([],[],[],[]).
build_plus_orig_minus_equates([+|Mode],[Var|PArgs],[Var|VListMNew],EquateList) :-
	build_plus_orig_minus_equates(Mode,PArgs,VListMNew,EquateList).
build_plus_orig_minus_equates([-|Mode],[Var|PArgs],[NV|VListMNew],[Var=NV|EquateList]) :-
	build_plus_orig_minus_equates(Mode,PArgs,VListMNew,EquateList).


/* generate code for multiple subsumptive tables in different argument
   orders for arbitrary indexing. */
table_index_trans_pred(Sym,Pscs,Oth,Indexes,ClList,SymTab) :-
	sym_name(Sym,Pred,Arity),
	collect_orders_from_indexes(Indexes,Pred,Arity,OrdSets,IndexOrds),
	collect_inds_and_perms(OrdSets,IndexOrds,IndPerms0),
	parsort(IndPerms0,[asc(2)],0,IndPerms1),
	simplify_ind_and_perms(IndPerms1,IndPerms),
%	console_writeln(simplify_ind_and_perms(IndPerms1,IndPerms)),
	gen_sel_clause(Pred,Arity,IndPerms,(SelHead:-SelBody)),
	ClList = [pred(Sym1,Pscs,Oth)|NewClList],
	%%print_rule(sel_rule,SelHead,SelBody), % dsw
	parse_preddef_closed(SelHead,SelBody,NewClList,SymTab),
	tail_var(NewClList,NewClList1),

	IndPerms = [ip(Set1,_,_)|IndPerms2],
	intersect_ips(IndPerms2,Set1,NonAbsPoss),
	upto_n(1,Arity,BaseOrdI),
	(select(BaseOrdI,IndexOrds,IOrds)
	 ->	BaseOrd = BaseOrdI
	 ;	IndexOrds = [BaseOrd|IOrds]
	),
	concat_atom([Pred,'$'],PredD),
	concat_atom([PredD,base],'_',OrigPred),
	functor(OrigHead,OrigPred,Arity),
	parse_preddef_closed(OrigHead,true,[pred(Sym1,_,_)],SymTab), % add to symtab
	
	gen_ind_clause(BaseOrd,BaseOrdI,Pred,Arity,NonAbsPoss,_BasePred,OrigPred,
		       (BaseCallHead:-BaseCallBody)),
	functor(BaseCallHead,BPred,Arity),
	sym_insert(BPred,Arity,[tabled,use_subsumptive_tabling],SymTab,_),
	%%print_rule(base_rule,BaseCallHead,BaseCallBody), %dsw
	parse_preddef_closed(BaseCallHead,BaseCallBody,NewClList1,SymTab),
	tail_var(NewClList1,NewClList2),

	concat_atom([PredD|BaseOrd],'_',BOPred),
	gen_ind_clauses(IOrds,BaseOrd,Pred,Arity,NonAbsPoss,BOPred,NewClList2,SymTab),
	
	% generate_rest_index_clauses
	ensure_imported_list([im(table_error/1,error_handler),
			      im(term_to_atom/3,string),
			      im(pred_table_pred/2,tables),
			      im(assert/1,assert),
			      im(retractall/1,assert),
			      im(numbervars/1,usermod)
			      ], SymTab).

gen_ind_clauses([],_BaseOrd,_Pred,_Arity,_NonAbsPoss,_OrdPred,_ClList,_SymTab).
gen_ind_clauses([IOrd|IOrds],BaseOrd,Pred,Arity,NonAbsPoss,OrdPred,ClList,SymTab) :-
	gen_ind_clause(IOrd,BaseOrd,Pred,Arity,NonAbsPoss,IPred,OrdPred,(Head:-Body)),
	functor(HeadCopy,IPred,Arity),
	parse_preddef_closed(HeadCopy,true,_,SymTab), % add to symtab	
	sym_insert(IPred,Arity,[tabled,use_subsumptive_tabling],SymTab,_),
	%%print_rule(ind_rule,Head,Body), % dsw
	parse_preddef_closed(Head,Body,ClList,SymTab),
	tail_var(ClList,ClList1),	
	gen_ind_clauses(IOrds,BaseOrd,Pred,Arity,NonAbsPoss,OrdPred,ClList1,SymTab).

gen_ind_clause(IOrd,BaseOrd,Pred,Arity,NonAbsPoss,IPred,OrdPred,
	       (Head:-(AbsVarTests->BaseCall;AbsGoal,EquateAbsVars))) :-
	concat_atom([Pred,'$'],PredD),
	concat_atom([PredD|IOrd],'_',IPred),
	functor(Head,IPred,Arity),
	Head =.. [IPred|HeadVars],

	compl(1,Arity,NonAbsPoss,AbsPoss),
	select_indir_vars(AbsPoss,IOrd,HeadVars,AbsVars),
	build_var_test(AbsVars,AbsVarTests),

	select_indir_vars(BaseOrd,IOrd,HeadVars,OrdVars),
	BaseCall =.. [OrdPred|OrdVars],

	functor(AbsGoal,IPred,Arity),
	AbsGoal =.. [IPred|AbsGoalVars],
	equate_vars(NonAbsPoss,IOrd,AbsGoalVars,HeadVars),
	build_equates_as_nec(AbsGoalVars,HeadVars,EquateAbsVars).

intersect_ips([],IntSet,IntSet).
intersect_ips([ip(Set,_,_)|IPSets],IntSet0,IntSet) :-
	intersect_set(Set,IntSet0,IntSet1),
	intersect_ips(IPSets,IntSet1,IntSet).

intersect_set(_,[],[]) :- !.
intersect_set([],_,[]).
intersect_set(S1,S2,Intersect) :-
	S1 = [E1|R1],
	S2 = [E2|R2],
	(E1 < E2
	 ->	intersect_set(R1,S2,Intersect)
	 ; E2 < E1
	 ->	intersect_set(S1,R2,Intersect)
	 ;	Intersect = [E1|Int1],
		intersect_set(R1,R2,Int1)
	).
gen_sel_clause(Pred,Arity,IndPerms,(Head:-Body)) :-
	functor(Head,Pred,Arity),
	Head =.. [Pred|HeadArgs],
	gen_tests(IndPerms,Pred,HeadArgs,Body).

%TES: better error handling.
gen_tests([IndPerm],Pred,HeadArgs,Condition) :- !,
	gen_test_call(IndPerm,Pred,HeadArgs,Test,Call),
	(Test == true
	 ->	Condition = Call
	 ;	length(HeadArgs,Arity),
		Msg = ['The table_indexed predicate ',Pred,'/',Arity,' executed a subgoal of non-declared mode (a predicate symbol suffix gives argument order): ',Atom],
		Condition = (Test->Call ; term_to_atom(Call,Atom,[numbervars(true)]),table_error(Msg))		
	).
gen_tests([IndPerm|IndPerms],Pred,HeadArgs,(Test->Call;Clause)) :-
	gen_test_call(IndPerm,Pred,HeadArgs,Test,Call),
	gen_tests(IndPerms,Pred,HeadArgs,Clause).

simplify_ind_and_perms([],[]).
simplify_ind_and_perms([IP],[IP]) :- !.
simplify_ind_and_perms([IP1|IPs0],IPs) :-
	(IP1 = ip(Set1,_,Ord),
	 IPs0 = [ip(Set2,_,Ord)|_],
	 subset(Set2,Set1)
	 ->	simplify_ind_and_perms(IPs0,IPs)
	 ;	IPs = [IP1|IPs1],
		simplify_ind_and_perms(IPs0,IPs1)
	).

equate_vars([],_,_,_).
equate_vars([Pos|Poss],OPoss,VarsA,VarsB) :-
	ith(NPos,OPoss,Pos),
	ith(NPos,VarsA,Var),
	ith(NPos,VarsB,Var),
	equate_vars(Poss,OPoss,VarsA,VarsB).
	
build_equates_as_nec([],[],true) :- !.
build_equates_as_nec([Var1|Vars1],[Var2|Vars2],Equates) :-
	(Vars1 == []
	 ->	(Var1 == Var2
		 ->	Equates = true
		 ;	Equates = (Var1=Var2)
		)
	 ;	(Var1 == Var2
		 ->	Equates = Equates1
		 ;	Equates = ((Var1=Var2),Equates1)
		),
		build_equates_as_nec(Vars1,Vars2,Equates1)
	).

select_indir_vars([],_,_,[]).
select_indir_vars([Pos|Poss],Ord,HeadVars,[Var|AbsVars]) :-
	ith(Ind,Ord,Pos),
	ith(Ind,HeadVars,Var),
	select_indir_vars(Poss,Ord,HeadVars,AbsVars).

build_var_test([],true) :- !.
build_var_test([Var],var(Var)) :- !.
build_var_test([Var|Vars],(var(Var),VarTests)) :-
	       build_var_test(Vars,VarTests).

compl(I,Arity,_,[]) :- I > Arity.
compl(I,Arity,List,Compl) :-
	I =< Arity,
	(member(I,List)
	 ->	Compl1 = Compl
	 ;	Compl = [I|Compl1]
	),
	I1 is I+1,
	compl(I1,Arity,List,Compl1).

gen_test_call(ip(IndPos,_,OrdPos),Pred,HeadArgs,NonVarTest,Call) :-
	gen_sel_nonvars(1,IndPos,HeadArgs,NonVarTest),
	concat_atom([Pred,'$'],PredD),
	concat_atom([PredD|OrdPos],'_',OrdPredName),
	select_vars(OrdPos,HeadArgs,OrdVars),
	Call =.. [OrdPredName|OrdVars].

select_vars([],_,[]).
select_vars([Pos|Poss],HeadVars,[Var|AbsVars]) :-
	ith(Pos,HeadVars,Var),
	select_vars(Poss,HeadVars,AbsVars).

gen_sel_nonvars(_,[],_,true) :- !.
gen_sel_nonvars(I,Poss,[Arg|RemArgs],NonVars) :-
	Poss = [P|Ps],
	I1 is I + 1,
	(I =:= P
	 ->	(Ps == []
		 ->	NonVars = nonvar(Arg)
		 ;	NonVars = (nonvar(Arg),VarTest),
			gen_sel_nonvars(I1,Ps,RemArgs,VarTest)
		)
	 ;	gen_sel_nonvars(I1,Poss,RemArgs,NonVars)
	).


:- mode collect_orders_from_indexes(?,+,+,?,?).
collect_orders_from_indexes(Indexes,Pred,Arity,OrdSets,IndexOrds) :-
	find_orders(Indexes,Pred,Arity,OrdSets),
	collect_arg_perms(OrdSets,Arity,IndexOrds).

tail_var(L,Tail) :- var(L), !, Tail = L.
tail_var([_|L],Tail) :- tail_var(L,Tail).

:- import parsort/4 from machine.

collect_inds_and_perms([],[],[]).
collect_inds_and_perms([OrderSet|OrdSets],[IndexOrder|IndexOrds],IndPerms) :-
	collect_inds_and_perms(OrdSets,IndexOrds,IndPerms0),
	findall(ip(IndSet,I,IndexOrder),member(set(_,I,IndSet),OrderSet),IndPerms1),
	append(IndPerms1,IndPerms0,IndPerms).

%% collect_arg_perms(+OrderSetList,+Arity,-ArgPermList) takes a list of
%% ordered index sets, and the arity of the predicate and returns the
%% corresponding list of argument permutations needed to provide all
%% the indexes.
collect_arg_perms([],_Arity,[]).
collect_arg_perms([OrderSet|OrdSets],Arity,[ArgPerm|ArgPerms]) :-
	OrderSet = [set(_,_,List1)|OrderSetTail],
	collect_arg_pos(OrderSetTail,List1,List1,ArgPoses),
	upto_n(1,Arity,AllArgs),
	sort(ArgPoses,SArgPoses),
	diff_list(SArgPoses,AllArgs,NotIndPosns),
	append(ArgPoses,NotIndPosns,ArgPerm),
	collect_arg_perms(OrdSets,Arity,ArgPerms).

collect_arg_pos([],_PList,AP,AP).
collect_arg_pos([set(_,_,List)|Sets],PList,ArgPoses0,ArgPoses) :-
	diff_list(PList,List,NewArgPoses),
	append(ArgPoses0,NewArgPoses,ArgPoses1),
	collect_arg_pos(Sets,List,ArgPoses1,ArgPoses).

%% find_orders(+Indexes,+Pred,Arity,-OrdSets) takes a list of indexes
%% and returns a list of lists, one for each required argument order.
%% Such a list is a list of terms set(Len,I,OrdArgPosList), where Len
%% is the length of OrdArgPosList, I is the position of this index in
%% the sequence of indexes, and OrdArgPosList is a list of indexed
%% argument positions, and each OrdArgPosList in such a list is a
%% (proper) subset of the following one.
find_orders(Indexes,Pred,Arity,OrdSets) :-
	%% strange, findall/3 doesn't correctly cause setof mod to be loaded(loops),
	%%     but findall/4 seems to????  So this is findall/4 (for call from table_index.)
	findall(set(Len,I,BoundArgs),
	      an_indexset(Indexes,Len,I,BoundArgs),
	      IndexSets0),
	check_for_bad_order(IndexSets0,Indexes,Pred,Arity),
	parsort(IndexSets0,[desc(1),asc(2),asc(3)],0,IndexSets),
	add_all_to_chains(IndexSets,[],OrdSets).

an_indexset(Indexes,Len,I,BoundArgs) :-
	ith(I,Indexes,Index),
	index_to_list(Index,BoundArgs0),
	sort(BoundArgs0,BoundArgs),
	length(BoundArgs,Len).	

check_for_bad_order(IndexSets,IndexSpec,Pred,Arity) :-
	(append(_,[set(_,_,IndList)|Rest],IndexSets),
	 member(set(_,_,IndList1),Rest),
	 subset(IndList,IndList1),
         plusify(IndList,PlusList),
         plusify(IndList1,PlusList1),
	 commafy(IndexSpec,CIndexSpec),
	 warning(('In the table ... as ... index declaration index([',CIndexSpec,']) for ',Pred/Arity,', the index ',PlusList1,
		 ' will never be used because of the earlier index ',PlusList)),
	 fail
	 ;
	 true
	).

commafy([],[]).
commafy([H],[H]):- !.
commafy([H|T],[H,', '|PlusT]):- 
    commafy(T,PlusT).

plusify([],[]).
plusify([H],H):- !.
plusify([H|T],H+PlusT):- 
    plusify(T,PlusT).

%% add_all_to_chains(+SetList,+ChainList,-NewChainList) iteratively
%% adds each "set" in SetList to the beginning of the appropriate chain
%% in ChainList, generating NewChainList.
add_all_to_chains([],Orders,Orders).
add_all_to_chains([ISet|ISets],Orders0,Orders) :-
	add_to_chains(Orders0,ISet,Orders1),
	add_all_to_chains(ISets,Orders1,Orders).

%% add_to_chains(+ChainList,+Set,-NewChainList) adds Set to the front
%% of the first chain in ChainList such that it is a subset of the
%% first element of that chain.  If there is no such chain, it is added
%% as a singleton chain at the end.  The updated chain list is returned
%% in NewChainList.
add_to_chains([],Set,[[Set]]).
add_to_chains([Chain|ChainList],Set,NewChainList) :-
	Chain = [SmallestSet|_],
	(proper_subset(Set,SmallestSet)
	 ->	NewChainList = [[Set|Chain]|ChainList]
	 ;	NewChainList = [Chain|NewChainList0],
		add_to_chains(ChainList,Set,NewChainList0)
	).

%% proper_subset(+S1,+S2) is true if "set" S1 (of form
%% set(Len,OrdList)) is a proper subset of "set" S2.
proper_subset(set(L1,_,S1),set(L2,_,S2)) :-
	L1 < L2,
	subset(S1,S2).

%% subset(+L1,+L2) succeeds if L1 is a subset of L2 (both must be
%% sorted)
subset([],_).
subset([I|Is],[J|Js]) :-
	(I =:= J
	 ->	subset(Is,Js)
	 ; I > J
	 ->	subset([I|Is],Js)
	).

%% index_to_list(+ISpec,-ArgList) generates a list of atoms from a
%% "sum" of atoms
index_to_list(0,[]) :- !.
index_to_list(I+J,[J|Is]) :-
	!,
	index_to_list(I,Is).
index_to_list(I,[I]).

%% upto_n(+I,+K,-List) generates a list of integers from I to K,
%% inclusive.
upto_n(I,K,List) :-
	(I =< K
	 ->	List = [I|List1],
		I1 is I+1,
		upto_n(I1,K,List1)
	 ;	List = []
	).

%% diff_list(+L1,+L2,-L3) L1 must be a sublist of L2, both ordered, and L3
%% is returned a difference: L2-L1.
diff_list([],Rest,Rest).
diff_list([I|Is],[J|Js],Rest) :-
	(I =\= J
	 ->	Rest = [J|Rest1],
		diff_list([I|Is],Js,Rest1)
	 ;	diff_list(Is,Js,Rest)
	).

parse_preddef_closed(Head,Body,PredList,SymTab) :-
	functor(Head, Pred, Arity),
	sym_insert(Pred, Arity, [defined], SymTab, PredSym),
	memberchk(pred(PredSym,Clauses,_), PredList),
	trans_head(Head, goal(_, ArgList), SymTab, 0, VidMed),
	trans_goals(Body, BodyForm, SymTab, VidMed, _VidOut),
	add_tail(clause(ArgList, BodyForm, _), Clauses),
	sym_propin(pred, PredSym).

add_tail(Clause,Clauses) :- var(Clauses), !, Clauses=[Clause|_].
add_tail(Clause,[_|Clauses]) :- add_tail(Clause,Clauses).

end_of_file.

approximation_transform(module(Symtab,DcList,ClList,QrList,Par),
		      module(Symtab,DcList,ClList,QrList,Par)):-
	approx_transform(ClList,Symtab).

approx_transform([],_Symtab).
approx_transform([pred(Sym,Pscs,Oth)|ClList0],Symtab):-
	sym_type(Sym,approximation(_Spec)),!,
%	approx_trans_pred(Pscs,Sym,Oth,_NewClList,Symtab),
	approx_transform(ClList0,Symtab).
approx_transform([pred(_Sym,_Pscs,_Oth)|ClList0],Symtab):-
	approx_transform(ClList0,Symtab).

approx_trans_pred([],_Sym,_Oth,_NewClList,_Symtab).
approx_trans_pred([Clause|R],Sym,Oth,NewClList,_Symtab):- 
	get_max_varocc(Clause,MaxV),
	approx_trans_pred(R,Sym,Oth,NewClList,_Symtab). 

% Original, before changes of Sept 2014
generate_agg_clauses(AggrSpec,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :-
	(AggrSpec = fold(OpSpec,Id)
	 ;
	 AggrSpec = lattice(OpSpec), Id = no_identity
	),
	!,
	functor(PredTerm,Pred,Arity),
	(spec_skel(OpSpec,OpSpec1,3)
	 ->	true
	 ;	syntax_error(['Improper join specification in tabling declaration for ',Pred,'/',Arity])
	),
	(Id = F/1
	 ->	GetID =.. [F,OVar],
		Init = (GetID, OpSpec1)
	 ; Id == no_identity
	 ->	Init = (Var1 = Var)
	 ;	Init = (OVar = Id, OpSpec1)
	),
	sym_propin(tabled,Sym),
	arg(1,OpSpec1,OVar), arg(2,OpSpec1,Var1), arg(3,OpSpec1,Var),
	get_actions_for_lattice_aggr_success(Sym,[Cs,Leaf,Var,OVar,Breg],Actions),
	Body = ('_$savecp'(Breg),
		breg_retskel(Breg,Arity,Skel,Cs),
		(nonvar(Var) -> instantiation_error(Pred/Arity,Arity,var) ; true),
		excess_vars(Skel,[Var|ExistVarList],[],Vars),
		copy_term(t(Vars,Skel,Var),t(Vars,OSkel,OVar)),
		Pred1Term,
		('_$$get_returns'(Cs,OSkel,Leaf)
		 ->	OpSpec1,
			Var \== OVar,
%			delete_return(Cs,Leaf)
			Actions
		 ;	Init
		)
	       ),
	parse_preddef_closed(PredTerm,Body,ClList2,SymTab),
	globalize('/',2,SymTab),
	globalize(var,0,SymTab),
	ensure_imported([im(breg_retskel/4,machine),
			 im(excess_vars/4,setof),
			 im(copy_term/2,basics),
			 im(instantiation_error/3,error_handler),
			 im('_$$get_returns'/3,tables),
			 im(delete_return/2,tables),
			 im('_$member'/2,basics),
			 im(early_complete_on_nth/2,tables),
			 im(early_complete_on_delta/4,tables)
			 ], SymTab).


Following is a template for doing lattice and fold in WFS (including
undefineds).

:- import breg_retskel/4 from machine.
:- import excess_vars/4 from setof.
:- import copy_term/2 from basics.
:- import instantiation_error/3 from error_handler.
:- import '_$$get_returns'/3 from tables.
:- import '_$$get_returns_and_tvs'/4 from tables.
:- import delete_return/2 from tables.
:- import member/2 from basics.
:- import xwam_state/2 from machine.

:- table p/2.
p(X,Var) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,2,Skel,Cs),
	excess_vars(Skel,[Var],[],Vars),
	copy_term(t(Vars,Skel,Var),t(Vars,OSkel,OVar)),
	p1(X,Var1),
	xwam_state(2,DelayReg),
	writeln(newans(Var1,DelayReg)),
	findall(t(OSkel,Leaf1,TV1),'_$$get_returns_and_tvs'(Cs,OSkel,Leaf1,TV1),RetList),
	writeln(findall(RetList)),
	(DelayReg =:= 0,  % all deletion before returning any ans, otw bug
	 member(t(OSkel,Leaf1,_),RetList),
	 Var \== OVar,
	 delete_return(Cs,Leaf1),
	 fail
	 ;
	 \+member(t(OSkel,Leaf1,t),RetList),
	 Var = Var1
	 ;
	 member(t(OSkel,Leaf1,TV1),RetList),
	 sum(OVar,Var1,Var),
	 Var \== OVar,
	 (DelayReg =:= 0
	  ->	 (TV1 == t
		  ->	 true
		  ;	 undefined
		 )
	  ;	 true
	 )
	), xwam_state(2,NDel),writeln(ans(Var,NDel)).


sum(X,Y,Z) :- Z is X+Y.

p1(1,4) :- undefined.
p1(1,5) :- undefined.
p1(1,1).
p1(1,2).
p1(1,4).


%% restore Original, before changes of Sept 2014
generate_agg_clauses(AggrSpec,Sym,PredTerm,Var1,ExistVarList,Pred1Term,Var,ClList2,SymTab) :-
	(AggrSpec = fold(OpSpec,Id)
	 ;
	 AggrSpec = lattice(OpSpec), Id = no_identity
	),
	!,
	functor(PredTerm,Pred,Arity),
	(spec_skel(OpSpec,OpSpec1,3)
	 ->	true
	 ;	syntax_error(['Improper join specification in tabling declaration for ',Pred,'/',Arity])
	),
	(Id = F/1
	 ->	GetID =.. [F,OVar],
		Init = (GetID, OpSpec1)
	 ; Id == no_identity
	 ->	Init = (Var1 = Var)
	 ;	Init = (OVar = Id, OpSpec1)
	),
	sym_propin(tabled,Sym),
	arg(1,OpSpec1,OVar), arg(2,OpSpec1,Var1), arg(3,OpSpec1,Var),
	get_actions_for_lattice_aggr_success(Sym,[Cs,Leaf,Var,OVar,Breg],Actions),
	Body = ('_$savecp'(Breg),
		breg_retskel(Breg,Arity,Skel,Cs),
		(nonvar(Var) -> instantiation_error(Pred/Arity,Arity,var) ; true),
		excess_vars(Skel,[Var|ExistVarList],[],Vars),
		copy_term(t(Vars,Skel,Var),t(Vars,OSkel,OVar)),
		Pred1Term,
		('_$$get_returns'(Cs,OSkel,Leaf)
		 ->	OpSpec1,
			Var \== OVar,
%			delete_return(Cs,Leaf)
			Actions
		 ;	Init
		)
	       ),
	globalize('/',2,SymTab),
	globalize(var,0,SymTab),
	ensure_imported_list([im(breg_retskel/4,machine),
			      im(excess_vars/4,setof),
			      im(copy_term/2,basics),
			      im(instantiation_error/3,error_handler),
			      im('_$$get_returns'/3,tables),
			      im(delete_return/2,tables),
			      im('_$member'/2,basics),
			      im(early_complete_on_nth/2,tables),
			      im(early_complete_on_delta/4,tables)
			      ], SymTab
			    ),
	parse_preddef_closed(PredTerm,Body,ClList2,SymTab).
