/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "msg.h"

msg * msg_serverhello::deep_copy()
{
  vector<int> pi (pindices);
  return new msg_serverhello(pi);
}

msg * msg_clienthello::deep_copy()
{
  vector<string> n (names);
  return new msg_clienthello(n);
}

msg * msg_player_disconnected::deep_copy()
{
  return new msg_player_disconnected(pnum);
}

msg * msg_start_castleselect::deep_copy()
{
  return new msg_start_castleselect();
}

msg * msg_disconnect::deep_copy()
{
  return new msg_disconnect();
}

msg * msg_start_shoot::deep_copy()
{
  return new msg_start_shoot();
}

msg * msg_game_over::deep_copy()
{
  return new msg_game_over();
}

msg * msg_end::deep_copy()
{
  return new msg_end();
}

msg * msg_start_placecan::deep_copy()
{
  return new msg_start_placecan();
}

msg * msg_start_endgame::deep_copy()
{
  return new msg_start_endgame();
}

msg * msg_start_endround::deep_copy()
{
  return new msg_start_endround();
}

msg * msg_start_repair::deep_copy()
{
  return new msg_start_repair();
}

msg * msg_start_game::deep_copy()
{
  return new msg_start_game(wins);
}

msg * msg_playerinfo::deep_copy()
{
  return new msg_playerinfo(index, name);
}

msg * msg_talkmessage::deep_copy()
{
  return new msg_talkmessage(owner, txt);
}

msg * msg_castle_location::deep_copy()
{
  return new msg_castle_location(pos, owner);
}

msg * msg_move_cannonplacer::deep_copy()
{
  return new msg_move_cannonplacer(pos, owner);
}

msg * msg_move_characterchooser::deep_copy()
{
  return new msg_move_characterchooser(pos, owner);
}

msg * msg_move_blockplacer::deep_copy()
{
  return new msg_move_blockplacer(pos, owner);
}

msg * msg_place_cannon::deep_copy()
{
  return new msg_place_cannon(owner, pos);
}

msg * msg_place_big_cannon::deep_copy()
{
  return new msg_place_big_cannon(owner, pos);
}

msg * msg_change_cannon::deep_copy()
{
  return new msg_change_cannon(owner);
}

msg * msg_choose_character::deep_copy()
{
  return new msg_choose_character(owner, pos, character);
}

msg * msg_chooser_lock::deep_copy()
{
  return new msg_chooser_lock(owner);
}

msg * msg_move_chooser::deep_copy()
{
  return new msg_move_chooser(owner, pos);
}

msg * msg_move_chooser_req::deep_copy()
{
  return new msg_move_chooser_req(owner, down);
}

msg * msg_rotate_block::deep_copy()
{
  return new msg_rotate_block(owner);
}

msg * msg_place_block::deep_copy()
{
  return new msg_place_block(owner, next_type, orientation);
}

msg * msg_shoot::deep_copy()
{
  return new msg_shoot(from, to, shooter, ammosize);
}

msg * msg_move_cursor::deep_copy()
{
  return new msg_move_cursor(pos, owner, lock);
}


msg* msg::recv (TCPsocket sock) throw (string)
{
  assert (sizeof(Uint32) == 4);

  msg *last_msg;

  Uint32 type = read_uint(sock);
  
  switch (type)
    {
    case msg::SERVERHELLO:
      last_msg = new msg_serverhello();
      break;
    case msg::CLIENTHELLO:
      last_msg = new msg_clienthello();
      break;
    case msg::PL_DISCONNECT:
      last_msg = new msg_player_disconnected();
      break;
    case msg::START_CSELECT:
      last_msg = new msg_start_castleselect();
      break;
    case msg::START_GAME:
      last_msg = new msg_start_game();
      break;
    case msg::PLAYERINFO:
      last_msg = new msg_playerinfo();
      break;
    case msg::TALKMESSAGE:
      last_msg = new msg_talkmessage();
      break;
    case msg::CASTLE_LOCATION:
      last_msg = new msg_castle_location();
      break;
    case msg::MOVE_CANNONPLACER:
      last_msg = new msg_move_cannonplacer();
      break;
    case msg::MOVE_CHARACTERCHOOSER:
      last_msg = new msg_move_characterchooser();
      break;
    case msg::PLACE_CANNON:
      last_msg = new msg_place_cannon();
      break;
    case msg::PLACE_BIG_CANNON:
      last_msg = new msg_place_big_cannon();
      break;
    case msg::CHANGE_CANNON:
      last_msg = new msg_change_cannon();
      break;
    case msg::CHOOSE_CHARACTER:
      last_msg = new msg_choose_character();
      break;
    case msg::START_SHOOT:
      last_msg = new msg_start_shoot();
      break;
    case msg::START_REPAIR:
      last_msg = new msg_start_repair();
      break;
    case msg::SHOOT:
      last_msg = new msg_shoot();
      break;
    case msg::MOVE_CURSOR:
      last_msg = new msg_move_cursor();
      break;
    case msg::MOVE_BLOCKPLACER:
      last_msg = new msg_move_blockplacer();
      break;
    case msg::PLACE_BLOCK:
      last_msg = new msg_place_block();
      break;
    case msg::ROTATE_BL:
      last_msg = new msg_rotate_block();
      break;
    case msg::START_PLACECAN:
      last_msg = new msg_start_placecan();
      break;
    case msg::START_ENDGAME:
      last_msg = new msg_start_endgame();
      break;
    case msg::GAME_OVER:
      last_msg = new msg_game_over();
      break;
    case msg::MOVE_CHOOSER:
      last_msg = new msg_move_chooser();
      break;
    case msg::CHOOSER_LOCK:
      last_msg = new msg_chooser_lock();
      break;
    case msg::MOVE_CHOOSER_REQ:
      last_msg = new msg_move_chooser_req();
      break;
    case msg::END:
      last_msg = new msg_end();
      break;
    case msg::START_ENDROUND:
      last_msg = new msg_start_endround();
      break;
    case msg::DISCONNECT:
      last_msg = new msg_disconnect();
    default:
      throw string("Error, received unknown message!");
      break;
    }
  last_msg->read_msgdata(sock);
  return last_msg;
}


msg::msg (const msg &m)
{
  msg_id = m.msg_id;
}

void msg::send (TCPsocket sock) const throw (string)
{
  assert (sizeof(Uint32) == 4);
  assert (msg_id != MSG_IDNOTSET);

  write_uint (sock, (Uint32) msg_id);
  write_msgdata(sock);
}

msg_clienthello::msg_clienthello (const vector<string> &_names)
{
  msg_id = CLIENTHELLO;
  names = _names;
}

msg_serverhello::msg_serverhello (const vector<int> &pind)
{
  assert(pind.size() > 0);

  msg_id = SERVERHELLO;
  pindices = pind;
}

msg_serverhello::msg_serverhello()
{
  msg_id = SERVERHELLO;
}

msg_clienthello::msg_clienthello()
{
  msg_id = CLIENTHELLO;
}

const vector<int> & msg_serverhello::get_player_indices()
{
  return pindices;
}

void msg::write_uint(TCPsocket sock, Uint32 d) throw (string)
{
  assert (sizeof(Uint32) == 4);
  char buf[4];
  SDLNet_Write32(d, buf);
  if (SDLNet_TCP_Send (sock, buf, 4) != 4)
    throw string ("Error sending uint: ") + string(SDLNet_GetError());
}

Uint32 msg::read_uint(TCPsocket sock) throw (string)
{
  char buf[sizeof(Uint32)];

  if (SDLNet_TCP_Recv (sock, buf, sizeof(Uint32)) != sizeof(Uint32))
    throw string ("Error receiving uint: ") + string( SDLNet_GetError());

  Uint32 ret = SDLNet_Read32 (buf);

  return ret;
}

void msg::write_coord(TCPsocket sock, const coord &c) throw (string)
{
  write_uint(sock, c.getPX());
  write_uint(sock, c.getPY());
}

coord msg::read_coord(TCPsocket sock) throw (string)
{
  Uint32 x = read_uint(sock);
  Uint32 y = read_uint(sock);
  return coord::pc (x,y);
}

void msg::write_string(TCPsocket sock, const string &s) throw (string)
{
  Uint32 slen = (Uint32) s.length();
  write_uint(sock, slen);
  if (SDLNet_TCP_Send (sock, (void *) s.c_str(), slen) != (int) slen)
    throw string ("Error sending string data: ") + string(SDLNet_GetError());    
}

string msg::read_string(TCPsocket sock) throw (string)
{
  Uint32 slen = read_uint(sock);

  char *tbuf = new char[slen+1];
  
  if (SDLNet_TCP_Recv (sock, tbuf, slen) != (int) slen)
    {
      delete tbuf;
      throw string ("Error receiving string data: ") + string(SDLNet_GetError());
    }
  tbuf[slen] = 0;
  string rets(tbuf);
  delete tbuf;
  return rets;
}

void msg_serverhello::write_msgdata(TCPsocket sock) const throw (string)
{
  unsigned int nump = pindices.size();
  write_uint(sock, (int) nump);
  for (unsigned int i=0;i<nump;i++)
    write_uint(sock, pindices[i]);
}
void msg_serverhello::read_msgdata(TCPsocket sock) throw (string)
{
  int nump = read_uint(sock);
  pindices = vector<int>(nump);
  for (int i=0;i<nump;i++)
    pindices[i] = read_uint(sock);
}

void msg_clienthello::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, names.size());
  for (unsigned int i=0;i<names.size();i++)
    write_string(sock, names[i]);
}
void msg_clienthello::read_msgdata(TCPsocket sock) throw (string)
{
  int num_pl = read_uint(sock);
  names = vector<string>(num_pl);
  for (int i=0;i<num_pl;i++)
    names[i] = read_string(sock);
}

msg_player_disconnected::msg_player_disconnected()
{
  msg_id = PL_DISCONNECT;
}
msg_player_disconnected::msg_player_disconnected (int _pnum)
{
  msg_id = PL_DISCONNECT;
  pnum = _pnum;
}

msg_start_castleselect::msg_start_castleselect()
{
  msg_id = START_CSELECT;
}

msg_start_game::msg_start_game()
{
  msg_id = START_GAME;
}

msg_start_game::msg_start_game(int _wins)
{
  msg_id = START_GAME;
  wins = _wins;
}

void msg_start_game::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, wins);
}
void msg_start_game::read_msgdata(TCPsocket sock) throw (string)
{
  wins = read_uint(sock);
}

msg_playerinfo::msg_playerinfo()
{
  msg_id = PLAYERINFO;
}

msg_playerinfo::msg_playerinfo(int _index, const string &_name)
{
  msg_id = PLAYERINFO;
  name = _name;
  index = _index;
}

void msg_playerinfo::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, index);
  write_string(sock, name);
}

void msg_playerinfo::read_msgdata(TCPsocket sock) throw (string)
{
  index = read_uint(sock);
  name = read_string(sock);
}

msg_talkmessage::msg_talkmessage()
{
  msg_id = TALKMESSAGE;
}

msg_talkmessage::msg_talkmessage(int _owner, const string &_txt)
{
  msg_id = TALKMESSAGE;
  txt = _txt;
  owner = _owner;
}

void msg_talkmessage::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_string(sock, txt);
}

void msg_talkmessage::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  txt = read_string(sock);
}

msg_castle_location::msg_castle_location()
{
  msg_id = CASTLE_LOCATION;
}

msg_castle_location::msg_castle_location(const coord &_pos, int _owner)
{
  msg_id = CASTLE_LOCATION;
  pos = _pos;
  owner = _owner;
}

void msg_castle_location::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord(sock, pos);
  write_uint(sock, owner);
}

void msg_castle_location::read_msgdata(TCPsocket sock) throw (string)
{
  pos = read_coord(sock);
  owner = read_uint(sock);
}


msg_move_cannonplacer::msg_move_cannonplacer()
{
  msg_id = MOVE_CANNONPLACER;
}

msg_move_cannonplacer::msg_move_cannonplacer(const coord &_pos, int _owner)
{
  msg_id = MOVE_CANNONPLACER;
  pos = _pos;
  owner = _owner;
}

void msg_move_cannonplacer::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord(sock, pos);
  write_uint(sock, owner);
}

void msg_move_cannonplacer::read_msgdata(TCPsocket sock) throw (string)
{
  pos = read_coord(sock);
  owner = read_uint(sock);
}

msg_move_characterchooser::msg_move_characterchooser()
{
  msg_id = MOVE_CHARACTERCHOOSER;
}

msg_move_characterchooser::msg_move_characterchooser(const coord &_pos, int _owner)
{
  msg_id = MOVE_CHARACTERCHOOSER;
  pos = _pos;
  owner = _owner;
}
void msg_move_characterchooser::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord(sock, pos);
  write_uint(sock, owner);
}

void msg_move_characterchooser::read_msgdata(TCPsocket sock) throw (string)
{
  pos = read_coord(sock);
  owner = read_uint(sock);
}



msg_move_blockplacer::msg_move_blockplacer()
{
  msg_id = MOVE_BLOCKPLACER;
}

msg_move_blockplacer::msg_move_blockplacer(const coord &_pos, int _owner)
{
  msg_id = MOVE_BLOCKPLACER;
  pos = _pos;
  owner = _owner;
}

void msg_move_blockplacer::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord(sock, pos);
  write_uint(sock, owner);
}

void msg_move_blockplacer::read_msgdata(TCPsocket sock) throw (string)
{
  pos = read_coord(sock);
  owner = read_uint(sock);
}

msg_place_cannon::msg_place_cannon()
{
  msg_id = PLACE_CANNON;
}

msg_place_cannon::msg_place_cannon(int _owner, const coord &_pos)
{
  msg_id = PLACE_CANNON;
  owner = _owner;
  pos = _pos;
}

msg_change_cannon::msg_change_cannon()
{
  msg_id = CHANGE_CANNON;
}

msg_change_cannon::msg_change_cannon(int _owner)
{
  msg_id = CHANGE_CANNON;
  owner = _owner;
}

msg_place_big_cannon::msg_place_big_cannon()
{
  msg_id = PLACE_BIG_CANNON;
}

msg_place_big_cannon::msg_place_big_cannon(int _owner, const coord &_pos)
{
  msg_id = PLACE_BIG_CANNON;
  owner = _owner;
  pos = _pos;
}

void msg_place_cannon::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_coord (sock, pos);
}

void msg_change_cannon::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
}

void msg_change_cannon::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
}

void msg_place_cannon::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  pos = read_coord(sock);
}

void msg_place_big_cannon::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_coord (sock, pos);
}

void msg_place_big_cannon::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  pos = read_coord(sock);
}

msg_choose_character::msg_choose_character()
{
  msg_id = CHOOSE_CHARACTER;
}

msg_choose_character::msg_choose_character(int _owner, const coord &_pos, int ch)
{
  msg_id = CHOOSE_CHARACTER;
  owner = _owner;
  pos = _pos;
  character = ch;
}

int msg_choose_character:: get_character()
{
  return character;
}

void msg_choose_character::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_uint(sock, character);
  write_coord (sock, pos);
}

void msg_choose_character::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  character = read_uint(sock);
  pos = read_coord(sock);
}

msg_chooser_lock::msg_chooser_lock()
{
  msg_id = CHOOSER_LOCK;
}

msg_chooser_lock::msg_chooser_lock(int _owner)
{
  msg_id = CHOOSER_LOCK;
  owner = _owner;
}

void msg_chooser_lock::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
}

void msg_chooser_lock::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
}

msg_move_chooser::msg_move_chooser()
{
  msg_id = MOVE_CHOOSER;
}

msg_move_chooser::msg_move_chooser(int _owner, const coord &new_pos)
{
  msg_id = MOVE_CHOOSER;
  owner = _owner;
  pos = new_pos;
}

void msg_move_chooser::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_coord(sock, pos);
}

void msg_move_chooser::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  pos = read_coord(sock);
}

msg_move_chooser_req::msg_move_chooser_req()
{
  msg_id = MOVE_CHOOSER_REQ;
}

msg_move_chooser_req::msg_move_chooser_req(int _owner, bool _down)
{
  msg_id = MOVE_CHOOSER_REQ;
  owner = _owner;
  down = _down;
}

void msg_move_chooser_req::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_uint(sock, down);
}

void msg_move_chooser_req::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  down = read_uint(sock);
}

msg_rotate_block::msg_rotate_block()
{
  msg_id = ROTATE_BL;
}

msg_rotate_block::msg_rotate_block(int _owner)
{
  msg_id = ROTATE_BL;
  owner = _owner;
}

void msg_rotate_block::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
}

void msg_rotate_block::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
}

msg_place_block::msg_place_block()
{
  msg_id = PLACE_BLOCK;
}

msg_place_block::msg_place_block(int _owner, int _next_type, int _orientation)
{
  msg_id = PLACE_BLOCK;
  owner = _owner;
  next_type = _next_type;
  orientation = _orientation;
}

void msg_place_block::write_msgdata(TCPsocket sock) const throw (string)
{
  write_uint(sock, owner);
  write_uint(sock, next_type);
  write_uint(sock, orientation);
}

void msg_place_block::read_msgdata(TCPsocket sock) throw (string)
{
  owner = read_uint(sock);
  next_type = read_uint(sock);
  orientation = read_uint(sock);
}

msg_shoot::msg_shoot()
{
  msg_id = SHOOT;
}

msg_shoot::msg_shoot(const coord &_from, const coord &_to, int _shooter_id, int _ammosize)
{
  msg_id = SHOOT;
  from = _from;
  to = _to;
  shooter = _shooter_id;
  ammosize=_ammosize;
}

void msg_shoot::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord (sock, from);
  write_coord (sock, to);
  write_uint (sock, shooter);
  write_uint (sock, ammosize);
}

void msg_shoot::read_msgdata(TCPsocket sock) throw (string)
{
  from = read_coord(sock);
  to = read_coord(sock);
  shooter = read_uint(sock);
  ammosize = read_uint(sock);
}

msg_move_cursor::msg_move_cursor()
{
  msg_id = MOVE_CURSOR;
}

msg_move_cursor::msg_move_cursor(const coord &_pos, int _owner, bool _lock)
{
  msg_id = MOVE_CURSOR;
  pos = _pos;
  owner = _owner;
  lock = _lock;
}

msg_disconnect::msg_disconnect()
{
  msg_id = DISCONNECT;
}

void msg_disconnect::write_msgdata(TCPsocket sock) const throw (string)
{
}

void msg_disconnect::read_msgdata(TCPsocket sock) throw (string)
{
}

void msg_move_cursor::write_msgdata(TCPsocket sock) const throw (string)
{
  write_coord(sock, pos);
  write_uint(sock, owner);
  write_uint(sock, lock);
  
}

void msg_move_cursor::read_msgdata(TCPsocket sock) throw (string)
{
  pos = read_coord(sock);
  owner = read_uint(sock);
  lock = read_uint(sock);
}
