/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "particle_efects.h"

Uint32 multiplyRGB (Uint32 col, double am)
{
  int r = (col & 0xFF0000) >> 16;
  int g = (col & 0x00FF00) >> 8;
  int b = col & 0x0000FF;

  r  = (int)(((double) r) * am);
  g  = (int)(((double) g) * am);
  b  = (int)(((double) b) * am);

  if (r > 255) r = 255;
  if (g > 255) g = 255;
  if (b > 255) b = 255;

  Uint32 ret = (r << 16) | (g << 8) | b;
  return ret;
}


coord particle::get_position() const
{
  return coord::pc ((int) posx, (int) posy);
}

particle::particle(const coord & _pos)
{
  dirx = (rand() / (double) RAND_MAX) * 10.0 -5.0;
  diry = (rand() / (double) RAND_MAX) * 10.0 -5.0;
  posx = (double) _pos.getPX();
  posy = (double) _pos.getPY();
  color = 0xFFFFFF;
  dead = false;
}

particle::particle(const coord & _pos, double _dirx, double _diry)
{
  dirx = _dirx;
  diry = _diry;
  posx = (double) _pos.getPX();
  posy = (double) _pos.getPY();
  color = 0xFFFFFF;
  dead = false;
}

particle::particle(const coord & _pos, double _dirx, double _diry, Uint32 col)
{
  dirx = _dirx;
  diry = _diry;
  posx = (double) _pos.getPX();
  posy = (double) _pos.getPY();
  color = col;
  dead = false;
}

particle::particle()
{
  dirx = (rand() / (double) RAND_MAX) * 10.0 -5.0;
  diry = (rand() / (double) RAND_MAX) * 10.0 -5.0;
  posx = 0.0;
  posy = 0.0;
  color = 0xFFFFFF;
  dead = false;
}

void particle::set_direction (double _dirx, double _diry)
{
  dirx = _dirx;
  diry = _diry;
}

void particle::move_decel(Uint32 time, double rate)
{
  if (dead) return;
  double am = (double) time / 15.0;
  
  posx += dirx * am;
  posy += diry * am;

  double dam = am / rate;

  //printf ("%f %f\n", am, dam);
  //  printf ("new pos %f %f\n", posx, posy);

  if (dirx == 0.0 && diry == 0.0)
    {
      dead = true;
      return;
    }
  if (dirx > 0.0)
    {
      dirx-=dam;
      if (dirx < 0.0) dirx = 0.0;
    }
  else if (dirx < 0.0)
    {
      dirx+=dam;
      if (dirx > 0.0) dirx = 0.0;
    }

  if (diry > 0.0)
    {
      diry-=dam;
      if (diry < 0.0) diry = 0.0;
    }
  else if (diry < 0.0)
    {
      diry+=dam;
      if (diry > 0.0) diry = 0.0;
    }

}



void particle::move(Uint32 time)
{
  if (dead) return;
  double am = (double) time / 100.0;
  
  posx += dirx * am;
  posy += diry * am;
}

void particle::decelerate(Uint32 time)
{
  double am = (double) time / 100.0;

  double xmul = dirx / (dirx+diry);
  double ymul = diry / (dirx+diry);
  
  dirx -= am * xmul;
  diry -= am * ymul;

  if (dirx == 0.0 && diry == 0.0)
    {
      dead = true;
      return;
    }
  if (dirx > 0.0)
    {
      dirx-=am;
      if (dirx < 0.0) dirx = 0.0;
    }
  else if (dirx < 0.0)
    {
      dirx+=am;
      if (dirx > 0.0) dirx = 0.0;
    }

  if (diry > 0.0)
    {
      diry-=am;
      if (diry < 0.0) diry = 0.0;
    }
  else if (diry < 0.0)
    {
      diry+=am;
      if (diry > 0.0) diry = 0.0;
    }
}

void particle::fade(Uint32 time)
{
  if (dead) return;

  double sptot = fabs(dirx) + fabs(diry);
  double dam = time * 10.0 / sptot;
  
  if (color == 0)
    {
      dead = true;
      return;
    }

  color -= (Uint32) dam;

  if (color < 0)
    color = 0;

  if (color > 255)
    color = 255;
}

void particle::move_const (Uint32 time, const coord &ssize)
{
  if (dead) return;

  double am = (double) time / 30.0;

  posx += dirx * am;
  posy += diry * am;

  //  printf ("new posx %lf posy %lf\n", posx, posy);
  if (posx >= (double) ssize.getPX())
    {
      posx = (double) (ssize.getPX()-1);
      dead = true;
    }
  else if (posx < 0.0)
    {
      posx = 0.0;
      dead = true;
    }
  
  if (posy >= ssize.getPY()) 
    {
      posy = (double) (ssize.getPY()-1);
      dead = true;
    }
  else if (posy < 0.0)
    {
      posy = 0.0;
      dead = true;
    }
}

void particle::move_accel (Uint32 time, const coord &ssize)
{
  if (dead) return;

  double am = (double) time / 30.0;

  posx += dirx * am;
  posy += diry * am;
  
  double dam = am / 10.0;

  color = multiplyRGB(color, 1.0+dam*2.0);

  //  printf ("new posx %lf posy %lf\n", posx, posy);
  if (posx >= (double) ssize.getPX())
    {
      posx = (double) (ssize.getPX()-1);
      dead = true;
    }
  else if (posx < 0.0)
    {
      posx = 0.0;
      dead = true;
    }
  
  if (posy >= ssize.getPY()) 
    {
      posy = (double) (ssize.getPY()-1);
      dead = true;
    }
  else if (posy < 0.0)
    {
      posy = 0.0;
      dead = true;
    }
  dirx*=1.0+dam;
  diry*=1.0+dam;
}

bool particle::is_dead()
{
  return dead;
}

void particle::set_position(const coord & _pos)
{
  posx = (double) _pos.getPX();
  posy = (double) _pos.getPY();
}

particle_efect::particle_efect(const coord &_center, int _num_particles)
  : game_obj (_center, OWNER_NOONE)
{
  num_particles = _num_particles;
  ptc = new particle[num_particles];
  for ( int i = 0; i < num_particles; i++ )
    ptc[i].set_position( _center );
}

particle_efect::~particle_efect()
{
  delete[] ptc;
}

explosion_efect::explosion_efect(const coord & center, SDL_Surface *til)
  : particle_efect (center, til->w * til->h)
{
  done = false;
  int i, j;
  Uint32 *pix = (Uint32*) til->pixels;
  for (i=0;i<til->h;i++)
    for (j=0;j<til->w;j++)
      {
	ptc[j+i*til->w].set_color (pix[til->w*i+j]);
	ptc[j+i*til->w].set_position(get_position()+coord::pc(j,i));
      }
}

void explosion_efect::move(Uint32 time)
{
  done = true;
  for (int i=0;i<num_particles;i++)
    {
      ptc[i].move_decel(time);
      if (done)
	if (!ptc[i].is_dead())
	  done = false;
    }
}

void particle_efect::draw(SDL_Surface *surf)
{

  if (SDL_MUSTLOCK(surf))
    SDL_LockSurface (surf);

  Uint32 *pix = (Uint32 *) surf->pixels;
  Uint32 w = surf->w;
  Uint32 h = surf->h;

  for (int i=0;i<num_particles;i++)
    if (!ptc[i].is_dead())
      {
	coord c = ptc[i].get_position();
	if (c.getPX() < 0 || c.getPY() < 0 || c.getPX() >= (int) w || c.getPY() >= (int) h)
	  continue;
	Uint32 col = ptc[i].get_color();
	pix[c.getPY()*w+c.getPX()] = col;
      }
  
  if (SDL_MUSTLOCK(surf))
    SDL_UnlockSurface(surf);
}

bool explosion_efect::is_finished() const
{
  return done;
}

starfield_efect::starfield_efect(const coord &_center, const coord &_scrsize, int _no_stars)
  : particle_efect (_center, 100)
{
  no_stars = _no_stars;
  scrsize = _scrsize;

  // spawn stars
  for (int i=0;i<no_stars;i++)
    {
      coord c = coord::pc(rand() % scrsize.getPX(), rand() % scrsize.getPY());
      ptc[i] = spawn_star (c);
    }
}

void starfield_efect::move(Uint32 time)
{
  for (int i=0;i<num_particles;i++)
    {
      ptc[i].move_accel(time, scrsize);
      if (ptc[i].is_dead())
	{
	  coord c = coord::pc(rand() % scrsize.getPX(), rand() % scrsize.getPY());
	  ptc[i] = spawn_star (c);
	  //	  ptc[i] = spawn_center_star ();
	}
    }
}

particle starfield_efect::spawn_star(const coord & pos)
{
  double dx = (double)(pos.getPX() - position.getPX()) / 25.0;
  double dy = (double)(pos.getPY() - position.getPY()) / 25.0;
  
  if (dx == 0.0 && dy == 0.0)
    {
      dx = 0.1;
      dy = 0.1;
    }

  return particle(pos, dx, dy, 0x111111);  
}

particle starfield_efect::spawn_center_star()
{
  double angle = (rand() / (double) RAND_MAX) * PI * 2.0;
  double dx = cos(angle) / 5.0;
  double dy = sin(angle) / 5.0;

  return particle (position, dx, dy, 0x111111);
}

bool starfield_efect::is_finished() const
{
  return false;
}

real_explosion_efect::real_explosion_efect(const coord & center, int num_part, double _speed)
  : particle_efect (center, num_part)
{
  done = false;
  speed = _speed;
  set_colormap_yellowredgray();
  for (int i=0;i<num_part;i++)
    {
      double angle = (rand() / (double) RAND_MAX) * PI * 2.0;
      double dx = cos(angle) / 5.0;
      double dy = sin(angle) / 5.0;

      double am = rand() / (double) RAND_MAX;

      dx *= speed * am;
      dy *= speed * am;

      //      printf ("%lf %lf \n", dx, dy);

      int rx = rand() % 5;
      int ry = rand() % 5;

      ptc[i].set_direction(dx, dy);
      ptc[i].set_position(center+coord::pc(rx,ry));
      ptc[i].set_color(255); // colormap index
    }
}

void real_explosion_efect::set_colormap_yellowred()
{
  cmap = colormap_yr;
}

void real_explosion_efect::set_colormap_yellowredgray()
{
  cmap = colormap_yrg;
}

void real_explosion_efect::create_colormaps()
{
  if (cmap_inited) return;

  SDL_Surface *s = SDL_GetVideoSurface();
  SDL_PixelFormat *f = s->format;

  Uint32 rs = f->Rshift;
  Uint32 gs = f->Gshift;
  Uint32 bs = f->Bshift;
  
  for (int i=0;i<128;i++)
    {
      int r = i*2;
      int g = 0;
      int b = 0;

      colormap_yr[i] = (r << rs) | (g << gs) | (b << bs);
      //      printf ("map %d %d %d %d\n", i, r, g, b);
    }
  for (int i=128;i<256;i++)
    {
      int r = 255;
      int g = (i-128)*2;
      int b = 0;

      colormap_yr[i] = (r << rs) | (g << gs) | (b << bs);
      //      printf ("map %d %d %d %d\n", i, r, g, b);
    }

  // yrg

  for (int i=0;i<64;i++)
    {
      int r = i*2;
      int g = i;
      int b = 0;

      colormap_yrg[i] = (r << rs) | (g << gs) | (b << bs);
      //      printf ("map %d %d %d %d\n", i, r, g, b);
    }

  for (int i=64;i<128;i++)
    {
      int r = i*2;
      int g = i;
      int b = 0;

      colormap_yrg[i] = (r << rs) | (g << gs) | (b << bs);
      //      printf ("map %d %d %d %d\n", i, r, g, b);
    }
  for (int i=128;i<256;i++)
    {
      int r = 255;
      int g = i;
      int b = 0;

      colormap_yrg[i] = (r << rs) | (g << gs) | (b << bs);
      //      printf ("map %d %d %d %d\n", i, r, g, b);
    }

  
  cmap_inited = true;
}

void real_explosion_efect::move(Uint32 time)
{
  done = true;
  for (int i=0;i<num_particles;i++)
    {
      ptc[i].move(time);
      ptc[i].fade(time);
      ptc[i].decelerate(time);
      //      int col = (int)ptc[i].get_color();
      //int minam = (int)(time/10*(speed*0.1));
      //      printf ("minam %d\n", minam);
      //col -= minam;
      //if (col < 0) col = 0;
      //ptc[i].set_color(col);
      //printf ("color %d\n", ptc[i].get_color());
      if (done)
	if (!ptc[i].is_dead())
	  done = false;
    }
}

bool real_explosion_efect::is_finished() const
{
  return done;
}

void real_explosion_efect::draw(SDL_Surface *surf)
{

  if (SDL_MUSTLOCK(surf))
    SDL_LockSurface (surf);
  
  Uint32 *pix = (Uint32 *) surf->pixels;
  Uint32 w = surf->w;
  Uint32 h = surf->h;

  // Pasi - MSVC: error: C2057: expected constant expression
  // Uint32 tbuf[w*h];
  Uint32 *tbuf = new Uint32[w * h];
  fill (tbuf, tbuf+w*h, 0);

  for (int i=0;i<num_particles;i++)
    if (true)//!ptc[i].is_dead())
      {
	coord c = ptc[i].get_position();
	if (c.getPX() < 0 || c.getPY() < 0 || c.getPX() >= (int) w || c.getPY() >= (int) h)
	  continue;
	Uint32 col = ptc[i].get_color();
	tbuf[c.getPY()*w+c.getPX()] = col;
	//printf ("drawn to %d %d with %d ind %d\n", c.x, c.y, cmap[col], col);
	//printf ("r %d g %d b %d\n", (cmap[col] & 0xFF0000) >> 16, 
	//	(cmap[col] & 0xFF00) >> 8, 
	//	(cmap[col] & 0xFF) >> 0);
      }

  // trilinear interpolation, draw
  for (unsigned int y=1;y<h-1;y++)
    for (unsigned int x=1;x<w-1;x++)
      {
	Uint32 tc = 
	  tbuf[(y-1)*w+x]+
	  tbuf[(y+1)*w+x]+
	  tbuf[y*w+x+1]+
	  tbuf[y*w+x-1]+
	  tbuf[(y-1)*w+x-1]+
	  tbuf[(y+1)*w+x+1]+
	  tbuf[(y-1)*w+x+1]+
	  tbuf[(y+1)*w+x-1];
	  //	  tbuf[y*w+x];
	tc /= 8;

	//	printf ("draw %d\n", tc);
	pix[y*w+x] = cmap[tc];
	  
      }
      
  delete[] tbuf;
  
  if (SDL_MUSTLOCK(surf))
    SDL_UnlockSurface(surf);
}

Uint32 real_explosion_efect::colormap_yr[256];
Uint32 real_explosion_efect::colormap_yrg[256];
bool real_explosion_efect::cmap_inited = false;
