/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "terrain.h"

tile::tile(int _type, int _owner)
  : type(_type), owner(_owner), checked(false), conquered(false), damage(0)
{
}

tile::tile()
  : type(TYPE_EMPTY), owner(OWNER_NOONE), checked(false), conquered(false), damage(0)
{
}

void tile::hit(const ammo &a)
{
  ammo am = a;
  
  if(am.get_size()==BIG)
    damage+=BIG_AMMO_DAMAGE;
  else
    damage+=AMMO_DAMAGE;

}

terrain::terrain (int _num_castles)
{
  init_terrain (_num_castles); 
}

void terrain::init_terrain (int _num_castles)
{
  num_castles = _num_castles;

  /* init random number gen with current time */
  time_t tt = time(NULL);
  srand (tt);
  

  if (num_castles == 0) return;

  /* Castle placing algorithm.
   */

/* THIS ALGORITHM NEEDS REDOING. WE NEED TO SPREAD THE CASTLES IN GAUSSIAN DISTRIBUTION -FASHION
	 WITH MULTIPLE KERNELS (random non-overlapping kernels, 1 castle / kernel) */
  /* place the castles randomly .. yes, this is a very good algorithm :) */
  static const int max_set_trys = 10000;
  for(int i=0;i<num_castles;i++)
    {
      bool ok = false;
      for (int j=0;j<max_set_trys;j++)
	{
	  int x = (rand() % (XTILES -2 - CASTLE_WALL_RADIUS*2)) + CASTLE_WALL_RADIUS+1;
	  int y = (rand() % (YTILES -2- CASTLE_WALL_RADIUS*2)) + CASTLE_WALL_RADIUS+1;
	  
	  if (!is_square_clear (coord::tc(x, y), CASTLE_WALL_RADIUS+3))
	    continue;
	  insert_castle (coord::tc(x, y), OWNER_NOONE);
	  ok = true;
	  break;
	}
      if (!ok)
	{
	  fprintf (stderr, "Cannot place all castles (no space?), quitting\n");
	  exit (1);
	}
    }
}


void terrain::insert_castle (const coord &c, int owner)
{
  int x = c.getTX();
  int y = c.getTY();
  /* pzq
  assert (x >= 0);
  assert (y >= 0);
  assert (x < XTILES);
  assert (y < YTILES);
  */

  for (int i=x-CASTLE_RADIUS;i<=x+CASTLE_RADIUS;i++)
	for (int j=y-CASTLE_RADIUS;j<=y+CASTLE_RADIUS;j++)
	  set_tile (coord::tc(i, j), tile::TYPE_CASTLE_A, owner);
  
  set_tile ( coord::tc(x, y), tile::TYPE_CASTLE_C, owner);
}

bool terrain::is_square_clear (const coord &c, int radius)
{
  int x = c.getTX();
  int y = c.getTY();
  for (int i=x-radius;i<=x+radius;i++)
	for (int j=y-radius;j<=y+radius;j++)
	  if (!is_tile_clear (coord::tc(i, j))) return false;
  return true;
}

bool terrain::is_tile_clear (const coord &c)
{
  int x = c.getTX();
  int y = c.getTY();
  /* pzq
  assert (x >= 0);
  assert (y >= 0);
  assert (x < XTILES);
  assert (y < YTILES);
  */
  tile& t = get_tile (coord::tc(x, y));
  return (t.type == tile::TYPE_EMPTY);
}

void terrain::set_tile (const coord &c, tile &t)
{
  /* pzq
  assert (c.x >= 0);
  assert (c.y >= 0);
  assert (c.x < XTILES);
  assert (c.y < YTILES);
  */
  int tx = c.getTX();
  int ty = c.getTY();
  tiles[ty][tx] = t;
  changed_tiles.push_back (coord::tc(tx, ty));
}

tile& terrain::get_tile (const coord &c)
{
  /* pzq
  assert (c.x >= 0);
  assert (c.y >= 0);
  assert (c.x< XTILES);
  assert (c.y < YTILES);
  */

  return tiles[c.getTY()][c.getTX()];
}

vector<coord> terrain::get_castle_centers() const
{
  vector<coord> rv;
  for (int y=0;y<YTILES;y++)
	for (int x=0;x<XTILES;x++)
	  {
		const tile &t = tiles[y][x];
		if (t.type != tile::TYPE_CASTLE_C)
		  continue;
		rv.push_back (coord::tc(x,y));
	  }
  return rv;
}

void terrain::check_conquered ()
{
  static const int W = 0;
  static const int NW = 1;
  static const int N = 2;
  static const int NE = 3;
  static const int E = 4;
  static const int SE = 5;
  static const int S = 6;
  static const int SW = 7;

  int x, y;

  int owner=-1;
  bool ok, ready;

  queue<coord> tiles_to_check;

  // initial pass
  for(y=0;y<YTILES;y++)
	for (x=0;x<XTILES;x++)
	  {
		tile &t = get_tile(coord::tc(x,y));
		
		//		printf ("tile type %d\n", t.type);
		
		t.checked = true;
		
		// wall-tile. always conquered
		if (t.type == tile::TYPE_WALL && t.owner != OWNER_NOONE)
		  {
			t.conquered = true;
			continue;
		  }
		
		// border-tile and not a wall => not conquered.
		if (y == 0 || x == 0 || y == YTILES-1 || x == XTILES-1)
		  {
			t.owner = OWNER_NOONE;
			t.conquered = false;
			continue;
		  }
		
		// Ok, this tile is not conquered, not checked yet, and is not a wall-tile.
		ok = true;
		// lets check all its adjacent tiles from west to north-east
		for (int dir = W;dir <= NE;dir++)
		  {
			int nx=0, ny=0;
			switch (dir)
			  {
			  case W:
				ny = y;
				nx = x-1;
				break;
			  case NW:
				ny = y-1;
				nx = x-1;
				break;
			  case N:
				ny = y-1;
				nx = x;
				break;
			  case NE:
				ny = y-1;
				nx = x+1;
				break;
			  } // end switch
			tile &t2 = get_tile(coord::tc(nx,ny));

			if (!t2.conquered && t2.checked)
			  {
				ok = false;
				break;
			  }
		  } // end dir-for
		if (!ok)
		  {
			// we found adjacent checked and not conquered tile =>
			// this tile is not conquered.
			// if it was set as conquered before, it is changed
			if (t.conquered)
			  changed_tiles.push_back (coord::tc (x, y));
			t.conquered = false;
			t.owner = OWNER_NOONE;
			continue;
		  }
		// adjacent tiles are all ok, this tile should be checked (as S, SE, E, SW tiles not checked yet!)
		tiles_to_check.push (coord::tc (x,y));
		t.owner = OWNER_NOONE;
		// if it was conquered before, check it
		if (t.conquered)
		  changed_tiles.push_back (coord::tc(x,y));
		t.conquered = false;
		t.checked = false;
	  } // end initial pass

  int last_checked = 1;
  int checked = 0;

  while (last_checked != checked)
	{
	  int i = 0;
	  last_checked = checked;
	  checked = tiles_to_check.size();

	  //	  printf ("%d tiles to check\n", checked);

	  while (i < checked)
		{
		  i++;

		  coord c = tiles_to_check.front();
		  tiles_to_check.pop();

		  tile &t = get_tile(c);

		  //		  printf ("checking %d %d\n", c.x, c.y);

		  t.checked = true;

		  // Ok, this tile is not conquered, not checked yet, and is not a wall-tile
		  ok = true;
		  ready = true;
		  // lets check all its adjacent tiles
		  for (int dir = W;dir <= SW;dir++)
			{
			  int nx=0, ny=0;
			  switch (dir)
				{
				case W:
				  ny = c.getTY();
				  nx = c.getTX()-1;
				  break;
				case NW:
				  ny = c.getTY()-1;
				  nx = c.getTX()-1;
				  break;
				case N:
				  ny = c.getTY()-1;
				  nx = c.getTX();
				  break;
				case NE:
				  ny = c.getTY()-1;
				  nx = c.getTX()+1;
				  break;
				case E:
				  ny = c.getTY();
				  nx = c.getTX()+1;
				  break;
				case SE:
				  ny = c.getTY()+1;
				  nx = c.getTX()+1;
				  break;
				case S:
				  ny = c.getTY()+1;
				  nx = c.getTX();
				  break;
				case SW:
				  ny = c.getTY()+1;
				  nx = c.getTX()-1;
				  break;
				} // end switch
			  tile &t2 = get_tile(coord::tc(nx,ny));
			  //			  printf ("adjacent tile (%d %d) type %d owner %d\n", nx, ny, t2.type, t2.owner);
			  if (!t2.checked)
				{
				  ready = false;
				  continue;
				}
			  if (!t2.conquered)
				{
				  ok = false;
				  break;
				}
			  // conquer this to owner, if all goes ok
			  owner = t2.owner;
			} // end dir-for
	  
		  if (!ok)
			{
			  // we found adjacent checked and not conquered tile =>
			  // this tile is not conquered.
			  t.conquered = false;
			  t.owner = OWNER_NOONE;
			  //			  changed_tiles.push_back (coord (x, y));
			  continue;
			}

		  if (!ready)
			{
			  // adjacent tiles are not ready, this tile should be checked next round
			  tiles_to_check.push (c);
			  t.checked = false;
			  continue;
			}
		
		  // at this point the tile should belong to conquered tiles
		  //		  printf ("tile %d %d conquered to owner %d\n", c.x, c.y, owner);
		  changed_tiles.push_back (c);
		  t.conquered = true;
		  t.owner = owner;	  
		} // end while
	} // end while

  //  printf ("%d tiles left unchecked\n", tiles_to_check.size());
  // ok, now all the tiles remaining at status "not checked" should be conquered tiles.
  while (!tiles_to_check.empty())
	{
	  coord c = tiles_to_check.front();
	  tiles_to_check.pop();

	  tile &t = get_tile(c);

	  // this tile is not checked, so the tile north of it SHOULD be checked.
	  tile &nt = get_tile(coord::tc(c.getTX(), c.getTY()-1));
	  assert (nt.checked);
	  t.checked = true;
	  t.owner = nt.owner;
	  t.conquered = true;
	  changed_tiles.push_back (c);
	  //  printf ("final tile at %d %d owner %d, up type %d\n", c.x, c.y, nt.owner, nt.type);
	}
}

vector<const tile *> terrain::conquered_tiles(int type, int owner)
{
  vector<const tile *> ve;

  for (int y=0;y<YTILES;y++)
	for (int x=0;x<XTILES;x++)
	  {
		tile &t = get_tile(coord::tc(x,y));
		if (!t.conquered)
		  continue;
		if (t.owner != owner)
		  continue;
		if (t.type != type)
		  continue;
		ve.push_back(&t);
	  }
  return ve;
}

vector<coord> terrain::conq_tiles(int type, int owner)
{
  vector<coord> vec;

  for (int y=0;y<YTILES;y++)
	for (int x=0;x<XTILES;x++)
	  {
		tile &t = tiles[y][x];
		if (!t.conquered)
		  continue;
		if (t.owner != owner)
		  continue;
		if (t.type == type)
		  vec.push_back(coord::tc(x,y));
	  }
  return vec;
}

vector<coord> terrain::get_changed_tiles()
{
  vector<coord> ct = changed_tiles;
  changed_tiles.clear();
  return ct;
}

int terrain::check_surrounding (const coord &c, int player_index)
{
  tile t;
  int x = c.getTX();
  int y = c.getTY();
  int own_found = 0;
  if (x >= 0) 
	{
	  t = tiles[y][x-1];
	  if (t.owner == player_index && t.type == tile::TYPE_WALL)
	    own_found = 1;
	  //	  if (t.owner != player_index && t.type == tile::TYPE_WALL)
	  //  return -1;
	  if (y  >= 0)
		{
		  t = tiles[y-1][x];
		  if (t.owner == player_index && t.type == tile::TYPE_WALL)
		    own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //  return -1;
		  //t = tiles[y-1][x-1];
		  //if (t.owner == player_index && t.type == tile::TYPE_WALL)
		  //  own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //  return -1;
		}
	  if (y < YTILES-1)
		{
		  t = tiles[y+1][x];
		  if (t.owner == player_index && t.type == tile::TYPE_WALL)
		    own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //  return -1;
		  //t = tiles[y+1][x-1];
		  //if (t.owner == player_index && t.type == tile::TYPE_WALL)
		  //  own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //  return -1;
		}
	}
  if (x < XTILES-1)
	{
	  t = tiles[y][x+1];
	  if (t.owner == player_index && t.type == tile::TYPE_WALL)
	    own_found = 1;
	  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
	  //  return -1;
	  if (y  >= 0)
		{
		  //t = tiles[y-1][x+1];
		  //if (t.owner == player_index && t.type == tile::TYPE_WALL)
		  //	own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //	return -1;
		}
	  if (y < YTILES-1)
		{
		  //t = tiles[y+1][x+1];
		  //if (t.owner == player_index && t.type == tile::TYPE_WALL)
		  //	own_found = 1;
		  //if (t.owner != player_index && t.type == tile::TYPE_WALL)
		  //	return -1;
		}
	}
  return own_found;
}

void terrain::set_tile (const coord &c, int type, int owner)
{
  tile t (type, owner);
  set_tile (c, t);
}

void terrain::dead_player_remove(int index)
{
  for (int x=0;x<XTILES;x++)
    for (int y=0;y<YTILES;y++)
      {
	tile &t = tiles[y][x];
	if (t.owner != index)
	  continue;
	switch (t.type)
	  {
	  case tile::TYPE_EMPTY:
	  case tile::TYPE_CASTLE_A:
	  case tile::TYPE_CASTLE_C:
	  case tile::TYPE_CANNON_A:
	  case tile::TYPE_CANNON_C:
	  case tile::TYPE_CANNON_A_DEST:
	  case tile::TYPE_CANNON_C_DEST:
	  case tile::TYPE_BIG_CANNON_A:
	  case tile::TYPE_BIG_CANNON_C:
	  case tile::TYPE_BIG_CANNON_A_DEST:
	  case tile::TYPE_BIG_CANNON_C_DEST:
	    t.owner = OWNER_NOONE;
	    t.conquered = false;
	    set_tile (coord::tc(x, y), t);
	    break;
	  case tile::TYPE_WALL:
	    t.owner = OWNER_NOONE;
	    t.conquered = false;
 	    t.type = tile::TYPE_EMPTY;
	    set_tile (coord::tc(x,y), t);
	    break;
	  default:
	    assert(0);
	  }
	
      }
}

coord terrain::get_center_tile (const coord &c)
{
  coord x = coord::tc (c.getTX(), c.getTY());
  tile &ot = get_tile(c);

  if (ot.type == tile::TYPE_CANNON_A)
    {
      if (x.getTY() > 0)
	{ // check upper
	  x.decTY();
	  tile &t = get_tile(x);
	  if (t.type == tile::TYPE_CANNON_C)
	    return x;
	  if (x.getTX() > 0)
	    {
	      x.decTX();
	      // check upper-left
	      tile &t2 = get_tile(x);
	      if (t2.type == tile::TYPE_CANNON_C)
		return x;
	    }
	}
      x = coord::tc(c.getTX(), c.getTY());
      x.decTX();
      return x; // it's left then
    }
  if (ot.type == tile::TYPE_CANNON_A_DEST)
    {
      if (x.getTY() > 0)
	{ // check upper
	  x.decTY();
	  tile &t = get_tile(x);
	  if (t.type == tile::TYPE_CANNON_C_DEST)
	    return x;
	  if (x.getTX() > 0)
	    {
	      x.decTX();
	      // check upper-left
	      tile &t2 = get_tile(x);
	      if (t2.type == tile::TYPE_CANNON_C_DEST)
		return x;
	    }
	}
      x = coord::tc(c.getTX(), c.getTY());
      x.decTX();
      return x; // it's left then
    }
  if (ot.type == tile::TYPE_BIG_CANNON_A)
    {
      if (x.getTY() > 0)
	{ // check upper
	  x.decTY();
	  tile &t = get_tile(x);
	  if (t.type == tile::TYPE_BIG_CANNON_C)
	    return x;
	  if (x.getTX() > 0)
	    {
	      x.decTX();
	      // check upper-left
	      tile &t2 = get_tile(x);
	      if (t2.type == tile::TYPE_BIG_CANNON_C)
		return x;
	    }
	}
      x = coord::tc(c.getTX(), c.getTY());
      x.decTX();
      return x; // it's left then
    }

  if (ot.type == tile::TYPE_BIG_CANNON_A_DEST)
    {
      if (x.getTY() > 0)
	{ // check upper
	  x.decTY();
	  tile &t = get_tile(x);
	  if (t.type == tile::TYPE_BIG_CANNON_C_DEST)
	    return x;
	  if (x.getTX() > 0)
	    {
	      x.decTX();
	      // check upper-left
	      tile &t2 = get_tile(x);
	      if (t2.type == tile::TYPE_BIG_CANNON_C_DEST)
		return x;
	    }
	}
      x = coord::tc(c.getTX(), c.getTY());
      x.decTX();
      return x; // it's left then
    }

  return coord::tc(c.getTX(), c.getTY()); // otherwise it's this tile.
}

void terrain::hit (const coord &c, const ammo &a)
{
  tile &t = get_tile (c);
  if (t.type == tile::TYPE_CANNON_C || t.type == tile::TYPE_CANNON_A)
    {
      coord cc = get_center_tile(c);
      tile &t = get_tile(cc);
      t.hit(a);
      if (t.get_damage() >= CANNON_HP)
	{
	  t.type = tile::TYPE_CANNON_C_DEST;
	  set_tile(cc,t);
	  cc.incTX();
	  tile &t2 = get_tile(cc);
	  t2.type = tile::TYPE_CANNON_A_DEST;
	  set_tile(cc,t2);
	  cc.incTY();
	  tile &t3 = get_tile(cc);
	  t3.type = tile::TYPE_CANNON_A_DEST;
	  set_tile(cc,t3);
	  cc.decTX();
	  tile &t4 = get_tile(cc);
	  t4.type = tile::TYPE_CANNON_A_DEST;
	  set_tile(cc,t4);
	}
    }
  else if (t.type == tile::TYPE_BIG_CANNON_C || t.type == tile::TYPE_BIG_CANNON_A)
    {
      coord cc = get_center_tile(c);
      tile &t = get_tile(cc);
      t.hit(a);
      if (t.get_damage() >= BIG_CANNON_HP)
	{
	  t.type = tile::TYPE_BIG_CANNON_C_DEST;
	  set_tile(cc,t);
	  cc.incTX();
	  tile &t2 = get_tile(cc);
	  t2.type = tile::TYPE_BIG_CANNON_A_DEST;
	  set_tile(cc,t2);
	  cc.incTY();
	  tile &t3 = get_tile(cc);
	  t3.type = tile::TYPE_BIG_CANNON_A_DEST;
	  set_tile(cc,t3);
	  cc.decTX();
	  tile &t4 = get_tile(cc);
	  t4.type = tile::TYPE_BIG_CANNON_A_DEST;
	  set_tile(cc,t4);
	}
    }
}
