package npm

import (
	"bufio"
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"strconv"
	"strings"

	"github.com/jfrog/build-info-go/build"
	biUtils "github.com/jfrog/build-info-go/build/utils"
	"github.com/jfrog/gofrog/version"
	commandUtils "github.com/jfrog/jfrog-cli-core/v2/artifactory/commands/utils"
	"github.com/jfrog/jfrog-cli-core/v2/artifactory/utils/npm"
	buildUtils "github.com/jfrog/jfrog-cli-core/v2/common/build"
	"github.com/jfrog/jfrog-cli-core/v2/common/project"
	"github.com/jfrog/jfrog-cli-core/v2/utils/config"
	"github.com/jfrog/jfrog-cli-core/v2/utils/coreutils"
	"github.com/jfrog/jfrog-cli-core/v2/utils/ioutils"
	"github.com/jfrog/jfrog-client-go/auth"
	"github.com/jfrog/jfrog-client-go/utils/errorutils"
	"github.com/jfrog/jfrog-client-go/utils/log"
	"github.com/spf13/viper"
)

const (
	npmrcFileName          = ".npmrc"
	npmrcBackupFileName    = "jfrog.npmrc.backup"
	minSupportedNpmVersion = "5.4.0"

	// Scoped authentication env var that sets the _auth or _authToken npm config variables.
	npmConfigAuthEnv                  = "npm_config_%s:%s"
	npmVersionSupportingScopedAuthEnv = "9.2.0"
	// Legacy un-scoped auth env vars doesn't support access tokens (with _authToken suffix).
	npmLegacyConfigAuthEnv = "npm_config__auth"
)

type NpmCommand struct {
	CommonArgs
	cmdName        string
	jsonOutput     bool
	executablePath string
	// Function to be called to restore the user's old npmrc and delete the one we created.
	restoreNpmrcFunc func() error
	workingDirectory string
	// Npm registry as exposed by Artifactory.
	registry string
	// Npm token generated by Artifactory using the user's provided credentials.
	npmAuth             string
	authArtDetails      auth.ServiceDetails
	npmVersion          *version.Version
	internalCommandName string
	configFilePath      string
	collectBuildInfo    bool
	buildInfoModule     *build.NpmModule
}

func NewNpmCommand(cmdName string, collectBuildInfo bool) *NpmCommand {
	return &NpmCommand{
		cmdName:             cmdName,
		collectBuildInfo:    collectBuildInfo,
		internalCommandName: "rt_npm_" + cmdName,
	}
}

func NewNpmInstallCommand() *NpmCommand {
	return &NpmCommand{cmdName: "install", internalCommandName: "rt_npm_install"}
}

func NewNpmCiCommand() *NpmCommand {
	return &NpmCommand{cmdName: "ci", internalCommandName: "rt_npm_ci"}
}

func (nc *NpmCommand) CommandName() string {
	return nc.internalCommandName
}

func (nc *NpmCommand) SetConfigFilePath(configFilePath string) *NpmCommand {
	nc.configFilePath = configFilePath
	return nc
}

func (nc *NpmCommand) SetArgs(args []string) *NpmCommand {
	nc.npmArgs = args
	return nc
}

func (nc *NpmCommand) SetRepoConfig(conf *project.RepositoryConfig) *NpmCommand {
	serverDetails, _ := conf.ServerDetails()
	nc.SetRepo(conf.TargetRepo()).SetServerDetails(serverDetails)
	return nc
}

func (nc *NpmCommand) SetServerDetails(serverDetails *config.ServerDetails) *NpmCommand {
	nc.serverDetails = serverDetails
	return nc
}

func (nc *NpmCommand) SetRepo(repo string) *NpmCommand {
	nc.repo = repo
	return nc
}

func (nc *NpmCommand) Init() error {
	// Read config file.
	log.Debug("Preparing to read the config file", nc.configFilePath)
	vConfig, err := project.ReadConfigFile(nc.configFilePath, project.YAML)
	if err != nil {
		return err
	}

	repoConfig, err := nc.getRepoConfig(vConfig)
	if err != nil {
		return err
	}
	_, _, _, filteredNpmArgs, buildConfiguration, err := commandUtils.ExtractNpmOptionsFromArgs(nc.npmArgs)
	if err != nil {
		return err
	}
	nc.SetRepoConfig(repoConfig).SetArgs(filteredNpmArgs).SetBuildConfiguration(buildConfiguration)
	return nil
}

// Get the repository configuration from the config file.
// Use the resolver prefix for all commands except for 'dist-tag' which use the deployer prefix.
func (nc *NpmCommand) getRepoConfig(vConfig *viper.Viper) (repoConfig *project.RepositoryConfig, err error) {
	prefix := project.ProjectConfigResolverPrefix
	// Aliases accepted by npm.
	if nc.cmdName == "dist-tag" || nc.cmdName == "dist-tags" {
		prefix = project.ProjectConfigDeployerPrefix
	}
	return project.GetRepoConfigByPrefix(nc.configFilePath, prefix, vConfig)
}

func (nc *NpmCommand) SetBuildConfiguration(buildConfiguration *buildUtils.BuildConfiguration) *NpmCommand {
	nc.buildConfiguration = buildConfiguration
	return nc
}

func (nc *NpmCommand) ServerDetails() (*config.ServerDetails, error) {
	return nc.serverDetails, nil
}

func (nc *NpmCommand) RestoreNpmrcFunc() func() error {
	return nc.restoreNpmrcFunc
}

func (nc *NpmCommand) PreparePrerequisites(repo string) error {
	log.Debug("Preparing prerequisites...")
	var err error
	nc.npmVersion, nc.executablePath, err = biUtils.GetNpmVersionAndExecPath(log.Logger)
	if err != nil {
		return err
	}
	if nc.npmVersion.Compare(minSupportedNpmVersion) > 0 {
		return errorutils.CheckErrorf(
			"JFrog CLI npm %s command requires npm client version %s or higher. The Current version is: %s", nc.cmdName, minSupportedNpmVersion, nc.npmVersion.GetVersion())
	}

	if err = nc.setJsonOutput(); err != nil {
		return err
	}

	nc.workingDirectory, err = coreutils.GetWorkingDirectory()
	if err != nil {
		return err
	}
	log.Debug("Working directory set to:", nc.workingDirectory)
	if err = nc.setArtifactoryAuth(); err != nil {
		return err
	}

	if err = nc.setNpmAuthRegistry(repo); err != nil {
		return err
	}

	return nc.setRestoreNpmrcFunc()
}

func (nc *NpmCommand) setNpmAuthRegistry(repo string) (err error) {
	nc.npmAuth, nc.registry, err = commandUtils.GetArtifactoryNpmRepoDetails(repo, nc.authArtDetails, !nc.isNpmVersionSupportsScopedAuthEnv())
	return
}

func (nc *NpmCommand) setRestoreNpmrcFunc() error {
	restoreNpmrcFunc, err := ioutils.BackupFile(filepath.Join(nc.workingDirectory, npmrcFileName), npmrcBackupFileName)
	if err != nil {
		return err
	}
	nc.restoreNpmrcFunc = func() error {
		if unsetEnvErr := os.Unsetenv(npmConfigAuthEnv); unsetEnvErr != nil {
			return unsetEnvErr
		}
		return restoreNpmrcFunc()
	}
	return nil
}

func (nc *NpmCommand) setArtifactoryAuth() error {
	authArtDetails, err := nc.serverDetails.CreateArtAuthConfig()
	if err != nil {
		return err
	}
	if authArtDetails.GetSshAuthHeaders() != nil {
		return errorutils.CheckErrorf("SSH authentication is not supported in this command")
	}
	nc.authArtDetails = authArtDetails
	return nil
}

func (nc *NpmCommand) setJsonOutput() error {
	jsonOutput, err := npm.ConfigGet(nc.npmArgs, "json", nc.executablePath)
	if err != nil {
		return err
	}

	// In case of --json=<not boolean>, the value of json is set to 'true', but the result from the command is not 'true'
	nc.jsonOutput = jsonOutput != "false"
	return nil
}

func (nc *NpmCommand) processConfigLine(configLine string) (filteredLine string, err error) {
	splitOption := strings.SplitN(configLine, "=", 2)
	key := strings.TrimSpace(splitOption[0])
	validLine := len(splitOption) == 2 && isValidKey(key)
	if !validLine {
		if strings.HasPrefix(splitOption[0], "@") {
			// Override scoped registries (@scope = xyz)
			return fmt.Sprintf("%s = %s\n", splitOption[0], nc.registry), nil
		}
		return
	}
	value := strings.TrimSpace(splitOption[1])
	if key == commandUtils.NpmConfigAuthKey || key == commandUtils.NpmConfigAuthTokenKey {
		return "", nc.setNpmConfigAuthEnv(value, key)
	}
	if strings.HasPrefix(value, "[") && strings.HasSuffix(value, "]") {
		return addArrayConfigs(key, value), nil
	}

	return fmt.Sprintf("%s\n", configLine), err
}

func (nc *NpmCommand) setNpmConfigAuthEnv(value, authKey string) error {
	// Check if the npm version supports scoped auth env vars.
	if nc.isNpmVersionSupportsScopedAuthEnv() {
		// Get registry name without the protocol name but including the '//'
		registryWithoutProtocolName := nc.registry[strings.Index(nc.registry, "://")+1:]
		// Set "npm_config_//<registry-url>:_auth" environment variable to allow authentication with Artifactory
		scopedRegistryEnv := fmt.Sprintf(npmConfigAuthEnv, registryWithoutProtocolName, authKey)
		return os.Setenv(scopedRegistryEnv, value)
	}
	// Set "npm_config__auth" environment variable to allow authentication with Artifactory when running post-install scripts on subdirectories.
	// For older versions, use un-scoped auth env vars.
	return os.Setenv(npmLegacyConfigAuthEnv, value)
}

func (nc *NpmCommand) isNpmVersionSupportsScopedAuthEnv() bool {
	return nc.npmVersion.Compare(npmVersionSupportingScopedAuthEnv) <= 0
}

func (nc *NpmCommand) prepareConfigData(data []byte) ([]byte, error) {
	var filteredConf []string
	configString := string(data) + "\n" + nc.npmAuth
	scanner := bufio.NewScanner(strings.NewReader(configString))
	for scanner.Scan() {
		currOption := scanner.Text()
		if currOption == "" {
			continue
		}
		filteredLine, err := nc.processConfigLine(currOption)
		if err != nil {
			return nil, errorutils.CheckError(err)
		}
		if filteredLine != "" {
			filteredConf = append(filteredConf, filteredLine)
		}
	}
	if err := scanner.Err(); err != nil {
		return nil, errorutils.CheckError(err)
	}

	filteredConf = append(filteredConf, "json = ", strconv.FormatBool(nc.jsonOutput), "\n")
	filteredConf = append(filteredConf, "registry = ", nc.registry, "\n")
	return []byte(strings.Join(filteredConf, "")), nil
}

func (nc *NpmCommand) CreateTempNpmrc() error {
	data, err := npm.GetConfigList(nc.npmArgs, nc.executablePath)
	if err != nil {
		return err
	}
	configData, err := nc.prepareConfigData(data)
	if err != nil {
		return errorutils.CheckError(err)
	}

	if err = removeNpmrcIfExists(nc.workingDirectory); err != nil {
		return err
	}
	log.Debug("Creating temporary .npmrc file.")
	return errorutils.CheckError(os.WriteFile(filepath.Join(nc.workingDirectory, npmrcFileName), configData, 0755))
}

func (nc *NpmCommand) Run() (err error) {
	if err = nc.PreparePrerequisites(nc.repo); err != nil {
		return
	}
	defer func() {
		err = errors.Join(err, nc.restoreNpmrcFunc())
	}()
	if err = nc.CreateTempNpmrc(); err != nil {
		return
	}

	if err = nc.prepareBuildInfoModule(); err != nil {
		return
	}

	err = nc.collectDependencies()
	return
}

func (nc *NpmCommand) prepareBuildInfoModule() error {
	var err error
	if nc.collectBuildInfo {
		nc.collectBuildInfo, err = nc.buildConfiguration.IsCollectBuildInfo()
		if err != nil {
			return err
		}
	}
	// Build-info should not be created when installing a single package (npm install <package name>).
	if nc.collectBuildInfo && len(filterFlags(nc.npmArgs)) > 0 {
		log.Info("Build-info dependencies collection is not supported for installations of single packages. Build-info creation is skipped.")
		nc.collectBuildInfo = false
	}
	buildName, err := nc.buildConfiguration.GetBuildName()
	if err != nil {
		return err
	}
	buildNumber, err := nc.buildConfiguration.GetBuildNumber()
	if err != nil {
		return err
	}
	buildInfoService := buildUtils.CreateBuildInfoService()
	npmBuild, err := buildInfoService.GetOrCreateBuildWithProject(buildName, buildNumber, nc.buildConfiguration.GetProject())
	if err != nil {
		return errorutils.CheckError(err)
	}
	nc.buildInfoModule, err = npmBuild.AddNpmModule(nc.workingDirectory)
	if err != nil {
		return errorutils.CheckError(err)
	}
	nc.buildInfoModule.SetCollectBuildInfo(nc.collectBuildInfo)
	if nc.buildConfiguration.GetModule() != "" {
		nc.buildInfoModule.SetName(nc.buildConfiguration.GetModule())
	}
	return nil
}

func (nc *NpmCommand) collectDependencies() error {
	nc.buildInfoModule.SetNpmArgs(append([]string{nc.cmdName}, nc.npmArgs...))
	return errorutils.CheckError(nc.buildInfoModule.Build())
}

// Gets a config with value which is an array
func addArrayConfigs(key, arrayValue string) string {
	if arrayValue == "[]" {
		return ""
	}

	values := strings.TrimPrefix(strings.TrimSuffix(arrayValue, "]"), "[")
	valuesSlice := strings.Split(values, ",")
	var configArrayValues strings.Builder
	for _, val := range valuesSlice {
		configArrayValues.WriteString(fmt.Sprintf("%s[] = %s\n", key, val))
	}

	return configArrayValues.String()
}

func removeNpmrcIfExists(workingDirectory string) error {
	if _, err := os.Stat(filepath.Join(workingDirectory, npmrcFileName)); err != nil {
		// The file does not exist, nothing to do.
		if os.IsNotExist(err) {
			return nil
		}
		return errorutils.CheckError(err)
	}

	log.Debug("Removing existing .npmrc file")
	return errorutils.CheckError(os.Remove(filepath.Join(workingDirectory, npmrcFileName)))
}

// To avoid writing configurations that are used by us
func isValidKey(key string) bool {
	return !strings.HasPrefix(key, "//") &&
		!strings.HasPrefix(key, ";") && // Comments
		!strings.HasPrefix(key, "@") && // Scoped configurations
		key != "registry" &&
		key != "metrics-registry" &&
		key != "json" // Handled separately because 'npm c ls' should run with json=false
}

func filterFlags(splitArgs []string) []string {
	var filteredArgs []string
	for _, arg := range splitArgs {
		if !strings.HasPrefix(arg, "-") {
			filteredArgs = append(filteredArgs, arg)
		}
	}
	return filteredArgs
}

func (nc *NpmCommand) GetRepo() string {
	return nc.repo
}

// Creates an .npmrc file in the project's directory in order to configure the provided Artifactory server as a resolution server
func SetArtifactoryAsResolutionServer(serverDetails *config.ServerDetails, depsRepo string) (clearResolutionServerFunc func() error, err error) {
	npmCmd := NewNpmInstallCommand().SetServerDetails(serverDetails)
	if err = npmCmd.PreparePrerequisites(depsRepo); err != nil {
		return
	}
	if err = npmCmd.CreateTempNpmrc(); err != nil {
		return
	}
	clearResolutionServerFunc = npmCmd.RestoreNpmrcFunc()
	log.Info(fmt.Sprintf("Resolving dependencies from '%s' from repo '%s'", serverDetails.Url, depsRepo))
	return
}
