/*
 *  svaf_amalgam.c: code for attaching multiple models into a single
 *                  combined array.
 *
 *  Copyright (c) 2006, Michael C. Martin
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - Neither the name 'Bumbershoot Software' nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 *  CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 *  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 *  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 */

#include <stdlib.h>
#include "svaf.h"
	
static void
shift_vertex_offset (svaf_model *m, int offset)
{
	int i, j;
	for (i = 0; i < m->num_commands; i++)
		if (m->commands[i].tag == SVAF_CMD_ELEMENTS)
			for (j = 0; j < m->commands[i].command.elements.count; j++)
				m->commands[i].command.elements.indices[j] += offset;
}

svaf_amalgam *
svaf_NewAmalgam (void)
{
	svaf_amalgam *a = (svaf_amalgam *)malloc (sizeof (svaf_amalgam));
	a->vertices = a->normals = NULL;
	a->num_vertices = 0;

	return a;
}

void
svaf_FreeAmalgam (svaf_amalgam *a)
{
	if (a)
	{
		if (a->vertices)
			free (a->vertices);
		if (a->normals)
			free (a->normals);
		free (a);
	}
}

void
svaf_RegisterGeometry (svaf_amalgam *a, svaf_model *m)
{
	int newVertCount = m->num_vertices;
	int oldCount = a->num_vertices * 3;
	int newCount = (a->num_vertices + newVertCount) * 3;
	GLfloat *newVertices = (GLfloat *) malloc (sizeof (GLfloat) * newCount * 3);
	GLfloat *newNormals = (GLfloat *) malloc (sizeof (GLfloat) * newCount * 3);
	GLfloat *addedVertices, *addedNormals;
	int i;

	for (i = 0; i < oldCount; i++) 
	{
		newVertices[i] = a->vertices[i];
		newNormals[i] = a->normals[i];
	}
	addedVertices = m->vertices;
	addedNormals = m->normals;

	for (i = oldCount; i < newCount; i++)
	{
		newVertices[i] = addedVertices[i-oldCount];
		newNormals[i] = addedNormals[i-oldCount];
	}
	if (a->vertices)
		free (a->vertices);
	if (a->normals)
		free (a->normals);
	a->vertices = newVertices;
	a->normals = newNormals;
	a->num_vertices = newCount / 3;

	free (m->vertices);
	free (m->normals);
	m->vertices = NULL;
	m->normals = NULL;
	
	shift_vertex_offset (m, oldCount / 3);
}

void
svaf_SelectAmalgam (svaf_amalgam *a)
{
	if (a->vertices != NULL)
	{
		glEnable(GL_NORMALIZE);

		glVertexPointer (3, GL_FLOAT, 0, a->vertices);
		glNormalPointer (GL_FLOAT, 0, a->normals);
	}
}
