/*
 * Decompiled with CFR 0.152.
 */
package jwbroek.cuelib;

import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jwbroek.cuelib.CueSheet;
import jwbroek.cuelib.CueSheetSerializer;
import jwbroek.cuelib.CueSheetToXmlSerializer;
import jwbroek.cuelib.FileData;
import jwbroek.cuelib.Index;
import jwbroek.cuelib.LineOfInput;
import jwbroek.cuelib.Message;
import jwbroek.cuelib.Position;
import jwbroek.cuelib.TrackData;
import jwbroek.io.FileSelector;
import jwbroek.util.LogUtil;

public final class CueParser {
    private static final Logger logger = Logger.getLogger(CueParser.class.getCanonicalName());
    private static final String WARNING_EMPTY_LINES = "Empty lines not allowed. Will ignore.";
    private static final String WARNING_UNPARSEABLE_INPUT = "Unparseable line. Will ignore.";
    private static final String WARNING_INVALID_CATALOG_NUMBER = "Invalid catalog number.";
    private static final String WARNING_NONCOMPLIANT_FILE_TYPE = "Noncompliant file type.";
    private static final String WARNING_NO_FLAGS = "No flags specified.";
    private static final String WARNING_NONCOMPLIANT_FLAG = "Noncompliant flag(s) specified.";
    private static final String WARNING_WRONG_NUMBER_OF_DIGITS = "Wrong number of digits in number.";
    private static final String WARNING_NONCOMPLIANT_ISRC_CODE = "ISRC code has noncompliant format.";
    private static final String WARNING_FIELD_LENGTH_OVER_80 = "The field is too long to burn as CD-TEXT. The maximum length is 80.";
    private static final String WARNING_NONCOMPLIANT_DATA_TYPE = "Noncompliant data type specified.";
    private static final String WARNING_TOKEN_NOT_UPPERCASE = "Token has wrong case. Uppercase was expected.";
    private static final String WARNING_INVALID_FRAMES_VALUE = "Position has invalid frame value. Should be 00-74.";
    private static final String WARNING_INVALID_SECONDS_VALUE = "Position has invalid seconds value. Should be 00-59.";
    private static final String WARNING_DATUM_APPEARS_TOO_OFTEN = "Datum appears too often.";
    private static final String WARNING_FILE_IN_WRONG_PLACE = "A FILE datum must come before everything else except REM and CATALOG.";
    private static final String WARNING_FLAGS_IN_WRONG_PLACE = "A FLAGS datum must come after a TRACK, but before any INDEX of that TRACK.";
    private static final String WARNING_NO_FILE_SPECIFIED = "Datum must appear in FILE, but no FILE specified.";
    private static final String WARNING_NO_TRACK_SPECIFIED = "Datum must appear in TRACK, but no TRACK specified.";
    private static final String WARNING_INVALID_INDEX_NUMBER = "Invalid index number. First number must be 0 or 1; all next ones sequential.";
    private static final String WARNING_INVALID_FIRST_POSITION = "Invalid position. First index must have position 00:00:00";
    private static final String WARNING_ISRC_IN_WRONG_PLACE = "An ISRC datum must come after TRACK, but before any INDEX of TRACK.";
    private static final String WARNING_PREGAP_IN_WRONG_PLACE = "A PREGAP datum must come after TRACK, but before any INDEX of that TRACK.";
    private static final String WARNING_INDEX_AFTER_POSTGAP = "A POSTGAP datum must come after all INDEX data of a TRACK.";
    private static final String WARNING_INVALID_TRACK_NUMBER = "Invalid track number. First number must be 1; all next ones sequential.";
    private static final String WARNING_INVALID_YEAR = "Invalid year. Should be a number from 1 to 9999 (inclusive).";
    private static final Pattern PATTERN_POSITION = Pattern.compile("^(\\d*):(\\d*):(\\d*)$");
    private static final Pattern PATTERN_CATALOG_NUMBER = Pattern.compile("^\\d{13}$");
    private static final Pattern PATTERN_FILE = Pattern.compile("^FILE\\s+((?:\"[^\"]*\")|\\S+)\\s+(\\S+)\\s*$", 2);
    private static final Pattern PATTERN_CDTEXTFILE = Pattern.compile("^CDTEXTFILE\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_FLAGS = Pattern.compile("^FLAGS(\\s+\\w+)*\\s*$", 2);
    private static final Pattern PATTERN_INDEX = Pattern.compile("^INDEX\\s+(\\d+)\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    private static final Pattern PATTERN_ISRC_CODE = Pattern.compile("^\\w{5}\\d{7}$");
    private static final Pattern PATTERN_PERFORMER = Pattern.compile("^PERFORMER\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_POSTGAP = Pattern.compile("^POSTGAP\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    private static final Pattern PATTERN_PREGAP = Pattern.compile("^PREGAP\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    private static final Pattern PATTERN_REM_COMMENT = Pattern.compile("^(REM\\s+COMMENT)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_REM_DATE = Pattern.compile("^(REM\\s+DATE)\\s+(\\d+)\\s*$", 2);
    private static final Pattern PATTERN_REM_DISCID = Pattern.compile("^(REM\\s+DISCID)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_REM_GENRE = Pattern.compile("^(REM\\s+GENRE)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_SONGWRITER = Pattern.compile("^SONGWRITER\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_TITLE = Pattern.compile("^TITLE\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    private static final Pattern PATTERN_TRACK = Pattern.compile("TRACK\\s+(\\d+)\\s+(\\S+)\\s*$", 2);
    private static final Set<String> COMPLIANT_FILE_TYPES = new TreeSet<String>(Arrays.asList("BINARY", "MOTOROLA", "AIFF", "WAVE", "MP3"));
    private static final Set<String> COMPLIANT_FLAGS = new TreeSet<String>(Arrays.asList("DCP", "4CH", "PRE", "SCMS", "DATA"));
    private static final Set<String> COMPLIANT_DATA_TYPES = new TreeSet<String>(Arrays.asList("AUDIO", "CDG", "MODE1/2048", "MODE1/2352", "MODE2/2336", "MODE2/2352", "CDI/2336", "CDI/2352"));

    private CueParser() {
        logger.entering(FileSelector.class.getCanonicalName(), "FileSelector(File)");
        logger.warning("jwbroek.cuelib.CueParser should not be initialized");
        logger.exiting(FileSelector.class.getCanonicalName(), "FileSelector(File)");
    }

    public static CueSheet parse(InputStream inputStream) throws IOException {
        logger.entering(CueParser.class.getCanonicalName(), "parse(InputStream)", inputStream);
        CueSheet cueSheet = CueParser.parse(new LineNumberReader(new InputStreamReader(inputStream)));
        logger.exiting(CueParser.class.getCanonicalName(), "parse(InputStream)", cueSheet);
        return cueSheet;
    }

    public static CueSheet parse(File file) throws IOException {
        logger.entering(CueParser.class.getCanonicalName(), "parse(File)", file);
        CueSheet cueSheet = CueParser.parse(new LineNumberReader(new FileReader(file)));
        logger.exiting(CueParser.class.getCanonicalName(), "parse(File)", cueSheet);
        return cueSheet;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static CueSheet parse(LineNumberReader lineNumberReader) throws IOException {
        logger.entering(CueParser.class.getCanonicalName(), "parse(LineNumberReader)", lineNumberReader);
        logger.fine("Parsing cue sheet.");
        CueSheet cueSheet = new CueSheet();
        try {
            String string = lineNumberReader.readLine();
            while (string != null) {
                logger.finest("Processing input line.");
                string = string.trim();
                LineOfInput lineOfInput = new LineOfInput(lineNumberReader.getLineNumber(), string, cueSheet);
                if (string.length() == 0) {
                    CueParser.addWarning(lineOfInput, WARNING_EMPTY_LINES);
                } else if (string.length() < 2) {
                    CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                } else {
                    block1 : switch (string.charAt(0)) {
                        case 'C': 
                        case 'c': {
                            switch (string.charAt(1)) {
                                case 'A': 
                                case 'a': {
                                    CueParser.parseCatalog(lineOfInput);
                                    break block1;
                                }
                                case 'D': 
                                case 'd': {
                                    CueParser.parseCdTextFile(lineOfInput);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'F': 
                        case 'f': {
                            switch (string.charAt(1)) {
                                case 'I': 
                                case 'i': {
                                    CueParser.parseFile(lineOfInput);
                                    break block1;
                                }
                                case 'L': 
                                case 'l': {
                                    CueParser.parseFlags(lineOfInput);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'I': 
                        case 'i': {
                            switch (string.charAt(1)) {
                                case 'N': 
                                case 'n': {
                                    CueParser.parseIndex(lineOfInput);
                                    break block1;
                                }
                                case 'S': 
                                case 's': {
                                    CueParser.parseIsrc(lineOfInput);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'P': 
                        case 'p': {
                            switch (string.charAt(1)) {
                                case 'E': 
                                case 'e': {
                                    CueParser.parsePerformer(lineOfInput);
                                    break block1;
                                }
                                case 'O': 
                                case 'o': {
                                    CueParser.parsePostgap(lineOfInput);
                                    break block1;
                                }
                                case 'R': 
                                case 'r': {
                                    CueParser.parsePregap(lineOfInput);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'R': 
                        case 'r': {
                            CueParser.parseRem(lineOfInput);
                            break;
                        }
                        case 'S': 
                        case 's': {
                            CueParser.parseSongwriter(lineOfInput);
                            break;
                        }
                        case 'T': 
                        case 't': {
                            switch (string.charAt(1)) {
                                case 'I': 
                                case 'i': {
                                    CueParser.parseTitle(lineOfInput);
                                    break block1;
                                }
                                case 'R': 
                                case 'r': {
                                    CueParser.parseTrack(lineOfInput);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        default: {
                            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
                        }
                    }
                }
                string = lineNumberReader.readLine();
            }
        }
        finally {
            logger.finest("Closing input reader.");
            lineNumberReader.close();
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parse(LineNumberReader)", cueSheet);
        return cueSheet;
    }

    private static boolean startsWith(LineOfInput lineOfInput, String string) {
        logger.entering(CueParser.class.getCanonicalName(), "startsWith(LineOfInput,String)", new Object[]{lineOfInput, string});
        if (lineOfInput.getInput().startsWith(string)) {
            logger.exiting(CueParser.class.getCanonicalName(), "startsWith(LineOfInput,String)", true);
            return true;
        }
        if (lineOfInput.getInput().substring(0, string.length()).equalsIgnoreCase(string)) {
            CueParser.addWarning(lineOfInput, WARNING_TOKEN_NOT_UPPERCASE);
            logger.exiting(CueParser.class.getCanonicalName(), "startsWith(LineOfInput,String)", true);
            return true;
        }
        logger.exiting(CueParser.class.getCanonicalName(), "startsWith(LineOfInput,String)", false);
        return false;
    }

    private static boolean contains(LineOfInput lineOfInput, Pattern pattern) {
        logger.entering(CueParser.class.getCanonicalName(), "contains(LineOfInput,Pattern)", new Object[]{lineOfInput, pattern});
        Matcher matcher = pattern.matcher(lineOfInput.getInput());
        if (matcher.find()) {
            if (matcher.groupCount() > 0 && !matcher.group(1).equals(matcher.group(1).toUpperCase())) {
                CueParser.addWarning(lineOfInput, WARNING_TOKEN_NOT_UPPERCASE);
            }
            logger.exiting(CueParser.class.getCanonicalName(), "contains(LineOfInput,Pattern)", true);
            return true;
        }
        logger.exiting(CueParser.class.getCanonicalName(), "contains(LineOfInput,Pattern)", false);
        return false;
    }

    private static void parseCatalog(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseCatalog(LineOfInput)", lineOfInput);
        if (CueParser.startsWith(lineOfInput, "CATALOG")) {
            String string = lineOfInput.getInput().substring("CATALOG".length()).trim();
            if (!PATTERN_CATALOG_NUMBER.matcher(string).matches()) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_CATALOG_NUMBER);
            }
            if (lineOfInput.getAssociatedSheet().getCatalog() != null) {
                CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            lineOfInput.getAssociatedSheet().setCatalog(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseCatalog(LineOfInput)");
    }

    private static void parseFile(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseFile(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_FILE.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "FILE") && matcher.matches()) {
            String string;
            if (!COMPLIANT_FILE_TYPES.contains(matcher.group(2))) {
                if (COMPLIANT_FILE_TYPES.contains(matcher.group(2).toUpperCase())) {
                    CueParser.addWarning(lineOfInput, WARNING_TOKEN_NOT_UPPERCASE);
                } else {
                    CueParser.addWarning(lineOfInput, WARNING_NONCOMPLIANT_FILE_TYPE);
                }
            }
            if ((string = matcher.group(1)).length() > 0 && string.charAt(0) == '\"' && string.charAt(string.length() - 1) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            lineOfInput.getAssociatedSheet().getFileData().add(new FileData(lineOfInput.getAssociatedSheet(), string, matcher.group(2).toUpperCase()));
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseFile(LineOfInput)");
    }

    private static void parseCdTextFile(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseCdTextFile(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_CDTEXTFILE.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "CDTEXTFILE") && matcher.matches()) {
            String string;
            if (lineOfInput.getAssociatedSheet().getCdTextFile() != null) {
                logger.warning(WARNING_DATUM_APPEARS_TOO_OFTEN);
                lineOfInput.getAssociatedSheet().addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            if ((string = matcher.group(1)).length() > 0 && string.charAt(0) == '\"' && string.charAt(string.length() - 1) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            lineOfInput.getAssociatedSheet().setCdTextFile(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseCdTextFile(LineOfInput)");
    }

    private static void parseFlags(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseFlags(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_FLAGS.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "FLAGS") && matcher.matches()) {
            if (null == matcher.group(1)) {
                CueParser.addWarning(lineOfInput, WARNING_NO_FLAGS);
            } else {
                Set<String> set;
                TrackData trackData = CueParser.getLastTrackData(lineOfInput);
                if (trackData.getIndices().size() > 0) {
                    CueParser.addWarning(lineOfInput, WARNING_FLAGS_IN_WRONG_PLACE);
                }
                if (!(set = trackData.getFlags()).isEmpty()) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                Scanner scanner = new Scanner(matcher.group(1));
                while (scanner.hasNext()) {
                    String string = scanner.next();
                    if (!COMPLIANT_FLAGS.contains(string)) {
                        CueParser.addWarning(lineOfInput, WARNING_NONCOMPLIANT_FLAG);
                    }
                    set.add(string);
                }
            }
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseFlags(LineOfInput)");
    }

    private static void parseIndex(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseIndex(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_INDEX.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "INDEX") && matcher.matches()) {
            TrackData trackData;
            List<Index> list;
            if (matcher.group(1).length() != 2) {
                CueParser.addWarning(lineOfInput, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            if ((list = (trackData = CueParser.getLastTrackData(lineOfInput)).getIndices()).isEmpty() && trackData.getPostgap() != null) {
                CueParser.addWarning(lineOfInput, WARNING_INDEX_AFTER_POSTGAP);
            }
            int n = Integer.parseInt(matcher.group(1));
            if (list.isEmpty() && n > 1 || !list.isEmpty() && list.get(list.size() - 1).getNumber() != n - 1) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_INDEX_NUMBER);
            }
            List<Index> list2 = CueParser.getLastFileData(lineOfInput).getAllIndices();
            Position position = CueParser.parsePosition(lineOfInput, matcher.group(2));
            if (list2.isEmpty() && (position.getMinutes() != 0 || position.getSeconds() != 0 || position.getFrames() != 0)) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_FIRST_POSITION);
            }
            list.add(new Index(n, position));
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseIndex(LineOfInput)");
    }

    private static void parseIsrc(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseIsrc(LineOfInput)", lineOfInput);
        if (CueParser.startsWith(lineOfInput, "ISRC")) {
            TrackData trackData;
            String string = lineOfInput.getInput().substring("ISRC".length()).trim();
            if (!PATTERN_ISRC_CODE.matcher(string).matches()) {
                CueParser.addWarning(lineOfInput, WARNING_NONCOMPLIANT_ISRC_CODE);
            }
            if ((trackData = CueParser.getLastTrackData(lineOfInput)).getIndices().size() > 0) {
                CueParser.addWarning(lineOfInput, WARNING_ISRC_IN_WRONG_PLACE);
            }
            if (trackData.getIsrcCode() != null) {
                CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            trackData.setIsrcCode(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseIsrc(LineOfInput)");
    }

    private static void parsePerformer(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parsePerformer(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_PERFORMER.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "PERFORMER") && matcher.matches()) {
            String string = matcher.group(1);
            if (string.charAt(0) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            if (string.length() > 80) {
                CueParser.addWarning(lineOfInput, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (lineOfInput.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(lineOfInput).getTrackData().size() == 0) {
                if (lineOfInput.getAssociatedSheet().getPerformer() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                lineOfInput.getAssociatedSheet().setPerformer(string);
            } else {
                TrackData trackData = CueParser.getLastTrackData(lineOfInput);
                if (trackData.getPerformer() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setPerformer(string);
            }
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parsePerformer(LineOfInput)");
    }

    private static void parsePostgap(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parsePostgap(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_POSTGAP.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "POSTGAP") && matcher.matches()) {
            TrackData trackData = CueParser.getLastTrackData(lineOfInput);
            if (trackData.getPostgap() != null) {
                CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            trackData.setPostgap(CueParser.parsePosition(lineOfInput, matcher.group(1)));
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parsePostgap(LineOfInput)");
    }

    private static void parsePregap(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parsePregap(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_PREGAP.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "PREGAP") && matcher.matches()) {
            TrackData trackData = CueParser.getLastTrackData(lineOfInput);
            if (trackData.getPregap() != null) {
                CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            if (trackData.getIndices().size() > 0) {
                CueParser.addWarning(lineOfInput, WARNING_PREGAP_IN_WRONG_PLACE);
            }
            trackData.setPregap(CueParser.parsePosition(lineOfInput, matcher.group(1)));
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parsePregap(LineOfInput)");
    }

    private static void parseRemComment(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseRemComment(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_REM_COMMENT.matcher(lineOfInput.getInput());
        if (matcher.find()) {
            String string = matcher.group(2);
            if (string.charAt(0) == '\"' && string.charAt(string.length() - 1) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            lineOfInput.getAssociatedSheet().setComment(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseRemComment(LineOfInput)");
    }

    private static void parseRemDate(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseRemDate(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_REM_DATE.matcher(lineOfInput.getInput());
        if (matcher.find()) {
            int n = Integer.parseInt(matcher.group(2));
            if (n < 1 || n > 9999) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_YEAR);
            }
            lineOfInput.getAssociatedSheet().setYear(n);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseRemDate(LineOfInput)");
    }

    private static void parseRemDiscid(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseRemDiscid(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_REM_DISCID.matcher(lineOfInput.getInput());
        if (matcher.find()) {
            String string = matcher.group(2);
            if (string.charAt(0) == '\"' && string.charAt(string.length() - 1) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            lineOfInput.getAssociatedSheet().setDiscid(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseRemDiscid(LineOfInput)");
    }

    private static void parseRemGenre(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseRemGenre(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_REM_GENRE.matcher(lineOfInput.getInput());
        if (matcher.find()) {
            String string = matcher.group(2);
            if (string.charAt(0) == '\"' && string.charAt(string.length() - 1) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            lineOfInput.getAssociatedSheet().setGenre(string);
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseRemGenre(LineOfInput)");
    }

    private static void parseRem(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseRem(LineOfInput)", lineOfInput);
        if (CueParser.startsWith(lineOfInput, "REM")) {
            String string = lineOfInput.getInput().substring("REM".length()).trim();
            switch (string.charAt(0)) {
                case 'C': 
                case 'c': {
                    if (!CueParser.contains(lineOfInput, PATTERN_REM_COMMENT)) break;
                    CueParser.parseRemComment(lineOfInput);
                    break;
                }
                case 'D': 
                case 'd': {
                    if (CueParser.contains(lineOfInput, PATTERN_REM_DATE)) {
                        CueParser.parseRemDate(lineOfInput);
                        break;
                    }
                    if (!CueParser.contains(lineOfInput, PATTERN_REM_DISCID)) break;
                    CueParser.parseRemDiscid(lineOfInput);
                    break;
                }
                case 'G': 
                case 'g': {
                    if (!CueParser.contains(lineOfInput, PATTERN_REM_GENRE)) break;
                    CueParser.parseRemGenre(lineOfInput);
                }
            }
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseRem(LineOfInput)");
    }

    private static void parseSongwriter(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseSongwriter(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_SONGWRITER.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "SONGWRITER") && matcher.matches()) {
            String string = matcher.group(1);
            if (string.charAt(0) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            if (string.length() > 80) {
                CueParser.addWarning(lineOfInput, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (lineOfInput.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(lineOfInput).getTrackData().size() == 0) {
                if (lineOfInput.getAssociatedSheet().getSongwriter() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                lineOfInput.getAssociatedSheet().setSongwriter(string);
            } else {
                TrackData trackData = CueParser.getLastTrackData(lineOfInput);
                if (trackData.getSongwriter() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setSongwriter(string);
            }
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseSongwriter(LineOfInput)");
    }

    private static void parseTitle(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseTitle(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_TITLE.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "TITLE") && matcher.matches()) {
            String string = matcher.group(1);
            if (string.charAt(0) == '\"') {
                string = string.substring(1, string.length() - 1);
            }
            if (string.length() > 80) {
                CueParser.addWarning(lineOfInput, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (lineOfInput.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(lineOfInput).getTrackData().size() == 0) {
                if (lineOfInput.getAssociatedSheet().getTitle() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                lineOfInput.getAssociatedSheet().setTitle(string);
            } else {
                TrackData trackData = CueParser.getLastTrackData(lineOfInput);
                if (trackData.getTitle() != null) {
                    CueParser.addWarning(lineOfInput, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setTitle(string);
            }
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseTitle(LineOfInput)");
    }

    private static void parseTrack(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "parseTrack(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_TRACK.matcher(lineOfInput.getInput());
        if (CueParser.startsWith(lineOfInput, "TRACK") && matcher.matches()) {
            List<TrackData> list;
            if (matcher.group(1).length() != 2) {
                CueParser.addWarning(lineOfInput, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            int n = Integer.parseInt(matcher.group(1));
            String string = matcher.group(2);
            if (!COMPLIANT_DATA_TYPES.contains(string)) {
                CueParser.addWarning(lineOfInput, WARNING_NONCOMPLIANT_DATA_TYPE);
            }
            if ((list = lineOfInput.getAssociatedSheet().getAllTrackData()).isEmpty() && n != 1 || !list.isEmpty() && list.get(list.size() - 1).getNumber() != n - 1) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_TRACK_NUMBER);
            }
            FileData fileData = CueParser.getLastFileData(lineOfInput);
            fileData.getTrackData().add(new TrackData(fileData, n, string));
        } else {
            CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "parseTrack(LineOfInput)");
    }

    private static Position parsePosition(LineOfInput lineOfInput, String string) {
        logger.entering(CueParser.class.getCanonicalName(), "parsePosition(LineOfInput)", lineOfInput);
        Matcher matcher = PATTERN_POSITION.matcher(string);
        if (matcher.matches()) {
            String string2 = matcher.group(1);
            String string3 = matcher.group(2);
            String string4 = matcher.group(3);
            int n = Integer.parseInt(string2);
            int n2 = Integer.parseInt(string3);
            int n3 = Integer.parseInt(string4);
            if (string2.length() != 2 || string3.length() != 2 || string4.length() != 2) {
                CueParser.addWarning(lineOfInput, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            if (n2 > 59) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_SECONDS_VALUE);
            }
            if (n3 > 74) {
                CueParser.addWarning(lineOfInput, WARNING_INVALID_FRAMES_VALUE);
            }
            Position position = new Position(n, n2, n3);
            logger.exiting(CueParser.class.getCanonicalName(), "parsePosition(LineOfInput)", position);
            return position;
        }
        CueParser.addWarning(lineOfInput, WARNING_UNPARSEABLE_INPUT);
        Position position = new Position();
        logger.exiting(CueParser.class.getCanonicalName(), "parsePosition(LineOfInput)", position);
        return position;
    }

    private static TrackData getLastTrackData(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "getLastTrackData(LineOfInput)", lineOfInput);
        FileData fileData = CueParser.getLastFileData(lineOfInput);
        List<TrackData> list = fileData.getTrackData();
        if (list.size() == 0) {
            list.add(new TrackData(fileData));
            CueParser.addWarning(lineOfInput, WARNING_NO_TRACK_SPECIFIED);
        }
        TrackData trackData = list.get(list.size() - 1);
        logger.exiting(CueParser.class.getCanonicalName(), "getLastTrackData(LineOfInput)", trackData);
        return trackData;
    }

    private static FileData getLastFileData(LineOfInput lineOfInput) {
        logger.entering(CueParser.class.getCanonicalName(), "getLastFileData(LineOfInput)", lineOfInput);
        List<FileData> list = lineOfInput.getAssociatedSheet().getFileData();
        if (list.size() == 0) {
            list.add(new FileData(lineOfInput.getAssociatedSheet()));
            CueParser.addWarning(lineOfInput, WARNING_NO_FILE_SPECIFIED);
        }
        FileData fileData = list.get(list.size() - 1);
        logger.exiting(CueParser.class.getCanonicalName(), "getLastFileData(LineOfInput)", fileData);
        return fileData;
    }

    private static void addWarning(LineOfInput lineOfInput, String string) {
        logger.warning(string);
        lineOfInput.getAssociatedSheet().addWarning(lineOfInput, string);
    }

    public static void main(String[] stringArray) {
        logger.entering(CueParser.class.getCanonicalName(), "main(String[])", stringArray);
        CueSheet cueSheet = null;
        try {
            CueSheetToXmlSerializer cueSheetToXmlSerializer = new CueSheetToXmlSerializer();
            FileFilter fileFilter = new FileFilter(){

                public boolean accept(File file) {
                    return file.getName().length() >= 4 && file.getName().substring(file.getName().length() - 4).equalsIgnoreCase(".cue");
                }
            };
            ArrayList<File> arrayList = new ArrayList<File>();
            File[] fileArray = null;
            File file = new File(System.getProperty("user.dir"));
            fileArray = file.listFiles(fileFilter);
            if (fileArray != null) {
                arrayList.addAll(Arrays.asList(fileArray));
            }
            for (File file2 : arrayList) {
                logger.info("Processing file: '" + file2.toString() + "'");
                cueSheet = CueParser.parse(file2);
                for (Message message : cueSheet.getMessages()) {
                    System.out.println(message);
                }
                System.out.println(new CueSheetSerializer().serializeCueSheet(cueSheet));
                cueSheetToXmlSerializer.serializeCueSheet(cueSheet, System.out);
            }
        }
        catch (Exception exception) {
            LogUtil.logStacktrace(logger, Level.SEVERE, exception);
        }
        logger.exiting(CueParser.class.getCanonicalName(), "main(String[])");
    }
}

