/*
*  qm_settings.cpp
*  QUIMUP settings window
*  © 2008-2024 Johan Spee
*  SPDX-License-Identifier: GPL-3.0-or-later
*/

#include "qm_settings.h"
#include "qm_modes_ids.h"


qm_Settings::qm_Settings(qm_Config *cfg)
{
    if (objectName().isEmpty())
        setObjectName("quimup settings");

    config = cfg;
    mpd_cmd = nullptr;
    b_mpd_connected = false;
    b_profile_readonly = false;
    dev1_isChecked = false;
    dev2_isChecked = false;
    dev3_isChecked = false;
    dev4_isChecked = false;
    dev5_isChecked = false;
    b_new_custom_colors = false;
    b_wayland = config->this_is_wayland;

    set_themed_icons(config->set_dark_theme);
    setup_widgets();

    lb_version->setText("<b>Quimup " + config->version + "</b>");

    retranslate();

    init_vars();
}


void qm_Settings::setup_widgets()
{
    if (objectName().isEmpty())
        setObjectName("settings_window");

    setWindowIcon(QPixmap(":/qm_main_icon"));
    setWindowTitle(tr("Quimup Settings") );

    this->setContentsMargins(0,0,0,0);
    main_widget = new QWidget(this);
    vbox_all = new QVBoxLayout(main_widget);
    vbox_all->setContentsMargins(8,8,8,8);
    setCentralWidget(main_widget);

    tab_widget = new QTabWidget();
    tab_widget->setContentsMargins(0,0,0,0);
    vbox_all->addWidget(tab_widget);
    hbox_buttons = new QHBoxLayout();
    hbox_buttons->setSpacing(8);
    hbox_buttons->setContentsMargins(4,4,4,4);
    hbox_buttons->setAlignment(Qt::AlignRight);
    bt_apply = new QPushButton();
    bt_apply->setFixedSize(QSize(120, 28));
    bt_apply->setAutoDefault(false);
    bt_apply->setDefault(false);
    bt_apply->setIcon(ic_br_check);
    hbox_buttons->addWidget(bt_apply);
    bt_close = new QPushButton();
    bt_close->setFixedSize(QSize(120, 28));
    bt_close->setAutoDefault(false);
    bt_close->setDefault(false);
    bt_close->setIcon(ic_br_reddelete);
    hbox_buttons->addWidget(bt_close);
    vbox_all->addLayout(hbox_buttons);

    // TAB CONNECT
    tab_con = new QWidget();
    vbox_tab_connect = new QVBoxLayout(tab_con);
    vbox_tab_connect->setSpacing(8);
    vbox_tab_connect->setContentsMargins(6,6,6,6);
    gbox_connect = new QGroupBox();
    vbox_timing = new QVBoxLayout(gbox_connect);
    vbox_timing->setContentsMargins(6,6,6,6);
    vbox_timing->setSpacing(4);
    cb_auto_conn = new QCheckBox();
    cb_auto_Re_conn = new QCheckBox();
    vbox_timing->addWidget(cb_auto_conn);
    vbox_timing->addWidget(cb_auto_Re_conn);

    hbox_pollfreq = new QHBoxLayout();
    hbox_pollfreq->setSpacing(8);
    sb_pollfreq = new QSpinBox();
    sb_pollfreq->setMaximum(10000);
    sb_pollfreq->setMinimum(100);
    sb_pollfreq->setSingleStep(100);
    sb_pollfreq->setMaximumSize(QSize(64, 26));
    lb_pollfreq = new QLabel();
    lb_pollfreq->setTextFormat(Qt::RichText);
    hbox_pollfreq->addWidget(sb_pollfreq);
    hbox_pollfreq->addWidget(lb_pollfreq);
    vbox_timing->addLayout(hbox_pollfreq);

    hbox_timeout = new QHBoxLayout();
    hbox_timeout->setSpacing(8);
    sb_timeout = new QSpinBox();
    sb_timeout->setMaximum(30);
    sb_timeout->setMinimum(1);
    sb_timeout->setSingleStep(1);
    sb_timeout->setMaximumSize(QSize(46, 26));
    lb_timeout = new QLabel();
    lb_timeout->setTextFormat(Qt::RichText);
    hbox_timeout->addWidget(sb_timeout);
    hbox_timeout->addWidget(lb_timeout);
    vbox_timing->addLayout(hbox_timeout);

    vbox_tab_connect->addWidget(gbox_connect);
    gbox_profiles = new QGroupBox();
    gbox_profiles->setContentsMargins(0,0,0,0);
    grid_connect = new QGridLayout(gbox_profiles);
    grid_connect->setSpacing(4);
    grid_connect->setContentsMargins(6,6,6,6);
    lb_profiles = new QLabel();
    grid_connect->addWidget(lb_profiles, 0, 0);
    cm_profiles = new QComboBox();
    cm_profiles->insertItems(0, QStringList() << "0" << "1" << "2" << "3" << "4");
    grid_connect->addWidget(cm_profiles, 0, 1);
    lb_profilename = new QLabel();
    grid_connect->addWidget(lb_profilename, 1, 0);
    le_note_text = new QLineEdit();
    grid_connect->addWidget(le_note_text, 1, 1);
    lb_host = new QLabel();
    grid_connect->addWidget(lb_host, 2, 0);
    le_host_text = new QLineEdit();
    grid_connect->addWidget(le_host_text, 2, 1);
    lb_port = new QLabel();
    grid_connect->addWidget(lb_port, 3, 0);
    le_port_text = new QLineEdit();
    grid_connect->addWidget(le_port_text, 3, 1);
    lb_passwd = new QLabel();
    grid_connect->addWidget(lb_passwd, 4, 0);
    le_pswd_text = new QLineEdit();
    //le_pswd_text->setEchoMode(QLineEdit::PasswordEchoOnEdit);
    grid_connect->addWidget(le_pswd_text, 4, 1);
    vbox_tab_connect->addWidget(gbox_profiles);
    lb_musicdir = new QLabel();
    grid_connect->addWidget(lb_musicdir, 5, 0);
    le_mdir_text = new QLineEdit();
    grid_connect->addWidget(le_mdir_text, 5, 1);
    bt_connect = new QPushButton();
    bt_connect->setMinimumSize(QSize(180, 32));
    bt_connect->setIcon(ic_bt_connect);
    grid_connect->addWidget(bt_connect, 6, 1);
    spacer_tab_con =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_tab_connect->addItem(spacer_tab_con);
    gbox_status = new QGroupBox;
    grid_connect_status = new QGridLayout(gbox_status);
    grid_connect_status->setSpacing(4);
    grid_connect_status->setContentsMargins(6,6,6,6);
    lb_conn_icon = new QLabel();
    lb_conn_icon->setScaledContents(true);
    lb_conn_icon->setFixedSize(QSize(32, 32));
    lb_conn_icon->setPixmap(pm_disconnected);
    lb_conn_icon->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    grid_connect_status->addWidget(lb_conn_icon, 0, 0);
    lb_conn_status_text = new QLabel();
    lb_conn_status_text->setTextFormat(Qt::RichText);
    lb_conn_status_text->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    lb_conn_status_text->setText(tr("Not connected"));
    grid_connect_status->addWidget(lb_conn_status_text, 0, 1);
    lb_musicdir_icon = new QLabel();
    lb_musicdir_icon->setFixedSize(QSize(32, 32));
    lb_musicdir_icon->setScaledContents(true);
    lb_musicdir_icon->setPixmap(pm_musicdir_nocon);
    grid_connect_status->addWidget(lb_musicdir_icon, 1, 0);
    lb_musicdir_status_text = new QLabel();
    lb_musicdir_status_text->setTextFormat(Qt::RichText);
    lb_musicdir_status_text->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    lb_musicdir_status_text->setText(tr("Music directory was not determined"));
    grid_connect_status->addWidget(lb_musicdir_status_text, 1, 1);
    lb_owner_icon = new QLabel();
    lb_owner_icon->setFixedSize(QSize(32, 32));
    lb_owner_icon->setScaledContents(true);
    lb_owner_icon->setPixmap(pm_owner_user);
    grid_connect_status->addWidget(lb_owner_icon, 2, 0);
    lb_owner_status_text = new QLabel();
    lb_owner_status_text->setTextFormat(Qt::RichText);
    lb_owner_status_text->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    lb_owner_status_text->setText(tr("MPD is not running"));
    grid_connect_status->addWidget(lb_owner_status_text, 2, 1);
    vbox_tab_connect->addWidget(gbox_status);
    tab_widget->addTab(tab_con, "");

    // TAB SERVER
    tab_server = new QWidget();
    vbox_tab_server = new QVBoxLayout(tab_server);
    vbox_tab_server->setSpacing(8);
    vbox_tab_server->setContentsMargins(6,6,6,6);
    gbox_xfade = new QGroupBox();
    gbox_xfade->setEnabled(false);
    hbox_xfade = new QHBoxLayout(gbox_xfade);
    hbox_xfade->setSpacing(8);
    sb_xfade = new QSpinBox();
    sb_xfade->setMaximumSize(QSize(46, 26));
    sb_xfade->setMaximum(30);
    sb_xfade->setMinimum(0);
    sb_xfade->setSingleStep(1);
    hbox_xfade->addWidget(sb_xfade);
    lb_xfade = new QLabel();
    hbox_xfade->addWidget(lb_xfade);
    vbox_tab_server->addWidget(gbox_xfade);
    gbox_replaygain = new QGroupBox();
    gbox_replaygain->setEnabled(false);
    hbox_replaygain = new QHBoxLayout(gbox_replaygain);
    hbox_replaygain->setSpacing(8);
    hbox_replaygain->setContentsMargins(6,6,6,6);
    hbox_replaygain->setAlignment(Qt::AlignLeft);
    rb_rpg_off = new QRadioButton();
    hbox_replaygain->addWidget(rb_rpg_off);
    rb_rpg_track = new QRadioButton();
    hbox_replaygain->addWidget(rb_rpg_track);
    rb_rpg_album = new QRadioButton();
    hbox_replaygain->addWidget(rb_rpg_album);
    rb_rpg_auto = new QRadioButton();
    hbox_replaygain->addWidget(rb_rpg_auto);
    vbox_tab_server->addWidget(gbox_replaygain);
    gbox_outputs = new QGroupBox();
    gbox_outputs->setEnabled(false);
    vbox_devices = new QVBoxLayout(gbox_outputs);
    vbox_devices->setSpacing(4);
    vbox_devices->setContentsMargins(6,6,6,6);
    cb_dev1 = new QCheckBox();
    vbox_devices->addWidget(cb_dev1);
    cb_dev2 = new QCheckBox();
    vbox_devices->addWidget(cb_dev2);
    cb_dev3 = new QCheckBox();
    vbox_devices->addWidget(cb_dev3);
    cb_dev4 = new QCheckBox();
    vbox_devices->addWidget(cb_dev4);
    cb_dev5 = new QCheckBox();
    vbox_devices->addWidget(cb_dev5);
    vbox_tab_server->addWidget(gbox_outputs);
    lb_mpdconnected = new QLabel();
    lb_mpdconnected->setAlignment(Qt::AlignRight);
    lb_mpdconnected->setTextFormat(Qt::RichText);
    lb_mpdconnected->setText("<small>" + tr ("Not connected to MPD") + "  </small>");
    vbox_tab_server->addWidget(lb_mpdconnected);
    spacer_tab_server =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_tab_server->addItem(spacer_tab_server);
    gbox_manage_mpd = new QGroupBox();
    grid_manage = new QGridLayout(gbox_manage_mpd);
    cb_startserver = new QCheckBox();
    grid_manage->addWidget(cb_startserver,0,0);
    grid_manage->setSpacing(4);
    grid_manage->setContentsMargins(6,6,6,6);
    vbox_startmpd = new QVBoxLayout();
    vbox_startmpd->setSpacing(4);
    vbox_startmpd->setContentsMargins(2,2,2,2);
    lb_startcomm = new QLabel();
    lb_startcomm->setTextFormat(Qt::RichText);
    vbox_startmpd->addWidget(lb_startcomm);
    le_startcommand = new QLineEdit();
    le_startcommand->setPlaceholderText("systemctl --user start mpd");
    vbox_startmpd->addWidget(le_startcommand);
    grid_manage->addLayout(vbox_startmpd,0,1);
    cb_killserver = new QCheckBox();
    grid_manage->addWidget(cb_killserver,1,0);
    vbox_killmpd = new QVBoxLayout();
    vbox_killmpd->setSpacing(4);
    vbox_killmpd->setContentsMargins(2,2,2,2);
    lb_killcomm = new QLabel();
    lb_killcomm->setTextFormat(Qt::RichText);
    vbox_killmpd->addWidget(lb_killcomm);
    le_killcommand = new QLineEdit();
    le_killcommand->setPlaceholderText("systemctl --user stop mpd");
    vbox_killmpd->addWidget(le_killcommand);
    grid_manage->addLayout(vbox_killmpd,1,1);
    vbox_tab_server->addWidget(gbox_manage_mpd);
    lb_mpdinstalled = new QLabel();
    lb_mpdinstalled->setAlignment(Qt::AlignRight);
    lb_mpdinstalled->setTextFormat(Qt::RichText);
    vbox_tab_server->addWidget(lb_mpdinstalled);
    tab_widget->addTab(tab_server, "");

    // TAB CLIENT
    tab_client = new QWidget();
    vbox_tab_client = new QVBoxLayout(tab_client);
    vbox_tab_client->setSpacing(8);
    vbox_tab_client->setContentsMargins(6,6,6,6);
    // General options
    gbox_general = new QGroupBox();
    vbox_general = new QVBoxLayout(gbox_general);
    vbox_general->setSpacing(4);
    vbox_general->setContentsMargins(6,6,6,6);
    cb_use_systray = new QCheckBox();
    vbox_general->addWidget(cb_use_systray);
    cb_start_minimized = new QCheckBox();
    vbox_general->addWidget(cb_start_minimized);
    cb_showTips = new QCheckBox();
    vbox_general->addWidget(cb_showTips);
    vbox_tab_client->addWidget(gbox_general);
    // library
    gbox_library = new QGroupBox();
    vbox_library = new QVBoxLayout(gbox_library);
    vbox_library->setSpacing(4);
    vbox_library->setContentsMargins(6,6,6,6);
    cb_sortbyyear = new QCheckBox();
    vbox_library->addWidget(cb_sortbyyear);
    cb_ignore_the = new QCheckBox();
    vbox_library->addWidget(cb_ignore_the);
    cb_library_autocols = new QCheckBox();
    vbox_library->addWidget(cb_library_autocols);
    vbox_tab_client->addWidget(gbox_library);
    // playlist
    gbox_playlist =  new QGroupBox();
    vbox_playlist = new QVBoxLayout(gbox_playlist);
    vbox_playlist->setSpacing(4);
    vbox_playlist->setContentsMargins(6,6,6,6);
    cb_auto_playfirst = new QCheckBox();
    vbox_playlist->addWidget(cb_auto_playfirst);
    cb_mark_whenplayed = new QCheckBox();
    vbox_playlist->addWidget(cb_mark_whenplayed);
    cb_playlist_autocols = new QCheckBox();
    vbox_playlist->addWidget(cb_playlist_autocols);
    vbox_tab_client->addWidget(gbox_playlist);
    //albumart
    gbox_albumart = new QGroupBox();
    vbox_art_size = new QVBoxLayout(gbox_albumart);
    vbox_art_size->setContentsMargins(6,6,6,6);
    vbox_art_size->setSpacing(4);
    hbox_art_size = new QHBoxLayout();
    hbox_art_size->setSpacing(8);
    sb_art_size = new QSpinBox();
    sb_art_size->setMaximumSize(QSize(58, 26));
    sb_art_size->setMaximum(400);
    sb_art_size->setMinimum(100);
    sb_art_size->setSingleStep(10);
    hbox_art_size->addWidget(sb_art_size);
    lb_art_size = new QLabel();
    hbox_art_size->addWidget(lb_art_size);
    vbox_art_size->addLayout(hbox_art_size);
    cb_custom_art = new QCheckBox();
    vbox_art_size->addWidget(cb_custom_art);
    le_custom_art = new QLineEdit();
    le_custom_art->setPlaceholderText("/home/user/../image");
    vbox_art_size->addWidget(le_custom_art);
    vbox_tab_client->addWidget(gbox_albumart);
    spacer_tab_client =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_tab_client->addItem(spacer_tab_client);
    tab_widget->addTab(tab_client, "");

    // TAB STYLE
    tab_style = new QWidget();
    vbox_tab_style = new QVBoxLayout(tab_style);
    vbox_tab_style->setSpacing(8);
    vbox_tab_style->setContentsMargins(6,6,6,6);
    //fonts
    gbox_fonts = new QGroupBox();
    grid_fonts = new QGridLayout(gbox_fonts);
    grid_fonts->setSpacing(4);
    grid_fonts->setContentsMargins(6,6,6,6);
    sb_titlefont = new QSpinBox();
    sb_titlefont->setMaximumSize(QSize(46, 26));
    sb_titlefont->setMaximum(16);
    sb_titlefont->setMinimum(6);
    sb_titlefont->setSingleStep(1);
    grid_fonts->addWidget(sb_titlefont,0,0);
    lb_titlefont = new QLabel();
    lb_titlefont->setTextFormat(Qt::RichText);
    grid_fonts->addWidget(lb_titlefont,0,1);
    hbox_title_b_i = new QHBoxLayout();
    hbox_title_b_i->setSpacing(8);
    cb_title_bold = new QCheckBox();
    hbox_title_b_i->addWidget(cb_title_bold);
    cb_title_italic = new QCheckBox();
    hbox_title_b_i->addWidget(cb_title_italic);
    grid_fonts->addLayout (hbox_title_b_i, 0, 2);
    sb_codecfont = new QSpinBox();
    sb_codecfont->setMaximumSize(QSize(46, 26));
    sb_codecfont->setMaximum(16);
    sb_codecfont->setMinimum(6);
    sb_codecfont->setSingleStep(1);
    grid_fonts->addWidget(sb_codecfont,1,0);
    lb_codecfont = new QLabel();
    grid_fonts->addWidget(lb_codecfont,1,1);
    cb_codec_bold = new QCheckBox();
    grid_fonts->addWidget(cb_codec_bold,1,2);
    sb_timefont = new QSpinBox();
    sb_timefont->setMaximumSize(QSize(46, 26));
    sb_timefont->setMaximum(16);
    sb_timefont->setMinimum(6);
    sb_timefont->setSingleStep(1);
    grid_fonts->addWidget(sb_timefont,2,0);
    lb_timefont = new QLabel();
    grid_fonts->addWidget(lb_timefont,2,1);
    cb_time_bold = new QCheckBox();
    grid_fonts->addWidget(cb_time_bold,2,2);
    sb_albumfont = new QSpinBox();
    sb_albumfont->setMaximumSize(QSize(46, 26));
    sb_albumfont->setMaximum(16);
    sb_albumfont->setMinimum(6);
    sb_albumfont->setSingleStep(1);
    grid_fonts->addWidget(sb_albumfont,3,0);
    lb_albumfont = new QLabel();
    grid_fonts->addWidget(lb_albumfont,3,1);
    cb_album_bold= new QCheckBox();
    grid_fonts->addWidget(cb_album_bold,3,2);
    sb_yearfont = new QSpinBox();
    sb_yearfont->setMaximumSize(QSize(46, 26));
    sb_yearfont->setMaximum(16);
    sb_yearfont->setMinimum(6);
    sb_yearfont->setSingleStep(1);
    grid_fonts->addWidget(sb_yearfont,4,0);
    lb_yearfont = new QLabel();
    grid_fonts->addWidget(lb_yearfont,4,1);
    cb_year_bold = new QCheckBox();
    grid_fonts->addWidget(cb_year_bold,4,2);
    sb_commentfont = new QSpinBox();
    sb_commentfont->setMaximumSize(QSize(46, 26));
    sb_commentfont->setMaximum(16);
    sb_commentfont->setMinimum(6);
    sb_commentfont->setSingleStep(1);
    grid_fonts->addWidget(sb_commentfont,5,0);
    lb_commentfont = new QLabel();
    grid_fonts->addWidget(lb_commentfont,5,1);
    cb_comment_italic = new QCheckBox();
    grid_fonts->addWidget(cb_comment_italic,5,2);
    sb_liblistfont = new QSpinBox();
    sb_liblistfont->setMaximumSize(QSize(46, 26));
    sb_liblistfont->setMaximum(16);
    sb_liblistfont->setMinimum(6);
    sb_liblistfont->setSingleStep(1);
    grid_fonts->addWidget(sb_liblistfont,6,0);
    lb_liblistfont = new QLabel();
    grid_fonts->addWidget(lb_liblistfont,6,1);
    bt_fonts_reset = new QPushButton();
    bt_fonts_reset->setMinimumSize(QSize(120, 30));
    bt_fonts_reset->setMaximumSize(QSize(180, 30));
    grid_fonts->addWidget(bt_fonts_reset,6,2);
    vbox_tab_style->addWidget(gbox_fonts);
    // colors
    gbox_colors = new QGroupBox();
    hbox_colors = new QHBoxLayout(gbox_colors);
    hbox_colors->setSpacing(12);
    hbox_colors->setContentsMargins(6,6,6,6);
    vbox_colors_left = new QVBoxLayout();
    vbox_colors_left->setAlignment(Qt::AlignBottom);
    vbox_colors_left->setSpacing(4);
    vbox_colors_left->setContentsMargins(0,0,0,0);
    rb_col_syst = new QRadioButton();
    vbox_colors_left->addWidget(rb_col_syst);
    rb_col_def = new QRadioButton();
    vbox_colors_left->addWidget(rb_col_def);
    rb_col_cust = new QRadioButton();
    vbox_colors_left->addWidget(rb_col_cust);
    spacer_l = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_colors_left->addItem(spacer_l);
    hbox_colors->addLayout(vbox_colors_left);
    vbox_colors_right = new QVBoxLayout();
    vbox_colors_right->setAlignment(Qt::AlignBottom);
    vbox_colors_right->setSpacing(4);
    vbox_colors_right->setContentsMargins(0,0,0,0);
    cb_colfor_browser = new QCheckBox();
    vbox_colors_left->addWidget(cb_colfor_browser);
    cb_colfor_albuminfo = new QCheckBox();
    vbox_colors_left->addWidget(cb_colfor_albuminfo);
    cb_colfor_pgsbar = new QCheckBox();
    vbox_colors_left->addWidget(cb_colfor_pgsbar);
    spacer_r = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_colors_right->addItem(spacer_r);
    vbox_custcolor_show  = new QVBoxLayout();
    vbox_custcolor_show->setContentsMargins(0,0,0,0);
    vbox_custcolor_show->setSpacing(0);
    lb_custcolor_title = new QLabel();
    lb_custcolor_title->setTextFormat(Qt::RichText);
    lb_custcolor_title->setFrameShape(QFrame::StyledPanel);
    lb_custcolor_title->setFrameShadow(QFrame::Sunken);
    lb_custcolor_title->setAlignment(Qt::AlignCenter);
    lb_custcolor_title->setTextFormat(Qt::RichText);
    lb_custcolor_title->setMinimumSize(QSize(120, 30));
    lb_custcolor_title->setMaximumSize(QSize(180, 30));
    lb_custcolor_title->setAutoFillBackground(true);
    vbox_custcolor_show->addWidget(lb_custcolor_title);
    lb_custcolor_album = new QLabel();
    lb_custcolor_album->setTextFormat(Qt::RichText);
    lb_custcolor_album->setFrameShape(QFrame::StyledPanel);
    lb_custcolor_album->setFrameShadow(QFrame::Sunken);
    lb_custcolor_album->setAlignment(Qt::AlignCenter);
    lb_custcolor_album->setTextFormat(Qt::RichText);
    lb_custcolor_album->setMinimumSize(QSize(120, 30));
    lb_custcolor_album->setMaximumSize(QSize(180, 30));
    lb_custcolor_album->setAutoFillBackground(true);
    vbox_custcolor_show->addWidget(lb_custcolor_album);
    lb_custcolor_show_text = new QLabel();
    lb_custcolor_show_text->setTextFormat(Qt::RichText);
    vbox_colors_right->addWidget(lb_custcolor_show_text);
    vbox_colors_right->addLayout(vbox_custcolor_show);
    bt_cust_color = new QPushButton();
    bt_cust_color->setMinimumSize(QSize(120, 30));
    bt_cust_color->setMaximumSize(QSize(180, 30));
    vbox_colors_right->addWidget(bt_cust_color);
    hbox_colors->addLayout(vbox_colors_right);
    vbox_tab_style->addWidget(gbox_colors);
    // theme
    gbox_theme = new QGroupBox();
    hbox_dark_theme = new QHBoxLayout(gbox_theme);
    hbox_dark_theme->setSpacing(8);
    hbox_dark_theme->setContentsMargins(6,6,6,6);
    cb_dark_icons = new QCheckBox();
    hbox_dark_theme->addWidget(cb_dark_icons);
    vbox_tab_style->addWidget(gbox_theme);
    spacer_tab_style =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_tab_style->addItem(spacer_tab_style);
    tab_widget->addTab(tab_style, "");

    // TAB ACTIONS
    tab_actions = new QWidget();
    vbox_tab_actions = new QVBoxLayout(tab_actions);
    vbox_tab_actions->setSpacing(8);
    vbox_tab_actions->setContentsMargins(6,6,6,6);
    // external applications
    gbox_external_actions = new QGroupBox();
    grid_external_actions = new QGridLayout(gbox_external_actions);
    grid_external_actions->setSpacing(4);
    grid_external_actions->setContentsMargins(6,6,6,6);

    lb_imageviewer = new QLabel();
    grid_external_actions->addWidget(lb_imageviewer, 0, 0);
    le_imageviewer = new QLineEdit();
    le_imageviewer->setPlaceholderText("qview, display");
    grid_external_actions->addWidget(le_imageviewer, 0, 1);

    lb_tageditor = new QLabel();
    grid_external_actions->addWidget(lb_tageditor, 1, 0);
    le_tageditor = new QLineEdit();
    le_tageditor->setPlaceholderText("kid3, easytag, picard");
    grid_external_actions->addWidget(le_tageditor, 1, 1);

    lb_filemanager = new QLabel();
    grid_external_actions->addWidget(lb_filemanager, 2, 0);
    le_filemanager = new QLineEdit();
    le_filemanager->setPlaceholderText("dolphin, nautilus, thunar");
    grid_external_actions->addWidget(le_filemanager, 2, 1);

    vbox_tab_actions->addWidget(gbox_external_actions);

    // user action 1
    gbox_usract1 = new QGroupBox();
    grid_usract1 = new QGridLayout(gbox_usract1);
    grid_usract1->setVerticalSpacing(4);
    grid_usract1->setHorizontalSpacing(8);
    grid_usract1->setContentsMargins(6,6,6,6);
    cb_usract1 = new QCheckBox;
    grid_usract1->addWidget(cb_usract1, 0, 0);
    hbox_rb_usract1 = new QHBoxLayout();
    spacer_usract1 =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Fixed);
    hbox_rb_usract1->addItem(spacer_usract1);
    rb_file_usract1 = new QRadioButton();
    rb_dir_usract1 = new QRadioButton();
    hbox_rb_usract1->addWidget(rb_file_usract1);
    hbox_rb_usract1->addWidget(rb_dir_usract1);
    grid_usract1->addLayout(hbox_rb_usract1, 0, 1);
    lb_name_usract1 = new QLabel();
    grid_usract1->addWidget(lb_name_usract1, 1, 0);
    le_label_usract1 = new QLineEdit();
    le_label_usract1->setPlaceholderText("mp3 gain, delete...");
    grid_usract1->addWidget(le_label_usract1, 1, 1);
    lb_cmd_usract1 = new QLabel();
    grid_usract1->addWidget(lb_cmd_usract1, 2, 0);
    le_cmd_usract1 = new QLineEdit();
    le_cmd_usract1->setPlaceholderText("konsole -e /home/user/...script.sh");
    grid_usract1->addWidget(le_cmd_usract1, 2, 1);
    vbox_tab_actions->addWidget(gbox_usract1);

    // user action 2
    gbox_usract2 = new QGroupBox();
    grid_usract2 = new QGridLayout(gbox_usract2);
    grid_usract2->setVerticalSpacing(4);
    grid_usract2->setHorizontalSpacing(8);
    grid_usract2->setContentsMargins(6,6,6,6);
    cb_usract2 = new QCheckBox;
    grid_usract2->addWidget(cb_usract2, 0, 0);
    hbox_rb_usract2 = new QHBoxLayout();
    spacer_usract2 =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Fixed);
    hbox_rb_usract2->addItem(spacer_usract2);
    rb_file_usract2 = new QRadioButton();
    rb_dir_usract2 = new QRadioButton();
    hbox_rb_usract2->addWidget(rb_file_usract2);
    hbox_rb_usract2->addWidget(rb_dir_usract2);
    grid_usract2->addLayout(hbox_rb_usract2, 0, 1);
    lb_name_usract2 = new QLabel();
    grid_usract2->addWidget(lb_name_usract2, 1, 0);
    le_label_usract2 = new QLineEdit();
    le_label_usract2->setPlaceholderText("mp3 gain, delete...");
    grid_usract2->addWidget(le_label_usract2, 1, 1);
    lb_cmd_usract2 = new QLabel();
    grid_usract2->addWidget(lb_cmd_usract2, 2, 0);
    le_cmd_usract2 = new QLineEdit();
    le_cmd_usract2->setPlaceholderText("konsole -e /home/user/...script.sh");
    grid_usract2->addWidget(le_cmd_usract2, 2, 1);
    vbox_tab_actions->addWidget(gbox_usract2);

    spacer_tab_actions =  new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    vbox_tab_actions->addItem(spacer_tab_actions);
    tab_widget->addTab(tab_actions, "");

    // TAB ABOUT
    tb_about = new QWidget();
    vbox_tab_about = new QVBoxLayout(tb_about);
    vbox_tab_about->setSpacing(8);
    vbox_tab_about->setContentsMargins(6,6,6,6);
    hbox_about_top = new QHBoxLayout();
    hbox_about_top->setSpacing(20);
    hbox_about_top->setContentsMargins(20,6,6,6);
    px_about = new QLabel();
    px_about->setFixedSize(32, 32);
    px_about->setScaledContents(true);
    px_about->setPixmap(QPixmap(":/qm_main_icon"));
    hbox_about_top->addWidget(px_about);
    lb_version = new QLabel();
    lb_version->setTextFormat(Qt::RichText);
    lb_version->setText("<b>Quimup</b>");
    hbox_about_top->addWidget(lb_version);
    vbox_tab_about->addLayout(hbox_about_top);
    sa_about = new QScrollArea;
    sa_about->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    sa_about->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    sa_about->setLineWidth(0);
    sa_about->setWidgetResizable(true);
    lb_about_info = new QLabel();
    lb_about_info->setTextFormat(Qt::RichText);
    lb_about_info->setContentsMargins(6,6,6,6);
    lb_about_info->setWordWrap(true);
    lb_about_info->setAlignment(Qt::AlignLeft|Qt::AlignTop);
    sa_about->setWidget(lb_about_info);
    vbox_tab_about->addWidget(sa_about);
    tab_widget->addTab(tb_about, "");

    colorpicker = new qm_colorPicker(nullptr, config);
    connect( bt_close, SIGNAL(clicked()), this, SLOT(cancel_changes()) );
    connect( bt_apply, SIGNAL(clicked()), this, SLOT(apply_settings()) );
    connect( cm_profiles, SIGNAL(currentIndexChanged(int)), this, SLOT(set_profile(int)) );
    connect( bt_connect, SIGNAL(clicked()), this, SLOT(connect_to_server()));
    connect( bt_fonts_reset, SIGNAL(clicked()), this, SLOT(reset_fonts()));
    connect( bt_cust_color, SIGNAL(clicked()), this, SLOT(show_colorpicker()));
    connect( colorpicker, SIGNAL(signal_newcolors()), this, SLOT(on_new_customcolors()) );

} // setup_widgets


void qm_Settings::set_profile(int profile)
{
    if ( profile != 0 && b_profile_readonly )
    {
        le_note_text->setReadOnly(false);
        le_port_text->setReadOnly(false);
        le_host_text->setReadOnly(false);
        le_pswd_text->setReadOnly(false);
        le_mdir_text->setReadOnly(false);
        b_profile_readonly = false;
    }

    switch (profile)
    {
        case 0:
        {
            if (!b_profile_readonly)
            {
                b_profile_readonly = true;
                le_note_text->setReadOnly(true);
                le_port_text->setReadOnly(true);
                le_host_text->setReadOnly(true);
                le_pswd_text->setReadOnly(true);
                le_mdir_text->setReadOnly(true);
            }

            le_note_text->setText(config->profile_note_0);

            if (config->profile_note_0 == "mpd.conf not found")
            {
                le_host_text->setText("");
                le_port_text->setText("");
                le_pswd_text->setText("");
                le_mdir_text->setText("");
            }
            else
            {
                le_host_text->setText(config->profile_host_0);
                le_port_text->setText(QString::number(config->profile_port_0));
                le_pswd_text->setText(config->profile_pswd_0);
                le_mdir_text->setText(config->profile_mdir_0);
            }
            break;
        }
        case 1:
        {
            le_note_text->setText(config->profile_note_1);
            le_port_text->setText(QString::number(config->profile_port_1));
            le_host_text->setText(config->profile_host_1);
            le_pswd_text->setText(config->profile_pswd_1);
            le_mdir_text->setText(config->profile_mdir_1);
            break;
        }
        case 2:
        {
            le_note_text->setText(config->profile_note_2);
            le_port_text->setText(QString::number(config->profile_port_2));
            le_host_text->setText(config->profile_host_2);
            le_pswd_text->setText(config->profile_pswd_2);
            le_mdir_text->setText(config->profile_mdir_2);
            break;
        }
        case 3:
        {
            le_note_text->setText(config->profile_note_3);
            le_port_text->setText(QString::number(config->profile_port_3));
            le_host_text->setText(config->profile_host_3);
            le_pswd_text->setText(config->profile_pswd_3);
            le_mdir_text->setText(config->profile_mdir_3);
            break;
        }
        case 4:
        {
            le_note_text->setText(config->profile_note_4);
            le_port_text->setText(QString::number(config->profile_port_4));
            le_host_text->setText(config->profile_host_4);
            le_pswd_text->setText(config->profile_pswd_4);
            le_mdir_text->setText(config->profile_mdir_4);
            break;
        }
        default:
            break;
    }
}


void qm_Settings::retranslate()
{
    if (config->show_tooltips)
    {
        cb_auto_conn->setToolTip(tr("Automatically connect to MPD when starting up") );
        cb_auto_Re_conn->setToolTip(tr("Automatically reconnect to MPD when the connection fails") );
        lb_pollfreq->setToolTip(tr("100 to 10000 msec. Default is 500") +"\n" + tr("Set higher for remote (wifi) connections") );
        lb_timeout->setToolTip(tr("Reconnect if there is no response") +"\n" + tr("Set above 10 for streams") );
        le_note_text->setToolTip(tr("A description of this profile") );
        le_port_text->setToolTip(tr("Higher than 1024, usually 6600") );
        le_host_text->setToolTip("localhost\n/home/USER/.mpd/socket\n/home/USER/.config/mpd/socket\n192.168.178.20\netc.") ;
        le_killcommand->setToolTip("\"mpd --kill\"\n\"systemctl --user stop mpd\"");
        cm_profiles->setToolTip(tr("Connection profiles") );
        le_startcommand->setToolTip("\"mpd\"\n\"systemctl --user start mpd\"");
        bt_cust_color->setToolTip(tr("Select and set custom colors"));
        le_imageviewer->setToolTip(tr("To open currently displayed album art"));
        le_tageditor->setToolTip(tr("To edit artist, title, album, genre, year etc."));
        le_filemanager->setToolTip(tr("To open song directory in file manager"));
        lb_xfade->setToolTip(tr("Fade songs in and out (overlapping)"));
        bt_close->setToolTip(tr("Close this window (does not apply or save the settings)"));
        bt_apply->setToolTip(tr("Apply and save current settings"));
        bt_connect->setToolTip(tr("Connect using this profile\nApply to save it"));
        lb_musicdir_icon->setToolTip(tr("Accessibility status of MPD's music directory"));
        lb_conn_icon->setToolTip(tr("Status of the connection with MPD"));
        le_custom_art->setToolTip(tr("Full path to an image\nTo be used when no album art is found"));
        lb_art_size->setToolTip(tr("Size in pixels of the shown album art\nApply and restart"));
        le_cmd_usract1->setToolTip(tr("Command to be executed") + "\n" +
                                   + "File or directory will be added as parameter");
        le_label_usract1->setToolTip(tr("Label to show in context menu"));
        le_cmd_usract2->setToolTip(tr("Command to be executed") + "\n" +
                                   + "File or directory will be added as parameter");
        le_label_usract2->setToolTip(tr("Label to show in context menu"));
        QString s_outputs = tr("Output, as defined in MPD's configuration");
        cb_dev1->setToolTip(s_outputs);
        cb_dev2->setToolTip(s_outputs);
        cb_dev3->setToolTip(s_outputs);
        cb_dev4->setToolTip(s_outputs);
        cb_dev5->setToolTip(s_outputs);
    }
    bt_close->setText(tr("Close"));
    bt_apply->setText(tr("Apply"));
    bt_connect->setText(tr("Connect now"));
    gbox_connect->setTitle(tr("Timing"));
    lb_profiles->setText(tr("Profile"));
    gbox_profiles->setTitle(tr("Connection profile"));
    lb_profilename->setText(tr("Description"));
    lb_passwd->setText(tr("Password"));
    lb_port->setText(tr("Port"));
    lb_host->setText(tr("Host"));
    lb_musicdir->setText(tr("Music directory"));
    cb_auto_conn->setText(tr("Connect on start-up"));
    cb_auto_Re_conn->setText(tr("Automatically reconnect"));
    lb_timeout->setText(tr("Timeout (sec.)"));
    lb_pollfreq->setText(tr("Interval to poll MPD (msec.)"));
    gbox_status->setTitle(tr("Status"));
    gbox_replaygain->setTitle(tr("Replaygain mode"));
    rb_rpg_off->setText(tr("Off"));
    rb_rpg_track->setText(tr("Track"));
    rb_rpg_album->setText(tr("Album"));
    rb_rpg_auto->setText(tr("Auto"));
    gbox_outputs->setTitle(tr("Ouput devices"));
    gbox_general->setTitle(tr("General"));
    gbox_library->setTitle(tr("Library"));
    gbox_playlist->setTitle(tr("Playlist"));
    gbox_manage_mpd->setTitle(tr("Manage MPD locally"));
    lb_killcomm->setText("<small>" + tr("command") + "</small>");
    le_killcommand->setText("systemctl --user stop mpd");
    cb_startserver->setText(tr("Start with quimup"));
    lb_startcomm->setText("<small>" + tr("command") + "</small>");
    le_startcommand->setText("systemctl --user start mpd");
    cb_killserver->setText(tr("Quit with quimup"));
     if (config->mpd_is_installed)
    {
        lb_mpdinstalled->setText("<small>" + tr("MPD is installed") + "  </small>");
        //gbox_manage_mpd->setEnabled(true);
    }
    else
    {
        lb_mpdinstalled->setText("<small>" + tr("MPD is not installed") + "  </small>");
        //gbox_manage_mpd->setEnabled(false);
    }
    gbox_fonts->setTitle(tr("Fonts"));
    gbox_xfade->setTitle(tr("Cross fade"));
    lb_xfade->setText(tr("Time (sec.)") + "<i> " + tr("Turn on-off in playlist") + "</i>");
    lb_titlefont->setText(tr("Artist : Title"));
    cb_title_bold->setText(tr("bold"));
    cb_title_italic->setText(tr("italic title"));
    lb_codecfont->setText(tr("MP3 kb/s KHz"));
    cb_codec_bold->setText(tr("bold"));
    lb_timefont->setText(tr("Time"));
    cb_time_bold->setText(tr("bold"));
    lb_albumfont->setText(tr("Album"));
    cb_album_bold->setText(tr("bold"));
    lb_yearfont->setText(tr("Year"));
    cb_year_bold->setText(tr("bold"));
    lb_commentfont->setText(tr("Comment"));
    cb_comment_italic->setText(tr("italic"));
    lb_liblistfont->setText(tr("Media Browser"));
    bt_fonts_reset->setText(tr("Reset fonts"));
    gbox_colors->setTitle(tr("Colors"));
    rb_col_syst->setText(tr("System colors"));
    rb_col_def->setText(tr("Quimup colors"));
    rb_col_cust->setText(tr("Custom colors"));
    bt_cust_color->setText(tr("Set custom colors"));
    lb_custcolor_title->setText("<b>" + tr("Artist : Title") + "</b>");
    lb_custcolor_album->setText(tr("album & year"));
    cb_use_systray->setText(tr("Use system tray icon (restart)"));
    cb_start_minimized->setText(tr("Start player minimized"));
    cb_showTips->setText(tr("Show tooltips (restart)"));
    cb_sortbyyear->setText(tr("In artist mode: Sort albums by year"));
    cb_ignore_the->setText(tr("Ignore leading 'The' when sorting artists"));
    cb_library_autocols->setText(tr("Automatic column width"));
    cb_auto_playfirst->setText(tr("Start first song after 'Open in new playlist'"));
    cb_mark_whenplayed->setText(tr("Mark songs that have played"));
    cb_playlist_autocols->setText(tr("Automatic column width"));
    cb_colfor_browser->setText(tr("Apply to browser"));
    cb_colfor_albuminfo->setText(tr("Apply to album info"));
    cb_colfor_pgsbar->setText(tr("Apply to progress bar"));
    lb_custcolor_show_text->setText("<small>" + tr("Custom colors") + "</small>");
    gbox_theme->setTitle(tr("Theme"));
    cb_dark_icons->setText(tr("Use icons and colors for a dark theme (restart)"));
    gbox_albumart->setTitle(tr("Album art"));
    lb_art_size->setText(tr("Album art size (restart)"));
    cb_custom_art->setText(tr("Custom default album art :"));
    gbox_external_actions->setTitle(tr("External apllications"));
    lb_imageviewer->setText(tr("Image viewer"));
    lb_tageditor->setText(tr("Tag editor"));
    lb_filemanager->setText(tr("File manager"));
    gbox_usract1->setTitle(tr("User action"));
    gbox_usract2->setTitle(tr("User action"));
    lb_cmd_usract1->setText(tr("Command"));
    lb_cmd_usract2->setText(tr("Command"));
    lb_name_usract1->setText(tr("Label"));
    lb_name_usract2->setText(tr("Label"));
    cb_usract1->setText(tr("Enable"));
    cb_usract2->setText(tr("Enable"));
    rb_file_usract1->setText(tr("File parameter"));
    rb_dir_usract1->setText(tr("Directory parameter"));
    rb_file_usract2->setText(tr("File parameter"));
    rb_dir_usract2->setText(tr("Directory parameter"));
    tab_widget->setTabText(tab_widget->indexOf(tab_con), tr("Connect") + " ");
    tab_widget->setTabText(tab_widget->indexOf(tab_server), tr("Server") + " ");
    tab_widget->setTabText(tab_widget->indexOf(tab_client), tr("Client") + " ");
    tab_widget->setTabText(tab_widget->indexOf(tab_style), tr("Style") + " ");
    tab_widget->setTabText(tab_widget->indexOf(tab_actions), tr("Actions") + " ");
    tab_widget->setTabText(tab_widget->indexOf(tb_about), tr("About") + " ");
    QString s_tips     = "<div align=\"center\"><br><b>Q</b>t <b>U</b>ser <b>I</b>nterface for the <b>MU</b>sic <b>P</b>layer daemon<br><br>© 2008-2024 Johan Spee<br>oubliette@xs4all.nl<br><br>";
            s_tips    += tr("License")
                       + "</div><div align=\"justify\"><small>"
                       + "This program is free software. You can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but without any warrenty; without even the implied warranty of merchantability or fitness for a particular purpose. See the GNU General Public License for details."
                       + "</small><br></div><div align=\"center\"><b><big>"
                       + tr("TIPS")
                       + "</div></big><hr></b><div align=\"justify\">"
                       + tr("Double-clicking the album art will <b>open the shown image</b> in the mage viewer (e.g. qview) that can be set here, in the 'Actions' tab.")
                       + "<hr>"
                       + tr("Right-clicking the player brings up a menu where you can open files in <b>external applications</b> (e.g. qview, kid3, dolphin), or run user actions (scripts) on files or folders. See the 'Actions' tab.")
                       + "<hr>"
                       + tr("To open files in such programs Quimup must be able to access MPD's <b>music directory</b> (the current status of which is shown in the 'Connect' tab). The directory is read from mpd.conf or can be set in the connection profile. To avoid problems set MPDCONF in /etc/default/mpd to <i>/path/to/your/mpd.conf</i>")
                       + "<hr>"
                       + tr("Quimup first queries MPD for <b>album art</b> (from tag, next from file). You need MPD 0.22, or above, for this. When the album directory is accessible it next searches there for a bitmap with a name starting with: ")
                       + " <i>cover, albumart, folder, front, booklet, portada, couverture, capa</i>,  "
                       + tr("or a name that contains the album name. in that order, case insensitive).")
                       + "<hr>"
                       + tr("The <b>album art size</b> can bet set in the 'Client' tab. After a restart the player will adapt to the new size.")
                       + "<hr>"
                       + tr("A <b>custom image</b> to be used as default (when no album art is found) can also be set in the 'Client' tab.")
                       + "<hr>"
                       + tr("Quimup will accept files that are <b>drag & drop</b>ped on the player or the playlist, are added on the command line or using '<b>open with</b>' in your file manager - but only if it is connected to MPD using a socket. Both <i>bind_to_address</i> in mpd.conf and <i>host</i> in Quimup's connection profile must point to <i>/your/path/to/.mpd/socket</i>.")
                       + "<hr>"
                       + tr("Clicking the time counter toggles the <b>time mode</b>, i.e. <i>time elapsed</i> and <i>time remaining</i>.")
                       + "<hr>"
                       + tr("Clicking the middle mouse-button on the system-tray icon toggles <b>play / pause.")
                       + "</b>"
                       + tr("Pressing the spacebar in the playlist scrolls the <b>current song</b> into view.")
                       + "<hr>"
                       + tr("A <b>scrolling title</b> speeds up (quite a bit) when the mouse cursor is moved over it, and can be paused with a mouse-press.")
                       + "<hr>"
                       + tr("The <b>album comments</b> can be scrolled with the mouse wheel when the comments are too large to fit the label.")
                       + "<hr>"
                       + tr("<b>Selecting</b> an item in the library tree will disable the selection of subitems. A selected artist's albums, for instance, can not be selected because the artist already includes those albums.")
                       + "<hr>"
                       + tr("The 'Style' tab of the settings window has an option to use icons and colors that are more suitable for <b>dark themes</b>.")
                       + "<hr>"
                       + tr("To make sure you have all necessary <b>user rights</b> it is recommended to NOT run MPD as a system service and start MPD as user: <i>systemctl --user start mpd</i>.")
                       + "<hr>"
                       + tr("Run <i>quimup -help</i> in a terminal for a list of the <b>command line parameters</b>.")
                       + "<hr>"
                       + tr("Run <i>quimup -e</i> in a terminal for more extensive <b>cout information</b>.")
                       + "<hr><br>"
                       + "<a href=\"file:////usr/share/doc/quimup/faq\">/usr/share/doc/quimup/faq</a><br>"
                       + "<a href=\"file:////usr/share/doc/quimup/readme\">/usr/share/doc/quimup/readme</a><br>"
                       + "</div><b><div align=\"center\">"
                       + "<br><br>"
                       + tr("Enjoy!")
                       + "<br><br></b></div>";
    lb_about_info->setText(s_tips);
    lb_about_info->setOpenExternalLinks(true);
} // retranslate


void qm_Settings::reset_fonts()
{
    sb_titlefont->setValue(11);
    sb_codecfont->setValue(8);
    sb_timefont->setValue(8);
    sb_albumfont->setValue(9);
    sb_yearfont->setValue(9);
    sb_commentfont->setValue(8);
    sb_liblistfont->setValue(10);
    cb_title_bold->setChecked(true);
    cb_title_italic->setChecked(true);
    cb_codec_bold->setChecked(false);
    cb_time_bold->setChecked(false);
    cb_album_bold->setChecked(true);
    cb_year_bold->setChecked(false);
    cb_comment_italic->setChecked(false);
}


void qm_Settings::cancel_changes()
{
    // Apply or save nothing.
    // Changes will be reset next time in init_vars()
    this->close();
}


void qm_Settings::apply_settings()
{
    // Apply ONLY the current tab
    // Changes in other tabs will be reset next time in init_vars()

    if (tab_widget->currentIndex() == 0) // CONNECT
    {
        // timing
        config->auto_connect = cb_auto_conn->isChecked();
        config->auto_re_connect = cb_auto_Re_conn->isChecked();
        config->mpd_pollfreq = sb_pollfreq->value();
        if (b_mpd_connected)
        config->mpd_timeout = sb_timeout->value();
        if (b_mpd_connected)
            mpd_cmd->set_timeout(sb_timeout->value());
        config->profile_current = cm_profiles->currentIndex();

        // profiles
        switch (config->profile_current)
        {
        case 1:
        {
            config->profile_note_1 = (le_note_text->text()).simplified();
            config->profile_port_1 = (le_port_text->text()).toUInt();
            config->profile_host_1 = (le_host_text->text()).simplified();
            config->profile_pswd_1 = (le_pswd_text->text()).simplified();
            config->profile_mdir_1 = (le_mdir_text->text()).simplified();
            break;
        }
        case 2:
        {
            config->profile_note_2 = (le_note_text->text()).simplified();
            config->profile_port_2 = (le_port_text->text()).toUInt();
            config->profile_host_2 = (le_host_text->text()).simplified();
            config->profile_pswd_2 = (le_pswd_text->text()).simplified();
            config->profile_mdir_2 = (le_mdir_text->text()).simplified();
            break;
        }
        case 3:
        {
            config->profile_note_3 = (le_note_text->text()).simplified();
            config->profile_port_3 = (le_port_text->text()).toUInt();
            config->profile_host_3 = (le_host_text->text()).simplified();
            config->profile_pswd_3 = (le_pswd_text->text()).simplified();
            config->profile_mdir_3 = (le_mdir_text->text()).simplified();
            break;
        }
        case 4:
        {
            config->profile_note_4 = (le_note_text->text()).simplified();
            config->profile_port_4 = (le_port_text->text()).toUInt();
            config->profile_host_4 = (le_host_text->text()).simplified();
            config->profile_pswd_4 = (le_pswd_text->text()).simplified();
            config->profile_mdir_4 = (le_mdir_text->text()).simplified();
            break;
        }
        default:
            break;
        }
    }
    else // SERVER
    if (tab_widget->currentIndex() == 1)
    {
        // x fade
        config->xfade_time = sb_xfade->value();
        if (config->xfade_is_on && b_mpd_connected)
            mpd_cmd->set_xfade(config->xfade_time);

        // replay gain
        bool b_set_rpgain = false;

        if (rb_rpg_off->isChecked() &&  config->replaygain_mode != "off")
        {
            config->replaygain_mode = "off";
            b_set_rpgain = true;
        }
        else
            if (rb_rpg_track->isChecked() && config->replaygain_mode != "track")
            {
                config->replaygain_mode = "track";
                b_set_rpgain = true;
            }
            else
                if (rb_rpg_album->isChecked() && config->replaygain_mode != "album")
                {
                    config->replaygain_mode = "album";
                    b_set_rpgain = true;
                }
                else
                    if (rb_rpg_auto->isChecked() && config->replaygain_mode != "auto")
                    {
                        config->replaygain_mode = "auto";
                        b_set_rpgain = true;
                    }

        if (b_set_rpgain)
            set_replaygain_mode();

        // output devices
        apply_output_devices();

        // manage mpd
        config->onquit_quitmpd = cb_killserver->isChecked();
        config->onquit_mpd_command = (le_killcommand->text()).simplified();
        config->onstart_startmpd = cb_startserver->isChecked();
        config->onstart_mpd_command = (le_startcommand->text()).simplified();
    }
    else // CLIENT
    if (tab_widget->currentIndex() == 2)
    {
        // general
        config->use_trayicon = cb_use_systray->isChecked();
        config->start_minimized = cb_start_minimized->isChecked();
        config->show_tooltips = cb_showTips->isChecked();

        // library
        config->ignore_leading_the = cb_ignore_the->isChecked();
        config->sort_albums_year = cb_sortbyyear->isChecked();

        if (config->library_autocols != cb_library_autocols->isChecked())
        {
            config->library_autocols = cb_library_autocols->isChecked();
            emit signal_library_auto_colwidth(); //to player (->browser->libview)
        }

        // playlist
        config->auto_playfirst = cb_auto_playfirst->isChecked();
        if (config->mark_whenplayed != cb_mark_whenplayed->isChecked())
        {
            config->mark_whenplayed = cb_mark_whenplayed->isChecked();
            emit signal_mark_whenplayed(); //to player (->browser->plistview)
        }
        if (config->playlist_autocols != cb_playlist_autocols->isChecked())
        {
            config->playlist_autocols = cb_playlist_autocols->isChecked();
            emit signal_playlist_auto_colwidth(); //to player (->browser->plistview)
        }

        // album art
        bool b_customart_changed = false;
        QString cpath = (le_custom_art->text()).simplified();
        if (config->custom_albumart_file != cpath)
        {
            if (cpath.startsWith("file://"))
                cpath = cpath.right(cpath.length() - 7);
            if (!cpath.startsWith("/") && !cpath.isEmpty())
                cpath.prepend("/");
            le_custom_art->setText(cpath);

            if (config->custom_albumart_file != cpath)
            {
                b_customart_changed = true;
                config->custom_albumart_file = cpath;
            }
        }

        if (cb_custom_art->isChecked() != config->albumart_custom)
        {
            b_customart_changed = true;
            config->albumart_custom = cb_custom_art->isChecked();
        }

        if (b_customart_changed)
            emit signal_custom_art(); //to player

        // apply without restart is not really possible
        config->albumart_size = sb_art_size->value();
    }
    else // STYLE
    if (tab_widget->currentIndex() == 3)
    {
        apply_fontsizes();
        apply_colors();
        config->set_dark_theme = cb_dark_icons->isChecked();
    }
    else // ACTIONS
    if (tab_widget->currentIndex() == 4)
    {
        bool b_app_change = false;

        // external applications
        QString app = (le_imageviewer->text()).simplified();
        if (config->image_viewer != app)
        {
            config->image_viewer = app;
            b_app_change = true;
        }
        app = (le_tageditor->text()).simplified();
        if (config->tag_editor != app)
        {
            config->tag_editor = app;
            b_app_change = true;
        }
        app = (le_filemanager->text()).simplified();
        if (config->file_manager != app)
        {
            config->file_manager = app;
            b_app_change = true;
        }

        // user actions
        if (config->usract1_fileparm != rb_file_usract1->isChecked())
        {
            b_app_change = true;
            config->usract1_fileparm = rb_file_usract1->isChecked();
        }
        if (config->usract2_fileparm != rb_file_usract2->isChecked())
        {
            b_app_change = true;
            config->usract2_fileparm = rb_file_usract2->isChecked();
        }
        if (config->usract1_enabled != cb_usract1->isChecked())
        {
            b_app_change = true;
            config->usract1_enabled = cb_usract1->isChecked();
        }
        if (config->usract2_enabled != cb_usract2->isChecked())
        {
            b_app_change = true;
            config->usract2_enabled = cb_usract2->isChecked();
        }
        if (config->usract1_command != le_cmd_usract1->text())
        {
            b_app_change = true;
            config->usract1_command = le_cmd_usract1->text();
        }
        if (config->usract2_command != le_cmd_usract2->text())
        {
            b_app_change = true;
            config->usract2_command = le_cmd_usract2->text();
        }
        if (config->usract1_label != le_label_usract1->text())
        {
            b_app_change = true;
            config->usract1_label = le_label_usract1->text();
        }
        if (config->usract2_label != le_label_usract2->text())
        {
            b_app_change = true;
            config->usract2_label = le_label_usract2->text();
        }

        if (b_app_change)
            emit signal_actions_changed(); // to player
    }
    else // ABOUT
        return;

    config->save_config();
}


void qm_Settings::connect_to_server()
{
    if (mpd_cmd == nullptr)
        return;

    config->auto_connect = cb_auto_conn->isChecked();
    config->auto_re_connect = cb_auto_Re_conn->isChecked();
    config->profile_current = cm_profiles->currentIndex();
    switch (config->profile_current)
    {
        case 1:
        {
            config->profile_note_1 = (le_note_text->text()).simplified();
            config->profile_port_1 = (le_port_text->text()).toUInt();
            config->profile_host_1 = (le_host_text->text()).simplified();
            config->profile_pswd_1 = (le_pswd_text->text()).simplified();
            config->profile_mdir_1 = (le_mdir_text->text()).simplified();

            break;
        }
        case 2:
        {
            config->profile_note_2 = (le_note_text->text()).simplified();
            config->profile_port_2 = (le_port_text->text()).toUInt();
            config->profile_host_2 = (le_host_text->text()).simplified();
            config->profile_pswd_2 = (le_pswd_text->text()).simplified();
            config->profile_mdir_2 = (le_mdir_text->text()).simplified();
            break;
        }
        case 3:
        {
            config->profile_note_3 = (le_note_text->text()).simplified();
            config->profile_port_3 = (le_port_text->text()).toUInt();
            config->profile_host_3 = (le_host_text->text()).simplified();
            config->profile_pswd_3 = (le_pswd_text->text()).simplified();
            config->profile_mdir_3 = (le_mdir_text->text()).simplified();
            break;
        }
        case 4:
        {
            config->profile_note_4 = (le_note_text->text()).simplified();
            config->profile_port_4 = (le_port_text->text()).toUInt();
            config->profile_host_4 = (le_host_text->text()).simplified();
            config->profile_pswd_4 = (le_pswd_text->text()).simplified();
            config->profile_mdir_4 = (le_mdir_text->text()).simplified();
            break;
        }
        default:
            break;
    }

    mpd_cmd->configure_connection();
    mpd_cmd->mpd_try_connect(true);
}


void qm_Settings::apply_fontsizes()
{
    bool fonts_changed = false;

    if (config->font_title_size != sb_titlefont->value())
    {
        config->font_title_size = sb_titlefont->value();
        fonts_changed = true;
    }
    if (config->font_title_bold != cb_title_bold->isChecked())
    {
        config->font_title_bold = cb_title_bold->isChecked();
        fonts_changed = true;
    }
    if (config->font_title_italic != cb_title_italic->isChecked())
    {
        config->font_title_italic = cb_title_italic->isChecked();
        fonts_changed = true;
    }
    if (config->font_codecinfo_size != sb_codecfont->value())
    {
        config->font_codecinfo_size = sb_codecfont->value();
        fonts_changed = true;
    }
    if (config->font_codecinfo_bold != cb_codec_bold->isChecked())
    {
        config->font_codecinfo_bold = cb_codec_bold->isChecked();
        fonts_changed = true;
    }
    if (config->font_time_size != sb_timefont->value())
    {
        config->font_time_size = sb_timefont->value();
        fonts_changed = true;
    }
    if (config->font_time_bold != cb_time_bold->isChecked())
    {
        config->font_time_bold = cb_time_bold->isChecked();
        fonts_changed = true;
    }
    if (config->font_album_size != sb_albumfont->value())
    {
        config->font_album_size = sb_albumfont->value();
        fonts_changed = true;
    }
    if (config->font_album_bold != cb_album_bold->isChecked())
    {
        config->font_album_bold = cb_album_bold->isChecked();
        fonts_changed = true;
    }
    if (config->font_year_size != sb_yearfont->value())
    {
        config->font_year_size = sb_yearfont->value();
        fonts_changed = true;
    }
    if (config->font_year_bold != cb_year_bold->isChecked())
    {
        config->font_year_bold = cb_year_bold->isChecked();
        fonts_changed = true;
    }
    if (config->font_comment_italic != cb_comment_italic->isChecked())
    {
        config->font_comment_italic = cb_comment_italic->isChecked();
        fonts_changed = true;
    }
    if (config->font_comment_size != sb_commentfont->value())
    {
        config->font_comment_size = sb_commentfont->value();
        fonts_changed = true;
    }

    if (config->font_browser_size != sb_liblistfont->value())
    {
        config->font_browser_size = sb_liblistfont->value();
        fonts_changed = true;
    }

    if (fonts_changed)
        emit signal_fonts(); // to player
}


void qm_Settings::apply_colors()
{
    bool colors_changed = false;

    if (config->colors_system != rb_col_syst->isChecked())
    {
        config->colors_system = rb_col_syst->isChecked();
        colors_changed = true;
    }

    if (config->colors_default != rb_col_def->isChecked())
    {
        config->colors_default = rb_col_def->isChecked();
        colors_changed = true;
    }

    if (config->colors_cutom != rb_col_cust->isChecked())
    {
        config->colors_cutom = rb_col_cust->isChecked();
        colors_changed = true;
    }
    if (config->colors_cutom && b_new_custom_colors)
        colors_changed = true;

    if (config->colfor_browser != cb_colfor_browser->isChecked())
    {
        config->colfor_browser = cb_colfor_browser->isChecked();
        colors_changed = true;
    }

    if (config->colfor_albuminfo != cb_colfor_albuminfo->isChecked())
    {
        config->colfor_albuminfo = cb_colfor_albuminfo->isChecked();
        colors_changed = true;
    }

    if (config->colfor_progressbar != cb_colfor_pgsbar->isChecked())
    {
        config->colfor_progressbar = cb_colfor_pgsbar->isChecked();
        colors_changed = true;
    }

    if (colors_changed)
        emit signal_colors(); // to player
}

// init from config / after show(): resets changes that were not applied earlier
void qm_Settings::init_vars()
{
    resize(config->settings_W, config->settings_H );

    QPalette pal1 (this->palette());
    pal1.setColor(QPalette::Window, QColor(config->color_cus_bg_title)); // background
    pal1.setColor(QPalette::WindowText, QColor(config->color_cus_fg_title)); // text
    lb_custcolor_title->setPalette(pal1);
    QPalette pal2 (this->palette());
    pal2.setColor(QPalette::Window, QColor(config->color_cus_bg_album)); // background
    pal2.setColor(QPalette::WindowText, QColor(config->color_cus_fg_album)); // text
    lb_custcolor_album->setPalette(pal2);

    sb_xfade->setValue(config->xfade_time);
    cb_auto_conn->setChecked(config->auto_connect);
    cb_auto_Re_conn->setChecked(config->auto_re_connect);
    sb_pollfreq->setValue(config->mpd_pollfreq);
    sb_timeout->setValue(config->mpd_timeout);
    QChar qc = QChar(0x2022); // bullet
    QString start = QString(" %1 ").arg(qc);
    cm_profiles->setItemText( 0, start + "Autodetect" );
    cm_profiles->setItemText( 1, start + config->profile_name_1 );
    cm_profiles->setItemText( 2, start + config->profile_name_2 );
    cm_profiles->setItemText( 3, start + config->profile_name_3 );
    cm_profiles->setItemText( 4, start + config->profile_name_4 );
    cm_profiles->setCurrentIndex(config->profile_current);
    set_profile(config->profile_current);
    cb_use_systray->setChecked(config->use_trayicon);
    cb_showTips->setChecked(config->show_tooltips);
    cb_sortbyyear->setChecked(config->sort_albums_year);
    cb_start_minimized->setChecked(config->start_minimized);
    cb_ignore_the->setChecked(config->ignore_leading_the);
    cb_playlist_autocols->setChecked(config->playlist_autocols);
    cb_library_autocols->setChecked(config->library_autocols);
    cb_auto_playfirst->setChecked(config->auto_playfirst);
    cb_mark_whenplayed->setChecked(config->mark_whenplayed);
    cb_dark_icons->setChecked(config->set_dark_theme);
    cb_custom_art->setChecked(config->albumart_custom);
    le_custom_art->setText(config->custom_albumart_file);
    sb_art_size->setValue(config->albumart_size);

    le_imageviewer->setText(config->image_viewer);
    le_tageditor->setText(config->tag_editor);
    le_filemanager->setText(config->file_manager);

    cb_killserver->setChecked(config->onquit_quitmpd);
    le_killcommand->setText(config->onquit_mpd_command);

    cb_startserver->setChecked(config->onstart_startmpd);
    le_startcommand->setText(config->onstart_mpd_command);

    sb_titlefont->setValue(config->font_title_size);
    cb_title_bold->setChecked(config->font_title_bold);
    cb_title_italic->setChecked(config->font_title_italic);

    sb_codecfont->setValue(config->font_codecinfo_size);
    cb_codec_bold->setChecked(config->font_codecinfo_bold);
    sb_timefont->setValue(config->font_time_size);
    cb_time_bold->setChecked(config->font_time_bold);
    sb_albumfont->setValue(config->font_album_size);
    cb_album_bold->setChecked(config->font_album_bold);
    sb_yearfont->setValue(config->font_year_size);
    cb_year_bold->setChecked(config->font_year_bold);
    sb_commentfont->setValue(config->font_comment_size);
    cb_comment_italic->setChecked(config->font_comment_italic);
    sb_liblistfont->setValue(config->font_browser_size);

    cb_colfor_browser->setChecked(config->colfor_browser);
    cb_colfor_albuminfo->setChecked(config->colfor_albuminfo);
    cb_colfor_pgsbar->setChecked(config->colfor_progressbar);

    if (config->colors_system)
        rb_col_syst->setChecked(true);
    else
        if (config->colors_default)
            rb_col_def->setChecked(true);
        else
            if (config->colors_cutom)
                rb_col_cust->setChecked(true);

    if (config->replaygain_mode == "track")
        rb_rpg_track->setChecked(true);
    else
        if (config->replaygain_mode == "album")
            rb_rpg_album->setChecked(true);
        else
            if (config->replaygain_mode == "auto")
                rb_rpg_auto->setChecked(true);
            else
                rb_rpg_off->setChecked(true); // default

    if (config->usract1_fileparm)
        rb_file_usract1->setChecked(true);
    else
        rb_dir_usract1->setChecked(true);
    if (config->usract2_fileparm)
        rb_file_usract2->setChecked(true);
    else
        rb_dir_usract2->setChecked(true);
    cb_usract1->setChecked(config->usract1_enabled);
    cb_usract2->setChecked(config->usract2_enabled);
    le_cmd_usract1->setText(config->usract1_command);
    le_cmd_usract2->setText(config->usract2_command);
    le_label_usract1->setText(config->usract1_label);
    le_label_usract2->setText(config->usract2_label);
}

// disable or enable stuff here, depending on connection
void qm_Settings::set_connected(qm_mpdCommand *cmd, bool isconnected, bool isremote)
{
    mpd_cmd = cmd;
    if (mpd_cmd == nullptr)
        b_mpd_connected = false;
    else
        b_mpd_connected = isconnected;

    if (!b_mpd_connected)
    {
        lb_mpdconnected->setText("<small>" + tr ("Not connected to MPD") + "  </small>");
        lb_conn_status_text->setText(tr("Not connected"));
        lb_conn_icon->setPixmap(pm_disconnected);

        gbox_outputs->setEnabled(false);
        cb_dev1->setText("");
        cb_dev1->setChecked(false);
        cb_dev2->setText("");
        cb_dev2->setChecked(false);
        cb_dev3->setText("");
        cb_dev3->setChecked(false);
        cb_dev4->setText("");
        cb_dev4->setChecked(false);
        cb_dev5->setText("");
        cb_dev5->setChecked(false);
        gbox_replaygain->setEnabled(false);
        gbox_xfade->setEnabled(false);

        lb_owner_icon->setEnabled(false);
        lb_owner_status_text->setEnabled(false);
        lb_musicdir_status_text->setEnabled(false);
        lb_musicdir_icon->setEnabled(false);

        if (!config->mpd_is_installed)
        {
            lb_owner_icon->setPixmap(pm_mpd_notactive);
            lb_owner_status_text->setText(tr("MPD is not installed on this system"));
        }
        else
        {
            if (config->mpd_process_owner.isEmpty())
            {
                lb_owner_icon->setPixmap(pm_mpd_notactive);
                lb_owner_status_text->setText(tr("MPD is not running on this system"));
            }
            else
            {
                if (config->mpd_process_owner == "root")
                {
                    lb_owner_icon->setPixmap(pm_owner_root);
                    lb_owner_status_text->setText(tr("MPD is a system service for 'root'"));
                }
                else
                {
                    lb_owner_icon->setPixmap(pm_owner_user);
                    lb_owner_status_text->setText(tr("MPD is a user service for") + " '" + config->mpd_process_owner + "'");
                }
            }
        }

        switch (config->mpd_musicdir_status)
        {
            case MDIR_NOT_DETERMINED:
            {
                lb_musicdir_status_text->setText(tr("Music directory was not determined"));
                lb_musicdir_icon->setPixmap(pm_musicdir_nocon);
                break;
            }
            case MDIR_ACCESSIBLE:
            {
                lb_musicdir_status_text->setText(tr("Music directory is accessible"));
                lb_musicdir_icon->setPixmap(pm_musicdir_ok);
                break;
            }
            case MDIR_NOTREADABLE:
            {
                lb_musicdir_status_text->setText(tr("Music directory is not readable"));
                lb_musicdir_icon->setPixmap(pm_musicdir_error);
                break;
            }
            case MDIR_NOTFOUND:
            {
                lb_musicdir_status_text->setText(tr("Music directory was not found"));
                lb_musicdir_icon->setPixmap(pm_musicdir_error);
                break;
            }
            default: // unknown status
            {
                lb_musicdir_status_text->setText(tr("Music directory has unknown status"));
                lb_musicdir_icon->setPixmap(pm_musicdir_error);
                break;
            }
        }
    }
    else
    {
        if (isremote)
        {
            lb_mpdconnected->setText("<small>" + tr("Connected to MPD / remote\n") + "  </small>");
            lb_conn_status_text->setText( tr("Connected to MPD") + " " + mpd_cmd->get_version(false) + " / remote server");
            lb_conn_icon->setPixmap(pm_connected);

            lb_owner_icon->setPixmap(pm_mpd_notactive);
            lb_owner_status_text->setText(tr("MPD is not a local service"));
            lb_owner_icon->setEnabled(false);
            lb_owner_status_text->setEnabled(false);

            lb_musicdir_icon->setPixmap(pm_musicdir_nocon);
            lb_musicdir_status_text->setText(tr("Music directory is remote"));
            lb_musicdir_status_text->setEnabled(false);
            lb_musicdir_icon->setEnabled(false);
        }
        else
        {
            lb_mpdconnected->setText("<small>" + tr("Connected to MPD / local\n") + "  </small>");
            lb_conn_status_text->setText( tr("Connected to MPD") + " " + mpd_cmd->get_version(false) + " / local server");
            lb_conn_icon->setPixmap(pm_connected);

            lb_owner_icon->setEnabled(true);
            lb_owner_status_text->setEnabled(true);
            lb_musicdir_status_text->setEnabled(true);
            lb_musicdir_icon->setEnabled(true);

            if (!config->mpd_is_installed)
            {
                lb_owner_icon->setPixmap(pm_mpd_notactive);
                lb_owner_status_text->setText(tr("MPD is not installed on this system"));
            }
            else
            {
                if (config->mpd_process_owner.isEmpty())
                {
                    lb_owner_icon->setPixmap(pm_mpd_notactive);
                    lb_owner_status_text->setText(tr("MPD is not running on this system"));
                }
                else
                {
                    if (config->mpd_process_owner == "root")
                    {
                        lb_owner_icon->setPixmap(pm_owner_root);
                        lb_owner_status_text->setText(tr("MPD is a system service for 'root'"));
                    }
                    else
                    {
                        lb_owner_icon->setPixmap(pm_owner_user);
                        lb_owner_status_text->setText(tr("MPD is a user service for") + " '" + config->mpd_process_owner + "'");
                    }
                }
            }

            switch (config->mpd_musicdir_status)
            {
                case MDIR_NOT_DETERMINED:
                {
                    lb_musicdir_status_text->setText(tr("Music directory was not determined"));
                    lb_musicdir_icon->setPixmap(pm_musicdir_nocon);
                    break;
                }
                case MDIR_ACCESSIBLE:
                {
                    lb_musicdir_status_text->setText(tr("Music directory is accessible"));
                    lb_musicdir_icon->setPixmap(pm_musicdir_ok);
                    break;
                }
                case MDIR_NOTREADABLE:
                {
                    lb_musicdir_status_text->setText(tr("Music directory is not readable"));
                    lb_musicdir_icon->setPixmap(pm_musicdir_error);
                    break;
                }
                case MDIR_NOTFOUND:
                {
                    lb_musicdir_status_text->setText(tr("Music directory was not found"));
                    lb_musicdir_icon->setPixmap(pm_musicdir_error);
                    break;
                }
                default: // unknown status
                {
                    lb_musicdir_status_text->setText(tr("Music directory has unknown status"));
                    lb_musicdir_icon->setPixmap(pm_musicdir_error);
                    break;
                }
            }
        }

        if (config->mpd_setrpgain_allowed)
        {
            gbox_replaygain->setEnabled(true);
            set_replaygain_mode();
        }
        else
            gbox_replaygain->setEnabled(false);

        if (config->mpd_xfade_allowed)
            gbox_xfade->setEnabled(true);
        else
            gbox_xfade->setEnabled(false);

        if (config->mpd_op_listall_allowed)
        {
            get_output_devices();

            if (config->mpd_op_disable_allowed || config->mpd_op_enable_allowed)
            {
                gbox_outputs->setEnabled(true);
            }
            else
                gbox_outputs->setEnabled(false);
        }
        else
            gbox_outputs->setEnabled(false);
    }
}


void qm_Settings::set_replaygain_mode()
{
    if (!b_mpd_connected)
        return;

    if(!config->mpd_setrpgain_allowed)
    {
        printf ("Replaygain not set: MPD does not allow it\n");
        return;
    }

    if (config->replaygain_mode == "off" )
    {
        mpd_cmd->set_replaygain_mode(0);
        if (config->cout_extensive)
            printf ("Replaygain mode set to \"off\"\n");
    }
    else
    if (config->replaygain_mode == "track" )
    {
        mpd_cmd->set_replaygain_mode(1);
        if (config->cout_extensive)
            printf ("Replaygain mode set to \"track\"\n");
    }
    else
    if (config->replaygain_mode == "album" )
    {
        mpd_cmd->set_replaygain_mode(2);
        if (config->cout_extensive)
            printf ("Replaygain mode set to \"album\"\n");
    }
    else
    if (config->replaygain_mode == "auto" )
    {
        mpd_cmd->set_replaygain_mode(3);
        if (config->cout_extensive)
            printf ("Replaygain mode set to \"auto\"\n");
    }
}

// send ouput device selectionss to mpd
void qm_Settings::apply_output_devices()
{
    if (!b_mpd_connected)
        return;

    bool devs_changed = false;
    qm_outputList outputlist;

    // dev1
    if (cb_dev1->isCheckable())
    {
        qm_output output;
        output.id = 0;
        output.enabled = cb_dev1->isChecked();
        outputlist.push_back(output);
        if (cb_dev1->isChecked()!= dev1_isChecked)
        {
            devs_changed = true;
            dev1_isChecked = cb_dev1->isChecked();
        }
    }
    // dev2
    if (cb_dev2->isCheckable() )
    {
        qm_output output;
        output.id = 1;
        output.enabled = cb_dev2->isChecked();
        outputlist.push_back(output);
        if (cb_dev2->isChecked()!= dev2_isChecked)
        {
            devs_changed = true;
            dev2_isChecked = cb_dev2->isChecked();
        }
    }
    // dev3
    if (cb_dev3->isCheckable())
    {
        qm_output output;
        output.id = 2;
        output.enabled = cb_dev3->isChecked();
        outputlist.push_back(output);
        if (cb_dev3->isChecked()!= dev3_isChecked)
        {
            devs_changed = true;
            dev3_isChecked = cb_dev3->isChecked();
        }
    }
    // dev4
    if (cb_dev4->isCheckable())
    {
        qm_output output;
        output.id = 3;
        output.enabled = cb_dev4->isChecked();
        outputlist.push_back(output);
        if (cb_dev4->isChecked()!= dev4_isChecked)
        {
            devs_changed = true;
            dev4_isChecked = cb_dev4->isChecked();
        }
    }
    // dev5
    if (cb_dev5->isCheckable())
    {
        qm_output output;
        output.id = 4;
        output.enabled = cb_dev5->isChecked();
        outputlist.push_back(output);
        if (cb_dev5->isChecked()!= dev5_isChecked)
        {
            devs_changed = true;
            dev5_isChecked = cb_dev5->isChecked();
        }
    }

    if (devs_changed)
    {
        if (mpd_cmd->set_outputs(outputlist))
        {
            if (config->cout_extensive)
                printf ("Setting MPD outputs succeeded\n");
        }
        else
            printf ("Setting MPD outputs failed!\n");
    }
}

// get the output devices from mpd
void qm_Settings::get_output_devices()
{
    if (!b_mpd_connected)
        return;

    qm_outputList outputlist;
    qm_output output;
    dev1_isChecked = false;
    dev2_isChecked = false;
    dev3_isChecked = false;
    dev4_isChecked = false;
    dev5_isChecked = false;

    outputlist = mpd_cmd->get_outputs();
    // Set up the output devices
    QListIterator<qm_output> i(outputlist);

    if (i.hasNext())
    {
        cb_dev1->setCheckable(true);
        output = i.next();
        cb_dev1->setText(output.name);
        cb_dev1->setChecked(output.enabled);
        dev1_isChecked = output.enabled;
    }
    else
    {
        cb_dev1->setCheckable(false);
        cb_dev1->setChecked(false);
        cb_dev1->setText( tr("(Not connected)") );
        cb_dev2->setCheckable(false);
        cb_dev2->setChecked(false);
        cb_dev2->setText("..");
        cb_dev3->setCheckable(false);
        cb_dev3->setChecked(false);
        cb_dev3->setText("..");
        cb_dev4->setCheckable(false);
        cb_dev4->setChecked(false);
        cb_dev4->setText("..");
        cb_dev5->setCheckable(false);
        cb_dev5->setChecked(false);
        cb_dev5->setText("..");
        return;
    }

    if (i.hasNext())
    {
        cb_dev2->setCheckable(true);
        output = i.next();
        cb_dev2->setText(output.name);
        cb_dev2->setChecked(output.enabled);
        dev2_isChecked = output.enabled;
    }
    else
    {
        cb_dev2->setCheckable(false);
        cb_dev2->setChecked(false);
        cb_dev2->setText("..");
        cb_dev3->setCheckable(false);
        cb_dev3->setChecked(false);
        cb_dev3->setText("..");
        cb_dev4->setCheckable(false);
        cb_dev4->setChecked(false);
        cb_dev4->setText("..");
        cb_dev5->setCheckable(false);
        cb_dev5->setChecked(false);
        cb_dev5->setText("..");
        return;
    }

    if (i.hasNext())
    {
        cb_dev3->setCheckable(true);
        output = i.next();
        cb_dev3->setText(output.name);
        cb_dev3->setChecked(output.enabled);
        dev3_isChecked = output.enabled;
    }
    else
    {
        cb_dev3->setCheckable(false);
        cb_dev3->setChecked(false);
        cb_dev3->setText("..");
        cb_dev4->setCheckable(false);
        cb_dev4->setChecked(false);
        cb_dev4->setText("..");
        cb_dev5->setCheckable(false);
        cb_dev5->setChecked(false);
        cb_dev5->setText("..");
        return;
    }

    if (i.hasNext())
    {
        cb_dev4->setCheckable(true);
        output = i.next();
        cb_dev4->setText(output.name);
        cb_dev4->setChecked(output.enabled);
        dev4_isChecked = output.enabled;
    }
    else
    {
        cb_dev4->setCheckable(false);
        cb_dev4->setChecked(false);
        cb_dev4->setText("..");
        cb_dev5->setCheckable(false);
        cb_dev5->setChecked(false);
        cb_dev5->setText("..");
        return;
    }

    if (i.hasNext())
    {
        cb_dev5->setCheckable(true);
        output = i.next();
        cb_dev5->setText(output.name);
        cb_dev5->setChecked(output.enabled);
        dev5_isChecked = output.enabled;
    }
    else
    {
        cb_dev5->setCheckable(false);
        cb_dev5->setChecked(false);
        cb_dev5->setText("..");
        return;
    }
}


void qm_Settings::closeEvent( QCloseEvent *event )
{
    this->hide();
    event->ignore(); // only hide, don't close
}


void qm_Settings::hideEvent(QHideEvent *event)
{
    if (colorpicker->isVisible())
        colorpicker->hide();

    if (!b_wayland)
    {
        config->settings_X = x();
        config->settings_Y = y();
    }
    config->settings_W = width();
    config->settings_H = height();
    event->accept();
}


void qm_Settings::showEvent(QShowEvent *event)
{
    if (!b_wayland)
        move(config->settings_X, config->settings_Y);

    // reset changes that were not applied/saved:
    init_vars();

    event->accept();
}


void qm_Settings::show_colorpicker()
{
    if (colorpicker->isVisible())
    {
        colorpicker->hide();
        return;
    }

    colorpicker->move(x()+160, y()+190);
    colorpicker->show();
}


void qm_Settings::on_new_customcolors()
{
    QPalette pal1 (this->palette());
    pal1.setColor(QPalette::Window, QColor(config->color_cus_bg_title)); // background
    pal1.setColor(QPalette::WindowText, QColor(config->color_cus_fg_title)); // text
    lb_custcolor_title->setPalette(pal1);
    QPalette pal2 (this->palette());
    pal2.setColor(QPalette::Window, QColor(config->color_cus_bg_album)); // background
    pal2.setColor(QPalette::WindowText, QColor(config->color_cus_fg_album)); // text
    lb_custcolor_album->setPalette(pal2);

    b_new_custom_colors = true;
}


void qm_Settings::set_themed_icons(bool dark_theme)
{
    if (!dark_theme)
    {
        pm_owner_root     = QPixmap(":/se_owner_root");
        pm_owner_user     = QPixmap(":/se_owner_user");
        pm_mpd_notactive  = QPixmap(":/se_mpd_notactive");
        pm_musicdir_ok    = QPixmap(":/se_mdir_ok");
        pm_musicdir_nocon = QPixmap(":/se_mdir_nocon");
        pm_musicdir_error = QPixmap(":/se_mdir_error");
    }
    else
    {
        pm_owner_root     = QPixmap(":/se_owner_root.alt");
        pm_owner_user     = QPixmap(":/se_owner_user.alt");
        pm_mpd_notactive  = QPixmap(":/se_mpd_notactive.alt");
        pm_musicdir_ok    = QPixmap(":/se_mdir_ok.alt");
        pm_musicdir_nocon = QPixmap(":/se_mdir_nocon.alt");
        pm_musicdir_error = QPixmap(":/se_mdir_error.alt");
    }
    // not themed
    ic_bt_connect = QIcon(":/se_connect");
    ic_br_check = QIcon(":/br_check");
    ic_br_reddelete = QIcon(":/br_reddelete");
    pm_disconnected = QPixmap( ":/se_disconnected");
    pm_connected = QPixmap( ":/se_connected");
}


qm_Settings::~qm_Settings()
{
    delete(colorpicker);
}
