#include "nsec_rr.h"

namespace ADNS {

		NSEC_RR::NSEC_RR()
		{
			rr_type = RR_TYPE::NSEC;
			rr_class = RR_CLASS::IN;
			UpdateRdata();
		}

		DOMAIN_NAME^ NSEC_RR::GetNextDomain()
		{
			return next_domain->Clone();
		}

		Void NSEC_RR::SetNextDomain(DOMAIN_NAME^ nd)
		{
			next_domain = nd->Clone();
			UpdateRdata();
		}

		array<Byte>^ NSEC_RR::GetBitmap()
		{
			array<Byte>^ bm;
			if (!bitmap)
				return nullptr;

			bm = gcnew array<Byte>(bitmap->Length);
			bitmap->CopyTo(bm,0);

			return bm;
		}


		Void NSEC_RR::SetBitmap(List<RR_TYPE>^ typelist)
		{
			size_t i = 0;
			unsigned short int bm_len = 0;
			unsigned short int i_type;
			unsigned short int max_windows = 0;
			Byte window_num = 0;
			Byte winlen = 0;
			int datapos = 0;
			array<Byte>^ data = nullptr;
			array<Byte>^ cur_data = gcnew array<Byte>(32);
			Byte cur_window = 0;
			Byte cur_window_max = 0;
			unsigned short int cur_data_size = 0;
			
			if (!typelist)
				return;

			i_type = 0;

			for (i = 0; i < typelist->Count; i++) {
				if ( i_type < (unsigned short int) typelist[i])
					i_type = (unsigned short int) typelist[i];
			}

			if (i_type < (unsigned short int) RR_TYPE::NSEC) {
				i_type = (unsigned short int) RR_TYPE::NSEC;
			}

			bm_len = i_type / 8 + 2;
			if (!data)
				data = gcnew array<Byte>(bm_len);
			else
				data->Resize(data,bm_len);

			data->Clear(bitmap,0,data->Length);

			for (i = 0; i < typelist->Count; i++) {
				SetBit(data,(unsigned short int) typelist[i],true);
			}
			
			bitmap = gcnew array<Byte>(1);
			datapos = 0;

			i = 0;
			window_num = 0;

			/* Fold it into windows */
			while (i < data->Length)
			{
				winlen = 32;
				if (i + winlen >= data->Length)
				{
					winlen = data->Length - i;
					break;
				}
				//Are there any set bits in the window?
				winlen = find_last_non_zero_byte(bitmap,i,winlen) + 1; //+1 since this is a length, not a zero based position
				if (winlen > 0)
				{
					bitmap->Resize(bitmap,bitmap->Length + winlen + 2); // +2 for windows number and windows length bytes
					bitmap[datapos++] = window_num;
					bitmap[datapos++] = winlen;
					data->Copy(data,i,bitmap,datapos,winlen);
				}
				window_num++;
				i += 32;
			}

			UpdateRdata();
		}
		
		List<RR_TYPE>^ NSEC_RR::GetTypesFromBitmap()
		{
			List<RR_TYPE>^ output = gcnew List<RR_TYPE>(0);
			int pos = 0;
			Byte window_num = 0;
			Byte window_len = 0;
			int i = 0;
			int j = 0;

			while (pos < bitmap->Length)
			{
				window_num = bitmap[pos++];
				window_len = bitmap[pos++];

				for (j = 0; j < window_len; j++)
				{
					if (bitmap[pos + j] != 0)
					{
						//Need to find the set bits, and reconstruct them into values
						for (i = 0; i < 8; ++i)
						{
							if (bitmap[pos + j] & (0x1 <<(7 - i)))
							{
								output->Add((RR_TYPE) ((window_num * 256) * (j * 8) + i));
							}
						}
					}
				}

				pos += window_len;
			}

			return output;
		}

		
		Void NSEC_RR::SetBitmap(array<Byte>^ bm)
		{
			if (bitmap == nullptr)
				bitmap = gcnew array<Byte>(bm->Length);
			else
				bitmap->Resize(bitmap,bm->Length);

			bm->CopyTo(bitmap,0);
			UpdateRdata();
		}

		Void NSEC_RR::UpdateRdata()
		{
			int rdata_len = 0;
			int pos = 0;

			if (bitmap)
				rdata_len += bitmap->Length;
			if (next_domain)
				rdata_len += next_domain->GetName()->Length;
			else
				rdata_len += 1;  //For the zero identifier

			if (!rdata)
				rdata = gcnew array<Byte>(rdata_len);
			else
				rdata->Resize(rdata,rdata_len);

			if (next_domain)
			{
				next_domain->GetName()->CopyTo(rdata,0);
				pos = next_domain->GetName()->Length;
			}
			else
			{
				rdata[0] = 0;
				pos = 1;
			}

			if (bitmap)
			{
				bitmap->CopyTo(rdata,pos);
			}

			return;
		}
			
		String^ NSEC_RR::Print()
		{
			String^ output;
			List<RR_TYPE>^ l;
			int i = 0;

			l = GetTypesFromBitmap();
			
			output = PrintHeader();
			output += " ";
			output += next_domain->Print();
			output += " ";
			for (i = 0; i < l->Count; ++i)
			{
				output += Enum::GetName(RR_TYPE::typeid,l[i]);
				output += " ";
			}
			
			return output;

		}
		
		bool NSEC_RR::BitmapCoversType(RR_TYPE t)
		{
			List<RR_TYPE>^ l;

			l = GetTypesFromBitmap();
			if (l->Contains(t))
				return true;

			return false;
		}

		Void NSEC_RR::ToCanonical()
		{
			next_domain->MakeCanonical();
			owner->MakeCanonical();
			UpdateRdata();
			return;
		}

		NSEC_RR^ NSEC_RR::Clone()
		{
			NSEC_RR^ newrr = gcnew NSEC_RR();
			newrr->rr_type = rr_type;
			newrr->owner = owner->Clone();
			newrr->ttl = ttl;
			newrr->rr_class = rr_class;
			newrr->next_domain = next_domain->Clone();
			newrr->bitmap = gcnew array<Byte>(bitmap->Length);
			bitmap->CopyTo(newrr->bitmap,0);
			newrr->UpdateRdata();
			return newrr;
		}

		ResourceRecord^ NSEC_RR::ParseResourceRecord(array<Byte>^ domainname, UInt16 rr_type, UInt16 rr_class, UInt32 ttl, UInt16 rdata_len, array<Byte>^ packet, int rdata_start)
		{
			NSEC_RR^ nsecout;
			array<Byte>^ tmparray;
			int pos;
			int len;
			int reallen;

			nsecout = gcnew NSEC_RR();
			nsecout->owner = gcnew DOMAIN_NAME(domainname);
			nsecout->rr_class = (RR_CLASS) rr_class;
			nsecout->ttl = ttl;
			tmparray = ReadDomainFromPacket(packet, rdata_start, reallen);
			nsecout->SetNextDomain(gcnew DOMAIN_NAME(tmparray));
			pos = rdata_start + reallen;
			len = rdata_start + rdata_len - pos;
			tmparray->Resize(tmparray,len);
			packet->Copy(packet,pos,tmparray,0,len);
			nsecout->SetBitmap(tmparray);
			return nsecout;
		}

	String^ NSEC_RR::PrintRR(ResourceRecord^ rec)
	{
		return safe_cast<NSEC_RR^>(rec)->Print();
	}

	ResourceRecord^ NSEC_RR::CloneRR(ResourceRecord^ rec)
	{
		return safe_cast<NSEC_RR^>(rec)->Clone();
	}
}