pub const ext = @import("ext.zig");
const ibus = @This();

const std = @import("std");
const compat = @import("compat");
const gio = @import("gio2");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
/// Array of IBusAttribute.
pub const AttrList = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.AttrListClass;
    f_parent: ibus.Serializable,
    /// GArray that holds `ibus.Attribute`.
    f_attributes: ?*glib.Array,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates an new `ibus.AttrList`.
    extern fn ibus_attr_list_new() *ibus.AttrList;
    pub const new = ibus_attr_list_new;

    /// Append an IBusAttribute to IBusAttrList, and increase reference.
    extern fn ibus_attr_list_append(p_attr_list: *AttrList, p_attr: *ibus.Attribute) void;
    pub const append = ibus_attr_list_append;

    /// Returns `ibus.Attribute` at given index. Borrowed reference.
    extern fn ibus_attr_list_get(p_attr_list: *AttrList, p_index: c_uint) *ibus.Attribute;
    pub const get = ibus_attr_list_get;

    extern fn ibus_attr_list_get_type() usize;
    pub const getGObjectType = ibus_attr_list_get_type;

    extern fn g_object_ref(p_self: *ibus.AttrList) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.AttrList) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AttrList, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusAttribute represents an attribute that associate to IBusText.
/// It decorates preedit buffer and auxiliary text with underline, foreground
/// and background colors.
pub const Attribute = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.AttributeClass;
    f_parent: ibus.Serializable,
    /// IBusAttributeType
    f_type: c_uint,
    /// Value for the type.
    f_value: c_uint,
    /// The starting index, inclusive.
    f_start_index: c_uint,
    /// The ending index, exclusive.
    f_end_index: c_uint,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new IBusAttribute.
    extern fn ibus_attribute_new(p_type: c_uint, p_value: c_uint, p_start_index: c_uint, p_end_index: c_uint) *ibus.Attribute;
    pub const new = ibus_attribute_new;

    /// Gets an enum of `ibus.AttrType`.
    extern fn ibus_attribute_get_attr_type(p_attr: *Attribute) c_uint;
    pub const getAttrType = ibus_attribute_get_attr_type;

    /// Gets an end unsigned index
    extern fn ibus_attribute_get_end_index(p_attr: *Attribute) c_uint;
    pub const getEndIndex = ibus_attribute_get_end_index;

    /// Gets a start unsigned index
    extern fn ibus_attribute_get_start_index(p_attr: *Attribute) c_uint;
    pub const getStartIndex = ibus_attribute_get_start_index;

    /// Gets an unsigned int value relative with `ibus.AttrType`.
    /// If the type is `IBUS_ATTR_TYPE_UNDERLINE`, the return value is
    /// `ibus.AttrUnderline`. If the type is `IBUS_ATTR_TYPE_FOREGROUND`,
    /// the return value is the color RGB.
    extern fn ibus_attribute_get_value(p_attr: *Attribute) c_uint;
    pub const getValue = ibus_attribute_get_value;

    extern fn ibus_attribute_get_type() usize;
    pub const getGObjectType = ibus_attribute_get_type;

    extern fn g_object_ref(p_self: *ibus.Attribute) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Attribute) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Attribute, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusBus connects with IBus daemon.
pub const Bus = extern struct {
    pub const Parent = ibus.Object;
    pub const Implements = [_]type{};
    pub const Class = ibus.BusClass;
    f_parent: ibus.Object,
    f_priv: ?*ibus.BusPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the `ibus.Bus` object is for client use only.
        pub const client_only = struct {
            pub const name = "client-only";

            pub const Type = c_int;
        };

        /// Whether the `ibus.Bus` object should connect asynchronously to the bus.
        pub const connect_async = struct {
            pub const name = "connect-async";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when `ibus.Bus` is connected to ibus-daemon.
        pub const connected = struct {
            pub const name = "connected";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Bus, p_instance))),
                    gobject.signalLookup("connected", Bus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when `ibus.Bus` is disconnected from ibus-daemon.
        pub const disconnected = struct {
            pub const name = "disconnected";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Bus, p_instance))),
                    gobject.signalLookup("disconnected", Bus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when global engine is changed.
        pub const global_engine_changed = struct {
            pub const name = "global-engine-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_name: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Bus, p_instance))),
                    gobject.signalLookup("global-engine-changed", Bus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when D-Bus name owner is changed.
        pub const name_owner_changed = struct {
            pub const name = "name-owner-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_name: [*:0]u8, p_old_owner: [*:0]u8, p_new_owner: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Bus, p_instance))),
                    gobject.signalLookup("name-owner-changed", Bus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `ibus.Bus` instance.
    extern fn ibus_bus_new() *ibus.Bus;
    pub const new = ibus_bus_new;

    /// Creates a new `ibus.Bus` instance. The instance will asynchronously connect
    /// to the IBus daemon.
    extern fn ibus_bus_new_async() *ibus.Bus;
    pub const newAsync = ibus_bus_new_async;

    /// Creates a new `ibus.Bus` instance for client use only. It will possibly
    /// be limited in what it can do.
    ///
    /// The instance will asynchronously connect to the IBus daemon.
    extern fn ibus_bus_new_async_client() *ibus.Bus;
    pub const newAsyncClient = ibus_bus_new_async_client;

    /// Add a match rule to an `ibus.Bus` synchronously.
    extern fn ibus_bus_add_match(p_bus: *Bus, p_rule: [*:0]const u8) c_int;
    pub const addMatch = ibus_bus_add_match;

    /// Add a match rule to an `ibus.Bus` asynchronously.
    extern fn ibus_bus_add_match_async(p_bus: *Bus, p_rule: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const addMatchAsync = ibus_bus_add_match_async;

    /// Finishes an operation started with `ibus.Bus.addMatchAsync`.
    extern fn ibus_bus_add_match_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const addMatchAsyncFinish = ibus_bus_add_match_async_finish;

    /// Create an input context for client synchronously.
    extern fn ibus_bus_create_input_context(p_bus: *Bus, p_client_name: [*:0]const u8) *ibus.InputContext;
    pub const createInputContext = ibus_bus_create_input_context;

    /// Create an input context for client asynchronously.
    extern fn ibus_bus_create_input_context_async(p_bus: *Bus, p_client_name: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const createInputContextAsync = ibus_bus_create_input_context_async;

    /// Finishes an operation started with `ibus.Bus.createInputContextAsync`.
    extern fn ibus_bus_create_input_context_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.InputContext;
    pub const createInputContextAsyncFinish = ibus_bus_create_input_context_async_finish;

    /// Get the current focused input context synchronously.
    extern fn ibus_bus_current_input_context(p_bus: *Bus) [*:0]u8;
    pub const currentInputContext = ibus_bus_current_input_context;

    /// Get the current focused input context asynchronously.
    extern fn ibus_bus_current_input_context_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const currentInputContextAsync = ibus_bus_current_input_context_async;

    /// Finishes an operation started with `ibus.Bus.currentInputContextAsync`.
    extern fn ibus_bus_current_input_context_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const currentInputContextAsyncFinish = ibus_bus_current_input_context_async_finish;

    /// Exit or restart ibus-daemon synchronously.
    extern fn ibus_bus_exit(p_bus: *Bus, p_restart: c_int) c_int;
    pub const exit = ibus_bus_exit;

    /// Exit or restart ibus-daemon asynchronously.
    extern fn ibus_bus_exit_async(p_bus: *Bus, p_restart: c_int, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const exitAsync = ibus_bus_exit_async;

    /// Finishes an operation started with `ibus.Bus.exitAsync`.
    extern fn ibus_bus_exit_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const exitAsyncFinish = ibus_bus_exit_async_finish;

    /// Get the config instance from `ibus.Bus`.
    extern fn ibus_bus_get_config(p_bus: *Bus) *ibus.Config;
    pub const getConfig = ibus_bus_get_config;

    /// Gets a `gio.DBusConnection` of an `ibus.Bus` instance.
    extern fn ibus_bus_get_connection(p_bus: *Bus) *gio.DBusConnection;
    pub const getConnection = ibus_bus_get_connection;

    /// Get engines by given names synchronously. If some engine names do not exist,
    /// this function will simply ignore them, and return rest of engines.
    /// TODO(penghuang): add asynchronous version
    extern fn ibus_bus_get_engines_by_names(p_bus: *Bus, p_names: [*]const [*:0]const u8) [*]*ibus.EngineDesc;
    pub const getEnginesByNames = ibus_bus_get_engines_by_names;

    /// Get the description of current global engine synchronously.
    extern fn ibus_bus_get_global_engine(p_bus: *Bus) *ibus.EngineDesc;
    pub const getGlobalEngine = ibus_bus_get_global_engine;

    /// Get the description of current global engine asynchronously.
    extern fn ibus_bus_get_global_engine_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getGlobalEngineAsync = ibus_bus_get_global_engine_async;

    /// Finishes an operation started with `ibus.Bus.getGlobalEngineAsyncFinish`.
    extern fn ibus_bus_get_global_engine_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.EngineDesc;
    pub const getGlobalEngineAsyncFinish = ibus_bus_get_global_engine_async_finish;

    /// Get org.freedesktop.DBus.Properties.
    extern fn ibus_bus_get_ibus_property(p_bus: *Bus, p_property_name: [*:0]const u8) *glib.Variant;
    pub const getIbusProperty = ibus_bus_get_ibus_property;

    /// Get org.freedesktop.DBus.Properties asynchronously.
    extern fn ibus_bus_get_ibus_property_async(p_bus: *Bus, p_property_name: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getIbusPropertyAsync = ibus_bus_get_ibus_property_async;

    /// Finishes an operation started with `ibus.Bus.getIbusPropertyAsync`.
    extern fn ibus_bus_get_ibus_property_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const getIbusPropertyAsyncFinish = ibus_bus_get_ibus_property_async_finish;

    /// Return the name owner synchronously.
    extern fn ibus_bus_get_name_owner(p_bus: *Bus, p_name: [*:0]const u8) [*:0]u8;
    pub const getNameOwner = ibus_bus_get_name_owner;

    /// Return the name owner asynchronously.
    extern fn ibus_bus_get_name_owner_async(p_bus: *Bus, p_name: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getNameOwnerAsync = ibus_bus_get_name_owner_async;

    /// Finishes an operation started with `ibus.Bus.getNameOwnerAsync`.
    extern fn ibus_bus_get_name_owner_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getNameOwnerAsyncFinish = ibus_bus_get_name_owner_async_finish;

    /// Return the main service name to use for calls on the ibus connection.
    extern fn ibus_bus_get_service_name(p_bus: *Bus) [*:0]const u8;
    pub const getServiceName = ibus_bus_get_service_name;

    /// Check if the bus's "use_global_engine" option is enabled or not
    /// synchronously.
    extern fn ibus_bus_get_use_global_engine(p_bus: *Bus) c_int;
    pub const getUseGlobalEngine = ibus_bus_get_use_global_engine;

    /// Check if the bus's "use_global_engine" option is enabled or not asynchronously.
    extern fn ibus_bus_get_use_global_engine_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getUseGlobalEngineAsync = ibus_bus_get_use_global_engine_async;

    /// Finishes an operation started with `ibus.Bus.getUseGlobalEngineAsync`.
    extern fn ibus_bus_get_use_global_engine_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const getUseGlobalEngineAsyncFinish = ibus_bus_get_use_global_engine_async_finish;

    /// Check if the bus's "use_sys_layout" option is enabled or not synchronously.
    extern fn ibus_bus_get_use_sys_layout(p_bus: *Bus) c_int;
    pub const getUseSysLayout = ibus_bus_get_use_sys_layout;

    /// Check if the bus's "use_sys_layout" option is enabled or not asynchronously.
    extern fn ibus_bus_get_use_sys_layout_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getUseSysLayoutAsync = ibus_bus_get_use_sys_layout_async;

    /// Finishes an operation started with `ibus.Bus.getUseSysLayoutAsync`.
    extern fn ibus_bus_get_use_sys_layout_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const getUseSysLayoutAsyncFinish = ibus_bus_get_use_sys_layout_async_finish;

    /// This function sends a "HELLO" message to DBus daemon,
    /// which replies the unique name of current IBus process.
    extern fn ibus_bus_hello(p_bus: *Bus) [*:0]const u8;
    pub const hello = ibus_bus_hello;

    /// Return `TRUE` if `bus` is connected to IBus daemon.
    extern fn ibus_bus_is_connected(p_bus: *Bus) c_int;
    pub const isConnected = ibus_bus_is_connected;

    /// Check if the current global engine is enabled or not synchronously.
    extern fn ibus_bus_is_global_engine_enabled(p_bus: *Bus) c_int;
    pub const isGlobalEngineEnabled = ibus_bus_is_global_engine_enabled;

    /// Check if the current global engine is enabled or not asynchronously.
    extern fn ibus_bus_is_global_engine_enabled_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const isGlobalEngineEnabledAsync = ibus_bus_is_global_engine_enabled_async;

    /// Finishes an operation started with `ibus.Bus.isGlobalEngineEnabledAsync`.
    extern fn ibus_bus_is_global_engine_enabled_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const isGlobalEngineEnabledAsyncFinish = ibus_bus_is_global_engine_enabled_async_finish;

    /// List active engines synchronously.
    extern fn ibus_bus_list_active_engines(p_bus: *Bus) *glib.List;
    pub const listActiveEngines = ibus_bus_list_active_engines;

    /// List active engines asynchronously.
    extern fn ibus_bus_list_active_engines_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const listActiveEnginesAsync = ibus_bus_list_active_engines_async;

    /// Finishes an operation started with `ibus.Bus.listActiveEnginesAsync`.
    extern fn ibus_bus_list_active_engines_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const listActiveEnginesAsyncFinish = ibus_bus_list_active_engines_async_finish;

    /// List engines synchronously.
    extern fn ibus_bus_list_engines(p_bus: *Bus) *glib.List;
    pub const listEngines = ibus_bus_list_engines;

    /// List engines asynchronously.
    extern fn ibus_bus_list_engines_async(p_bus: *Bus, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const listEnginesAsync = ibus_bus_list_engines_async;

    /// Finishes an operation started with `ibus.Bus.listEnginesAsync`.
    extern fn ibus_bus_list_engines_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const listEnginesAsyncFinish = ibus_bus_list_engines_async_finish;

    /// Return lists that attached to `bus`.
    /// <note><para>[FixMe] Not implemented yet, only return `NULL`.</para></note>
    /// <note><para>[FixMe] Add async version.</para></note>
    extern fn ibus_bus_list_names(p_bus: *Bus) *glib.List;
    pub const listNames = ibus_bus_list_names;

    /// Lists the unique bus names of connections currently queued for a bus name.
    /// FIXME add an asynchronous version.
    extern fn ibus_bus_list_queued_owners(p_bus: *Bus, p_name: [*:0]const u8) *glib.List;
    pub const listQueuedOwners = ibus_bus_list_queued_owners;

    /// Checks whether the name has owner synchronously.
    extern fn ibus_bus_name_has_owner(p_bus: *Bus, p_name: [*:0]const u8) c_int;
    pub const nameHasOwner = ibus_bus_name_has_owner;

    /// Checks whether the name has owner asynchronously.
    extern fn ibus_bus_name_has_owner_async(p_bus: *Bus, p_name: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const nameHasOwnerAsync = ibus_bus_name_has_owner_async;

    /// Finishes an operation started with `ibus.Bus.nameHasOwnerAsync`.
    extern fn ibus_bus_name_has_owner_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const nameHasOwnerAsyncFinish = ibus_bus_name_has_owner_async_finish;

    /// Start bus components by engine names synchronously.
    extern fn ibus_bus_preload_engines(p_bus: *Bus, p_names: [*]const [*:0]const u8) c_int;
    pub const preloadEngines = ibus_bus_preload_engines;

    /// Start bus components by engine names asynchronously.
    extern fn ibus_bus_preload_engines_async(p_bus: *Bus, p_names: [*]const [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const preloadEnginesAsync = ibus_bus_preload_engines_async;

    /// Finishes an operation started with `ibus.Bus.preloadEnginesAsync`.
    extern fn ibus_bus_preload_engines_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const preloadEnginesAsyncFinish = ibus_bus_preload_engines_async_finish;

    /// Register a component to an `ibus.Bus` synchronously.
    extern fn ibus_bus_register_component(p_bus: *Bus, p_component: *ibus.Component) c_int;
    pub const registerComponent = ibus_bus_register_component;

    /// Register a component to an `ibus.Bus` asynchronously.
    extern fn ibus_bus_register_component_async(p_bus: *Bus, p_component: *ibus.Component, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const registerComponentAsync = ibus_bus_register_component_async;

    /// Finishes an operation started with `ibus.Bus.registerComponentAsync`.
    extern fn ibus_bus_register_component_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const registerComponentAsyncFinish = ibus_bus_register_component_async_finish;

    /// Release a name to IBus daemon synchronously.
    extern fn ibus_bus_release_name(p_bus: *Bus, p_name: [*:0]const u8) c_uint;
    pub const releaseName = ibus_bus_release_name;

    /// Release a name to IBus daemon asynchronously.
    extern fn ibus_bus_release_name_async(p_bus: *Bus, p_name: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const releaseNameAsync = ibus_bus_release_name_async;

    /// Finishes an operation started with `ibus.Bus.releaseNameAsync`.
    extern fn ibus_bus_release_name_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_uint;
    pub const releaseNameAsyncFinish = ibus_bus_release_name_async_finish;

    /// Remove a match rule to an `ibus.Bus` synchronously.
    extern fn ibus_bus_remove_match(p_bus: *Bus, p_rule: [*:0]const u8) c_int;
    pub const removeMatch = ibus_bus_remove_match;

    /// Remove a match rule to an IBusBus asynchronously.
    extern fn ibus_bus_remove_match_async(p_bus: *Bus, p_rule: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const removeMatchAsync = ibus_bus_remove_match_async;

    /// Finishes an operation started with `ibus.Bus.removeMatchAsync`.
    extern fn ibus_bus_remove_match_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const removeMatchAsyncFinish = ibus_bus_remove_match_async_finish;

    /// Request a name from IBus daemon synchronously.
    extern fn ibus_bus_request_name(p_bus: *Bus, p_name: [*:0]const u8, p_flags: u32) u32;
    pub const requestName = ibus_bus_request_name;

    /// Request a name from IBus daemon asynchronously.
    extern fn ibus_bus_request_name_async(p_bus: *Bus, p_name: [*:0]const u8, p_flags: c_uint, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const requestNameAsync = ibus_bus_request_name_async;

    /// Finishes an operation started with `ibus.Bus.requestNameAsync`.
    extern fn ibus_bus_request_name_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_uint;
    pub const requestNameAsyncFinish = ibus_bus_request_name_async_finish;

    /// Set current global engine synchronously.
    extern fn ibus_bus_set_global_engine(p_bus: *Bus, p_global_engine: [*:0]const u8) c_int;
    pub const setGlobalEngine = ibus_bus_set_global_engine;

    /// Set current global engine asynchronously.
    extern fn ibus_bus_set_global_engine_async(p_bus: *Bus, p_global_engine: [*:0]const u8, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setGlobalEngineAsync = ibus_bus_set_global_engine_async;

    /// Finishes an operation started with `ibus.Bus.setGlobalEngineAsync`.
    extern fn ibus_bus_set_global_engine_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setGlobalEngineAsyncFinish = ibus_bus_set_global_engine_async_finish;

    /// Set org.freedesktop.DBus.Properties.
    extern fn ibus_bus_set_ibus_property(p_bus: *Bus, p_property_name: [*:0]const u8, p_value: *glib.Variant) void;
    pub const setIbusProperty = ibus_bus_set_ibus_property;

    /// Set org.freedesktop.DBus.Properties asynchronously.
    extern fn ibus_bus_set_ibus_property_async(p_bus: *Bus, p_property_name: [*:0]const u8, p_value: *glib.Variant, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setIbusPropertyAsync = ibus_bus_set_ibus_property_async;

    /// Finishes an operation started with `ibus.Bus.setIbusPropertyAsync`.
    extern fn ibus_bus_set_ibus_property_async_finish(p_bus: *Bus, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setIbusPropertyAsyncFinish = ibus_bus_set_ibus_property_async_finish;

    /// Start or stop watching the NameOwnerChanged DBus signal.
    extern fn ibus_bus_set_watch_dbus_signal(p_bus: *Bus, p_watch: c_int) void;
    pub const setWatchDbusSignal = ibus_bus_set_watch_dbus_signal;

    /// Start or stop watching the GlobalEngineChanged IBus signal.
    extern fn ibus_bus_set_watch_ibus_signal(p_bus: *Bus, p_watch: c_int) void;
    pub const setWatchIbusSignal = ibus_bus_set_watch_ibus_signal;

    extern fn ibus_bus_get_type() usize;
    pub const getGObjectType = ibus_bus_get_type;

    extern fn g_object_ref(p_self: *ibus.Bus) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Bus) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Bus, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusComponent is an executable program.
/// It provides services such as user interface, configuration,
/// and input method engine (IME).
///
/// It is recommended that IME developers provide
/// a component XML file and
/// load the XML file by `ibus.Component.newFromFile`.
///
/// The format of a component XML file is described  at
/// <ulink url="https://github.com/ibus/ibus/wiki/DevXML">https://github.com/ibus/ibus/wiki/DevXML</ulink>
pub const Component = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.ComponentClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.ComponentPrivate,
    f_pdummy: [7]*anyopaque,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The author of component
        pub const author = struct {
            pub const name = "author";

            pub const Type = ?[*:0]u8;
        };

        /// The exec path of component
        pub const command_line = struct {
            pub const name = "command-line";

            pub const Type = ?[*:0]u8;
        };

        /// The description of component
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The homepage of component
        pub const homepage = struct {
            pub const name = "homepage";

            pub const Type = ?[*:0]u8;
        };

        /// The license of component
        pub const license = struct {
            pub const name = "license";

            pub const Type = ?[*:0]u8;
        };

        /// The name of component
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The textdomain of component
        pub const textdomain = struct {
            pub const name = "textdomain";

            pub const Type = ?[*:0]u8;
        };

        /// The version of component
        pub const version = struct {
            pub const name = "version";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `ibus.Component`.
    extern fn ibus_component_new(p_name: [*:0]const u8, p_description: [*:0]const u8, p_version: [*:0]const u8, p_license: [*:0]const u8, p_author: [*:0]const u8, p_homepage: [*:0]const u8, p_command_line: [*:0]const u8, p_textdomain: [*:0]const u8) *ibus.Component;
    pub const new = ibus_component_new;

    /// Creates a new `ibus.Component` from an XML file.
    /// Note that a component file usually contains engine descriptions,
    /// if it does, `ibus.EngineDesc.newFromXmlNode` will be called
    /// to load the engine descriptions.
    extern fn ibus_component_new_from_file(p_filename: [*:0]const u8) *ibus.Component;
    pub const newFromFile = ibus_component_new_from_file;

    /// Creates a new `ibus.Component` from an XML tree.
    extern fn ibus_component_new_from_xml_node(p_node: *ibus.XML) *ibus.Component;
    pub const newFromXmlNode = ibus_component_new_from_xml_node;

    /// Creates a new `ibus.Component`.
    /// `ibus.Component.newVarargs` supports the va_list format.
    /// name property is required. e.g.
    /// IBusComponent *component = ibus_component_new_varargs ("name", "ibus-foo",
    ///                                                        "command_line", "/usr/libexec/ibus-engine-foo --ibus",
    ///                                                     NULL)
    extern fn ibus_component_new_varargs(p_first_property_name: [*:0]const u8, ...) *ibus.Component;
    pub const newVarargs = ibus_component_new_varargs;

    /// Add an engine to `ibus.Component` according to the description in `engine`.
    extern fn ibus_component_add_engine(p_component: *Component, p_engine: *ibus.EngineDesc) void;
    pub const addEngine = ibus_component_add_engine;

    /// Add an observed path to `ibus.Component`.
    extern fn ibus_component_add_observed_path(p_component: *Component, p_path: [*:0]const u8, p_access_fs: c_int) void;
    pub const addObservedPath = ibus_component_add_observed_path;

    /// Check whether the observed paths of component is modified.
    extern fn ibus_component_check_modification(p_component: *Component) c_int;
    pub const checkModification = ibus_component_check_modification;

    /// Gets the author property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_author(p_component: *Component) [*:0]const u8;
    pub const getAuthor = ibus_component_get_author;

    /// Gets the description property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_description(p_component: *Component) [*:0]const u8;
    pub const getDescription = ibus_component_get_description;

    /// Gets the engines of this component.
    extern fn ibus_component_get_engines(p_component: *Component) *glib.List;
    pub const getEngines = ibus_component_get_engines;

    /// Gets the exec property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_exec(p_component: *Component) [*:0]const u8;
    pub const getExec = ibus_component_get_exec;

    /// Gets the homepage property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_homepage(p_component: *Component) [*:0]const u8;
    pub const getHomepage = ibus_component_get_homepage;

    /// Gets the license property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_license(p_component: *Component) [*:0]const u8;
    pub const getLicense = ibus_component_get_license;

    /// Gets the name property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_name(p_component: *Component) [*:0]const u8;
    pub const getName = ibus_component_get_name;

    /// Gets the observed paths of this component.
    extern fn ibus_component_get_observed_paths(p_component: *Component) *glib.List;
    pub const getObservedPaths = ibus_component_get_observed_paths;

    /// Gets the textdomain property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_textdomain(p_component: *Component) [*:0]const u8;
    pub const getTextdomain = ibus_component_get_textdomain;

    /// Gets the version property in `ibus.Component`. It should not be freed.
    extern fn ibus_component_get_version(p_component: *Component) [*:0]const u8;
    pub const getVersion = ibus_component_get_version;

    /// Output `ibus.Component` as an XML-formatted string.
    /// The output string can be then shown on the screen or written to file.
    extern fn ibus_component_output(p_component: *Component, p_output: *glib.String, p_indent: c_int) void;
    pub const output = ibus_component_output;

    /// Output engine description  as an XML-formatted string.
    /// The output string can be then shown on the screen or written to file.
    extern fn ibus_component_output_engines(p_component: *Component, p_output: *glib.String, p_indent: c_int) void;
    pub const outputEngines = ibus_component_output_engines;

    extern fn ibus_component_get_type() usize;
    pub const getGObjectType = ibus_component_get_type;

    extern fn g_object_ref(p_self: *ibus.Component) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Component) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Component, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusConfig provides engine configuration methods
/// such as get and set the configure settings to configuration file.
///
/// Currently, IBusConfig supports gconf.
pub const Config = extern struct {
    pub const Parent = ibus.Proxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = ibus.ConfigClass;
    f_parent: ibus.Proxy,
    f_priv: ?*ibus.ConfigPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when configuration value is changed.
        /// <note><para>Argument `user_data` is ignored in this function.</para></note>
        pub const value_changed = struct {
            pub const name = "value-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_section: [*:0]u8, p_name: [*:0]u8, p_value: *glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Config, p_instance))),
                    gobject.signalLookup("value-changed", Config.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// New an `ibus.Config` asynchronously.
    extern fn ibus_config_new_async(p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newAsync = ibus_config_new_async;

    /// Create a new `ibus.Config` from existing `gio.DBusConnection`.
    extern fn ibus_config_new(p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*ibus.Config;
    pub const new = ibus_config_new;

    /// Finishes an operation started with `ibus.Config.newAsync`.
    extern fn ibus_config_new_async_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.Config;
    pub const newAsyncFinish = ibus_config_new_async_finish;

    /// Get the value of a configuration option synchronously.
    ///
    /// GConf stores configure options in a tree-like structure,
    /// and the IBus related setting is at /desktop/ibus,
    /// thus, `section` here is a path from there,
    /// while `name` is the key of that configuration option.
    ///
    /// ibus-chewing, for example, stores its setting in /desktop/ibus/engine/Chewing,
    /// so the section name for it is "engine/Chewing".
    /// See also: `ibus.Config.setValue`.
    extern fn ibus_config_get_value(p_config: *Config, p_section: [*:0]const u8, p_name: [*:0]const u8) *glib.Variant;
    pub const getValue = ibus_config_get_value;

    /// Get the value of a configuration option asynchronously.
    ///
    /// See also: `ibus.Config.getValue`.
    extern fn ibus_config_get_value_async(p_config: *Config, p_section: [*:0]const u8, p_name: [*:0]const u8, p_timeout_ms: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getValueAsync = ibus_config_get_value_async;

    /// Finish get value of a configuration option.
    extern fn ibus_config_get_value_async_finish(p_config: *Config, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const getValueAsyncFinish = ibus_config_get_value_async_finish;

    /// Get all values in a section synchronously.
    extern fn ibus_config_get_values(p_config: *Config, p_section: [*:0]const u8) *glib.Variant;
    pub const getValues = ibus_config_get_values;

    /// Get all values in a section asynchronously.
    ///
    /// See also: `ibus.Config.getValues`.
    extern fn ibus_config_get_values_async(p_config: *Config, p_section: [*:0]const u8, p_timeout_ms: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getValuesAsync = ibus_config_get_values_async;

    /// Finish get values in a section.
    extern fn ibus_config_get_values_async_finish(p_config: *Config, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const getValuesAsyncFinish = ibus_config_get_values_async_finish;

    /// Set the value of a configuration option synchronously.
    extern fn ibus_config_set_value(p_config: *Config, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant) c_int;
    pub const setValue = ibus_config_set_value;

    /// Set the value of a configuration option asynchronously.
    ///
    /// See also: `ibus.Config.setValue`.
    extern fn ibus_config_set_value_async(p_config: *Config, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant, p_timeout_ms: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setValueAsync = ibus_config_set_value_async;

    /// Finish set value of a configuration option.
    extern fn ibus_config_set_value_async_finish(p_config: *Config, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setValueAsyncFinish = ibus_config_set_value_async_finish;

    /// Remove an entry of a configuration option.
    extern fn ibus_config_unset(p_config: *Config, p_section: [*:0]const u8, p_name: [*:0]const u8) c_int;
    pub const unset = ibus_config_unset;

    /// Unsubscribe from the configuration option change notification.
    extern fn ibus_config_unwatch(p_config: *Config, p_section: ?[*:0]const u8, p_name: ?[*:0]const u8) c_int;
    pub const unwatch = ibus_config_unwatch;

    /// Subscribe to the configuration option change notification.
    ///
    /// Until this function is called, every change will be notified to the
    /// client through `ibus.Config.signals.value`-changed signal.  Clients should
    /// call `ibus.Config.watch` with the sections they are interested in,
    /// to reduce the number of D-Bus messages.
    extern fn ibus_config_watch(p_config: *Config, p_section: ?[*:0]const u8, p_name: ?[*:0]const u8) c_int;
    pub const watch = ibus_config_watch;

    extern fn ibus_config_get_type() usize;
    pub const getGObjectType = ibus_config_get_type;

    extern fn g_object_ref(p_self: *ibus.Config) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Config) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Config, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusConfigService is a base class for other configuration services such as GConf.
/// Currently, directly known sub class is IBusConfigGConf.
///
/// IBusConfigServiceClass has following member functions:
/// <itemizedlist>
///     <listitem>
///         <para>gboolean set_value(IBusConfigService *config, const gchar *section, const gchar *name,
///             const GValue *value, IBusError **error)
///         </para>
///         <variablelist>
///             <varlistentry>
///                 <term>config:</term>
///                 <listitem>A configure service</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>section:</term>
///                 <listitem>Section name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>name:</term>
///                 <listitem>Name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>value:</term>
///                 <listitem>GValue that holds the value.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>error:</term>
///                 <listitem>Error outputs here.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>Returns:</term>
///                 <listitem>TRUE if succeed; FALSE otherwise.</listitem>
///             </varlistentry>
///         </variablelist>
///         <para>Set a value to a configuration option.
///         </para>
///     </listitem>
///     <listitem>
///         <para>gboolean get_value(IBusConfigService *config, const gchar *section, const gchar *name,
///             GValue *value, IBusError **error)
///         </para>
///         <variablelist>
///             <varlistentry>
///                 <term>config:</term>
///                 <listitem>A configure service</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>section:</term>
///                 <listitem>Section name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>name:</term>
///                 <listitem>Name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>value:</term>
///                 <listitem>GValue that holds the value.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>error:</term>
///                 <listitem>Error outputs here.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>Returns:</term>
///                 <listitem>TRUE if succeed; FALSE otherwise.</listitem>
///             </varlistentry>
///        </variablelist>
///        <para>Get value of a configuration option.
///        </para>
///     </listitem>
///     <listitem>
///         <para>gboolean unset(IBusConfigService *config, const gchar *section, const gchar *name,
///             IBusError **error)
///         </para>
///         <variablelist>
///             <varlistentry>
///                 <term>config:</term>
///                 <listitem>A configure service</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>section:</term>
///                 <listitem>Section name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>name:</term>
///                 <listitem>Name of the configuration option.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>error:</term>
///                 <listitem>Error outputs here.</listitem>
///             </varlistentry>
///             <varlistentry>
///                 <term>Returns:</term>
///                 <listitem>TRUE if succeed; FALSE otherwise.</listitem>
///             </varlistentry>
///         </variablelist>
///         <para>Remove an entry to a configuration option.
///         </para>
///     </listitem>
/// </itemizedlist>
pub const ConfigService = extern struct {
    pub const Parent = ibus.Service;
    pub const Implements = [_]type{};
    pub const Class = ibus.ConfigServiceClass;
    f_parent: ibus.Service,

    pub const virtual_methods = struct {
        pub const get_value = struct {
            pub fn call(p_class: anytype, p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.Variant {
                return gobject.ext.as(ConfigService.Class, p_class).f_get_value.?(gobject.ext.as(ConfigService, p_config), p_section, p_name, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(ConfigService.Class, p_class).f_get_value = @ptrCast(p_implementation);
            }
        };

        pub const get_values = struct {
            pub fn call(p_class: anytype, p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.Variant {
                return gobject.ext.as(ConfigService.Class, p_class).f_get_values.?(gobject.ext.as(ConfigService, p_config), p_section, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(ConfigService.Class, p_class).f_get_values = @ptrCast(p_implementation);
            }
        };

        pub const set_value = struct {
            pub fn call(p_class: anytype, p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(ConfigService.Class, p_class).f_set_value.?(gobject.ext.as(ConfigService, p_config), p_section, p_name, p_value, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(ConfigService.Class, p_class).f_set_value = @ptrCast(p_implementation);
            }
        };

        pub const unset_value = struct {
            pub fn call(p_class: anytype, p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(ConfigService.Class, p_class).f_unset_value.?(gobject.ext.as(ConfigService, p_config), p_section, p_name, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_config: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(ConfigService.Class, p_class).f_unset_value = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates an new `ibus.ConfigService` from an `gio.DBusConnection`.
    extern fn ibus_config_service_new(p_connection: *gio.DBusConnection) *ibus.ConfigService;
    pub const new = ibus_config_service_new;

    /// Change a value of a configuration option
    /// by sending a "ValueChanged" message to IBus service.
    extern fn ibus_config_service_value_changed(p_config: *ConfigService, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant) void;
    pub const valueChanged = ibus_config_service_value_changed;

    extern fn ibus_config_service_get_type() usize;
    pub const getGObjectType = ibus_config_service_get_type;

    extern fn g_object_ref(p_self: *ibus.ConfigService) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.ConfigService) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ConfigService, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Emoji data likes emoji unicode, annotations, description, category.
/// You can get extended values with g_object_get_properties.
pub const EmojiData = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.EmojiDataClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.EmojiDataPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The emoji annotations
        pub const annotations = struct {
            pub const name = "annotations";

            pub const Type = ?*anyopaque;
        };

        /// The emoji category
        pub const category = struct {
            pub const name = "category";

            pub const Type = ?[*:0]u8;
        };

        /// The emoji description
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The emoji character
        pub const emoji = struct {
            pub const name = "emoji";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn ibus_emoji_data_load(p_path: [*:0]const u8) *glib.SList;
    pub const load = ibus_emoji_data_load;

    /// Save the list of `ibus.EmojiData` to the cache file.
    extern fn ibus_emoji_data_save(p_path: [*:0]const u8, p_list: *glib.SList) void;
    pub const save = ibus_emoji_data_save;

    /// Creates a new `ibus.EmojiData`.
    /// emoji property is required. e.g.
    /// ibus_emoji_data_new ("emoji", "😁", NULL)
    extern fn ibus_emoji_data_new(p_first_property_name: [*:0]const u8, ...) *ibus.EmojiData;
    pub const new = ibus_emoji_data_new;

    /// Gets the annotation list in `ibus.EmojiData`. It should not be freed.
    extern fn ibus_emoji_data_get_annotations(p_emoji: *EmojiData) *glib.SList;
    pub const getAnnotations = ibus_emoji_data_get_annotations;

    /// Gets the emoji category in `ibus.EmojiData`. It should not be freed.
    extern fn ibus_emoji_data_get_category(p_emoji: *EmojiData) [*:0]const u8;
    pub const getCategory = ibus_emoji_data_get_category;

    /// Gets the emoji description in `ibus.EmojiData`. It should not be freed.
    extern fn ibus_emoji_data_get_description(p_emoji: *EmojiData) [*:0]const u8;
    pub const getDescription = ibus_emoji_data_get_description;

    /// Gets the emoji character in `ibus.EmojiData`. It should not be freed.
    extern fn ibus_emoji_data_get_emoji(p_emoji: *EmojiData) [*:0]const u8;
    pub const getEmoji = ibus_emoji_data_get_emoji;

    /// Sets the annotation list in `ibus.EmojiData`.
    extern fn ibus_emoji_data_set_annotations(p_emoji: *EmojiData, p_annotations: *glib.SList) void;
    pub const setAnnotations = ibus_emoji_data_set_annotations;

    /// Sets the description in `ibus.EmojiData`.
    extern fn ibus_emoji_data_set_description(p_emoji: *EmojiData, p_description: [*:0]const u8) void;
    pub const setDescription = ibus_emoji_data_set_description;

    extern fn ibus_emoji_data_get_type() usize;
    pub const getGObjectType = ibus_emoji_data_get_type;

    extern fn g_object_ref(p_self: *ibus.EmojiData) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.EmojiData) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EmojiData, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusEngine provides infrastructure for input method engine.
/// Developers can "extend" this class for input method engine development.
///
/// see_also: `ibus.Component`, `ibus.EngineDesc`
pub const Engine = extern struct {
    pub const Parent = ibus.Service;
    pub const Implements = [_]type{};
    pub const Class = ibus.EngineClass;
    f_parent: ibus.Service,
    f_priv: ?*ibus.EnginePrivate,
    /// Whether the engine is enabled.
    f_enabled: c_int,
    /// Whether the engine has focus.
    f_has_focus: c_int,
    /// Area of cursor.
    f_cursor_area: ibus.Rectangle,
    /// IBusCapabilite (client capabilities) flags.
    f_client_capabilities: c_uint,

    pub const virtual_methods = struct {
        pub const cancel_hand_writing = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_strokes: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_cancel_hand_writing.?(gobject.ext.as(Engine, p_engine), p_n_strokes);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_strokes: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_cancel_hand_writing = @ptrCast(p_implementation);
            }
        };

        pub const candidate_clicked = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_button: c_uint, p_state: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_candidate_clicked.?(gobject.ext.as(Engine, p_engine), p_index, p_button, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_button: c_uint, p_state: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_candidate_clicked = @ptrCast(p_implementation);
            }
        };

        pub const cursor_down = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_cursor_down.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_cursor_down = @ptrCast(p_implementation);
            }
        };

        pub const cursor_up = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_cursor_up.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_cursor_up = @ptrCast(p_implementation);
            }
        };

        pub const disable = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_disable.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_disable = @ptrCast(p_implementation);
            }
        };

        pub const enable = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_enable.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_enable = @ptrCast(p_implementation);
            }
        };

        pub const focus_in = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_focus_in.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_focus_in = @ptrCast(p_implementation);
            }
        };

        pub const focus_in_id = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_path: [*:0]const u8, p_client: [*:0]const u8) void {
                return gobject.ext.as(Engine.Class, p_class).f_focus_in_id.?(gobject.ext.as(Engine, p_engine), p_object_path, p_client);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_path: [*:0]const u8, p_client: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_focus_in_id = @ptrCast(p_implementation);
            }
        };

        pub const focus_out = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_focus_out.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_focus_out = @ptrCast(p_implementation);
            }
        };

        pub const focus_out_id = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_path: [*:0]const u8) void {
                return gobject.ext.as(Engine.Class, p_class).f_focus_out_id.?(gobject.ext.as(Engine, p_engine), p_object_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_focus_out_id = @ptrCast(p_implementation);
            }
        };

        pub const page_down = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_page_down.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_page_down = @ptrCast(p_implementation);
            }
        };

        pub const page_up = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_page_up.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_page_up = @ptrCast(p_implementation);
            }
        };

        pub const process_hand_writing_event = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_coordinates: *const f64, p_coordinates_len: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_process_hand_writing_event.?(gobject.ext.as(Engine, p_engine), p_coordinates, p_coordinates_len);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_coordinates: *const f64, p_coordinates_len: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_process_hand_writing_event = @ptrCast(p_implementation);
            }
        };

        pub const process_key_event = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) c_int {
                return gobject.ext.as(Engine.Class, p_class).f_process_key_event.?(gobject.ext.as(Engine, p_engine), p_keyval, p_keycode, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) callconv(.C) c_int) void {
                gobject.ext.as(Engine.Class, p_class).f_process_key_event = @ptrCast(p_implementation);
            }
        };

        pub const property_activate = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8, p_prop_state: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_property_activate.?(gobject.ext.as(Engine, p_engine), p_prop_name, p_prop_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8, p_prop_state: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_property_activate = @ptrCast(p_implementation);
            }
        };

        pub const property_hide = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8) void {
                return gobject.ext.as(Engine.Class, p_class).f_property_hide.?(gobject.ext.as(Engine, p_engine), p_prop_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_property_hide = @ptrCast(p_implementation);
            }
        };

        pub const property_show = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8) void {
                return gobject.ext.as(Engine.Class, p_class).f_property_show.?(gobject.ext.as(Engine, p_engine), p_prop_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_name: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_property_show = @ptrCast(p_implementation);
            }
        };

        pub const reset = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Engine.Class, p_class).f_reset.?(gobject.ext.as(Engine, p_engine));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_reset = @ptrCast(p_implementation);
            }
        };

        pub const set_capabilities = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_caps: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_set_capabilities.?(gobject.ext.as(Engine, p_engine), p_caps);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_caps: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_set_capabilities = @ptrCast(p_implementation);
            }
        };

        pub const set_content_type = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_purpose: c_uint, p_hints: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_set_content_type.?(gobject.ext.as(Engine, p_engine), p_purpose, p_hints);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_purpose: c_uint, p_hints: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_set_content_type = @ptrCast(p_implementation);
            }
        };

        pub const set_cursor_location = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) void {
                return gobject.ext.as(Engine.Class, p_class).f_set_cursor_location.?(gobject.ext.as(Engine, p_engine), p_x, p_y, p_w, p_h);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_set_cursor_location = @ptrCast(p_implementation);
            }
        };

        pub const set_surrounding_text = struct {
            pub fn call(p_class: anytype, p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_cursor_index: c_uint, p_anchor_pos: c_uint) void {
                return gobject.ext.as(Engine.Class, p_class).f_set_surrounding_text.?(gobject.ext.as(Engine, p_engine), p_text, p_cursor_index, p_anchor_pos);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_engine: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_cursor_index: c_uint, p_anchor_pos: c_uint) callconv(.C) void) void {
                gobject.ext.as(Engine.Class, p_class).f_set_surrounding_text = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// When this property is set to `TRUE`, "RequireSurroundingText" D-Bus
        /// signal will be called by ibus-daemon on every focus-in/out event, with
        /// no need for the engine to call `ibus.Engine.getSurroundingText`.
        /// This property can only be set at construct time.
        pub const active_surrounding_text = struct {
            pub const name = "active-surrounding-text";

            pub const Type = c_int;
        };

        /// Name of this IBusEngine.
        pub const engine_name = struct {
            pub const name = "engine-name";

            pub const Type = ?[*:0]u8;
        };

        /// Use `ibus.Engine.signals.focus_in_id``focus_in_id`/`focus_out_id` class method insteads of
        /// `focus_in`/`focus_out` class methods when this property is set to `TRUE`.
        /// Otherwise, use `ibus.Engine.signals.focus_in``focus_in`/focus_out class methods.
        /// This property can only be set at construct time.
        ///
        /// See also: IBusEngine::focus-in-id
        pub const has_focus_id = struct {
            pub const name = "has-focus-id";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when a hand writing operation is cancelled.
        /// Implement the member function IBusEngineClass::cancel_hand_writing
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const cancel_hand_writing = struct {
            pub const name = "cancel-hand-writing";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_n_strokes: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("cancel-hand-writing", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when candidate on lookup table is clicked.
        /// Implement the member function IBusEngineClass::candidate_clicked
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const candidate_clicked = struct {
            pub const name = "candidate-clicked";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_index: c_uint, p_button: c_uint, p_state: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("candidate-clicked", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the down cursor button is pressed.
        /// Implement the member function IBusEngineClass::cursor_down
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const cursor_down = struct {
            pub const name = "cursor-down";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("cursor-down", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the up cursor button is pressed.
        /// Implement the member function IBusEngineClass::cursor_up
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const cursor_up = struct {
            pub const name = "cursor-up";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("cursor-up", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the IME is disabled.
        /// Implement the member function IBusEngineClass::disable
        /// in extended class to receive this signal.
        ///
        /// See also:  `ibus.Bus.setGlobalEngine`.
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const disable = struct {
            pub const name = "disable";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("disable", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the IME is enabled.
        /// Implement the member function IBusEngineClass::enable
        /// in extended class to receive this signal.
        ///
        /// See also:  `ibus.Bus.setGlobalEngine`.
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const enable = struct {
            pub const name = "enable";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("enable", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the focus.
        /// Implement the member function IBusEngineClass::focus_in
        /// in extended class to receive this signal.
        ///
        /// See also: `ibus.InputContext.focusIn`
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_in = struct {
            pub const name = "focus-in";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("focus-in", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the focus.
        /// Implement the member function IBusEngineClass::focus_in
        /// in extended class to receive this signal.
        /// `object_path` is a unique id by input context.
        /// `client` indicates a client type:
        /// 'fake':    focus is on desktop background or other programs where no
        ///            input is possible
        /// 'xim':     old X11 programs like xterm, emacs, ...
        ///            GTK3 programs in a Gnome Xorg session when GTK_IM_MODULE
        ///            is unset also use xim
        /// 'gtk-im:&lt;client-name&gt;':  Gtk2 input module is used
        /// 'gtk3-im:&lt;client-name&gt;': Gtk3 input module is used
        /// 'gtk4-im:&lt;client-name&gt;': Gtk4 input module is used
        ///            In case of the Gtk input modules, the name of the
        ///            client is also shown after the “:”, for example
        ///            like 'gtk3-im:firefox', 'gtk4-im:gnome-text-editor', …
        /// 'gnome-shell': Entries handled by gnome-shell
        ///                (like the command line dialog opened with Alt+F2
        ///                or the search field when pressing the Super key.)
        ///                When GTK_IM_MODULE is unset in a Gnome Wayland session
        ///                all programs which would show 'gtk3-im' or 'gtk4-im'
        ///                with GTK_IM_MODULE=ibus then show 'gnome-shell'
        ///                instead.
        /// 'Qt':      Qt4 programs like keepassx-2.0.3 …
        /// 'QIBusInputContext': Qt5 programs like keepassxc-2.7.1, anki-2.1.15
        ///                      telegram-desktop-3.7.3,
        ///
        /// You need to set `ibus.Engine.signals.has`-focus-id property to `TRUE` when you
        /// construct an `ibus.Engine` to use this class method.
        ///
        /// See also: `ibus.InputContext.focusIn`
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_in_id = struct {
            pub const name = "focus-in-id";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object_path: [*:0]u8, p_client: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("focus-in-id", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application  lost the focus.
        /// Implement the member function IBusEngineClass::focus_out
        /// in extended class to receive this signal.
        ///
        /// See also: `ibus.InputContext.focusOut`
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_out = struct {
            pub const name = "focus-out";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("focus-out", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application  lost the focus.
        /// Implement the member function IBusEngineClass::focus_out
        /// in extended class to receive this signal.
        /// `object_path` is a unique id by input context.
        /// You need to set `ibus.Engine.signals.has`-focus-id property to `TRUE` when you
        /// construct an `ibus.Engine` to use this class method.
        ///
        /// See also: `ibus.InputContext.focusOut`
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_out_id = struct {
            pub const name = "focus-out-id";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("focus-out-id", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the page-down button is pressed.
        /// Implement the member function IBusEngineClass::page_down
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const page_down = struct {
            pub const name = "page-down";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("page-down", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the page-up button is pressed.
        /// Implement the member function IBusEngineClass::page_up
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const page_up = struct {
            pub const name = "page-up";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("page-up", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a hand writing operation is cancelled.
        /// Implement the member function IBusEngineClass::cancel_hand_writing
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const process_hand_writing_event = struct {
            pub const name = "process-hand-writing-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_coordinates: ?*anyopaque, p_coordinates_len: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("process-hand-writing-event", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a key event is received.
        /// Implement the member function IBusEngineClass::process_key_event
        /// in extended class to receive this signal.
        /// Both the key symbol and keycode are passed to the member function.
        /// See `ibus.InputContext.processKeyEvent` for further explanation of
        /// key symbol, keycode and which to use.
        pub const process_key_event = struct {
            pub const name = "process-key-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("process-key-event", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a property is activated or change changed.
        /// Implement the member function IBusEngineClass::property_activate
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const property_activate = struct {
            pub const name = "property-activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_name: [*:0]u8, p_state: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("property-activate", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a property is hidden.
        /// Implement the member function IBusEngineClass::property_hide
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const property_hide = struct {
            pub const name = "property-hide";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_name: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("property-hide", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a property is shown.
        /// Implement the member function IBusEngineClass::property_side
        /// in extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const property_show = struct {
            pub const name = "property-show";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_name: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("property-show", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the IME is reset.
        /// Implement the member function IBusEngineClass::reset
        /// in extended class to receive this signal.
        ///
        /// See also:  `ibus.InputContext.reset`.
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const reset = struct {
            pub const name = "reset";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("reset", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application capabilities is set.
        /// Implement the member function IBusEngineClass::set_capabilities
        /// in extended class to receive this signal.
        ///
        /// See also:  `ibus.InputContext.setCapabilities`.
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_capabilities = struct {
            pub const name = "set-capabilities";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_caps: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("set-capabilities", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application content-type (primary
        /// purpose and hints) is set.  The engine could change the
        /// behavior according to the content-type.  Implement the member
        /// function IBusEngineClass::set_content_type
        /// in extended class to receive this signal.
        ///
        /// For example, if the client application wants to restrict input
        /// to numbers, this signal will be emitted with `purpose` set to
        /// `IBUS_INPUT_PURPOSE_NUMBER`, so the engine can switch the input
        /// mode to latin.
        ///
        /// <note><para>Argument `user_data` is ignored in this
        /// function.</para></note>
        pub const set_content_type = struct {
            pub const name = "set-content-type";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_purpose: c_uint, p_hints: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("set-content-type", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the location of IME is set.
        /// Implement the member function IBusEngineClass::set_cursor_location
        /// in extended class to receive this signal.
        ///
        /// See also:  `ibus.InputContext.setCursorLocation`.
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_cursor_location = struct {
            pub const name = "set-cursor-location";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("set-cursor-location", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a surrounding text is set.
        /// Implement the member function IBusEngineClass::set_surrounding_text
        /// in extended class to receive this signal.
        /// If anchor_pos equals to cursor_pos, it means "there are no selection"
        /// or "does not support selection retrival".
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_surrounding_text = struct {
            pub const name = "set-surrounding-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *gobject.Object, p_cursor_pos: c_uint, p_anchor_pos: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Engine, p_instance))),
                    gobject.signalLookup("set-surrounding-text", Engine.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new `ibus.Engine`.
    extern fn ibus_engine_new(p_engine_name: [*:0]const u8, p_object_path: [*:0]const u8, p_connection: *gio.DBusConnection) *ibus.Engine;
    pub const new = ibus_engine_new;

    /// Create a new `ibus.Engine`.
    extern fn ibus_engine_new_with_type(p_engine_type: usize, p_engine_name: [*:0]const u8, p_object_path: [*:0]const u8, p_connection: *gio.DBusConnection) *ibus.Engine;
    pub const newWithType = ibus_engine_new_with_type;

    /// Commit output of input method to IBus client.
    ///
    /// (Note: The text object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_commit_text(p_engine: *Engine, p_text: *ibus.Text) void;
    pub const commitText = ibus_engine_commit_text;

    /// Delete surrounding text.
    extern fn ibus_engine_delete_surrounding_text(p_engine: *Engine, p_offset: c_int, p_nchars: c_uint) void;
    pub const deleteSurroundingText = ibus_engine_delete_surrounding_text;

    /// Forward the key event.
    extern fn ibus_engine_forward_key_event(p_engine: *Engine, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) void;
    pub const forwardKeyEvent = ibus_engine_forward_key_event;

    /// Get content-type (primary purpose and hints) of the current input
    /// context.
    ///
    /// See also: `ibus.Engine.signals.set`-content-type
    extern fn ibus_engine_get_content_type(p_engine: *Engine, p_purpose: ?*c_uint, p_hints: ?*c_uint) void;
    pub const getContentType = ibus_engine_get_content_type;

    /// Return the name of `ibus.Engine`.
    extern fn ibus_engine_get_name(p_engine: *Engine) [*:0]const u8;
    pub const getName = ibus_engine_get_name;

    /// Get surrounding text.
    ///
    /// It is also used to tell the input-context that the engine will
    /// utilize surrounding-text.  In that case, it must be called in
    /// `ibus.Engine.signals.enable` handler, with both `text` and `cursor` set to
    /// `NULL`.
    ///
    /// See also: `ibus.Engine.signals.set`-surrounding-text
    extern fn ibus_engine_get_surrounding_text(p_engine: *Engine, p_text: ?**ibus.Text, p_cursor_pos: ?*c_uint, p_anchor_pos: ?*c_uint) void;
    pub const getSurroundingText = ibus_engine_get_surrounding_text;

    /// Hide the auxiliary bar.
    extern fn ibus_engine_hide_auxiliary_text(p_engine: *Engine) void;
    pub const hideAuxiliaryText = ibus_engine_hide_auxiliary_text;

    /// Hide the lookup table.
    extern fn ibus_engine_hide_lookup_table(p_engine: *Engine) void;
    pub const hideLookupTable = ibus_engine_hide_lookup_table;

    /// Hide the pre-edit buffer.
    extern fn ibus_engine_hide_preedit_text(p_engine: *Engine) void;
    pub const hidePreeditText = ibus_engine_hide_preedit_text;

    /// Register and show properties in language bar.
    ///
    /// (Note: The prop_list object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_register_properties(p_engine: *Engine, p_prop_list: *ibus.PropList) void;
    pub const registerProperties = ibus_engine_register_properties;

    /// Show the auxiliary bar.
    extern fn ibus_engine_show_auxiliary_text(p_engine: *Engine) void;
    pub const showAuxiliaryText = ibus_engine_show_auxiliary_text;

    /// Show the lookup table.
    extern fn ibus_engine_show_lookup_table(p_engine: *Engine) void;
    pub const showLookupTable = ibus_engine_show_lookup_table;

    /// Show the pre-edit buffer.
    extern fn ibus_engine_show_preedit_text(p_engine: *Engine) void;
    pub const showPreeditText = ibus_engine_show_preedit_text;

    /// Update the auxiliary bar.
    ///
    /// (Note: The text object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_auxiliary_text(p_engine: *Engine, p_text: *ibus.Text, p_visible: c_int) void;
    pub const updateAuxiliaryText = ibus_engine_update_auxiliary_text;

    /// Update the lookup table.
    ///
    /// (Note: The table object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_lookup_table(p_engine: *Engine, p_lookup_table: *ibus.LookupTable, p_visible: c_int) void;
    pub const updateLookupTable = ibus_engine_update_lookup_table;

    /// Fast update for big lookup table.
    ///
    /// If size of lookup table is not over table page size *4,
    /// then it calls `ibus.Engine.updateLookupTable`.
    ///
    /// (Note: The table object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_lookup_table_fast(p_engine: *Engine, p_lookup_table: *ibus.LookupTable, p_visible: c_int) void;
    pub const updateLookupTableFast = ibus_engine_update_lookup_table_fast;

    /// Update the pre-edit buffer.
    ///
    /// (Note: The text object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_preedit_text(p_engine: *Engine, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int) void;
    pub const updatePreeditText = ibus_engine_update_preedit_text;

    /// Update the pre-edit buffer with commit mode. Similar to
    /// `ibus.Engine.updatePreeditText`, this function allows users to specify
    /// the behavior on focus out when the pre-edit buffer is visible.
    ///
    /// If `mode` is IBUS_ENGINE_PREEDIT_COMMIT, contents of the pre-edit buffer
    /// will be committed and cleared.
    /// If `mode` is IBUS_ENGINE_PREEDIT_CLEAR, contents of the pre-edit buffer
    /// will be cleared only.
    ///
    /// (Note: The text object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_preedit_text_with_mode(p_engine: *Engine, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int, p_mode: ibus.PreeditFocusMode) void;
    pub const updatePreeditTextWithMode = ibus_engine_update_preedit_text_with_mode;

    /// Update the state displayed in language bar.
    ///
    /// (Note: The prop object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_engine_update_property(p_engine: *Engine, p_prop: *ibus.Property) void;
    pub const updateProperty = ibus_engine_update_property;

    extern fn ibus_engine_get_type() usize;
    pub const getGObjectType = ibus_engine_get_type;

    extern fn g_object_ref(p_self: *ibus.Engine) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Engine) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Engine, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusEngineDesc stores description data of IBusEngine.
/// The description data can either be passed to `ibus.EngineDesc.new`,
/// or loaded from an XML node through `ibus.EngineDesc.newFromXmlNode`
/// to construct IBusEngineDesc.
///
/// However, the recommended way to load engine description data is
/// using `ibus.Component.newFromFile` to load a component file,
/// which also includes engine description data.
///
/// see_also: `ibus.Component`, `ibus.Engine`
pub const EngineDesc = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.EngineDescClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.EngineDescPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The author of engine description
        pub const author = struct {
            pub const name = "author";

            pub const Type = ?[*:0]u8;
        };

        /// The description of engine description
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The hotkeys of engine description
        pub const hotkeys = struct {
            pub const name = "hotkeys";

            pub const Type = ?[*:0]u8;
        };

        /// The icon of engine description
        pub const icon = struct {
            pub const name = "icon";

            pub const Type = ?[*:0]u8;
        };

        /// The key of IBusProperty to change panel icon dynamically.
        pub const icon_prop_key = struct {
            pub const name = "icon-prop-key";

            pub const Type = ?[*:0]u8;
        };

        /// The language of engine description
        pub const language = struct {
            pub const name = "language";

            pub const Type = ?[*:0]u8;
        };

        /// The layout of engine description
        pub const layout = struct {
            pub const name = "layout";

            pub const Type = ?[*:0]u8;
        };

        /// The keyboard option of engine description
        pub const layout_option = struct {
            pub const name = "layout-option";

            pub const Type = ?[*:0]u8;
        };

        /// The keyboard variant of engine description
        pub const layout_variant = struct {
            pub const name = "layout-variant";

            pub const Type = ?[*:0]u8;
        };

        /// The license of engine description
        pub const license = struct {
            pub const name = "license";

            pub const Type = ?[*:0]u8;
        };

        /// The longname of engine description
        pub const longname = struct {
            pub const name = "longname";

            pub const Type = ?[*:0]u8;
        };

        /// The name of engine description
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The rank of engine description
        pub const rank = struct {
            pub const name = "rank";

            pub const Type = c_uint;
        };

        /// The exec lists of the engine setup command
        pub const setup = struct {
            pub const name = "setup";

            pub const Type = ?[*:0]u8;
        };

        /// The symbol chars of engine description instead of icon image
        pub const symbol = struct {
            pub const name = "symbol";

            pub const Type = ?[*:0]u8;
        };

        /// The textdomain of engine description
        pub const textdomain = struct {
            pub const name = "textdomain";

            pub const Type = ?[*:0]u8;
        };

        /// The version number of engine description
        pub const version = struct {
            pub const name = "version";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `ibus.EngineDesc`.
    /// If layout is "default", the engine inherits the current layout and
    /// does not change the layout. The layouts "default" and "" are same.
    /// E.g. If you switch JP XKB engine and an input method engine (IME),
    /// the IME inherits the JP layout.
    extern fn ibus_engine_desc_new(p_name: [*:0]const u8, p_longname: [*:0]const u8, p_description: [*:0]const u8, p_language: [*:0]const u8, p_license: [*:0]const u8, p_author: [*:0]const u8, p_icon: [*:0]const u8, p_layout: [*:0]const u8) *ibus.EngineDesc;
    pub const new = ibus_engine_desc_new;

    /// Creates a new IBusEngineDesc from an XML node.
    /// <note><para>This function is called by `ibus.Component.newFromFile`,
    ///  so developers normally do not need to call it directly.
    /// </para></note>
    extern fn ibus_engine_desc_new_from_xml_node(p_node: *ibus.XML) *ibus.EngineDesc;
    pub const newFromXmlNode = ibus_engine_desc_new_from_xml_node;

    /// Creates a new `ibus.EngineDesc`.
    /// `ibus.EngineDesc.newVarargs` supports the va_list format.
    /// name property is required. e.g.
    /// ibus_engine_desc_new_varargs("name", "ibus-foo", "language", "us", NULL)
    /// If layout is "default", the engine inherits the current layout and
    /// does not change the layout. The layouts "default" and "" are same.
    /// E.g. If you switch JP XKB engine and an input method engine (IME),
    /// the IME inherits the JP layout.
    extern fn ibus_engine_desc_new_varargs(p_first_property_name: [*:0]const u8, ...) *ibus.EngineDesc;
    pub const newVarargs = ibus_engine_desc_new_varargs;

    /// Gets the author property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_author(p_info: *EngineDesc) [*:0]const u8;
    pub const getAuthor = ibus_engine_desc_get_author;

    /// Gets the description property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_description(p_info: *EngineDesc) [*:0]const u8;
    pub const getDescription = ibus_engine_desc_get_description;

    /// Gets the hotkeys property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_hotkeys(p_info: *EngineDesc) [*:0]const u8;
    pub const getHotkeys = ibus_engine_desc_get_hotkeys;

    /// Gets the icon property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_icon(p_info: *EngineDesc) [*:0]const u8;
    pub const getIcon = ibus_engine_desc_get_icon;

    /// Gets the key of IBusProperty to load the panel icon dynamically
    /// in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_icon_prop_key(p_info: *EngineDesc) [*:0]const u8;
    pub const getIconPropKey = ibus_engine_desc_get_icon_prop_key;

    /// Gets the language property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_language(p_info: *EngineDesc) [*:0]const u8;
    pub const getLanguage = ibus_engine_desc_get_language;

    /// Gets the layout property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_layout(p_info: *EngineDesc) [*:0]const u8;
    pub const getLayout = ibus_engine_desc_get_layout;

    /// Gets the keyboard option property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_layout_option(p_info: *EngineDesc) [*:0]const u8;
    pub const getLayoutOption = ibus_engine_desc_get_layout_option;

    /// Gets the keyboard variant property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_layout_variant(p_info: *EngineDesc) [*:0]const u8;
    pub const getLayoutVariant = ibus_engine_desc_get_layout_variant;

    /// Gets the license property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_license(p_info: *EngineDesc) [*:0]const u8;
    pub const getLicense = ibus_engine_desc_get_license;

    /// Gets the longname property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_longname(p_info: *EngineDesc) [*:0]const u8;
    pub const getLongname = ibus_engine_desc_get_longname;

    /// Gets the name property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_name(p_info: *EngineDesc) [*:0]const u8;
    pub const getName = ibus_engine_desc_get_name;

    /// Gets the rank property in IBusEngineDesc.
    extern fn ibus_engine_desc_get_rank(p_info: *EngineDesc) c_uint;
    pub const getRank = ibus_engine_desc_get_rank;

    /// Gets the setup property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_setup(p_info: *EngineDesc) [*:0]const u8;
    pub const getSetup = ibus_engine_desc_get_setup;

    /// Gets the symbol property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_symbol(p_info: *EngineDesc) [*:0]const u8;
    pub const getSymbol = ibus_engine_desc_get_symbol;

    /// Gets the textdomain property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_textdomain(p_info: *EngineDesc) [*:0]const u8;
    pub const getTextdomain = ibus_engine_desc_get_textdomain;

    /// Gets the version property in IBusEngineDesc. It should not be freed.
    extern fn ibus_engine_desc_get_version(p_info: *EngineDesc) [*:0]const u8;
    pub const getVersion = ibus_engine_desc_get_version;

    /// Output XML-formatted input method engine description.
    /// The result will be append to GString specified in `output`.
    extern fn ibus_engine_desc_output(p_info: *EngineDesc, p_output: *glib.String, p_indent: c_int) void;
    pub const output = ibus_engine_desc_output;

    extern fn ibus_engine_desc_get_type() usize;
    pub const getGObjectType = ibus_engine_desc_get_type;

    extern fn g_object_ref(p_self: *ibus.EngineDesc) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.EngineDesc) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EngineDesc, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusEngineSimple provides table-based input method logic.
///
/// see_also: `ibus.Engine`
pub const EngineSimple = extern struct {
    pub const Parent = ibus.Engine;
    pub const Implements = [_]type{};
    pub const Class = ibus.EngineSimpleClass;
    f_parent: ibus.Engine,
    f_priv: ?*ibus.EngineSimplePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Call `ibus.EngineSimple.addTable` internally by locale.
    extern fn ibus_engine_simple_add_compose_file(p_simple: *EngineSimple, p_file: [*:0]const u8) c_int;
    pub const addComposeFile = ibus_engine_simple_add_compose_file;

    /// Adds an additional table to search to the engine. Each row of the table
    /// consists of max_seq_len key symbols followed by two guint16 interpreted as
    /// the high and low words of a gunicode value. Tables are searched starting from
    /// the last added.
    ///
    /// The table must be sorted in dictionary order on the numeric value of the key
    /// symbol fields. (Values beyond the length of the sequence should be zero.)
    extern fn ibus_engine_simple_add_table(p_simple: *EngineSimple, p_data: [*]const u16, p_max_seq_len: c_int, p_n_seqs: c_int) void;
    pub const addTable = ibus_engine_simple_add_table;

    /// Call `ibus.EngineSimple.addTable` internally by locale.
    extern fn ibus_engine_simple_add_table_by_locale(p_simple: *EngineSimple, p_locale: ?[*:0]const u8) c_int;
    pub const addTableByLocale = ibus_engine_simple_add_table_by_locale;

    extern fn ibus_engine_simple_get_type() usize;
    pub const getGObjectType = ibus_engine_simple_get_type;

    extern fn g_object_ref(p_self: *ibus.EngineSimple) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.EngineSimple) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EngineSimple, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// IBusExtensionEvent properties.
pub const ExtensionEvent = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.ExtensionEventClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.ExtensionEventPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// `TRUE` if the extension is enabled in the `ibus.ExtensionEvent`.
        pub const is_enabled = struct {
            pub const name = "is-enabled";

            pub const Type = c_int;
        };

        /// `TRUE` if the `ibus.ExtensionEvent` is called by an extension.
        /// `FALSE` if the `ibus.ExtensionEvent` is called by an active engine or
        /// panel.
        /// If this value is `TRUE`, the event is send to ibus-daemon, an active
        /// engine. If it's `FALSE`, the event is sned to ibus-daemon, panels.
        pub const is_extension = struct {
            pub const name = "is-extension";

            pub const Type = c_int;
        };

        /// Name of the extension in the `ibus.ExtensionEvent`.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// Parameters to enable the extension in the `ibus.ExtensionEvent`.
        pub const params = struct {
            pub const name = "params";

            pub const Type = ?[*:0]u8;
        };

        /// Version of the `ibus.ExtensionEvent`.
        pub const version = struct {
            pub const name = "version";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Create a new `ibus.ExtensionEvent`.
    extern fn ibus_extension_event_new(p_first_property_name: [*:0]const u8, ...) *ibus.ExtensionEvent;
    pub const new = ibus_extension_event_new;

    extern fn ibus_extension_event_get_name(p_event: *ExtensionEvent) [*:0]const u8;
    pub const getName = ibus_extension_event_get_name;

    extern fn ibus_extension_event_get_params(p_event: *ExtensionEvent) [*:0]const u8;
    pub const getParams = ibus_extension_event_get_params;

    extern fn ibus_extension_event_get_version(p_event: *ExtensionEvent) c_uint;
    pub const getVersion = ibus_extension_event_get_version;

    extern fn ibus_extension_event_is_enabled(p_event: *ExtensionEvent) c_int;
    pub const isEnabled = ibus_extension_event_is_enabled;

    extern fn ibus_extension_event_is_extension(p_event: *ExtensionEvent) c_int;
    pub const isExtension = ibus_extension_event_is_extension;

    extern fn ibus_extension_event_get_type() usize;
    pub const getGObjectType = ibus_extension_event_get_type;

    extern fn g_object_ref(p_self: *ibus.ExtensionEvent) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.ExtensionEvent) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ExtensionEvent, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusFactory is an `ibus.Service` that creates input method engine (IME) instance.
/// It provides CreateEngine remote method, which creates an IME instance by name,
/// and returns the D-Bus object path to IBus daemon.
///
/// see_also: `ibus.Engine`
pub const Factory = extern struct {
    pub const Parent = ibus.Service;
    pub const Implements = [_]type{};
    pub const Class = ibus.FactoryClass;
    f_parent: ibus.Service,
    f_priv: ?*ibus.FactoryPrivate,

    pub const virtual_methods = struct {
        /// Creates an `ibus.Engine` with `engine_name`.
        pub const create_engine = struct {
            pub fn call(p_class: anytype, p_factory: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_engine_name: [*:0]const u8) *ibus.Engine {
                return gobject.ext.as(Factory.Class, p_class).f_create_engine.?(gobject.ext.as(Factory, p_factory), p_engine_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_factory: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_engine_name: [*:0]const u8) callconv(.C) *ibus.Engine) void {
                gobject.ext.as(Factory.Class, p_class).f_create_engine = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// The ::create-engine signal is a signal to create IBusEngine
        /// with `engine_name`, which gets emitted when IBusFactory
        /// received CreateEngine dbus method. The callback functions
        /// will be called until a callback returns a non-null object
        /// of IBusEngine.
        pub const create_engine = struct {
            pub const name = "create-engine";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_engine_name: [*:0]u8, P_Data) callconv(.C) ?*ibus.Engine, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Factory, p_instance))),
                    gobject.signalLookup("create-engine", Factory.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `ibus.Factory`.
    extern fn ibus_factory_new(p_connection: *gio.DBusConnection) *ibus.Factory;
    pub const new = ibus_factory_new;

    /// Add an engine to the factory.
    extern fn ibus_factory_add_engine(p_factory: *Factory, p_engine_name: [*:0]const u8, p_engine_type: usize) void;
    pub const addEngine = ibus_factory_add_engine;

    /// Creates an `ibus.Engine` with `engine_name`.
    extern fn ibus_factory_create_engine(p_factory: *Factory, p_engine_name: [*:0]const u8) *ibus.Engine;
    pub const createEngine = ibus_factory_create_engine;

    extern fn ibus_factory_get_type() usize;
    pub const getGObjectType = ibus_factory_get_type;

    extern fn g_object_ref(p_self: *ibus.Factory) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Factory) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Factory, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An opaque data type representing an IBusHotkeyProfile.
pub const HotkeyProfile = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.HotkeyProfileClass;
    f_parent: ibus.Serializable,

    pub const virtual_methods = struct {
        pub const trigger = struct {
            pub fn call(p_class: anytype, p_profile: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: glib.Quark, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(HotkeyProfile.Class, p_class).f_trigger.?(gobject.ext.as(HotkeyProfile, p_profile), p_event, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_profile: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: glib.Quark, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(HotkeyProfile.Class, p_class).f_trigger = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when a hotkey is pressed and the hotkey is in profile.
        /// Implement the member function `trigger` in extended class to receive this signal.
        ///
        /// <note><para>The last parameter, user_data is not actually a valid parameter. It is displayed because of GtkDoc bug.</para></note>
        pub const trigger = struct {
            pub const name = "trigger";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_event: c_uint, p_user_data: ?*anyopaque, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HotkeyProfile, p_instance))),
                    gobject.signalLookup("trigger", HotkeyProfile.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `ibus.HotkeyProfile`.
    extern fn ibus_hotkey_profile_new() *ibus.HotkeyProfile;
    pub const new = ibus_hotkey_profile_new;

    /// Adds a hotkey and its associated event to an `ibus.HotkeyProfile`.
    extern fn ibus_hotkey_profile_add_hotkey(p_profile: *HotkeyProfile, p_keyval: c_uint, p_modifiers: c_uint, p_event: glib.Quark) c_int;
    pub const addHotkey = ibus_hotkey_profile_add_hotkey;

    /// Adds a hotkey and its associated event to an `ibus.HotkeyProfile`.
    /// The hotkey is in string format, such like
    /// <constant>Control+Shift+A</constant>.
    extern fn ibus_hotkey_profile_add_hotkey_from_string(p_profile: *HotkeyProfile, p_str: [*:0]const u8, p_event: glib.Quark) c_int;
    pub const addHotkeyFromString = ibus_hotkey_profile_add_hotkey_from_string;

    /// Emits a <constant>::trigger</constant> signal when a hotkey is in a profile.
    extern fn ibus_hotkey_profile_filter_key_event(p_profile: *HotkeyProfile, p_keyval: c_uint, p_modifiers: c_uint, p_prev_keyval: c_uint, p_prev_modifiers: c_uint, p_user_data: ?*anyopaque) glib.Quark;
    pub const filterKeyEvent = ibus_hotkey_profile_filter_key_event;

    extern fn ibus_hotkey_profile_lookup_hotkey(p_profile: *HotkeyProfile, p_keyval: c_uint, p_modifiers: c_uint) glib.Quark;
    pub const lookupHotkey = ibus_hotkey_profile_lookup_hotkey;

    /// Removes the hotkey for an `ibus.HotkeyProfile`.
    extern fn ibus_hotkey_profile_remove_hotkey(p_profile: *HotkeyProfile, p_keyval: c_uint, p_modifiers: c_uint) c_int;
    pub const removeHotkey = ibus_hotkey_profile_remove_hotkey;

    /// Removes the hotkey for an `ibus.HotkeyProfile` by event.
    extern fn ibus_hotkey_profile_remove_hotkey_by_event(p_profile: *HotkeyProfile, p_event: glib.Quark) c_int;
    pub const removeHotkeyByEvent = ibus_hotkey_profile_remove_hotkey_by_event;

    extern fn ibus_hotkey_profile_get_type() usize;
    pub const getGObjectType = ibus_hotkey_profile_get_type;

    extern fn g_object_ref(p_self: *ibus.HotkeyProfile) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.HotkeyProfile) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HotkeyProfile, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusInputContext is a proxy object of BusInputContext,
/// which manages the context for input methods that supports
/// text input in various natural languages.
///
/// Clients call the IBusInputContext to invoke BusInputContext,
/// through which invokes IBusEngine.
pub const InputContext = extern struct {
    pub const Parent = ibus.Proxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = ibus.InputContextClass;
    f_parent: ibus.Proxy,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when the text is going to be committed.
        ///
        /// (Note: The text object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const commit_text = struct {
            pub const name = "commit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("commit-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to select next candidate of lookup table.
        pub const cursor_down_lookup_table = struct {
            pub const name = "cursor-down-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("cursor-down-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to select previous candidate of lookup table.
        pub const cursor_up_lookup_table = struct {
            pub const name = "cursor-up-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("cursor-up-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to delete surrounding text event from IME to client of IME.
        pub const delete_surrounding_text = struct {
            pub const name = "delete-surrounding-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_offset: c_int, p_n_chars: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("delete-surrounding-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when an IME is disabled.
        pub const disabled = struct {
            pub const name = "disabled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("disabled", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when an IME is enabled.
        pub const enabled = struct {
            pub const name = "enabled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("enabled", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to forward key event from IME to client of IME.
        pub const forward_key_event = struct {
            pub const name = "forward-key-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_keycode: c_uint, p_modifiers: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("forward-key-event", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to hide auxiliary text.
        pub const hide_auxiliary_text = struct {
            pub const name = "hide-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("hide-auxiliary-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to hide lookup table.
        pub const hide_lookup_table = struct {
            pub const name = "hide-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("hide-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to hide preedit text.
        pub const hide_preedit_text = struct {
            pub const name = "hide-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("hide-preedit-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to view the next page of lookup table.
        pub const page_down_lookup_table = struct {
            pub const name = "page-down-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("page-down-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to view the previous page of lookup table.
        pub const page_up_lookup_table = struct {
            pub const name = "page-up-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("page-up-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to register the properties in `props`.
        ///
        /// (Note: The props object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const register_properties = struct {
            pub const name = "register-properties";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_props: *ibus.PropList, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("register-properties", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to receive the RequireSurroundingText signal from the daemon.
        pub const require_surrounding_text = struct {
            pub const name = "require-surrounding-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("require-surrounding-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to show auxiliary text.
        pub const show_auxiliary_text = struct {
            pub const name = "show-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("show-auxiliary-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to show lookup table.
        pub const show_lookup_table = struct {
            pub const name = "show-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("show-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to show preedit text.
        pub const show_preedit_text = struct {
            pub const name = "show-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("show-preedit-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to hide auxilary text.
        ///
        /// (Note: The text object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const update_auxiliary_text = struct {
            pub const name = "update-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("update-auxiliary-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to update lookup table.
        ///
        /// (Note: The table object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const update_lookup_table = struct {
            pub const name = "update-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_table: *ibus.LookupTable, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("update-lookup-table", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to update preedit text.
        ///
        /// (Note: The text object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const update_preedit_text = struct {
            pub const name = "update-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("update-preedit-text", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to update preedit text with the mode.
        ///
        /// (Note: The text object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const update_preedit_text_with_mode = struct {
            pub const name = "update-preedit-text-with-mode";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int, p_mode: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("update-preedit-text-with-mode", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to update the property `prop`.
        ///
        /// (Note: The prop object is floating, and it will be released after the
        ///  signal. If signal handler wants to keep the object, the handler should
        ///  use `gobject.Object.refSink` to get the ownership of the object.)
        pub const update_property = struct {
            pub const name = "update-property";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_prop: *ibus.Property, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(InputContext, p_instance))),
                    gobject.signalLookup("update-property", InputContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Gets an existing IBusInputContext.
    extern fn ibus_input_context_get_input_context(p_path: [*:0]const u8, p_connection: *gio.DBusConnection) *ibus.InputContext;
    pub const getInputContext = ibus_input_context_get_input_context;

    /// Gets an existing `ibus.InputContext` asynchronously.
    extern fn ibus_input_context_get_input_context_async(p_path: [*:0]const u8, p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getInputContextAsync = ibus_input_context_get_input_context_async;

    /// Finishes an operation started with
    /// `ibus.InputContext.getInputContextAsync`.
    extern fn ibus_input_context_get_input_context_async_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.InputContext;
    pub const getInputContextAsyncFinish = ibus_input_context_get_input_context_async_finish;

    /// Creates a new `ibus.InputContext` asynchronously.
    extern fn ibus_input_context_new_async(p_path: [*:0]const u8, p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newAsync = ibus_input_context_new_async;

    /// Creates a new `ibus.InputContext`.
    extern fn ibus_input_context_new(p_path: [*:0]const u8, p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*ibus.InputContext;
    pub const new = ibus_input_context_new;

    /// Finishes an operation started with `ibus.InputContext.newAsync`.
    extern fn ibus_input_context_new_async_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.InputContext;
    pub const newAsyncFinish = ibus_input_context_new_async_finish;

    /// Clear handwriting stroke(s) in the current input method engine.
    ///
    /// see_also: `ibus.Engine.signals.cancel`-hand-writing
    extern fn ibus_input_context_cancel_hand_writing(p_context: *InputContext, p_n_strokes: c_uint) void;
    pub const cancelHandWriting = ibus_input_context_cancel_hand_writing;

    /// Invoked when the client application get focus. An asynchronous IPC will
    /// be performed.
    ///
    /// see_also: `ibus.Engine.signals.focus_in`.
    extern fn ibus_input_context_focus_in(p_context: *InputContext) void;
    pub const focusIn = ibus_input_context_focus_in;

    /// Invoked when the client application get focus. An asynchronous IPC will be performed.
    ///
    /// see_also: `ibus.Engine.signals.focus_out`.
    extern fn ibus_input_context_focus_out(p_context: *InputContext) void;
    pub const focusOut = ibus_input_context_focus_out;

    /// Gets an IME engine description for the context.
    /// A synchronous IPC will be performed.
    extern fn ibus_input_context_get_engine(p_context: *InputContext) *ibus.EngineDesc;
    pub const getEngine = ibus_input_context_get_engine;

    /// An asynchronous IPC will be performed.
    extern fn ibus_input_context_get_engine_async(p_context: *InputContext, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getEngineAsync = ibus_input_context_get_engine_async;

    /// Finishes an operation started with `ibus.InputContext.getEngineAsync`.
    extern fn ibus_input_context_get_engine_async_finish(p_context: *InputContext, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*ibus.EngineDesc;
    pub const getEngineAsyncFinish = ibus_input_context_get_engine_async_finish;

    /// Check whether the current engine requires surrounding-text.
    extern fn ibus_input_context_needs_surrounding_text(p_context: *InputContext) c_int;
    pub const needsSurroundingText = ibus_input_context_needs_surrounding_text;

    /// Pass a handwriting stroke to an input method engine.
    ///
    /// In this API, a coordinate (0.0, 0.0) represents the top-left corner of an area for
    /// handwriting, and (1.0, 1.0) does the bottom-right. Therefore, for example, if
    /// a user writes a character 'L', the array would be something like [0.0, 0.0, 0.0, 1.0, 1.0, 1.0]
    /// and coordinates_len would be 6.
    ///
    /// The function is usually called when a user releases the mouse button in a hand
    /// writing area.
    ///
    /// see_also: `ibus.Engine.signals.process`-hand-writing-event
    extern fn ibus_input_context_process_hand_writing_event(p_context: *InputContext, p_coordinates: *const f64, p_coordinates_len: c_uint) void;
    pub const processHandWritingEvent = ibus_input_context_process_hand_writing_event;

    /// Pass the key event to input method engine and wait for the reply from
    /// ibus (i.e. synchronous IPC).
    extern fn ibus_input_context_process_key_event(p_context: *InputContext, p_keyval: u32, p_keycode: u32, p_state: u32) c_int;
    pub const processKeyEvent = ibus_input_context_process_key_event;

    /// Pass the key event to input method engine.
    ///
    /// Key symbols are characters/symbols produced by key press, for example,
    /// pressing "s" generates key symbol "s"; pressing shift-"s" generates key symbol "S".
    /// Same key on keyboard may produce different key symbols on different keyboard layout.
    /// e.g., "s" key on QWERTY keyboard produces "o" in DVORAK layout.
    ///
    /// Unlike key symbol, keycode is only determined by the location of the key, and
    /// irrelevant of the keyboard layout.
    ///
    /// Briefly speaking, input methods that expect certain keyboard layout should use
    /// keycode; otherwise keyval is sufficient.
    /// For example, Chewing, Cangjie, Wubi expect an en-US QWERTY keyboard, these should
    /// use keycode; while pinyin can rely on keyval only, as it is less sensitive to
    /// the keyboard layout change, DVORAK users can still use DVORAK layout to input pinyin.
    ///
    /// Use `ibus.Keymap.lookupKeysym` to convert keycode to keysym in given keyboard layout.
    ///
    /// see_also: `ibus.Engine.signals.process`-key-event
    extern fn ibus_input_context_process_key_event_async(p_context: *InputContext, p_keyval: u32, p_keycode: u32, p_state: u32, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const processKeyEventAsync = ibus_input_context_process_key_event_async;

    /// Finishes an operation started with
    ///      `ibus.InputContext.processKeyEventAsync`.
    extern fn ibus_input_context_process_key_event_async_finish(p_context: *InputContext, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const processKeyEventAsyncFinish = ibus_input_context_process_key_event_async_finish;

    /// Activate the property asynchronously.
    ///
    /// See also: `ibus.Engine.signals.property_activate`
    extern fn ibus_input_context_property_activate(p_context: *InputContext, p_prop_name: [*:0]const u8, p_state: u32) void;
    pub const propertyActivate = ibus_input_context_property_activate;

    /// Invoked when the IME is reset. An asynchronous IPC will be performed.
    ///
    /// see_also: `ibus.Engine.signals.reset`
    extern fn ibus_input_context_reset(p_context: *InputContext) void;
    pub const reset = ibus_input_context_reset;

    /// Set the capabilities flags of client application asynchronously.
    /// When IBUS_CAP_FOCUS is not set, IBUS_CAP_PREEDIT_TEXT, IBUS_CAP_AUXILIARY_TEXT, IBUS_CAP_LOOKUP_TABLE, and IBUS_CAP_PROPERTY have to be all set.
    /// The panel component does nothing for an application that doesn't support focus.
    ///
    /// see_also: `ibus.Engine.signals.set`-capabilities
    extern fn ibus_input_context_set_capabilities(p_context: *InputContext, p_capabilities: u32) void;
    pub const setCapabilities = ibus_input_context_set_capabilities;

    /// Set whether `ibus.InputContext` commits pre-edit texts or not.
    /// If `TRUE`, 'update-preedit-text-with-mode' signal is emitted
    /// instead of 'update-preedit-text' signal.
    /// If your client receives the 'update-preedit-text-with-mode' signal,
    /// the client needs to implement `commit_text` of pre-edit text when
    /// GtkIMContextClass.`focus_out` is called in case an IME desires that
    /// behavior but it depends on each IME.
    ///
    /// See also `ibus.Engine.updatePreeditTextWithMode`.
    extern fn ibus_input_context_set_client_commit_preedit(p_context: *InputContext, p_client_commit: c_int) void;
    pub const setClientCommitPreedit = ibus_input_context_set_client_commit_preedit;

    /// Set content-type (primary purpose and hints) of the context.  This
    /// information is particularly useful to implement intelligent
    /// behavior in engines, such as automatic input-mode switch and text
    /// prediction.  For example, to restrict input to numbers, the client
    /// can call this function with `purpose` set to
    /// `IBUS_INPUT_PURPOSE_NUMBER`.
    ///
    /// See also: `ibus.Engine.signals.set`-content-type
    extern fn ibus_input_context_set_content_type(p_context: *InputContext, p_purpose: c_uint, p_hints: c_uint) void;
    pub const setContentType = ibus_input_context_set_content_type;

    /// Set the cursor location of IBus input context asynchronously.
    ///
    /// see_also: `ibus.Engine.signals.set`-cursor-location
    extern fn ibus_input_context_set_cursor_location(p_context: *InputContext, p_x: i32, p_y: i32, p_w: i32, p_h: i32) void;
    pub const setCursorLocation = ibus_input_context_set_cursor_location;

    /// Set the relative cursor location of IBus input context asynchronously.
    extern fn ibus_input_context_set_cursor_location_relative(p_context: *InputContext, p_x: i32, p_y: i32, p_w: i32, p_h: i32) void;
    pub const setCursorLocationRelative = ibus_input_context_set_cursor_location_relative;

    /// Invoked when the IME engine is changed.
    /// An asynchronous IPC will be performed.
    extern fn ibus_input_context_set_engine(p_context: *InputContext, p_name: [*:0]const u8) void;
    pub const setEngine = ibus_input_context_set_engine;

    extern fn ibus_input_context_set_surrounding_text(p_context: *InputContext, p_text: *ibus.Text, p_cursor_pos: u32, p_anchor_pos: u32) void;
    pub const setSurroundingText = ibus_input_context_set_surrounding_text;

    extern fn ibus_input_context_get_type() usize;
    pub const getGObjectType = ibus_input_context_get_type;

    extern fn g_object_ref(p_self: *ibus.InputContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.InputContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *InputContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusKeymap defines the mapping between keyboard scancodes and
/// keyboard symbols such as numbers, alphabets, and punctuation marks.
///
/// Some input methods assume certain keyboard layout
/// (such as Chewing and Wubi requires an US-QWERTY layout),
/// and expect key symbols to be arranged in that order.
/// These input methods should new an IBusKeymap
/// instance and define the keyboard layout.
/// Then `ibus.Keymap.lookupKeysym` can
/// convert scancodes back to the key symbols.
///
/// see_also: `ibus.Component`, `ibus.EngineDesc`
pub const Keymap = extern struct {
    pub const Parent = ibus.Object;
    pub const Implements = [_]type{};
    pub const Class = ibus.KeymapClass;
    f_parent: ibus.Object,
    /// The name of the keymap, such as 'us', 'jp'.
    f_name: ?[*:0]u8,
    /// Keymap table. IME developers normally don have to touch this.
    f_keymap: [1792]c_uint,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Get an IBusKeymap associated with the giving name.
    ///
    /// This function loads the keymap file specified in `name`
    /// in the IBUS_DATA_DIR/keymaps directory.
    extern fn ibus_keymap_get(p_name: [*:0]const u8) *ibus.Keymap;
    pub const get = ibus_keymap_get;

    /// Get an `ibus.Keymap` associated with the giving name.
    ///
    /// This function loads the keymap file specified in `name`
    /// in the IBUS_DATA_DIR/keymaps directory.
    extern fn ibus_keymap_new(p_name: [*:0]const u8) *ibus.Keymap;
    pub const new = ibus_keymap_new;

    /// Converts the scancode to keysym, given the keymap.
    extern fn ibus_keymap_lookup_keysym(p_keymap: *Keymap, p_keycode: u16, p_state: u32) c_uint;
    pub const lookupKeysym = ibus_keymap_lookup_keysym;

    extern fn ibus_keymap_get_type() usize;
    pub const getGObjectType = ibus_keymap_get_type;

    extern fn g_object_ref(p_self: *ibus.Keymap) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Keymap) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Keymap, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusLookuptable stores the candidate words or phrases for users to
/// choose from.
///
/// Use `ibus.Engine.updateLookupTable`, `ibus.Engine.showLookupTable`,
/// and `ibus.Engine.hideLookupTable` to update, show and hide the lookup
/// table.
///
/// see_also: `ibus.Engine`
pub const LookupTable = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.LookupTableClass;
    f_parent: ibus.Serializable,
    /// number of candidate shown per page.
    f_page_size: c_uint,
    /// position index of cursor.
    f_cursor_pos: c_uint,
    /// whether the cursor is visible.
    f_cursor_visible: c_int,
    /// TRUE for lookup table wrap around.
    f_round: c_int,
    /// orientation of the table.
    f_orientation: c_int,
    /// Candidate words/phrases.
    f_candidates: ?*glib.Array,
    /// Candidate labels which identify individual candidates in the same page. Default is 1, 2, 3, 4 ...
    f_labels: ?*glib.Array,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Craetes a new `ibus.LookupTable`.
    extern fn ibus_lookup_table_new(p_page_size: c_uint, p_cursor_pos: c_uint, p_cursor_visible: c_int, p_round: c_int) *ibus.LookupTable;
    pub const new = ibus_lookup_table_new;

    /// Append a candidate word/phrase to IBusLookupTable, and increase reference.
    extern fn ibus_lookup_table_append_candidate(p_table: *LookupTable, p_text: *ibus.Text) void;
    pub const appendCandidate = ibus_lookup_table_append_candidate;

    /// Append a candidate word/phrase to IBusLookupTable, and increase reference.
    /// This function is needed if the input method select candidate with
    /// non-numeric keys such as "asdfghjkl;".
    extern fn ibus_lookup_table_append_label(p_table: *LookupTable, p_text: *ibus.Text) void;
    pub const appendLabel = ibus_lookup_table_append_label;

    /// Clear and remove all candidate from an IBusLookupTable.
    extern fn ibus_lookup_table_clear(p_table: *LookupTable) void;
    pub const clear = ibus_lookup_table_clear;

    /// Go to next candidate of an `ibus.LookupTable`.
    ///
    /// It returns FALSE if it is already at the last candidate,
    /// unless  <code>table&gt;-round==TRUE</code>, where it will go
    /// to the first candidate.
    extern fn ibus_lookup_table_cursor_down(p_table: *LookupTable) c_int;
    pub const cursorDown = ibus_lookup_table_cursor_down;

    /// Go to previous candidate of an `ibus.LookupTable`.
    ///
    /// It returns FALSE if it is already at the first candidate,
    /// unless  <code>table&gt;-round==TRUE</code>, where it will go
    /// to the last candidate.
    extern fn ibus_lookup_table_cursor_up(p_table: *LookupTable) c_int;
    pub const cursorUp = ibus_lookup_table_cursor_up;

    /// Return `ibus.Text` at the given index. Borrowed reference.
    extern fn ibus_lookup_table_get_candidate(p_table: *LookupTable, p_index: c_uint) *ibus.Text;
    pub const getCandidate = ibus_lookup_table_get_candidate;

    /// Gets the cursor position in current page of `ibus.LookupTable`.
    extern fn ibus_lookup_table_get_cursor_in_page(p_table: *LookupTable) c_uint;
    pub const getCursorInPage = ibus_lookup_table_get_cursor_in_page;

    /// Gets the cursor position of `ibus.LookupTable`.
    extern fn ibus_lookup_table_get_cursor_pos(p_table: *LookupTable) c_uint;
    pub const getCursorPos = ibus_lookup_table_get_cursor_pos;

    /// Return `ibus.Text` at the given index. Borrowed reference.
    extern fn ibus_lookup_table_get_label(p_table: *LookupTable, p_index: c_uint) *ibus.Text;
    pub const getLabel = ibus_lookup_table_get_label;

    /// Return the number of candidate in the table.
    extern fn ibus_lookup_table_get_number_of_candidates(p_table: *LookupTable) c_uint;
    pub const getNumberOfCandidates = ibus_lookup_table_get_number_of_candidates;

    /// Returns the orientation of the `ibus.LookupTable`.
    extern fn ibus_lookup_table_get_orientation(p_table: *LookupTable) c_int;
    pub const getOrientation = ibus_lookup_table_get_orientation;

    /// Gets the number of candidate shown per page.
    extern fn ibus_lookup_table_get_page_size(p_table: *LookupTable) c_uint;
    pub const getPageSize = ibus_lookup_table_get_page_size;

    /// Returns whether the cursor of an `ibus.LookupTable` is visible.
    extern fn ibus_lookup_table_is_cursor_visible(p_table: *LookupTable) c_int;
    pub const isCursorVisible = ibus_lookup_table_is_cursor_visible;

    /// Returns whether the `ibus.LookupTable` is round.
    extern fn ibus_lookup_table_is_round(p_table: *LookupTable) c_int;
    pub const isRound = ibus_lookup_table_is_round;

    /// Go to next page of an `ibus.LookupTable`.
    ///
    /// It returns FALSE if it is already at the last page,
    /// unless  <code>table&gt;-round==TRUE</code>, where it will go
    /// to the first page.
    extern fn ibus_lookup_table_page_down(p_table: *LookupTable) c_int;
    pub const pageDown = ibus_lookup_table_page_down;

    /// Go to previous page of an `ibus.LookupTable`.
    ///
    /// It returns FALSE if it is already at the first page,
    /// unless  <code>table&gt;-round==TRUE</code>, where it will go
    /// to the last page.
    extern fn ibus_lookup_table_page_up(p_table: *LookupTable) c_int;
    pub const pageUp = ibus_lookup_table_page_up;

    /// Set the cursor position of IBusLookupTable.
    extern fn ibus_lookup_table_set_cursor_pos(p_table: *LookupTable, p_cursor_pos: c_uint) void;
    pub const setCursorPos = ibus_lookup_table_set_cursor_pos;

    /// Set whether to make the cursor of an IBusLookupTable visible or not.
    extern fn ibus_lookup_table_set_cursor_visible(p_table: *LookupTable, p_visible: c_int) void;
    pub const setCursorVisible = ibus_lookup_table_set_cursor_visible;

    /// Append a candidate word/phrase to IBusLookupTable, and increase reference.
    /// This function is needed if the input method select candidate with
    /// non-numeric keys such as "asdfghjkl;".
    extern fn ibus_lookup_table_set_label(p_table: *LookupTable, p_index: c_uint, p_text: *ibus.Text) void;
    pub const setLabel = ibus_lookup_table_set_label;

    /// Set the orientation.
    extern fn ibus_lookup_table_set_orientation(p_table: *LookupTable, p_orientation: c_int) void;
    pub const setOrientation = ibus_lookup_table_set_orientation;

    /// Set the number of candidate shown per page.
    extern fn ibus_lookup_table_set_page_size(p_table: *LookupTable, p_page_size: c_uint) void;
    pub const setPageSize = ibus_lookup_table_set_page_size;

    /// Set whether to make the IBusLookupTable round or not.
    extern fn ibus_lookup_table_set_round(p_table: *LookupTable, p_round: c_int) void;
    pub const setRound = ibus_lookup_table_set_round;

    extern fn ibus_lookup_table_get_type() usize;
    pub const getGObjectType = ibus_lookup_table_get_type;

    extern fn g_object_ref(p_self: *ibus.LookupTable) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.LookupTable) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *LookupTable, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// IBusObject is the base object for all objects in IBus.
pub const Object = extern struct {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{};
    pub const Class = ibus.ObjectClass;
    f_parent: gobject.InitiallyUnowned,
    f_flags: u32,
    f_priv: ?*ibus.ObjectPrivate,

    pub const virtual_methods = struct {
        /// Emit the "destroy" signal notifying all reference holders that they should
        /// release the `ibus.Object`.
        ///
        /// The memory for the object itself won't be deleted until its reference count
        /// actually drops to 0; ibus_object_destroy merely asks reference holders to
        /// release their references. It does not free the object.
        pub const destroy = struct {
            pub fn call(p_class: anytype, p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Object.Class, p_class).f_destroy.?(gobject.ext.as(Object, p_object));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Object.Class, p_class).f_destroy = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Destroy and free an IBusObject
        ///
        /// See also:  `ibus.Object.destroy`.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para></note>
        pub const destroy = struct {
            pub const name = "destroy";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Object, p_instance))),
                    gobject.signalLookup("destroy", Object.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates  a new `ibus.Object`.
    extern fn ibus_object_new() *ibus.Object;
    pub const new = ibus_object_new;

    /// Emit the "destroy" signal notifying all reference holders that they should
    /// release the `ibus.Object`.
    ///
    /// The memory for the object itself won't be deleted until its reference count
    /// actually drops to 0; ibus_object_destroy merely asks reference holders to
    /// release their references. It does not free the object.
    extern fn ibus_object_destroy(p_object: *Object) void;
    pub const destroy = ibus_object_destroy;

    extern fn ibus_object_get_type() usize;
    pub const getGObjectType = ibus_object_get_type;

    extern fn g_object_ref(p_self: *ibus.Object) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Object) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Object, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// IBusObservedPath provides methods for file path manipulation,
/// such as monitor modification, directory tree traversal.
pub const ObservedPath = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.ObservedPathClass;
    f_parent: ibus.Serializable,
    /// Path to be handled.
    f_path: ?[*:0]u8,
    /// Modified time.
    f_mtime: c_long,
    /// Whether the file is the path directory.
    f_is_dir: c_int,
    /// Whether the file exists.
    f_is_exist: c_int,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `ibus.ObservedPath` from an XML node.
    extern fn ibus_observed_path_new(p_path: [*:0]const u8, p_fill_stat: c_int) *ibus.ObservedPath;
    pub const new = ibus_observed_path_new;

    /// Creates an new `ibus.ObservedPath` from an XML node.
    extern fn ibus_observed_path_new_from_xml_node(p_node: *ibus.XML, p_fill_stat: c_int) *ibus.ObservedPath;
    pub const newFromXmlNode = ibus_observed_path_new_from_xml_node;

    /// Checks whether the path is modified by comparing the mtime in object and
    /// mtime in file system.
    extern fn ibus_observed_path_check_modification(p_path: *ObservedPath) c_int;
    pub const checkModification = ibus_observed_path_check_modification;

    /// Append the observed path to a string with following format:
    /// &lt;path mtime="&lt;i&gt;modified time&lt;/i&gt;" &gt;&lt;i&gt;path&lt;/i&gt;&lt;/path&gt;
    extern fn ibus_observed_path_output(p_path: *ObservedPath, p_output: *glib.String, p_indent: c_int) void;
    pub const output = ibus_observed_path_output;

    /// Recursively traverse the path and put the files and subdirectory in to
    /// a newly allocated
    /// GLists, if the `path` is a directory. Otherwise returns NULL.
    extern fn ibus_observed_path_traverse(p_path: *ObservedPath, p_dir_only: c_int) *glib.List;
    pub const traverse = ibus_observed_path_traverse;

    extern fn ibus_observed_path_get_type() usize;
    pub const getGObjectType = ibus_observed_path_get_type;

    extern fn g_object_ref(p_self: *ibus.ObservedPath) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.ObservedPath) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ObservedPath, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusPanelService is a base class for UI services.
/// Developers can "extend" this class for panel UI development.
pub const PanelService = extern struct {
    pub const Parent = ibus.Service;
    pub const Implements = [_]type{};
    pub const Class = ibus.PanelServiceClass;
    f_parent: ibus.Service,

    pub const virtual_methods = struct {
        pub const candidate_clicked_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_button: c_uint, p_state: c_uint) void {
                return gobject.ext.as(PanelService.Class, p_class).f_candidate_clicked_lookup_table.?(gobject.ext.as(PanelService, p_panel), p_index, p_button, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_button: c_uint, p_state: c_uint) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_candidate_clicked_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const commit_text_received = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text) void {
                return gobject.ext.as(PanelService.Class, p_class).f_commit_text_received.?(gobject.ext.as(PanelService, p_panel), p_text);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_commit_text_received = @ptrCast(p_implementation);
            }
        };

        pub const cursor_down_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_cursor_down_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_cursor_down_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const cursor_up_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_cursor_up_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_cursor_up_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const destroy_context = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) void {
                return gobject.ext.as(PanelService.Class, p_class).f_destroy_context.?(gobject.ext.as(PanelService, p_panel), p_input_context_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_destroy_context = @ptrCast(p_implementation);
            }
        };

        pub const focus_in = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) void {
                return gobject.ext.as(PanelService.Class, p_class).f_focus_in.?(gobject.ext.as(PanelService, p_panel), p_input_context_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_focus_in = @ptrCast(p_implementation);
            }
        };

        pub const focus_out = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) void {
                return gobject.ext.as(PanelService.Class, p_class).f_focus_out.?(gobject.ext.as(PanelService, p_panel), p_input_context_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_input_context_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_focus_out = @ptrCast(p_implementation);
            }
        };

        pub const hide_auxiliary_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_hide_auxiliary_text.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_hide_auxiliary_text = @ptrCast(p_implementation);
            }
        };

        pub const hide_language_bar = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_hide_language_bar.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_hide_language_bar = @ptrCast(p_implementation);
            }
        };

        pub const hide_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_hide_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_hide_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const hide_preedit_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_hide_preedit_text.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_hide_preedit_text = @ptrCast(p_implementation);
            }
        };

        pub const page_down_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_page_down_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_page_down_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const page_up_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_page_up_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_page_up_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const panel_extension_received = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *ibus.ExtensionEvent) void {
                return gobject.ext.as(PanelService.Class, p_class).f_panel_extension_received.?(gobject.ext.as(PanelService, p_panel), p_event);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *ibus.ExtensionEvent) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_panel_extension_received = @ptrCast(p_implementation);
            }
        };

        pub const process_key_event = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) c_int {
                return gobject.ext.as(PanelService.Class, p_class).f_process_key_event.?(gobject.ext.as(PanelService, p_panel), p_keyval, p_keycode, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) callconv(.C) c_int) void {
                gobject.ext.as(PanelService.Class, p_class).f_process_key_event = @ptrCast(p_implementation);
            }
        };

        pub const register_properties = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_list: *ibus.PropList) void {
                return gobject.ext.as(PanelService.Class, p_class).f_register_properties.?(gobject.ext.as(PanelService, p_panel), p_prop_list);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop_list: *ibus.PropList) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_register_properties = @ptrCast(p_implementation);
            }
        };

        pub const reset = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_reset.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_reset = @ptrCast(p_implementation);
            }
        };

        pub const set_content_type = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_purpose: c_uint, p_hints: c_uint) void {
                return gobject.ext.as(PanelService.Class, p_class).f_set_content_type.?(gobject.ext.as(PanelService, p_panel), p_purpose, p_hints);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_purpose: c_uint, p_hints: c_uint) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_set_content_type = @ptrCast(p_implementation);
            }
        };

        pub const set_cursor_location = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) void {
                return gobject.ext.as(PanelService.Class, p_class).f_set_cursor_location.?(gobject.ext.as(PanelService, p_panel), p_x, p_y, p_w, p_h);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_set_cursor_location = @ptrCast(p_implementation);
            }
        };

        pub const set_cursor_location_relative = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) void {
                return gobject.ext.as(PanelService.Class, p_class).f_set_cursor_location_relative.?(gobject.ext.as(PanelService, p_panel), p_x, p_y, p_w, p_h);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_set_cursor_location_relative = @ptrCast(p_implementation);
            }
        };

        pub const show_auxiliary_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_show_auxiliary_text.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_show_auxiliary_text = @ptrCast(p_implementation);
            }
        };

        pub const show_language_bar = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_show_language_bar.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_show_language_bar = @ptrCast(p_implementation);
            }
        };

        pub const show_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_show_lookup_table.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_show_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const show_preedit_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_show_preedit_text.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_show_preedit_text = @ptrCast(p_implementation);
            }
        };

        pub const start_setup = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_start_setup.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_start_setup = @ptrCast(p_implementation);
            }
        };

        pub const state_changed = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(PanelService.Class, p_class).f_state_changed.?(gobject.ext.as(PanelService, p_panel));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_state_changed = @ptrCast(p_implementation);
            }
        };

        pub const update_auxiliary_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_visible: c_int) void {
                return gobject.ext.as(PanelService.Class, p_class).f_update_auxiliary_text.?(gobject.ext.as(PanelService, p_panel), p_text, p_visible);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_visible: c_int) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_update_auxiliary_text = @ptrCast(p_implementation);
            }
        };

        pub const update_lookup_table = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lookup_table: *ibus.LookupTable, p_visible: c_int) void {
                return gobject.ext.as(PanelService.Class, p_class).f_update_lookup_table.?(gobject.ext.as(PanelService, p_panel), p_lookup_table, p_visible);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lookup_table: *ibus.LookupTable, p_visible: c_int) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_update_lookup_table = @ptrCast(p_implementation);
            }
        };

        pub const update_preedit_text = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int) void {
                return gobject.ext.as(PanelService.Class, p_class).f_update_preedit_text.?(gobject.ext.as(PanelService, p_panel), p_text, p_cursor_pos, p_visible);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_update_preedit_text = @ptrCast(p_implementation);
            }
        };

        pub const update_property = struct {
            pub fn call(p_class: anytype, p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop: *ibus.Property) void {
                return gobject.ext.as(PanelService.Class, p_class).f_update_property.?(gobject.ext.as(PanelService, p_panel), p_prop);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_panel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prop: *ibus.Property) callconv(.C) void) void {
                gobject.ext.as(PanelService.Class, p_class).f_update_property = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        pub const candidate_clicked_lookup_table = struct {
            pub const name = "candidate-clicked-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: c_uint, p_p0: c_uint, p_p1: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("candidate-clicked-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::commit-text-received.
        /// Implement the member function
        /// IBusPanelServiceClass::commit_text_received in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const commit_text_received = struct {
            pub const name = "commit-text-received";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("commit-text-received", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::cursor-down-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::cursor_down_lookup_table in extended
        /// class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const cursor_down_lookup_table = struct {
            pub const name = "cursor-down-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("cursor-down-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::cursor-up-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::cursor_up_lookup_table in extended
        /// class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const cursor_up_lookup_table = struct {
            pub const name = "cursor-up-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("cursor-up-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application destroys.
        /// Implement the member function
        /// IBusPanelServiceClass::destroy_context in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const destroy_context = struct {
            pub const name = "destroy-context";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_input_context_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("destroy-context", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::focus-in.
        /// Implement the member function
        /// IBusPanelServiceClass::focus_in in extended class to receive
        /// this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_in = struct {
            pub const name = "focus-in";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_input_context_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("focus-in", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::focus-out.
        /// Implement the member function
        /// IBusPanelServiceClass::focus_out in extended class to receive
        /// this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const focus_out = struct {
            pub const name = "focus-out";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_input_context_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("focus-out", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::hide-auxiliary-text.
        /// Implement the member function
        /// IBusPanelServiceClass::hide_auxiliary_text in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const hide_auxiliary_text = struct {
            pub const name = "hide-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("hide-auxiliary-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::hide-language-bar.
        /// Implement the member function
        /// IBusPanelServiceClass::hide_language_bar in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const hide_language_bar = struct {
            pub const name = "hide-language-bar";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("hide-language-bar", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::hide-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::hide_lookup_table in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const hide_lookup_table = struct {
            pub const name = "hide-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("hide-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::hide-preedit-text.
        /// Implement the member function
        /// IBusPanelServiceClass::hide_preedit_text in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const hide_preedit_text = struct {
            pub const name = "hide-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("hide-preedit-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::page-down-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::page_down_lookup_table in extended
        /// class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const page_down_lookup_table = struct {
            pub const name = "page-down-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("page-down-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::page-up-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::page_up_lookup_table in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const page_up_lookup_table = struct {
            pub const name = "page-up-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("page-up-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::panel-extension-received.
        /// Implement the member function
        /// IBusPanelServiceClass::panel_extension_received in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const panel_extension_received = struct {
            pub const name = "panel-extension-received";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_data: *ibus.ExtensionEvent, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("panel-extension-received", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a key event is received.
        /// Implement the member function IBusPanelServiceClass::process_key_event
        /// in extended class to receive this signal.
        /// Both the key symbol and keycode are passed to the member function.
        /// See `ibus.InputContext.processKeyEvent` for further explanation of
        /// key symbol, keycode and which to use.
        pub const process_key_event = struct {
            pub const name = "process-key-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("process-key-event", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::register-properties.
        /// Implement the member function
        /// IBusPanelServiceClass::register_properties in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const register_properties = struct {
            pub const name = "register-properties";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_prop_list: *ibus.PropList, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("register-properties", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::reset.
        /// Implement the member function
        /// IBusPanelServiceClass::reset in extended class to receive this
        /// signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const reset = struct {
            pub const name = "reset";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("reset", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::set-content-type.
        /// Implement the member function
        /// IBusPanelServiceClass::set_content_type in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_content_type = struct {
            pub const name = "set-content-type";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_purpose: c_uint, p_hints: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("set-content-type", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::set-cursor-location.
        /// Implement the member function
        /// IBusPanelServiceClass::set_cursor_location in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_cursor_location = struct {
            pub const name = "set-cursor-location";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("set-cursor-location", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the set-cursor-location-relative.
        /// Implement the member function `set_cursor_location_relative` in
        /// extended class to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const set_cursor_location_relative = struct {
            pub const name = "set-cursor-location-relative";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("set-cursor-location-relative", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::show-auxiliary-text.
        /// Implement the member function
        /// IBusPanelServiceClass::show_auxiliary_text in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const show_auxiliary_text = struct {
            pub const name = "show-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("show-auxiliary-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::show-language-bar.
        /// Implement the member function
        /// IBusPanelServiceClass::show_language_bar in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const show_language_bar = struct {
            pub const name = "show-language-bar";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("show-language-bar", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::show-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::show_lookup_table in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const show_lookup_table = struct {
            pub const name = "show-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("show-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::show-preedit-text.
        /// Implement the member function
        /// IBusPanelServiceClass::show_preedit_text in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const show_preedit_text = struct {
            pub const name = "show-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("show-preedit-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::start-setup.
        /// Implement the member function
        /// IBusPanelServiceClass::start_setup in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const start_setup = struct {
            pub const name = "start-setup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("start-setup", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::state-changed.
        /// Implement the member function
        /// IBusPanelServiceClass::state_changed in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const state_changed = struct {
            pub const name = "state-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("state-changed", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::update-auxiliary-text.
        /// Implement the member function
        /// IBusPanelServiceClass::update_auxiliary_text in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const update_auxiliary_text = struct {
            pub const name = "update-auxiliary-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("update-auxiliary-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::update-lookup-table.
        /// Implement the member function
        /// IBusPanelServiceClass::update_lookup_table in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const update_lookup_table = struct {
            pub const name = "update-lookup-table";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_lookup_table: *ibus.LookupTable, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("update-lookup-table", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::update-preedit-text.
        /// Implement the member function
        /// IBusPanelServiceClass::update_preedit_text in extended class
        /// to receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const update_preedit_text = struct {
            pub const name = "update-preedit-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("update-preedit-text", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the client application get the ::update-property.
        /// Implement the member function
        /// IBusPanelServiceClass::update_property in extended class to
        /// receive this signal.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para>
        /// </note>
        pub const update_property = struct {
            pub const name = "update-property";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_prop: *ibus.Property, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(PanelService, p_instance))),
                    gobject.signalLookup("update-property", PanelService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `ibus.PanelService` from an `gio.DBusConnection`.
    extern fn ibus_panel_service_new(p_connection: *gio.DBusConnection) *ibus.PanelService;
    pub const new = ibus_panel_service_new;

    /// Notify that a candidate is clicked
    /// by sending a "CandidateClicked" to IBus service.
    extern fn ibus_panel_service_candidate_clicked(p_panel: *PanelService, p_index: c_uint, p_button: c_uint, p_state: c_uint) void;
    pub const candidateClicked = ibus_panel_service_candidate_clicked;

    /// Notify that a text is sent
    /// by sending a "CommitText" message to IBus service.
    extern fn ibus_panel_service_commit_text(p_panel: *PanelService, p_text: *ibus.Text) void;
    pub const commitText = ibus_panel_service_commit_text;

    /// Notify that the cursor is down
    /// by sending a "CursorDown" to IBus service.
    extern fn ibus_panel_service_cursor_down(p_panel: *PanelService) void;
    pub const cursorDown = ibus_panel_service_cursor_down;

    /// Notify that the cursor is up
    /// by sending a "CursorUp" to IBus service.
    extern fn ibus_panel_service_cursor_up(p_panel: *PanelService) void;
    pub const cursorUp = ibus_panel_service_cursor_up;

    /// Notify that the preedit is hidden by the panel extension
    extern fn ibus_panel_service_hide_preedit_text_received(p_panel: *PanelService) void;
    pub const hidePreeditTextReceived = ibus_panel_service_hide_preedit_text_received;

    /// Notify that the page is down
    /// by sending a "PageDown" to IBus service.
    extern fn ibus_panel_service_page_down(p_panel: *PanelService) void;
    pub const pageDown = ibus_panel_service_page_down;

    /// Notify that the page is up
    /// by sending a "PageUp" to IBus service.
    extern fn ibus_panel_service_page_up(p_panel: *PanelService) void;
    pub const pageUp = ibus_panel_service_page_up;

    /// Enable or disable a panel extension with `ibus.ExtensionEvent`.
    /// Notify that a data is sent
    /// by sending a "PanelExtension" message to IBus panel extension service.
    extern fn ibus_panel_service_panel_extension(p_panel: *PanelService, p_event: *ibus.ExtensionEvent) void;
    pub const panelExtension = ibus_panel_service_panel_extension;

    /// Register shortcut keys to enable panel extensions with `ibus.ExtensionEvent`.
    /// Notify that a data is sent
    /// by sending a "PanelExtensionRegisterKeys" message to IBus panel extension
    /// service. Seems Vala does not support uint[][3] and use
    /// IBusProcessKeyEventData[]. E.g.
    /// IBusProcessKeyEventData[] keys = {{
    ///         IBUS_KEY_e, 0, IBUS_SHIFT_MASK | IBUS_SUPER_MASK }};
    /// ibus_panel_service_panel_extension_register_keys(panel, "emoji", keys, NULL);
    extern fn ibus_panel_service_panel_extension_register_keys(p_panel: *PanelService, p_first_property_name: [*:0]const u8, ...) void;
    pub const panelExtensionRegisterKeys = ibus_panel_service_panel_extension_register_keys;

    /// Notify that a property is active
    /// by sending a "PropertyActivate" message to IBus service.
    extern fn ibus_panel_service_property_activate(p_panel: *PanelService, p_prop_name: [*:0]const u8, p_prop_state: c_uint) void;
    pub const propertyActivate = ibus_panel_service_property_activate;

    /// Notify that a property is hidden
    /// by sending a "ValueChanged" message to IBus service.
    extern fn ibus_panel_service_property_hide(p_panel: *PanelService, p_prop_name: [*:0]const u8) void;
    pub const propertyHide = ibus_panel_service_property_hide;

    /// Notify that a property is shown
    /// by sending a "ValueChanged" message to IBus service.
    extern fn ibus_panel_service_property_show(p_panel: *PanelService, p_prop_name: [*:0]const u8) void;
    pub const propertyShow = ibus_panel_service_property_show;

    /// Notify that the preedit is shown by the panel extension
    extern fn ibus_panel_service_show_preedit_text_received(p_panel: *PanelService) void;
    pub const showPreeditTextReceived = ibus_panel_service_show_preedit_text_received;

    /// Notify that the auxilirary is updated by the panel extension.
    ///
    /// (Note: The table object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_panel_service_update_auxiliary_text_received(p_panel: *PanelService, p_text: *ibus.Text, p_visible: c_int) void;
    pub const updateAuxiliaryTextReceived = ibus_panel_service_update_auxiliary_text_received;

    /// Notify that the lookup table is updated by the panel extension.
    ///
    /// (Note: The table object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_panel_service_update_lookup_table_received(p_panel: *PanelService, p_table: *ibus.LookupTable, p_visible: c_int) void;
    pub const updateLookupTableReceived = ibus_panel_service_update_lookup_table_received;

    /// Notify that the preedit is updated by the panel extension
    ///
    /// (Note: The table object will be released, if it is floating.
    ///  If caller want to keep the object, caller should make the object
    ///  sink by g_object_ref_sink.)
    extern fn ibus_panel_service_update_preedit_text_received(p_panel: *PanelService, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int) void;
    pub const updatePreeditTextReceived = ibus_panel_service_update_preedit_text_received;

    extern fn ibus_panel_service_get_type() usize;
    pub const getGObjectType = ibus_panel_service_get_type;

    extern fn g_object_ref(p_self: *ibus.PanelService) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.PanelService) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PanelService, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// See_also: `ibus.Property`, `ibus.Engine`
pub const PropList = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.PropListClass;
    f_parent: ibus.Serializable,
    /// GArray that holds IBusProperties.
    f_properties: ?*glib.Array,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Create a new `ibus.PropList`.
    extern fn ibus_prop_list_new() *ibus.PropList;
    pub const new = ibus_prop_list_new;

    /// Append an IBusProperty to an IBusPropList, and increase reference.
    extern fn ibus_prop_list_append(p_prop_list: *PropList, p_prop: *ibus.Property) void;
    pub const append = ibus_prop_list_append;

    /// Gets `ibus.Property` at given index. Borrowed reference.
    extern fn ibus_prop_list_get(p_prop_list: *PropList, p_index: c_uint) *ibus.Property;
    pub const get = ibus_prop_list_get;

    /// Update an IBusProperty in IBusPropList.
    extern fn ibus_prop_list_update_property(p_prop_list: *PropList, p_prop: *ibus.Property) c_int;
    pub const updateProperty = ibus_prop_list_update_property;

    extern fn ibus_prop_list_get_type() usize;
    pub const getGObjectType = ibus_prop_list_get_type;

    extern fn g_object_ref(p_self: *ibus.PropList) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.PropList) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PropList, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusProperty is an UI component like a button or a menu item
/// which shows the status of corresponding input method engine property.
/// End user can operate and see the current status of IME through these components.
/// For example, ibus-chewing users change the English/Chinese input mode by
/// pressing ctrl-space or click on the Eng/Chi switch button.
/// And the IBusProperty shows the change correspondingly.
///
/// see_also: `ibus.PropList`, `ibus.Engine`
pub const Property = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.PropertyClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.PropertyPrivate,
    f_pdummy: [7]*anyopaque,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const icon = struct {
            pub const name = "icon";

            pub const Type = ?[*:0]u8;
        };

        pub const key = struct {
            pub const name = "key";

            pub const Type = ?[*:0]u8;
        };

        pub const label = struct {
            pub const name = "label";

            pub const Type = ?*ibus.Text;
        };

        pub const prop_type = struct {
            pub const name = "prop-type";

            pub const Type = ibus.PropType;
        };

        pub const sensitive = struct {
            pub const name = "sensitive";

            pub const Type = c_int;
        };

        pub const state = struct {
            pub const name = "state";

            pub const Type = ibus.PropState;
        };

        pub const sub_props = struct {
            pub const name = "sub-props";

            pub const Type = ?*ibus.PropList;
        };

        pub const symbol = struct {
            pub const name = "symbol";

            pub const Type = ?*ibus.Text;
        };

        pub const tooltip = struct {
            pub const name = "tooltip";

            pub const Type = ?*ibus.Text;
        };

        pub const visible = struct {
            pub const name = "visible";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `ibus.Property`.
    extern fn ibus_property_new(p_key: [*:0]const u8, p_type: ibus.PropType, p_label: *ibus.Text, p_icon: ?[*:0]const u8, p_tooltip: *ibus.Text, p_sensitive: c_int, p_visible: c_int, p_state: ibus.PropState, p_prop_list: ?*ibus.PropList) *ibus.Property;
    pub const new = ibus_property_new;

    /// Creates a new `ibus.Property`.
    /// `ibus.Property.newVarargs` supports the va_list format.
    /// name property is required. e.g.
    /// ibus_property_new_varargs("key", "TypingMode", "type", PROP_TYPE_MENU, NULL)
    extern fn ibus_property_new_varargs(p_first_property_name: [*:0]const u8, ...) *ibus.Property;
    pub const newVarargs = ibus_property_new_varargs;

    /// Get the icon of `ibus.Property`.
    extern fn ibus_property_get_icon(p_prop: *Property) [*:0]const u8;
    pub const getIcon = ibus_property_get_icon;

    /// Get the key of `ibus.Property`.
    extern fn ibus_property_get_key(p_prop: *Property) [*:0]const u8;
    pub const getKey = ibus_property_get_key;

    /// Get the label of `ibus.Property`.
    extern fn ibus_property_get_label(p_prop: *Property) *ibus.Text;
    pub const getLabel = ibus_property_get_label;

    extern fn ibus_property_get_prop_type(p_prop: *Property) ibus.PropType;
    pub const getPropType = ibus_property_get_prop_type;

    /// Get the sensitive of `ibus.Property`.
    extern fn ibus_property_get_sensitive(p_prop: *Property) c_int;
    pub const getSensitive = ibus_property_get_sensitive;

    /// Get the state of `ibus.Property`.
    extern fn ibus_property_get_state(p_prop: *Property) ibus.PropState;
    pub const getState = ibus_property_get_state;

    /// Get the IBusPropList of `ibus.Property`.
    extern fn ibus_property_get_sub_props(p_prop: *Property) *ibus.PropList;
    pub const getSubProps = ibus_property_get_sub_props;

    /// Get the symbol of `ibus.Property`.
    extern fn ibus_property_get_symbol(p_prop: *Property) *ibus.Text;
    pub const getSymbol = ibus_property_get_symbol;

    /// Get the tooltip of `ibus.Property`.
    extern fn ibus_property_get_tooltip(p_prop: *Property) *ibus.Text;
    pub const getTooltip = ibus_property_get_tooltip;

    /// Get the visible of `ibus.Property`.
    extern fn ibus_property_get_visible(p_prop: *Property) c_int;
    pub const getVisible = ibus_property_get_visible;

    /// Set the icon of `ibus.Property`.
    extern fn ibus_property_set_icon(p_prop: *Property, p_icon: [*:0]const u8) void;
    pub const setIcon = ibus_property_set_icon;

    /// Set the label of `ibus.Property`.
    extern fn ibus_property_set_label(p_prop: *Property, p_label: *ibus.Text) void;
    pub const setLabel = ibus_property_set_label;

    /// Set whether the `ibus.Property` is sensitive.
    extern fn ibus_property_set_sensitive(p_prop: *Property, p_sensitive: c_int) void;
    pub const setSensitive = ibus_property_set_sensitive;

    /// Set the state of the `ibus.Property`.
    extern fn ibus_property_set_state(p_prop: *Property, p_state: ibus.PropState) void;
    pub const setState = ibus_property_set_state;

    /// Set the sub IBusProperties.
    extern fn ibus_property_set_sub_props(p_prop: *Property, p_prop_list: *ibus.PropList) void;
    pub const setSubProps = ibus_property_set_sub_props;

    /// Set the symbol of `ibus.Property`.
    extern fn ibus_property_set_symbol(p_prop: *Property, p_symbol: *ibus.Text) void;
    pub const setSymbol = ibus_property_set_symbol;

    /// Set the tooltip of `ibus.Property`.
    extern fn ibus_property_set_tooltip(p_prop: *Property, p_tooltip: *ibus.Text) void;
    pub const setTooltip = ibus_property_set_tooltip;

    /// Set whether the `ibus.Property` is visible.
    extern fn ibus_property_set_visible(p_prop: *Property, p_visible: c_int) void;
    pub const setVisible = ibus_property_set_visible;

    /// Update the content of an `ibus.Property`.
    /// `ibus.Property` `prop_update` can either be sub-property of `prop`,
    /// or holds new values for `prop`.
    extern fn ibus_property_update(p_prop: *Property, p_prop_update: *ibus.Property) c_int;
    pub const update = ibus_property_update;

    extern fn ibus_property_get_type() usize;
    pub const getGObjectType = ibus_property_get_type;

    extern fn g_object_ref(p_self: *ibus.Property) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Property) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Property, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusProxy is the base of all proxy objects,
/// which communicate the corresponding `IBusServices` on the other end of
/// IBusConnection.
/// For example, IBus clients (such as editors, web browsers) invoke the proxy
/// object,
/// IBusInputContext to communicate with the InputContext service of the
/// ibus-daemon.
///
/// Almost all services have corresponding proxies, except very simple services.
pub const Proxy = extern struct {
    pub const Parent = gio.DBusProxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = ibus.ProxyClass;
    f_parent: gio.DBusProxy,
    f_flags: u32,
    f_own: c_int,

    pub const virtual_methods = struct {
        /// Dispose the proxy object. If the dbus connection is alive and the own
        /// variable above is TRUE (which is the default),
        /// org.freedesktop.IBus.Service.Destroy method will be called.
        /// Note that "destroy" signal might be emitted when ibus_proxy_destroy is
        /// called or the underlying dbus connection for the proxy is terminated.
        /// In the callback of the destroy signal, you might have to call something
        /// like 'g_object_unref(the_proxy);'.
        pub const destroy = struct {
            pub fn call(p_class: anytype, p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Proxy.Class, p_class).f_destroy.?(gobject.ext.as(Proxy, p_proxy));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Proxy.Class, p_class).f_destroy = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Destroy and free an IBusProxy
        ///
        /// See also:  `ibus.Proxy.destroy`.
        ///
        /// <note><para>Argument `user_data` is ignored in this function.</para></note>
        pub const destroy = struct {
            pub const name = "destroy";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Proxy, p_instance))),
                    gobject.signalLookup("destroy", Proxy.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Dispose the proxy object. If the dbus connection is alive and the own
    /// variable above is TRUE (which is the default),
    /// org.freedesktop.IBus.Service.Destroy method will be called.
    /// Note that "destroy" signal might be emitted when ibus_proxy_destroy is
    /// called or the underlying dbus connection for the proxy is terminated.
    /// In the callback of the destroy signal, you might have to call something
    /// like 'g_object_unref(the_proxy);'.
    extern fn ibus_proxy_destroy(p_proxy: *Proxy) void;
    pub const destroy = ibus_proxy_destroy;

    extern fn ibus_proxy_get_type() usize;
    pub const getGObjectType = ibus_proxy_get_type;

    extern fn g_object_ref(p_self: *ibus.Proxy) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Proxy) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Proxy, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An `ibus.Registry` loads IBus component files and generates the cache files.
///
/// see_also: `ibus.Component`
pub const Registry = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.RegistryClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.RegistryPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when any observed paths are changed.
        /// A method is not associated in this class. the "changed"
        /// signal would be handled in other classes.
        ///
        /// See also: `ibus.Registry.startMonitorChanges`.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Registry, p_instance))),
                    gobject.signalLookup("changed", Registry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `ibus.Registry`
    extern fn ibus_registry_new() *ibus.Registry;
    pub const new = ibus_registry_new;

    /// Check if the registry is updated.
    extern fn ibus_registry_check_modification(p_registry: *Registry) c_int;
    pub const checkModification = ibus_registry_check_modification;

    /// List components.
    extern fn ibus_registry_get_components(p_registry: *Registry) *glib.List;
    pub const getComponents = ibus_registry_get_components;

    /// List observed paths.
    extern fn ibus_registry_get_observed_paths(p_registry: *Registry) *glib.List;
    pub const getObservedPaths = ibus_registry_get_observed_paths;

    /// Read all XML files in a IBus component directory (typically
    /// /usr/share/ibus/component/ *.xml) and update the registry object.
    /// IBUS_COMPONENT_PATH environment valuable is also available for
    /// the custom component directories, whose delimiter is ':'.
    extern fn ibus_registry_load(p_registry: *Registry) void;
    pub const load = ibus_registry_load;

    /// Load the user or system registry cache.
    extern fn ibus_registry_load_cache(p_registry: *Registry, p_is_user: c_int) c_int;
    pub const loadCache = ibus_registry_load_cache;

    /// Load the registry cache `filename`.
    extern fn ibus_registry_load_cache_file(p_registry: *Registry, p_filename: [*:0]const u8) c_int;
    pub const loadCacheFile = ibus_registry_load_cache_file;

    /// Read all XML files in `dirname`, create a `ibus.Component` object for each file,
    /// and add the component objects to the registry.
    /// If `dirname` is "/usr/share/ibus/component", this API and
    /// `ibus.Registry.load` are same.
    extern fn ibus_registry_load_in_dir(p_registry: *Registry, p_dirname: [*:0]const u8) void;
    pub const loadInDir = ibus_registry_load_in_dir;

    /// Output `ibus.Registry` as an XML-formatted string.
    /// The output string can be then shown on the screen or written to file.
    extern fn ibus_registry_output(p_registry: *Registry, p_output: *glib.String, p_indent: c_int) void;
    pub const output = ibus_registry_output;

    /// Save the registry in a user directory or system directory.
    extern fn ibus_registry_save_cache(p_registry: *Registry, p_is_user: c_int) c_int;
    pub const saveCache = ibus_registry_save_cache;

    /// Save the registry cache `filename`.
    extern fn ibus_registry_save_cache_file(p_registry: *Registry, p_filename: [*:0]const u8) c_int;
    pub const saveCacheFile = ibus_registry_save_cache_file;

    /// Start to monitor observed paths.
    extern fn ibus_registry_start_monitor_changes(p_registry: *Registry) void;
    pub const startMonitorChanges = ibus_registry_start_monitor_changes;

    extern fn ibus_registry_get_type() usize;
    pub const getGObjectType = ibus_registry_get_type;

    extern fn g_object_ref(p_self: *ibus.Registry) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Registry) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Registry, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An `ibus.Serializable` is an IBus object which can be serialized, that is,
/// to be to and from a `glib.Variant`.
///
/// This class is to be extended by other class that requires serialization.
/// An extended class should overrides following methods:
/// <itemizedlist>
///    <listitem>
///       <para><function>serialize(object,iter)</function>: for serialize.</para>
///    </listitem>
///    <listitem>
///       <para><function>deserialize(object,iter)</function>: for deserialize.</para>
///    </listitem>
///    <listitem>
///       <para><function>copy(desc,src)</function>: for copy between IBusSerializable.</para>
///    </listitem>
/// </itemizedlist>
/// See `ibus.SerializableSerializeFunc`, `ibus.SerializableDeserializeFunc`, `ibus.SerializableCopyFunc`
/// for function prototype.
pub const Serializable = extern struct {
    pub const Parent = ibus.Object;
    pub const Implements = [_]type{};
    pub const Class = ibus.SerializableClass;
    f_parent: ibus.Object,
    f_priv: ?*ibus.SerializablePrivate,

    pub const virtual_methods = struct {
        pub const copy = struct {
            pub fn call(p_class: anytype, p_dest: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_src: *const ibus.Serializable) c_int {
                return gobject.ext.as(Serializable.Class, p_class).f_copy.?(gobject.ext.as(Serializable, p_dest), p_src);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_dest: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_src: *const ibus.Serializable) callconv(.C) c_int) void {
                gobject.ext.as(Serializable.Class, p_class).f_copy = @ptrCast(p_implementation);
            }
        };

        pub const deserialize = struct {
            pub fn call(p_class: anytype, p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_variant: *glib.Variant) c_int {
                return gobject.ext.as(Serializable.Class, p_class).f_deserialize.?(gobject.ext.as(Serializable, p_object), p_variant);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_variant: *glib.Variant) callconv(.C) c_int) void {
                gobject.ext.as(Serializable.Class, p_class).f_deserialize = @ptrCast(p_implementation);
            }
        };

        pub const serialize = struct {
            pub fn call(p_class: anytype, p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *glib.VariantBuilder) c_int {
                return gobject.ext.as(Serializable.Class, p_class).f_serialize.?(gobject.ext.as(Serializable, p_object), p_builder);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *glib.VariantBuilder) callconv(.C) c_int) void {
                gobject.ext.as(Serializable.Class, p_class).f_serialize = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Deserialize a `glib.Variant` to an `ibus.Serializable`/
    /// The deserialize method should be implemented in extended class.
    extern fn ibus_serializable_deserialize_object(p_variant: *glib.Variant) *ibus.Serializable;
    pub const deserializeObject = ibus_serializable_deserialize_object;

    /// Creates a new instance of an `ibus.Serializable`.
    extern fn ibus_serializable_new() *ibus.Serializable;
    pub const new = ibus_serializable_new;

    /// Clone an `ibus.Serializable`.
    /// The copy method should be implemented in extended class.
    extern fn ibus_serializable_copy(p_serializable: *Serializable) *ibus.Serializable;
    pub const copy = ibus_serializable_copy;

    /// Gets a value from attachment of an `ibus.Serializable`.
    extern fn ibus_serializable_get_qattachment(p_serializable: *Serializable, p_key: glib.Quark) *glib.Variant;
    pub const getQattachment = ibus_serializable_get_qattachment;

    /// Remove a value from attachment of an `ibus.Serializable`.
    /// See also: `ibus_serializable_remove_attachment`.
    extern fn ibus_serializable_remove_qattachment(p_serializable: *Serializable, p_key: glib.Quark) void;
    pub const removeQattachment = ibus_serializable_remove_qattachment;

    /// Serialize an `ibus.Serializable` to a `glib.Variant`.
    /// The serialize method should be implemented in extended class.
    extern fn ibus_serializable_serialize_object(p_serializable: *Serializable) *glib.Variant;
    pub const serializeObject = ibus_serializable_serialize_object;

    /// Attach a value to an `ibus.Serializable`. If the value is floating,
    /// the serializable will take the ownership.
    ///
    /// See also: `ibus_serializable_set_attachment`.
    extern fn ibus_serializable_set_qattachment(p_serializable: *Serializable, p_key: glib.Quark, p_value: *glib.Variant) void;
    pub const setQattachment = ibus_serializable_set_qattachment;

    extern fn ibus_serializable_get_type() usize;
    pub const getGObjectType = ibus_serializable_get_type;

    extern fn g_object_ref(p_self: *ibus.Serializable) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Serializable) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Serializable, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusService is a base class for services.
pub const Service = extern struct {
    pub const Parent = ibus.Object;
    pub const Implements = [_]type{};
    pub const Class = ibus.ServiceClass;
    f_parent: ibus.Object,
    f_priv: ?*ibus.ServicePrivate,

    pub const virtual_methods = struct {
        /// The ::service_get_property class method is to connect
        /// `gio.DBusInterfaceGetPropertyFunc`.
        pub const service_get_property = struct {
            pub fn call(p_class: anytype, p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.Variant {
                return gobject.ext.as(Service.Class, p_class).f_service_get_property.?(gobject.ext.as(Service, p_service), p_connection, p_sender, p_object_path, p_interface_name, p_property_name, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(Service.Class, p_class).f_service_get_property = @ptrCast(p_implementation);
            }
        };

        /// The ::service_method_call class method is to connect
        /// `gio.DBusInterfaceMethodCallFunc`.
        pub const service_method_call = struct {
            pub fn call(p_class: anytype, p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: *glib.Variant, p_invocation: *gio.DBusMethodInvocation) void {
                return gobject.ext.as(Service.Class, p_class).f_service_method_call.?(gobject.ext.as(Service, p_service), p_connection, p_sender, p_object_path, p_interface_name, p_method_name, p_parameters, p_invocation);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: *glib.Variant, p_invocation: *gio.DBusMethodInvocation) callconv(.C) void) void {
                gobject.ext.as(Service.Class, p_class).f_service_method_call = @ptrCast(p_implementation);
            }
        };

        /// The ::service_set_property class method is to connect
        /// `gio.DBusInterfaceSetPropertyFunc`.
        pub const service_set_property = struct {
            pub fn call(p_class: anytype, p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Service.Class, p_class).f_service_set_property.?(gobject.ext.as(Service, p_service), p_connection, p_sender, p_object_path, p_interface_name, p_property_name, p_value, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Service.Class, p_class).f_service_set_property = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The connection of service object.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// The path of service object.
        pub const object_path = struct {
            pub const name = "object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creantes a new `ibus.Service`.
    extern fn ibus_service_new(p_connection: *gio.DBusConnection, p_path: [*:0]const u8) *ibus.Service;
    pub const new = ibus_service_new;

    extern fn ibus_service_emit_signal(p_service: *Service, p_dest_bus_name: [*:0]const u8, p_interface_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: *glib.Variant, p_error: ?*?*glib.Error) c_int;
    pub const emitSignal = ibus_service_emit_signal;

    /// Gets a connections.
    extern fn ibus_service_get_connection(p_service: *Service) *gio.DBusConnection;
    pub const getConnection = ibus_service_get_connection;

    /// Gets the object path of an IBusService.
    extern fn ibus_service_get_object_path(p_service: *Service) [*:0]const u8;
    pub const getObjectPath = ibus_service_get_object_path;

    /// Registers service to a connection.
    extern fn ibus_service_register(p_service: *Service, p_connection: *gio.DBusConnection, p_error: ?*?*glib.Error) c_int;
    pub const register = ibus_service_register;

    /// Unregisters service from a connection.
    extern fn ibus_service_unregister(p_service: *Service, p_connection: *gio.DBusConnection) void;
    pub const unregister = ibus_service_unregister;

    extern fn ibus_service_get_type() usize;
    pub const getGObjectType = ibus_service_get_type;

    extern fn g_object_ref(p_self: *ibus.Service) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Service) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Service, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusText is the main text object in IBus.
/// The text is decorated according to associated IBusAttribute,
/// e.g. the foreground/background color, underline, and
/// applied scope.
///
/// see_also: `ibus.Attribute`
pub const Text = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.TextClass;
    f_parent: ibus.Serializable,
    /// Whether `text` is static, i.e., no need and will not be freed. Only TRUE if IBusText is newed from `ibus.Text.newFromStaticString`.
    f_is_static: c_int,
    /// The string content of IBusText in UTF-8.
    f_text: ?[*:0]u8,
    /// Associated IBusAttributes.
    f_attrs: ?*ibus.AttrList,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `ibus.Text` from a printf expression.
    ///
    /// The result of printf expression is stored in the new IBusText instance.
    extern fn ibus_text_new_from_printf(p_fmt: [*:0]const u8, ...) *ibus.Text;
    pub const newFromPrintf = ibus_text_new_from_printf;

    /// Creates a new `ibus.Text` from a static string.
    ///
    /// Since `str` is a static string which won't be freed.
    /// This function will NOT duplicate `str`.
    extern fn ibus_text_new_from_static_string(p_str: [*:0]const u8) *ibus.Text;
    pub const newFromStaticString = ibus_text_new_from_static_string;

    /// Creates a new `ibus.Text` from a string.
    /// `str` will be duplicated in `ibus.Text`, so feel free to free `str` after this
    /// function.
    extern fn ibus_text_new_from_string(p_str: [*:0]const u8) *ibus.Text;
    pub const newFromString = ibus_text_new_from_string;

    /// Creates a new `ibus.Text` from an UCS-4 encoded string.
    /// `str` will be duplicated in IBusText, so feel free to free `str` after this
    /// function.
    extern fn ibus_text_new_from_ucs4(p_str: *const u32) *ibus.Text;
    pub const newFromUcs4 = ibus_text_new_from_ucs4;

    /// Creates a new `ibus.Text` from a single UCS4-encoded character.
    extern fn ibus_text_new_from_unichar(p_c: u32) *ibus.Text;
    pub const newFromUnichar = ibus_text_new_from_unichar;

    /// Append an IBusAttribute for IBusText.
    extern fn ibus_text_append_attribute(p_text: *Text, p_type: c_uint, p_value: c_uint, p_start_index: c_uint, p_end_index: c_int) void;
    pub const appendAttribute = ibus_text_append_attribute;

    /// Return the attributes in an `ibus.Text`. Should not be freed.
    extern fn ibus_text_get_attributes(p_text: *Text) *ibus.AttrList;
    pub const getAttributes = ibus_text_get_attributes;

    /// Return the is_static in an `ibus.Text`.
    extern fn ibus_text_get_is_static(p_text: *Text) c_int;
    pub const getIsStatic = ibus_text_get_is_static;

    /// Return number of characters in an `ibus.Text`.
    /// This function is based on `glib.utf8Strlen`, so unlike `strlen`,
    /// it does not count by bytes but characters instead.
    extern fn ibus_text_get_length(p_text: *Text) c_uint;
    pub const getLength = ibus_text_get_length;

    /// Return the text in an `ibus.Text`. Should not be freed.
    extern fn ibus_text_get_text(p_text: *Text) [*:0]const u8;
    pub const getText = ibus_text_get_text;

    extern fn ibus_text_set_attributes(p_text: *Text, p_attrs: *ibus.AttrList) void;
    pub const setAttributes = ibus_text_set_attributes;

    extern fn ibus_text_get_type() usize;
    pub const getGObjectType = ibus_text_get_type;

    extern fn g_object_ref(p_self: *ibus.Text) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.Text) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Text, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const UnicodeBlock = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.UnicodeBlockClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.UnicodeBlockPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The Uniode end code point
        pub const end = struct {
            pub const name = "end";

            pub const Type = c_uint;
        };

        /// The Uniode block name
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The Uniode start code point
        pub const start = struct {
            pub const name = "start";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    extern fn ibus_unicode_block_load(p_path: [*:0]const u8) *glib.SList;
    pub const load = ibus_unicode_block_load;

    /// Save the list of `ibus.UnicodeBlock` to the cache file.
    extern fn ibus_unicode_block_save(p_path: [*:0]const u8, p_list: *glib.SList) void;
    pub const save = ibus_unicode_block_save;

    /// Creates a new `ibus.UnicodeBlock`.
    /// block property is required. e.g.
    /// ibus_unicode_block_new ("start", 0x0000, "end", "0x007f", "name", "basic",
    /// NULL)
    extern fn ibus_unicode_block_new(p_first_property_name: [*:0]const u8, ...) *ibus.UnicodeBlock;
    pub const new = ibus_unicode_block_new;

    /// Gets the end code point in `ibus.UnicodeBlock`.
    extern fn ibus_unicode_block_get_end(p_block: *UnicodeBlock) u32;
    pub const getEnd = ibus_unicode_block_get_end;

    /// Gets the name in `ibus.UnicodeBlock`. It should not be freed.
    extern fn ibus_unicode_block_get_name(p_block: *UnicodeBlock) [*:0]const u8;
    pub const getName = ibus_unicode_block_get_name;

    /// Gets the start code point in `ibus.UnicodeBlock`.
    extern fn ibus_unicode_block_get_start(p_block: *UnicodeBlock) u32;
    pub const getStart = ibus_unicode_block_get_start;

    extern fn ibus_unicode_block_get_type() usize;
    pub const getGObjectType = ibus_unicode_block_get_type;

    extern fn g_object_ref(p_self: *ibus.UnicodeBlock) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.UnicodeBlock) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *UnicodeBlock, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Unicode data likes code, name, alias, block-name.
/// You can get extended values with g_object_get_properties.
pub const UnicodeData = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.UnicodeDataClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.UnicodeDataPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The Uniode alias name
        pub const alias = struct {
            pub const name = "alias";

            pub const Type = ?[*:0]u8;
        };

        /// The Uniode block name
        pub const block_name = struct {
            pub const name = "block-name";

            pub const Type = ?[*:0]u8;
        };

        /// The Uniode code point
        pub const code = struct {
            pub const name = "code";

            pub const Type = c_uint;
        };

        /// The Uniode name
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn ibus_unicode_data_load(p_path: [*:0]const u8, p_object: ?*gobject.Object) *glib.SList;
    pub const load = ibus_unicode_data_load;

    /// IBusUnicodeDataLoadAsyncFinish can receive the list of `ibus.UnicodeData`.
    extern fn ibus_unicode_data_load_async(p_path: [*:0]const u8, p_object: ?*gobject.Object, p_cancellable: ?*gio.Cancellable, p_callback: ibus.UnicodeDataLoadAsyncFinish, p_user_data: ?*anyopaque) void;
    pub const loadAsync = ibus_unicode_data_load_async;

    /// Save the list of `ibus.UnicodeData` to the cache file.
    extern fn ibus_unicode_data_save(p_path: [*:0]const u8, p_list: *glib.SList) void;
    pub const save = ibus_unicode_data_save;

    /// Creates a new `ibus.UnicodeData`.
    /// code property is required. e.g.
    /// ibus_unicode_data_new ("code", 0x3042, NULL)
    extern fn ibus_unicode_data_new(p_first_property_name: [*:0]const u8, ...) *ibus.UnicodeData;
    pub const new = ibus_unicode_data_new;

    /// Gets the alias in `ibus.UnicodeData`. It should not be freed.
    extern fn ibus_unicode_data_get_alias(p_unicode: *UnicodeData) [*:0]const u8;
    pub const getAlias = ibus_unicode_data_get_alias;

    /// Gets the block name in `ibus.UnicodeData`. It should not be freed.
    extern fn ibus_unicode_data_get_block_name(p_unicode: *UnicodeData) [*:0]const u8;
    pub const getBlockName = ibus_unicode_data_get_block_name;

    /// Gets the code point in `ibus.UnicodeData`.
    extern fn ibus_unicode_data_get_code(p_unicode: *UnicodeData) u32;
    pub const getCode = ibus_unicode_data_get_code;

    /// Gets the name in `ibus.UnicodeData`. It should not be freed.
    extern fn ibus_unicode_data_get_name(p_unicode: *UnicodeData) [*:0]const u8;
    pub const getName = ibus_unicode_data_get_name;

    /// Sets the block name in `ibus.UnicodeData`.
    extern fn ibus_unicode_data_set_block_name(p_unicode: *UnicodeData, p_block_name: [*:0]const u8) void;
    pub const setBlockName = ibus_unicode_data_set_block_name;

    extern fn ibus_unicode_data_get_type() usize;
    pub const getGObjectType = ibus_unicode_data_get_type;

    extern fn g_object_ref(p_self: *ibus.UnicodeData) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.UnicodeData) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *UnicodeData, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IBusXEvent provides a wrapper of XEvent.
///
/// see_also: `ibus.Component`, `ibus.EngineDesc`
pub const XEvent = extern struct {
    pub const Parent = ibus.Serializable;
    pub const Implements = [_]type{};
    pub const Class = ibus.XEventClass;
    f_parent: ibus.Serializable,
    f_priv: ?*ibus.XEventPrivate,
    f_event_type: ibus.XEventType,
    f_window: c_uint,
    f_send_event: i8,
    f_serial: c_ulong,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// IBusXEventType of this IBusXEvent.
        pub const event_type = struct {
            pub const name = "event-type";

            pub const Type = c_int;
        };

        /// group of this IBusXEvent.
        pub const group = struct {
            pub const name = "group";

            pub const Type = c_uint;
        };

        /// hardware keycode of this IBusXEvent.
        pub const hardware_keycode = struct {
            pub const name = "hardware-keycode";

            pub const Type = c_uint;
        };

        /// is_modifier of this IBusXEvent.
        pub const is_modifier = struct {
            pub const name = "is-modifier";

            pub const Type = c_int;
        };

        /// keyval of this IBusXEvent.
        pub const keyval = struct {
            pub const name = "keyval";

            pub const Type = c_uint;
        };

        /// keyval of this IBusXEvent.
        pub const length = struct {
            pub const name = "length";

            pub const Type = c_int;
        };

        /// purpose of this IBusXEvent.
        pub const purpose = struct {
            pub const name = "purpose";

            pub const Type = ?[*:0]u8;
        };

        /// root window of this IBusXEvent.
        pub const root = struct {
            pub const name = "root";

            pub const Type = c_uint;
        };

        /// same_screen of this IBusXEvent.
        pub const same_screen = struct {
            pub const name = "same-screen";

            pub const Type = c_int;
        };

        /// send_event of this IBusXEvent.
        pub const send_event = struct {
            pub const name = "send-event";

            pub const Type = c_int;
        };

        /// serial of this IBusXEvent.
        pub const serial = struct {
            pub const name = "serial";

            pub const Type = c_ulong;
        };

        /// state of this IBusXEvent.
        pub const state = struct {
            pub const name = "state";

            pub const Type = c_uint;
        };

        /// string of this IBusXEvent.
        pub const string = struct {
            pub const name = "string";

            pub const Type = ?[*:0]u8;
        };

        /// subwindow of this IBusXEvent.
        pub const subwindow = struct {
            pub const name = "subwindow";

            pub const Type = c_uint;
        };

        /// time of this IBusXEvent.
        pub const time = struct {
            pub const name = "time";

            pub const Type = c_uint;
        };

        /// Version of this IBusXEvent.
        pub const version = struct {
            pub const name = "version";

            pub const Type = c_uint;
        };

        /// window of this IBusXEvent.
        pub const window = struct {
            pub const name = "window";

            pub const Type = c_uint;
        };

        /// x of this IBusXEvent.
        pub const x = struct {
            pub const name = "x";

            pub const Type = c_int;
        };

        /// root-x of this IBusXEvent.
        pub const x_root = struct {
            pub const name = "x-root";

            pub const Type = c_int;
        };

        /// x of this IBusXEvent.
        pub const y = struct {
            pub const name = "y";

            pub const Type = c_int;
        };

        /// root-y of this IBusXEvent.
        pub const y_root = struct {
            pub const name = "y-root";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Create a new `ibus.XEvent`.
    extern fn ibus_x_event_new(p_first_property_name: [*:0]const u8, ...) *ibus.XEvent;
    pub const new = ibus_x_event_new;

    extern fn ibus_x_event_get_event_type(p_event: *XEvent) ibus.XEventType;
    pub const getEventType = ibus_x_event_get_event_type;

    extern fn ibus_x_event_get_group(p_event: *XEvent) u8;
    pub const getGroup = ibus_x_event_get_group;

    extern fn ibus_x_event_get_hardware_keycode(p_event: *XEvent) u16;
    pub const getHardwareKeycode = ibus_x_event_get_hardware_keycode;

    extern fn ibus_x_event_get_is_modifier(p_event: *XEvent) c_int;
    pub const getIsModifier = ibus_x_event_get_is_modifier;

    extern fn ibus_x_event_get_keyval(p_event: *XEvent) c_uint;
    pub const getKeyval = ibus_x_event_get_keyval;

    extern fn ibus_x_event_get_length(p_event: *XEvent) c_int;
    pub const getLength = ibus_x_event_get_length;

    extern fn ibus_x_event_get_purpose(p_event: *XEvent) [*:0]const u8;
    pub const getPurpose = ibus_x_event_get_purpose;

    extern fn ibus_x_event_get_root(p_event: *XEvent) u32;
    pub const getRoot = ibus_x_event_get_root;

    extern fn ibus_x_event_get_same_screen(p_event: *XEvent) c_int;
    pub const getSameScreen = ibus_x_event_get_same_screen;

    extern fn ibus_x_event_get_send_event(p_event: *XEvent) i8;
    pub const getSendEvent = ibus_x_event_get_send_event;

    extern fn ibus_x_event_get_serial(p_event: *XEvent) c_ulong;
    pub const getSerial = ibus_x_event_get_serial;

    extern fn ibus_x_event_get_state(p_event: *XEvent) c_uint;
    pub const getState = ibus_x_event_get_state;

    extern fn ibus_x_event_get_string(p_event: *XEvent) [*:0]const u8;
    pub const getString = ibus_x_event_get_string;

    extern fn ibus_x_event_get_subwindow(p_event: *XEvent) u32;
    pub const getSubwindow = ibus_x_event_get_subwindow;

    extern fn ibus_x_event_get_time(p_event: *XEvent) u32;
    pub const getTime = ibus_x_event_get_time;

    extern fn ibus_x_event_get_version(p_event: *XEvent) c_uint;
    pub const getVersion = ibus_x_event_get_version;

    extern fn ibus_x_event_get_window(p_event: *XEvent) u32;
    pub const getWindow = ibus_x_event_get_window;

    extern fn ibus_x_event_get_x(p_event: *XEvent) c_int;
    pub const getX = ibus_x_event_get_x;

    extern fn ibus_x_event_get_x_root(p_event: *XEvent) c_int;
    pub const getXRoot = ibus_x_event_get_x_root;

    extern fn ibus_x_event_get_y(p_event: *XEvent) c_int;
    pub const getY = ibus_x_event_get_y;

    extern fn ibus_x_event_get_y_root(p_event: *XEvent) c_int;
    pub const getYRoot = ibus_x_event_get_y_root;

    extern fn ibus_x_event_get_type() usize;
    pub const getGObjectType = ibus_x_event_get_type;

    extern fn g_object_ref(p_self: *ibus.XEvent) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *ibus.XEvent) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *XEvent, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const AttrListClass = extern struct {
    pub const Instance = ibus.AttrList;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *AttrListClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const AttributeClass = extern struct {
    pub const Instance = ibus.Attribute;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *AttributeClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BusClass = extern struct {
    pub const Instance = ibus.Bus;

    f_parent: ibus.ObjectClass,

    pub fn as(p_instance: *BusClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BusPrivate = opaque {};

pub const ComponentClass = extern struct {
    pub const Instance = ibus.Component;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *ComponentClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ComponentPrivate = opaque {};

pub const ConfigClass = extern struct {
    pub const Instance = ibus.Config;

    f_parent: ibus.ProxyClass,

    pub fn as(p_instance: *ConfigClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ConfigPrivate = opaque {};

pub const ConfigServiceClass = extern struct {
    pub const Instance = ibus.ConfigService;

    f_parent: ibus.ServiceClass,
    f_set_value: ?*const fn (p_config: *ibus.ConfigService, p_section: [*:0]const u8, p_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) callconv(.C) c_int,
    f_get_value: ?*const fn (p_config: *ibus.ConfigService, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant,
    f_unset_value: ?*const fn (p_config: *ibus.ConfigService, p_section: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) c_int,
    f_get_values: ?*const fn (p_config: *ibus.ConfigService, p_section: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant,
    f_pdummy: [12]*anyopaque,

    pub fn as(p_instance: *ConfigServiceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EmojiDataClass = extern struct {
    pub const Instance = ibus.EmojiData;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *EmojiDataClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EmojiDataPrivate = opaque {};

pub const EngineClass = extern struct {
    pub const Instance = ibus.Engine;

    f_parent: ibus.ServiceClass,
    f_process_key_event: ?*const fn (p_engine: *ibus.Engine, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) callconv(.C) c_int,
    f_focus_in: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_focus_out: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_reset: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_enable: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_disable: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_set_cursor_location: ?*const fn (p_engine: *ibus.Engine, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void,
    f_set_capabilities: ?*const fn (p_engine: *ibus.Engine, p_caps: c_uint) callconv(.C) void,
    f_page_up: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_page_down: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_cursor_up: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_cursor_down: ?*const fn (p_engine: *ibus.Engine) callconv(.C) void,
    f_property_activate: ?*const fn (p_engine: *ibus.Engine, p_prop_name: [*:0]const u8, p_prop_state: c_uint) callconv(.C) void,
    f_property_show: ?*const fn (p_engine: *ibus.Engine, p_prop_name: [*:0]const u8) callconv(.C) void,
    f_property_hide: ?*const fn (p_engine: *ibus.Engine, p_prop_name: [*:0]const u8) callconv(.C) void,
    f_candidate_clicked: ?*const fn (p_engine: *ibus.Engine, p_index: c_uint, p_button: c_uint, p_state: c_uint) callconv(.C) void,
    f_set_surrounding_text: ?*const fn (p_engine: *ibus.Engine, p_text: *ibus.Text, p_cursor_index: c_uint, p_anchor_pos: c_uint) callconv(.C) void,
    f_process_hand_writing_event: ?*const fn (p_engine: *ibus.Engine, p_coordinates: *const f64, p_coordinates_len: c_uint) callconv(.C) void,
    f_cancel_hand_writing: ?*const fn (p_engine: *ibus.Engine, p_n_strokes: c_uint) callconv(.C) void,
    f_set_content_type: ?*const fn (p_engine: *ibus.Engine, p_purpose: c_uint, p_hints: c_uint) callconv(.C) void,
    f_focus_in_id: ?*const fn (p_engine: *ibus.Engine, p_object_path: [*:0]const u8, p_client: [*:0]const u8) callconv(.C) void,
    f_focus_out_id: ?*const fn (p_engine: *ibus.Engine, p_object_path: [*:0]const u8) callconv(.C) void,
    f_pdummy: [2]*anyopaque,

    pub fn as(p_instance: *EngineClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EngineDescClass = extern struct {
    pub const Instance = ibus.EngineDesc;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *EngineDescClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EngineDescPrivate = opaque {};

pub const EnginePrivate = opaque {};

pub const EngineSimpleClass = extern struct {
    pub const Instance = ibus.EngineSimple;

    f_parent: ibus.EngineClass,
    f_pdummy: [8]*anyopaque,

    pub fn as(p_instance: *EngineSimpleClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EngineSimplePrivate = opaque {};

pub const ExtensionEventClass = extern struct {
    pub const Instance = ibus.ExtensionEvent;

    f_parent: ibus.SerializableClass,
    f_pdummy: [10]*anyopaque,

    pub fn as(p_instance: *ExtensionEventClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ExtensionEventPrivate = opaque {};

pub const FactoryClass = extern struct {
    pub const Instance = ibus.Factory;

    f_parent: ibus.ServiceClass,
    f_create_engine: ?*const fn (p_factory: *ibus.Factory, p_engine_name: [*:0]const u8) callconv(.C) *ibus.Engine,
    f_pdummy: [7]*anyopaque,

    pub fn as(p_instance: *FactoryClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FactoryPrivate = opaque {};

pub const HotkeyProfileClass = extern struct {
    pub const Instance = ibus.HotkeyProfile;

    f_parent: ibus.SerializableClass,
    f_trigger: ?*const fn (p_profile: *ibus.HotkeyProfile, p_event: glib.Quark, p_user_data: ?*anyopaque) callconv(.C) void,

    pub fn as(p_instance: *HotkeyProfileClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const InputContextClass = extern struct {
    pub const Instance = ibus.InputContext;

    f_parent: ibus.ProxyClass,
    f_pdummy: [24]*anyopaque,

    pub fn as(p_instance: *InputContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const KeymapClass = extern struct {
    pub const Instance = ibus.Keymap;

    f_parent: ibus.ObjectClass,

    pub fn as(p_instance: *KeymapClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const LookupTableClass = extern struct {
    pub const Instance = ibus.LookupTable;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *LookupTableClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ObjectClass = extern struct {
    pub const Instance = ibus.Object;

    f_parent: gobject.InitiallyUnownedClass,
    f_destroy: ?*const fn (p_object: *ibus.Object) callconv(.C) void,
    f_pdummy: [7]*anyopaque,

    pub fn as(p_instance: *ObjectClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ObjectPrivate = opaque {};

pub const ObservedPathClass = extern struct {
    pub const Instance = ibus.ObservedPath;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *ObservedPathClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PanelServiceClass = extern struct {
    pub const Instance = ibus.PanelService;

    f_parent: ibus.ServiceClass,
    f_focus_in: ?*const fn (p_panel: *ibus.PanelService, p_input_context_path: [*:0]const u8) callconv(.C) void,
    f_focus_out: ?*const fn (p_panel: *ibus.PanelService, p_input_context_path: [*:0]const u8) callconv(.C) void,
    f_register_properties: ?*const fn (p_panel: *ibus.PanelService, p_prop_list: *ibus.PropList) callconv(.C) void,
    f_set_cursor_location: ?*const fn (p_panel: *ibus.PanelService, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void,
    f_update_auxiliary_text: ?*const fn (p_panel: *ibus.PanelService, p_text: *ibus.Text, p_visible: c_int) callconv(.C) void,
    f_update_lookup_table: ?*const fn (p_panel: *ibus.PanelService, p_lookup_table: *ibus.LookupTable, p_visible: c_int) callconv(.C) void,
    f_update_preedit_text: ?*const fn (p_panel: *ibus.PanelService, p_text: *ibus.Text, p_cursor_pos: c_uint, p_visible: c_int) callconv(.C) void,
    f_update_property: ?*const fn (p_panel: *ibus.PanelService, p_prop: *ibus.Property) callconv(.C) void,
    f_cursor_down_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_cursor_up_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_hide_auxiliary_text: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_hide_language_bar: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_hide_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_hide_preedit_text: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_page_down_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_page_up_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_reset: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_show_auxiliary_text: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_show_language_bar: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_show_lookup_table: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_show_preedit_text: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_start_setup: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_state_changed: ?*const fn (p_panel: *ibus.PanelService) callconv(.C) void,
    f_destroy_context: ?*const fn (p_panel: *ibus.PanelService, p_input_context_path: [*:0]const u8) callconv(.C) void,
    f_set_content_type: ?*const fn (p_panel: *ibus.PanelService, p_purpose: c_uint, p_hints: c_uint) callconv(.C) void,
    f_set_cursor_location_relative: ?*const fn (p_panel: *ibus.PanelService, p_x: c_int, p_y: c_int, p_w: c_int, p_h: c_int) callconv(.C) void,
    f_panel_extension_received: ?*const fn (p_panel: *ibus.PanelService, p_event: *ibus.ExtensionEvent) callconv(.C) void,
    f_process_key_event: ?*const fn (p_panel: *ibus.PanelService, p_keyval: c_uint, p_keycode: c_uint, p_state: c_uint) callconv(.C) c_int,
    f_commit_text_received: ?*const fn (p_panel: *ibus.PanelService, p_text: *ibus.Text) callconv(.C) void,
    f_candidate_clicked_lookup_table: ?*const fn (p_panel: *ibus.PanelService, p_index: c_uint, p_button: c_uint, p_state: c_uint) callconv(.C) void,
    f_pdummy: [2]*anyopaque,

    pub fn as(p_instance: *PanelServiceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// IBuProcessKeyEventData properties.
pub const ProcessKeyEventData = extern struct {
    f_keyval: c_uint,
    f_keycode: c_uint,
    f_state: c_uint,
};

/// Class structure for `ibus.PropList`.
pub const PropListClass = extern struct {
    pub const Instance = ibus.PropList;

    /// The parent class.
    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *PropListClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PropertyClass = extern struct {
    pub const Instance = ibus.Property;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *PropertyClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PropertyPrivate = opaque {};

pub const ProxyClass = extern struct {
    pub const Instance = ibus.Proxy;

    f_parent: gio.DBusProxyClass,
    f_destroy: ?*const fn (p_proxy: *ibus.Proxy) callconv(.C) void,
    f_pdummy: [7]*anyopaque,

    pub fn as(p_instance: *ProxyClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Rectangle definition.
pub const Rectangle = extern struct {
    /// x coordinate.
    f_x: c_int,
    /// y coordinate.
    f_y: c_int,
    /// width of the rectangle.
    f_width: c_int,
    /// height of the renctangl.
    f_height: c_int,
};

pub const RegistryClass = extern struct {
    pub const Instance = ibus.Registry;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *RegistryClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const RegistryPrivate = opaque {};

pub const SerializableClass = extern struct {
    pub const Instance = ibus.Serializable;

    f_parent: ibus.ObjectClass,
    f_serialize: ?*const fn (p_object: *ibus.Serializable, p_builder: *glib.VariantBuilder) callconv(.C) c_int,
    f_deserialize: ?*const fn (p_object: *ibus.Serializable, p_variant: *glib.Variant) callconv(.C) c_int,
    f_copy: ?*const fn (p_dest: *ibus.Serializable, p_src: *const ibus.Serializable) callconv(.C) c_int,
    f_pdummy: [5]*anyopaque,

    pub fn as(p_instance: *SerializableClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SerializablePrivate = opaque {};

pub const ServiceClass = extern struct {
    pub const Instance = ibus.Service;

    f_parent: ibus.ObjectClass,
    f_service_method_call: ?*const fn (p_service: *ibus.Service, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: *glib.Variant, p_invocation: *gio.DBusMethodInvocation) callconv(.C) void,
    f_service_get_property: ?*const fn (p_service: *ibus.Service, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant,
    f_service_set_property: ?*const fn (p_service: *ibus.Service, p_connection: *gio.DBusConnection, p_sender: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_property_name: [*:0]const u8, p_value: *glib.Variant, p_error: ?*?*glib.Error) callconv(.C) c_int,
    f_interfaces: ?*glib.Array,
    f_pdummy: [4]*anyopaque,

    /// Set the interface introspection information with the service class.
    extern fn ibus_service_class_add_interfaces(p_klass: *ServiceClass, p_xml_data: [*:0]const u8) c_int;
    pub const addInterfaces = ibus_service_class_add_interfaces;

    pub fn as(p_instance: *ServiceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ServicePrivate = opaque {};

pub const TextClass = extern struct {
    pub const Instance = ibus.Text;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *TextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const UnicodeBlockClass = extern struct {
    pub const Instance = ibus.UnicodeBlock;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *UnicodeBlockClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const UnicodeBlockPrivate = opaque {};

pub const UnicodeDataClass = extern struct {
    pub const Instance = ibus.UnicodeData;

    f_parent: ibus.SerializableClass,

    pub fn as(p_instance: *UnicodeDataClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const UnicodeDataPrivate = opaque {};

pub const XEventClass = extern struct {
    pub const Instance = ibus.XEvent;

    f_parent: ibus.SerializableClass,
    f_pdummy: [10]*anyopaque,

    pub fn as(p_instance: *XEventClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const XEventPrivate = opaque {};

/// IBusXML lists data structure and handling function for XML in IBus.
pub const XML = extern struct {
    /// Name of XML tag.
    f_name: ?[*:0]u8,
    /// Text enclosed by XML start tag and end tag. i.e. <tag>text</tag>.
    f_text: ?[*:0]u8,
    /// Attributes of the XML node.
    f_attributes: ?*[*:0]u8,
    /// Children node of this XML node.
    f_sub_nodes: ?*glib.List,

    /// Parse a string buffer which contains an XML-formatted string,
    /// and return a corresponding XML tree.
    extern fn ibus_xml_parse_buffer(p_buffer: [*:0]const u8) *ibus.XML;
    pub const parseBuffer = ibus_xml_parse_buffer;

    /// Parse an XML file and return a corresponding XML tree.
    extern fn ibus_xml_parse_file(p_name: [*:0]const u8) *ibus.XML;
    pub const parseFile = ibus_xml_parse_file;

    /// Creates a copy of `node`, which should be freed with
    /// `ibus.XML.free`. Primarily used by language bindings,
    /// not that useful otherwise (since `node` can just be copied
    /// by assignment in C).
    extern fn ibus_xml_copy(p_node: *const XML) *ibus.XML;
    pub const copy = ibus_xml_copy;

    /// Free an XML tree.
    extern fn ibus_xml_free(p_node: *XML) void;
    pub const free = ibus_xml_free;

    /// Output an XML tree to a GString.
    extern fn ibus_xml_output(p_node: *const XML, p_output: *glib.String) void;
    pub const output = ibus_xml_output;

    extern fn ibus_xml_get_type() usize;
    pub const getGObjectType = ibus_xml_get_type;
};

/// Type enumeration of IBusText attribute.
pub const AttrType = enum(c_int) {
    underline = 1,
    foreground = 2,
    background = 3,
    _,

    extern fn ibus_attr_type_get_type() usize;
    pub const getGObjectType = ibus_attr_type_get_type;
};

/// Type of IBusText attribute.
pub const AttrUnderline = enum(c_int) {
    none = 0,
    single = 1,
    double = 2,
    low = 3,
    @"error" = 4,
    _,

    /// Creates a new underline `ibus.Attribute`.
    extern fn ibus_attr_underline_new(p_underline_type: c_uint, p_start_index: c_uint, p_end_index: c_uint) *ibus.Attribute;
    pub const new = ibus_attr_underline_new;

    extern fn ibus_attr_underline_get_type() usize;
    pub const getGObjectType = ibus_attr_underline_get_type;
};

pub const BusRequestNameReply = enum(c_int) {
    primary_owner = 1,
    in_queue = 2,
    exists = 3,
    already_owner = 4,
    _,

    extern fn ibus_bus_request_name_reply_get_type() usize;
    pub const getGObjectType = ibus_bus_request_name_reply_get_type;
};

pub const BusStartServiceByNameReply = enum(c_int) {
    success = 1,
    already_running = 2,
    _,

    extern fn ibus_bus_start_service_by_name_reply_get_type() usize;
    pub const getGObjectType = ibus_bus_start_service_by_name_reply_get_type;
};

pub const Error = enum(c_int) {
    no_engine = 0,
    no_config = 1,
    failed = 2,
    _,

    extern fn ibus_error_quark() glib.Quark;
    pub const quark = ibus_error_quark;

    extern fn ibus_error_get_type() usize;
    pub const getGObjectType = ibus_error_get_type;
};

/// Describes primary purpose of the input context.  This information
/// is particularly useful to implement intelligent behavior in
/// engines, such as automatic input-mode switch and text prediction.
///
/// Note that the purpose is not meant to impose a totally strict rule
/// about allowed characters, and does not replace input validation.
/// It is fine for an on-screen keyboard to let the user override the
/// character set restriction that is expressed by the purpose. The
/// application is expected to validate the entry contents, even if
/// it specified a purpose.
///
/// The difference between `IBUS_INPUT_PURPOSE_DIGITS` and
/// `IBUS_INPUT_PURPOSE_NUMBER` is that the former accepts only digits
/// while the latter also some punctuation (like commas or points, plus,
/// minus) and “e” or “E” as in 3.14E+000.
///
/// This enumeration may be extended in the future; engines should
/// interpret unknown values as 'free form'.
pub const InputPurpose = enum(c_int) {
    free_form = 0,
    alpha = 1,
    digits = 2,
    number = 3,
    phone = 4,
    url = 5,
    email = 6,
    name = 7,
    password = 8,
    pin = 9,
    terminal = 10,
    _,

    extern fn ibus_input_purpose_get_type() usize;
    pub const getGObjectType = ibus_input_purpose_get_type;
};

/// Orientation of UI.
pub const Orientation = enum(c_int) {
    horizontal = 0,
    vertical = 1,
    system = 2,
    _,

    extern fn ibus_orientation_get_type() usize;
    pub const getGObjectType = ibus_orientation_get_type;
};

/// Pre-edit commit mode when the focus is lost.
pub const PreeditFocusMode = enum(c_int) {
    clear = 0,
    commit = 1,
    _,

    extern fn ibus_preedit_focus_mode_get_type() usize;
    pub const getGObjectType = ibus_preedit_focus_mode_get_type;
};

/// State of `ibus.Property`. The actual effect depends on `ibus.PropType` of the
/// IBusProperty.
///
/// <variablelist>
///     <varlistentry>
///         <term>PROP_TYPE_TOGGLE</term>
///         <listitem><para>Emphasized if PROP_STATE_CHECKED, normal otherwise.</para></listitem>
///     </varlistentry>
///     <varlistentry>
///         <term>PROP_TYPE_RADIO</term>
///         <listitem><para>Option checked if PROP_STATE_CHECKED, unchecked otherwise.</para></listitem>
///     </varlistentry>
/// </variablelist>
/// No effect on other types.
pub const PropState = enum(c_int) {
    unchecked = 0,
    checked = 1,
    inconsistent = 2,
    _,

    extern fn ibus_prop_state_get_type() usize;
    pub const getGObjectType = ibus_prop_state_get_type;
};

/// Type enumeration of IBusProperty.
pub const PropType = enum(c_int) {
    normal = 0,
    toggle = 1,
    radio = 2,
    menu = 3,
    separator = 4,
    _,

    extern fn ibus_prop_type_get_type() usize;
    pub const getGObjectType = ibus_prop_type_get_type;
};

pub const XEventType = enum(c_int) {
    nothing = -1,
    key_press = 0,
    key_release = 1,
    other = 2,
    event_last = 3,
    _,

    extern fn ibus_xevent_type_get_type() usize;
    pub const getGObjectType = ibus_xevent_type_get_type;
};

pub const BusNameFlag = packed struct(c_uint) {
    allow_replacement: bool = false,
    replace_existing: bool = false,
    do_not_queue: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_allow_replacement: BusNameFlag = @bitCast(@as(c_uint, 1));
    const flags_replace_existing: BusNameFlag = @bitCast(@as(c_uint, 2));
    const flags_do_not_queue: BusNameFlag = @bitCast(@as(c_uint, 4));
    extern fn ibus_bus_name_flag_get_type() usize;
    pub const getGObjectType = ibus_bus_name_flag_get_type;
};

/// Capability flags of UI.
pub const Capabilite = packed struct(c_uint) {
    preedit_text: bool = false,
    auxiliary_text: bool = false,
    lookup_table: bool = false,
    focus: bool = false,
    property: bool = false,
    surrounding_text: bool = false,
    osk: bool = false,
    sync_process_key: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_preedit_text: Capabilite = @bitCast(@as(c_uint, 1));
    const flags_auxiliary_text: Capabilite = @bitCast(@as(c_uint, 2));
    const flags_lookup_table: Capabilite = @bitCast(@as(c_uint, 4));
    const flags_focus: Capabilite = @bitCast(@as(c_uint, 8));
    const flags_property: Capabilite = @bitCast(@as(c_uint, 16));
    const flags_surrounding_text: Capabilite = @bitCast(@as(c_uint, 32));
    const flags_osk: Capabilite = @bitCast(@as(c_uint, 64));
    const flags_sync_process_key: Capabilite = @bitCast(@as(c_uint, 128));
    const flags_sync_process_key_v2: Capabilite = @bitCast(@as(c_uint, 128));
    extern fn ibus_capabilite_get_type() usize;
    pub const getGObjectType = ibus_capabilite_get_type;
};

/// Describes hints that might be taken into account by engines.  Note
/// that engines may already tailor their behaviour according to the
/// `ibus.InputPurpose` of the entry.
///
/// Some common sense is expected when using these flags - mixing
/// `IBUS_INPUT_HINT_LOWERCASE` with any of the uppercase hints makes no sense.
///
/// This enumeration may be extended in the future; engines should
/// ignore unknown values.
pub const InputHints = packed struct(c_uint) {
    spellcheck: bool = false,
    no_spellcheck: bool = false,
    word_completion: bool = false,
    lowercase: bool = false,
    uppercase_chars: bool = false,
    uppercase_words: bool = false,
    uppercase_sentences: bool = false,
    inhibit_osk: bool = false,
    vertical_writing: bool = false,
    emoji: bool = false,
    no_emoji: bool = false,
    private: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: InputHints = @bitCast(@as(c_uint, 0));
    const flags_spellcheck: InputHints = @bitCast(@as(c_uint, 1));
    const flags_no_spellcheck: InputHints = @bitCast(@as(c_uint, 2));
    const flags_word_completion: InputHints = @bitCast(@as(c_uint, 4));
    const flags_lowercase: InputHints = @bitCast(@as(c_uint, 8));
    const flags_uppercase_chars: InputHints = @bitCast(@as(c_uint, 16));
    const flags_uppercase_words: InputHints = @bitCast(@as(c_uint, 32));
    const flags_uppercase_sentences: InputHints = @bitCast(@as(c_uint, 64));
    const flags_inhibit_osk: InputHints = @bitCast(@as(c_uint, 128));
    const flags_vertical_writing: InputHints = @bitCast(@as(c_uint, 256));
    const flags_emoji: InputHints = @bitCast(@as(c_uint, 512));
    const flags_no_emoji: InputHints = @bitCast(@as(c_uint, 1024));
    const flags_private: InputHints = @bitCast(@as(c_uint, 2048));
    extern fn ibus_input_hints_get_type() usize;
    pub const getGObjectType = ibus_input_hints_get_type;
};

/// Handles key modifier such as control, shift and alt and release event.
/// Note that nits 15 - 25 are currently unused, while bit 29 is used internally.
pub const ModifierType = packed struct(c_uint) {
    shift_mask: bool = false,
    lock_mask: bool = false,
    control_mask: bool = false,
    mod1_mask: bool = false,
    mod2_mask: bool = false,
    mod3_mask: bool = false,
    mod4_mask: bool = false,
    mod5_mask: bool = false,
    button1_mask: bool = false,
    button2_mask: bool = false,
    button3_mask: bool = false,
    button4_mask: bool = false,
    button5_mask: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    handled_mask: bool = false,
    forward_mask: bool = false,
    super_mask: bool = false,
    hyper_mask: bool = false,
    meta_mask: bool = false,
    _padding29: bool = false,
    release_mask: bool = false,
    _padding31: bool = false,

    const flags_shift_mask: ModifierType = @bitCast(@as(c_uint, 1));
    const flags_lock_mask: ModifierType = @bitCast(@as(c_uint, 2));
    const flags_control_mask: ModifierType = @bitCast(@as(c_uint, 4));
    const flags_mod1_mask: ModifierType = @bitCast(@as(c_uint, 8));
    const flags_mod2_mask: ModifierType = @bitCast(@as(c_uint, 16));
    const flags_mod3_mask: ModifierType = @bitCast(@as(c_uint, 32));
    const flags_mod4_mask: ModifierType = @bitCast(@as(c_uint, 64));
    const flags_mod5_mask: ModifierType = @bitCast(@as(c_uint, 128));
    const flags_button1_mask: ModifierType = @bitCast(@as(c_uint, 256));
    const flags_button2_mask: ModifierType = @bitCast(@as(c_uint, 512));
    const flags_button3_mask: ModifierType = @bitCast(@as(c_uint, 1024));
    const flags_button4_mask: ModifierType = @bitCast(@as(c_uint, 2048));
    const flags_button5_mask: ModifierType = @bitCast(@as(c_uint, 4096));
    const flags_handled_mask: ModifierType = @bitCast(@as(c_uint, 16777216));
    const flags_forward_mask: ModifierType = @bitCast(@as(c_uint, 33554432));
    const flags_ignored_mask: ModifierType = @bitCast(@as(c_uint, 33554432));
    const flags_super_mask: ModifierType = @bitCast(@as(c_uint, 67108864));
    const flags_hyper_mask: ModifierType = @bitCast(@as(c_uint, 134217728));
    const flags_meta_mask: ModifierType = @bitCast(@as(c_uint, 268435456));
    const flags_release_mask: ModifierType = @bitCast(@as(c_uint, 1073741824));
    const flags_modifier_mask: ModifierType = @bitCast(@as(c_uint, 1593843711));
    extern fn ibus_modifier_type_get_type() usize;
    pub const getGObjectType = ibus_modifier_type_get_type;
};

/// The flags are used internally.
pub const ObjectFlags = packed struct(c_uint) {
    in_destruction: bool = false,
    destroyed: bool = false,
    reserved_1: bool = false,
    reserved_2: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_in_destruction: ObjectFlags = @bitCast(@as(c_uint, 1));
    const flags_destroyed: ObjectFlags = @bitCast(@as(c_uint, 2));
    const flags_reserved_1: ObjectFlags = @bitCast(@as(c_uint, 4));
    const flags_reserved_2: ObjectFlags = @bitCast(@as(c_uint, 8));
    extern fn ibus_object_flags_get_type() usize;
    pub const getGObjectType = ibus_object_flags_get_type;
};

/// Converts an accelerator keyval and modifier mask into a string
/// parseable by `gtk_accelerator_parse`. For example, if you pass in
/// `IBUS_KEY_q` and `IBUS_CONTROL_MASK`, this function returns “&lt;Control&gt;q”.
///
/// If you need to display accelerators in the user interface,
/// see `gtk_accelerator_get_label`.
extern fn ibus_accelerator_name(p_accelerator_key: c_uint, p_accelerator_mods: ibus.ModifierType) [*:0]u8;
pub const acceleratorName = ibus_accelerator_name;

/// Parses a string representing an accelerator. The format looks like
/// “&lt;Control&gt;a” or “&lt;Shift&gt;&lt;Alt&gt;F1” or “&lt;Release`gt`;z”
/// (the last one is for key release).
///
/// The parser is fairly liberal and allows lower or upper case, and also
/// abbreviations such as “&lt;Ctl&gt;” and “&lt;Ctrl&gt;”. Key names are
/// parsed using `gdk_keyval_from_name`. For character keys the name is not the
/// symbol, but the lowercase name, e.g. one would use “&lt;Ctrl&gt;minus”
/// instead of “&lt;Ctrl&gt;-”.
///
/// If the parse fails, `accelerator_key` and `accelerator_mods` will
/// be set to 0 (zero).
extern fn ibus_accelerator_parse(p_accelerator: [*:0]const u8, p_accelerator_key: ?*c_uint, p_accelerator_mods: ?*ibus.ModifierType) void;
pub const acceleratorParse = ibus_accelerator_parse;

/// Determines whether a given keyval and modifier mask constitute
/// a valid keyboard accelerator. For example, the `IBUS_KEY_a` keyval
/// plus `IBUS_CONTROL_MASK` is valid - this is a “Ctrl+a” accelerator.
/// But, you can't, for instance, use the `IBUS_KEY_Control_L` keyval
/// as an accelerator.
extern fn ibus_accelerator_valid(p_keyval: c_uint, p_modifiers: ibus.ModifierType) c_int;
pub const acceleratorValid = ibus_accelerator_valid;

/// Creates a new background `ibus.Attribute`.
extern fn ibus_attr_background_new(p_color: c_uint, p_start_index: c_uint, p_end_index: c_uint) *ibus.Attribute;
pub const attrBackgroundNew = ibus_attr_background_new;

/// Creates a new foreground `ibus.Attribute`.
extern fn ibus_attr_foreground_new(p_color: c_uint, p_start_index: c_uint, p_end_index: c_uint) *ibus.Attribute;
pub const attrForegroundNew = ibus_attr_foreground_new;

extern fn ibus_emoji_dict_load(p_path: [*:0]const u8) *glib.HashTable;
pub const emojiDictLoad = ibus_emoji_dict_load;

extern fn ibus_emoji_dict_lookup(p_dict: *glib.HashTable, p_emoji: [*:0]const u8) *ibus.EmojiData;
pub const emojiDictLookup = ibus_emoji_dict_lookup;

/// Saves the Emoji dictionary to the cache file.
/// Recommend to use `ibus.EmojiData.save` instead becase GSList in
/// GHashTable does not work with Gir and Vala.
/// Calls `ibus.EmojiData.save` internally. The format of the hash table
/// changed and now is { emoji character, `ibus.EmojiData` object }.
extern fn ibus_emoji_dict_save(p_path: [*:0]const u8, p_dict: *glib.HashTable) void;
pub const emojiDictSave = ibus_emoji_dict_save;

/// Free a list of strings.
extern fn ibus_free_strv(p_strv: *[*:0]u8) void;
pub const freeStrv = ibus_free_strv;

/// Return the D-Bus address of IBus.
/// It will find the address from following source:
/// <orderedlist>
///    <listitem><para>Environment variable IBUS_ADDRESS</para></listitem>
///    <listitem><para>Socket file under ~/.config/ibus/bus/</para></listitem>
/// </orderedlist>
extern fn ibus_get_address() [*:0]const u8;
pub const getAddress = ibus_get_address;

/// Get UID of ibus-daemon.
extern fn ibus_get_daemon_uid() c_long;
pub const getDaemonUid = ibus_get_daemon_uid;

extern fn ibus_get_language_name(p__locale: [*:0]const u8) [*:0]u8;
pub const getLanguageName = ibus_get_language_name;

/// Obtains the machine UUID of the machine this process is running on.
extern fn ibus_get_local_machine_id() [*:0]const u8;
pub const getLocalMachineId = ibus_get_local_machine_id;

/// Get the path of socket file.
extern fn ibus_get_socket_path() [*:0]const u8;
pub const getSocketPath = ibus_get_socket_path;

/// Get the GDBus timeout in milliseconds. The timeout is for clients (e.g.
/// im-ibus.so), not for ibus-daemon.
/// Note that the timeout for ibus-daemon could be set by --timeout command
/// line option of the daemon.
extern fn ibus_get_timeout() c_int;
pub const getTimeout = ibus_get_timeout;

extern fn ibus_get_untranslated_language_name(p__locale: [*:0]const u8) [*:0]u8;
pub const getUntranslatedLanguageName = ibus_get_untranslated_language_name;

/// Get the current user name.
/// It is determined by:
/// <orderedlist>
///    <listitem><para>`getlogin`</para></listitem>
///    <listitem><para>Environment variable SUDO_USER</para></listitem>
///    <listitem><para>Environment variable USERHELPER_UID</para></listitem>
///    <listitem><para>Environment variable USERNAME</para></listitem>
///    <listitem><para>Environment variable LOGNAME</para></listitem>
///    <listitem><para>Environment variable USER</para></listitem>
///    <listitem><para>Environment variable LNAME</para></listitem>
/// </orderedlist>
extern fn ibus_get_user_name() [*:0]const u8;
pub const getUserName = ibus_get_user_name;

/// Initialize the ibus types.
extern fn ibus_init() void;
pub const init = ibus_init;

/// Parse key event string and return key symbol and modifiers.
extern fn ibus_key_event_from_string(p_string: [*:0]const u8, p_keyval: *c_uint, p_modifiers: *c_uint) c_int;
pub const keyEventFromString = ibus_key_event_from_string;

/// Return the name of a key symbol and modifiers.
///
/// For example, if press ctrl, shift, and enter, then this function returns:
/// Shift+Control+enter.
extern fn ibus_key_event_to_string(p_keyval: c_uint, p_modifiers: c_uint) [*:0]u8;
pub const keyEventToString = ibus_key_event_to_string;

/// Obtains the upper- and lower-case versions of the keyval `symbol`.
/// Examples of keyvals are `IBUS_KEY_a`, `IBUS_KEY_Return`, `IBUS_KEY_F1`, etc.
extern fn ibus_keyval_convert_case(p_symbol: c_uint, p_lower: *c_uint, p_upper: *c_uint) void;
pub const keyvalConvertCase = ibus_keyval_convert_case;

/// Return the key symbol that associate with the key name.
extern fn ibus_keyval_from_name(p_keyval_name: [*:0]const u8) c_uint;
pub const keyvalFromName = ibus_keyval_from_name;

/// Return the name of a key symbol.
///
/// Note that the returned string is used internally, so don't free it.
extern fn ibus_keyval_name(p_keyval: c_uint) [*:0]const u8;
pub const keyvalName = ibus_keyval_name;

/// Converts a key value to lower case, if applicable.
extern fn ibus_keyval_to_lower(p_keyval: c_uint) c_uint;
pub const keyvalToLower = ibus_keyval_to_lower;

/// Convert from an IBus key symbol to the corresponding ISO10646 (Unicode)
/// character.
extern fn ibus_keyval_to_unicode(p_keyval: c_uint) u32;
pub const keyvalToUnicode = ibus_keyval_to_unicode;

/// Converts a key value to upper case, if applicable.
extern fn ibus_keyval_to_upper(p_keyval: c_uint) c_uint;
pub const keyvalToUpper = ibus_keyval_to_upper;

/// Runs an IBus main loop until `ibus.quit` is called in the loop.
///
/// See also: `ibus.quit`.
extern fn ibus_main() void;
pub const main = ibus_main;

/// Stops an IBus from running.
///
/// Any calls to `ibus.quit` for the loop will return.
/// See also: `ibus.main`.
extern fn ibus_quit() void;
pub const quit = ibus_quit;

/// Set the display address.
extern fn ibus_set_display(p_display: [*:0]const u8) void;
pub const setDisplay = ibus_set_display;

/// Sets GLIB's log handler to ours. Our log handler adds time info
/// including hour, minute, second, and microsecond, like:
///
/// (ibus-daemon:7088): IBUS-DEBUG: 18:06:45.822819: ibus-daemon started
///
/// If `verbose` is `TRUE`, all levels of messages will be logged. Otherwise,
/// DEBUG and WARNING messages will be ignored.  The function is used in
/// ibus-daemon, but can be useful for IBus client programs as well for
/// debugging. It's totally fine for not calling this function. If you
/// don't set a custom GLIB log handler, the default GLIB log handler will
/// be used.
extern fn ibus_set_log_handler(p_verbose: c_int) void;
pub const setLogHandler = ibus_set_log_handler;

/// Convert from a ISO10646 character to a key symbol.
extern fn ibus_unicode_to_keyval(p_wc: u32) c_uint;
pub const unicodeToKeyval = ibus_unicode_to_keyval;

/// Remove the log handler which is set by ibus_set_log_handler.
extern fn ibus_unset_log_handler() void;
pub const unsetLogHandler = ibus_unset_log_handler;

/// Write D-Bus address to socket file.
///
/// See also: `ibus.getAddress`.
extern fn ibus_write_address(p_address: [*:0]const u8) void;
pub const writeAddress = ibus_write_address;

/// Free function prototype.
pub const FreeFunc = *const fn (p_object: ?*anyopaque) callconv(.C) void;

pub const ObjectDestroyFunc = *const fn (p_object: *ibus.Object) callconv(.C) void;

/// Prototype of copy function.
/// Copy function copy from source `ibus.Serializable` to the destination one.
/// Returns a gboolean value which indicates whether the copying is success.
pub const SerializableCopyFunc = *const fn (p_dest: *ibus.Serializable, p_src: *const ibus.Serializable) callconv(.C) c_int;

/// Prototype of deserialize function.
/// Deserialize function convert a `glib.Variant` to `ibus.Serializable`.
/// Returns an integer value which indicates how many values in
/// the variant(tuple) are consumed.
pub const SerializableDeserializeFunc = *const fn (p_serializable: *ibus.Serializable, p_variant: *glib.Variant) callconv(.C) c_int;

/// Prototype of serialize function.
/// Serialize function convert an `ibus.Serializable` to `glib.VariantBuilder`.
/// Returns a gboolean value which indicates whether the conversion is success.
/// Return `TRUE` if succeed.
pub const SerializableSerializeFunc = *const fn (p_serializable: *ibus.Serializable, p_builder: *glib.VariantBuilder) callconv(.C) c_int;

/// This callback can receive the list of `ibus.UnicodeData`.
pub const UnicodeDataLoadAsyncFinish = *const fn (p_data_list: *glib.SList, p_user_data: ?*anyopaque) callconv(.C) void;

pub const @"0" = 48;
pub const @"1" = 49;
pub const @"2" = 50;
pub const @"3" = 51;
pub const @"3270_AltCursor" = 64784;
pub const @"3270_Attn" = 64782;
pub const @"3270_BackTab" = 64773;
pub const @"3270_ChangeScreen" = 64793;
pub const @"3270_Copy" = 64789;
pub const @"3270_CursorBlink" = 64783;
pub const @"3270_CursorSelect" = 64796;
pub const @"3270_DeleteWord" = 64794;
pub const @"3270_Duplicate" = 64769;
pub const @"3270_Enter" = 64798;
pub const @"3270_EraseEOF" = 64774;
pub const @"3270_EraseInput" = 64775;
pub const @"3270_ExSelect" = 64795;
pub const @"3270_FieldMark" = 64770;
pub const @"3270_Ident" = 64787;
pub const @"3270_Jump" = 64786;
pub const @"3270_KeyClick" = 64785;
pub const @"3270_Left2" = 64772;
pub const @"3270_PA1" = 64778;
pub const @"3270_PA2" = 64779;
pub const @"3270_PA3" = 64780;
pub const @"3270_Play" = 64790;
pub const @"3270_PrintScreen" = 64797;
pub const @"3270_Quit" = 64777;
pub const @"3270_Record" = 64792;
pub const @"3270_Reset" = 64776;
pub const @"3270_Right2" = 64771;
pub const @"3270_Rule" = 64788;
pub const @"3270_Setup" = 64791;
pub const @"3270_Test" = 64781;
pub const @"4" = 52;
pub const @"5" = 53;
pub const @"6" = 54;
pub const @"7" = 55;
pub const @"8" = 56;
pub const @"9" = 57;
pub const A = 65;
pub const AE = 198;
pub const Aacute = 193;
pub const Abelowdot = 16785056;
pub const Abreve = 451;
pub const Abreveacute = 16785070;
pub const Abrevebelowdot = 16785078;
pub const Abrevegrave = 16785072;
pub const Abrevehook = 16785074;
pub const Abrevetilde = 16785076;
pub const AccessX_Enable = 65136;
pub const AccessX_Feedback_Enable = 65137;
pub const Acircumflex = 194;
pub const Acircumflexacute = 16785060;
pub const Acircumflexbelowdot = 16785068;
pub const Acircumflexgrave = 16785062;
pub const Acircumflexhook = 16785064;
pub const Acircumflextilde = 16785066;
pub const Adiaeresis = 196;
pub const Agrave = 192;
pub const Ahook = 16785058;
pub const Alt_L = 65513;
pub const Alt_R = 65514;
pub const Amacron = 960;
pub const Aogonek = 417;
pub const Arabic_0 = 16778848;
pub const Arabic_1 = 16778849;
pub const Arabic_2 = 16778850;
pub const Arabic_3 = 16778851;
pub const Arabic_4 = 16778852;
pub const Arabic_5 = 16778853;
pub const Arabic_6 = 16778854;
pub const Arabic_7 = 16778855;
pub const Arabic_8 = 16778856;
pub const Arabic_9 = 16778857;
pub const Arabic_ain = 1497;
pub const Arabic_alef = 1479;
pub const Arabic_alefmaksura = 1513;
pub const Arabic_beh = 1480;
pub const Arabic_comma = 1452;
pub const Arabic_dad = 1494;
pub const Arabic_dal = 1487;
pub const Arabic_damma = 1519;
pub const Arabic_dammatan = 1516;
pub const Arabic_ddal = 16778888;
pub const Arabic_farsi_yeh = 16778956;
pub const Arabic_fatha = 1518;
pub const Arabic_fathatan = 1515;
pub const Arabic_feh = 1505;
pub const Arabic_fullstop = 16778964;
pub const Arabic_gaf = 16778927;
pub const Arabic_ghain = 1498;
pub const Arabic_ha = 1511;
pub const Arabic_hah = 1485;
pub const Arabic_hamza = 1473;
pub const Arabic_hamza_above = 16778836;
pub const Arabic_hamza_below = 16778837;
pub const Arabic_hamzaonalef = 1475;
pub const Arabic_hamzaonwaw = 1476;
pub const Arabic_hamzaonyeh = 1478;
pub const Arabic_hamzaunderalef = 1477;
pub const Arabic_heh = 1511;
pub const Arabic_heh_doachashmee = 16778942;
pub const Arabic_heh_goal = 16778945;
pub const Arabic_jeem = 1484;
pub const Arabic_jeh = 16778904;
pub const Arabic_kaf = 1507;
pub const Arabic_kasra = 1520;
pub const Arabic_kasratan = 1517;
pub const Arabic_keheh = 16778921;
pub const Arabic_khah = 1486;
pub const Arabic_lam = 1508;
pub const Arabic_madda_above = 16778835;
pub const Arabic_maddaonalef = 1474;
pub const Arabic_meem = 1509;
pub const Arabic_noon = 1510;
pub const Arabic_noon_ghunna = 16778938;
pub const Arabic_peh = 16778878;
pub const Arabic_percent = 16778858;
pub const Arabic_qaf = 1506;
pub const Arabic_question_mark = 1471;
pub const Arabic_ra = 1489;
pub const Arabic_rreh = 16778897;
pub const Arabic_sad = 1493;
pub const Arabic_seen = 1491;
pub const Arabic_semicolon = 1467;
pub const Arabic_shadda = 1521;
pub const Arabic_sheen = 1492;
pub const Arabic_sukun = 1522;
pub const Arabic_superscript_alef = 16778864;
pub const Arabic_switch = 65406;
pub const Arabic_tah = 1495;
pub const Arabic_tatweel = 1504;
pub const Arabic_tcheh = 16778886;
pub const Arabic_teh = 1482;
pub const Arabic_tehmarbuta = 1481;
pub const Arabic_thal = 1488;
pub const Arabic_theh = 1483;
pub const Arabic_tteh = 16778873;
pub const Arabic_veh = 16778916;
pub const Arabic_waw = 1512;
pub const Arabic_yeh = 1514;
pub const Arabic_yeh_baree = 16778962;
pub const Arabic_zah = 1496;
pub const Arabic_zain = 1490;
pub const Aring = 197;
pub const Armenian_AT = 16778552;
pub const Armenian_AYB = 16778545;
pub const Armenian_BEN = 16778546;
pub const Armenian_CHA = 16778569;
pub const Armenian_DA = 16778548;
pub const Armenian_DZA = 16778561;
pub const Armenian_E = 16778551;
pub const Armenian_FE = 16778582;
pub const Armenian_GHAT = 16778562;
pub const Armenian_GIM = 16778547;
pub const Armenian_HI = 16778565;
pub const Armenian_HO = 16778560;
pub const Armenian_INI = 16778555;
pub const Armenian_JE = 16778571;
pub const Armenian_KE = 16778580;
pub const Armenian_KEN = 16778559;
pub const Armenian_KHE = 16778557;
pub const Armenian_LYUN = 16778556;
pub const Armenian_MEN = 16778564;
pub const Armenian_NU = 16778566;
pub const Armenian_O = 16778581;
pub const Armenian_PE = 16778570;
pub const Armenian_PYUR = 16778579;
pub const Armenian_RA = 16778572;
pub const Armenian_RE = 16778576;
pub const Armenian_SE = 16778573;
pub const Armenian_SHA = 16778567;
pub const Armenian_TCHE = 16778563;
pub const Armenian_TO = 16778553;
pub const Armenian_TSA = 16778558;
pub const Armenian_TSO = 16778577;
pub const Armenian_TYUN = 16778575;
pub const Armenian_VEV = 16778574;
pub const Armenian_VO = 16778568;
pub const Armenian_VYUN = 16778578;
pub const Armenian_YECH = 16778549;
pub const Armenian_ZA = 16778550;
pub const Armenian_ZHE = 16778554;
pub const Armenian_accent = 16778587;
pub const Armenian_amanak = 16778588;
pub const Armenian_apostrophe = 16778586;
pub const Armenian_at = 16778600;
pub const Armenian_ayb = 16778593;
pub const Armenian_ben = 16778594;
pub const Armenian_but = 16778589;
pub const Armenian_cha = 16778617;
pub const Armenian_da = 16778596;
pub const Armenian_dza = 16778609;
pub const Armenian_e = 16778599;
pub const Armenian_exclam = 16778588;
pub const Armenian_fe = 16778630;
pub const Armenian_full_stop = 16778633;
pub const Armenian_ghat = 16778610;
pub const Armenian_gim = 16778595;
pub const Armenian_hi = 16778613;
pub const Armenian_ho = 16778608;
pub const Armenian_hyphen = 16778634;
pub const Armenian_ini = 16778603;
pub const Armenian_je = 16778619;
pub const Armenian_ke = 16778628;
pub const Armenian_ken = 16778607;
pub const Armenian_khe = 16778605;
pub const Armenian_ligature_ew = 16778631;
pub const Armenian_lyun = 16778604;
pub const Armenian_men = 16778612;
pub const Armenian_nu = 16778614;
pub const Armenian_o = 16778629;
pub const Armenian_paruyk = 16778590;
pub const Armenian_pe = 16778618;
pub const Armenian_pyur = 16778627;
pub const Armenian_question = 16778590;
pub const Armenian_ra = 16778620;
pub const Armenian_re = 16778624;
pub const Armenian_se = 16778621;
pub const Armenian_separation_mark = 16778589;
pub const Armenian_sha = 16778615;
pub const Armenian_shesht = 16778587;
pub const Armenian_tche = 16778611;
pub const Armenian_to = 16778601;
pub const Armenian_tsa = 16778606;
pub const Armenian_tso = 16778625;
pub const Armenian_tyun = 16778623;
pub const Armenian_verjaket = 16778633;
pub const Armenian_vev = 16778622;
pub const Armenian_vo = 16778616;
pub const Armenian_vyun = 16778626;
pub const Armenian_yech = 16778597;
pub const Armenian_yentamna = 16778634;
pub const Armenian_za = 16778598;
pub const Armenian_zhe = 16778602;
pub const Atilde = 195;
pub const AudibleBell_Enable = 65146;
pub const B = 66;
pub const Babovedot = 16784898;
pub const BackSpace = 65288;
pub const Begin = 65368;
pub const BounceKeys_Enable = 65140;
pub const Break = 65387;
pub const Byelorussian_SHORTU = 1726;
pub const Byelorussian_shortu = 1710;
pub const C = 67;
pub const Cabovedot = 709;
pub const Cacute = 454;
pub const Cancel = 65385;
pub const Caps_Lock = 65509;
pub const Ccaron = 456;
pub const Ccedilla = 199;
pub const Ccircumflex = 710;
pub const Clear = 65291;
/// International and multi-key character composition.
pub const Codeinput = 65335;
pub const ColonSign = 16785569;
pub const Control_L = 65507;
pub const Control_R = 65508;
pub const CruzeiroSign = 16785570;
pub const Cyrillic_A = 1761;
pub const Cyrillic_BE = 1762;
pub const Cyrillic_CHE = 1790;
pub const Cyrillic_CHE_descender = 16778422;
pub const Cyrillic_CHE_vertstroke = 16778424;
pub const Cyrillic_DE = 1764;
pub const Cyrillic_DZHE = 1727;
pub const Cyrillic_E = 1788;
pub const Cyrillic_EF = 1766;
pub const Cyrillic_EL = 1772;
pub const Cyrillic_EM = 1773;
pub const Cyrillic_EN = 1774;
pub const Cyrillic_EN_descender = 16778402;
pub const Cyrillic_ER = 1778;
pub const Cyrillic_ES = 1779;
pub const Cyrillic_GHE = 1767;
pub const Cyrillic_GHE_bar = 16778386;
pub const Cyrillic_HA = 1768;
pub const Cyrillic_HARDSIGN = 1791;
pub const Cyrillic_HA_descender = 16778418;
pub const Cyrillic_I = 1769;
pub const Cyrillic_IE = 1765;
pub const Cyrillic_IO = 1715;
pub const Cyrillic_I_macron = 16778466;
pub const Cyrillic_JE = 1720;
pub const Cyrillic_KA = 1771;
pub const Cyrillic_KA_descender = 16778394;
pub const Cyrillic_KA_vertstroke = 16778396;
pub const Cyrillic_LJE = 1721;
pub const Cyrillic_NJE = 1722;
pub const Cyrillic_O = 1775;
pub const Cyrillic_O_bar = 16778472;
pub const Cyrillic_PE = 1776;
pub const Cyrillic_SCHWA = 16778456;
pub const Cyrillic_SHA = 1787;
pub const Cyrillic_SHCHA = 1789;
pub const Cyrillic_SHHA = 16778426;
pub const Cyrillic_SHORTI = 1770;
pub const Cyrillic_SOFTSIGN = 1784;
pub const Cyrillic_TE = 1780;
pub const Cyrillic_TSE = 1763;
pub const Cyrillic_U = 1781;
pub const Cyrillic_U_macron = 16778478;
pub const Cyrillic_U_straight = 16778414;
pub const Cyrillic_U_straight_bar = 16778416;
pub const Cyrillic_VE = 1783;
pub const Cyrillic_YA = 1777;
pub const Cyrillic_YERU = 1785;
pub const Cyrillic_YU = 1760;
pub const Cyrillic_ZE = 1786;
pub const Cyrillic_ZHE = 1782;
pub const Cyrillic_ZHE_descender = 16778390;
pub const Cyrillic_a = 1729;
pub const Cyrillic_be = 1730;
pub const Cyrillic_che = 1758;
pub const Cyrillic_che_descender = 16778423;
pub const Cyrillic_che_vertstroke = 16778425;
pub const Cyrillic_de = 1732;
pub const Cyrillic_dzhe = 1711;
pub const Cyrillic_e = 1756;
pub const Cyrillic_ef = 1734;
pub const Cyrillic_el = 1740;
pub const Cyrillic_em = 1741;
pub const Cyrillic_en = 1742;
pub const Cyrillic_en_descender = 16778403;
pub const Cyrillic_er = 1746;
pub const Cyrillic_es = 1747;
pub const Cyrillic_ghe = 1735;
pub const Cyrillic_ghe_bar = 16778387;
pub const Cyrillic_ha = 1736;
pub const Cyrillic_ha_descender = 16778419;
pub const Cyrillic_hardsign = 1759;
pub const Cyrillic_i = 1737;
pub const Cyrillic_i_macron = 16778467;
pub const Cyrillic_ie = 1733;
pub const Cyrillic_io = 1699;
pub const Cyrillic_je = 1704;
pub const Cyrillic_ka = 1739;
pub const Cyrillic_ka_descender = 16778395;
pub const Cyrillic_ka_vertstroke = 16778397;
pub const Cyrillic_lje = 1705;
pub const Cyrillic_nje = 1706;
pub const Cyrillic_o = 1743;
pub const Cyrillic_o_bar = 16778473;
pub const Cyrillic_pe = 1744;
pub const Cyrillic_schwa = 16778457;
pub const Cyrillic_sha = 1755;
pub const Cyrillic_shcha = 1757;
pub const Cyrillic_shha = 16778427;
pub const Cyrillic_shorti = 1738;
pub const Cyrillic_softsign = 1752;
pub const Cyrillic_te = 1748;
pub const Cyrillic_tse = 1731;
pub const Cyrillic_u = 1749;
pub const Cyrillic_u_macron = 16778479;
pub const Cyrillic_u_straight = 16778415;
pub const Cyrillic_u_straight_bar = 16778417;
pub const Cyrillic_ve = 1751;
pub const Cyrillic_ya = 1745;
pub const Cyrillic_yeru = 1753;
pub const Cyrillic_yu = 1728;
pub const Cyrillic_ze = 1754;
pub const Cyrillic_zhe = 1750;
pub const Cyrillic_zhe_descender = 16778391;
pub const D = 68;
pub const Dabovedot = 16784906;
pub const Dcaron = 463;
pub const Delete = 65535;
pub const DongSign = 16785579;
pub const Down = 65364;
pub const Dstroke = 464;
pub const E = 69;
pub const ENG = 957;
pub const ETH = 208;
pub const Eabovedot = 972;
pub const Eacute = 201;
pub const Ebelowdot = 16785080;
pub const Ecaron = 460;
pub const Ecircumflex = 202;
pub const Ecircumflexacute = 16785086;
pub const Ecircumflexbelowdot = 16785094;
pub const Ecircumflexgrave = 16785088;
pub const Ecircumflexhook = 16785090;
pub const Ecircumflextilde = 16785092;
pub const EcuSign = 16785568;
pub const Ediaeresis = 203;
pub const Egrave = 200;
pub const Ehook = 16785082;
pub const Eisu_Shift = 65327;
pub const Eisu_toggle = 65328;
pub const Emacron = 938;
pub const End = 65367;
pub const Eogonek = 458;
pub const Escape = 65307;
pub const Eth = 208;
pub const Etilde = 16785084;
pub const EuroSign = 8364;
pub const Execute = 65378;
pub const F = 70;
pub const F1 = 65470;
pub const F10 = 65479;
pub const F11 = 65480;
pub const F12 = 65481;
pub const F13 = 65482;
pub const F14 = 65483;
pub const F15 = 65484;
pub const F16 = 65485;
pub const F17 = 65486;
pub const F18 = 65487;
pub const F19 = 65488;
pub const F2 = 65471;
pub const F20 = 65489;
pub const F21 = 65490;
pub const F22 = 65491;
pub const F23 = 65492;
pub const F24 = 65493;
pub const F25 = 65494;
pub const F26 = 65495;
pub const F27 = 65496;
pub const F28 = 65497;
pub const F29 = 65498;
pub const F3 = 65472;
pub const F30 = 65499;
pub const F31 = 65500;
pub const F32 = 65501;
pub const F33 = 65502;
pub const F34 = 65503;
pub const F35 = 65504;
pub const F4 = 65473;
pub const F5 = 65474;
pub const F6 = 65475;
pub const F7 = 65476;
pub const F8 = 65477;
pub const F9 = 65478;
pub const FFrancSign = 16785571;
pub const Fabovedot = 16784926;
pub const Farsi_0 = 16778992;
pub const Farsi_1 = 16778993;
pub const Farsi_2 = 16778994;
pub const Farsi_3 = 16778995;
pub const Farsi_4 = 16778996;
pub const Farsi_5 = 16778997;
pub const Farsi_6 = 16778998;
pub const Farsi_7 = 16778999;
pub const Farsi_8 = 16779000;
pub const Farsi_9 = 16779001;
pub const Farsi_yeh = 16778956;
pub const Find = 65384;
pub const First_Virtual_Screen = 65232;
pub const G = 71;
pub const Gabovedot = 725;
pub const Gbreve = 683;
pub const Gcaron = 16777702;
pub const Gcedilla = 939;
pub const Gcircumflex = 728;
pub const Georgian_an = 16781520;
pub const Georgian_ban = 16781521;
pub const Georgian_can = 16781546;
pub const Georgian_char = 16781549;
pub const Georgian_chin = 16781545;
pub const Georgian_cil = 16781548;
pub const Georgian_don = 16781523;
pub const Georgian_en = 16781524;
pub const Georgian_fi = 16781558;
pub const Georgian_gan = 16781522;
pub const Georgian_ghan = 16781542;
pub const Georgian_hae = 16781552;
pub const Georgian_har = 16781556;
pub const Georgian_he = 16781553;
pub const Georgian_hie = 16781554;
pub const Georgian_hoe = 16781557;
pub const Georgian_in = 16781528;
pub const Georgian_jhan = 16781551;
pub const Georgian_jil = 16781547;
pub const Georgian_kan = 16781529;
pub const Georgian_khar = 16781541;
pub const Georgian_las = 16781530;
pub const Georgian_man = 16781531;
pub const Georgian_nar = 16781532;
pub const Georgian_on = 16781533;
pub const Georgian_par = 16781534;
pub const Georgian_phar = 16781540;
pub const Georgian_qar = 16781543;
pub const Georgian_rae = 16781536;
pub const Georgian_san = 16781537;
pub const Georgian_shin = 16781544;
pub const Georgian_tan = 16781527;
pub const Georgian_tar = 16781538;
pub const Georgian_un = 16781539;
pub const Georgian_vin = 16781525;
pub const Georgian_we = 16781555;
pub const Georgian_xan = 16781550;
pub const Georgian_zen = 16781526;
pub const Georgian_zhar = 16781535;
pub const Greek_ALPHA = 1985;
pub const Greek_ALPHAaccent = 1953;
pub const Greek_BETA = 1986;
pub const Greek_CHI = 2007;
pub const Greek_DELTA = 1988;
pub const Greek_EPSILON = 1989;
pub const Greek_EPSILONaccent = 1954;
pub const Greek_ETA = 1991;
pub const Greek_ETAaccent = 1955;
pub const Greek_GAMMA = 1987;
pub const Greek_IOTA = 1993;
pub const Greek_IOTAaccent = 1956;
pub const Greek_IOTAdiaeresis = 1957;
pub const Greek_IOTAdieresis = 1957;
pub const Greek_KAPPA = 1994;
pub const Greek_LAMBDA = 1995;
pub const Greek_LAMDA = 1995;
pub const Greek_MU = 1996;
pub const Greek_NU = 1997;
pub const Greek_OMEGA = 2009;
pub const Greek_OMEGAaccent = 1963;
pub const Greek_OMICRON = 1999;
pub const Greek_OMICRONaccent = 1959;
pub const Greek_PHI = 2006;
pub const Greek_PI = 2000;
pub const Greek_PSI = 2008;
pub const Greek_RHO = 2001;
pub const Greek_SIGMA = 2002;
pub const Greek_TAU = 2004;
pub const Greek_THETA = 1992;
pub const Greek_UPSILON = 2005;
pub const Greek_UPSILONaccent = 1960;
pub const Greek_UPSILONdieresis = 1961;
pub const Greek_XI = 1998;
pub const Greek_ZETA = 1990;
pub const Greek_accentdieresis = 1966;
pub const Greek_alpha = 2017;
pub const Greek_alphaaccent = 1969;
pub const Greek_beta = 2018;
pub const Greek_chi = 2039;
pub const Greek_delta = 2020;
pub const Greek_epsilon = 2021;
pub const Greek_epsilonaccent = 1970;
pub const Greek_eta = 2023;
pub const Greek_etaaccent = 1971;
pub const Greek_finalsmallsigma = 2035;
pub const Greek_gamma = 2019;
pub const Greek_horizbar = 1967;
pub const Greek_iota = 2025;
pub const Greek_iotaaccent = 1972;
pub const Greek_iotaaccentdieresis = 1974;
pub const Greek_iotadieresis = 1973;
pub const Greek_kappa = 2026;
pub const Greek_lambda = 2027;
pub const Greek_lamda = 2027;
pub const Greek_mu = 2028;
pub const Greek_nu = 2029;
pub const Greek_omega = 2041;
pub const Greek_omegaaccent = 1979;
pub const Greek_omicron = 2031;
pub const Greek_omicronaccent = 1975;
pub const Greek_phi = 2038;
pub const Greek_pi = 2032;
pub const Greek_psi = 2040;
pub const Greek_rho = 2033;
pub const Greek_sigma = 2034;
pub const Greek_switch = 65406;
pub const Greek_tau = 2036;
pub const Greek_theta = 2024;
pub const Greek_upsilon = 2037;
pub const Greek_upsilonaccent = 1976;
pub const Greek_upsilonaccentdieresis = 1978;
pub const Greek_upsilondieresis = 1977;
pub const Greek_xi = 2030;
pub const Greek_zeta = 2022;
pub const H = 72;
pub const Hangul = 65329;
pub const Hangul_A = 3775;
pub const Hangul_AE = 3776;
pub const Hangul_AraeA = 3830;
pub const Hangul_AraeAE = 3831;
pub const Hangul_Banja = 65337;
pub const Hangul_Cieuc = 3770;
pub const Hangul_Codeinput = 65335;
pub const Hangul_Dikeud = 3751;
pub const Hangul_E = 3780;
pub const Hangul_EO = 3779;
pub const Hangul_EU = 3793;
pub const Hangul_End = 65331;
pub const Hangul_Hanja = 65332;
pub const Hangul_Hieuh = 3774;
pub const Hangul_I = 3795;
pub const Hangul_Ieung = 3767;
pub const Hangul_J_Cieuc = 3818;
pub const Hangul_J_Dikeud = 3802;
pub const Hangul_J_Hieuh = 3822;
pub const Hangul_J_Ieung = 3816;
pub const Hangul_J_Jieuj = 3817;
pub const Hangul_J_Khieuq = 3819;
pub const Hangul_J_Kiyeog = 3796;
pub const Hangul_J_KiyeogSios = 3798;
pub const Hangul_J_KkogjiDalrinIeung = 3833;
pub const Hangul_J_Mieum = 3811;
pub const Hangul_J_Nieun = 3799;
pub const Hangul_J_NieunHieuh = 3801;
pub const Hangul_J_NieunJieuj = 3800;
pub const Hangul_J_PanSios = 3832;
pub const Hangul_J_Phieuf = 3821;
pub const Hangul_J_Pieub = 3812;
pub const Hangul_J_PieubSios = 3813;
pub const Hangul_J_Rieul = 3803;
pub const Hangul_J_RieulHieuh = 3810;
pub const Hangul_J_RieulKiyeog = 3804;
pub const Hangul_J_RieulMieum = 3805;
pub const Hangul_J_RieulPhieuf = 3809;
pub const Hangul_J_RieulPieub = 3806;
pub const Hangul_J_RieulSios = 3807;
pub const Hangul_J_RieulTieut = 3808;
pub const Hangul_J_Sios = 3814;
pub const Hangul_J_SsangKiyeog = 3797;
pub const Hangul_J_SsangSios = 3815;
pub const Hangul_J_Tieut = 3820;
pub const Hangul_J_YeorinHieuh = 3834;
pub const Hangul_Jamo = 65333;
pub const Hangul_Jeonja = 65336;
pub const Hangul_Jieuj = 3768;
pub const Hangul_Khieuq = 3771;
pub const Hangul_Kiyeog = 3745;
pub const Hangul_KiyeogSios = 3747;
pub const Hangul_KkogjiDalrinIeung = 3827;
pub const Hangul_Mieum = 3761;
pub const Hangul_MultipleCandidate = 65341;
pub const Hangul_Nieun = 3748;
pub const Hangul_NieunHieuh = 3750;
pub const Hangul_NieunJieuj = 3749;
pub const Hangul_O = 3783;
pub const Hangul_OE = 3786;
pub const Hangul_PanSios = 3826;
pub const Hangul_Phieuf = 3773;
pub const Hangul_Pieub = 3762;
pub const Hangul_PieubSios = 3764;
pub const Hangul_PostHanja = 65339;
pub const Hangul_PreHanja = 65338;
pub const Hangul_PreviousCandidate = 65342;
pub const Hangul_Rieul = 3753;
pub const Hangul_RieulHieuh = 3760;
pub const Hangul_RieulKiyeog = 3754;
pub const Hangul_RieulMieum = 3755;
pub const Hangul_RieulPhieuf = 3759;
pub const Hangul_RieulPieub = 3756;
pub const Hangul_RieulSios = 3757;
pub const Hangul_RieulTieut = 3758;
pub const Hangul_RieulYeorinHieuh = 3823;
pub const Hangul_Romaja = 65334;
pub const Hangul_SingleCandidate = 65340;
pub const Hangul_Sios = 3765;
pub const Hangul_Special = 65343;
pub const Hangul_SsangDikeud = 3752;
pub const Hangul_SsangJieuj = 3769;
pub const Hangul_SsangKiyeog = 3746;
pub const Hangul_SsangPieub = 3763;
pub const Hangul_SsangSios = 3766;
pub const Hangul_Start = 65330;
pub const Hangul_SunkyeongeumMieum = 3824;
pub const Hangul_SunkyeongeumPhieuf = 3828;
pub const Hangul_SunkyeongeumPieub = 3825;
pub const Hangul_Tieut = 3772;
pub const Hangul_U = 3788;
pub const Hangul_WA = 3784;
pub const Hangul_WAE = 3785;
pub const Hangul_WE = 3790;
pub const Hangul_WEO = 3789;
pub const Hangul_WI = 3791;
pub const Hangul_YA = 3777;
pub const Hangul_YAE = 3778;
pub const Hangul_YE = 3782;
pub const Hangul_YEO = 3781;
pub const Hangul_YI = 3794;
pub const Hangul_YO = 3787;
pub const Hangul_YU = 3792;
pub const Hangul_YeorinHieuh = 3829;
pub const Hangul_switch = 65406;
pub const Hankaku = 65321;
pub const Hcircumflex = 678;
pub const Hebrew_switch = 65406;
pub const Help = 65386;
/// Japanese keyboard support.
pub const Henkan = 65315;
/// Japanese keyboard support.
pub const Henkan_Mode = 65315;
/// Japanese keyboard support.
pub const Hiragana = 65317;
/// Japanese keyboard support.
pub const Hiragana_Katakana = 65319;
pub const Home = 65360;
pub const Hstroke = 673;
pub const Hyper_L = 65517;
pub const Hyper_R = 65518;
pub const I = 73;
/// D-Bus interface for IBus config.
pub const INTERFACE_CONFIG = "org.freedesktop.IBus.Config";
/// D-Bus interface for IBus engine.
pub const INTERFACE_ENGINE = "org.freedesktop.IBus.Engine";
/// D-Bus interface for IBus factory.
pub const INTERFACE_FACTORY = "org.freedesktop.IBus.Factory";
/// D-Bus interface for IBus.
pub const INTERFACE_IBUS = "org.freedesktop.IBus";
/// D-Bus interface for IBus input context.
pub const INTERFACE_INPUT_CONTEXT = "org.freedesktop.IBus.InputContext";
/// D-Bus interface for IBus notifications.
pub const INTERFACE_NOTIFICATIONS = "org.freedesktop.IBus.Notifications";
/// D-Bus interface for IBus panel.
pub const INTERFACE_PANEL = "org.freedesktop.IBus.Panel";
/// D-Bus interface for IBus portal.
pub const INTERFACE_PORTAL = "org.freedesktop.IBus.Portal";
pub const ISO_Center_Object = 65075;
pub const ISO_Continuous_Underline = 65072;
pub const ISO_Discontinuous_Underline = 65073;
pub const ISO_Emphasize = 65074;
pub const ISO_Enter = 65076;
pub const ISO_Fast_Cursor_Down = 65071;
pub const ISO_Fast_Cursor_Left = 65068;
pub const ISO_Fast_Cursor_Right = 65069;
pub const ISO_Fast_Cursor_Up = 65070;
pub const ISO_First_Group = 65036;
pub const ISO_First_Group_Lock = 65037;
pub const ISO_Group_Latch = 65030;
pub const ISO_Group_Lock = 65031;
pub const ISO_Group_Shift = 65406;
pub const ISO_Last_Group = 65038;
pub const ISO_Last_Group_Lock = 65039;
pub const ISO_Left_Tab = 65056;
pub const ISO_Level2_Latch = 65026;
pub const ISO_Level3_Latch = 65028;
pub const ISO_Level3_Lock = 65029;
pub const ISO_Level3_Shift = 65027;
pub const ISO_Level5_Latch = 65042;
pub const ISO_Level5_Lock = 65043;
pub const ISO_Level5_Shift = 65041;
pub const ISO_Lock = 65025;
pub const ISO_Move_Line_Down = 65058;
pub const ISO_Move_Line_Up = 65057;
pub const ISO_Next_Group = 65032;
pub const ISO_Next_Group_Lock = 65033;
pub const ISO_Partial_Line_Down = 65060;
pub const ISO_Partial_Line_Up = 65059;
pub const ISO_Partial_Space_Left = 65061;
pub const ISO_Partial_Space_Right = 65062;
pub const ISO_Prev_Group = 65034;
pub const ISO_Prev_Group_Lock = 65035;
pub const ISO_Release_Both_Margins = 65067;
pub const ISO_Release_Margin_Left = 65065;
pub const ISO_Release_Margin_Right = 65066;
pub const ISO_Set_Margin_Left = 65063;
pub const ISO_Set_Margin_Right = 65064;
pub const Iabovedot = 681;
pub const Iacute = 205;
pub const Ibelowdot = 16785098;
pub const Ibreve = 16777516;
pub const Icircumflex = 206;
pub const Idiaeresis = 207;
pub const Igrave = 204;
pub const Ihook = 16785096;
pub const Imacron = 975;
pub const Insert = 65379;
pub const Iogonek = 967;
pub const Itilde = 933;
pub const J = 74;
pub const Jcircumflex = 684;
pub const K = 75;
pub const KEY_0 = 48;
pub const KEY_1 = 49;
pub const KEY_2 = 50;
pub const KEY_3 = 51;
pub const KEY_3270_AltCursor = 64784;
pub const KEY_3270_Attn = 64782;
pub const KEY_3270_BackTab = 64773;
pub const KEY_3270_ChangeScreen = 64793;
pub const KEY_3270_Copy = 64789;
pub const KEY_3270_CursorBlink = 64783;
pub const KEY_3270_CursorSelect = 64796;
pub const KEY_3270_DeleteWord = 64794;
pub const KEY_3270_Duplicate = 64769;
pub const KEY_3270_Enter = 64798;
pub const KEY_3270_EraseEOF = 64774;
pub const KEY_3270_EraseInput = 64775;
pub const KEY_3270_ExSelect = 64795;
pub const KEY_3270_FieldMark = 64770;
pub const KEY_3270_Ident = 64787;
pub const KEY_3270_Jump = 64786;
pub const KEY_3270_KeyClick = 64785;
pub const KEY_3270_Left2 = 64772;
pub const KEY_3270_PA1 = 64778;
pub const KEY_3270_PA2 = 64779;
pub const KEY_3270_PA3 = 64780;
pub const KEY_3270_Play = 64790;
pub const KEY_3270_PrintScreen = 64797;
pub const KEY_3270_Quit = 64777;
pub const KEY_3270_Record = 64792;
pub const KEY_3270_Reset = 64776;
pub const KEY_3270_Right2 = 64771;
pub const KEY_3270_Rule = 64788;
pub const KEY_3270_Setup = 64791;
pub const KEY_3270_Test = 64781;
pub const KEY_4 = 52;
pub const KEY_5 = 53;
pub const KEY_6 = 54;
pub const KEY_7 = 55;
pub const KEY_8 = 56;
pub const KEY_9 = 57;
pub const KEY_A = 65;
pub const KEY_AE = 198;
pub const KEY_Aacute = 193;
pub const KEY_Abelowdot = 16785056;
pub const KEY_Abreve = 451;
pub const KEY_Abreveacute = 16785070;
pub const KEY_Abrevebelowdot = 16785078;
pub const KEY_Abrevegrave = 16785072;
pub const KEY_Abrevehook = 16785074;
pub const KEY_Abrevetilde = 16785076;
pub const KEY_AccessX_Enable = 65136;
pub const KEY_AccessX_Feedback_Enable = 65137;
pub const KEY_Acircumflex = 194;
pub const KEY_Acircumflexacute = 16785060;
pub const KEY_Acircumflexbelowdot = 16785068;
pub const KEY_Acircumflexgrave = 16785062;
pub const KEY_Acircumflexhook = 16785064;
pub const KEY_Acircumflextilde = 16785066;
pub const KEY_AddFavorite = 269025081;
pub const KEY_Adiaeresis = 196;
pub const KEY_Agrave = 192;
pub const KEY_Ahook = 16785058;
pub const KEY_Alt_L = 65513;
pub const KEY_Alt_R = 65514;
pub const KEY_Amacron = 960;
pub const KEY_Aogonek = 417;
pub const KEY_ApplicationLeft = 269025104;
pub const KEY_ApplicationRight = 269025105;
pub const KEY_Arabic_0 = 16778848;
pub const KEY_Arabic_1 = 16778849;
pub const KEY_Arabic_2 = 16778850;
pub const KEY_Arabic_3 = 16778851;
pub const KEY_Arabic_4 = 16778852;
pub const KEY_Arabic_5 = 16778853;
pub const KEY_Arabic_6 = 16778854;
pub const KEY_Arabic_7 = 16778855;
pub const KEY_Arabic_8 = 16778856;
pub const KEY_Arabic_9 = 16778857;
pub const KEY_Arabic_ain = 1497;
pub const KEY_Arabic_alef = 1479;
pub const KEY_Arabic_alefmaksura = 1513;
pub const KEY_Arabic_beh = 1480;
pub const KEY_Arabic_comma = 1452;
pub const KEY_Arabic_dad = 1494;
pub const KEY_Arabic_dal = 1487;
pub const KEY_Arabic_damma = 1519;
pub const KEY_Arabic_dammatan = 1516;
pub const KEY_Arabic_ddal = 16778888;
pub const KEY_Arabic_farsi_yeh = 16778956;
pub const KEY_Arabic_fatha = 1518;
pub const KEY_Arabic_fathatan = 1515;
pub const KEY_Arabic_feh = 1505;
pub const KEY_Arabic_fullstop = 16778964;
pub const KEY_Arabic_gaf = 16778927;
pub const KEY_Arabic_ghain = 1498;
pub const KEY_Arabic_ha = 1511;
pub const KEY_Arabic_hah = 1485;
pub const KEY_Arabic_hamza = 1473;
pub const KEY_Arabic_hamza_above = 16778836;
pub const KEY_Arabic_hamza_below = 16778837;
pub const KEY_Arabic_hamzaonalef = 1475;
pub const KEY_Arabic_hamzaonwaw = 1476;
pub const KEY_Arabic_hamzaonyeh = 1478;
pub const KEY_Arabic_hamzaunderalef = 1477;
pub const KEY_Arabic_heh = 1511;
pub const KEY_Arabic_heh_doachashmee = 16778942;
pub const KEY_Arabic_heh_goal = 16778945;
pub const KEY_Arabic_jeem = 1484;
pub const KEY_Arabic_jeh = 16778904;
pub const KEY_Arabic_kaf = 1507;
pub const KEY_Arabic_kasra = 1520;
pub const KEY_Arabic_kasratan = 1517;
pub const KEY_Arabic_keheh = 16778921;
pub const KEY_Arabic_khah = 1486;
pub const KEY_Arabic_lam = 1508;
pub const KEY_Arabic_madda_above = 16778835;
pub const KEY_Arabic_maddaonalef = 1474;
pub const KEY_Arabic_meem = 1509;
pub const KEY_Arabic_noon = 1510;
pub const KEY_Arabic_noon_ghunna = 16778938;
pub const KEY_Arabic_peh = 16778878;
pub const KEY_Arabic_percent = 16778858;
pub const KEY_Arabic_qaf = 1506;
pub const KEY_Arabic_question_mark = 1471;
pub const KEY_Arabic_ra = 1489;
pub const KEY_Arabic_rreh = 16778897;
pub const KEY_Arabic_sad = 1493;
pub const KEY_Arabic_seen = 1491;
pub const KEY_Arabic_semicolon = 1467;
pub const KEY_Arabic_shadda = 1521;
pub const KEY_Arabic_sheen = 1492;
pub const KEY_Arabic_sukun = 1522;
pub const KEY_Arabic_superscript_alef = 16778864;
pub const KEY_Arabic_switch = 65406;
pub const KEY_Arabic_tah = 1495;
pub const KEY_Arabic_tatweel = 1504;
pub const KEY_Arabic_tcheh = 16778886;
pub const KEY_Arabic_teh = 1482;
pub const KEY_Arabic_tehmarbuta = 1481;
pub const KEY_Arabic_thal = 1488;
pub const KEY_Arabic_theh = 1483;
pub const KEY_Arabic_tteh = 16778873;
pub const KEY_Arabic_veh = 16778916;
pub const KEY_Arabic_waw = 1512;
pub const KEY_Arabic_yeh = 1514;
pub const KEY_Arabic_yeh_baree = 16778962;
pub const KEY_Arabic_zah = 1496;
pub const KEY_Arabic_zain = 1490;
pub const KEY_Aring = 197;
pub const KEY_Armenian_AT = 16778552;
pub const KEY_Armenian_AYB = 16778545;
pub const KEY_Armenian_BEN = 16778546;
pub const KEY_Armenian_CHA = 16778569;
pub const KEY_Armenian_DA = 16778548;
pub const KEY_Armenian_DZA = 16778561;
pub const KEY_Armenian_E = 16778551;
pub const KEY_Armenian_FE = 16778582;
pub const KEY_Armenian_GHAT = 16778562;
pub const KEY_Armenian_GIM = 16778547;
pub const KEY_Armenian_HI = 16778565;
pub const KEY_Armenian_HO = 16778560;
pub const KEY_Armenian_INI = 16778555;
pub const KEY_Armenian_JE = 16778571;
pub const KEY_Armenian_KE = 16778580;
pub const KEY_Armenian_KEN = 16778559;
pub const KEY_Armenian_KHE = 16778557;
pub const KEY_Armenian_LYUN = 16778556;
pub const KEY_Armenian_MEN = 16778564;
pub const KEY_Armenian_NU = 16778566;
pub const KEY_Armenian_O = 16778581;
pub const KEY_Armenian_PE = 16778570;
pub const KEY_Armenian_PYUR = 16778579;
pub const KEY_Armenian_RA = 16778572;
pub const KEY_Armenian_RE = 16778576;
pub const KEY_Armenian_SE = 16778573;
pub const KEY_Armenian_SHA = 16778567;
pub const KEY_Armenian_TCHE = 16778563;
pub const KEY_Armenian_TO = 16778553;
pub const KEY_Armenian_TSA = 16778558;
pub const KEY_Armenian_TSO = 16778577;
pub const KEY_Armenian_TYUN = 16778575;
pub const KEY_Armenian_VEV = 16778574;
pub const KEY_Armenian_VO = 16778568;
pub const KEY_Armenian_VYUN = 16778578;
pub const KEY_Armenian_YECH = 16778549;
pub const KEY_Armenian_ZA = 16778550;
pub const KEY_Armenian_ZHE = 16778554;
pub const KEY_Armenian_accent = 16778587;
pub const KEY_Armenian_amanak = 16778588;
pub const KEY_Armenian_apostrophe = 16778586;
pub const KEY_Armenian_at = 16778600;
pub const KEY_Armenian_ayb = 16778593;
pub const KEY_Armenian_ben = 16778594;
pub const KEY_Armenian_but = 16778589;
pub const KEY_Armenian_cha = 16778617;
pub const KEY_Armenian_da = 16778596;
pub const KEY_Armenian_dza = 16778609;
pub const KEY_Armenian_e = 16778599;
pub const KEY_Armenian_exclam = 16778588;
pub const KEY_Armenian_fe = 16778630;
pub const KEY_Armenian_full_stop = 16778633;
pub const KEY_Armenian_ghat = 16778610;
pub const KEY_Armenian_gim = 16778595;
pub const KEY_Armenian_hi = 16778613;
pub const KEY_Armenian_ho = 16778608;
pub const KEY_Armenian_hyphen = 16778634;
pub const KEY_Armenian_ini = 16778603;
pub const KEY_Armenian_je = 16778619;
pub const KEY_Armenian_ke = 16778628;
pub const KEY_Armenian_ken = 16778607;
pub const KEY_Armenian_khe = 16778605;
pub const KEY_Armenian_ligature_ew = 16778631;
pub const KEY_Armenian_lyun = 16778604;
pub const KEY_Armenian_men = 16778612;
pub const KEY_Armenian_nu = 16778614;
pub const KEY_Armenian_o = 16778629;
pub const KEY_Armenian_paruyk = 16778590;
pub const KEY_Armenian_pe = 16778618;
pub const KEY_Armenian_pyur = 16778627;
pub const KEY_Armenian_question = 16778590;
pub const KEY_Armenian_ra = 16778620;
pub const KEY_Armenian_re = 16778624;
pub const KEY_Armenian_se = 16778621;
pub const KEY_Armenian_separation_mark = 16778589;
pub const KEY_Armenian_sha = 16778615;
pub const KEY_Armenian_shesht = 16778587;
pub const KEY_Armenian_tche = 16778611;
pub const KEY_Armenian_to = 16778601;
pub const KEY_Armenian_tsa = 16778606;
pub const KEY_Armenian_tso = 16778625;
pub const KEY_Armenian_tyun = 16778623;
pub const KEY_Armenian_verjaket = 16778633;
pub const KEY_Armenian_vev = 16778622;
pub const KEY_Armenian_vo = 16778616;
pub const KEY_Armenian_vyun = 16778626;
pub const KEY_Armenian_yech = 16778597;
pub const KEY_Armenian_yentamna = 16778634;
pub const KEY_Armenian_za = 16778598;
pub const KEY_Armenian_zhe = 16778602;
pub const KEY_Atilde = 195;
pub const KEY_AudibleBell_Enable = 65146;
pub const KEY_AudioCycleTrack = 269025179;
pub const KEY_AudioForward = 269025175;
pub const KEY_AudioLowerVolume = 269025041;
pub const KEY_AudioMedia = 269025074;
pub const KEY_AudioMicMute = 269025202;
pub const KEY_AudioMute = 269025042;
pub const KEY_AudioNext = 269025047;
pub const KEY_AudioPause = 269025073;
pub const KEY_AudioPlay = 269025044;
pub const KEY_AudioPreset = 269025206;
pub const KEY_AudioPrev = 269025046;
pub const KEY_AudioRaiseVolume = 269025043;
pub const KEY_AudioRandomPlay = 269025177;
pub const KEY_AudioRecord = 269025052;
pub const KEY_AudioRepeat = 269025176;
pub const KEY_AudioRewind = 269025086;
pub const KEY_AudioStop = 269025045;
pub const KEY_Away = 269025165;
pub const KEY_B = 66;
pub const KEY_Babovedot = 16784898;
pub const KEY_Back = 269025062;
pub const KEY_BackForward = 269025087;
pub const KEY_BackSpace = 65288;
pub const KEY_Battery = 269025171;
pub const KEY_Begin = 65368;
pub const KEY_Blue = 269025190;
pub const KEY_Bluetooth = 269025172;
pub const KEY_Book = 269025106;
pub const KEY_BounceKeys_Enable = 65140;
pub const KEY_Break = 65387;
pub const KEY_BrightnessAdjust = 269025083;
pub const KEY_Byelorussian_SHORTU = 1726;
pub const KEY_Byelorussian_shortu = 1710;
pub const KEY_C = 67;
pub const KEY_CD = 269025107;
pub const KEY_CH = 65186;
pub const KEY_C_H = 65189;
pub const KEY_C_h = 65188;
pub const KEY_Cabovedot = 709;
pub const KEY_Cacute = 454;
pub const KEY_Calculator = 269025053;
pub const KEY_Calendar = 269025056;
pub const KEY_Cancel = 65385;
pub const KEY_Caps_Lock = 65509;
pub const KEY_Ccaron = 456;
pub const KEY_Ccedilla = 199;
pub const KEY_Ccircumflex = 710;
pub const KEY_Ch = 65185;
pub const KEY_Clear = 65291;
pub const KEY_ClearGrab = 269024801;
pub const KEY_Close = 269025110;
pub const KEY_Codeinput = 65335;
pub const KEY_ColonSign = 16785569;
pub const KEY_Community = 269025085;
pub const KEY_ContrastAdjust = 269025058;
pub const KEY_Control_L = 65507;
pub const KEY_Control_R = 65508;
pub const KEY_Copy = 269025111;
pub const KEY_CruzeiroSign = 16785570;
pub const KEY_Cut = 269025112;
pub const KEY_CycleAngle = 269025180;
pub const KEY_Cyrillic_A = 1761;
pub const KEY_Cyrillic_BE = 1762;
pub const KEY_Cyrillic_CHE = 1790;
pub const KEY_Cyrillic_CHE_descender = 16778422;
pub const KEY_Cyrillic_CHE_vertstroke = 16778424;
pub const KEY_Cyrillic_DE = 1764;
pub const KEY_Cyrillic_DZHE = 1727;
pub const KEY_Cyrillic_E = 1788;
pub const KEY_Cyrillic_EF = 1766;
pub const KEY_Cyrillic_EL = 1772;
pub const KEY_Cyrillic_EM = 1773;
pub const KEY_Cyrillic_EN = 1774;
pub const KEY_Cyrillic_EN_descender = 16778402;
pub const KEY_Cyrillic_ER = 1778;
pub const KEY_Cyrillic_ES = 1779;
pub const KEY_Cyrillic_GHE = 1767;
pub const KEY_Cyrillic_GHE_bar = 16778386;
pub const KEY_Cyrillic_HA = 1768;
pub const KEY_Cyrillic_HARDSIGN = 1791;
pub const KEY_Cyrillic_HA_descender = 16778418;
pub const KEY_Cyrillic_I = 1769;
pub const KEY_Cyrillic_IE = 1765;
pub const KEY_Cyrillic_IO = 1715;
pub const KEY_Cyrillic_I_macron = 16778466;
pub const KEY_Cyrillic_JE = 1720;
pub const KEY_Cyrillic_KA = 1771;
pub const KEY_Cyrillic_KA_descender = 16778394;
pub const KEY_Cyrillic_KA_vertstroke = 16778396;
pub const KEY_Cyrillic_LJE = 1721;
pub const KEY_Cyrillic_NJE = 1722;
pub const KEY_Cyrillic_O = 1775;
pub const KEY_Cyrillic_O_bar = 16778472;
pub const KEY_Cyrillic_PE = 1776;
pub const KEY_Cyrillic_SCHWA = 16778456;
pub const KEY_Cyrillic_SHA = 1787;
pub const KEY_Cyrillic_SHCHA = 1789;
pub const KEY_Cyrillic_SHHA = 16778426;
pub const KEY_Cyrillic_SHORTI = 1770;
pub const KEY_Cyrillic_SOFTSIGN = 1784;
pub const KEY_Cyrillic_TE = 1780;
pub const KEY_Cyrillic_TSE = 1763;
pub const KEY_Cyrillic_U = 1781;
pub const KEY_Cyrillic_U_macron = 16778478;
pub const KEY_Cyrillic_U_straight = 16778414;
pub const KEY_Cyrillic_U_straight_bar = 16778416;
pub const KEY_Cyrillic_VE = 1783;
pub const KEY_Cyrillic_YA = 1777;
pub const KEY_Cyrillic_YERU = 1785;
pub const KEY_Cyrillic_YU = 1760;
pub const KEY_Cyrillic_ZE = 1786;
pub const KEY_Cyrillic_ZHE = 1782;
pub const KEY_Cyrillic_ZHE_descender = 16778390;
pub const KEY_Cyrillic_a = 1729;
pub const KEY_Cyrillic_be = 1730;
pub const KEY_Cyrillic_che = 1758;
pub const KEY_Cyrillic_che_descender = 16778423;
pub const KEY_Cyrillic_che_vertstroke = 16778425;
pub const KEY_Cyrillic_de = 1732;
pub const KEY_Cyrillic_dzhe = 1711;
pub const KEY_Cyrillic_e = 1756;
pub const KEY_Cyrillic_ef = 1734;
pub const KEY_Cyrillic_el = 1740;
pub const KEY_Cyrillic_em = 1741;
pub const KEY_Cyrillic_en = 1742;
pub const KEY_Cyrillic_en_descender = 16778403;
pub const KEY_Cyrillic_er = 1746;
pub const KEY_Cyrillic_es = 1747;
pub const KEY_Cyrillic_ghe = 1735;
pub const KEY_Cyrillic_ghe_bar = 16778387;
pub const KEY_Cyrillic_ha = 1736;
pub const KEY_Cyrillic_ha_descender = 16778419;
pub const KEY_Cyrillic_hardsign = 1759;
pub const KEY_Cyrillic_i = 1737;
pub const KEY_Cyrillic_i_macron = 16778467;
pub const KEY_Cyrillic_ie = 1733;
pub const KEY_Cyrillic_io = 1699;
pub const KEY_Cyrillic_je = 1704;
pub const KEY_Cyrillic_ka = 1739;
pub const KEY_Cyrillic_ka_descender = 16778395;
pub const KEY_Cyrillic_ka_vertstroke = 16778397;
pub const KEY_Cyrillic_lje = 1705;
pub const KEY_Cyrillic_nje = 1706;
pub const KEY_Cyrillic_o = 1743;
pub const KEY_Cyrillic_o_bar = 16778473;
pub const KEY_Cyrillic_pe = 1744;
pub const KEY_Cyrillic_schwa = 16778457;
pub const KEY_Cyrillic_sha = 1755;
pub const KEY_Cyrillic_shcha = 1757;
pub const KEY_Cyrillic_shha = 16778427;
pub const KEY_Cyrillic_shorti = 1738;
pub const KEY_Cyrillic_softsign = 1752;
pub const KEY_Cyrillic_te = 1748;
pub const KEY_Cyrillic_tse = 1731;
pub const KEY_Cyrillic_u = 1749;
pub const KEY_Cyrillic_u_macron = 16778479;
pub const KEY_Cyrillic_u_straight = 16778415;
pub const KEY_Cyrillic_u_straight_bar = 16778417;
pub const KEY_Cyrillic_ve = 1751;
pub const KEY_Cyrillic_ya = 1745;
pub const KEY_Cyrillic_yeru = 1753;
pub const KEY_Cyrillic_yu = 1728;
pub const KEY_Cyrillic_ze = 1754;
pub const KEY_Cyrillic_zhe = 1750;
pub const KEY_Cyrillic_zhe_descender = 16778391;
pub const KEY_D = 68;
pub const KEY_DOS = 269025114;
pub const KEY_Dabovedot = 16784906;
pub const KEY_Dcaron = 463;
pub const KEY_Delete = 65535;
pub const KEY_Display = 269025113;
pub const KEY_Documents = 269025115;
pub const KEY_DongSign = 16785579;
pub const KEY_Down = 65364;
pub const KEY_Dstroke = 464;
pub const KEY_E = 69;
pub const KEY_ENG = 957;
pub const KEY_ETH = 208;
pub const KEY_EZH = 16777655;
pub const KEY_Eabovedot = 972;
pub const KEY_Eacute = 201;
pub const KEY_Ebelowdot = 16785080;
pub const KEY_Ecaron = 460;
pub const KEY_Ecircumflex = 202;
pub const KEY_Ecircumflexacute = 16785086;
pub const KEY_Ecircumflexbelowdot = 16785094;
pub const KEY_Ecircumflexgrave = 16785088;
pub const KEY_Ecircumflexhook = 16785090;
pub const KEY_Ecircumflextilde = 16785092;
pub const KEY_EcuSign = 16785568;
pub const KEY_Ediaeresis = 203;
pub const KEY_Egrave = 200;
pub const KEY_Ehook = 16785082;
pub const KEY_Eisu_Shift = 65327;
pub const KEY_Eisu_toggle = 65328;
pub const KEY_Eject = 269025068;
pub const KEY_Emacron = 938;
pub const KEY_End = 65367;
pub const KEY_Eogonek = 458;
pub const KEY_Escape = 65307;
pub const KEY_Eth = 208;
pub const KEY_Etilde = 16785084;
pub const KEY_EuroSign = 8364;
pub const KEY_Excel = 269025116;
pub const KEY_Execute = 65378;
pub const KEY_Explorer = 269025117;
pub const KEY_F = 70;
pub const KEY_F1 = 65470;
pub const KEY_F10 = 65479;
pub const KEY_F11 = 65480;
pub const KEY_F12 = 65481;
pub const KEY_F13 = 65482;
pub const KEY_F14 = 65483;
pub const KEY_F15 = 65484;
pub const KEY_F16 = 65485;
pub const KEY_F17 = 65486;
pub const KEY_F18 = 65487;
pub const KEY_F19 = 65488;
pub const KEY_F2 = 65471;
pub const KEY_F20 = 65489;
pub const KEY_F21 = 65490;
pub const KEY_F22 = 65491;
pub const KEY_F23 = 65492;
pub const KEY_F24 = 65493;
pub const KEY_F25 = 65494;
pub const KEY_F26 = 65495;
pub const KEY_F27 = 65496;
pub const KEY_F28 = 65497;
pub const KEY_F29 = 65498;
pub const KEY_F3 = 65472;
pub const KEY_F30 = 65499;
pub const KEY_F31 = 65500;
pub const KEY_F32 = 65501;
pub const KEY_F33 = 65502;
pub const KEY_F34 = 65503;
pub const KEY_F35 = 65504;
pub const KEY_F4 = 65473;
pub const KEY_F5 = 65474;
pub const KEY_F6 = 65475;
pub const KEY_F7 = 65476;
pub const KEY_F8 = 65477;
pub const KEY_F9 = 65478;
pub const KEY_FFrancSign = 16785571;
pub const KEY_Fabovedot = 16784926;
pub const KEY_Farsi_0 = 16778992;
pub const KEY_Farsi_1 = 16778993;
pub const KEY_Farsi_2 = 16778994;
pub const KEY_Farsi_3 = 16778995;
pub const KEY_Farsi_4 = 16778996;
pub const KEY_Farsi_5 = 16778997;
pub const KEY_Farsi_6 = 16778998;
pub const KEY_Farsi_7 = 16778999;
pub const KEY_Farsi_8 = 16779000;
pub const KEY_Farsi_9 = 16779001;
pub const KEY_Farsi_yeh = 16778956;
pub const KEY_Favorites = 269025072;
pub const KEY_Finance = 269025084;
pub const KEY_Find = 65384;
pub const KEY_First_Virtual_Screen = 65232;
pub const KEY_Forward = 269025063;
pub const KEY_FrameBack = 269025181;
pub const KEY_FrameForward = 269025182;
pub const KEY_G = 71;
pub const KEY_Gabovedot = 725;
pub const KEY_Game = 269025118;
pub const KEY_Gbreve = 683;
pub const KEY_Gcaron = 16777702;
pub const KEY_Gcedilla = 939;
pub const KEY_Gcircumflex = 728;
pub const KEY_Georgian_an = 16781520;
pub const KEY_Georgian_ban = 16781521;
pub const KEY_Georgian_can = 16781546;
pub const KEY_Georgian_char = 16781549;
pub const KEY_Georgian_chin = 16781545;
pub const KEY_Georgian_cil = 16781548;
pub const KEY_Georgian_don = 16781523;
pub const KEY_Georgian_en = 16781524;
pub const KEY_Georgian_fi = 16781558;
pub const KEY_Georgian_gan = 16781522;
pub const KEY_Georgian_ghan = 16781542;
pub const KEY_Georgian_hae = 16781552;
pub const KEY_Georgian_har = 16781556;
pub const KEY_Georgian_he = 16781553;
pub const KEY_Georgian_hie = 16781554;
pub const KEY_Georgian_hoe = 16781557;
pub const KEY_Georgian_in = 16781528;
pub const KEY_Georgian_jhan = 16781551;
pub const KEY_Georgian_jil = 16781547;
pub const KEY_Georgian_kan = 16781529;
pub const KEY_Georgian_khar = 16781541;
pub const KEY_Georgian_las = 16781530;
pub const KEY_Georgian_man = 16781531;
pub const KEY_Georgian_nar = 16781532;
pub const KEY_Georgian_on = 16781533;
pub const KEY_Georgian_par = 16781534;
pub const KEY_Georgian_phar = 16781540;
pub const KEY_Georgian_qar = 16781543;
pub const KEY_Georgian_rae = 16781536;
pub const KEY_Georgian_san = 16781537;
pub const KEY_Georgian_shin = 16781544;
pub const KEY_Georgian_tan = 16781527;
pub const KEY_Georgian_tar = 16781538;
pub const KEY_Georgian_un = 16781539;
pub const KEY_Georgian_vin = 16781525;
pub const KEY_Georgian_we = 16781555;
pub const KEY_Georgian_xan = 16781550;
pub const KEY_Georgian_zen = 16781526;
pub const KEY_Georgian_zhar = 16781535;
pub const KEY_Go = 269025119;
pub const KEY_Greek_ALPHA = 1985;
pub const KEY_Greek_ALPHAaccent = 1953;
pub const KEY_Greek_BETA = 1986;
pub const KEY_Greek_CHI = 2007;
pub const KEY_Greek_DELTA = 1988;
pub const KEY_Greek_EPSILON = 1989;
pub const KEY_Greek_EPSILONaccent = 1954;
pub const KEY_Greek_ETA = 1991;
pub const KEY_Greek_ETAaccent = 1955;
pub const KEY_Greek_GAMMA = 1987;
pub const KEY_Greek_IOTA = 1993;
pub const KEY_Greek_IOTAaccent = 1956;
pub const KEY_Greek_IOTAdiaeresis = 1957;
pub const KEY_Greek_IOTAdieresis = 1957;
pub const KEY_Greek_KAPPA = 1994;
pub const KEY_Greek_LAMBDA = 1995;
pub const KEY_Greek_LAMDA = 1995;
pub const KEY_Greek_MU = 1996;
pub const KEY_Greek_NU = 1997;
pub const KEY_Greek_OMEGA = 2009;
pub const KEY_Greek_OMEGAaccent = 1963;
pub const KEY_Greek_OMICRON = 1999;
pub const KEY_Greek_OMICRONaccent = 1959;
pub const KEY_Greek_PHI = 2006;
pub const KEY_Greek_PI = 2000;
pub const KEY_Greek_PSI = 2008;
pub const KEY_Greek_RHO = 2001;
pub const KEY_Greek_SIGMA = 2002;
pub const KEY_Greek_TAU = 2004;
pub const KEY_Greek_THETA = 1992;
pub const KEY_Greek_UPSILON = 2005;
pub const KEY_Greek_UPSILONaccent = 1960;
pub const KEY_Greek_UPSILONdieresis = 1961;
pub const KEY_Greek_XI = 1998;
pub const KEY_Greek_ZETA = 1990;
pub const KEY_Greek_accentdieresis = 1966;
pub const KEY_Greek_alpha = 2017;
pub const KEY_Greek_alphaaccent = 1969;
pub const KEY_Greek_beta = 2018;
pub const KEY_Greek_chi = 2039;
pub const KEY_Greek_delta = 2020;
pub const KEY_Greek_epsilon = 2021;
pub const KEY_Greek_epsilonaccent = 1970;
pub const KEY_Greek_eta = 2023;
pub const KEY_Greek_etaaccent = 1971;
pub const KEY_Greek_finalsmallsigma = 2035;
pub const KEY_Greek_gamma = 2019;
pub const KEY_Greek_horizbar = 1967;
pub const KEY_Greek_iota = 2025;
pub const KEY_Greek_iotaaccent = 1972;
pub const KEY_Greek_iotaaccentdieresis = 1974;
pub const KEY_Greek_iotadieresis = 1973;
pub const KEY_Greek_kappa = 2026;
pub const KEY_Greek_lambda = 2027;
pub const KEY_Greek_lamda = 2027;
pub const KEY_Greek_mu = 2028;
pub const KEY_Greek_nu = 2029;
pub const KEY_Greek_omega = 2041;
pub const KEY_Greek_omegaaccent = 1979;
pub const KEY_Greek_omicron = 2031;
pub const KEY_Greek_omicronaccent = 1975;
pub const KEY_Greek_phi = 2038;
pub const KEY_Greek_pi = 2032;
pub const KEY_Greek_psi = 2040;
pub const KEY_Greek_rho = 2033;
pub const KEY_Greek_sigma = 2034;
pub const KEY_Greek_switch = 65406;
pub const KEY_Greek_tau = 2036;
pub const KEY_Greek_theta = 2024;
pub const KEY_Greek_upsilon = 2037;
pub const KEY_Greek_upsilonaccent = 1976;
pub const KEY_Greek_upsilonaccentdieresis = 1978;
pub const KEY_Greek_upsilondieresis = 1977;
pub const KEY_Greek_xi = 2030;
pub const KEY_Greek_zeta = 2022;
pub const KEY_Green = 269025188;
pub const KEY_H = 72;
pub const KEY_Hangul = 65329;
pub const KEY_Hangul_A = 3775;
pub const KEY_Hangul_AE = 3776;
pub const KEY_Hangul_AraeA = 3830;
pub const KEY_Hangul_AraeAE = 3831;
pub const KEY_Hangul_Banja = 65337;
pub const KEY_Hangul_Cieuc = 3770;
pub const KEY_Hangul_Codeinput = 65335;
pub const KEY_Hangul_Dikeud = 3751;
pub const KEY_Hangul_E = 3780;
pub const KEY_Hangul_EO = 3779;
pub const KEY_Hangul_EU = 3793;
pub const KEY_Hangul_End = 65331;
pub const KEY_Hangul_Hanja = 65332;
pub const KEY_Hangul_Hieuh = 3774;
pub const KEY_Hangul_I = 3795;
pub const KEY_Hangul_Ieung = 3767;
pub const KEY_Hangul_J_Cieuc = 3818;
pub const KEY_Hangul_J_Dikeud = 3802;
pub const KEY_Hangul_J_Hieuh = 3822;
pub const KEY_Hangul_J_Ieung = 3816;
pub const KEY_Hangul_J_Jieuj = 3817;
pub const KEY_Hangul_J_Khieuq = 3819;
pub const KEY_Hangul_J_Kiyeog = 3796;
pub const KEY_Hangul_J_KiyeogSios = 3798;
pub const KEY_Hangul_J_KkogjiDalrinIeung = 3833;
pub const KEY_Hangul_J_Mieum = 3811;
pub const KEY_Hangul_J_Nieun = 3799;
pub const KEY_Hangul_J_NieunHieuh = 3801;
pub const KEY_Hangul_J_NieunJieuj = 3800;
pub const KEY_Hangul_J_PanSios = 3832;
pub const KEY_Hangul_J_Phieuf = 3821;
pub const KEY_Hangul_J_Pieub = 3812;
pub const KEY_Hangul_J_PieubSios = 3813;
pub const KEY_Hangul_J_Rieul = 3803;
pub const KEY_Hangul_J_RieulHieuh = 3810;
pub const KEY_Hangul_J_RieulKiyeog = 3804;
pub const KEY_Hangul_J_RieulMieum = 3805;
pub const KEY_Hangul_J_RieulPhieuf = 3809;
pub const KEY_Hangul_J_RieulPieub = 3806;
pub const KEY_Hangul_J_RieulSios = 3807;
pub const KEY_Hangul_J_RieulTieut = 3808;
pub const KEY_Hangul_J_Sios = 3814;
pub const KEY_Hangul_J_SsangKiyeog = 3797;
pub const KEY_Hangul_J_SsangSios = 3815;
pub const KEY_Hangul_J_Tieut = 3820;
pub const KEY_Hangul_J_YeorinHieuh = 3834;
pub const KEY_Hangul_Jamo = 65333;
pub const KEY_Hangul_Jeonja = 65336;
pub const KEY_Hangul_Jieuj = 3768;
pub const KEY_Hangul_Khieuq = 3771;
pub const KEY_Hangul_Kiyeog = 3745;
pub const KEY_Hangul_KiyeogSios = 3747;
pub const KEY_Hangul_KkogjiDalrinIeung = 3827;
pub const KEY_Hangul_Mieum = 3761;
pub const KEY_Hangul_MultipleCandidate = 65341;
pub const KEY_Hangul_Nieun = 3748;
pub const KEY_Hangul_NieunHieuh = 3750;
pub const KEY_Hangul_NieunJieuj = 3749;
pub const KEY_Hangul_O = 3783;
pub const KEY_Hangul_OE = 3786;
pub const KEY_Hangul_PanSios = 3826;
pub const KEY_Hangul_Phieuf = 3773;
pub const KEY_Hangul_Pieub = 3762;
pub const KEY_Hangul_PieubSios = 3764;
pub const KEY_Hangul_PostHanja = 65339;
pub const KEY_Hangul_PreHanja = 65338;
pub const KEY_Hangul_PreviousCandidate = 65342;
pub const KEY_Hangul_Rieul = 3753;
pub const KEY_Hangul_RieulHieuh = 3760;
pub const KEY_Hangul_RieulKiyeog = 3754;
pub const KEY_Hangul_RieulMieum = 3755;
pub const KEY_Hangul_RieulPhieuf = 3759;
pub const KEY_Hangul_RieulPieub = 3756;
pub const KEY_Hangul_RieulSios = 3757;
pub const KEY_Hangul_RieulTieut = 3758;
pub const KEY_Hangul_RieulYeorinHieuh = 3823;
pub const KEY_Hangul_Romaja = 65334;
pub const KEY_Hangul_SingleCandidate = 65340;
pub const KEY_Hangul_Sios = 3765;
pub const KEY_Hangul_Special = 65343;
pub const KEY_Hangul_SsangDikeud = 3752;
pub const KEY_Hangul_SsangJieuj = 3769;
pub const KEY_Hangul_SsangKiyeog = 3746;
pub const KEY_Hangul_SsangPieub = 3763;
pub const KEY_Hangul_SsangSios = 3766;
pub const KEY_Hangul_Start = 65330;
pub const KEY_Hangul_SunkyeongeumMieum = 3824;
pub const KEY_Hangul_SunkyeongeumPhieuf = 3828;
pub const KEY_Hangul_SunkyeongeumPieub = 3825;
pub const KEY_Hangul_Tieut = 3772;
pub const KEY_Hangul_U = 3788;
pub const KEY_Hangul_WA = 3784;
pub const KEY_Hangul_WAE = 3785;
pub const KEY_Hangul_WE = 3790;
pub const KEY_Hangul_WEO = 3789;
pub const KEY_Hangul_WI = 3791;
pub const KEY_Hangul_YA = 3777;
pub const KEY_Hangul_YAE = 3778;
pub const KEY_Hangul_YE = 3782;
pub const KEY_Hangul_YEO = 3781;
pub const KEY_Hangul_YI = 3794;
pub const KEY_Hangul_YO = 3787;
pub const KEY_Hangul_YU = 3792;
pub const KEY_Hangul_YeorinHieuh = 3829;
pub const KEY_Hangul_switch = 65406;
pub const KEY_Hankaku = 65321;
pub const KEY_Hcircumflex = 678;
pub const KEY_Hebrew_switch = 65406;
pub const KEY_Help = 65386;
pub const KEY_Henkan = 65315;
pub const KEY_Henkan_Mode = 65315;
pub const KEY_Hibernate = 269025192;
pub const KEY_Hiragana = 65317;
pub const KEY_Hiragana_Katakana = 65319;
pub const KEY_History = 269025079;
pub const KEY_Home = 65360;
pub const KEY_HomePage = 269025048;
pub const KEY_HotLinks = 269025082;
pub const KEY_Hstroke = 673;
pub const KEY_Hyper_L = 65517;
pub const KEY_Hyper_R = 65518;
pub const KEY_I = 73;
pub const KEY_ISO_Center_Object = 65075;
pub const KEY_ISO_Continuous_Underline = 65072;
pub const KEY_ISO_Discontinuous_Underline = 65073;
pub const KEY_ISO_Emphasize = 65074;
pub const KEY_ISO_Enter = 65076;
pub const KEY_ISO_Fast_Cursor_Down = 65071;
pub const KEY_ISO_Fast_Cursor_Left = 65068;
pub const KEY_ISO_Fast_Cursor_Right = 65069;
pub const KEY_ISO_Fast_Cursor_Up = 65070;
pub const KEY_ISO_First_Group = 65036;
pub const KEY_ISO_First_Group_Lock = 65037;
pub const KEY_ISO_Group_Latch = 65030;
pub const KEY_ISO_Group_Lock = 65031;
pub const KEY_ISO_Group_Shift = 65406;
pub const KEY_ISO_Last_Group = 65038;
pub const KEY_ISO_Last_Group_Lock = 65039;
pub const KEY_ISO_Left_Tab = 65056;
pub const KEY_ISO_Level2_Latch = 65026;
pub const KEY_ISO_Level3_Latch = 65028;
pub const KEY_ISO_Level3_Lock = 65029;
pub const KEY_ISO_Level3_Shift = 65027;
pub const KEY_ISO_Level5_Latch = 65042;
pub const KEY_ISO_Level5_Lock = 65043;
pub const KEY_ISO_Level5_Shift = 65041;
pub const KEY_ISO_Lock = 65025;
pub const KEY_ISO_Move_Line_Down = 65058;
pub const KEY_ISO_Move_Line_Up = 65057;
pub const KEY_ISO_Next_Group = 65032;
pub const KEY_ISO_Next_Group_Lock = 65033;
pub const KEY_ISO_Partial_Line_Down = 65060;
pub const KEY_ISO_Partial_Line_Up = 65059;
pub const KEY_ISO_Partial_Space_Left = 65061;
pub const KEY_ISO_Partial_Space_Right = 65062;
pub const KEY_ISO_Prev_Group = 65034;
pub const KEY_ISO_Prev_Group_Lock = 65035;
pub const KEY_ISO_Release_Both_Margins = 65067;
pub const KEY_ISO_Release_Margin_Left = 65065;
pub const KEY_ISO_Release_Margin_Right = 65066;
pub const KEY_ISO_Set_Margin_Left = 65063;
pub const KEY_ISO_Set_Margin_Right = 65064;
pub const KEY_Iabovedot = 681;
pub const KEY_Iacute = 205;
pub const KEY_Ibelowdot = 16785098;
pub const KEY_Ibreve = 16777516;
pub const KEY_Icircumflex = 206;
pub const KEY_Idiaeresis = 207;
pub const KEY_Igrave = 204;
pub const KEY_Ihook = 16785096;
pub const KEY_Imacron = 975;
pub const KEY_Insert = 65379;
pub const KEY_Iogonek = 967;
pub const KEY_Itilde = 933;
pub const KEY_J = 74;
pub const KEY_Jcircumflex = 684;
pub const KEY_K = 75;
pub const KEY_KP_0 = 65456;
pub const KEY_KP_1 = 65457;
pub const KEY_KP_2 = 65458;
pub const KEY_KP_3 = 65459;
pub const KEY_KP_4 = 65460;
pub const KEY_KP_5 = 65461;
pub const KEY_KP_6 = 65462;
pub const KEY_KP_7 = 65463;
pub const KEY_KP_8 = 65464;
pub const KEY_KP_9 = 65465;
pub const KEY_KP_Add = 65451;
pub const KEY_KP_Begin = 65437;
pub const KEY_KP_Decimal = 65454;
pub const KEY_KP_Delete = 65439;
pub const KEY_KP_Divide = 65455;
pub const KEY_KP_Down = 65433;
pub const KEY_KP_End = 65436;
pub const KEY_KP_Enter = 65421;
pub const KEY_KP_Equal = 65469;
pub const KEY_KP_F1 = 65425;
pub const KEY_KP_F2 = 65426;
pub const KEY_KP_F3 = 65427;
pub const KEY_KP_F4 = 65428;
pub const KEY_KP_Home = 65429;
pub const KEY_KP_Insert = 65438;
pub const KEY_KP_Left = 65430;
pub const KEY_KP_Multiply = 65450;
pub const KEY_KP_Next = 65435;
pub const KEY_KP_Page_Down = 65435;
pub const KEY_KP_Page_Up = 65434;
pub const KEY_KP_Prior = 65434;
pub const KEY_KP_Right = 65432;
pub const KEY_KP_Separator = 65452;
pub const KEY_KP_Space = 65408;
pub const KEY_KP_Subtract = 65453;
pub const KEY_KP_Tab = 65417;
pub const KEY_KP_Up = 65431;
pub const KEY_Kana_Lock = 65325;
pub const KEY_Kana_Shift = 65326;
pub const KEY_Kanji = 65313;
pub const KEY_Kanji_Bangou = 65335;
pub const KEY_Katakana = 65318;
pub const KEY_KbdBrightnessDown = 269025030;
pub const KEY_KbdBrightnessUp = 269025029;
pub const KEY_KbdLightOnOff = 269025028;
pub const KEY_Kcedilla = 979;
pub const KEY_Keyboard = 269025203;
pub const KEY_Korean_Won = 3839;
pub const KEY_L = 76;
pub const KEY_L1 = 65480;
pub const KEY_L10 = 65489;
pub const KEY_L2 = 65481;
pub const KEY_L3 = 65482;
pub const KEY_L4 = 65483;
pub const KEY_L5 = 65484;
pub const KEY_L6 = 65485;
pub const KEY_L7 = 65486;
pub const KEY_L8 = 65487;
pub const KEY_L9 = 65488;
pub const KEY_Lacute = 453;
pub const KEY_Last_Virtual_Screen = 65236;
pub const KEY_Launch0 = 269025088;
pub const KEY_Launch1 = 269025089;
pub const KEY_Launch2 = 269025090;
pub const KEY_Launch3 = 269025091;
pub const KEY_Launch4 = 269025092;
pub const KEY_Launch5 = 269025093;
pub const KEY_Launch6 = 269025094;
pub const KEY_Launch7 = 269025095;
pub const KEY_Launch8 = 269025096;
pub const KEY_Launch9 = 269025097;
pub const KEY_LaunchA = 269025098;
pub const KEY_LaunchB = 269025099;
pub const KEY_LaunchC = 269025100;
pub const KEY_LaunchD = 269025101;
pub const KEY_LaunchE = 269025102;
pub const KEY_LaunchF = 269025103;
pub const KEY_Lbelowdot = 16784950;
pub const KEY_Lcaron = 421;
pub const KEY_Lcedilla = 934;
pub const KEY_Left = 65361;
pub const KEY_LightBulb = 269025077;
pub const KEY_Linefeed = 65290;
pub const KEY_LiraSign = 16785572;
pub const KEY_LogGrabInfo = 269024805;
pub const KEY_LogOff = 269025121;
pub const KEY_LogWindowTree = 269024804;
pub const KEY_Lstroke = 419;
pub const KEY_M = 77;
pub const KEY_Mabovedot = 16784960;
pub const KEY_Macedonia_DSE = 1717;
pub const KEY_Macedonia_GJE = 1714;
pub const KEY_Macedonia_KJE = 1724;
pub const KEY_Macedonia_dse = 1701;
pub const KEY_Macedonia_gje = 1698;
pub const KEY_Macedonia_kje = 1708;
pub const KEY_Mae_Koho = 65342;
pub const KEY_Mail = 269025049;
pub const KEY_MailForward = 269025168;
pub const KEY_Market = 269025122;
pub const KEY_Massyo = 65324;
pub const KEY_Meeting = 269025123;
pub const KEY_Memo = 269025054;
pub const KEY_Menu = 65383;
pub const KEY_MenuKB = 269025125;
pub const KEY_MenuPB = 269025126;
pub const KEY_Messenger = 269025166;
pub const KEY_Meta_L = 65511;
pub const KEY_Meta_R = 65512;
pub const KEY_MillSign = 16785573;
pub const KEY_ModeLock = 269025025;
pub const KEY_Mode_switch = 65406;
pub const KEY_MonBrightnessDown = 269025027;
pub const KEY_MonBrightnessUp = 269025026;
pub const KEY_MouseKeys_Accel_Enable = 65143;
pub const KEY_MouseKeys_Enable = 65142;
pub const KEY_Muhenkan = 65314;
pub const KEY_Multi_key = 65312;
pub const KEY_MultipleCandidate = 65341;
pub const KEY_Music = 269025170;
pub const KEY_MyComputer = 269025075;
pub const KEY_MySites = 269025127;
pub const KEY_N = 78;
pub const KEY_Nacute = 465;
pub const KEY_NairaSign = 16785574;
pub const KEY_Ncaron = 466;
pub const KEY_Ncedilla = 977;
pub const KEY_New = 269025128;
pub const KEY_NewSheqelSign = 16785578;
pub const KEY_News = 269025129;
pub const KEY_Next = 65366;
pub const KEY_Next_VMode = 269024802;
pub const KEY_Next_Virtual_Screen = 65234;
pub const KEY_Ntilde = 209;
pub const KEY_Num_Lock = 65407;
pub const KEY_O = 79;
pub const KEY_OE = 5052;
pub const KEY_Oacute = 211;
pub const KEY_Obarred = 16777631;
pub const KEY_Obelowdot = 16785100;
pub const KEY_Ocaron = 16777681;
pub const KEY_Ocircumflex = 212;
pub const KEY_Ocircumflexacute = 16785104;
pub const KEY_Ocircumflexbelowdot = 16785112;
pub const KEY_Ocircumflexgrave = 16785106;
pub const KEY_Ocircumflexhook = 16785108;
pub const KEY_Ocircumflextilde = 16785110;
pub const KEY_Odiaeresis = 214;
pub const KEY_Odoubleacute = 469;
pub const KEY_OfficeHome = 269025130;
pub const KEY_Ograve = 210;
pub const KEY_Ohook = 16785102;
pub const KEY_Ohorn = 16777632;
pub const KEY_Ohornacute = 16785114;
pub const KEY_Ohornbelowdot = 16785122;
pub const KEY_Ohorngrave = 16785116;
pub const KEY_Ohornhook = 16785118;
pub const KEY_Ohorntilde = 16785120;
pub const KEY_Omacron = 978;
pub const KEY_Ooblique = 216;
pub const KEY_Open = 269025131;
pub const KEY_OpenURL = 269025080;
pub const KEY_Option = 269025132;
pub const KEY_Oslash = 216;
pub const KEY_Otilde = 213;
pub const KEY_Overlay1_Enable = 65144;
pub const KEY_Overlay2_Enable = 65145;
pub const KEY_P = 80;
pub const KEY_Pabovedot = 16784982;
pub const KEY_Page_Down = 65366;
pub const KEY_Page_Up = 65365;
pub const KEY_Paste = 269025133;
pub const KEY_Pause = 65299;
pub const KEY_PesetaSign = 16785575;
pub const KEY_Phone = 269025134;
pub const KEY_Pictures = 269025169;
pub const KEY_Pointer_Accelerate = 65274;
pub const KEY_Pointer_Button1 = 65257;
pub const KEY_Pointer_Button2 = 65258;
pub const KEY_Pointer_Button3 = 65259;
pub const KEY_Pointer_Button4 = 65260;
pub const KEY_Pointer_Button5 = 65261;
pub const KEY_Pointer_Button_Dflt = 65256;
pub const KEY_Pointer_DblClick1 = 65263;
pub const KEY_Pointer_DblClick2 = 65264;
pub const KEY_Pointer_DblClick3 = 65265;
pub const KEY_Pointer_DblClick4 = 65266;
pub const KEY_Pointer_DblClick5 = 65267;
pub const KEY_Pointer_DblClick_Dflt = 65262;
pub const KEY_Pointer_DfltBtnNext = 65275;
pub const KEY_Pointer_DfltBtnPrev = 65276;
pub const KEY_Pointer_Down = 65251;
pub const KEY_Pointer_DownLeft = 65254;
pub const KEY_Pointer_DownRight = 65255;
pub const KEY_Pointer_Drag1 = 65269;
pub const KEY_Pointer_Drag2 = 65270;
pub const KEY_Pointer_Drag3 = 65271;
pub const KEY_Pointer_Drag4 = 65272;
pub const KEY_Pointer_Drag5 = 65277;
pub const KEY_Pointer_Drag_Dflt = 65268;
pub const KEY_Pointer_EnableKeys = 65273;
pub const KEY_Pointer_Left = 65248;
pub const KEY_Pointer_Right = 65249;
pub const KEY_Pointer_Up = 65250;
pub const KEY_Pointer_UpLeft = 65252;
pub const KEY_Pointer_UpRight = 65253;
pub const KEY_PowerDown = 269025057;
pub const KEY_PowerOff = 269025066;
pub const KEY_Prev_VMode = 269024803;
pub const KEY_Prev_Virtual_Screen = 65233;
pub const KEY_PreviousCandidate = 65342;
pub const KEY_Print = 65377;
pub const KEY_Prior = 65365;
pub const KEY_Q = 81;
pub const KEY_R = 82;
pub const KEY_R1 = 65490;
pub const KEY_R10 = 65499;
pub const KEY_R11 = 65500;
pub const KEY_R12 = 65501;
pub const KEY_R13 = 65502;
pub const KEY_R14 = 65503;
pub const KEY_R15 = 65504;
pub const KEY_R2 = 65491;
pub const KEY_R3 = 65492;
pub const KEY_R4 = 65493;
pub const KEY_R5 = 65494;
pub const KEY_R6 = 65495;
pub const KEY_R7 = 65496;
pub const KEY_R8 = 65497;
pub const KEY_R9 = 65498;
pub const KEY_RFKill = 269025205;
pub const KEY_Racute = 448;
pub const KEY_Rcaron = 472;
pub const KEY_Rcedilla = 931;
pub const KEY_Red = 269025187;
pub const KEY_Redo = 65382;
pub const KEY_Refresh = 269025065;
pub const KEY_Reload = 269025139;
pub const KEY_RepeatKeys_Enable = 65138;
pub const KEY_Reply = 269025138;
pub const KEY_Return = 65293;
pub const KEY_Right = 65363;
pub const KEY_RockerDown = 269025060;
pub const KEY_RockerEnter = 269025061;
pub const KEY_RockerUp = 269025059;
pub const KEY_Romaji = 65316;
pub const KEY_RotateWindows = 269025140;
pub const KEY_RotationKB = 269025142;
pub const KEY_RotationPB = 269025141;
pub const KEY_RupeeSign = 16785576;
pub const KEY_S = 83;
pub const KEY_SCHWA = 16777615;
pub const KEY_Sabovedot = 16784992;
pub const KEY_Sacute = 422;
pub const KEY_Save = 269025143;
pub const KEY_Scaron = 425;
pub const KEY_Scedilla = 426;
pub const KEY_Scircumflex = 734;
pub const KEY_ScreenSaver = 269025069;
pub const KEY_ScrollClick = 269025146;
pub const KEY_ScrollDown = 269025145;
pub const KEY_ScrollUp = 269025144;
pub const KEY_Scroll_Lock = 65300;
pub const KEY_Search = 269025051;
pub const KEY_Select = 65376;
pub const KEY_SelectButton = 269025184;
pub const KEY_Send = 269025147;
pub const KEY_Serbian_DJE = 1713;
pub const KEY_Serbian_DZE = 1727;
pub const KEY_Serbian_JE = 1720;
pub const KEY_Serbian_LJE = 1721;
pub const KEY_Serbian_NJE = 1722;
pub const KEY_Serbian_TSHE = 1723;
pub const KEY_Serbian_dje = 1697;
pub const KEY_Serbian_dze = 1711;
pub const KEY_Serbian_je = 1704;
pub const KEY_Serbian_lje = 1705;
pub const KEY_Serbian_nje = 1706;
pub const KEY_Serbian_tshe = 1707;
pub const KEY_Shift_L = 65505;
pub const KEY_Shift_Lock = 65510;
pub const KEY_Shift_R = 65506;
pub const KEY_Shop = 269025078;
pub const KEY_SingleCandidate = 65340;
pub const KEY_Sinh_a = 16780677;
pub const KEY_Sinh_aa = 16780678;
pub const KEY_Sinh_aa2 = 16780751;
pub const KEY_Sinh_ae = 16780679;
pub const KEY_Sinh_ae2 = 16780752;
pub const KEY_Sinh_aee = 16780680;
pub const KEY_Sinh_aee2 = 16780753;
pub const KEY_Sinh_ai = 16780691;
pub const KEY_Sinh_ai2 = 16780763;
pub const KEY_Sinh_al = 16780746;
pub const KEY_Sinh_au = 16780694;
pub const KEY_Sinh_au2 = 16780766;
pub const KEY_Sinh_ba = 16780726;
pub const KEY_Sinh_bha = 16780727;
pub const KEY_Sinh_ca = 16780704;
pub const KEY_Sinh_cha = 16780705;
pub const KEY_Sinh_dda = 16780713;
pub const KEY_Sinh_ddha = 16780714;
pub const KEY_Sinh_dha = 16780719;
pub const KEY_Sinh_dhha = 16780720;
pub const KEY_Sinh_e = 16780689;
pub const KEY_Sinh_e2 = 16780761;
pub const KEY_Sinh_ee = 16780690;
pub const KEY_Sinh_ee2 = 16780762;
pub const KEY_Sinh_fa = 16780742;
pub const KEY_Sinh_ga = 16780700;
pub const KEY_Sinh_gha = 16780701;
pub const KEY_Sinh_h2 = 16780675;
pub const KEY_Sinh_ha = 16780740;
pub const KEY_Sinh_i = 16780681;
pub const KEY_Sinh_i2 = 16780754;
pub const KEY_Sinh_ii = 16780682;
pub const KEY_Sinh_ii2 = 16780755;
pub const KEY_Sinh_ja = 16780706;
pub const KEY_Sinh_jha = 16780707;
pub const KEY_Sinh_jnya = 16780709;
pub const KEY_Sinh_ka = 16780698;
pub const KEY_Sinh_kha = 16780699;
pub const KEY_Sinh_kunddaliya = 16780788;
pub const KEY_Sinh_la = 16780733;
pub const KEY_Sinh_lla = 16780741;
pub const KEY_Sinh_lu = 16780687;
pub const KEY_Sinh_lu2 = 16780767;
pub const KEY_Sinh_luu = 16780688;
pub const KEY_Sinh_luu2 = 16780787;
pub const KEY_Sinh_ma = 16780728;
pub const KEY_Sinh_mba = 16780729;
pub const KEY_Sinh_na = 16780721;
pub const KEY_Sinh_ndda = 16780716;
pub const KEY_Sinh_ndha = 16780723;
pub const KEY_Sinh_ng = 16780674;
pub const KEY_Sinh_ng2 = 16780702;
pub const KEY_Sinh_nga = 16780703;
pub const KEY_Sinh_nja = 16780710;
pub const KEY_Sinh_nna = 16780715;
pub const KEY_Sinh_nya = 16780708;
pub const KEY_Sinh_o = 16780692;
pub const KEY_Sinh_o2 = 16780764;
pub const KEY_Sinh_oo = 16780693;
pub const KEY_Sinh_oo2 = 16780765;
pub const KEY_Sinh_pa = 16780724;
pub const KEY_Sinh_pha = 16780725;
pub const KEY_Sinh_ra = 16780731;
pub const KEY_Sinh_ri = 16780685;
pub const KEY_Sinh_rii = 16780686;
pub const KEY_Sinh_ru2 = 16780760;
pub const KEY_Sinh_ruu2 = 16780786;
pub const KEY_Sinh_sa = 16780739;
pub const KEY_Sinh_sha = 16780737;
pub const KEY_Sinh_ssha = 16780738;
pub const KEY_Sinh_tha = 16780717;
pub const KEY_Sinh_thha = 16780718;
pub const KEY_Sinh_tta = 16780711;
pub const KEY_Sinh_ttha = 16780712;
pub const KEY_Sinh_u = 16780683;
pub const KEY_Sinh_u2 = 16780756;
pub const KEY_Sinh_uu = 16780684;
pub const KEY_Sinh_uu2 = 16780758;
pub const KEY_Sinh_va = 16780736;
pub const KEY_Sinh_ya = 16780730;
pub const KEY_Sleep = 269025071;
pub const KEY_SlowKeys_Enable = 65139;
pub const KEY_Spell = 269025148;
pub const KEY_SplitScreen = 269025149;
pub const KEY_Standby = 269025040;
pub const KEY_Start = 269025050;
pub const KEY_StickyKeys_Enable = 65141;
pub const KEY_Stop = 269025064;
pub const KEY_Subtitle = 269025178;
pub const KEY_Super_L = 65515;
pub const KEY_Super_R = 65516;
pub const KEY_Support = 269025150;
pub const KEY_Suspend = 269025191;
pub const KEY_Switch_VT_1 = 269024769;
pub const KEY_Switch_VT_10 = 269024778;
pub const KEY_Switch_VT_11 = 269024779;
pub const KEY_Switch_VT_12 = 269024780;
pub const KEY_Switch_VT_2 = 269024770;
pub const KEY_Switch_VT_3 = 269024771;
pub const KEY_Switch_VT_4 = 269024772;
pub const KEY_Switch_VT_5 = 269024773;
pub const KEY_Switch_VT_6 = 269024774;
pub const KEY_Switch_VT_7 = 269024775;
pub const KEY_Switch_VT_8 = 269024776;
pub const KEY_Switch_VT_9 = 269024777;
pub const KEY_Sys_Req = 65301;
pub const KEY_T = 84;
pub const KEY_THORN = 222;
pub const KEY_Tab = 65289;
pub const KEY_Tabovedot = 16785002;
pub const KEY_TaskPane = 269025151;
pub const KEY_Tcaron = 427;
pub const KEY_Tcedilla = 478;
pub const KEY_Terminal = 269025152;
pub const KEY_Terminate_Server = 65237;
pub const KEY_Thai_baht = 3551;
pub const KEY_Thai_bobaimai = 3514;
pub const KEY_Thai_chochan = 3496;
pub const KEY_Thai_chochang = 3498;
pub const KEY_Thai_choching = 3497;
pub const KEY_Thai_chochoe = 3500;
pub const KEY_Thai_dochada = 3502;
pub const KEY_Thai_dodek = 3508;
pub const KEY_Thai_fofa = 3517;
pub const KEY_Thai_fofan = 3519;
pub const KEY_Thai_hohip = 3531;
pub const KEY_Thai_honokhuk = 3534;
pub const KEY_Thai_khokhai = 3490;
pub const KEY_Thai_khokhon = 3493;
pub const KEY_Thai_khokhuat = 3491;
pub const KEY_Thai_khokhwai = 3492;
pub const KEY_Thai_khorakhang = 3494;
pub const KEY_Thai_kokai = 3489;
pub const KEY_Thai_lakkhangyao = 3557;
pub const KEY_Thai_lekchet = 3575;
pub const KEY_Thai_lekha = 3573;
pub const KEY_Thai_lekhok = 3574;
pub const KEY_Thai_lekkao = 3577;
pub const KEY_Thai_leknung = 3569;
pub const KEY_Thai_lekpaet = 3576;
pub const KEY_Thai_leksam = 3571;
pub const KEY_Thai_leksi = 3572;
pub const KEY_Thai_leksong = 3570;
pub const KEY_Thai_leksun = 3568;
pub const KEY_Thai_lochula = 3532;
pub const KEY_Thai_loling = 3525;
pub const KEY_Thai_lu = 3526;
pub const KEY_Thai_maichattawa = 3563;
pub const KEY_Thai_maiek = 3560;
pub const KEY_Thai_maihanakat = 3537;
pub const KEY_Thai_maihanakat_maitho = 3550;
pub const KEY_Thai_maitaikhu = 3559;
pub const KEY_Thai_maitho = 3561;
pub const KEY_Thai_maitri = 3562;
pub const KEY_Thai_maiyamok = 3558;
pub const KEY_Thai_moma = 3521;
pub const KEY_Thai_ngongu = 3495;
pub const KEY_Thai_nikhahit = 3565;
pub const KEY_Thai_nonen = 3507;
pub const KEY_Thai_nonu = 3513;
pub const KEY_Thai_oang = 3533;
pub const KEY_Thai_paiyannoi = 3535;
pub const KEY_Thai_phinthu = 3546;
pub const KEY_Thai_phophan = 3518;
pub const KEY_Thai_phophung = 3516;
pub const KEY_Thai_phosamphao = 3520;
pub const KEY_Thai_popla = 3515;
pub const KEY_Thai_rorua = 3523;
pub const KEY_Thai_ru = 3524;
pub const KEY_Thai_saraa = 3536;
pub const KEY_Thai_saraaa = 3538;
pub const KEY_Thai_saraae = 3553;
pub const KEY_Thai_saraaimaimalai = 3556;
pub const KEY_Thai_saraaimaimuan = 3555;
pub const KEY_Thai_saraam = 3539;
pub const KEY_Thai_sarae = 3552;
pub const KEY_Thai_sarai = 3540;
pub const KEY_Thai_saraii = 3541;
pub const KEY_Thai_sarao = 3554;
pub const KEY_Thai_sarau = 3544;
pub const KEY_Thai_saraue = 3542;
pub const KEY_Thai_sarauee = 3543;
pub const KEY_Thai_sarauu = 3545;
pub const KEY_Thai_sorusi = 3529;
pub const KEY_Thai_sosala = 3528;
pub const KEY_Thai_soso = 3499;
pub const KEY_Thai_sosua = 3530;
pub const KEY_Thai_thanthakhat = 3564;
pub const KEY_Thai_thonangmontho = 3505;
pub const KEY_Thai_thophuthao = 3506;
pub const KEY_Thai_thothahan = 3511;
pub const KEY_Thai_thothan = 3504;
pub const KEY_Thai_thothong = 3512;
pub const KEY_Thai_thothung = 3510;
pub const KEY_Thai_topatak = 3503;
pub const KEY_Thai_totao = 3509;
pub const KEY_Thai_wowaen = 3527;
pub const KEY_Thai_yoyak = 3522;
pub const KEY_Thai_yoying = 3501;
pub const KEY_Thorn = 222;
pub const KEY_Time = 269025183;
pub const KEY_ToDoList = 269025055;
pub const KEY_Tools = 269025153;
pub const KEY_TopMenu = 269025186;
pub const KEY_TouchpadOff = 269025201;
pub const KEY_TouchpadOn = 269025200;
pub const KEY_TouchpadToggle = 269025193;
pub const KEY_Touroku = 65323;
pub const KEY_Travel = 269025154;
pub const KEY_Tslash = 940;
pub const KEY_U = 85;
pub const KEY_UWB = 269025174;
pub const KEY_Uacute = 218;
pub const KEY_Ubelowdot = 16785124;
pub const KEY_Ubreve = 733;
pub const KEY_Ucircumflex = 219;
pub const KEY_Udiaeresis = 220;
pub const KEY_Udoubleacute = 475;
pub const KEY_Ugrave = 217;
pub const KEY_Uhook = 16785126;
pub const KEY_Uhorn = 16777647;
pub const KEY_Uhornacute = 16785128;
pub const KEY_Uhornbelowdot = 16785136;
pub const KEY_Uhorngrave = 16785130;
pub const KEY_Uhornhook = 16785132;
pub const KEY_Uhorntilde = 16785134;
pub const KEY_Ukrainian_GHE_WITH_UPTURN = 1725;
pub const KEY_Ukrainian_I = 1718;
pub const KEY_Ukrainian_IE = 1716;
pub const KEY_Ukrainian_YI = 1719;
pub const KEY_Ukrainian_ghe_with_upturn = 1709;
pub const KEY_Ukrainian_i = 1702;
pub const KEY_Ukrainian_ie = 1700;
pub const KEY_Ukrainian_yi = 1703;
pub const KEY_Ukranian_I = 1718;
pub const KEY_Ukranian_JE = 1716;
pub const KEY_Ukranian_YI = 1719;
pub const KEY_Ukranian_i = 1702;
pub const KEY_Ukranian_je = 1700;
pub const KEY_Ukranian_yi = 1703;
pub const KEY_Umacron = 990;
pub const KEY_Undo = 65381;
pub const KEY_Ungrab = 269024800;
pub const KEY_Uogonek = 985;
pub const KEY_Up = 65362;
pub const KEY_Uring = 473;
pub const KEY_User1KB = 269025157;
pub const KEY_User2KB = 269025158;
pub const KEY_UserPB = 269025156;
pub const KEY_Utilde = 989;
pub const KEY_V = 86;
pub const KEY_VendorHome = 269025076;
pub const KEY_Video = 269025159;
pub const KEY_View = 269025185;
pub const KEY_VoidSymbol = 16777215;
pub const KEY_W = 87;
pub const KEY_WLAN = 269025173;
pub const KEY_WWAN = 269025204;
pub const KEY_WWW = 269025070;
pub const KEY_Wacute = 16785026;
pub const KEY_WakeUp = 269025067;
pub const KEY_Wcircumflex = 16777588;
pub const KEY_Wdiaeresis = 16785028;
pub const KEY_WebCam = 269025167;
pub const KEY_Wgrave = 16785024;
pub const KEY_WheelButton = 269025160;
pub const KEY_WindowClear = 269025109;
pub const KEY_WonSign = 16785577;
pub const KEY_Word = 269025161;
pub const KEY_X = 88;
pub const KEY_Xabovedot = 16785034;
pub const KEY_Xfer = 269025162;
pub const KEY_Y = 89;
pub const KEY_Yacute = 221;
pub const KEY_Ybelowdot = 16785140;
pub const KEY_Ycircumflex = 16777590;
pub const KEY_Ydiaeresis = 5054;
pub const KEY_Yellow = 269025189;
pub const KEY_Ygrave = 16785138;
pub const KEY_Yhook = 16785142;
pub const KEY_Ytilde = 16785144;
pub const KEY_Z = 90;
pub const KEY_Zabovedot = 431;
pub const KEY_Zacute = 428;
pub const KEY_Zcaron = 430;
pub const KEY_Zen_Koho = 65341;
pub const KEY_Zenkaku = 65320;
pub const KEY_Zenkaku_Hankaku = 65322;
pub const KEY_ZoomIn = 269025163;
pub const KEY_ZoomOut = 269025164;
pub const KEY_Zstroke = 16777653;
pub const KEY_a = 97;
pub const KEY_aacute = 225;
pub const KEY_abelowdot = 16785057;
pub const KEY_abovedot = 511;
pub const KEY_abreve = 483;
pub const KEY_abreveacute = 16785071;
pub const KEY_abrevebelowdot = 16785079;
pub const KEY_abrevegrave = 16785073;
pub const KEY_abrevehook = 16785075;
pub const KEY_abrevetilde = 16785077;
pub const KEY_acircumflex = 226;
pub const KEY_acircumflexacute = 16785061;
pub const KEY_acircumflexbelowdot = 16785069;
pub const KEY_acircumflexgrave = 16785063;
pub const KEY_acircumflexhook = 16785065;
pub const KEY_acircumflextilde = 16785067;
pub const KEY_acute = 180;
pub const KEY_adiaeresis = 228;
pub const KEY_ae = 230;
pub const KEY_agrave = 224;
pub const KEY_ahook = 16785059;
pub const KEY_amacron = 992;
pub const KEY_ampersand = 38;
pub const KEY_aogonek = 433;
pub const KEY_apostrophe = 39;
pub const KEY_approxeq = 16785992;
pub const KEY_approximate = 2248;
pub const KEY_aring = 229;
pub const KEY_asciicircum = 94;
pub const KEY_asciitilde = 126;
pub const KEY_asterisk = 42;
pub const KEY_at = 64;
pub const KEY_atilde = 227;
pub const KEY_b = 98;
pub const KEY_babovedot = 16784899;
pub const KEY_backslash = 92;
pub const KEY_ballotcross = 2804;
pub const KEY_bar = 124;
pub const KEY_because = 16785973;
pub const KEY_blank = 2527;
pub const KEY_botintegral = 2213;
pub const KEY_botleftparens = 2220;
pub const KEY_botleftsqbracket = 2216;
pub const KEY_botleftsummation = 2226;
pub const KEY_botrightparens = 2222;
pub const KEY_botrightsqbracket = 2218;
pub const KEY_botrightsummation = 2230;
pub const KEY_bott = 2550;
pub const KEY_botvertsummationconnector = 2228;
pub const KEY_braceleft = 123;
pub const KEY_braceright = 125;
pub const KEY_bracketleft = 91;
pub const KEY_bracketright = 93;
pub const KEY_braille_blank = 16787456;
pub const KEY_braille_dot_1 = 65521;
pub const KEY_braille_dot_10 = 65530;
pub const KEY_braille_dot_2 = 65522;
pub const KEY_braille_dot_3 = 65523;
pub const KEY_braille_dot_4 = 65524;
pub const KEY_braille_dot_5 = 65525;
pub const KEY_braille_dot_6 = 65526;
pub const KEY_braille_dot_7 = 65527;
pub const KEY_braille_dot_8 = 65528;
pub const KEY_braille_dot_9 = 65529;
pub const KEY_braille_dots_1 = 16787457;
pub const KEY_braille_dots_12 = 16787459;
pub const KEY_braille_dots_123 = 16787463;
pub const KEY_braille_dots_1234 = 16787471;
pub const KEY_braille_dots_12345 = 16787487;
pub const KEY_braille_dots_123456 = 16787519;
pub const KEY_braille_dots_1234567 = 16787583;
pub const KEY_braille_dots_12345678 = 16787711;
pub const KEY_braille_dots_1234568 = 16787647;
pub const KEY_braille_dots_123457 = 16787551;
pub const KEY_braille_dots_1234578 = 16787679;
pub const KEY_braille_dots_123458 = 16787615;
pub const KEY_braille_dots_12346 = 16787503;
pub const KEY_braille_dots_123467 = 16787567;
pub const KEY_braille_dots_1234678 = 16787695;
pub const KEY_braille_dots_123468 = 16787631;
pub const KEY_braille_dots_12347 = 16787535;
pub const KEY_braille_dots_123478 = 16787663;
pub const KEY_braille_dots_12348 = 16787599;
pub const KEY_braille_dots_1235 = 16787479;
pub const KEY_braille_dots_12356 = 16787511;
pub const KEY_braille_dots_123567 = 16787575;
pub const KEY_braille_dots_1235678 = 16787703;
pub const KEY_braille_dots_123568 = 16787639;
pub const KEY_braille_dots_12357 = 16787543;
pub const KEY_braille_dots_123578 = 16787671;
pub const KEY_braille_dots_12358 = 16787607;
pub const KEY_braille_dots_1236 = 16787495;
pub const KEY_braille_dots_12367 = 16787559;
pub const KEY_braille_dots_123678 = 16787687;
pub const KEY_braille_dots_12368 = 16787623;
pub const KEY_braille_dots_1237 = 16787527;
pub const KEY_braille_dots_12378 = 16787655;
pub const KEY_braille_dots_1238 = 16787591;
pub const KEY_braille_dots_124 = 16787467;
pub const KEY_braille_dots_1245 = 16787483;
pub const KEY_braille_dots_12456 = 16787515;
pub const KEY_braille_dots_124567 = 16787579;
pub const KEY_braille_dots_1245678 = 16787707;
pub const KEY_braille_dots_124568 = 16787643;
pub const KEY_braille_dots_12457 = 16787547;
pub const KEY_braille_dots_124578 = 16787675;
pub const KEY_braille_dots_12458 = 16787611;
pub const KEY_braille_dots_1246 = 16787499;
pub const KEY_braille_dots_12467 = 16787563;
pub const KEY_braille_dots_124678 = 16787691;
pub const KEY_braille_dots_12468 = 16787627;
pub const KEY_braille_dots_1247 = 16787531;
pub const KEY_braille_dots_12478 = 16787659;
pub const KEY_braille_dots_1248 = 16787595;
pub const KEY_braille_dots_125 = 16787475;
pub const KEY_braille_dots_1256 = 16787507;
pub const KEY_braille_dots_12567 = 16787571;
pub const KEY_braille_dots_125678 = 16787699;
pub const KEY_braille_dots_12568 = 16787635;
pub const KEY_braille_dots_1257 = 16787539;
pub const KEY_braille_dots_12578 = 16787667;
pub const KEY_braille_dots_1258 = 16787603;
pub const KEY_braille_dots_126 = 16787491;
pub const KEY_braille_dots_1267 = 16787555;
pub const KEY_braille_dots_12678 = 16787683;
pub const KEY_braille_dots_1268 = 16787619;
pub const KEY_braille_dots_127 = 16787523;
pub const KEY_braille_dots_1278 = 16787651;
pub const KEY_braille_dots_128 = 16787587;
pub const KEY_braille_dots_13 = 16787461;
pub const KEY_braille_dots_134 = 16787469;
pub const KEY_braille_dots_1345 = 16787485;
pub const KEY_braille_dots_13456 = 16787517;
pub const KEY_braille_dots_134567 = 16787581;
pub const KEY_braille_dots_1345678 = 16787709;
pub const KEY_braille_dots_134568 = 16787645;
pub const KEY_braille_dots_13457 = 16787549;
pub const KEY_braille_dots_134578 = 16787677;
pub const KEY_braille_dots_13458 = 16787613;
pub const KEY_braille_dots_1346 = 16787501;
pub const KEY_braille_dots_13467 = 16787565;
pub const KEY_braille_dots_134678 = 16787693;
pub const KEY_braille_dots_13468 = 16787629;
pub const KEY_braille_dots_1347 = 16787533;
pub const KEY_braille_dots_13478 = 16787661;
pub const KEY_braille_dots_1348 = 16787597;
pub const KEY_braille_dots_135 = 16787477;
pub const KEY_braille_dots_1356 = 16787509;
pub const KEY_braille_dots_13567 = 16787573;
pub const KEY_braille_dots_135678 = 16787701;
pub const KEY_braille_dots_13568 = 16787637;
pub const KEY_braille_dots_1357 = 16787541;
pub const KEY_braille_dots_13578 = 16787669;
pub const KEY_braille_dots_1358 = 16787605;
pub const KEY_braille_dots_136 = 16787493;
pub const KEY_braille_dots_1367 = 16787557;
pub const KEY_braille_dots_13678 = 16787685;
pub const KEY_braille_dots_1368 = 16787621;
pub const KEY_braille_dots_137 = 16787525;
pub const KEY_braille_dots_1378 = 16787653;
pub const KEY_braille_dots_138 = 16787589;
pub const KEY_braille_dots_14 = 16787465;
pub const KEY_braille_dots_145 = 16787481;
pub const KEY_braille_dots_1456 = 16787513;
pub const KEY_braille_dots_14567 = 16787577;
pub const KEY_braille_dots_145678 = 16787705;
pub const KEY_braille_dots_14568 = 16787641;
pub const KEY_braille_dots_1457 = 16787545;
pub const KEY_braille_dots_14578 = 16787673;
pub const KEY_braille_dots_1458 = 16787609;
pub const KEY_braille_dots_146 = 16787497;
pub const KEY_braille_dots_1467 = 16787561;
pub const KEY_braille_dots_14678 = 16787689;
pub const KEY_braille_dots_1468 = 16787625;
pub const KEY_braille_dots_147 = 16787529;
pub const KEY_braille_dots_1478 = 16787657;
pub const KEY_braille_dots_148 = 16787593;
pub const KEY_braille_dots_15 = 16787473;
pub const KEY_braille_dots_156 = 16787505;
pub const KEY_braille_dots_1567 = 16787569;
pub const KEY_braille_dots_15678 = 16787697;
pub const KEY_braille_dots_1568 = 16787633;
pub const KEY_braille_dots_157 = 16787537;
pub const KEY_braille_dots_1578 = 16787665;
pub const KEY_braille_dots_158 = 16787601;
pub const KEY_braille_dots_16 = 16787489;
pub const KEY_braille_dots_167 = 16787553;
pub const KEY_braille_dots_1678 = 16787681;
pub const KEY_braille_dots_168 = 16787617;
pub const KEY_braille_dots_17 = 16787521;
pub const KEY_braille_dots_178 = 16787649;
pub const KEY_braille_dots_18 = 16787585;
pub const KEY_braille_dots_2 = 16787458;
pub const KEY_braille_dots_23 = 16787462;
pub const KEY_braille_dots_234 = 16787470;
pub const KEY_braille_dots_2345 = 16787486;
pub const KEY_braille_dots_23456 = 16787518;
pub const KEY_braille_dots_234567 = 16787582;
pub const KEY_braille_dots_2345678 = 16787710;
pub const KEY_braille_dots_234568 = 16787646;
pub const KEY_braille_dots_23457 = 16787550;
pub const KEY_braille_dots_234578 = 16787678;
pub const KEY_braille_dots_23458 = 16787614;
pub const KEY_braille_dots_2346 = 16787502;
pub const KEY_braille_dots_23467 = 16787566;
pub const KEY_braille_dots_234678 = 16787694;
pub const KEY_braille_dots_23468 = 16787630;
pub const KEY_braille_dots_2347 = 16787534;
pub const KEY_braille_dots_23478 = 16787662;
pub const KEY_braille_dots_2348 = 16787598;
pub const KEY_braille_dots_235 = 16787478;
pub const KEY_braille_dots_2356 = 16787510;
pub const KEY_braille_dots_23567 = 16787574;
pub const KEY_braille_dots_235678 = 16787702;
pub const KEY_braille_dots_23568 = 16787638;
pub const KEY_braille_dots_2357 = 16787542;
pub const KEY_braille_dots_23578 = 16787670;
pub const KEY_braille_dots_2358 = 16787606;
pub const KEY_braille_dots_236 = 16787494;
pub const KEY_braille_dots_2367 = 16787558;
pub const KEY_braille_dots_23678 = 16787686;
pub const KEY_braille_dots_2368 = 16787622;
pub const KEY_braille_dots_237 = 16787526;
pub const KEY_braille_dots_2378 = 16787654;
pub const KEY_braille_dots_238 = 16787590;
pub const KEY_braille_dots_24 = 16787466;
pub const KEY_braille_dots_245 = 16787482;
pub const KEY_braille_dots_2456 = 16787514;
pub const KEY_braille_dots_24567 = 16787578;
pub const KEY_braille_dots_245678 = 16787706;
pub const KEY_braille_dots_24568 = 16787642;
pub const KEY_braille_dots_2457 = 16787546;
pub const KEY_braille_dots_24578 = 16787674;
pub const KEY_braille_dots_2458 = 16787610;
pub const KEY_braille_dots_246 = 16787498;
pub const KEY_braille_dots_2467 = 16787562;
pub const KEY_braille_dots_24678 = 16787690;
pub const KEY_braille_dots_2468 = 16787626;
pub const KEY_braille_dots_247 = 16787530;
pub const KEY_braille_dots_2478 = 16787658;
pub const KEY_braille_dots_248 = 16787594;
pub const KEY_braille_dots_25 = 16787474;
pub const KEY_braille_dots_256 = 16787506;
pub const KEY_braille_dots_2567 = 16787570;
pub const KEY_braille_dots_25678 = 16787698;
pub const KEY_braille_dots_2568 = 16787634;
pub const KEY_braille_dots_257 = 16787538;
pub const KEY_braille_dots_2578 = 16787666;
pub const KEY_braille_dots_258 = 16787602;
pub const KEY_braille_dots_26 = 16787490;
pub const KEY_braille_dots_267 = 16787554;
pub const KEY_braille_dots_2678 = 16787682;
pub const KEY_braille_dots_268 = 16787618;
pub const KEY_braille_dots_27 = 16787522;
pub const KEY_braille_dots_278 = 16787650;
pub const KEY_braille_dots_28 = 16787586;
pub const KEY_braille_dots_3 = 16787460;
pub const KEY_braille_dots_34 = 16787468;
pub const KEY_braille_dots_345 = 16787484;
pub const KEY_braille_dots_3456 = 16787516;
pub const KEY_braille_dots_34567 = 16787580;
pub const KEY_braille_dots_345678 = 16787708;
pub const KEY_braille_dots_34568 = 16787644;
pub const KEY_braille_dots_3457 = 16787548;
pub const KEY_braille_dots_34578 = 16787676;
pub const KEY_braille_dots_3458 = 16787612;
pub const KEY_braille_dots_346 = 16787500;
pub const KEY_braille_dots_3467 = 16787564;
pub const KEY_braille_dots_34678 = 16787692;
pub const KEY_braille_dots_3468 = 16787628;
pub const KEY_braille_dots_347 = 16787532;
pub const KEY_braille_dots_3478 = 16787660;
pub const KEY_braille_dots_348 = 16787596;
pub const KEY_braille_dots_35 = 16787476;
pub const KEY_braille_dots_356 = 16787508;
pub const KEY_braille_dots_3567 = 16787572;
pub const KEY_braille_dots_35678 = 16787700;
pub const KEY_braille_dots_3568 = 16787636;
pub const KEY_braille_dots_357 = 16787540;
pub const KEY_braille_dots_3578 = 16787668;
pub const KEY_braille_dots_358 = 16787604;
pub const KEY_braille_dots_36 = 16787492;
pub const KEY_braille_dots_367 = 16787556;
pub const KEY_braille_dots_3678 = 16787684;
pub const KEY_braille_dots_368 = 16787620;
pub const KEY_braille_dots_37 = 16787524;
pub const KEY_braille_dots_378 = 16787652;
pub const KEY_braille_dots_38 = 16787588;
pub const KEY_braille_dots_4 = 16787464;
pub const KEY_braille_dots_45 = 16787480;
pub const KEY_braille_dots_456 = 16787512;
pub const KEY_braille_dots_4567 = 16787576;
pub const KEY_braille_dots_45678 = 16787704;
pub const KEY_braille_dots_4568 = 16787640;
pub const KEY_braille_dots_457 = 16787544;
pub const KEY_braille_dots_4578 = 16787672;
pub const KEY_braille_dots_458 = 16787608;
pub const KEY_braille_dots_46 = 16787496;
pub const KEY_braille_dots_467 = 16787560;
pub const KEY_braille_dots_4678 = 16787688;
pub const KEY_braille_dots_468 = 16787624;
pub const KEY_braille_dots_47 = 16787528;
pub const KEY_braille_dots_478 = 16787656;
pub const KEY_braille_dots_48 = 16787592;
pub const KEY_braille_dots_5 = 16787472;
pub const KEY_braille_dots_56 = 16787504;
pub const KEY_braille_dots_567 = 16787568;
pub const KEY_braille_dots_5678 = 16787696;
pub const KEY_braille_dots_568 = 16787632;
pub const KEY_braille_dots_57 = 16787536;
pub const KEY_braille_dots_578 = 16787664;
pub const KEY_braille_dots_58 = 16787600;
pub const KEY_braille_dots_6 = 16787488;
pub const KEY_braille_dots_67 = 16787552;
pub const KEY_braille_dots_678 = 16787680;
pub const KEY_braille_dots_68 = 16787616;
pub const KEY_braille_dots_7 = 16787520;
pub const KEY_braille_dots_78 = 16787648;
pub const KEY_braille_dots_8 = 16787584;
pub const KEY_breve = 418;
pub const KEY_brokenbar = 166;
pub const KEY_c = 99;
pub const KEY_c_h = 65187;
pub const KEY_cabovedot = 741;
pub const KEY_cacute = 486;
pub const KEY_careof = 2744;
pub const KEY_caret = 2812;
pub const KEY_caron = 439;
pub const KEY_ccaron = 488;
pub const KEY_ccedilla = 231;
pub const KEY_ccircumflex = 742;
pub const KEY_cedilla = 184;
pub const KEY_cent = 162;
pub const KEY_ch = 65184;
pub const KEY_checkerboard = 2529;
pub const KEY_checkmark = 2803;
pub const KEY_circle = 3023;
pub const KEY_club = 2796;
pub const KEY_colon = 58;
pub const KEY_comma = 44;
pub const KEY_containsas = 16785931;
pub const KEY_copyright = 169;
pub const KEY_cr = 2532;
pub const KEY_crossinglines = 2542;
pub const KEY_cuberoot = 16785947;
pub const KEY_currency = 164;
pub const KEY_cursor = 2815;
pub const KEY_d = 100;
pub const KEY_dabovedot = 16784907;
pub const KEY_dagger = 2801;
pub const KEY_dcaron = 495;
pub const KEY_dead_A = 65153;
pub const KEY_dead_E = 65155;
pub const KEY_dead_I = 65157;
pub const KEY_dead_O = 65159;
pub const KEY_dead_U = 65161;
pub const KEY_dead_a = 65152;
pub const KEY_dead_abovecomma = 65124;
pub const KEY_dead_abovedot = 65110;
pub const KEY_dead_abovereversedcomma = 65125;
pub const KEY_dead_abovering = 65112;
pub const KEY_dead_aboveverticalline = 65169;
pub const KEY_dead_acute = 65105;
pub const KEY_dead_belowbreve = 65131;
pub const KEY_dead_belowcircumflex = 65129;
pub const KEY_dead_belowcomma = 65134;
pub const KEY_dead_belowdiaeresis = 65132;
pub const KEY_dead_belowdot = 65120;
pub const KEY_dead_belowmacron = 65128;
pub const KEY_dead_belowring = 65127;
pub const KEY_dead_belowtilde = 65130;
pub const KEY_dead_belowverticalline = 65170;
pub const KEY_dead_breve = 65109;
pub const KEY_dead_capital_schwa = 65163;
pub const KEY_dead_caron = 65114;
pub const KEY_dead_cedilla = 65115;
pub const KEY_dead_circumflex = 65106;
pub const KEY_dead_currency = 65135;
pub const KEY_dead_dasia = 65125;
pub const KEY_dead_diaeresis = 65111;
pub const KEY_dead_doubleacute = 65113;
pub const KEY_dead_doublegrave = 65126;
pub const KEY_dead_e = 65154;
pub const KEY_dead_grave = 65104;
pub const KEY_dead_greek = 65164;
pub const KEY_dead_hook = 65121;
pub const KEY_dead_horn = 65122;
pub const KEY_dead_i = 65156;
pub const KEY_dead_invertedbreve = 65133;
pub const KEY_dead_iota = 65117;
pub const KEY_dead_longsolidusoverlay = 65171;
pub const KEY_dead_lowline = 65168;
pub const KEY_dead_macron = 65108;
pub const KEY_dead_o = 65158;
pub const KEY_dead_ogonek = 65116;
pub const KEY_dead_perispomeni = 65107;
pub const KEY_dead_psili = 65124;
pub const KEY_dead_semivoiced_sound = 65119;
pub const KEY_dead_small_schwa = 65162;
pub const KEY_dead_stroke = 65123;
pub const KEY_dead_tilde = 65107;
pub const KEY_dead_u = 65160;
pub const KEY_dead_voiced_sound = 65118;
pub const KEY_decimalpoint = 2749;
pub const KEY_degree = 176;
pub const KEY_diaeresis = 168;
pub const KEY_diamond = 2797;
pub const KEY_digitspace = 2725;
pub const KEY_dintegral = 16785964;
pub const KEY_division = 247;
pub const KEY_dollar = 36;
pub const KEY_doubbaselinedot = 2735;
pub const KEY_doubleacute = 445;
pub const KEY_doubledagger = 2802;
pub const KEY_doublelowquotemark = 2814;
pub const KEY_downarrow = 2302;
pub const KEY_downcaret = 2984;
pub const KEY_downshoe = 3030;
pub const KEY_downstile = 3012;
pub const KEY_downtack = 3010;
pub const KEY_dstroke = 496;
pub const KEY_e = 101;
pub const KEY_eabovedot = 1004;
pub const KEY_eacute = 233;
pub const KEY_ebelowdot = 16785081;
pub const KEY_ecaron = 492;
pub const KEY_ecircumflex = 234;
pub const KEY_ecircumflexacute = 16785087;
pub const KEY_ecircumflexbelowdot = 16785095;
pub const KEY_ecircumflexgrave = 16785089;
pub const KEY_ecircumflexhook = 16785091;
pub const KEY_ecircumflextilde = 16785093;
pub const KEY_ediaeresis = 235;
pub const KEY_egrave = 232;
pub const KEY_ehook = 16785083;
pub const KEY_eightsubscript = 16785544;
pub const KEY_eightsuperior = 16785528;
pub const KEY_elementof = 16785928;
pub const KEY_ellipsis = 2734;
pub const KEY_em3space = 2723;
pub const KEY_em4space = 2724;
pub const KEY_emacron = 954;
pub const KEY_emdash = 2729;
pub const KEY_emfilledcircle = 2782;
pub const KEY_emfilledrect = 2783;
pub const KEY_emopencircle = 2766;
pub const KEY_emopenrectangle = 2767;
pub const KEY_emptyset = 16785925;
pub const KEY_emspace = 2721;
pub const KEY_endash = 2730;
pub const KEY_enfilledcircbullet = 2790;
pub const KEY_enfilledsqbullet = 2791;
pub const KEY_eng = 959;
pub const KEY_enopencircbullet = 2784;
pub const KEY_enopensquarebullet = 2785;
pub const KEY_enspace = 2722;
pub const KEY_eogonek = 490;
pub const KEY_equal = 61;
pub const KEY_eth = 240;
pub const KEY_etilde = 16785085;
pub const KEY_exclam = 33;
pub const KEY_exclamdown = 161;
pub const KEY_ezh = 16777874;
pub const KEY_f = 102;
pub const KEY_fabovedot = 16784927;
pub const KEY_femalesymbol = 2808;
pub const KEY_ff = 2531;
pub const KEY_figdash = 2747;
pub const KEY_filledlefttribullet = 2780;
pub const KEY_filledrectbullet = 2779;
pub const KEY_filledrighttribullet = 2781;
pub const KEY_filledtribulletdown = 2793;
pub const KEY_filledtribulletup = 2792;
pub const KEY_fiveeighths = 2757;
pub const KEY_fivesixths = 2743;
pub const KEY_fivesubscript = 16785541;
pub const KEY_fivesuperior = 16785525;
pub const KEY_fourfifths = 2741;
pub const KEY_foursubscript = 16785540;
pub const KEY_foursuperior = 16785524;
pub const KEY_fourthroot = 16785948;
pub const KEY_function = 2294;
pub const KEY_g = 103;
pub const KEY_gabovedot = 757;
pub const KEY_gbreve = 699;
pub const KEY_gcaron = 16777703;
pub const KEY_gcedilla = 955;
pub const KEY_gcircumflex = 760;
pub const KEY_grave = 96;
pub const KEY_greater = 62;
pub const KEY_greaterthanequal = 2238;
pub const KEY_guillemotleft = 171;
pub const KEY_guillemotright = 187;
pub const KEY_h = 104;
pub const KEY_hairspace = 2728;
pub const KEY_hcircumflex = 694;
pub const KEY_heart = 2798;
pub const KEY_hebrew_aleph = 3296;
pub const KEY_hebrew_ayin = 3314;
pub const KEY_hebrew_bet = 3297;
pub const KEY_hebrew_beth = 3297;
pub const KEY_hebrew_chet = 3303;
pub const KEY_hebrew_dalet = 3299;
pub const KEY_hebrew_daleth = 3299;
pub const KEY_hebrew_doublelowline = 3295;
pub const KEY_hebrew_finalkaph = 3306;
pub const KEY_hebrew_finalmem = 3309;
pub const KEY_hebrew_finalnun = 3311;
pub const KEY_hebrew_finalpe = 3315;
pub const KEY_hebrew_finalzade = 3317;
pub const KEY_hebrew_finalzadi = 3317;
pub const KEY_hebrew_gimel = 3298;
pub const KEY_hebrew_gimmel = 3298;
pub const KEY_hebrew_he = 3300;
pub const KEY_hebrew_het = 3303;
pub const KEY_hebrew_kaph = 3307;
pub const KEY_hebrew_kuf = 3319;
pub const KEY_hebrew_lamed = 3308;
pub const KEY_hebrew_mem = 3310;
pub const KEY_hebrew_nun = 3312;
pub const KEY_hebrew_pe = 3316;
pub const KEY_hebrew_qoph = 3319;
pub const KEY_hebrew_resh = 3320;
pub const KEY_hebrew_samech = 3313;
pub const KEY_hebrew_samekh = 3313;
pub const KEY_hebrew_shin = 3321;
pub const KEY_hebrew_taf = 3322;
pub const KEY_hebrew_taw = 3322;
pub const KEY_hebrew_tet = 3304;
pub const KEY_hebrew_teth = 3304;
pub const KEY_hebrew_waw = 3301;
pub const KEY_hebrew_yod = 3305;
pub const KEY_hebrew_zade = 3318;
pub const KEY_hebrew_zadi = 3318;
pub const KEY_hebrew_zain = 3302;
pub const KEY_hebrew_zayin = 3302;
pub const KEY_hexagram = 2778;
pub const KEY_horizconnector = 2211;
pub const KEY_horizlinescan1 = 2543;
pub const KEY_horizlinescan3 = 2544;
pub const KEY_horizlinescan5 = 2545;
pub const KEY_horizlinescan7 = 2546;
pub const KEY_horizlinescan9 = 2547;
pub const KEY_hstroke = 689;
pub const KEY_ht = 2530;
pub const KEY_hyphen = 173;
pub const KEY_i = 105;
pub const KEY_iTouch = 269025120;
pub const KEY_iacute = 237;
pub const KEY_ibelowdot = 16785099;
pub const KEY_ibreve = 16777517;
pub const KEY_icircumflex = 238;
pub const KEY_identical = 2255;
pub const KEY_idiaeresis = 239;
pub const KEY_idotless = 697;
pub const KEY_ifonlyif = 2253;
pub const KEY_igrave = 236;
pub const KEY_ihook = 16785097;
pub const KEY_imacron = 1007;
pub const KEY_implies = 2254;
pub const KEY_includedin = 2266;
pub const KEY_includes = 2267;
pub const KEY_infinity = 2242;
pub const KEY_integral = 2239;
pub const KEY_intersection = 2268;
pub const KEY_iogonek = 999;
pub const KEY_itilde = 949;
pub const KEY_j = 106;
pub const KEY_jcircumflex = 700;
pub const KEY_jot = 3018;
pub const KEY_k = 107;
pub const KEY_kana_A = 1201;
pub const KEY_kana_CHI = 1217;
pub const KEY_kana_E = 1204;
pub const KEY_kana_FU = 1228;
pub const KEY_kana_HA = 1226;
pub const KEY_kana_HE = 1229;
pub const KEY_kana_HI = 1227;
pub const KEY_kana_HO = 1230;
pub const KEY_kana_HU = 1228;
pub const KEY_kana_I = 1202;
pub const KEY_kana_KA = 1206;
pub const KEY_kana_KE = 1209;
pub const KEY_kana_KI = 1207;
pub const KEY_kana_KO = 1210;
pub const KEY_kana_KU = 1208;
pub const KEY_kana_MA = 1231;
pub const KEY_kana_ME = 1234;
pub const KEY_kana_MI = 1232;
pub const KEY_kana_MO = 1235;
pub const KEY_kana_MU = 1233;
pub const KEY_kana_N = 1245;
pub const KEY_kana_NA = 1221;
pub const KEY_kana_NE = 1224;
pub const KEY_kana_NI = 1222;
pub const KEY_kana_NO = 1225;
pub const KEY_kana_NU = 1223;
pub const KEY_kana_O = 1205;
pub const KEY_kana_RA = 1239;
pub const KEY_kana_RE = 1242;
pub const KEY_kana_RI = 1240;
pub const KEY_kana_RO = 1243;
pub const KEY_kana_RU = 1241;
pub const KEY_kana_SA = 1211;
pub const KEY_kana_SE = 1214;
pub const KEY_kana_SHI = 1212;
pub const KEY_kana_SO = 1215;
pub const KEY_kana_SU = 1213;
pub const KEY_kana_TA = 1216;
pub const KEY_kana_TE = 1219;
pub const KEY_kana_TI = 1217;
pub const KEY_kana_TO = 1220;
pub const KEY_kana_TSU = 1218;
pub const KEY_kana_TU = 1218;
pub const KEY_kana_U = 1203;
pub const KEY_kana_WA = 1244;
pub const KEY_kana_WO = 1190;
pub const KEY_kana_YA = 1236;
pub const KEY_kana_YO = 1238;
pub const KEY_kana_YU = 1237;
pub const KEY_kana_a = 1191;
pub const KEY_kana_closingbracket = 1187;
pub const KEY_kana_comma = 1188;
pub const KEY_kana_conjunctive = 1189;
pub const KEY_kana_e = 1194;
pub const KEY_kana_fullstop = 1185;
pub const KEY_kana_i = 1192;
pub const KEY_kana_middledot = 1189;
pub const KEY_kana_o = 1195;
pub const KEY_kana_openingbracket = 1186;
pub const KEY_kana_switch = 65406;
pub const KEY_kana_tsu = 1199;
pub const KEY_kana_tu = 1199;
pub const KEY_kana_u = 1193;
pub const KEY_kana_ya = 1196;
pub const KEY_kana_yo = 1198;
pub const KEY_kana_yu = 1197;
pub const KEY_kappa = 930;
pub const KEY_kcedilla = 1011;
pub const KEY_kra = 930;
pub const KEY_l = 108;
pub const KEY_lacute = 485;
pub const KEY_latincross = 2777;
pub const KEY_lbelowdot = 16784951;
pub const KEY_lcaron = 437;
pub const KEY_lcedilla = 950;
pub const KEY_leftanglebracket = 2748;
pub const KEY_leftarrow = 2299;
pub const KEY_leftcaret = 2979;
pub const KEY_leftdoublequotemark = 2770;
pub const KEY_leftmiddlecurlybrace = 2223;
pub const KEY_leftopentriangle = 2764;
pub const KEY_leftpointer = 2794;
pub const KEY_leftradical = 2209;
pub const KEY_leftshoe = 3034;
pub const KEY_leftsinglequotemark = 2768;
pub const KEY_leftt = 2548;
pub const KEY_lefttack = 3036;
pub const KEY_less = 60;
pub const KEY_lessthanequal = 2236;
pub const KEY_lf = 2533;
pub const KEY_logicaland = 2270;
pub const KEY_logicalor = 2271;
pub const KEY_lowleftcorner = 2541;
pub const KEY_lowrightcorner = 2538;
pub const KEY_lstroke = 435;
pub const KEY_m = 109;
pub const KEY_mabovedot = 16784961;
pub const KEY_macron = 175;
pub const KEY_malesymbol = 2807;
pub const KEY_maltesecross = 2800;
pub const KEY_marker = 2751;
pub const KEY_masculine = 186;
pub const KEY_minus = 45;
pub const KEY_minutes = 2774;
pub const KEY_mu = 181;
pub const KEY_multiply = 215;
pub const KEY_musicalflat = 2806;
pub const KEY_musicalsharp = 2805;
pub const KEY_n = 110;
pub const KEY_nabla = 2245;
pub const KEY_nacute = 497;
pub const KEY_ncaron = 498;
pub const KEY_ncedilla = 1009;
pub const KEY_ninesubscript = 16785545;
pub const KEY_ninesuperior = 16785529;
pub const KEY_nl = 2536;
pub const KEY_nobreakspace = 160;
pub const KEY_notapproxeq = 16785991;
pub const KEY_notelementof = 16785929;
pub const KEY_notequal = 2237;
pub const KEY_notidentical = 16786018;
pub const KEY_notsign = 172;
pub const KEY_ntilde = 241;
pub const KEY_numbersign = 35;
pub const KEY_numerosign = 1712;
pub const KEY_o = 111;
pub const KEY_oacute = 243;
pub const KEY_obarred = 16777845;
pub const KEY_obelowdot = 16785101;
pub const KEY_ocaron = 16777682;
pub const KEY_ocircumflex = 244;
pub const KEY_ocircumflexacute = 16785105;
pub const KEY_ocircumflexbelowdot = 16785113;
pub const KEY_ocircumflexgrave = 16785107;
pub const KEY_ocircumflexhook = 16785109;
pub const KEY_ocircumflextilde = 16785111;
pub const KEY_odiaeresis = 246;
pub const KEY_odoubleacute = 501;
pub const KEY_oe = 5053;
pub const KEY_ogonek = 434;
pub const KEY_ograve = 242;
pub const KEY_ohook = 16785103;
pub const KEY_ohorn = 16777633;
pub const KEY_ohornacute = 16785115;
pub const KEY_ohornbelowdot = 16785123;
pub const KEY_ohorngrave = 16785117;
pub const KEY_ohornhook = 16785119;
pub const KEY_ohorntilde = 16785121;
pub const KEY_omacron = 1010;
pub const KEY_oneeighth = 2755;
pub const KEY_onefifth = 2738;
pub const KEY_onehalf = 189;
pub const KEY_onequarter = 188;
pub const KEY_onesixth = 2742;
pub const KEY_onesubscript = 16785537;
pub const KEY_onesuperior = 185;
pub const KEY_onethird = 2736;
pub const KEY_ooblique = 248;
pub const KEY_openrectbullet = 2786;
pub const KEY_openstar = 2789;
pub const KEY_opentribulletdown = 2788;
pub const KEY_opentribulletup = 2787;
pub const KEY_ordfeminine = 170;
pub const KEY_oslash = 248;
pub const KEY_otilde = 245;
pub const KEY_overbar = 3008;
pub const KEY_overline = 1150;
pub const KEY_p = 112;
pub const KEY_pabovedot = 16784983;
pub const KEY_paragraph = 182;
pub const KEY_parenleft = 40;
pub const KEY_parenright = 41;
pub const KEY_partdifferential = 16785922;
pub const KEY_partialderivative = 2287;
pub const KEY_percent = 37;
pub const KEY_period = 46;
pub const KEY_periodcentered = 183;
pub const KEY_permille = 2773;
pub const KEY_phonographcopyright = 2811;
pub const KEY_plus = 43;
pub const KEY_plusminus = 177;
pub const KEY_prescription = 2772;
pub const KEY_prolongedsound = 1200;
pub const KEY_punctspace = 2726;
pub const KEY_q = 113;
pub const KEY_quad = 3020;
pub const KEY_question = 63;
pub const KEY_questiondown = 191;
pub const KEY_quotedbl = 34;
pub const KEY_quoteleft = 96;
pub const KEY_quoteright = 39;
pub const KEY_r = 114;
pub const KEY_racute = 480;
pub const KEY_radical = 2262;
pub const KEY_rcaron = 504;
pub const KEY_rcedilla = 947;
pub const KEY_registered = 174;
pub const KEY_rightanglebracket = 2750;
pub const KEY_rightarrow = 2301;
pub const KEY_rightcaret = 2982;
pub const KEY_rightdoublequotemark = 2771;
pub const KEY_rightmiddlecurlybrace = 2224;
pub const KEY_rightmiddlesummation = 2231;
pub const KEY_rightopentriangle = 2765;
pub const KEY_rightpointer = 2795;
pub const KEY_rightshoe = 3032;
pub const KEY_rightsinglequotemark = 2769;
pub const KEY_rightt = 2549;
pub const KEY_righttack = 3068;
pub const KEY_s = 115;
pub const KEY_sabovedot = 16784993;
pub const KEY_sacute = 438;
pub const KEY_scaron = 441;
pub const KEY_scedilla = 442;
pub const KEY_schwa = 16777817;
pub const KEY_scircumflex = 766;
pub const KEY_script_switch = 65406;
pub const KEY_seconds = 2775;
pub const KEY_section = 167;
pub const KEY_semicolon = 59;
pub const KEY_semivoicedsound = 1247;
pub const KEY_seveneighths = 2758;
pub const KEY_sevensubscript = 16785543;
pub const KEY_sevensuperior = 16785527;
pub const KEY_signaturemark = 2762;
pub const KEY_signifblank = 2732;
pub const KEY_similarequal = 2249;
pub const KEY_singlelowquotemark = 2813;
pub const KEY_sixsubscript = 16785542;
pub const KEY_sixsuperior = 16785526;
pub const KEY_slash = 47;
pub const KEY_soliddiamond = 2528;
pub const KEY_space = 32;
pub const KEY_squareroot = 16785946;
pub const KEY_ssharp = 223;
pub const KEY_sterling = 163;
pub const KEY_stricteq = 16786019;
pub const KEY_t = 116;
pub const KEY_tabovedot = 16785003;
pub const KEY_tcaron = 443;
pub const KEY_tcedilla = 510;
pub const KEY_telephone = 2809;
pub const KEY_telephonerecorder = 2810;
pub const KEY_therefore = 2240;
pub const KEY_thinspace = 2727;
pub const KEY_thorn = 254;
pub const KEY_threeeighths = 2756;
pub const KEY_threefifths = 2740;
pub const KEY_threequarters = 190;
pub const KEY_threesubscript = 16785539;
pub const KEY_threesuperior = 179;
pub const KEY_tintegral = 16785965;
pub const KEY_topintegral = 2212;
pub const KEY_topleftparens = 2219;
pub const KEY_topleftradical = 2210;
pub const KEY_topleftsqbracket = 2215;
pub const KEY_topleftsummation = 2225;
pub const KEY_toprightparens = 2221;
pub const KEY_toprightsqbracket = 2217;
pub const KEY_toprightsummation = 2229;
pub const KEY_topt = 2551;
pub const KEY_topvertsummationconnector = 2227;
pub const KEY_trademark = 2761;
pub const KEY_trademarkincircle = 2763;
pub const KEY_tslash = 956;
pub const KEY_twofifths = 2739;
pub const KEY_twosubscript = 16785538;
pub const KEY_twosuperior = 178;
pub const KEY_twothirds = 2737;
pub const KEY_u = 117;
pub const KEY_uacute = 250;
pub const KEY_ubelowdot = 16785125;
pub const KEY_ubreve = 765;
pub const KEY_ucircumflex = 251;
pub const KEY_udiaeresis = 252;
pub const KEY_udoubleacute = 507;
pub const KEY_ugrave = 249;
pub const KEY_uhook = 16785127;
pub const KEY_uhorn = 16777648;
pub const KEY_uhornacute = 16785129;
pub const KEY_uhornbelowdot = 16785137;
pub const KEY_uhorngrave = 16785131;
pub const KEY_uhornhook = 16785133;
pub const KEY_uhorntilde = 16785135;
pub const KEY_umacron = 1022;
pub const KEY_underbar = 3014;
pub const KEY_underscore = 95;
pub const KEY_union = 2269;
pub const KEY_uogonek = 1017;
pub const KEY_uparrow = 2300;
pub const KEY_upcaret = 2985;
pub const KEY_upleftcorner = 2540;
pub const KEY_uprightcorner = 2539;
pub const KEY_upshoe = 3011;
pub const KEY_upstile = 3027;
pub const KEY_uptack = 3022;
pub const KEY_uring = 505;
pub const KEY_utilde = 1021;
pub const KEY_v = 118;
pub const KEY_variation = 2241;
pub const KEY_vertbar = 2552;
pub const KEY_vertconnector = 2214;
pub const KEY_voicedsound = 1246;
pub const KEY_vt = 2537;
pub const KEY_w = 119;
pub const KEY_wacute = 16785027;
pub const KEY_wcircumflex = 16777589;
pub const KEY_wdiaeresis = 16785029;
pub const KEY_wgrave = 16785025;
pub const KEY_x = 120;
pub const KEY_xabovedot = 16785035;
pub const KEY_y = 121;
pub const KEY_yacute = 253;
pub const KEY_ybelowdot = 16785141;
pub const KEY_ycircumflex = 16777591;
pub const KEY_ydiaeresis = 255;
pub const KEY_yen = 165;
pub const KEY_ygrave = 16785139;
pub const KEY_yhook = 16785143;
pub const KEY_ytilde = 16785145;
pub const KEY_z = 122;
pub const KEY_zabovedot = 447;
pub const KEY_zacute = 444;
pub const KEY_zcaron = 446;
pub const KEY_zerosubscript = 16785536;
pub const KEY_zerosuperior = 16785520;
pub const KEY_zstroke = 16777654;
pub const KP_0 = 65456;
pub const KP_1 = 65457;
pub const KP_2 = 65458;
pub const KP_3 = 65459;
pub const KP_4 = 65460;
pub const KP_5 = 65461;
pub const KP_6 = 65462;
pub const KP_7 = 65463;
pub const KP_8 = 65464;
pub const KP_9 = 65465;
pub const KP_Add = 65451;
pub const KP_Begin = 65437;
pub const KP_Decimal = 65454;
pub const KP_Delete = 65439;
pub const KP_Divide = 65455;
pub const KP_Down = 65433;
pub const KP_End = 65436;
pub const KP_Enter = 65421;
pub const KP_Equal = 65469;
pub const KP_F1 = 65425;
pub const KP_F2 = 65426;
pub const KP_F3 = 65427;
pub const KP_F4 = 65428;
pub const KP_Home = 65429;
pub const KP_Insert = 65438;
pub const KP_Left = 65430;
pub const KP_Multiply = 65450;
pub const KP_Next = 65435;
pub const KP_Page_Down = 65435;
pub const KP_Page_Up = 65434;
pub const KP_Prior = 65434;
pub const KP_Right = 65432;
pub const KP_Separator = 65452;
pub const KP_Space = 65408;
pub const KP_Subtract = 65453;
pub const KP_Tab = 65417;
pub const KP_Up = 65431;
pub const Kana_Lock = 65325;
pub const Kana_Shift = 65326;
/// Japanese keyboard support.
pub const Kanji = 65313;
pub const Kanji_Bangou = 65335;
/// Japanese keyboard support.
pub const Katakana = 65318;
pub const Kcedilla = 979;
pub const Korean_Won = 3839;
pub const L = 76;
pub const L1 = 65480;
pub const L10 = 65489;
pub const L2 = 65481;
pub const L3 = 65482;
pub const L4 = 65483;
pub const L5 = 65484;
pub const L6 = 65485;
pub const L7 = 65486;
pub const L8 = 65487;
pub const L9 = 65488;
pub const Lacute = 453;
pub const Last_Virtual_Screen = 65236;
pub const Lbelowdot = 16784950;
pub const Lcaron = 421;
pub const Lcedilla = 934;
pub const Left = 65361;
pub const Linefeed = 65290;
pub const LiraSign = 16785572;
pub const Lstroke = 419;
pub const M = 77;
/// IBus major version.
pub const MAJOR_VERSION = 1;
pub const MAX_COMPOSE_LEN = 255;
/// IBus micro version.
pub const MICRO_VERSION = 28;
/// IBus minor version.
pub const MINOR_VERSION = 5;
pub const Mabovedot = 16784960;
pub const Macedonia_DSE = 1717;
pub const Macedonia_GJE = 1714;
pub const Macedonia_KJE = 1724;
pub const Macedonia_dse = 1701;
pub const Macedonia_gje = 1698;
pub const Macedonia_kje = 1708;
pub const Mae_Koho = 65342;
pub const Massyo = 65324;
pub const Menu = 65383;
pub const Meta_L = 65511;
pub const Meta_R = 65512;
pub const MillSign = 16785573;
pub const Mode_switch = 65406;
pub const MouseKeys_Accel_Enable = 65143;
pub const MouseKeys_Enable = 65142;
/// Japanese keyboard support.
pub const Muhenkan = 65314;
/// Key for composing characters.
/// A.k.a. Compose Key.
pub const Multi_key = 65312;
pub const MultipleCandidate = 65341;
pub const N = 78;
pub const Nacute = 465;
pub const NairaSign = 16785574;
pub const Ncaron = 466;
pub const Ncedilla = 977;
pub const NewSheqelSign = 16785578;
pub const Next = 65366;
pub const Next_Virtual_Screen = 65234;
pub const Ntilde = 209;
pub const Num_Lock = 65407;
pub const O = 79;
pub const OE = 5052;
pub const Oacute = 211;
pub const Obarred = 16777631;
pub const Obelowdot = 16785100;
pub const Ocaron = 16777681;
pub const Ocircumflex = 212;
pub const Ocircumflexacute = 16785104;
pub const Ocircumflexbelowdot = 16785112;
pub const Ocircumflexgrave = 16785106;
pub const Ocircumflexhook = 16785108;
pub const Ocircumflextilde = 16785110;
pub const Odiaeresis = 214;
pub const Odoubleacute = 469;
pub const Ograve = 210;
pub const Ohook = 16785102;
pub const Ohorn = 16777632;
pub const Ohornacute = 16785114;
pub const Ohornbelowdot = 16785122;
pub const Ohorngrave = 16785116;
pub const Ohornhook = 16785118;
pub const Ohorntilde = 16785120;
pub const Omacron = 978;
pub const Ooblique = 216;
pub const Oslash = 216;
pub const Otilde = 213;
pub const Overlay1_Enable = 65144;
pub const Overlay2_Enable = 65145;
pub const P = 80;
/// D-Bus path for IBus config.
pub const PATH_CONFIG = "/org/freedesktop/IBus/Config";
/// D-Bus path for IBus factory.
pub const PATH_FACTORY = "/org/freedesktop/IBus/Factory";
/// D-Bus path for IBus
pub const PATH_IBUS = "/org/freedesktop/IBus";
/// Template of D-Bus path for IBus input context.
pub const PATH_INPUT_CONTEXT = "/org/freedesktop/IBus/InputContext_%d";
/// D-Bus path for IBus notifications.
pub const PATH_NOTIFICATIONS = "/org/freedesktop/IBus/Notifications";
/// D-Bus path for IBus panel.
pub const PATH_PANEL = "/org/freedesktop/IBus/Panel";
/// D-Bus path for IBus extension panel for emoji.
/// This service provides emoji, Unicode code point, Unicode name features.
pub const PATH_PANEL_EXTENSION_EMOJI = "/org/freedesktop/IBus/Panel/Extension/Emoji";
pub const Pabovedot = 16784982;
pub const Page_Down = 65366;
pub const Page_Up = 65365;
pub const Pause = 65299;
pub const PesetaSign = 16785575;
pub const Pointer_Accelerate = 65274;
pub const Pointer_Button1 = 65257;
pub const Pointer_Button2 = 65258;
pub const Pointer_Button3 = 65259;
pub const Pointer_Button4 = 65260;
pub const Pointer_Button5 = 65261;
pub const Pointer_Button_Dflt = 65256;
pub const Pointer_DblClick1 = 65263;
pub const Pointer_DblClick2 = 65264;
pub const Pointer_DblClick3 = 65265;
pub const Pointer_DblClick4 = 65266;
pub const Pointer_DblClick5 = 65267;
pub const Pointer_DblClick_Dflt = 65262;
pub const Pointer_DfltBtnNext = 65275;
pub const Pointer_DfltBtnPrev = 65276;
pub const Pointer_Down = 65251;
pub const Pointer_DownLeft = 65254;
pub const Pointer_DownRight = 65255;
pub const Pointer_Drag1 = 65269;
pub const Pointer_Drag2 = 65270;
pub const Pointer_Drag3 = 65271;
pub const Pointer_Drag4 = 65272;
pub const Pointer_Drag5 = 65277;
pub const Pointer_Drag_Dflt = 65268;
pub const Pointer_EnableKeys = 65273;
pub const Pointer_Left = 65248;
pub const Pointer_Right = 65249;
pub const Pointer_Up = 65250;
pub const Pointer_UpLeft = 65252;
pub const Pointer_UpRight = 65253;
pub const Prev_Virtual_Screen = 65233;
pub const PreviousCandidate = 65342;
pub const Print = 65377;
pub const Prior = 65365;
pub const Q = 81;
pub const R = 82;
pub const R1 = 65490;
pub const R10 = 65499;
pub const R11 = 65500;
pub const R12 = 65501;
pub const R13 = 65502;
pub const R14 = 65503;
pub const R15 = 65504;
pub const R2 = 65491;
pub const R3 = 65492;
pub const R4 = 65493;
pub const R5 = 65494;
pub const R6 = 65495;
pub const R7 = 65496;
pub const R8 = 65497;
pub const R9 = 65498;
pub const Racute = 448;
pub const Rcaron = 472;
pub const Rcedilla = 931;
pub const Redo = 65382;
pub const RepeatKeys_Enable = 65138;
pub const Return = 65293;
pub const Right = 65363;
/// Japanese keyboard support.
pub const Romaji = 65316;
pub const RupeeSign = 16785576;
pub const S = 83;
pub const SCHWA = 16777615;
/// Address of IBus config service.
pub const SERVICE_CONFIG = "org.freedesktop.IBus.Config";
/// Address of IBus service.
pub const SERVICE_IBUS = "org.freedesktop.IBus";
/// Address of IBus notification service.
pub const SERVICE_NOTIFICATIONS = "org.freedesktop.IBus.Notifications";
/// Address of IBus panel service.
pub const SERVICE_PANEL = "org.freedesktop.IBus.Panel";
/// Address of IBus panel extension service.
pub const SERVICE_PANEL_EXTENSION = "org.freedesktop.IBus.Panel.Extension";
/// Address of IBus panel extension service for emoji.
/// This service provides emoji, Unicode code point, Unicode name features.
pub const SERVICE_PANEL_EXTENSION_EMOJI = "org.freedesktop.IBus.Panel.Extension.Emoji";
/// Address of IBus portalservice.
pub const SERVICE_PORTAL = "org.freedesktop.portal.IBus";
pub const Sabovedot = 16784992;
pub const Sacute = 422;
pub const Scaron = 425;
pub const Scedilla = 426;
pub const Scircumflex = 734;
pub const Scroll_Lock = 65300;
pub const Select = 65376;
pub const Serbian_DJE = 1713;
pub const Serbian_DZE = 1727;
pub const Serbian_JE = 1720;
pub const Serbian_LJE = 1721;
pub const Serbian_NJE = 1722;
pub const Serbian_TSHE = 1723;
pub const Serbian_dje = 1697;
pub const Serbian_dze = 1711;
pub const Serbian_je = 1704;
pub const Serbian_lje = 1705;
pub const Serbian_nje = 1706;
pub const Serbian_tshe = 1707;
pub const Shift_L = 65505;
pub const Shift_Lock = 65510;
pub const Shift_R = 65506;
pub const SingleCandidate = 65340;
pub const SlowKeys_Enable = 65139;
pub const StickyKeys_Enable = 65141;
pub const Super_L = 65515;
pub const Super_R = 65516;
pub const Sys_Req = 65301;
pub const T = 84;
pub const THORN = 222;
pub const Tab = 65289;
pub const Tabovedot = 16785002;
pub const Tcaron = 427;
pub const Tcedilla = 478;
pub const Terminate_Server = 65237;
pub const Thai_baht = 3551;
pub const Thai_bobaimai = 3514;
pub const Thai_chochan = 3496;
pub const Thai_chochang = 3498;
pub const Thai_choching = 3497;
pub const Thai_chochoe = 3500;
pub const Thai_dochada = 3502;
pub const Thai_dodek = 3508;
pub const Thai_fofa = 3517;
pub const Thai_fofan = 3519;
pub const Thai_hohip = 3531;
pub const Thai_honokhuk = 3534;
pub const Thai_khokhai = 3490;
pub const Thai_khokhon = 3493;
pub const Thai_khokhuat = 3491;
pub const Thai_khokhwai = 3492;
pub const Thai_khorakhang = 3494;
pub const Thai_kokai = 3489;
pub const Thai_lakkhangyao = 3557;
pub const Thai_lekchet = 3575;
pub const Thai_lekha = 3573;
pub const Thai_lekhok = 3574;
pub const Thai_lekkao = 3577;
pub const Thai_leknung = 3569;
pub const Thai_lekpaet = 3576;
pub const Thai_leksam = 3571;
pub const Thai_leksi = 3572;
pub const Thai_leksong = 3570;
pub const Thai_leksun = 3568;
pub const Thai_lochula = 3532;
pub const Thai_loling = 3525;
pub const Thai_lu = 3526;
pub const Thai_maichattawa = 3563;
pub const Thai_maiek = 3560;
pub const Thai_maihanakat = 3537;
pub const Thai_maihanakat_maitho = 3550;
pub const Thai_maitaikhu = 3559;
pub const Thai_maitho = 3561;
pub const Thai_maitri = 3562;
pub const Thai_maiyamok = 3558;
pub const Thai_moma = 3521;
pub const Thai_ngongu = 3495;
pub const Thai_nikhahit = 3565;
pub const Thai_nonen = 3507;
pub const Thai_nonu = 3513;
pub const Thai_oang = 3533;
pub const Thai_paiyannoi = 3535;
pub const Thai_phinthu = 3546;
pub const Thai_phophan = 3518;
pub const Thai_phophung = 3516;
pub const Thai_phosamphao = 3520;
pub const Thai_popla = 3515;
pub const Thai_rorua = 3523;
pub const Thai_ru = 3524;
pub const Thai_saraa = 3536;
pub const Thai_saraaa = 3538;
pub const Thai_saraae = 3553;
pub const Thai_saraaimaimalai = 3556;
pub const Thai_saraaimaimuan = 3555;
pub const Thai_saraam = 3539;
pub const Thai_sarae = 3552;
pub const Thai_sarai = 3540;
pub const Thai_saraii = 3541;
pub const Thai_sarao = 3554;
pub const Thai_sarau = 3544;
pub const Thai_saraue = 3542;
pub const Thai_sarauee = 3543;
pub const Thai_sarauu = 3545;
pub const Thai_sorusi = 3529;
pub const Thai_sosala = 3528;
pub const Thai_soso = 3499;
pub const Thai_sosua = 3530;
pub const Thai_thanthakhat = 3564;
pub const Thai_thonangmontho = 3505;
pub const Thai_thophuthao = 3506;
pub const Thai_thothahan = 3511;
pub const Thai_thothan = 3504;
pub const Thai_thothong = 3512;
pub const Thai_thothung = 3510;
pub const Thai_topatak = 3503;
pub const Thai_totao = 3509;
pub const Thai_wowaen = 3527;
pub const Thai_yoyak = 3522;
pub const Thai_yoying = 3501;
pub const Thorn = 222;
pub const Touroku = 65323;
pub const Tslash = 940;
pub const U = 85;
pub const Uacute = 218;
pub const Ubelowdot = 16785124;
pub const Ubreve = 733;
pub const Ucircumflex = 219;
pub const Udiaeresis = 220;
pub const Udoubleacute = 475;
pub const Ugrave = 217;
pub const Uhook = 16785126;
pub const Uhorn = 16777647;
pub const Uhornacute = 16785128;
pub const Uhornbelowdot = 16785136;
pub const Uhorngrave = 16785130;
pub const Uhornhook = 16785132;
pub const Uhorntilde = 16785134;
pub const Ukrainian_GHE_WITH_UPTURN = 1725;
pub const Ukrainian_I = 1718;
pub const Ukrainian_IE = 1716;
pub const Ukrainian_YI = 1719;
pub const Ukrainian_ghe_with_upturn = 1709;
pub const Ukrainian_i = 1702;
pub const Ukrainian_ie = 1700;
pub const Ukrainian_yi = 1703;
pub const Ukranian_I = 1718;
pub const Ukranian_JE = 1716;
pub const Ukranian_YI = 1719;
pub const Ukranian_i = 1702;
pub const Ukranian_je = 1700;
pub const Ukranian_yi = 1703;
pub const Umacron = 990;
pub const Undo = 65381;
pub const Uogonek = 985;
pub const Up = 65362;
pub const Uring = 473;
pub const Utilde = 989;
pub const V = 86;
pub const VoidSymbol = 16777215;
pub const W = 87;
pub const Wacute = 16785026;
pub const Wcircumflex = 16777588;
pub const Wdiaeresis = 16785028;
pub const Wgrave = 16785024;
pub const WonSign = 16785577;
pub const X = 88;
pub const Xabovedot = 16785034;
pub const Y = 89;
pub const Yacute = 221;
pub const Ybelowdot = 16785140;
pub const Ycircumflex = 16777590;
pub const Ydiaeresis = 5054;
pub const Ygrave = 16785138;
pub const Yhook = 16785142;
pub const Ytilde = 16785144;
pub const Z = 90;
pub const Zabovedot = 431;
pub const Zacute = 428;
pub const Zcaron = 430;
pub const Zen_Koho = 65341;
pub const Zenkaku = 65320;
pub const Zenkaku_Hankaku = 65322;
pub const Zstroke = 16777653;
pub const a = 97;
pub const aacute = 225;
pub const abelowdot = 16785057;
pub const abovedot = 511;
pub const abreve = 483;
pub const abreveacute = 16785071;
pub const abrevebelowdot = 16785079;
pub const abrevegrave = 16785073;
pub const abrevehook = 16785075;
pub const abrevetilde = 16785077;
pub const acircumflex = 226;
pub const acircumflexacute = 16785061;
pub const acircumflexbelowdot = 16785069;
pub const acircumflexgrave = 16785063;
pub const acircumflexhook = 16785065;
pub const acircumflextilde = 16785067;
pub const acute = 180;
pub const adiaeresis = 228;
pub const ae = 230;
pub const agrave = 224;
pub const ahook = 16785059;
pub const amacron = 992;
pub const ampersand = 38;
pub const aogonek = 433;
pub const apostrophe = 39;
pub const approxeq = 16785992;
pub const approximate = 2248;
pub const aring = 229;
pub const asciicircum = 94;
pub const asciitilde = 126;
pub const asterisk = 42;
pub const at = 64;
pub const atilde = 227;
pub const b = 98;
pub const babovedot = 16784899;
pub const backslash = 92;
pub const ballotcross = 2804;
pub const bar = 124;
pub const because = 16785973;
pub const blank = 2527;
pub const botintegral = 2213;
pub const botleftparens = 2220;
pub const botleftsqbracket = 2216;
pub const botleftsummation = 2226;
pub const botrightparens = 2222;
pub const botrightsqbracket = 2218;
pub const botrightsummation = 2230;
pub const bott = 2550;
pub const botvertsummationconnector = 2228;
pub const braceleft = 123;
pub const braceright = 125;
pub const bracketleft = 91;
pub const bracketright = 93;
pub const braille_blank = 16787456;
pub const braille_dot_1 = 65521;
pub const braille_dot_10 = 65530;
pub const braille_dot_2 = 65522;
pub const braille_dot_3 = 65523;
pub const braille_dot_4 = 65524;
pub const braille_dot_5 = 65525;
pub const braille_dot_6 = 65526;
pub const braille_dot_7 = 65527;
pub const braille_dot_8 = 65528;
pub const braille_dot_9 = 65529;
pub const braille_dots_1 = 16787457;
pub const braille_dots_12 = 16787459;
pub const braille_dots_123 = 16787463;
pub const braille_dots_1234 = 16787471;
pub const braille_dots_12345 = 16787487;
pub const braille_dots_123456 = 16787519;
pub const braille_dots_1234567 = 16787583;
pub const braille_dots_12345678 = 16787711;
pub const braille_dots_1234568 = 16787647;
pub const braille_dots_123457 = 16787551;
pub const braille_dots_1234578 = 16787679;
pub const braille_dots_123458 = 16787615;
pub const braille_dots_12346 = 16787503;
pub const braille_dots_123467 = 16787567;
pub const braille_dots_1234678 = 16787695;
pub const braille_dots_123468 = 16787631;
pub const braille_dots_12347 = 16787535;
pub const braille_dots_123478 = 16787663;
pub const braille_dots_12348 = 16787599;
pub const braille_dots_1235 = 16787479;
pub const braille_dots_12356 = 16787511;
pub const braille_dots_123567 = 16787575;
pub const braille_dots_1235678 = 16787703;
pub const braille_dots_123568 = 16787639;
pub const braille_dots_12357 = 16787543;
pub const braille_dots_123578 = 16787671;
pub const braille_dots_12358 = 16787607;
pub const braille_dots_1236 = 16787495;
pub const braille_dots_12367 = 16787559;
pub const braille_dots_123678 = 16787687;
pub const braille_dots_12368 = 16787623;
pub const braille_dots_1237 = 16787527;
pub const braille_dots_12378 = 16787655;
pub const braille_dots_1238 = 16787591;
pub const braille_dots_124 = 16787467;
pub const braille_dots_1245 = 16787483;
pub const braille_dots_12456 = 16787515;
pub const braille_dots_124567 = 16787579;
pub const braille_dots_1245678 = 16787707;
pub const braille_dots_124568 = 16787643;
pub const braille_dots_12457 = 16787547;
pub const braille_dots_124578 = 16787675;
pub const braille_dots_12458 = 16787611;
pub const braille_dots_1246 = 16787499;
pub const braille_dots_12467 = 16787563;
pub const braille_dots_124678 = 16787691;
pub const braille_dots_12468 = 16787627;
pub const braille_dots_1247 = 16787531;
pub const braille_dots_12478 = 16787659;
pub const braille_dots_1248 = 16787595;
pub const braille_dots_125 = 16787475;
pub const braille_dots_1256 = 16787507;
pub const braille_dots_12567 = 16787571;
pub const braille_dots_125678 = 16787699;
pub const braille_dots_12568 = 16787635;
pub const braille_dots_1257 = 16787539;
pub const braille_dots_12578 = 16787667;
pub const braille_dots_1258 = 16787603;
pub const braille_dots_126 = 16787491;
pub const braille_dots_1267 = 16787555;
pub const braille_dots_12678 = 16787683;
pub const braille_dots_1268 = 16787619;
pub const braille_dots_127 = 16787523;
pub const braille_dots_1278 = 16787651;
pub const braille_dots_128 = 16787587;
pub const braille_dots_13 = 16787461;
pub const braille_dots_134 = 16787469;
pub const braille_dots_1345 = 16787485;
pub const braille_dots_13456 = 16787517;
pub const braille_dots_134567 = 16787581;
pub const braille_dots_1345678 = 16787709;
pub const braille_dots_134568 = 16787645;
pub const braille_dots_13457 = 16787549;
pub const braille_dots_134578 = 16787677;
pub const braille_dots_13458 = 16787613;
pub const braille_dots_1346 = 16787501;
pub const braille_dots_13467 = 16787565;
pub const braille_dots_134678 = 16787693;
pub const braille_dots_13468 = 16787629;
pub const braille_dots_1347 = 16787533;
pub const braille_dots_13478 = 16787661;
pub const braille_dots_1348 = 16787597;
pub const braille_dots_135 = 16787477;
pub const braille_dots_1356 = 16787509;
pub const braille_dots_13567 = 16787573;
pub const braille_dots_135678 = 16787701;
pub const braille_dots_13568 = 16787637;
pub const braille_dots_1357 = 16787541;
pub const braille_dots_13578 = 16787669;
pub const braille_dots_1358 = 16787605;
pub const braille_dots_136 = 16787493;
pub const braille_dots_1367 = 16787557;
pub const braille_dots_13678 = 16787685;
pub const braille_dots_1368 = 16787621;
pub const braille_dots_137 = 16787525;
pub const braille_dots_1378 = 16787653;
pub const braille_dots_138 = 16787589;
pub const braille_dots_14 = 16787465;
pub const braille_dots_145 = 16787481;
pub const braille_dots_1456 = 16787513;
pub const braille_dots_14567 = 16787577;
pub const braille_dots_145678 = 16787705;
pub const braille_dots_14568 = 16787641;
pub const braille_dots_1457 = 16787545;
pub const braille_dots_14578 = 16787673;
pub const braille_dots_1458 = 16787609;
pub const braille_dots_146 = 16787497;
pub const braille_dots_1467 = 16787561;
pub const braille_dots_14678 = 16787689;
pub const braille_dots_1468 = 16787625;
pub const braille_dots_147 = 16787529;
pub const braille_dots_1478 = 16787657;
pub const braille_dots_148 = 16787593;
pub const braille_dots_15 = 16787473;
pub const braille_dots_156 = 16787505;
pub const braille_dots_1567 = 16787569;
pub const braille_dots_15678 = 16787697;
pub const braille_dots_1568 = 16787633;
pub const braille_dots_157 = 16787537;
pub const braille_dots_1578 = 16787665;
pub const braille_dots_158 = 16787601;
pub const braille_dots_16 = 16787489;
pub const braille_dots_167 = 16787553;
pub const braille_dots_1678 = 16787681;
pub const braille_dots_168 = 16787617;
pub const braille_dots_17 = 16787521;
pub const braille_dots_178 = 16787649;
pub const braille_dots_18 = 16787585;
pub const braille_dots_2 = 16787458;
pub const braille_dots_23 = 16787462;
pub const braille_dots_234 = 16787470;
pub const braille_dots_2345 = 16787486;
pub const braille_dots_23456 = 16787518;
pub const braille_dots_234567 = 16787582;
pub const braille_dots_2345678 = 16787710;
pub const braille_dots_234568 = 16787646;
pub const braille_dots_23457 = 16787550;
pub const braille_dots_234578 = 16787678;
pub const braille_dots_23458 = 16787614;
pub const braille_dots_2346 = 16787502;
pub const braille_dots_23467 = 16787566;
pub const braille_dots_234678 = 16787694;
pub const braille_dots_23468 = 16787630;
pub const braille_dots_2347 = 16787534;
pub const braille_dots_23478 = 16787662;
pub const braille_dots_2348 = 16787598;
pub const braille_dots_235 = 16787478;
pub const braille_dots_2356 = 16787510;
pub const braille_dots_23567 = 16787574;
pub const braille_dots_235678 = 16787702;
pub const braille_dots_23568 = 16787638;
pub const braille_dots_2357 = 16787542;
pub const braille_dots_23578 = 16787670;
pub const braille_dots_2358 = 16787606;
pub const braille_dots_236 = 16787494;
pub const braille_dots_2367 = 16787558;
pub const braille_dots_23678 = 16787686;
pub const braille_dots_2368 = 16787622;
pub const braille_dots_237 = 16787526;
pub const braille_dots_2378 = 16787654;
pub const braille_dots_238 = 16787590;
pub const braille_dots_24 = 16787466;
pub const braille_dots_245 = 16787482;
pub const braille_dots_2456 = 16787514;
pub const braille_dots_24567 = 16787578;
pub const braille_dots_245678 = 16787706;
pub const braille_dots_24568 = 16787642;
pub const braille_dots_2457 = 16787546;
pub const braille_dots_24578 = 16787674;
pub const braille_dots_2458 = 16787610;
pub const braille_dots_246 = 16787498;
pub const braille_dots_2467 = 16787562;
pub const braille_dots_24678 = 16787690;
pub const braille_dots_2468 = 16787626;
pub const braille_dots_247 = 16787530;
pub const braille_dots_2478 = 16787658;
pub const braille_dots_248 = 16787594;
pub const braille_dots_25 = 16787474;
pub const braille_dots_256 = 16787506;
pub const braille_dots_2567 = 16787570;
pub const braille_dots_25678 = 16787698;
pub const braille_dots_2568 = 16787634;
pub const braille_dots_257 = 16787538;
pub const braille_dots_2578 = 16787666;
pub const braille_dots_258 = 16787602;
pub const braille_dots_26 = 16787490;
pub const braille_dots_267 = 16787554;
pub const braille_dots_2678 = 16787682;
pub const braille_dots_268 = 16787618;
pub const braille_dots_27 = 16787522;
pub const braille_dots_278 = 16787650;
pub const braille_dots_28 = 16787586;
pub const braille_dots_3 = 16787460;
pub const braille_dots_34 = 16787468;
pub const braille_dots_345 = 16787484;
pub const braille_dots_3456 = 16787516;
pub const braille_dots_34567 = 16787580;
pub const braille_dots_345678 = 16787708;
pub const braille_dots_34568 = 16787644;
pub const braille_dots_3457 = 16787548;
pub const braille_dots_34578 = 16787676;
pub const braille_dots_3458 = 16787612;
pub const braille_dots_346 = 16787500;
pub const braille_dots_3467 = 16787564;
pub const braille_dots_34678 = 16787692;
pub const braille_dots_3468 = 16787628;
pub const braille_dots_347 = 16787532;
pub const braille_dots_3478 = 16787660;
pub const braille_dots_348 = 16787596;
pub const braille_dots_35 = 16787476;
pub const braille_dots_356 = 16787508;
pub const braille_dots_3567 = 16787572;
pub const braille_dots_35678 = 16787700;
pub const braille_dots_3568 = 16787636;
pub const braille_dots_357 = 16787540;
pub const braille_dots_3578 = 16787668;
pub const braille_dots_358 = 16787604;
pub const braille_dots_36 = 16787492;
pub const braille_dots_367 = 16787556;
pub const braille_dots_3678 = 16787684;
pub const braille_dots_368 = 16787620;
pub const braille_dots_37 = 16787524;
pub const braille_dots_378 = 16787652;
pub const braille_dots_38 = 16787588;
pub const braille_dots_4 = 16787464;
pub const braille_dots_45 = 16787480;
pub const braille_dots_456 = 16787512;
pub const braille_dots_4567 = 16787576;
pub const braille_dots_45678 = 16787704;
pub const braille_dots_4568 = 16787640;
pub const braille_dots_457 = 16787544;
pub const braille_dots_4578 = 16787672;
pub const braille_dots_458 = 16787608;
pub const braille_dots_46 = 16787496;
pub const braille_dots_467 = 16787560;
pub const braille_dots_4678 = 16787688;
pub const braille_dots_468 = 16787624;
pub const braille_dots_47 = 16787528;
pub const braille_dots_478 = 16787656;
pub const braille_dots_48 = 16787592;
pub const braille_dots_5 = 16787472;
pub const braille_dots_56 = 16787504;
pub const braille_dots_567 = 16787568;
pub const braille_dots_5678 = 16787696;
pub const braille_dots_568 = 16787632;
pub const braille_dots_57 = 16787536;
pub const braille_dots_578 = 16787664;
pub const braille_dots_58 = 16787600;
pub const braille_dots_6 = 16787488;
pub const braille_dots_67 = 16787552;
pub const braille_dots_678 = 16787680;
pub const braille_dots_68 = 16787616;
pub const braille_dots_7 = 16787520;
pub const braille_dots_78 = 16787648;
pub const braille_dots_8 = 16787584;
pub const breve = 418;
pub const brokenbar = 166;
pub const c = 99;
pub const cabovedot = 741;
pub const cacute = 486;
pub const careof = 2744;
pub const caret = 2812;
pub const caron = 439;
pub const ccaron = 488;
pub const ccedilla = 231;
pub const ccircumflex = 742;
pub const cedilla = 184;
pub const cent = 162;
pub const checkerboard = 2529;
pub const checkmark = 2803;
pub const circle = 3023;
pub const club = 2796;
pub const colon = 58;
pub const comma = 44;
pub const containsas = 16785931;
pub const copyright = 169;
pub const cr = 2532;
pub const crossinglines = 2542;
pub const cuberoot = 16785947;
pub const currency = 164;
pub const cursor = 2815;
pub const d = 100;
pub const dabovedot = 16784907;
pub const dagger = 2801;
pub const dcaron = 495;
pub const dead_abovecomma = 65124;
pub const dead_abovedot = 65110;
pub const dead_abovereversedcomma = 65125;
pub const dead_abovering = 65112;
pub const dead_acute = 65105;
pub const dead_belowbreve = 65131;
pub const dead_belowcircumflex = 65129;
pub const dead_belowdiaeresis = 65132;
pub const dead_belowdot = 65120;
pub const dead_belowmacron = 65128;
pub const dead_belowring = 65127;
pub const dead_belowtilde = 65130;
pub const dead_breve = 65109;
pub const dead_caron = 65114;
pub const dead_cedilla = 65115;
pub const dead_circumflex = 65106;
pub const dead_dasia = 65125;
pub const dead_diaeresis = 65111;
pub const dead_doubleacute = 65113;
pub const dead_grave = 65104;
pub const dead_hook = 65121;
pub const dead_horn = 65122;
pub const dead_iota = 65117;
pub const dead_macron = 65108;
pub const dead_ogonek = 65116;
pub const dead_perispomeni = 65107;
pub const dead_psili = 65124;
pub const dead_semivoiced_sound = 65119;
pub const dead_stroke = 65123;
pub const dead_tilde = 65107;
pub const dead_voiced_sound = 65118;
pub const decimalpoint = 2749;
pub const degree = 176;
pub const diaeresis = 168;
pub const diamond = 2797;
pub const digitspace = 2725;
pub const dintegral = 16785964;
pub const division = 247;
pub const dollar = 36;
pub const doubbaselinedot = 2735;
pub const doubleacute = 445;
pub const doubledagger = 2802;
pub const doublelowquotemark = 2814;
pub const downarrow = 2302;
pub const downcaret = 2984;
pub const downshoe = 3030;
pub const downstile = 3012;
pub const downtack = 3010;
pub const dstroke = 496;
pub const e = 101;
pub const eabovedot = 1004;
pub const eacute = 233;
pub const ebelowdot = 16785081;
pub const ecaron = 492;
pub const ecircumflex = 234;
pub const ecircumflexacute = 16785087;
pub const ecircumflexbelowdot = 16785095;
pub const ecircumflexgrave = 16785089;
pub const ecircumflexhook = 16785091;
pub const ecircumflextilde = 16785093;
pub const ediaeresis = 235;
pub const egrave = 232;
pub const ehook = 16785083;
pub const eightsubscript = 16785544;
pub const eightsuperior = 16785528;
pub const elementof = 16785928;
pub const ellipsis = 2734;
pub const em3space = 2723;
pub const em4space = 2724;
pub const emacron = 954;
pub const emdash = 2729;
pub const emfilledcircle = 2782;
pub const emfilledrect = 2783;
pub const emopencircle = 2766;
pub const emopenrectangle = 2767;
pub const emptyset = 16785925;
pub const emspace = 2721;
pub const endash = 2730;
pub const enfilledcircbullet = 2790;
pub const enfilledsqbullet = 2791;
pub const eng = 959;
pub const enopencircbullet = 2784;
pub const enopensquarebullet = 2785;
pub const enspace = 2722;
pub const eogonek = 490;
pub const equal = 61;
pub const eth = 240;
pub const etilde = 16785085;
pub const exclam = 33;
pub const exclamdown = 161;
pub const f = 102;
pub const fabovedot = 16784927;
pub const femalesymbol = 2808;
pub const ff = 2531;
pub const figdash = 2747;
pub const filledlefttribullet = 2780;
pub const filledrectbullet = 2779;
pub const filledrighttribullet = 2781;
pub const filledtribulletdown = 2793;
pub const filledtribulletup = 2792;
pub const fiveeighths = 2757;
pub const fivesixths = 2743;
pub const fivesubscript = 16785541;
pub const fivesuperior = 16785525;
pub const fourfifths = 2741;
pub const foursubscript = 16785540;
pub const foursuperior = 16785524;
pub const fourthroot = 16785948;
pub const function = 2294;
pub const g = 103;
pub const gabovedot = 757;
pub const gbreve = 699;
pub const gcaron = 16777703;
pub const gcedilla = 955;
pub const gcircumflex = 760;
pub const grave = 96;
pub const greater = 62;
pub const greaterthanequal = 2238;
pub const guillemotleft = 171;
pub const guillemotright = 187;
pub const h = 104;
pub const hairspace = 2728;
pub const hcircumflex = 694;
pub const heart = 2798;
pub const hebrew_aleph = 3296;
pub const hebrew_ayin = 3314;
pub const hebrew_bet = 3297;
pub const hebrew_beth = 3297;
pub const hebrew_chet = 3303;
pub const hebrew_dalet = 3299;
pub const hebrew_daleth = 3299;
pub const hebrew_doublelowline = 3295;
pub const hebrew_finalkaph = 3306;
pub const hebrew_finalmem = 3309;
pub const hebrew_finalnun = 3311;
pub const hebrew_finalpe = 3315;
pub const hebrew_finalzade = 3317;
pub const hebrew_finalzadi = 3317;
pub const hebrew_gimel = 3298;
pub const hebrew_gimmel = 3298;
pub const hebrew_he = 3300;
pub const hebrew_het = 3303;
pub const hebrew_kaph = 3307;
pub const hebrew_kuf = 3319;
pub const hebrew_lamed = 3308;
pub const hebrew_mem = 3310;
pub const hebrew_nun = 3312;
pub const hebrew_pe = 3316;
pub const hebrew_qoph = 3319;
pub const hebrew_resh = 3320;
pub const hebrew_samech = 3313;
pub const hebrew_samekh = 3313;
pub const hebrew_shin = 3321;
pub const hebrew_taf = 3322;
pub const hebrew_taw = 3322;
pub const hebrew_tet = 3304;
pub const hebrew_teth = 3304;
pub const hebrew_waw = 3301;
pub const hebrew_yod = 3305;
pub const hebrew_zade = 3318;
pub const hebrew_zadi = 3318;
pub const hebrew_zain = 3302;
pub const hebrew_zayin = 3302;
pub const hexagram = 2778;
pub const horizconnector = 2211;
pub const horizlinescan1 = 2543;
pub const horizlinescan3 = 2544;
pub const horizlinescan5 = 2545;
pub const horizlinescan7 = 2546;
pub const horizlinescan9 = 2547;
pub const hstroke = 689;
pub const ht = 2530;
pub const hyphen = 173;
pub const i = 105;
pub const iacute = 237;
pub const ibelowdot = 16785099;
pub const ibreve = 16777517;
pub const icircumflex = 238;
pub const identical = 2255;
pub const idiaeresis = 239;
pub const idotless = 697;
pub const ifonlyif = 2253;
pub const igrave = 236;
pub const ihook = 16785097;
pub const imacron = 1007;
pub const implies = 2254;
pub const includedin = 2266;
pub const includes = 2267;
pub const infinity = 2242;
pub const integral = 2239;
pub const intersection = 2268;
pub const iogonek = 999;
pub const itilde = 949;
pub const j = 106;
pub const jcircumflex = 700;
pub const jot = 3018;
pub const k = 107;
pub const kana_A = 1201;
pub const kana_CHI = 1217;
pub const kana_E = 1204;
pub const kana_FU = 1228;
pub const kana_HA = 1226;
pub const kana_HE = 1229;
pub const kana_HI = 1227;
pub const kana_HO = 1230;
pub const kana_HU = 1228;
pub const kana_I = 1202;
pub const kana_KA = 1206;
pub const kana_KE = 1209;
pub const kana_KI = 1207;
pub const kana_KO = 1210;
pub const kana_KU = 1208;
pub const kana_MA = 1231;
pub const kana_ME = 1234;
pub const kana_MI = 1232;
pub const kana_MO = 1235;
pub const kana_MU = 1233;
pub const kana_N = 1245;
pub const kana_NA = 1221;
pub const kana_NE = 1224;
pub const kana_NI = 1222;
pub const kana_NO = 1225;
pub const kana_NU = 1223;
pub const kana_O = 1205;
pub const kana_RA = 1239;
pub const kana_RE = 1242;
pub const kana_RI = 1240;
pub const kana_RO = 1243;
pub const kana_RU = 1241;
pub const kana_SA = 1211;
pub const kana_SE = 1214;
pub const kana_SHI = 1212;
pub const kana_SO = 1215;
pub const kana_SU = 1213;
pub const kana_TA = 1216;
pub const kana_TE = 1219;
pub const kana_TI = 1217;
pub const kana_TO = 1220;
pub const kana_TSU = 1218;
pub const kana_TU = 1218;
pub const kana_U = 1203;
pub const kana_WA = 1244;
pub const kana_WO = 1190;
pub const kana_YA = 1236;
pub const kana_YO = 1238;
pub const kana_YU = 1237;
pub const kana_a = 1191;
pub const kana_closingbracket = 1187;
pub const kana_comma = 1188;
pub const kana_conjunctive = 1189;
pub const kana_e = 1194;
pub const kana_fullstop = 1185;
pub const kana_i = 1192;
pub const kana_middledot = 1189;
pub const kana_o = 1195;
pub const kana_openingbracket = 1186;
pub const kana_switch = 65406;
pub const kana_tsu = 1199;
pub const kana_tu = 1199;
pub const kana_u = 1193;
pub const kana_ya = 1196;
pub const kana_yo = 1198;
pub const kana_yu = 1197;
pub const kappa = 930;
pub const kcedilla = 1011;
pub const kra = 930;
pub const l = 108;
pub const lacute = 485;
pub const latincross = 2777;
pub const lbelowdot = 16784951;
pub const lcaron = 437;
pub const lcedilla = 950;
pub const leftanglebracket = 2748;
pub const leftarrow = 2299;
pub const leftcaret = 2979;
pub const leftdoublequotemark = 2770;
pub const leftmiddlecurlybrace = 2223;
pub const leftopentriangle = 2764;
pub const leftpointer = 2794;
pub const leftradical = 2209;
pub const leftshoe = 3034;
pub const leftsinglequotemark = 2768;
pub const leftt = 2548;
pub const lefttack = 3036;
pub const less = 60;
pub const lessthanequal = 2236;
pub const lf = 2533;
pub const logicaland = 2270;
pub const logicalor = 2271;
pub const lowleftcorner = 2541;
pub const lowrightcorner = 2538;
pub const lstroke = 435;
pub const m = 109;
pub const mabovedot = 16784961;
pub const macron = 175;
pub const malesymbol = 2807;
pub const maltesecross = 2800;
pub const marker = 2751;
pub const masculine = 186;
pub const minus = 45;
pub const minutes = 2774;
pub const mu = 181;
pub const multiply = 215;
pub const musicalflat = 2806;
pub const musicalsharp = 2805;
pub const n = 110;
pub const nabla = 2245;
pub const nacute = 497;
pub const ncaron = 498;
pub const ncedilla = 1009;
pub const ninesubscript = 16785545;
pub const ninesuperior = 16785529;
pub const nl = 2536;
pub const nobreakspace = 160;
pub const notapproxeq = 16785991;
pub const notelementof = 16785929;
pub const notequal = 2237;
pub const notidentical = 16786018;
pub const notsign = 172;
pub const ntilde = 241;
pub const numbersign = 35;
pub const numerosign = 1712;
pub const o = 111;
pub const oacute = 243;
pub const obarred = 16777845;
pub const obelowdot = 16785101;
pub const ocaron = 16777682;
pub const ocircumflex = 244;
pub const ocircumflexacute = 16785105;
pub const ocircumflexbelowdot = 16785113;
pub const ocircumflexgrave = 16785107;
pub const ocircumflexhook = 16785109;
pub const ocircumflextilde = 16785111;
pub const odiaeresis = 246;
pub const odoubleacute = 501;
pub const oe = 5053;
pub const ogonek = 434;
pub const ograve = 242;
pub const ohook = 16785103;
pub const ohorn = 16777633;
pub const ohornacute = 16785115;
pub const ohornbelowdot = 16785123;
pub const ohorngrave = 16785117;
pub const ohornhook = 16785119;
pub const ohorntilde = 16785121;
pub const omacron = 1010;
pub const oneeighth = 2755;
pub const onefifth = 2738;
pub const onehalf = 189;
pub const onequarter = 188;
pub const onesixth = 2742;
pub const onesubscript = 16785537;
pub const onesuperior = 185;
pub const onethird = 2736;
pub const ooblique = 248;
pub const openrectbullet = 2786;
pub const openstar = 2789;
pub const opentribulletdown = 2788;
pub const opentribulletup = 2787;
pub const ordfeminine = 170;
pub const oslash = 248;
pub const otilde = 245;
pub const overbar = 3008;
pub const overline = 1150;
pub const p = 112;
pub const pabovedot = 16784983;
pub const paragraph = 182;
pub const parenleft = 40;
pub const parenright = 41;
pub const partdifferential = 16785922;
pub const partialderivative = 2287;
pub const percent = 37;
pub const period = 46;
pub const periodcentered = 183;
pub const phonographcopyright = 2811;
pub const plus = 43;
pub const plusminus = 177;
pub const prescription = 2772;
pub const prolongedsound = 1200;
pub const punctspace = 2726;
pub const q = 113;
pub const quad = 3020;
pub const question = 63;
pub const questiondown = 191;
pub const quotedbl = 34;
pub const quoteleft = 96;
pub const quoteright = 39;
pub const r = 114;
pub const racute = 480;
pub const radical = 2262;
pub const rcaron = 504;
pub const rcedilla = 947;
pub const registered = 174;
pub const rightanglebracket = 2750;
pub const rightarrow = 2301;
pub const rightcaret = 2982;
pub const rightdoublequotemark = 2771;
pub const rightmiddlecurlybrace = 2224;
pub const rightmiddlesummation = 2231;
pub const rightopentriangle = 2765;
pub const rightpointer = 2795;
pub const rightshoe = 3032;
pub const rightsinglequotemark = 2769;
pub const rightt = 2549;
pub const righttack = 3068;
pub const s = 115;
pub const sabovedot = 16784993;
pub const sacute = 438;
pub const scaron = 441;
pub const scedilla = 442;
pub const schwa = 16777817;
pub const scircumflex = 766;
pub const script_switch = 65406;
pub const seconds = 2775;
pub const section = 167;
pub const semicolon = 59;
pub const semivoicedsound = 1247;
pub const seveneighths = 2758;
pub const sevensubscript = 16785543;
pub const sevensuperior = 16785527;
pub const signaturemark = 2762;
pub const signifblank = 2732;
pub const similarequal = 2249;
pub const singlelowquotemark = 2813;
pub const sixsubscript = 16785542;
pub const sixsuperior = 16785526;
pub const slash = 47;
pub const soliddiamond = 2528;
pub const space = 32;
pub const squareroot = 16785946;
pub const ssharp = 223;
pub const sterling = 163;
pub const stricteq = 16786019;
pub const t = 116;
pub const tabovedot = 16785003;
pub const tcaron = 443;
pub const tcedilla = 510;
pub const telephone = 2809;
pub const telephonerecorder = 2810;
pub const therefore = 2240;
pub const thinspace = 2727;
pub const thorn = 254;
pub const threeeighths = 2756;
pub const threefifths = 2740;
pub const threequarters = 190;
pub const threesubscript = 16785539;
pub const threesuperior = 179;
pub const tintegral = 16785965;
pub const topintegral = 2212;
pub const topleftparens = 2219;
pub const topleftradical = 2210;
pub const topleftsqbracket = 2215;
pub const topleftsummation = 2225;
pub const toprightparens = 2221;
pub const toprightsqbracket = 2217;
pub const toprightsummation = 2229;
pub const topt = 2551;
pub const topvertsummationconnector = 2227;
pub const trademark = 2761;
pub const trademarkincircle = 2763;
pub const tslash = 956;
pub const twofifths = 2739;
pub const twosubscript = 16785538;
pub const twosuperior = 178;
pub const twothirds = 2737;
pub const u = 117;
pub const uacute = 250;
pub const ubelowdot = 16785125;
pub const ubreve = 765;
pub const ucircumflex = 251;
pub const udiaeresis = 252;
pub const udoubleacute = 507;
pub const ugrave = 249;
pub const uhook = 16785127;
pub const uhorn = 16777648;
pub const uhornacute = 16785129;
pub const uhornbelowdot = 16785137;
pub const uhorngrave = 16785131;
pub const uhornhook = 16785133;
pub const uhorntilde = 16785135;
pub const umacron = 1022;
pub const underbar = 3014;
pub const underscore = 95;
pub const @"union" = 2269;
pub const uogonek = 1017;
pub const uparrow = 2300;
pub const upcaret = 2985;
pub const upleftcorner = 2540;
pub const uprightcorner = 2539;
pub const upshoe = 3011;
pub const upstile = 3027;
pub const uptack = 3022;
pub const uring = 505;
pub const utilde = 1021;
pub const v = 118;
pub const variation = 2241;
pub const vertbar = 2552;
pub const vertconnector = 2214;
pub const voicedsound = 1246;
pub const vt = 2537;
pub const w = 119;
pub const wacute = 16785027;
pub const wcircumflex = 16777589;
pub const wdiaeresis = 16785029;
pub const wgrave = 16785025;
pub const x = 120;
pub const xabovedot = 16785035;
pub const y = 121;
pub const yacute = 253;
pub const ybelowdot = 16785141;
pub const ycircumflex = 16777591;
pub const ydiaeresis = 255;
pub const yen = 165;
pub const ygrave = 16785139;
pub const yhook = 16785143;
pub const ytilde = 16785145;
pub const z = 122;
pub const zabovedot = 447;
pub const zacute = 444;
pub const zcaron = 446;
pub const zerosubscript = 16785536;
pub const zerosuperior = 16785520;
pub const zstroke = 16777654;
