/*-------------------------------------------------------------------------
 * getmd.c
 *
 * A utility to get md devices.  
 * Either get all md devices in raidtab (-a option), or 
 * get the one that matches a disk partition device pattern.
 *
 * Author:  Andy Cress  <arcress@users.sourceforge.net>
 *
 * 09/06/01 ARCress - created 
 * 01/30/02 ARCress - added capability to get devices that match an md device
 * 07/10/02 ARCress - set default devpattn to "sd" to be more flexible
 *-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002, Intel Corporation
Copyright (c) 2009, Kontron America, Inc.
All rights reserved.
Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:
  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
// char raidfile[] = "/etc/raidtab";
// char raidstr[] = "raiddev";
char fdebug = 0;
char fall = 0;

/* 
 * findmatch 
 * returns offset in buffer of the match, if found, 
 * or -1 if not found.  
 */
static int findmatch(const char *buffer, int sbuf, char *pattern, int spattern, 
		char figncase)
{
    int c, i, j, imatch;

    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) return(-1);
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if ((figncase == 1) &&
                   ((c & 0x5f) == (pattern[imatch] & 0x5f))) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else {
            if (imatch > 0) {
               imatch = 0;
               if (j > 0) j--; /* try again with the first match char */
	    }
        }
        if (imatch == spattern) break;
    }
    if (imatch == spattern) {
        i = (j+1) - imatch;  /*buffer[i] is the match */
        return(i);
    } else return (-1);  /*not found*/
}

static int get_line(FILE * fd, char *buf, int len)
{
    int i;
    int ch;
    for (i = 0; i < len; i++) {
	ch = fgetc(fd);
	if (ch == EOF)
	    return (i);
	buf[i] = (char) ch;
	if (ch == 0x0a)
	    return (++i);
    }
    return (i);
}

int getmd(char *devpattn, int sdevpattn, char *mddev, char *mdpart, char findmd)
{
    char buf[80];
    char rdev[75];
    int rlen, i, j;
    FILE *fd;
    char whspace[] = " \t\n";
    char raidfile[] = "/etc/raidtab";
    char raidstr[] = "raiddev";
    int sraidstr;
    int foundit = 0;
    int fhavemd = 0;
    if (sdevpattn >= 7) {
	if (devpattn[6] == 'g')
	    devpattn[6] = 'd';	/* swap sg -> sd */
    }
    fd = fopen(raidfile, "r");
    if (fd == NULL) return (errno);

    // sdevpattn = strlen(devpattn);
    sraidstr = strlen(raidstr);
    strcpy(mdpart, "/dev/sda1");	/* default */
    strcpy(rdev, "/dev/md0");	/* default */
    while ((rlen = get_line(fd, buf, 80)) > 0) {	/* process each line */
	buf[rlen] = 0;		/* stringify */
	/* look for raiddev string */
	i = findmatch(buf, rlen, raidstr, sraidstr, 0);
	/* if so, get value for rdev */
	if (i >= 0) {
	    i += sraidstr;
	    j = strspn(&buf[i], whspace);   /* skip leading spaces */
	    i += j;
	    j = strcspn(&buf[i], whspace);  /* find end (w spaces) */
	    strncpy(rdev, &buf[i], j);
	    rdev[j] = 0;
	    if (fdebug) printf("raiddev: %s\n",rdev);
	    if (findmatch(rdev, j, devpattn, sdevpattn, 0) >= 0) {
		fhavemd = 1;
	        if (fdebug) printf(" fhavemd: %s ",&buf[i]);
		}
	    if (fall || fhavemd == 1)
		printf("%s ", rdev);
	}
	if (findmd) {  /* show all configured devices for this md */
	  if (fhavemd) { 
	     foundit = 1;
	     i = findmatch(buf, rlen, "device", 6, 0);
	     if (i >= 0) {
		i += 6;
		j = strspn(&buf[i], whspace);   /* skip leading spaces */
		i += j;
		j = strcspn(&buf[i], whspace);  /* find end (w spaces) */
	        if (fdebug) printf(" device: %s ",&buf[i]);
		strncpy(mdpart, &buf[i], j);
		mdpart[j] = 0;
		printf("%s ", mdpart);
		}
	     if (fhavemd == 2) {  /* not the first time */
	        i = findmatch(buf, rlen, raidstr, sraidstr, 0);
	        if (i >= 0) break;   /* end of device list for this md */
	     } else fhavemd = 2;  /* first time, skip this, then look */
	  }
	}
	else {
	   /* look for devpattn */
	   i = findmatch(buf, rlen, devpattn, sdevpattn, 0);
	   if (i >= 0) {		/* if so, stop and show rdev & mdpart */
	       foundit = 1;
	       j = strcspn(&buf[i], whspace);
	       strncpy(mdpart, &buf[i], j);
	       mdpart[j] = 0;
	       if (!fall) break;
	   }
	}
    }
    fclose(fd);
    if (foundit) {
	strcpy(mddev, rdev);
	return (0);
    } else {
	strcpy(mddev, rdev);
	return (-1);
    }
}				/*end getmd() */

int main(int argc, char **argv)
{
    char *devpattn;
    char mddev[75];
    char mdpart[40];
    int rc;
    char *progname;
    char fmd;
    progname = argv[0];
    devpattn = "sd";
    while (argc > 1) {
	if (argv[1][0] == '-') {	/* options */
	    switch (argv[1][1]) {
	    case 'a':
		fall = 1;
		break;
	    case 'x':
		fdebug = 1;
		break;
	    default:
		printf("Usage: %s [-a -x] [device pattern]\n", progname);
		exit(1);
	    }
	} /*endif options */
	else
	    devpattn = argv[1];
	argv++;
	argc--;
    }
    if (findmatch(devpattn, strlen(devpattn), "md", 2, 0) >= 0) fmd = 1;
    else fmd = 0;
    if (fdebug)
	printf("%s argc = %d, pattern = %s, all=%d, debug=%d, md=%d\n",
	       progname, argc, devpattn, fall, fdebug, fmd);
    rc = getmd(devpattn, strlen(devpattn), mddev, mdpart, fmd);
    if (fall || fmd)
	printf("\n");
    else if (!fmd) 
	printf("%s %s\n", mddev, mdpart);
    if (fdebug) {
	    printf("%s status = %d, mddev = %s, mdpart = %s\n",
		   progname, rc, mddev, mdpart);
    }
    exit(rc);
}
