// Copyright 2018 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License.

package xform_test

import (
	"flag"
	"strings"
	"sync"
	"testing"

	"github.com/cockroachdb/cockroach/pkg/settings/cluster"
	"github.com/cockroachdb/cockroach/pkg/sql/opt"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/memo"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/norm"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/testutils"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/testutils/opttester"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/testutils/testcat"
	"github.com/cockroachdb/cockroach/pkg/sql/opt/xform"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/testutils/datadriven"
	"github.com/cockroachdb/cockroach/pkg/util/leaktest"
)

func TestDetachMemo(t *testing.T) {
	defer leaktest.AfterTest(t)()
	catalog := testcat.New()
	if _, err := catalog.ExecuteDDL("CREATE TABLE abc (a INT PRIMARY KEY, b INT, c STRING, INDEX (c))"); err != nil {
		t.Fatal(err)
	}

	var o xform.Optimizer
	evalCtx := tree.MakeTestingEvalContext(cluster.MakeTestingClusterSettings())
	testutils.BuildQuery(t, &o, catalog, &evalCtx, "SELECT * FROM abc WHERE c=$1")

	before := o.DetachMemo()

	if !o.Memo().IsEmpty() {
		t.Error("memo expression should be reinitialized by DetachMemo")
	}

	testutils.BuildQuery(t, &o, catalog, &evalCtx, "SELECT a=$1 FROM abc")

	after := o.Memo()
	if after == before {
		t.Error("after memo cannot be the same as the detached memo")
	}

	if !strings.Contains(after.RootExpr().String(), "variable: a [type=int]") {
		t.Error("after memo did not contain expected operator")
	}

	if after.RootExpr().(memo.RelExpr).Memo() != after {
		t.Error("after memo expression does not reference the after memo")
	}

	if before == o.Memo() {
		t.Error("detached memo should not be reused")
	}

	if before.RootExpr().(memo.RelExpr).Memo() != before {
		t.Error("detached memo expression does not reference the detached memo")
	}

	if !strings.Contains(before.RootExpr().String(), "variable: c [type=string]") {
		t.Error("detached memo did not contain expected operator")
	}
}

// TestDetachMemoRace reproduces the condition in #34904: a detached memo still
// aliases table annotations in the metadata. The problematic annotation is a
// statistics object. Construction of new expression can trigger calculation of
// new statistics.
func TestDetachMemoRace(t *testing.T) {
	defer leaktest.AfterTest(t)()

	catalog := testcat.New()

	_, err := catalog.ExecuteDDL("CREATE TABLE abc (a INT, b INT, c INT, d INT)")
	if err != nil {
		t.Fatal(err)
	}
	var o xform.Optimizer
	evalCtx := tree.MakeTestingEvalContext(cluster.MakeTestingClusterSettings())
	testutils.BuildQuery(t, &o, catalog, &evalCtx, "SELECT * FROM abc WHERE a = $1")
	mem := o.DetachMemo()

	var wg sync.WaitGroup
	for i := 0; i < 4; i++ {
		col := opt.ColumnID(i + 1)
		wg.Add(1)
		go func() {
			var o xform.Optimizer
			evalCtx := tree.MakeTestingEvalContext(cluster.MakeTestingClusterSettings())
			o.Init(&evalCtx)
			f := o.Factory()
			var replaceFn norm.ReplaceFunc
			replaceFn = func(e opt.Expr) opt.Expr {
				if sel, ok := e.(*memo.SelectExpr); ok {
					return f.ConstructSelect(
						f.CopyAndReplaceDefault(sel.Input, replaceFn).(memo.RelExpr),
						memo.FiltersExpr{{
							Condition: f.ConstructEq(
								f.ConstructVariable(col),
								f.ConstructConst(tree.NewDInt(10)),
							),
						}},
					)
				}
				return f.CopyAndReplaceDefault(e, replaceFn)
			}
			// Rewrite the filter to use a different column, which will trigger creation
			// of new table statistics. If the statistics object is aliased, this will
			// be racy.
			f.CopyAndReplace(mem.RootExpr().(memo.RelExpr), mem.RootProps(), replaceFn)
			wg.Done()
		}()
	}
	wg.Wait()
}

// TestCoster files can be run separately like this:
//   make test PKG=./pkg/sql/opt/xform TESTS="TestCoster/sort"
//   make test PKG=./pkg/sql/opt/xform TESTS="TestCoster/scan"
//   ...
func TestCoster(t *testing.T) {
	defer leaktest.AfterTest(t)()
	runDataDrivenTest(
		t, "testdata/coster/",
		memo.ExprFmtHideRuleProps|memo.ExprFmtHideQualifications|memo.ExprFmtHideScalars,
	)
}

// TestPhysicalProps files can be run separately like this:
//   make test PKG=./pkg/sql/opt/xform TESTS="TestPhysicalPropsFactory/ordering"
//   make test PKG=./pkg/sql/opt/xform TESTS="TestPhysicalPropsFactory/presentation"
//   ...
func TestPhysicalProps(t *testing.T) {
	defer leaktest.AfterTest(t)()
	runDataDrivenTest(
		t, "testdata/physprops/",
		memo.ExprFmtHideMiscProps|
			memo.ExprFmtHideConstraints|
			memo.ExprFmtHideFuncDeps|
			memo.ExprFmtHideRuleProps|
			memo.ExprFmtHideStats|
			memo.ExprFmtHideCost|
			memo.ExprFmtHideQualifications|
			memo.ExprFmtHideScalars,
	)
}

// TestRuleProps files can be run separately like this:
//   make test PKG=./pkg/sql/opt/xform TESTS="TestRuleProps/orderings"
//   ...
func TestRuleProps(t *testing.T) {
	defer leaktest.AfterTest(t)()
	datadriven.Walk(t, "testdata/ruleprops", func(t *testing.T, path string) {
		catalog := testcat.New()
		datadriven.RunTest(t, path, func(d *datadriven.TestData) string {
			tester := opttester.New(catalog, d.Input)
			tester.Flags.ExprFormat = memo.ExprFmtHideStats | memo.ExprFmtHideCost |
				memo.ExprFmtHideQualifications | memo.ExprFmtHideScalars
			return tester.RunCommand(t, d)
		})
	})
}

// TestRules files can be run separately like this:
//   make test PKG=./pkg/sql/opt/xform TESTS="TestRules/scan"
//   make test PKG=./pkg/sql/opt/xform TESTS="TestRules/select"
//   ...
func TestRules(t *testing.T) {
	defer leaktest.AfterTest(t)()
	runDataDrivenTest(
		t,
		"testdata/rules/",
		memo.ExprFmtHideStats|memo.ExprFmtHideCost|memo.ExprFmtHideRuleProps|
			memo.ExprFmtHideQualifications|memo.ExprFmtHideScalars,
	)
}

var externalTestData = flag.String(
	"d", "testdata/external/", "test files directory for TestExternal",
)

// TestExternal contains test cases from external customers and external
// benchmarks (like TPCH), so that changes in their query plans can be monitored
// over time.
//
// TestExternal files can be run separately like this:
//   make test PKG=./pkg/sql/opt/xform TESTS="TestExternal/tpch"
//   ...
//
// Test files from another location can be run using the -d flag:
//   make test PKG=./pkg/sql/opt/xform TESTS=TestExternal TESTFLAGS='-d /some-dir'
//
func TestExternal(t *testing.T) {
	defer leaktest.AfterTest(t)()
	runDataDrivenTest(
		t,
		*externalTestData,
		memo.ExprFmtHideStats|memo.ExprFmtHideCost|memo.ExprFmtHideRuleProps|
			memo.ExprFmtHideQualifications|memo.ExprFmtHideScalars,
	)
}

// runDataDrivenTest runs data-driven testcases of the form
//   <command>
//   <SQL statement>
//   ----
//   <expected results>
//
// See OptTester.Handle for supported commands.
func runDataDrivenTest(t *testing.T, path string, fmtFlags memo.ExprFmtFlags) {
	datadriven.Walk(t, path, func(t *testing.T, path string) {
		catalog := testcat.New()
		datadriven.RunTest(t, path, func(d *datadriven.TestData) string {
			tester := opttester.New(catalog, d.Input)
			tester.Flags.ExprFormat = fmtFlags
			return tester.RunCommand(t, d)
		})
	})
}
