//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cMainMenu.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cMainMenu.hpp"

#include <SDL.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "cGameCore.hpp"
#include "cDisplayManager.hpp"
#include "cTextureManager.hpp"
#include "cMixer.hpp"
#include "cOptions.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

enum Buttons
{
	Button_Player1Keys = 100,
	Button_Player2Keys,
	Button_Save,
	Button_MouseConfig
};

enum Toggles
{
	Toggle_Fullscreen = 1,
	Toggle_LowRes,
	Toggle_Sound,
	Toggle_Music,
	Toggle_FrameSkip,
	Toggle_Delay,
    Toggle_MouseTurning,
    Toggle_MousePlayer1
};

//! Constructor
cMainMenu::cMainMenu(cGameCore &gameCore):
mGameCore (gameCore),
mDisplayManager (cDisplayManager::GetInstance ()),
mTextureManager (cTextureManager::GetInstance ()),
mOptions (cOptions::GetInstance ()),
mSelectedItem (0),
mShowOptions (false),
mShowKeys (false),
mWaitKey (false),
mNeedsRestart (false)
{
	mMenuItems.push_back ("Single player");
	mMenuItems.push_back ("Split screen");
	mMenuItems.push_back ("Options");
	mMenuItems.push_back ("Quit game");

	// Make this better
	mDemoStrings.push_back ("2ndPoint Ry proudly presents");
	mDemoStrings.push_back ("an open source action game");
	mDemoStrings.push_back ("released under the GNU GPL");
	mDemoStrings.push_back ("   ");
	mDemoStrings.push_back ("Shooting Star");
	mDemoStrings.push_back ("   ");
	mDemoStrings.push_back ("Shooting Star developers:");
	mDemoStrings.push_back ("Jarmo Hekkanen (code, maps)");
	mDemoStrings.push_back ("Tuomas Peippo (code, sounds, cvs)");
	mDemoStrings.push_back ("Lauri Hakkarainen (graphics)");
	mDemoStrings.push_back ("Santeri Pilli (music)");
	mDemoStrings.push_back ("      ");
	
	// Options
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_Fullscreen, "Fullscreen", mOptions.mFullscreen));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_LowRes, "640x480 video mode", mOptions.mLowRes));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_Sound, "Sound", mOptions.mSound));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_Music, "Music", mOptions.mMusic));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_FrameSkip, "Frame skipping", mOptions.mFrameSkip));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_Delay, "Frame delay", mOptions.mDelay));
	mItems.push_back (cMenuItem (cMenuItem::Type_Button, Button_Player1Keys, "Config player 1 controls"));
	mItems.push_back (cMenuItem (cMenuItem::Type_Button, Button_Player2Keys, "Config player 2 controls"));
	mItems.push_back (cMenuItem (cMenuItem::Type_Button, Button_MouseConfig, "Config mouse"));
	mItems.push_back (cMenuItem (cMenuItem::Type_Button, Button_Save, "Save options"));
	mOptionsSelection = 0;
    mNumberOfMenuItems = mItems.size ();    // Rest of the items belong to mouse menu

	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_MouseTurning, "Mouse turning", mOptions.mMouseTurning));
	mItems.push_back (cMenuItem (cMenuItem::Type_Toggle, Toggle_MousePlayer1, "Mouse player", mOptions.mMousePlayer1));
	mItems.back ().SetToggleNames ("player 1", "player 2");
	
	mpKeys = &mOptions.mPlayer1Keys;
	mSelectedKey = 0;

	mKeyNames.push_back ("Turn left");
	mKeyNames.push_back ("Turn right");
	mKeyNames.push_back ("Walk forward");
	mKeyNames.push_back ("Walk backward");
    mKeyNames.push_back ("Walk left");
    mKeyNames.push_back ("Walk right");
	mKeyNames.push_back ("Fire");
	mKeyNames.push_back ("Next weapon");
	mKeyNames.push_back ("Previous weapon");
	mKeyNames.push_back ("Aim mode");
	mKeyNames.push_back ("Strafe mode");
	dbg::assertion (DBG_ASSERTION (mKeyNames.size () == tPlayerKeys::NumberOfKeys));
};

//! Destructor
cMainMenu::~cMainMenu (void)
{
	// Empty
};

cMainMenu::MenuSelection
cMainMenu::Run (void)
{
	// Load textures
	mTextureManager.FreeAllTextures ();
	mLogoLeft = mTextureManager.LoadTexture ("logo_left.png");	
	mLogoRight = mTextureManager.LoadTexture ("logo_right.png");
	mMainMenu = mTextureManager.LoadTexture ("mainmenu.png");
	mMenuSelector = mTextureManager.LoadTexture ("menu_selector.png");		

	mFont.Initialize ("font_11x20.png", 11, 20);
	
	glEnable (GL_TEXTURE_2D);

	mLastUpdate = SDL_GetTicks ();
	mLastLetter = SDL_GetTicks ();
	mCurrentString = 0;
	
	bool running = true;
	SDL_Event event;
	while ( running )
	{

		while ( SDL_PollEvent (&event) != 0 )
		{
			switch ( event.type )
			{
				case SDL_QUIT:
					running = false;
					break;
				case SDL_KEYDOWN:
					switch ( event.key.keysym.sym )
					{
						case SDLK_ESCAPE:
							running = false;
							break;
						case SDLK_UP:
							MoveSelectionUp ();
							break;
						case SDLK_DOWN:
							MoveSelectionDown ();
							break;
						case SDLK_RETURN:
							// Same as SDLK_SPACE
						case SDLK_SPACE:
							if ( mShowOptions )
								break;
							if ( mMenuItems[mSelectedItem] == "Split screen" )
							{
								Fade ();
								return SplitScreen;
							}
							if ( mMenuItems[mSelectedItem] == "Single player" )
							{
								Fade ();
								return SinglePlayer;
							}

							if ( mMenuItems[mSelectedItem] == "Quit game" )
							{
								Fade ();
								return QuitGame;
							}
							if ( mMenuItems[mSelectedItem] == "Options" )
								Options ();
							break;
						default:
							break;
					}
					break;
				default:
					break;
			}
		}
		
		Render ();
		
	}
	Fade ();
	return QuitGame;
}

void 
cMainMenu::MoveSelectionUp (void)
{

	if ( !mShowOptions )
	{
		if ( mSelectedItem == 0 )
			mSelectedItem = mMenuItems.size () - 1;
		else
			mSelectedItem--;
	}
	else if ( mShowKeys )
	{
		if ( mSelectedKey == 0 )
			mSelectedKey = tPlayerKeys::NumberOfKeys - 1;
		else
			mSelectedKey--;
	
	}
	else if ( mShowMouse )
  	{
		if ( mOptionsSelection == mNumberOfMenuItems )
			mOptionsSelection = mItems.size ();
		else
			mOptionsSelection--;
	}
    else
	{
		if ( mOptionsSelection == 0 )
			mOptionsSelection = mNumberOfMenuItems - 1;
		else
			mOptionsSelection--;
	}
}

void 
cMainMenu::MoveSelectionDown (void)
{

	if ( !mShowOptions )
	{
		if ( mSelectedItem == mMenuItems.size () - 1 )
			mSelectedItem = 0;
		else
			mSelectedItem++;
	}
	else if ( mShowKeys )
	{
		if ( mSelectedKey == tPlayerKeys::NumberOfKeys - 1 )
			 mSelectedKey = 0;
		else
			mSelectedKey++;
	}		
    else if ( mShowMouse )
	{
		if ( mOptionsSelection == mItems.size ())
			 mOptionsSelection = mNumberOfMenuItems;
		else
			mOptionsSelection++;
	}
	else
	{
		if ( mOptionsSelection == mNumberOfMenuItems - 1 )
			 mOptionsSelection = 0;
		else
			mOptionsSelection++;
	}
}

void 
cMainMenu::Render (void)
{
	SDL_Delay (SDL_TIMESLICE);
	mDisplayManager.BeginFrame ();
	
	RenderDemo ();
	
	RenderOptions ();
	
	glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
	
	glPushMatrix ();
	glTranslatef (30.0f, 30.0f, 0.0f);
	
	// Render Shooting Star logo (LEFT)
	glBindTexture (GL_TEXTURE_2D, mLogoLeft);		
	glBegin (GL_QUADS);
		glTexCoord2f (0.0f, 0.0f);	glVertex2f (0.0f, 0.0f);
		glTexCoord2f (0.0f, 1.0f);	glVertex2f (0.0f, 128.0f);
		glTexCoord2f (1.0f, 1.0f);	glVertex2f (256.0f, 128.0f);
		glTexCoord2f (1.0f, 0.0f);	glVertex2f (256.0f, 0.0f);
	glEnd ();

	// Render Shooting Star logo (RIGHT)
	glBindTexture (GL_TEXTURE_2D, mLogoRight);		
	glBegin (GL_QUADS);
		glTexCoord2f (0.0f, 0.0f);	glVertex2f (256.0f, 0.0f);
		glTexCoord2f (0.0f, 1.0f);	glVertex2f (256.0f, 128.0f);
		glTexCoord2f (1.0f, 1.0f);	glVertex2f (512.0f, 128.0f);
		glTexCoord2f (1.0f, 0.0f);	glVertex2f (512.0f, 0.0f);
	glEnd ();
	
	glPopMatrix ();
	
	         
	glPushMatrix ();
	glTranslatef (800.0f - 256.0f - 5.0f,
					600.0f - 128.0f - 5.0f,
					0.0f);
					
	// Render main menu
	glBindTexture (GL_TEXTURE_2D, mMainMenu);		
	glBegin (GL_QUADS);
		glTexCoord2f (0.0f, 0.0f);	glVertex2f (0.0f, 0.0f);
		glTexCoord2f (0.0f, 1.0f);	glVertex2f (0.0f, 128.0f);
		glTexCoord2f (1.0f, 1.0f);	glVertex2f (256.0f, 128.0f);
		glTexCoord2f (1.0f, 0.0f);	glVertex2f (256.0f, 0.0f);
	glEnd ();
	
	glTranslatef (-32.0f, mSelectedItem * 30.0f + 2.0f, 0.0f);
	
	// Render menu selection
	glBindTexture (GL_TEXTURE_2D, mMenuSelector);		
	glBegin (GL_QUADS);
		glTexCoord2f (0.0f, 0.0f);	glVertex2f (0.0f, 0.0f);
		glTexCoord2f (0.0f, 1.0f);	glVertex2f (0.0f, 32.0f);
		glTexCoord2f (1.0f, 1.0f);	glVertex2f (32.0f, 32.0f);
		glTexCoord2f (1.0f, 0.0f);	glVertex2f (32.0f, 0.0f);
	glEnd ();		
	
	glPopMatrix ();
	
	mDisplayManager.EndFrame ();
	
	GLenum error = glGetError ();
	while ( error != GL_NO_ERROR )
	{
		dbgError () << "cMainMenu: OpenGL error: " << gluErrorString (error) << '\n';
		error = glGetError ();
	}
}

void
cMainMenu::RenderDemo (void)
{
	if ( mDemoStrings.empty () )
		return;

	Uint32 now = SDL_GetTicks ();
	Uint32 deltaTime = now - mLastUpdate;
	mLastUpdate = now;

	glPushMatrix ();

	glTranslatef (300.0f, 300.0f, 0.0f);
	
	static bool fader = false;

	if ( fader )
	{
		static float d = 0.0f;
		float tmp;
		d += 0.0005f * deltaTime;
		if ( d >= 1.0f )
		{
			d = 0.0f;
			tmp	= 1.0f;
			fader = false;
		}
		else
			tmp = d;
		
		unsigned int faderString = mCurrentString - 1;
		if ( mCurrentString == 0 )
			faderString = mDemoStrings.size () - 1;
	
		glPushMatrix ();
			glColor4f (1.0f, 1.0f, 1.0f, 1.0f - tmp);
			glTranslatef (tmp * -80.0f, tmp * -80.0f, 0.0f);
			mFont.PrintString (mDemoStrings[faderString]);
		glPopMatrix ();
	}
	
	string &text = mDemoStrings[mCurrentString];
	
	static unsigned int i = 0;
	if ( mLastLetter + 100 < now )
	{
		i++;
		if ( i > text.size () )
		{
			i = 0;
			fader = true;
			mLastLetter = now + 1500;
			mCurrentString++;
			if ( mCurrentString >= mDemoStrings.size () )
				mCurrentString = 0;
		}
		else
			mLastLetter = now;
	}
	
	glTranslatef ((i - 2) * 11.0f, 0.0f, 0.0f);
	if ( i - 1 > 0 )
	{
		glColor4f (1.0f, 1.0f, 1.0f, 0.5f);
		mFont.Print ("%c", text[i - 2]);
	}
	glTranslatef (11.0f, 0.0f, 0.0f);
	if ( i > 0 )
	{
		glColor4f (1.0f, 1.0f, 1.0f, 0.75f);
		mFont.Print ("%c", text[i - 1]);
	}
	glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
	glTranslatef (11.0f, 0.0f, 0.0f);
	mFont.Print ("%c", text[i]);

	glPopMatrix ();
}

void 
cMainMenu::RenderOptions (void)
{
	// TEMP code
	
	if ( !mShowOptions )
		return;
	
	glPushMatrix ();
	glPushAttrib (GL_CURRENT_BIT|GL_ENABLE_BIT);
	glDisable (GL_TEXTURE_2D);
	
	glTranslatef (30.0f, 150.0f, 0.0f);
	
	glColor4f (0.0f, 0.0f, 0.0f, 0.5f);
	glBegin (GL_QUADS);
		glVertex2f (0.0f, 0.0f);
		glVertex2f (0.0f, 300.0f);
		glVertex2f (640.0f, 300.0f);
		glVertex2f (640.0f, 0.0f);
	glEnd ();
	
	glTranslatef (5.0f, 5.0f, 0.0f);
	
	glColor4f (1.0f, 0.5f, 0.0f, 1.0f);
	glTranslatef (0.0f, 20.0f, 0.0f);
	if ( mShowMouse )
		mFont.Print ("CONFIG MOUSE");
	else if ( mShowKeys )
		mFont.Print ("CONFIG KEYS");
	else
		mFont.Print ("OPTIONS");
	
	glTranslatef (0.0f, 5.0f, 0.0f);
	
	if ( !mShowKeys && !mShowMouse )
	{
		for ( unsigned int i = 0; i < mNumberOfMenuItems; i++ )
		{
			glTranslatef (0.0f, 20.0f, 0.0f);
			if ( mOptionsSelection == i )
				glColor4f (1.0f, 1.0f, 0.0f, 1.0f);
			else
				glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
			switch ( mItems[i].mType )
			{
				case cMenuItem::Type_Button:
					mFont.PrintString (mItems[i].mText);
					break;
				case cMenuItem::Type_Toggle:
					mFont.Print ("%-20s %s", mItems[i].mText.c_str (), mItems[i].GetToggleText ().c_str ());
					break;
				default:
					break;
			}
		}
		if ( mNeedsRestart )
		{
			glTranslatef (0.0f, 25.0f, 0.0f);
			glColor4f (1.0f, 0.5f, 0.0f, 1.0f);
			mFont.PrintString ("YOU NEED TO SAVE AND RESTART");
		}
	}
	else if ( mShowKeys )
	{
		for ( int i = 0; i < tPlayerKeys::NumberOfKeys; i++ )
		{
			glTranslatef (0.0f, 20.0f, 0.0f);
			if ( mSelectedKey == i )
			{
				glColor4f (1.0f, 1.0f, 0.0f, 1.0f);
			}
			else
				glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
			
			if ( mWaitKey && mSelectedKey == i )
				mFont.Print ("%-20s [KEY]", mKeyNames[i].c_str ());
			else if ( mpKeys->buttons[i] == 0 )
				mFont.Print ("%-20s %s", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
			else
			{
				/* 
					#define SDL_BUTTON_LEFT		1
					#define SDL_BUTTON_MIDDLE	2
					#define SDL_BUTTON_RIGHT	3
					#define SDL_BUTTON_WHEELUP	4
					#define SDL_BUTTON_WHEELDOWN	5
				*/
				switch ( mpKeys->buttons[i] )
				{
					case SDL_BUTTON_LEFT:
						mFont.Print ("%-20s %s or mouse left", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
					case SDL_BUTTON_MIDDLE:
						mFont.Print ("%-20s %s or mouse middle", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
					case SDL_BUTTON_RIGHT:
						mFont.Print ("%-20s %s or mouse right", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
					case SDL_BUTTON_WHEELUP:
						mFont.Print ("%-20s %s or mouse wheel up", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
					case SDL_BUTTON_WHEELDOWN:
						mFont.Print ("%-20s %s or mouse wheel down", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
					default:
						mFont.Print ("%-20s %s or mouse unknown", mKeyNames[i].c_str (), SDL_GetKeyName (mpKeys->keys[i]));
						break;
				}
			}
		}
        glColor4f (1.0f, 0.5f, 0.0f, 1.0f);
        glTranslatef (0.0f, 25.0f, 0.0f);
        mFont.Print ("ENTER to bind key or mouse button");
        glTranslatef (0.0f, 20.0f, 0.0f);
        mFont.Print ("BACKSPACE to remove key binding");
		
		glColor3f (1.0f, 0.1f, 0.0f);
	    glTranslatef (0.0f, 25.0f, 0.0f);
        mFont.Print ("Mouse bindings for turning");
		glTranslatef (0.0f, 20.0f, 0.0f);
		mFont.Print ("and walking don't work");

	}
	else    // Mouse stuff
	{
		glTranslatef (0.0f, 20.0f, 0.0f);
		glColor3f (1.0f, 1.0f, 1.0f);
		
        for ( unsigned int i = mNumberOfMenuItems; i < mItems.size (); i++ )
		{
			glTranslatef (0.0f, 20.0f, 0.0f);
			if ( mOptionsSelection == i )
				glColor4f (1.0f, 1.0f, 0.0f, 1.0f);
			else
				glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
			switch ( mItems[i].mType )
			{
				case cMenuItem::Type_Button:
					mFont.PrintString (mItems[i].mText);
					break;
				case cMenuItem::Type_Toggle:
					mFont.Print ("%-20s %s", mItems[i].mText.c_str (), mItems[i].GetToggleText ().c_str ());
					break;
				default:
					break;
			}
		}
        glTranslatef (0.0f, 20.0f, 0.0f);
    	if ( mOptionsSelection == mItems.size () )
	        glColor4f (1.0f, 1.0f, 0.0f, 1.0f);
		else
			glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
        mFont.Print ("Sensitivity %3i  ", mOptions.mMouseSensitivity);
    }
		
	glPopAttrib ();
	glPopMatrix ();
}

void 
cMainMenu::Fade (void)
{
	int width = 800;
	int height = 600;
	
	glDisable (GL_TEXTURE_2D);
	
	float alpha = 0.05f;
	for ( int i = 0; i < 50; i++ )
	{
		mDisplayManager.BeginFrame (false);
		
		glColor4f (0.0f, 0.0f, 0.0f, alpha);
		glBegin (GL_QUADS);
			glVertex2i (0, 0);
			glVertex2i (0, height);
			glVertex2i (width, height);
			glVertex2i (width, 0);
		glEnd ();
		alpha += 0.95f / 70.0f;
	
		mDisplayManager.EndFrame ();
		SDL_Delay (20);
	}
	
	glEnable (GL_TEXTURE_2D);
}

void 
cMainMenu::Options (void)
{
	mShowOptions = true;
	bool running = true;
	CheckOptions ();
    Uint8 *pKeys = SDL_GetKeyState (NULL);
	SDL_Event event;
    Uint32 lastChange = SDL_GetTicks (), now, changeDelay = 70;
    
	while ( running )
	{
		while ( SDL_PollEvent (&event) != 0 )
		{
			switch ( event.type )
			{
				case SDL_QUIT:
					running = false;
					break;
				case SDL_MOUSEBUTTONDOWN:
					if ( mWaitKey )
					{
						ClearButtons (event.button.button);
						mpKeys->buttons[mSelectedKey] = event.button.button;
						mWaitKey = false;
					}
					break;
				case SDL_KEYDOWN:
					if ( mWaitKey )
					{
						ClearKeys (event.key.keysym.sym);
						mpKeys->keys[mSelectedKey] = event.key.keysym.sym;	
						mWaitKey = false;
						break;
					}
					switch ( event.key.keysym.sym )
					{
						case SDLK_ESCAPE:
							if ( mShowMouse )
                            {
								mShowMouse = false;
                                mOptionsSelection = mNumberOfMenuItems - 2;
                            }
							else if ( mShowKeys )
								mShowKeys = false;
							else
								running = false;
							break;
						case SDLK_DOWN:
							MoveSelectionDown ();
							break;
						case SDLK_UP:
							MoveSelectionUp ();
							break;
						case SDLK_RETURN:
							// Same as SDLK_SPACE
						case SDLK_SPACE:
							if ( mShowKeys )
							{
								mWaitKey = true;
								break;
							}
							else
							{
                                if ( mOptionsSelection >= mItems.size () )
                                    break;
								switch ( mItems[mOptionsSelection].mType )
								{
									case cMenuItem::Type_Toggle:
										mItems[mOptionsSelection].mToggleValue = 
											!mItems[mOptionsSelection].mToggleValue;
										CheckOptions ();
										break;
									case cMenuItem::Type_Button:
										ButtonHandler ();
										break;
									default:
										break;
								}
							}
							break;
                        case SDLK_BACKSPACE:
                            if ( mShowKeys )
                            {
                                mpKeys->keys[mSelectedKey] = SDLK_UNKNOWN;
								mpKeys->buttons[mSelectedKey] = 0;
                            }
						default:
							break;
					}
					break;
				default:
					break;
			}
		}
        
        now = SDL_GetTicks ();
        if ( mOptionsSelection == mItems.size () && now - lastChange > changeDelay)
        {
            if ( pKeys[SDLK_LEFT] == SDL_PRESSED && mOptions.mMouseSensitivity > -128 )
                mOptions.mMouseSensitivity--;
            else if ( pKeys[SDLK_RIGHT] == SDL_PRESSED && mOptions.mMouseSensitivity < 127 )
                mOptions.mMouseSensitivity++;
            lastChange = now;
            if ( changeDelay > 10 )
                changeDelay -= 2;
        }
        if ( pKeys[SDLK_LEFT] == SDL_RELEASED && pKeys[SDLK_RIGHT] == SDL_RELEASED )
            changeDelay = 70;
		
		Render ();
  	}
	mShowOptions = false;
}

void 
cMainMenu::ButtonHandler (void)
{
	switch ( mItems[mOptionsSelection].mID )
	{
		case Button_Player1Keys:
			mShowKeys = true;
			mpKeys = &mOptions.mPlayer1Keys;
			mSelectedKey = 0;
			break;
		case Button_Player2Keys:
			mShowKeys = true;
			mpKeys = &mOptions.mPlayer2Keys;
			mSelectedKey = 0;
			break;
		case Button_MouseConfig:
			mShowMouse = true;
            mOptionsSelection = mNumberOfMenuItems;
			break;
		case Button_Save:
			try
			{
				CheckOptions (true);
				mOptions.Save (mGameCore.GetOptionsFile ());
			}
			catch ( ... )
			{
				dbgError () << "Unable to save options" << endl;
			}
			break;
		default:
			dbgError () << "Unkonwn ID in cMainMenu::ButtonHandler" << endl;
			break;
	}
}

void 
cMainMenu::ClearKeys (SDLKey key)
{
	for ( int i = 0; i < tPlayerKeys::NumberOfKeys; i++ )
	{
		if ( key == mOptions.mPlayer1Keys.keys[i] )
			mOptions.mPlayer1Keys.keys[i] = SDLK_UNKNOWN;
		if ( key == mOptions.mPlayer2Keys.keys[i] )
			mOptions.mPlayer2Keys.keys[i] = SDLK_UNKNOWN;
	}
}

void 
cMainMenu::ClearButtons (Uint8 button)
{
	for ( int i = 0; i < tPlayerKeys::NumberOfKeys; i++ )
	{
		if ( button == mOptions.mPlayer1Keys.buttons[i] )
			mOptions.mPlayer1Keys.buttons[i] = 0;
		if ( button == mOptions.mPlayer2Keys.buttons[i] )
			mOptions.mPlayer2Keys.buttons[i] = 0;
	}
}

void 
cMainMenu::CheckOptions (bool flush)
{
	mNeedsRestart = false;
	for ( unsigned int i = 0; i < mItems.size (); i++ )
	{
		switch ( mItems[i].mID )
		{
			case Toggle_Fullscreen:
				if ( flush )
					mOptions.mFullscreen = mItems[i].mToggleValue;
				else if ( mOptions.mFullscreen != mItems[i].mToggleValue )
				{
					mNeedsRestart = true;
					return;
				}
				break;
			case Toggle_LowRes:
				if ( flush )
					mOptions.mLowRes = mItems[i].mToggleValue;
				if ( mOptions.mLowRes != mItems[i].mToggleValue )
				{
					mNeedsRestart = true;
					return;
				}
				break;
			case Toggle_Sound:
				mOptions.mSound = mItems[i].mToggleValue;
				cMixer::GetInstance ().EnableSounds (mOptions.mSound);
				break;
			case Toggle_Music:
				if ( mOptions.mMusic != mItems[i].mToggleValue )
				{
					mOptions.mMusic = mItems[i].mToggleValue;
					cMixer::GetInstance ().EnableMusic (mOptions.mMusic);
					if ( !mOptions.mMusic )
						cMixer::GetInstance ().FadeMusic ();
					else
						cMixer::GetInstance ().PlayMusic ("menu.ogg");
				}
				break;
			case Toggle_FrameSkip:
				mOptions.mFrameSkip = mItems[i].mToggleValue;
				break;
			case Toggle_Delay:
				mOptions.mDelay = mItems[i].mToggleValue;
				break;
            case Toggle_MouseTurning:
                mOptions.mMouseTurning = mItems[i].mToggleValue;
                break;
            case Toggle_MousePlayer1:
                mOptions.mMousePlayer1 = mItems[i].mToggleValue;
                break;
			default:
				break;
		}
	}
}

//==============================================================================
// EOF
//==============================================================================
