//nolint:golint,lll,misspell
package util

// An HTML5Entity struct represents HTML5 entitites.
type HTML5Entity struct {
	Name       string
	CodePoints []int
	Characters []byte
}

// LookUpHTML5EntityByName returns (an HTML5Entity, true) if an entity named
// given name is found, otherwise (nil, false).
func LookUpHTML5EntityByName(name string) (*HTML5Entity, bool) {
	v, ok := html5entities[name]
	return v, ok
}

var html5entities = map[string]*HTML5Entity{
	"AElig":                           {Name: "AElig", CodePoints: []int{198}, Characters: []byte{0xc3, 0x86}},
	"AMP":                             {Name: "AMP", CodePoints: []int{38}, Characters: []byte{0x26}},
	"Aacute":                          {Name: "Aacute", CodePoints: []int{193}, Characters: []byte{0xc3, 0x81}},
	"Acirc":                           {Name: "Acirc", CodePoints: []int{194}, Characters: []byte{0xc3, 0x82}},
	"Acy":                             {Name: "Acy", CodePoints: []int{1040}, Characters: []byte{0xd0, 0x90}},
	"Afr":                             {Name: "Afr", CodePoints: []int{120068}, Characters: []byte{0xf0, 0x9d, 0x94, 0x84}},
	"Agrave":                          {Name: "Agrave", CodePoints: []int{192}, Characters: []byte{0xc3, 0x80}},
	"Alpha":                           {Name: "Alpha", CodePoints: []int{913}, Characters: []byte{0xce, 0x91}},
	"Amacr":                           {Name: "Amacr", CodePoints: []int{256}, Characters: []byte{0xc4, 0x80}},
	"And":                             {Name: "And", CodePoints: []int{10835}, Characters: []byte{0xe2, 0xa9, 0x93}},
	"Aogon":                           {Name: "Aogon", CodePoints: []int{260}, Characters: []byte{0xc4, 0x84}},
	"Aopf":                            {Name: "Aopf", CodePoints: []int{120120}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb8}},
	"ApplyFunction":                   {Name: "ApplyFunction", CodePoints: []int{8289}, Characters: []byte{0xe2, 0x81, 0xa1}},
	"Aring":                           {Name: "Aring", CodePoints: []int{197}, Characters: []byte{0xc3, 0x85}},
	"Ascr":                            {Name: "Ascr", CodePoints: []int{119964}, Characters: []byte{0xf0, 0x9d, 0x92, 0x9c}},
	"Assign":                          {Name: "Assign", CodePoints: []int{8788}, Characters: []byte{0xe2, 0x89, 0x94}},
	"Atilde":                          {Name: "Atilde", CodePoints: []int{195}, Characters: []byte{0xc3, 0x83}},
	"Auml":                            {Name: "Auml", CodePoints: []int{196}, Characters: []byte{0xc3, 0x84}},
	"Backslash":                       {Name: "Backslash", CodePoints: []int{8726}, Characters: []byte{0xe2, 0x88, 0x96}},
	"Barv":                            {Name: "Barv", CodePoints: []int{10983}, Characters: []byte{0xe2, 0xab, 0xa7}},
	"Barwed":                          {Name: "Barwed", CodePoints: []int{8966}, Characters: []byte{0xe2, 0x8c, 0x86}},
	"Bcy":                             {Name: "Bcy", CodePoints: []int{1041}, Characters: []byte{0xd0, 0x91}},
	"Because":                         {Name: "Because", CodePoints: []int{8757}, Characters: []byte{0xe2, 0x88, 0xb5}},
	"Bernoullis":                      {Name: "Bernoullis", CodePoints: []int{8492}, Characters: []byte{0xe2, 0x84, 0xac}},
	"Beta":                            {Name: "Beta", CodePoints: []int{914}, Characters: []byte{0xce, 0x92}},
	"Bfr":                             {Name: "Bfr", CodePoints: []int{120069}, Characters: []byte{0xf0, 0x9d, 0x94, 0x85}},
	"Bopf":                            {Name: "Bopf", CodePoints: []int{120121}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb9}},
	"Breve":                           {Name: "Breve", CodePoints: []int{728}, Characters: []byte{0xcb, 0x98}},
	"Bscr":                            {Name: "Bscr", CodePoints: []int{8492}, Characters: []byte{0xe2, 0x84, 0xac}},
	"Bumpeq":                          {Name: "Bumpeq", CodePoints: []int{8782}, Characters: []byte{0xe2, 0x89, 0x8e}},
	"CHcy":                            {Name: "CHcy", CodePoints: []int{1063}, Characters: []byte{0xd0, 0xa7}},
	"COPY":                            {Name: "COPY", CodePoints: []int{169}, Characters: []byte{0xc2, 0xa9}},
	"Cacute":                          {Name: "Cacute", CodePoints: []int{262}, Characters: []byte{0xc4, 0x86}},
	"Cap":                             {Name: "Cap", CodePoints: []int{8914}, Characters: []byte{0xe2, 0x8b, 0x92}},
	"CapitalDifferentialD":            {Name: "CapitalDifferentialD", CodePoints: []int{8517}, Characters: []byte{0xe2, 0x85, 0x85}},
	"Cayleys":                         {Name: "Cayleys", CodePoints: []int{8493}, Characters: []byte{0xe2, 0x84, 0xad}},
	"Ccaron":                          {Name: "Ccaron", CodePoints: []int{268}, Characters: []byte{0xc4, 0x8c}},
	"Ccedil":                          {Name: "Ccedil", CodePoints: []int{199}, Characters: []byte{0xc3, 0x87}},
	"Ccirc":                           {Name: "Ccirc", CodePoints: []int{264}, Characters: []byte{0xc4, 0x88}},
	"Cconint":                         {Name: "Cconint", CodePoints: []int{8752}, Characters: []byte{0xe2, 0x88, 0xb0}},
	"Cdot":                            {Name: "Cdot", CodePoints: []int{266}, Characters: []byte{0xc4, 0x8a}},
	"Cedilla":                         {Name: "Cedilla", CodePoints: []int{184}, Characters: []byte{0xc2, 0xb8}},
	"CenterDot":                       {Name: "CenterDot", CodePoints: []int{183}, Characters: []byte{0xc2, 0xb7}},
	"Cfr":                             {Name: "Cfr", CodePoints: []int{8493}, Characters: []byte{0xe2, 0x84, 0xad}},
	"Chi":                             {Name: "Chi", CodePoints: []int{935}, Characters: []byte{0xce, 0xa7}},
	"CircleDot":                       {Name: "CircleDot", CodePoints: []int{8857}, Characters: []byte{0xe2, 0x8a, 0x99}},
	"CircleMinus":                     {Name: "CircleMinus", CodePoints: []int{8854}, Characters: []byte{0xe2, 0x8a, 0x96}},
	"CirclePlus":                      {Name: "CirclePlus", CodePoints: []int{8853}, Characters: []byte{0xe2, 0x8a, 0x95}},
	"CircleTimes":                     {Name: "CircleTimes", CodePoints: []int{8855}, Characters: []byte{0xe2, 0x8a, 0x97}},
	"ClockwiseContourIntegral":        {Name: "ClockwiseContourIntegral", CodePoints: []int{8754}, Characters: []byte{0xe2, 0x88, 0xb2}},
	"CloseCurlyDoubleQuote":           {Name: "CloseCurlyDoubleQuote", CodePoints: []int{8221}, Characters: []byte{0xe2, 0x80, 0x9d}},
	"CloseCurlyQuote":                 {Name: "CloseCurlyQuote", CodePoints: []int{8217}, Characters: []byte{0xe2, 0x80, 0x99}},
	"Colon":                           {Name: "Colon", CodePoints: []int{8759}, Characters: []byte{0xe2, 0x88, 0xb7}},
	"Colone":                          {Name: "Colone", CodePoints: []int{10868}, Characters: []byte{0xe2, 0xa9, 0xb4}},
	"Congruent":                       {Name: "Congruent", CodePoints: []int{8801}, Characters: []byte{0xe2, 0x89, 0xa1}},
	"Conint":                          {Name: "Conint", CodePoints: []int{8751}, Characters: []byte{0xe2, 0x88, 0xaf}},
	"ContourIntegral":                 {Name: "ContourIntegral", CodePoints: []int{8750}, Characters: []byte{0xe2, 0x88, 0xae}},
	"Copf":                            {Name: "Copf", CodePoints: []int{8450}, Characters: []byte{0xe2, 0x84, 0x82}},
	"Coproduct":                       {Name: "Coproduct", CodePoints: []int{8720}, Characters: []byte{0xe2, 0x88, 0x90}},
	"CounterClockwiseContourIntegral": {Name: "CounterClockwiseContourIntegral", CodePoints: []int{8755}, Characters: []byte{0xe2, 0x88, 0xb3}},
	"Cross":                           {Name: "Cross", CodePoints: []int{10799}, Characters: []byte{0xe2, 0xa8, 0xaf}},
	"Cscr":                            {Name: "Cscr", CodePoints: []int{119966}, Characters: []byte{0xf0, 0x9d, 0x92, 0x9e}},
	"Cup":                             {Name: "Cup", CodePoints: []int{8915}, Characters: []byte{0xe2, 0x8b, 0x93}},
	"CupCap":                          {Name: "CupCap", CodePoints: []int{8781}, Characters: []byte{0xe2, 0x89, 0x8d}},
	"DD":                              {Name: "DD", CodePoints: []int{8517}, Characters: []byte{0xe2, 0x85, 0x85}},
	"DDotrahd":                        {Name: "DDotrahd", CodePoints: []int{10513}, Characters: []byte{0xe2, 0xa4, 0x91}},
	"DJcy":                            {Name: "DJcy", CodePoints: []int{1026}, Characters: []byte{0xd0, 0x82}},
	"DScy":                            {Name: "DScy", CodePoints: []int{1029}, Characters: []byte{0xd0, 0x85}},
	"DZcy":                            {Name: "DZcy", CodePoints: []int{1039}, Characters: []byte{0xd0, 0x8f}},
	"Dagger":                          {Name: "Dagger", CodePoints: []int{8225}, Characters: []byte{0xe2, 0x80, 0xa1}},
	"Darr":                            {Name: "Darr", CodePoints: []int{8609}, Characters: []byte{0xe2, 0x86, 0xa1}},
	"Dashv":                           {Name: "Dashv", CodePoints: []int{10980}, Characters: []byte{0xe2, 0xab, 0xa4}},
	"Dcaron":                          {Name: "Dcaron", CodePoints: []int{270}, Characters: []byte{0xc4, 0x8e}},
	"Dcy":                             {Name: "Dcy", CodePoints: []int{1044}, Characters: []byte{0xd0, 0x94}},
	"Del":                             {Name: "Del", CodePoints: []int{8711}, Characters: []byte{0xe2, 0x88, 0x87}},
	"Delta":                           {Name: "Delta", CodePoints: []int{916}, Characters: []byte{0xce, 0x94}},
	"Dfr":                             {Name: "Dfr", CodePoints: []int{120071}, Characters: []byte{0xf0, 0x9d, 0x94, 0x87}},
	"DiacriticalAcute":                {Name: "DiacriticalAcute", CodePoints: []int{180}, Characters: []byte{0xc2, 0xb4}},
	"DiacriticalDot":                  {Name: "DiacriticalDot", CodePoints: []int{729}, Characters: []byte{0xcb, 0x99}},
	"DiacriticalDoubleAcute":          {Name: "DiacriticalDoubleAcute", CodePoints: []int{733}, Characters: []byte{0xcb, 0x9d}},
	"DiacriticalGrave":                {Name: "DiacriticalGrave", CodePoints: []int{96}, Characters: []byte{0x60}},
	"DiacriticalTilde":                {Name: "DiacriticalTilde", CodePoints: []int{732}, Characters: []byte{0xcb, 0x9c}},
	"Diamond":                         {Name: "Diamond", CodePoints: []int{8900}, Characters: []byte{0xe2, 0x8b, 0x84}},
	"DifferentialD":                   {Name: "DifferentialD", CodePoints: []int{8518}, Characters: []byte{0xe2, 0x85, 0x86}},
	"Dopf":                            {Name: "Dopf", CodePoints: []int{120123}, Characters: []byte{0xf0, 0x9d, 0x94, 0xbb}},
	"Dot":                             {Name: "Dot", CodePoints: []int{168}, Characters: []byte{0xc2, 0xa8}},
	"DotDot":                          {Name: "DotDot", CodePoints: []int{8412}, Characters: []byte{0xe2, 0x83, 0x9c}},
	"DotEqual":                        {Name: "DotEqual", CodePoints: []int{8784}, Characters: []byte{0xe2, 0x89, 0x90}},
	"DoubleContourIntegral":           {Name: "DoubleContourIntegral", CodePoints: []int{8751}, Characters: []byte{0xe2, 0x88, 0xaf}},
	"DoubleDot":                       {Name: "DoubleDot", CodePoints: []int{168}, Characters: []byte{0xc2, 0xa8}},
	"DoubleDownArrow":                 {Name: "DoubleDownArrow", CodePoints: []int{8659}, Characters: []byte{0xe2, 0x87, 0x93}},
	"DoubleLeftArrow":                 {Name: "DoubleLeftArrow", CodePoints: []int{8656}, Characters: []byte{0xe2, 0x87, 0x90}},
	"DoubleLeftRightArrow":            {Name: "DoubleLeftRightArrow", CodePoints: []int{8660}, Characters: []byte{0xe2, 0x87, 0x94}},
	"DoubleLeftTee":                   {Name: "DoubleLeftTee", CodePoints: []int{10980}, Characters: []byte{0xe2, 0xab, 0xa4}},
	"DoubleLongLeftArrow":             {Name: "DoubleLongLeftArrow", CodePoints: []int{10232}, Characters: []byte{0xe2, 0x9f, 0xb8}},
	"DoubleLongLeftRightArrow":        {Name: "DoubleLongLeftRightArrow", CodePoints: []int{10234}, Characters: []byte{0xe2, 0x9f, 0xba}},
	"DoubleLongRightArrow":            {Name: "DoubleLongRightArrow", CodePoints: []int{10233}, Characters: []byte{0xe2, 0x9f, 0xb9}},
	"DoubleRightArrow":                {Name: "DoubleRightArrow", CodePoints: []int{8658}, Characters: []byte{0xe2, 0x87, 0x92}},
	"DoubleRightTee":                  {Name: "DoubleRightTee", CodePoints: []int{8872}, Characters: []byte{0xe2, 0x8a, 0xa8}},
	"DoubleUpArrow":                   {Name: "DoubleUpArrow", CodePoints: []int{8657}, Characters: []byte{0xe2, 0x87, 0x91}},
	"DoubleUpDownArrow":               {Name: "DoubleUpDownArrow", CodePoints: []int{8661}, Characters: []byte{0xe2, 0x87, 0x95}},
	"DoubleVerticalBar":               {Name: "DoubleVerticalBar", CodePoints: []int{8741}, Characters: []byte{0xe2, 0x88, 0xa5}},
	"DownArrow":                       {Name: "DownArrow", CodePoints: []int{8595}, Characters: []byte{0xe2, 0x86, 0x93}},
	"DownArrowBar":                    {Name: "DownArrowBar", CodePoints: []int{10515}, Characters: []byte{0xe2, 0xa4, 0x93}},
	"DownArrowUpArrow":                {Name: "DownArrowUpArrow", CodePoints: []int{8693}, Characters: []byte{0xe2, 0x87, 0xb5}},
	"DownBreve":                       {Name: "DownBreve", CodePoints: []int{785}, Characters: []byte{0xcc, 0x91}},
	"DownLeftRightVector":             {Name: "DownLeftRightVector", CodePoints: []int{10576}, Characters: []byte{0xe2, 0xa5, 0x90}},
	"DownLeftTeeVector":               {Name: "DownLeftTeeVector", CodePoints: []int{10590}, Characters: []byte{0xe2, 0xa5, 0x9e}},
	"DownLeftVector":                  {Name: "DownLeftVector", CodePoints: []int{8637}, Characters: []byte{0xe2, 0x86, 0xbd}},
	"DownLeftVectorBar":               {Name: "DownLeftVectorBar", CodePoints: []int{10582}, Characters: []byte{0xe2, 0xa5, 0x96}},
	"DownRightTeeVector":              {Name: "DownRightTeeVector", CodePoints: []int{10591}, Characters: []byte{0xe2, 0xa5, 0x9f}},
	"DownRightVector":                 {Name: "DownRightVector", CodePoints: []int{8641}, Characters: []byte{0xe2, 0x87, 0x81}},
	"DownRightVectorBar":              {Name: "DownRightVectorBar", CodePoints: []int{10583}, Characters: []byte{0xe2, 0xa5, 0x97}},
	"DownTee":                         {Name: "DownTee", CodePoints: []int{8868}, Characters: []byte{0xe2, 0x8a, 0xa4}},
	"DownTeeArrow":                    {Name: "DownTeeArrow", CodePoints: []int{8615}, Characters: []byte{0xe2, 0x86, 0xa7}},
	"Downarrow":                       {Name: "Downarrow", CodePoints: []int{8659}, Characters: []byte{0xe2, 0x87, 0x93}},
	"Dscr":                            {Name: "Dscr", CodePoints: []int{119967}, Characters: []byte{0xf0, 0x9d, 0x92, 0x9f}},
	"Dstrok":                          {Name: "Dstrok", CodePoints: []int{272}, Characters: []byte{0xc4, 0x90}},
	"ENG":                             {Name: "ENG", CodePoints: []int{330}, Characters: []byte{0xc5, 0x8a}},
	"ETH":                             {Name: "ETH", CodePoints: []int{208}, Characters: []byte{0xc3, 0x90}},
	"Eacute":                          {Name: "Eacute", CodePoints: []int{201}, Characters: []byte{0xc3, 0x89}},
	"Ecaron":                          {Name: "Ecaron", CodePoints: []int{282}, Characters: []byte{0xc4, 0x9a}},
	"Ecirc":                           {Name: "Ecirc", CodePoints: []int{202}, Characters: []byte{0xc3, 0x8a}},
	"Ecy":                             {Name: "Ecy", CodePoints: []int{1069}, Characters: []byte{0xd0, 0xad}},
	"Edot":                            {Name: "Edot", CodePoints: []int{278}, Characters: []byte{0xc4, 0x96}},
	"Efr":                             {Name: "Efr", CodePoints: []int{120072}, Characters: []byte{0xf0, 0x9d, 0x94, 0x88}},
	"Egrave":                          {Name: "Egrave", CodePoints: []int{200}, Characters: []byte{0xc3, 0x88}},
	"Element":                         {Name: "Element", CodePoints: []int{8712}, Characters: []byte{0xe2, 0x88, 0x88}},
	"Emacr":                           {Name: "Emacr", CodePoints: []int{274}, Characters: []byte{0xc4, 0x92}},
	"EmptySmallSquare":                {Name: "EmptySmallSquare", CodePoints: []int{9723}, Characters: []byte{0xe2, 0x97, 0xbb}},
	"EmptyVerySmallSquare":            {Name: "EmptyVerySmallSquare", CodePoints: []int{9643}, Characters: []byte{0xe2, 0x96, 0xab}},
	"Eogon":                           {Name: "Eogon", CodePoints: []int{280}, Characters: []byte{0xc4, 0x98}},
	"Eopf":                            {Name: "Eopf", CodePoints: []int{120124}, Characters: []byte{0xf0, 0x9d, 0x94, 0xbc}},
	"Epsilon":                         {Name: "Epsilon", CodePoints: []int{917}, Characters: []byte{0xce, 0x95}},
	"Equal":                           {Name: "Equal", CodePoints: []int{10869}, Characters: []byte{0xe2, 0xa9, 0xb5}},
	"EqualTilde":                      {Name: "EqualTilde", CodePoints: []int{8770}, Characters: []byte{0xe2, 0x89, 0x82}},
	"Equilibrium":                     {Name: "Equilibrium", CodePoints: []int{8652}, Characters: []byte{0xe2, 0x87, 0x8c}},
	"Escr":                            {Name: "Escr", CodePoints: []int{8496}, Characters: []byte{0xe2, 0x84, 0xb0}},
	"Esim":                            {Name: "Esim", CodePoints: []int{10867}, Characters: []byte{0xe2, 0xa9, 0xb3}},
	"Eta":                             {Name: "Eta", CodePoints: []int{919}, Characters: []byte{0xce, 0x97}},
	"Euml":                            {Name: "Euml", CodePoints: []int{203}, Characters: []byte{0xc3, 0x8b}},
	"Exists":                          {Name: "Exists", CodePoints: []int{8707}, Characters: []byte{0xe2, 0x88, 0x83}},
	"ExponentialE":                    {Name: "ExponentialE", CodePoints: []int{8519}, Characters: []byte{0xe2, 0x85, 0x87}},
	"Fcy":                             {Name: "Fcy", CodePoints: []int{1060}, Characters: []byte{0xd0, 0xa4}},
	"Ffr":                             {Name: "Ffr", CodePoints: []int{120073}, Characters: []byte{0xf0, 0x9d, 0x94, 0x89}},
	"FilledSmallSquare":               {Name: "FilledSmallSquare", CodePoints: []int{9724}, Characters: []byte{0xe2, 0x97, 0xbc}},
	"FilledVerySmallSquare":           {Name: "FilledVerySmallSquare", CodePoints: []int{9642}, Characters: []byte{0xe2, 0x96, 0xaa}},
	"Fopf":                            {Name: "Fopf", CodePoints: []int{120125}, Characters: []byte{0xf0, 0x9d, 0x94, 0xbd}},
	"ForAll":                          {Name: "ForAll", CodePoints: []int{8704}, Characters: []byte{0xe2, 0x88, 0x80}},
	"Fouriertrf":                      {Name: "Fouriertrf", CodePoints: []int{8497}, Characters: []byte{0xe2, 0x84, 0xb1}},
	"Fscr":                            {Name: "Fscr", CodePoints: []int{8497}, Characters: []byte{0xe2, 0x84, 0xb1}},
	"GJcy":                            {Name: "GJcy", CodePoints: []int{1027}, Characters: []byte{0xd0, 0x83}},
	"GT":                              {Name: "GT", CodePoints: []int{62}, Characters: []byte{0x3e}},
	"Gamma":                           {Name: "Gamma", CodePoints: []int{915}, Characters: []byte{0xce, 0x93}},
	"Gammad":                          {Name: "Gammad", CodePoints: []int{988}, Characters: []byte{0xcf, 0x9c}},
	"Gbreve":                          {Name: "Gbreve", CodePoints: []int{286}, Characters: []byte{0xc4, 0x9e}},
	"Gcedil":                          {Name: "Gcedil", CodePoints: []int{290}, Characters: []byte{0xc4, 0xa2}},
	"Gcirc":                           {Name: "Gcirc", CodePoints: []int{284}, Characters: []byte{0xc4, 0x9c}},
	"Gcy":                             {Name: "Gcy", CodePoints: []int{1043}, Characters: []byte{0xd0, 0x93}},
	"Gdot":                            {Name: "Gdot", CodePoints: []int{288}, Characters: []byte{0xc4, 0xa0}},
	"Gfr":                             {Name: "Gfr", CodePoints: []int{120074}, Characters: []byte{0xf0, 0x9d, 0x94, 0x8a}},
	"Gg":                              {Name: "Gg", CodePoints: []int{8921}, Characters: []byte{0xe2, 0x8b, 0x99}},
	"Gopf":                            {Name: "Gopf", CodePoints: []int{120126}, Characters: []byte{0xf0, 0x9d, 0x94, 0xbe}},
	"GreaterEqual":                    {Name: "GreaterEqual", CodePoints: []int{8805}, Characters: []byte{0xe2, 0x89, 0xa5}},
	"GreaterEqualLess":                {Name: "GreaterEqualLess", CodePoints: []int{8923}, Characters: []byte{0xe2, 0x8b, 0x9b}},
	"GreaterFullEqual":                {Name: "GreaterFullEqual", CodePoints: []int{8807}, Characters: []byte{0xe2, 0x89, 0xa7}},
	"GreaterGreater":                  {Name: "GreaterGreater", CodePoints: []int{10914}, Characters: []byte{0xe2, 0xaa, 0xa2}},
	"GreaterLess":                     {Name: "GreaterLess", CodePoints: []int{8823}, Characters: []byte{0xe2, 0x89, 0xb7}},
	"GreaterSlantEqual":               {Name: "GreaterSlantEqual", CodePoints: []int{10878}, Characters: []byte{0xe2, 0xa9, 0xbe}},
	"GreaterTilde":                    {Name: "GreaterTilde", CodePoints: []int{8819}, Characters: []byte{0xe2, 0x89, 0xb3}},
	"Gscr":                            {Name: "Gscr", CodePoints: []int{119970}, Characters: []byte{0xf0, 0x9d, 0x92, 0xa2}},
	"Gt":                              {Name: "Gt", CodePoints: []int{8811}, Characters: []byte{0xe2, 0x89, 0xab}},
	"HARDcy":                          {Name: "HARDcy", CodePoints: []int{1066}, Characters: []byte{0xd0, 0xaa}},
	"Hacek":                           {Name: "Hacek", CodePoints: []int{711}, Characters: []byte{0xcb, 0x87}},
	"Hat":                             {Name: "Hat", CodePoints: []int{94}, Characters: []byte{0x5e}},
	"Hcirc":                           {Name: "Hcirc", CodePoints: []int{292}, Characters: []byte{0xc4, 0xa4}},
	"Hfr":                             {Name: "Hfr", CodePoints: []int{8460}, Characters: []byte{0xe2, 0x84, 0x8c}},
	"HilbertSpace":                    {Name: "HilbertSpace", CodePoints: []int{8459}, Characters: []byte{0xe2, 0x84, 0x8b}},
	"Hopf":                            {Name: "Hopf", CodePoints: []int{8461}, Characters: []byte{0xe2, 0x84, 0x8d}},
	"HorizontalLine":                  {Name: "HorizontalLine", CodePoints: []int{9472}, Characters: []byte{0xe2, 0x94, 0x80}},
	"Hscr":                            {Name: "Hscr", CodePoints: []int{8459}, Characters: []byte{0xe2, 0x84, 0x8b}},
	"Hstrok":                          {Name: "Hstrok", CodePoints: []int{294}, Characters: []byte{0xc4, 0xa6}},
	"HumpDownHump":                    {Name: "HumpDownHump", CodePoints: []int{8782}, Characters: []byte{0xe2, 0x89, 0x8e}},
	"HumpEqual":                       {Name: "HumpEqual", CodePoints: []int{8783}, Characters: []byte{0xe2, 0x89, 0x8f}},
	"IEcy":                            {Name: "IEcy", CodePoints: []int{1045}, Characters: []byte{0xd0, 0x95}},
	"IJlig":                           {Name: "IJlig", CodePoints: []int{306}, Characters: []byte{0xc4, 0xb2}},
	"IOcy":                            {Name: "IOcy", CodePoints: []int{1025}, Characters: []byte{0xd0, 0x81}},
	"Iacute":                          {Name: "Iacute", CodePoints: []int{205}, Characters: []byte{0xc3, 0x8d}},
	"Icirc":                           {Name: "Icirc", CodePoints: []int{206}, Characters: []byte{0xc3, 0x8e}},
	"Icy":                             {Name: "Icy", CodePoints: []int{1048}, Characters: []byte{0xd0, 0x98}},
	"Idot":                            {Name: "Idot", CodePoints: []int{304}, Characters: []byte{0xc4, 0xb0}},
	"Ifr":                             {Name: "Ifr", CodePoints: []int{8465}, Characters: []byte{0xe2, 0x84, 0x91}},
	"Igrave":                          {Name: "Igrave", CodePoints: []int{204}, Characters: []byte{0xc3, 0x8c}},
	"Im":                              {Name: "Im", CodePoints: []int{8465}, Characters: []byte{0xe2, 0x84, 0x91}},
	"Imacr":                           {Name: "Imacr", CodePoints: []int{298}, Characters: []byte{0xc4, 0xaa}},
	"ImaginaryI":                      {Name: "ImaginaryI", CodePoints: []int{8520}, Characters: []byte{0xe2, 0x85, 0x88}},
	"Implies":                         {Name: "Implies", CodePoints: []int{8658}, Characters: []byte{0xe2, 0x87, 0x92}},
	"Int":                             {Name: "Int", CodePoints: []int{8748}, Characters: []byte{0xe2, 0x88, 0xac}},
	"Integral":                        {Name: "Integral", CodePoints: []int{8747}, Characters: []byte{0xe2, 0x88, 0xab}},
	"Intersection":                    {Name: "Intersection", CodePoints: []int{8898}, Characters: []byte{0xe2, 0x8b, 0x82}},
	"InvisibleComma":                  {Name: "InvisibleComma", CodePoints: []int{8291}, Characters: []byte{0xe2, 0x81, 0xa3}},
	"InvisibleTimes":                  {Name: "InvisibleTimes", CodePoints: []int{8290}, Characters: []byte{0xe2, 0x81, 0xa2}},
	"Iogon":                           {Name: "Iogon", CodePoints: []int{302}, Characters: []byte{0xc4, 0xae}},
	"Iopf":                            {Name: "Iopf", CodePoints: []int{120128}, Characters: []byte{0xf0, 0x9d, 0x95, 0x80}},
	"Iota":                            {Name: "Iota", CodePoints: []int{921}, Characters: []byte{0xce, 0x99}},
	"Iscr":                            {Name: "Iscr", CodePoints: []int{8464}, Characters: []byte{0xe2, 0x84, 0x90}},
	"Itilde":                          {Name: "Itilde", CodePoints: []int{296}, Characters: []byte{0xc4, 0xa8}},
	"Iukcy":                           {Name: "Iukcy", CodePoints: []int{1030}, Characters: []byte{0xd0, 0x86}},
	"Iuml":                            {Name: "Iuml", CodePoints: []int{207}, Characters: []byte{0xc3, 0x8f}},
	"Jcirc":                           {Name: "Jcirc", CodePoints: []int{308}, Characters: []byte{0xc4, 0xb4}},
	"Jcy":                             {Name: "Jcy", CodePoints: []int{1049}, Characters: []byte{0xd0, 0x99}},
	"Jfr":                             {Name: "Jfr", CodePoints: []int{120077}, Characters: []byte{0xf0, 0x9d, 0x94, 0x8d}},
	"Jopf":                            {Name: "Jopf", CodePoints: []int{120129}, Characters: []byte{0xf0, 0x9d, 0x95, 0x81}},
	"Jscr":                            {Name: "Jscr", CodePoints: []int{119973}, Characters: []byte{0xf0, 0x9d, 0x92, 0xa5}},
	"Jsercy":                          {Name: "Jsercy", CodePoints: []int{1032}, Characters: []byte{0xd0, 0x88}},
	"Jukcy":                           {Name: "Jukcy", CodePoints: []int{1028}, Characters: []byte{0xd0, 0x84}},
	"KHcy":                            {Name: "KHcy", CodePoints: []int{1061}, Characters: []byte{0xd0, 0xa5}},
	"KJcy":                            {Name: "KJcy", CodePoints: []int{1036}, Characters: []byte{0xd0, 0x8c}},
	"Kappa":                           {Name: "Kappa", CodePoints: []int{922}, Characters: []byte{0xce, 0x9a}},
	"Kcedil":                          {Name: "Kcedil", CodePoints: []int{310}, Characters: []byte{0xc4, 0xb6}},
	"Kcy":                             {Name: "Kcy", CodePoints: []int{1050}, Characters: []byte{0xd0, 0x9a}},
	"Kfr":                             {Name: "Kfr", CodePoints: []int{120078}, Characters: []byte{0xf0, 0x9d, 0x94, 0x8e}},
	"Kopf":                            {Name: "Kopf", CodePoints: []int{120130}, Characters: []byte{0xf0, 0x9d, 0x95, 0x82}},
	"Kscr":                            {Name: "Kscr", CodePoints: []int{119974}, Characters: []byte{0xf0, 0x9d, 0x92, 0xa6}},
	"LJcy":                            {Name: "LJcy", CodePoints: []int{1033}, Characters: []byte{0xd0, 0x89}},
	"LT":                              {Name: "LT", CodePoints: []int{60}, Characters: []byte{0x3c}},
	"Lacute":                          {Name: "Lacute", CodePoints: []int{313}, Characters: []byte{0xc4, 0xb9}},
	"Lambda":                          {Name: "Lambda", CodePoints: []int{923}, Characters: []byte{0xce, 0x9b}},
	"Lang":                            {Name: "Lang", CodePoints: []int{10218}, Characters: []byte{0xe2, 0x9f, 0xaa}},
	"Laplacetrf":                      {Name: "Laplacetrf", CodePoints: []int{8466}, Characters: []byte{0xe2, 0x84, 0x92}},
	"Larr":                            {Name: "Larr", CodePoints: []int{8606}, Characters: []byte{0xe2, 0x86, 0x9e}},
	"Lcaron":                          {Name: "Lcaron", CodePoints: []int{317}, Characters: []byte{0xc4, 0xbd}},
	"Lcedil":                          {Name: "Lcedil", CodePoints: []int{315}, Characters: []byte{0xc4, 0xbb}},
	"Lcy":                             {Name: "Lcy", CodePoints: []int{1051}, Characters: []byte{0xd0, 0x9b}},
	"LeftAngleBracket":                {Name: "LeftAngleBracket", CodePoints: []int{10216}, Characters: []byte{0xe2, 0x9f, 0xa8}},
	"LeftArrow":                       {Name: "LeftArrow", CodePoints: []int{8592}, Characters: []byte{0xe2, 0x86, 0x90}},
	"LeftArrowBar":                    {Name: "LeftArrowBar", CodePoints: []int{8676}, Characters: []byte{0xe2, 0x87, 0xa4}},
	"LeftArrowRightArrow":             {Name: "LeftArrowRightArrow", CodePoints: []int{8646}, Characters: []byte{0xe2, 0x87, 0x86}},
	"LeftCeiling":                     {Name: "LeftCeiling", CodePoints: []int{8968}, Characters: []byte{0xe2, 0x8c, 0x88}},
	"LeftDoubleBracket":               {Name: "LeftDoubleBracket", CodePoints: []int{10214}, Characters: []byte{0xe2, 0x9f, 0xa6}},
	"LeftDownTeeVector":               {Name: "LeftDownTeeVector", CodePoints: []int{10593}, Characters: []byte{0xe2, 0xa5, 0xa1}},
	"LeftDownVector":                  {Name: "LeftDownVector", CodePoints: []int{8643}, Characters: []byte{0xe2, 0x87, 0x83}},
	"LeftDownVectorBar":               {Name: "LeftDownVectorBar", CodePoints: []int{10585}, Characters: []byte{0xe2, 0xa5, 0x99}},
	"LeftFloor":                       {Name: "LeftFloor", CodePoints: []int{8970}, Characters: []byte{0xe2, 0x8c, 0x8a}},
	"LeftRightArrow":                  {Name: "LeftRightArrow", CodePoints: []int{8596}, Characters: []byte{0xe2, 0x86, 0x94}},
	"LeftRightVector":                 {Name: "LeftRightVector", CodePoints: []int{10574}, Characters: []byte{0xe2, 0xa5, 0x8e}},
	"LeftTee":                         {Name: "LeftTee", CodePoints: []int{8867}, Characters: []byte{0xe2, 0x8a, 0xa3}},
	"LeftTeeArrow":                    {Name: "LeftTeeArrow", CodePoints: []int{8612}, Characters: []byte{0xe2, 0x86, 0xa4}},
	"LeftTeeVector":                   {Name: "LeftTeeVector", CodePoints: []int{10586}, Characters: []byte{0xe2, 0xa5, 0x9a}},
	"LeftTriangle":                    {Name: "LeftTriangle", CodePoints: []int{8882}, Characters: []byte{0xe2, 0x8a, 0xb2}},
	"LeftTriangleBar":                 {Name: "LeftTriangleBar", CodePoints: []int{10703}, Characters: []byte{0xe2, 0xa7, 0x8f}},
	"LeftTriangleEqual":               {Name: "LeftTriangleEqual", CodePoints: []int{8884}, Characters: []byte{0xe2, 0x8a, 0xb4}},
	"LeftUpDownVector":                {Name: "LeftUpDownVector", CodePoints: []int{10577}, Characters: []byte{0xe2, 0xa5, 0x91}},
	"LeftUpTeeVector":                 {Name: "LeftUpTeeVector", CodePoints: []int{10592}, Characters: []byte{0xe2, 0xa5, 0xa0}},
	"LeftUpVector":                    {Name: "LeftUpVector", CodePoints: []int{8639}, Characters: []byte{0xe2, 0x86, 0xbf}},
	"LeftUpVectorBar":                 {Name: "LeftUpVectorBar", CodePoints: []int{10584}, Characters: []byte{0xe2, 0xa5, 0x98}},
	"LeftVector":                      {Name: "LeftVector", CodePoints: []int{8636}, Characters: []byte{0xe2, 0x86, 0xbc}},
	"LeftVectorBar":                   {Name: "LeftVectorBar", CodePoints: []int{10578}, Characters: []byte{0xe2, 0xa5, 0x92}},
	"Leftarrow":                       {Name: "Leftarrow", CodePoints: []int{8656}, Characters: []byte{0xe2, 0x87, 0x90}},
	"Leftrightarrow":                  {Name: "Leftrightarrow", CodePoints: []int{8660}, Characters: []byte{0xe2, 0x87, 0x94}},
	"LessEqualGreater":                {Name: "LessEqualGreater", CodePoints: []int{8922}, Characters: []byte{0xe2, 0x8b, 0x9a}},
	"LessFullEqual":                   {Name: "LessFullEqual", CodePoints: []int{8806}, Characters: []byte{0xe2, 0x89, 0xa6}},
	"LessGreater":                     {Name: "LessGreater", CodePoints: []int{8822}, Characters: []byte{0xe2, 0x89, 0xb6}},
	"LessLess":                        {Name: "LessLess", CodePoints: []int{10913}, Characters: []byte{0xe2, 0xaa, 0xa1}},
	"LessSlantEqual":                  {Name: "LessSlantEqual", CodePoints: []int{10877}, Characters: []byte{0xe2, 0xa9, 0xbd}},
	"LessTilde":                       {Name: "LessTilde", CodePoints: []int{8818}, Characters: []byte{0xe2, 0x89, 0xb2}},
	"Lfr":                             {Name: "Lfr", CodePoints: []int{120079}, Characters: []byte{0xf0, 0x9d, 0x94, 0x8f}},
	"Ll":                              {Name: "Ll", CodePoints: []int{8920}, Characters: []byte{0xe2, 0x8b, 0x98}},
	"Lleftarrow":                      {Name: "Lleftarrow", CodePoints: []int{8666}, Characters: []byte{0xe2, 0x87, 0x9a}},
	"Lmidot":                          {Name: "Lmidot", CodePoints: []int{319}, Characters: []byte{0xc4, 0xbf}},
	"LongLeftArrow":                   {Name: "LongLeftArrow", CodePoints: []int{10229}, Characters: []byte{0xe2, 0x9f, 0xb5}},
	"LongLeftRightArrow":              {Name: "LongLeftRightArrow", CodePoints: []int{10231}, Characters: []byte{0xe2, 0x9f, 0xb7}},
	"LongRightArrow":                  {Name: "LongRightArrow", CodePoints: []int{10230}, Characters: []byte{0xe2, 0x9f, 0xb6}},
	"Longleftarrow":                   {Name: "Longleftarrow", CodePoints: []int{10232}, Characters: []byte{0xe2, 0x9f, 0xb8}},
	"Longleftrightarrow":              {Name: "Longleftrightarrow", CodePoints: []int{10234}, Characters: []byte{0xe2, 0x9f, 0xba}},
	"Longrightarrow":                  {Name: "Longrightarrow", CodePoints: []int{10233}, Characters: []byte{0xe2, 0x9f, 0xb9}},
	"Lopf":                            {Name: "Lopf", CodePoints: []int{120131}, Characters: []byte{0xf0, 0x9d, 0x95, 0x83}},
	"LowerLeftArrow":                  {Name: "LowerLeftArrow", CodePoints: []int{8601}, Characters: []byte{0xe2, 0x86, 0x99}},
	"LowerRightArrow":                 {Name: "LowerRightArrow", CodePoints: []int{8600}, Characters: []byte{0xe2, 0x86, 0x98}},
	"Lscr":                            {Name: "Lscr", CodePoints: []int{8466}, Characters: []byte{0xe2, 0x84, 0x92}},
	"Lsh":                             {Name: "Lsh", CodePoints: []int{8624}, Characters: []byte{0xe2, 0x86, 0xb0}},
	"Lstrok":                          {Name: "Lstrok", CodePoints: []int{321}, Characters: []byte{0xc5, 0x81}},
	"Lt":                              {Name: "Lt", CodePoints: []int{8810}, Characters: []byte{0xe2, 0x89, 0xaa}},
	"Map":                             {Name: "Map", CodePoints: []int{10501}, Characters: []byte{0xe2, 0xa4, 0x85}},
	"Mcy":                             {Name: "Mcy", CodePoints: []int{1052}, Characters: []byte{0xd0, 0x9c}},
	"MediumSpace":                     {Name: "MediumSpace", CodePoints: []int{8287}, Characters: []byte{0xe2, 0x81, 0x9f}},
	"Mellintrf":                       {Name: "Mellintrf", CodePoints: []int{8499}, Characters: []byte{0xe2, 0x84, 0xb3}},
	"Mfr":                             {Name: "Mfr", CodePoints: []int{120080}, Characters: []byte{0xf0, 0x9d, 0x94, 0x90}},
	"MinusPlus":                       {Name: "MinusPlus", CodePoints: []int{8723}, Characters: []byte{0xe2, 0x88, 0x93}},
	"Mopf":                            {Name: "Mopf", CodePoints: []int{120132}, Characters: []byte{0xf0, 0x9d, 0x95, 0x84}},
	"Mscr":                            {Name: "Mscr", CodePoints: []int{8499}, Characters: []byte{0xe2, 0x84, 0xb3}},
	"Mu":                              {Name: "Mu", CodePoints: []int{924}, Characters: []byte{0xce, 0x9c}},
	"NJcy":                            {Name: "NJcy", CodePoints: []int{1034}, Characters: []byte{0xd0, 0x8a}},
	"Nacute":                          {Name: "Nacute", CodePoints: []int{323}, Characters: []byte{0xc5, 0x83}},
	"Ncaron":                          {Name: "Ncaron", CodePoints: []int{327}, Characters: []byte{0xc5, 0x87}},
	"Ncedil":                          {Name: "Ncedil", CodePoints: []int{325}, Characters: []byte{0xc5, 0x85}},
	"Ncy":                             {Name: "Ncy", CodePoints: []int{1053}, Characters: []byte{0xd0, 0x9d}},
	"NegativeMediumSpace":             {Name: "NegativeMediumSpace", CodePoints: []int{8203}, Characters: []byte{0xe2, 0x80, 0x8b}},
	"NegativeThickSpace":              {Name: "NegativeThickSpace", CodePoints: []int{8203}, Characters: []byte{0xe2, 0x80, 0x8b}},
	"NegativeThinSpace":               {Name: "NegativeThinSpace", CodePoints: []int{8203}, Characters: []byte{0xe2, 0x80, 0x8b}},
	"NegativeVeryThinSpace":           {Name: "NegativeVeryThinSpace", CodePoints: []int{8203}, Characters: []byte{0xe2, 0x80, 0x8b}},
	"NestedGreaterGreater":            {Name: "NestedGreaterGreater", CodePoints: []int{8811}, Characters: []byte{0xe2, 0x89, 0xab}},
	"NestedLessLess":                  {Name: "NestedLessLess", CodePoints: []int{8810}, Characters: []byte{0xe2, 0x89, 0xaa}},
	"NewLine":                         {Name: "NewLine", CodePoints: []int{10}, Characters: []byte{0xa}},
	"Nfr":                             {Name: "Nfr", CodePoints: []int{120081}, Characters: []byte{0xf0, 0x9d, 0x94, 0x91}},
	"NoBreak":                         {Name: "NoBreak", CodePoints: []int{8288}, Characters: []byte{0xe2, 0x81, 0xa0}},
	"NonBreakingSpace":                {Name: "NonBreakingSpace", CodePoints: []int{160}, Characters: []byte{0xc2, 0xa0}},
	"Nopf":                            {Name: "Nopf", CodePoints: []int{8469}, Characters: []byte{0xe2, 0x84, 0x95}},
	"Not":                             {Name: "Not", CodePoints: []int{10988}, Characters: []byte{0xe2, 0xab, 0xac}},
	"NotCongruent":                    {Name: "NotCongruent", CodePoints: []int{8802}, Characters: []byte{0xe2, 0x89, 0xa2}},
	"NotCupCap":                       {Name: "NotCupCap", CodePoints: []int{8813}, Characters: []byte{0xe2, 0x89, 0xad}},
	"NotDoubleVerticalBar":            {Name: "NotDoubleVerticalBar", CodePoints: []int{8742}, Characters: []byte{0xe2, 0x88, 0xa6}},
	"NotElement":                      {Name: "NotElement", CodePoints: []int{8713}, Characters: []byte{0xe2, 0x88, 0x89}},
	"NotEqual":                        {Name: "NotEqual", CodePoints: []int{8800}, Characters: []byte{0xe2, 0x89, 0xa0}},
	"NotEqualTilde":                   {Name: "NotEqualTilde", CodePoints: []int{8770, 824}, Characters: []byte{0xe2, 0x89, 0x82, 0xcc, 0xb8}},
	"NotExists":                       {Name: "NotExists", CodePoints: []int{8708}, Characters: []byte{0xe2, 0x88, 0x84}},
	"NotGreater":                      {Name: "NotGreater", CodePoints: []int{8815}, Characters: []byte{0xe2, 0x89, 0xaf}},
	"NotGreaterEqual":                 {Name: "NotGreaterEqual", CodePoints: []int{8817}, Characters: []byte{0xe2, 0x89, 0xb1}},
	"NotGreaterFullEqual":             {Name: "NotGreaterFullEqual", CodePoints: []int{8807, 824}, Characters: []byte{0xe2, 0x89, 0xa7, 0xcc, 0xb8}},
	"NotGreaterGreater":               {Name: "NotGreaterGreater", CodePoints: []int{8811, 824}, Characters: []byte{0xe2, 0x89, 0xab, 0xcc, 0xb8}},
	"NotGreaterLess":                  {Name: "NotGreaterLess", CodePoints: []int{8825}, Characters: []byte{0xe2, 0x89, 0xb9}},
	"NotGreaterSlantEqual":            {Name: "NotGreaterSlantEqual", CodePoints: []int{10878, 824}, Characters: []byte{0xe2, 0xa9, 0xbe, 0xcc, 0xb8}},
	"NotGreaterTilde":                 {Name: "NotGreaterTilde", CodePoints: []int{8821}, Characters: []byte{0xe2, 0x89, 0xb5}},
	"NotHumpDownHump":                 {Name: "NotHumpDownHump", CodePoints: []int{8782, 824}, Characters: []byte{0xe2, 0x89, 0x8e, 0xcc, 0xb8}},
	"NotHumpEqual":                    {Name: "NotHumpEqual", CodePoints: []int{8783, 824}, Characters: []byte{0xe2, 0x89, 0x8f, 0xcc, 0xb8}},
	"NotLeftTriangle":                 {Name: "NotLeftTriangle", CodePoints: []int{8938}, Characters: []byte{0xe2, 0x8b, 0xaa}},
	"NotLeftTriangleBar":              {Name: "NotLeftTriangleBar", CodePoints: []int{10703, 824}, Characters: []byte{0xe2, 0xa7, 0x8f, 0xcc, 0xb8}},
	"NotLeftTriangleEqual":            {Name: "NotLeftTriangleEqual", CodePoints: []int{8940}, Characters: []byte{0xe2, 0x8b, 0xac}},
	"NotLess":                         {Name: "NotLess", CodePoints: []int{8814}, Characters: []byte{0xe2, 0x89, 0xae}},
	"NotLessEqual":                    {Name: "NotLessEqual", CodePoints: []int{8816}, Characters: []byte{0xe2, 0x89, 0xb0}},
	"NotLessGreater":                  {Name: "NotLessGreater", CodePoints: []int{8824}, Characters: []byte{0xe2, 0x89, 0xb8}},
	"NotLessLess":                     {Name: "NotLessLess", CodePoints: []int{8810, 824}, Characters: []byte{0xe2, 0x89, 0xaa, 0xcc, 0xb8}},
	"NotLessSlantEqual":               {Name: "NotLessSlantEqual", CodePoints: []int{10877, 824}, Characters: []byte{0xe2, 0xa9, 0xbd, 0xcc, 0xb8}},
	"NotLessTilde":                    {Name: "NotLessTilde", CodePoints: []int{8820}, Characters: []byte{0xe2, 0x89, 0xb4}},
	"NotNestedGreaterGreater":         {Name: "NotNestedGreaterGreater", CodePoints: []int{10914, 824}, Characters: []byte{0xe2, 0xaa, 0xa2, 0xcc, 0xb8}},
	"NotNestedLessLess":               {Name: "NotNestedLessLess", CodePoints: []int{10913, 824}, Characters: []byte{0xe2, 0xaa, 0xa1, 0xcc, 0xb8}},
	"NotPrecedes":                     {Name: "NotPrecedes", CodePoints: []int{8832}, Characters: []byte{0xe2, 0x8a, 0x80}},
	"NotPrecedesEqual":                {Name: "NotPrecedesEqual", CodePoints: []int{10927, 824}, Characters: []byte{0xe2, 0xaa, 0xaf, 0xcc, 0xb8}},
	"NotPrecedesSlantEqual":           {Name: "NotPrecedesSlantEqual", CodePoints: []int{8928}, Characters: []byte{0xe2, 0x8b, 0xa0}},
	"NotReverseElement":               {Name: "NotReverseElement", CodePoints: []int{8716}, Characters: []byte{0xe2, 0x88, 0x8c}},
	"NotRightTriangle":                {Name: "NotRightTriangle", CodePoints: []int{8939}, Characters: []byte{0xe2, 0x8b, 0xab}},
	"NotRightTriangleBar":             {Name: "NotRightTriangleBar", CodePoints: []int{10704, 824}, Characters: []byte{0xe2, 0xa7, 0x90, 0xcc, 0xb8}},
	"NotRightTriangleEqual":           {Name: "NotRightTriangleEqual", CodePoints: []int{8941}, Characters: []byte{0xe2, 0x8b, 0xad}},
	"NotSquareSubset":                 {Name: "NotSquareSubset", CodePoints: []int{8847, 824}, Characters: []byte{0xe2, 0x8a, 0x8f, 0xcc, 0xb8}},
	"NotSquareSubsetEqual":            {Name: "NotSquareSubsetEqual", CodePoints: []int{8930}, Characters: []byte{0xe2, 0x8b, 0xa2}},
	"NotSquareSuperset":               {Name: "NotSquareSuperset", CodePoints: []int{8848, 824}, Characters: []byte{0xe2, 0x8a, 0x90, 0xcc, 0xb8}},
	"NotSquareSupersetEqual":          {Name: "NotSquareSupersetEqual", CodePoints: []int{8931}, Characters: []byte{0xe2, 0x8b, 0xa3}},
	"NotSubset":                       {Name: "NotSubset", CodePoints: []int{8834, 8402}, Characters: []byte{0xe2, 0x8a, 0x82, 0xe2, 0x83, 0x92}},
	"NotSubsetEqual":                  {Name: "NotSubsetEqual", CodePoints: []int{8840}, Characters: []byte{0xe2, 0x8a, 0x88}},
	"NotSucceeds":                     {Name: "NotSucceeds", CodePoints: []int{8833}, Characters: []byte{0xe2, 0x8a, 0x81}},
	"NotSucceedsEqual":                {Name: "NotSucceedsEqual", CodePoints: []int{10928, 824}, Characters: []byte{0xe2, 0xaa, 0xb0, 0xcc, 0xb8}},
	"NotSucceedsSlantEqual":           {Name: "NotSucceedsSlantEqual", CodePoints: []int{8929}, Characters: []byte{0xe2, 0x8b, 0xa1}},
	"NotSucceedsTilde":                {Name: "NotSucceedsTilde", CodePoints: []int{8831, 824}, Characters: []byte{0xe2, 0x89, 0xbf, 0xcc, 0xb8}},
	"NotSuperset":                     {Name: "NotSuperset", CodePoints: []int{8835, 8402}, Characters: []byte{0xe2, 0x8a, 0x83, 0xe2, 0x83, 0x92}},
	"NotSupersetEqual":                {Name: "NotSupersetEqual", CodePoints: []int{8841}, Characters: []byte{0xe2, 0x8a, 0x89}},
	"NotTilde":                        {Name: "NotTilde", CodePoints: []int{8769}, Characters: []byte{0xe2, 0x89, 0x81}},
	"NotTildeEqual":                   {Name: "NotTildeEqual", CodePoints: []int{8772}, Characters: []byte{0xe2, 0x89, 0x84}},
	"NotTildeFullEqual":               {Name: "NotTildeFullEqual", CodePoints: []int{8775}, Characters: []byte{0xe2, 0x89, 0x87}},
	"NotTildeTilde":                   {Name: "NotTildeTilde", CodePoints: []int{8777}, Characters: []byte{0xe2, 0x89, 0x89}},
	"NotVerticalBar":                  {Name: "NotVerticalBar", CodePoints: []int{8740}, Characters: []byte{0xe2, 0x88, 0xa4}},
	"Nscr":                            {Name: "Nscr", CodePoints: []int{119977}, Characters: []byte{0xf0, 0x9d, 0x92, 0xa9}},
	"Ntilde":                          {Name: "Ntilde", CodePoints: []int{209}, Characters: []byte{0xc3, 0x91}},
	"Nu":                              {Name: "Nu", CodePoints: []int{925}, Characters: []byte{0xce, 0x9d}},
	"OElig":                           {Name: "OElig", CodePoints: []int{338}, Characters: []byte{0xc5, 0x92}},
	"Oacute":                          {Name: "Oacute", CodePoints: []int{211}, Characters: []byte{0xc3, 0x93}},
	"Ocirc":                           {Name: "Ocirc", CodePoints: []int{212}, Characters: []byte{0xc3, 0x94}},
	"Ocy":                             {Name: "Ocy", CodePoints: []int{1054}, Characters: []byte{0xd0, 0x9e}},
	"Odblac":                          {Name: "Odblac", CodePoints: []int{336}, Characters: []byte{0xc5, 0x90}},
	"Ofr":                             {Name: "Ofr", CodePoints: []int{120082}, Characters: []byte{0xf0, 0x9d, 0x94, 0x92}},
	"Ograve":                          {Name: "Ograve", CodePoints: []int{210}, Characters: []byte{0xc3, 0x92}},
	"Omacr":                           {Name: "Omacr", CodePoints: []int{332}, Characters: []byte{0xc5, 0x8c}},
	"Omega":                           {Name: "Omega", CodePoints: []int{937}, Characters: []byte{0xce, 0xa9}},
	"Omicron":                         {Name: "Omicron", CodePoints: []int{927}, Characters: []byte{0xce, 0x9f}},
	"Oopf":                            {Name: "Oopf", CodePoints: []int{120134}, Characters: []byte{0xf0, 0x9d, 0x95, 0x86}},
	"OpenCurlyDoubleQuote":            {Name: "OpenCurlyDoubleQuote", CodePoints: []int{8220}, Characters: []byte{0xe2, 0x80, 0x9c}},
	"OpenCurlyQuote":                  {Name: "OpenCurlyQuote", CodePoints: []int{8216}, Characters: []byte{0xe2, 0x80, 0x98}},
	"Or":                              {Name: "Or", CodePoints: []int{10836}, Characters: []byte{0xe2, 0xa9, 0x94}},
	"Oscr":                            {Name: "Oscr", CodePoints: []int{119978}, Characters: []byte{0xf0, 0x9d, 0x92, 0xaa}},
	"Oslash":                          {Name: "Oslash", CodePoints: []int{216}, Characters: []byte{0xc3, 0x98}},
	"Otilde":                          {Name: "Otilde", CodePoints: []int{213}, Characters: []byte{0xc3, 0x95}},
	"Otimes":                          {Name: "Otimes", CodePoints: []int{10807}, Characters: []byte{0xe2, 0xa8, 0xb7}},
	"Ouml":                            {Name: "Ouml", CodePoints: []int{214}, Characters: []byte{0xc3, 0x96}},
	"OverBar":                         {Name: "OverBar", CodePoints: []int{8254}, Characters: []byte{0xe2, 0x80, 0xbe}},
	"OverBrace":                       {Name: "OverBrace", CodePoints: []int{9182}, Characters: []byte{0xe2, 0x8f, 0x9e}},
	"OverBracket":                     {Name: "OverBracket", CodePoints: []int{9140}, Characters: []byte{0xe2, 0x8e, 0xb4}},
	"OverParenthesis":                 {Name: "OverParenthesis", CodePoints: []int{9180}, Characters: []byte{0xe2, 0x8f, 0x9c}},
	"PartialD":                        {Name: "PartialD", CodePoints: []int{8706}, Characters: []byte{0xe2, 0x88, 0x82}},
	"Pcy":                             {Name: "Pcy", CodePoints: []int{1055}, Characters: []byte{0xd0, 0x9f}},
	"Pfr":                             {Name: "Pfr", CodePoints: []int{120083}, Characters: []byte{0xf0, 0x9d, 0x94, 0x93}},
	"Phi":                             {Name: "Phi", CodePoints: []int{934}, Characters: []byte{0xce, 0xa6}},
	"Pi":                              {Name: "Pi", CodePoints: []int{928}, Characters: []byte{0xce, 0xa0}},
	"PlusMinus":                       {Name: "PlusMinus", CodePoints: []int{177}, Characters: []byte{0xc2, 0xb1}},
	"Poincareplane":                   {Name: "Poincareplane", CodePoints: []int{8460}, Characters: []byte{0xe2, 0x84, 0x8c}},
	"Popf":                            {Name: "Popf", CodePoints: []int{8473}, Characters: []byte{0xe2, 0x84, 0x99}},
	"Pr":                              {Name: "Pr", CodePoints: []int{10939}, Characters: []byte{0xe2, 0xaa, 0xbb}},
	"Precedes":                        {Name: "Precedes", CodePoints: []int{8826}, Characters: []byte{0xe2, 0x89, 0xba}},
	"PrecedesEqual":                   {Name: "PrecedesEqual", CodePoints: []int{10927}, Characters: []byte{0xe2, 0xaa, 0xaf}},
	"PrecedesSlantEqual":              {Name: "PrecedesSlantEqual", CodePoints: []int{8828}, Characters: []byte{0xe2, 0x89, 0xbc}},
	"PrecedesTilde":                   {Name: "PrecedesTilde", CodePoints: []int{8830}, Characters: []byte{0xe2, 0x89, 0xbe}},
	"Prime":                           {Name: "Prime", CodePoints: []int{8243}, Characters: []byte{0xe2, 0x80, 0xb3}},
	"Product":                         {Name: "Product", CodePoints: []int{8719}, Characters: []byte{0xe2, 0x88, 0x8f}},
	"Proportion":                      {Name: "Proportion", CodePoints: []int{8759}, Characters: []byte{0xe2, 0x88, 0xb7}},
	"Proportional":                    {Name: "Proportional", CodePoints: []int{8733}, Characters: []byte{0xe2, 0x88, 0x9d}},
	"Pscr":                            {Name: "Pscr", CodePoints: []int{119979}, Characters: []byte{0xf0, 0x9d, 0x92, 0xab}},
	"Psi":                             {Name: "Psi", CodePoints: []int{936}, Characters: []byte{0xce, 0xa8}},
	"QUOT":                            {Name: "QUOT", CodePoints: []int{34}, Characters: []byte{0x22}},
	"Qfr":                             {Name: "Qfr", CodePoints: []int{120084}, Characters: []byte{0xf0, 0x9d, 0x94, 0x94}},
	"Qopf":                            {Name: "Qopf", CodePoints: []int{8474}, Characters: []byte{0xe2, 0x84, 0x9a}},
	"Qscr":                            {Name: "Qscr", CodePoints: []int{119980}, Characters: []byte{0xf0, 0x9d, 0x92, 0xac}},
	"RBarr":                           {Name: "RBarr", CodePoints: []int{10512}, Characters: []byte{0xe2, 0xa4, 0x90}},
	"REG":                             {Name: "REG", CodePoints: []int{174}, Characters: []byte{0xc2, 0xae}},
	"Racute":                          {Name: "Racute", CodePoints: []int{340}, Characters: []byte{0xc5, 0x94}},
	"Rang":                            {Name: "Rang", CodePoints: []int{10219}, Characters: []byte{0xe2, 0x9f, 0xab}},
	"Rarr":                            {Name: "Rarr", CodePoints: []int{8608}, Characters: []byte{0xe2, 0x86, 0xa0}},
	"Rarrtl":                          {Name: "Rarrtl", CodePoints: []int{10518}, Characters: []byte{0xe2, 0xa4, 0x96}},
	"Rcaron":                          {Name: "Rcaron", CodePoints: []int{344}, Characters: []byte{0xc5, 0x98}},
	"Rcedil":                          {Name: "Rcedil", CodePoints: []int{342}, Characters: []byte{0xc5, 0x96}},
	"Rcy":                             {Name: "Rcy", CodePoints: []int{1056}, Characters: []byte{0xd0, 0xa0}},
	"Re":                              {Name: "Re", CodePoints: []int{8476}, Characters: []byte{0xe2, 0x84, 0x9c}},
	"ReverseElement":                  {Name: "ReverseElement", CodePoints: []int{8715}, Characters: []byte{0xe2, 0x88, 0x8b}},
	"ReverseEquilibrium":              {Name: "ReverseEquilibrium", CodePoints: []int{8651}, Characters: []byte{0xe2, 0x87, 0x8b}},
	"ReverseUpEquilibrium":            {Name: "ReverseUpEquilibrium", CodePoints: []int{10607}, Characters: []byte{0xe2, 0xa5, 0xaf}},
	"Rfr":                             {Name: "Rfr", CodePoints: []int{8476}, Characters: []byte{0xe2, 0x84, 0x9c}},
	"Rho":                             {Name: "Rho", CodePoints: []int{929}, Characters: []byte{0xce, 0xa1}},
	"RightAngleBracket":               {Name: "RightAngleBracket", CodePoints: []int{10217}, Characters: []byte{0xe2, 0x9f, 0xa9}},
	"RightArrow":                      {Name: "RightArrow", CodePoints: []int{8594}, Characters: []byte{0xe2, 0x86, 0x92}},
	"RightArrowBar":                   {Name: "RightArrowBar", CodePoints: []int{8677}, Characters: []byte{0xe2, 0x87, 0xa5}},
	"RightArrowLeftArrow":             {Name: "RightArrowLeftArrow", CodePoints: []int{8644}, Characters: []byte{0xe2, 0x87, 0x84}},
	"RightCeiling":                    {Name: "RightCeiling", CodePoints: []int{8969}, Characters: []byte{0xe2, 0x8c, 0x89}},
	"RightDoubleBracket":              {Name: "RightDoubleBracket", CodePoints: []int{10215}, Characters: []byte{0xe2, 0x9f, 0xa7}},
	"RightDownTeeVector":              {Name: "RightDownTeeVector", CodePoints: []int{10589}, Characters: []byte{0xe2, 0xa5, 0x9d}},
	"RightDownVector":                 {Name: "RightDownVector", CodePoints: []int{8642}, Characters: []byte{0xe2, 0x87, 0x82}},
	"RightDownVectorBar":              {Name: "RightDownVectorBar", CodePoints: []int{10581}, Characters: []byte{0xe2, 0xa5, 0x95}},
	"RightFloor":                      {Name: "RightFloor", CodePoints: []int{8971}, Characters: []byte{0xe2, 0x8c, 0x8b}},
	"RightTee":                        {Name: "RightTee", CodePoints: []int{8866}, Characters: []byte{0xe2, 0x8a, 0xa2}},
	"RightTeeArrow":                   {Name: "RightTeeArrow", CodePoints: []int{8614}, Characters: []byte{0xe2, 0x86, 0xa6}},
	"RightTeeVector":                  {Name: "RightTeeVector", CodePoints: []int{10587}, Characters: []byte{0xe2, 0xa5, 0x9b}},
	"RightTriangle":                   {Name: "RightTriangle", CodePoints: []int{8883}, Characters: []byte{0xe2, 0x8a, 0xb3}},
	"RightTriangleBar":                {Name: "RightTriangleBar", CodePoints: []int{10704}, Characters: []byte{0xe2, 0xa7, 0x90}},
	"RightTriangleEqual":              {Name: "RightTriangleEqual", CodePoints: []int{8885}, Characters: []byte{0xe2, 0x8a, 0xb5}},
	"RightUpDownVector":               {Name: "RightUpDownVector", CodePoints: []int{10575}, Characters: []byte{0xe2, 0xa5, 0x8f}},
	"RightUpTeeVector":                {Name: "RightUpTeeVector", CodePoints: []int{10588}, Characters: []byte{0xe2, 0xa5, 0x9c}},
	"RightUpVector":                   {Name: "RightUpVector", CodePoints: []int{8638}, Characters: []byte{0xe2, 0x86, 0xbe}},
	"RightUpVectorBar":                {Name: "RightUpVectorBar", CodePoints: []int{10580}, Characters: []byte{0xe2, 0xa5, 0x94}},
	"RightVector":                     {Name: "RightVector", CodePoints: []int{8640}, Characters: []byte{0xe2, 0x87, 0x80}},
	"RightVectorBar":                  {Name: "RightVectorBar", CodePoints: []int{10579}, Characters: []byte{0xe2, 0xa5, 0x93}},
	"Rightarrow":                      {Name: "Rightarrow", CodePoints: []int{8658}, Characters: []byte{0xe2, 0x87, 0x92}},
	"Ropf":                            {Name: "Ropf", CodePoints: []int{8477}, Characters: []byte{0xe2, 0x84, 0x9d}},
	"RoundImplies":                    {Name: "RoundImplies", CodePoints: []int{10608}, Characters: []byte{0xe2, 0xa5, 0xb0}},
	"Rrightarrow":                     {Name: "Rrightarrow", CodePoints: []int{8667}, Characters: []byte{0xe2, 0x87, 0x9b}},
	"Rscr":                            {Name: "Rscr", CodePoints: []int{8475}, Characters: []byte{0xe2, 0x84, 0x9b}},
	"Rsh":                             {Name: "Rsh", CodePoints: []int{8625}, Characters: []byte{0xe2, 0x86, 0xb1}},
	"RuleDelayed":                     {Name: "RuleDelayed", CodePoints: []int{10740}, Characters: []byte{0xe2, 0xa7, 0xb4}},
	"SHCHcy":                          {Name: "SHCHcy", CodePoints: []int{1065}, Characters: []byte{0xd0, 0xa9}},
	"SHcy":                            {Name: "SHcy", CodePoints: []int{1064}, Characters: []byte{0xd0, 0xa8}},
	"SOFTcy":                          {Name: "SOFTcy", CodePoints: []int{1068}, Characters: []byte{0xd0, 0xac}},
	"Sacute":                          {Name: "Sacute", CodePoints: []int{346}, Characters: []byte{0xc5, 0x9a}},
	"Sc":                              {Name: "Sc", CodePoints: []int{10940}, Characters: []byte{0xe2, 0xaa, 0xbc}},
	"Scaron":                          {Name: "Scaron", CodePoints: []int{352}, Characters: []byte{0xc5, 0xa0}},
	"Scedil":                          {Name: "Scedil", CodePoints: []int{350}, Characters: []byte{0xc5, 0x9e}},
	"Scirc":                           {Name: "Scirc", CodePoints: []int{348}, Characters: []byte{0xc5, 0x9c}},
	"Scy":                             {Name: "Scy", CodePoints: []int{1057}, Characters: []byte{0xd0, 0xa1}},
	"Sfr":                             {Name: "Sfr", CodePoints: []int{120086}, Characters: []byte{0xf0, 0x9d, 0x94, 0x96}},
	"ShortDownArrow":                  {Name: "ShortDownArrow", CodePoints: []int{8595}, Characters: []byte{0xe2, 0x86, 0x93}},
	"ShortLeftArrow":                  {Name: "ShortLeftArrow", CodePoints: []int{8592}, Characters: []byte{0xe2, 0x86, 0x90}},
	"ShortRightArrow":                 {Name: "ShortRightArrow", CodePoints: []int{8594}, Characters: []byte{0xe2, 0x86, 0x92}},
	"ShortUpArrow":                    {Name: "ShortUpArrow", CodePoints: []int{8593}, Characters: []byte{0xe2, 0x86, 0x91}},
	"Sigma":                           {Name: "Sigma", CodePoints: []int{931}, Characters: []byte{0xce, 0xa3}},
	"SmallCircle":                     {Name: "SmallCircle", CodePoints: []int{8728}, Characters: []byte{0xe2, 0x88, 0x98}},
	"Sopf":                            {Name: "Sopf", CodePoints: []int{120138}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8a}},
	"Sqrt":                            {Name: "Sqrt", CodePoints: []int{8730}, Characters: []byte{0xe2, 0x88, 0x9a}},
	"Square":                          {Name: "Square", CodePoints: []int{9633}, Characters: []byte{0xe2, 0x96, 0xa1}},
	"SquareIntersection":              {Name: "SquareIntersection", CodePoints: []int{8851}, Characters: []byte{0xe2, 0x8a, 0x93}},
	"SquareSubset":                    {Name: "SquareSubset", CodePoints: []int{8847}, Characters: []byte{0xe2, 0x8a, 0x8f}},
	"SquareSubsetEqual":               {Name: "SquareSubsetEqual", CodePoints: []int{8849}, Characters: []byte{0xe2, 0x8a, 0x91}},
	"SquareSuperset":                  {Name: "SquareSuperset", CodePoints: []int{8848}, Characters: []byte{0xe2, 0x8a, 0x90}},
	"SquareSupersetEqual":             {Name: "SquareSupersetEqual", CodePoints: []int{8850}, Characters: []byte{0xe2, 0x8a, 0x92}},
	"SquareUnion":                     {Name: "SquareUnion", CodePoints: []int{8852}, Characters: []byte{0xe2, 0x8a, 0x94}},
	"Sscr":                            {Name: "Sscr", CodePoints: []int{119982}, Characters: []byte{0xf0, 0x9d, 0x92, 0xae}},
	"Star":                            {Name: "Star", CodePoints: []int{8902}, Characters: []byte{0xe2, 0x8b, 0x86}},
	"Sub":                             {Name: "Sub", CodePoints: []int{8912}, Characters: []byte{0xe2, 0x8b, 0x90}},
	"Subset":                          {Name: "Subset", CodePoints: []int{8912}, Characters: []byte{0xe2, 0x8b, 0x90}},
	"SubsetEqual":                     {Name: "SubsetEqual", CodePoints: []int{8838}, Characters: []byte{0xe2, 0x8a, 0x86}},
	"Succeeds":                        {Name: "Succeeds", CodePoints: []int{8827}, Characters: []byte{0xe2, 0x89, 0xbb}},
	"SucceedsEqual":                   {Name: "SucceedsEqual", CodePoints: []int{10928}, Characters: []byte{0xe2, 0xaa, 0xb0}},
	"SucceedsSlantEqual":              {Name: "SucceedsSlantEqual", CodePoints: []int{8829}, Characters: []byte{0xe2, 0x89, 0xbd}},
	"SucceedsTilde":                   {Name: "SucceedsTilde", CodePoints: []int{8831}, Characters: []byte{0xe2, 0x89, 0xbf}},
	"SuchThat":                        {Name: "SuchThat", CodePoints: []int{8715}, Characters: []byte{0xe2, 0x88, 0x8b}},
	"Sum":                             {Name: "Sum", CodePoints: []int{8721}, Characters: []byte{0xe2, 0x88, 0x91}},
	"Sup":                             {Name: "Sup", CodePoints: []int{8913}, Characters: []byte{0xe2, 0x8b, 0x91}},
	"Superset":                        {Name: "Superset", CodePoints: []int{8835}, Characters: []byte{0xe2, 0x8a, 0x83}},
	"SupersetEqual":                   {Name: "SupersetEqual", CodePoints: []int{8839}, Characters: []byte{0xe2, 0x8a, 0x87}},
	"Supset":                          {Name: "Supset", CodePoints: []int{8913}, Characters: []byte{0xe2, 0x8b, 0x91}},
	"THORN":                           {Name: "THORN", CodePoints: []int{222}, Characters: []byte{0xc3, 0x9e}},
	"TRADE":                           {Name: "TRADE", CodePoints: []int{8482}, Characters: []byte{0xe2, 0x84, 0xa2}},
	"TSHcy":                           {Name: "TSHcy", CodePoints: []int{1035}, Characters: []byte{0xd0, 0x8b}},
	"TScy":                            {Name: "TScy", CodePoints: []int{1062}, Characters: []byte{0xd0, 0xa6}},
	"Tab":                             {Name: "Tab", CodePoints: []int{9}, Characters: []byte{0x9}},
	"Tau":                             {Name: "Tau", CodePoints: []int{932}, Characters: []byte{0xce, 0xa4}},
	"Tcaron":                          {Name: "Tcaron", CodePoints: []int{356}, Characters: []byte{0xc5, 0xa4}},
	"Tcedil":                          {Name: "Tcedil", CodePoints: []int{354}, Characters: []byte{0xc5, 0xa2}},
	"Tcy":                             {Name: "Tcy", CodePoints: []int{1058}, Characters: []byte{0xd0, 0xa2}},
	"Tfr":                             {Name: "Tfr", CodePoints: []int{120087}, Characters: []byte{0xf0, 0x9d, 0x94, 0x97}},
	"Therefore":                       {Name: "Therefore", CodePoints: []int{8756}, Characters: []byte{0xe2, 0x88, 0xb4}},
	"Theta":                           {Name: "Theta", CodePoints: []int{920}, Characters: []byte{0xce, 0x98}},
	"ThickSpace":                      {Name: "ThickSpace", CodePoints: []int{8287, 8202}, Characters: []byte{0xe2, 0x81, 0x9f, 0xe2, 0x80, 0x8a}},
	"ThinSpace":                       {Name: "ThinSpace", CodePoints: []int{8201}, Characters: []byte{0xe2, 0x80, 0x89}},
	"Tilde":                           {Name: "Tilde", CodePoints: []int{8764}, Characters: []byte{0xe2, 0x88, 0xbc}},
	"TildeEqual":                      {Name: "TildeEqual", CodePoints: []int{8771}, Characters: []byte{0xe2, 0x89, 0x83}},
	"TildeFullEqual":                  {Name: "TildeFullEqual", CodePoints: []int{8773}, Characters: []byte{0xe2, 0x89, 0x85}},
	"TildeTilde":                      {Name: "TildeTilde", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"Topf":                            {Name: "Topf", CodePoints: []int{120139}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8b}},
	"TripleDot":                       {Name: "TripleDot", CodePoints: []int{8411}, Characters: []byte{0xe2, 0x83, 0x9b}},
	"Tscr":                            {Name: "Tscr", CodePoints: []int{119983}, Characters: []byte{0xf0, 0x9d, 0x92, 0xaf}},
	"Tstrok":                          {Name: "Tstrok", CodePoints: []int{358}, Characters: []byte{0xc5, 0xa6}},
	"Uacute":                          {Name: "Uacute", CodePoints: []int{218}, Characters: []byte{0xc3, 0x9a}},
	"Uarr":                            {Name: "Uarr", CodePoints: []int{8607}, Characters: []byte{0xe2, 0x86, 0x9f}},
	"Uarrocir":                        {Name: "Uarrocir", CodePoints: []int{10569}, Characters: []byte{0xe2, 0xa5, 0x89}},
	"Ubrcy":                           {Name: "Ubrcy", CodePoints: []int{1038}, Characters: []byte{0xd0, 0x8e}},
	"Ubreve":                          {Name: "Ubreve", CodePoints: []int{364}, Characters: []byte{0xc5, 0xac}},
	"Ucirc":                           {Name: "Ucirc", CodePoints: []int{219}, Characters: []byte{0xc3, 0x9b}},
	"Ucy":                             {Name: "Ucy", CodePoints: []int{1059}, Characters: []byte{0xd0, 0xa3}},
	"Udblac":                          {Name: "Udblac", CodePoints: []int{368}, Characters: []byte{0xc5, 0xb0}},
	"Ufr":                             {Name: "Ufr", CodePoints: []int{120088}, Characters: []byte{0xf0, 0x9d, 0x94, 0x98}},
	"Ugrave":                          {Name: "Ugrave", CodePoints: []int{217}, Characters: []byte{0xc3, 0x99}},
	"Umacr":                           {Name: "Umacr", CodePoints: []int{362}, Characters: []byte{0xc5, 0xaa}},
	"UnderBar":                        {Name: "UnderBar", CodePoints: []int{95}, Characters: []byte{0x5f}},
	"UnderBrace":                      {Name: "UnderBrace", CodePoints: []int{9183}, Characters: []byte{0xe2, 0x8f, 0x9f}},
	"UnderBracket":                    {Name: "UnderBracket", CodePoints: []int{9141}, Characters: []byte{0xe2, 0x8e, 0xb5}},
	"UnderParenthesis":                {Name: "UnderParenthesis", CodePoints: []int{9181}, Characters: []byte{0xe2, 0x8f, 0x9d}},
	"Union":                           {Name: "Union", CodePoints: []int{8899}, Characters: []byte{0xe2, 0x8b, 0x83}},
	"UnionPlus":                       {Name: "UnionPlus", CodePoints: []int{8846}, Characters: []byte{0xe2, 0x8a, 0x8e}},
	"Uogon":                           {Name: "Uogon", CodePoints: []int{370}, Characters: []byte{0xc5, 0xb2}},
	"Uopf":                            {Name: "Uopf", CodePoints: []int{120140}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8c}},
	"UpArrow":                         {Name: "UpArrow", CodePoints: []int{8593}, Characters: []byte{0xe2, 0x86, 0x91}},
	"UpArrowBar":                      {Name: "UpArrowBar", CodePoints: []int{10514}, Characters: []byte{0xe2, 0xa4, 0x92}},
	"UpArrowDownArrow":                {Name: "UpArrowDownArrow", CodePoints: []int{8645}, Characters: []byte{0xe2, 0x87, 0x85}},
	"UpDownArrow":                     {Name: "UpDownArrow", CodePoints: []int{8597}, Characters: []byte{0xe2, 0x86, 0x95}},
	"UpEquilibrium":                   {Name: "UpEquilibrium", CodePoints: []int{10606}, Characters: []byte{0xe2, 0xa5, 0xae}},
	"UpTee":                           {Name: "UpTee", CodePoints: []int{8869}, Characters: []byte{0xe2, 0x8a, 0xa5}},
	"UpTeeArrow":                      {Name: "UpTeeArrow", CodePoints: []int{8613}, Characters: []byte{0xe2, 0x86, 0xa5}},
	"Uparrow":                         {Name: "Uparrow", CodePoints: []int{8657}, Characters: []byte{0xe2, 0x87, 0x91}},
	"Updownarrow":                     {Name: "Updownarrow", CodePoints: []int{8661}, Characters: []byte{0xe2, 0x87, 0x95}},
	"UpperLeftArrow":                  {Name: "UpperLeftArrow", CodePoints: []int{8598}, Characters: []byte{0xe2, 0x86, 0x96}},
	"UpperRightArrow":                 {Name: "UpperRightArrow", CodePoints: []int{8599}, Characters: []byte{0xe2, 0x86, 0x97}},
	"Upsi":                            {Name: "Upsi", CodePoints: []int{978}, Characters: []byte{0xcf, 0x92}},
	"Upsilon":                         {Name: "Upsilon", CodePoints: []int{933}, Characters: []byte{0xce, 0xa5}},
	"Uring":                           {Name: "Uring", CodePoints: []int{366}, Characters: []byte{0xc5, 0xae}},
	"Uscr":                            {Name: "Uscr", CodePoints: []int{119984}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb0}},
	"Utilde":                          {Name: "Utilde", CodePoints: []int{360}, Characters: []byte{0xc5, 0xa8}},
	"Uuml":                            {Name: "Uuml", CodePoints: []int{220}, Characters: []byte{0xc3, 0x9c}},
	"VDash":                           {Name: "VDash", CodePoints: []int{8875}, Characters: []byte{0xe2, 0x8a, 0xab}},
	"Vbar":                            {Name: "Vbar", CodePoints: []int{10987}, Characters: []byte{0xe2, 0xab, 0xab}},
	"Vcy":                             {Name: "Vcy", CodePoints: []int{1042}, Characters: []byte{0xd0, 0x92}},
	"Vdash":                           {Name: "Vdash", CodePoints: []int{8873}, Characters: []byte{0xe2, 0x8a, 0xa9}},
	"Vdashl":                          {Name: "Vdashl", CodePoints: []int{10982}, Characters: []byte{0xe2, 0xab, 0xa6}},
	"Vee":                             {Name: "Vee", CodePoints: []int{8897}, Characters: []byte{0xe2, 0x8b, 0x81}},
	"Verbar":                          {Name: "Verbar", CodePoints: []int{8214}, Characters: []byte{0xe2, 0x80, 0x96}},
	"Vert":                            {Name: "Vert", CodePoints: []int{8214}, Characters: []byte{0xe2, 0x80, 0x96}},
	"VerticalBar":                     {Name: "VerticalBar", CodePoints: []int{8739}, Characters: []byte{0xe2, 0x88, 0xa3}},
	"VerticalLine":                    {Name: "VerticalLine", CodePoints: []int{124}, Characters: []byte{0x7c}},
	"VerticalSeparator":               {Name: "VerticalSeparator", CodePoints: []int{10072}, Characters: []byte{0xe2, 0x9d, 0x98}},
	"VerticalTilde":                   {Name: "VerticalTilde", CodePoints: []int{8768}, Characters: []byte{0xe2, 0x89, 0x80}},
	"VeryThinSpace":                   {Name: "VeryThinSpace", CodePoints: []int{8202}, Characters: []byte{0xe2, 0x80, 0x8a}},
	"Vfr":                             {Name: "Vfr", CodePoints: []int{120089}, Characters: []byte{0xf0, 0x9d, 0x94, 0x99}},
	"Vopf":                            {Name: "Vopf", CodePoints: []int{120141}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8d}},
	"Vscr":                            {Name: "Vscr", CodePoints: []int{119985}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb1}},
	"Vvdash":                          {Name: "Vvdash", CodePoints: []int{8874}, Characters: []byte{0xe2, 0x8a, 0xaa}},
	"Wcirc":                           {Name: "Wcirc", CodePoints: []int{372}, Characters: []byte{0xc5, 0xb4}},
	"Wedge":                           {Name: "Wedge", CodePoints: []int{8896}, Characters: []byte{0xe2, 0x8b, 0x80}},
	"Wfr":                             {Name: "Wfr", CodePoints: []int{120090}, Characters: []byte{0xf0, 0x9d, 0x94, 0x9a}},
	"Wopf":                            {Name: "Wopf", CodePoints: []int{120142}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8e}},
	"Wscr":                            {Name: "Wscr", CodePoints: []int{119986}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb2}},
	"Xfr":                             {Name: "Xfr", CodePoints: []int{120091}, Characters: []byte{0xf0, 0x9d, 0x94, 0x9b}},
	"Xi":                              {Name: "Xi", CodePoints: []int{926}, Characters: []byte{0xce, 0x9e}},
	"Xopf":                            {Name: "Xopf", CodePoints: []int{120143}, Characters: []byte{0xf0, 0x9d, 0x95, 0x8f}},
	"Xscr":                            {Name: "Xscr", CodePoints: []int{119987}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb3}},
	"YAcy":                            {Name: "YAcy", CodePoints: []int{1071}, Characters: []byte{0xd0, 0xaf}},
	"YIcy":                            {Name: "YIcy", CodePoints: []int{1031}, Characters: []byte{0xd0, 0x87}},
	"YUcy":                            {Name: "YUcy", CodePoints: []int{1070}, Characters: []byte{0xd0, 0xae}},
	"Yacute":                          {Name: "Yacute", CodePoints: []int{221}, Characters: []byte{0xc3, 0x9d}},
	"Ycirc":                           {Name: "Ycirc", CodePoints: []int{374}, Characters: []byte{0xc5, 0xb6}},
	"Ycy":                             {Name: "Ycy", CodePoints: []int{1067}, Characters: []byte{0xd0, 0xab}},
	"Yfr":                             {Name: "Yfr", CodePoints: []int{120092}, Characters: []byte{0xf0, 0x9d, 0x94, 0x9c}},
	"Yopf":                            {Name: "Yopf", CodePoints: []int{120144}, Characters: []byte{0xf0, 0x9d, 0x95, 0x90}},
	"Yscr":                            {Name: "Yscr", CodePoints: []int{119988}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb4}},
	"Yuml":                            {Name: "Yuml", CodePoints: []int{376}, Characters: []byte{0xc5, 0xb8}},
	"ZHcy":                            {Name: "ZHcy", CodePoints: []int{1046}, Characters: []byte{0xd0, 0x96}},
	"Zacute":                          {Name: "Zacute", CodePoints: []int{377}, Characters: []byte{0xc5, 0xb9}},
	"Zcaron":                          {Name: "Zcaron", CodePoints: []int{381}, Characters: []byte{0xc5, 0xbd}},
	"Zcy":                             {Name: "Zcy", CodePoints: []int{1047}, Characters: []byte{0xd0, 0x97}},
	"Zdot":                            {Name: "Zdot", CodePoints: []int{379}, Characters: []byte{0xc5, 0xbb}},
	"ZeroWidthSpace":                  {Name: "ZeroWidthSpace", CodePoints: []int{8203}, Characters: []byte{0xe2, 0x80, 0x8b}},
	"Zeta":                            {Name: "Zeta", CodePoints: []int{918}, Characters: []byte{0xce, 0x96}},
	"Zfr":                             {Name: "Zfr", CodePoints: []int{8488}, Characters: []byte{0xe2, 0x84, 0xa8}},
	"Zopf":                            {Name: "Zopf", CodePoints: []int{8484}, Characters: []byte{0xe2, 0x84, 0xa4}},
	"Zscr":                            {Name: "Zscr", CodePoints: []int{119989}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb5}},
	"aacute":                          {Name: "aacute", CodePoints: []int{225}, Characters: []byte{0xc3, 0xa1}},
	"abreve":                          {Name: "abreve", CodePoints: []int{259}, Characters: []byte{0xc4, 0x83}},
	"ac":                              {Name: "ac", CodePoints: []int{8766}, Characters: []byte{0xe2, 0x88, 0xbe}},
	"acE":                             {Name: "acE", CodePoints: []int{8766, 819}, Characters: []byte{0xe2, 0x88, 0xbe, 0xcc, 0xb3}},
	"acd":                             {Name: "acd", CodePoints: []int{8767}, Characters: []byte{0xe2, 0x88, 0xbf}},
	"acirc":                           {Name: "acirc", CodePoints: []int{226}, Characters: []byte{0xc3, 0xa2}},
	"acute":                           {Name: "acute", CodePoints: []int{180}, Characters: []byte{0xc2, 0xb4}},
	"acy":                             {Name: "acy", CodePoints: []int{1072}, Characters: []byte{0xd0, 0xb0}},
	"aelig":                           {Name: "aelig", CodePoints: []int{230}, Characters: []byte{0xc3, 0xa6}},
	"af":                              {Name: "af", CodePoints: []int{8289}, Characters: []byte{0xe2, 0x81, 0xa1}},
	"afr":                             {Name: "afr", CodePoints: []int{120094}, Characters: []byte{0xf0, 0x9d, 0x94, 0x9e}},
	"agrave":                          {Name: "agrave", CodePoints: []int{224}, Characters: []byte{0xc3, 0xa0}},
	"alefsym":                         {Name: "alefsym", CodePoints: []int{8501}, Characters: []byte{0xe2, 0x84, 0xb5}},
	"aleph":                           {Name: "aleph", CodePoints: []int{8501}, Characters: []byte{0xe2, 0x84, 0xb5}},
	"alpha":                           {Name: "alpha", CodePoints: []int{945}, Characters: []byte{0xce, 0xb1}},
	"amacr":                           {Name: "amacr", CodePoints: []int{257}, Characters: []byte{0xc4, 0x81}},
	"amalg":                           {Name: "amalg", CodePoints: []int{10815}, Characters: []byte{0xe2, 0xa8, 0xbf}},
	"amp":                             {Name: "amp", CodePoints: []int{38}, Characters: []byte{0x26}},
	"and":                             {Name: "and", CodePoints: []int{8743}, Characters: []byte{0xe2, 0x88, 0xa7}},
	"andand":                          {Name: "andand", CodePoints: []int{10837}, Characters: []byte{0xe2, 0xa9, 0x95}},
	"andd":                            {Name: "andd", CodePoints: []int{10844}, Characters: []byte{0xe2, 0xa9, 0x9c}},
	"andslope":                        {Name: "andslope", CodePoints: []int{10840}, Characters: []byte{0xe2, 0xa9, 0x98}},
	"andv":                            {Name: "andv", CodePoints: []int{10842}, Characters: []byte{0xe2, 0xa9, 0x9a}},
	"ang":                             {Name: "ang", CodePoints: []int{8736}, Characters: []byte{0xe2, 0x88, 0xa0}},
	"ange":                            {Name: "ange", CodePoints: []int{10660}, Characters: []byte{0xe2, 0xa6, 0xa4}},
	"angle":                           {Name: "angle", CodePoints: []int{8736}, Characters: []byte{0xe2, 0x88, 0xa0}},
	"angmsd":                          {Name: "angmsd", CodePoints: []int{8737}, Characters: []byte{0xe2, 0x88, 0xa1}},
	"angmsdaa":                        {Name: "angmsdaa", CodePoints: []int{10664}, Characters: []byte{0xe2, 0xa6, 0xa8}},
	"angmsdab":                        {Name: "angmsdab", CodePoints: []int{10665}, Characters: []byte{0xe2, 0xa6, 0xa9}},
	"angmsdac":                        {Name: "angmsdac", CodePoints: []int{10666}, Characters: []byte{0xe2, 0xa6, 0xaa}},
	"angmsdad":                        {Name: "angmsdad", CodePoints: []int{10667}, Characters: []byte{0xe2, 0xa6, 0xab}},
	"angmsdae":                        {Name: "angmsdae", CodePoints: []int{10668}, Characters: []byte{0xe2, 0xa6, 0xac}},
	"angmsdaf":                        {Name: "angmsdaf", CodePoints: []int{10669}, Characters: []byte{0xe2, 0xa6, 0xad}},
	"angmsdag":                        {Name: "angmsdag", CodePoints: []int{10670}, Characters: []byte{0xe2, 0xa6, 0xae}},
	"angmsdah":                        {Name: "angmsdah", CodePoints: []int{10671}, Characters: []byte{0xe2, 0xa6, 0xaf}},
	"angrt":                           {Name: "angrt", CodePoints: []int{8735}, Characters: []byte{0xe2, 0x88, 0x9f}},
	"angrtvb":                         {Name: "angrtvb", CodePoints: []int{8894}, Characters: []byte{0xe2, 0x8a, 0xbe}},
	"angrtvbd":                        {Name: "angrtvbd", CodePoints: []int{10653}, Characters: []byte{0xe2, 0xa6, 0x9d}},
	"angsph":                          {Name: "angsph", CodePoints: []int{8738}, Characters: []byte{0xe2, 0x88, 0xa2}},
	"angst":                           {Name: "angst", CodePoints: []int{197}, Characters: []byte{0xc3, 0x85}},
	"angzarr":                         {Name: "angzarr", CodePoints: []int{9084}, Characters: []byte{0xe2, 0x8d, 0xbc}},
	"aogon":                           {Name: "aogon", CodePoints: []int{261}, Characters: []byte{0xc4, 0x85}},
	"aopf":                            {Name: "aopf", CodePoints: []int{120146}, Characters: []byte{0xf0, 0x9d, 0x95, 0x92}},
	"ap":                              {Name: "ap", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"apE":                             {Name: "apE", CodePoints: []int{10864}, Characters: []byte{0xe2, 0xa9, 0xb0}},
	"apacir":                          {Name: "apacir", CodePoints: []int{10863}, Characters: []byte{0xe2, 0xa9, 0xaf}},
	"ape":                             {Name: "ape", CodePoints: []int{8778}, Characters: []byte{0xe2, 0x89, 0x8a}},
	"apid":                            {Name: "apid", CodePoints: []int{8779}, Characters: []byte{0xe2, 0x89, 0x8b}},
	"apos":                            {Name: "apos", CodePoints: []int{39}, Characters: []byte{0x27}},
	"approx":                          {Name: "approx", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"approxeq":                        {Name: "approxeq", CodePoints: []int{8778}, Characters: []byte{0xe2, 0x89, 0x8a}},
	"aring":                           {Name: "aring", CodePoints: []int{229}, Characters: []byte{0xc3, 0xa5}},
	"ascr":                            {Name: "ascr", CodePoints: []int{119990}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb6}},
	"ast":                             {Name: "ast", CodePoints: []int{42}, Characters: []byte{0x2a}},
	"asymp":                           {Name: "asymp", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"asympeq":                         {Name: "asympeq", CodePoints: []int{8781}, Characters: []byte{0xe2, 0x89, 0x8d}},
	"atilde":                          {Name: "atilde", CodePoints: []int{227}, Characters: []byte{0xc3, 0xa3}},
	"auml":                            {Name: "auml", CodePoints: []int{228}, Characters: []byte{0xc3, 0xa4}},
	"awconint":                        {Name: "awconint", CodePoints: []int{8755}, Characters: []byte{0xe2, 0x88, 0xb3}},
	"awint":                           {Name: "awint", CodePoints: []int{10769}, Characters: []byte{0xe2, 0xa8, 0x91}},
	"bNot":                            {Name: "bNot", CodePoints: []int{10989}, Characters: []byte{0xe2, 0xab, 0xad}},
	"backcong":                        {Name: "backcong", CodePoints: []int{8780}, Characters: []byte{0xe2, 0x89, 0x8c}},
	"backepsilon":                     {Name: "backepsilon", CodePoints: []int{1014}, Characters: []byte{0xcf, 0xb6}},
	"backprime":                       {Name: "backprime", CodePoints: []int{8245}, Characters: []byte{0xe2, 0x80, 0xb5}},
	"backsim":                         {Name: "backsim", CodePoints: []int{8765}, Characters: []byte{0xe2, 0x88, 0xbd}},
	"backsimeq":                       {Name: "backsimeq", CodePoints: []int{8909}, Characters: []byte{0xe2, 0x8b, 0x8d}},
	"barvee":                          {Name: "barvee", CodePoints: []int{8893}, Characters: []byte{0xe2, 0x8a, 0xbd}},
	"barwed":                          {Name: "barwed", CodePoints: []int{8965}, Characters: []byte{0xe2, 0x8c, 0x85}},
	"barwedge":                        {Name: "barwedge", CodePoints: []int{8965}, Characters: []byte{0xe2, 0x8c, 0x85}},
	"bbrk":                            {Name: "bbrk", CodePoints: []int{9141}, Characters: []byte{0xe2, 0x8e, 0xb5}},
	"bbrktbrk":                        {Name: "bbrktbrk", CodePoints: []int{9142}, Characters: []byte{0xe2, 0x8e, 0xb6}},
	"bcong":                           {Name: "bcong", CodePoints: []int{8780}, Characters: []byte{0xe2, 0x89, 0x8c}},
	"bcy":                             {Name: "bcy", CodePoints: []int{1073}, Characters: []byte{0xd0, 0xb1}},
	"bdquo":                           {Name: "bdquo", CodePoints: []int{8222}, Characters: []byte{0xe2, 0x80, 0x9e}},
	"becaus":                          {Name: "becaus", CodePoints: []int{8757}, Characters: []byte{0xe2, 0x88, 0xb5}},
	"because":                         {Name: "because", CodePoints: []int{8757}, Characters: []byte{0xe2, 0x88, 0xb5}},
	"bemptyv":                         {Name: "bemptyv", CodePoints: []int{10672}, Characters: []byte{0xe2, 0xa6, 0xb0}},
	"bepsi":                           {Name: "bepsi", CodePoints: []int{1014}, Characters: []byte{0xcf, 0xb6}},
	"bernou":                          {Name: "bernou", CodePoints: []int{8492}, Characters: []byte{0xe2, 0x84, 0xac}},
	"beta":                            {Name: "beta", CodePoints: []int{946}, Characters: []byte{0xce, 0xb2}},
	"beth":                            {Name: "beth", CodePoints: []int{8502}, Characters: []byte{0xe2, 0x84, 0xb6}},
	"between":                         {Name: "between", CodePoints: []int{8812}, Characters: []byte{0xe2, 0x89, 0xac}},
	"bfr":                             {Name: "bfr", CodePoints: []int{120095}, Characters: []byte{0xf0, 0x9d, 0x94, 0x9f}},
	"bigcap":                          {Name: "bigcap", CodePoints: []int{8898}, Characters: []byte{0xe2, 0x8b, 0x82}},
	"bigcirc":                         {Name: "bigcirc", CodePoints: []int{9711}, Characters: []byte{0xe2, 0x97, 0xaf}},
	"bigcup":                          {Name: "bigcup", CodePoints: []int{8899}, Characters: []byte{0xe2, 0x8b, 0x83}},
	"bigodot":                         {Name: "bigodot", CodePoints: []int{10752}, Characters: []byte{0xe2, 0xa8, 0x80}},
	"bigoplus":                        {Name: "bigoplus", CodePoints: []int{10753}, Characters: []byte{0xe2, 0xa8, 0x81}},
	"bigotimes":                       {Name: "bigotimes", CodePoints: []int{10754}, Characters: []byte{0xe2, 0xa8, 0x82}},
	"bigsqcup":                        {Name: "bigsqcup", CodePoints: []int{10758}, Characters: []byte{0xe2, 0xa8, 0x86}},
	"bigstar":                         {Name: "bigstar", CodePoints: []int{9733}, Characters: []byte{0xe2, 0x98, 0x85}},
	"bigtriangledown":                 {Name: "bigtriangledown", CodePoints: []int{9661}, Characters: []byte{0xe2, 0x96, 0xbd}},
	"bigtriangleup":                   {Name: "bigtriangleup", CodePoints: []int{9651}, Characters: []byte{0xe2, 0x96, 0xb3}},
	"biguplus":                        {Name: "biguplus", CodePoints: []int{10756}, Characters: []byte{0xe2, 0xa8, 0x84}},
	"bigvee":                          {Name: "bigvee", CodePoints: []int{8897}, Characters: []byte{0xe2, 0x8b, 0x81}},
	"bigwedge":                        {Name: "bigwedge", CodePoints: []int{8896}, Characters: []byte{0xe2, 0x8b, 0x80}},
	"bkarow":                          {Name: "bkarow", CodePoints: []int{10509}, Characters: []byte{0xe2, 0xa4, 0x8d}},
	"blacklozenge":                    {Name: "blacklozenge", CodePoints: []int{10731}, Characters: []byte{0xe2, 0xa7, 0xab}},
	"blacksquare":                     {Name: "blacksquare", CodePoints: []int{9642}, Characters: []byte{0xe2, 0x96, 0xaa}},
	"blacktriangle":                   {Name: "blacktriangle", CodePoints: []int{9652}, Characters: []byte{0xe2, 0x96, 0xb4}},
	"blacktriangledown":               {Name: "blacktriangledown", CodePoints: []int{9662}, Characters: []byte{0xe2, 0x96, 0xbe}},
	"blacktriangleleft":               {Name: "blacktriangleleft", CodePoints: []int{9666}, Characters: []byte{0xe2, 0x97, 0x82}},
	"blacktriangleright":              {Name: "blacktriangleright", CodePoints: []int{9656}, Characters: []byte{0xe2, 0x96, 0xb8}},
	"blank":                           {Name: "blank", CodePoints: []int{9251}, Characters: []byte{0xe2, 0x90, 0xa3}},
	"blk12":                           {Name: "blk12", CodePoints: []int{9618}, Characters: []byte{0xe2, 0x96, 0x92}},
	"blk14":                           {Name: "blk14", CodePoints: []int{9617}, Characters: []byte{0xe2, 0x96, 0x91}},
	"blk34":                           {Name: "blk34", CodePoints: []int{9619}, Characters: []byte{0xe2, 0x96, 0x93}},
	"block":                           {Name: "block", CodePoints: []int{9608}, Characters: []byte{0xe2, 0x96, 0x88}},
	"bne":                             {Name: "bne", CodePoints: []int{61, 8421}, Characters: []byte{0x3d, 0xe2, 0x83, 0xa5}},
	"bnequiv":                         {Name: "bnequiv", CodePoints: []int{8801, 8421}, Characters: []byte{0xe2, 0x89, 0xa1, 0xe2, 0x83, 0xa5}},
	"bnot":                            {Name: "bnot", CodePoints: []int{8976}, Characters: []byte{0xe2, 0x8c, 0x90}},
	"bopf":                            {Name: "bopf", CodePoints: []int{120147}, Characters: []byte{0xf0, 0x9d, 0x95, 0x93}},
	"bot":                             {Name: "bot", CodePoints: []int{8869}, Characters: []byte{0xe2, 0x8a, 0xa5}},
	"bottom":                          {Name: "bottom", CodePoints: []int{8869}, Characters: []byte{0xe2, 0x8a, 0xa5}},
	"bowtie":                          {Name: "bowtie", CodePoints: []int{8904}, Characters: []byte{0xe2, 0x8b, 0x88}},
	"boxDL":                           {Name: "boxDL", CodePoints: []int{9559}, Characters: []byte{0xe2, 0x95, 0x97}},
	"boxDR":                           {Name: "boxDR", CodePoints: []int{9556}, Characters: []byte{0xe2, 0x95, 0x94}},
	"boxDl":                           {Name: "boxDl", CodePoints: []int{9558}, Characters: []byte{0xe2, 0x95, 0x96}},
	"boxDr":                           {Name: "boxDr", CodePoints: []int{9555}, Characters: []byte{0xe2, 0x95, 0x93}},
	"boxH":                            {Name: "boxH", CodePoints: []int{9552}, Characters: []byte{0xe2, 0x95, 0x90}},
	"boxHD":                           {Name: "boxHD", CodePoints: []int{9574}, Characters: []byte{0xe2, 0x95, 0xa6}},
	"boxHU":                           {Name: "boxHU", CodePoints: []int{9577}, Characters: []byte{0xe2, 0x95, 0xa9}},
	"boxHd":                           {Name: "boxHd", CodePoints: []int{9572}, Characters: []byte{0xe2, 0x95, 0xa4}},
	"boxHu":                           {Name: "boxHu", CodePoints: []int{9575}, Characters: []byte{0xe2, 0x95, 0xa7}},
	"boxUL":                           {Name: "boxUL", CodePoints: []int{9565}, Characters: []byte{0xe2, 0x95, 0x9d}},
	"boxUR":                           {Name: "boxUR", CodePoints: []int{9562}, Characters: []byte{0xe2, 0x95, 0x9a}},
	"boxUl":                           {Name: "boxUl", CodePoints: []int{9564}, Characters: []byte{0xe2, 0x95, 0x9c}},
	"boxUr":                           {Name: "boxUr", CodePoints: []int{9561}, Characters: []byte{0xe2, 0x95, 0x99}},
	"boxV":                            {Name: "boxV", CodePoints: []int{9553}, Characters: []byte{0xe2, 0x95, 0x91}},
	"boxVH":                           {Name: "boxVH", CodePoints: []int{9580}, Characters: []byte{0xe2, 0x95, 0xac}},
	"boxVL":                           {Name: "boxVL", CodePoints: []int{9571}, Characters: []byte{0xe2, 0x95, 0xa3}},
	"boxVR":                           {Name: "boxVR", CodePoints: []int{9568}, Characters: []byte{0xe2, 0x95, 0xa0}},
	"boxVh":                           {Name: "boxVh", CodePoints: []int{9579}, Characters: []byte{0xe2, 0x95, 0xab}},
	"boxVl":                           {Name: "boxVl", CodePoints: []int{9570}, Characters: []byte{0xe2, 0x95, 0xa2}},
	"boxVr":                           {Name: "boxVr", CodePoints: []int{9567}, Characters: []byte{0xe2, 0x95, 0x9f}},
	"boxbox":                          {Name: "boxbox", CodePoints: []int{10697}, Characters: []byte{0xe2, 0xa7, 0x89}},
	"boxdL":                           {Name: "boxdL", CodePoints: []int{9557}, Characters: []byte{0xe2, 0x95, 0x95}},
	"boxdR":                           {Name: "boxdR", CodePoints: []int{9554}, Characters: []byte{0xe2, 0x95, 0x92}},
	"boxdl":                           {Name: "boxdl", CodePoints: []int{9488}, Characters: []byte{0xe2, 0x94, 0x90}},
	"boxdr":                           {Name: "boxdr", CodePoints: []int{9484}, Characters: []byte{0xe2, 0x94, 0x8c}},
	"boxh":                            {Name: "boxh", CodePoints: []int{9472}, Characters: []byte{0xe2, 0x94, 0x80}},
	"boxhD":                           {Name: "boxhD", CodePoints: []int{9573}, Characters: []byte{0xe2, 0x95, 0xa5}},
	"boxhU":                           {Name: "boxhU", CodePoints: []int{9576}, Characters: []byte{0xe2, 0x95, 0xa8}},
	"boxhd":                           {Name: "boxhd", CodePoints: []int{9516}, Characters: []byte{0xe2, 0x94, 0xac}},
	"boxhu":                           {Name: "boxhu", CodePoints: []int{9524}, Characters: []byte{0xe2, 0x94, 0xb4}},
	"boxminus":                        {Name: "boxminus", CodePoints: []int{8863}, Characters: []byte{0xe2, 0x8a, 0x9f}},
	"boxplus":                         {Name: "boxplus", CodePoints: []int{8862}, Characters: []byte{0xe2, 0x8a, 0x9e}},
	"boxtimes":                        {Name: "boxtimes", CodePoints: []int{8864}, Characters: []byte{0xe2, 0x8a, 0xa0}},
	"boxuL":                           {Name: "boxuL", CodePoints: []int{9563}, Characters: []byte{0xe2, 0x95, 0x9b}},
	"boxuR":                           {Name: "boxuR", CodePoints: []int{9560}, Characters: []byte{0xe2, 0x95, 0x98}},
	"boxul":                           {Name: "boxul", CodePoints: []int{9496}, Characters: []byte{0xe2, 0x94, 0x98}},
	"boxur":                           {Name: "boxur", CodePoints: []int{9492}, Characters: []byte{0xe2, 0x94, 0x94}},
	"boxv":                            {Name: "boxv", CodePoints: []int{9474}, Characters: []byte{0xe2, 0x94, 0x82}},
	"boxvH":                           {Name: "boxvH", CodePoints: []int{9578}, Characters: []byte{0xe2, 0x95, 0xaa}},
	"boxvL":                           {Name: "boxvL", CodePoints: []int{9569}, Characters: []byte{0xe2, 0x95, 0xa1}},
	"boxvR":                           {Name: "boxvR", CodePoints: []int{9566}, Characters: []byte{0xe2, 0x95, 0x9e}},
	"boxvh":                           {Name: "boxvh", CodePoints: []int{9532}, Characters: []byte{0xe2, 0x94, 0xbc}},
	"boxvl":                           {Name: "boxvl", CodePoints: []int{9508}, Characters: []byte{0xe2, 0x94, 0xa4}},
	"boxvr":                           {Name: "boxvr", CodePoints: []int{9500}, Characters: []byte{0xe2, 0x94, 0x9c}},
	"bprime":                          {Name: "bprime", CodePoints: []int{8245}, Characters: []byte{0xe2, 0x80, 0xb5}},
	"breve":                           {Name: "breve", CodePoints: []int{728}, Characters: []byte{0xcb, 0x98}},
	"brvbar":                          {Name: "brvbar", CodePoints: []int{166}, Characters: []byte{0xc2, 0xa6}},
	"bscr":                            {Name: "bscr", CodePoints: []int{119991}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb7}},
	"bsemi":                           {Name: "bsemi", CodePoints: []int{8271}, Characters: []byte{0xe2, 0x81, 0x8f}},
	"bsim":                            {Name: "bsim", CodePoints: []int{8765}, Characters: []byte{0xe2, 0x88, 0xbd}},
	"bsime":                           {Name: "bsime", CodePoints: []int{8909}, Characters: []byte{0xe2, 0x8b, 0x8d}},
	"bsol":                            {Name: "bsol", CodePoints: []int{92}, Characters: []byte{0x5c}},
	"bsolb":                           {Name: "bsolb", CodePoints: []int{10693}, Characters: []byte{0xe2, 0xa7, 0x85}},
	"bsolhsub":                        {Name: "bsolhsub", CodePoints: []int{10184}, Characters: []byte{0xe2, 0x9f, 0x88}},
	"bull":                            {Name: "bull", CodePoints: []int{8226}, Characters: []byte{0xe2, 0x80, 0xa2}},
	"bullet":                          {Name: "bullet", CodePoints: []int{8226}, Characters: []byte{0xe2, 0x80, 0xa2}},
	"bump":                            {Name: "bump", CodePoints: []int{8782}, Characters: []byte{0xe2, 0x89, 0x8e}},
	"bumpE":                           {Name: "bumpE", CodePoints: []int{10926}, Characters: []byte{0xe2, 0xaa, 0xae}},
	"bumpe":                           {Name: "bumpe", CodePoints: []int{8783}, Characters: []byte{0xe2, 0x89, 0x8f}},
	"bumpeq":                          {Name: "bumpeq", CodePoints: []int{8783}, Characters: []byte{0xe2, 0x89, 0x8f}},
	"cacute":                          {Name: "cacute", CodePoints: []int{263}, Characters: []byte{0xc4, 0x87}},
	"cap":                             {Name: "cap", CodePoints: []int{8745}, Characters: []byte{0xe2, 0x88, 0xa9}},
	"capand":                          {Name: "capand", CodePoints: []int{10820}, Characters: []byte{0xe2, 0xa9, 0x84}},
	"capbrcup":                        {Name: "capbrcup", CodePoints: []int{10825}, Characters: []byte{0xe2, 0xa9, 0x89}},
	"capcap":                          {Name: "capcap", CodePoints: []int{10827}, Characters: []byte{0xe2, 0xa9, 0x8b}},
	"capcup":                          {Name: "capcup", CodePoints: []int{10823}, Characters: []byte{0xe2, 0xa9, 0x87}},
	"capdot":                          {Name: "capdot", CodePoints: []int{10816}, Characters: []byte{0xe2, 0xa9, 0x80}},
	"caps":                            {Name: "caps", CodePoints: []int{8745, 65024}, Characters: []byte{0xe2, 0x88, 0xa9, 0xef, 0xb8, 0x80}},
	"caret":                           {Name: "caret", CodePoints: []int{8257}, Characters: []byte{0xe2, 0x81, 0x81}},
	"caron":                           {Name: "caron", CodePoints: []int{711}, Characters: []byte{0xcb, 0x87}},
	"ccaps":                           {Name: "ccaps", CodePoints: []int{10829}, Characters: []byte{0xe2, 0xa9, 0x8d}},
	"ccaron":                          {Name: "ccaron", CodePoints: []int{269}, Characters: []byte{0xc4, 0x8d}},
	"ccedil":                          {Name: "ccedil", CodePoints: []int{231}, Characters: []byte{0xc3, 0xa7}},
	"ccirc":                           {Name: "ccirc", CodePoints: []int{265}, Characters: []byte{0xc4, 0x89}},
	"ccups":                           {Name: "ccups", CodePoints: []int{10828}, Characters: []byte{0xe2, 0xa9, 0x8c}},
	"ccupssm":                         {Name: "ccupssm", CodePoints: []int{10832}, Characters: []byte{0xe2, 0xa9, 0x90}},
	"cdot":                            {Name: "cdot", CodePoints: []int{267}, Characters: []byte{0xc4, 0x8b}},
	"cedil":                           {Name: "cedil", CodePoints: []int{184}, Characters: []byte{0xc2, 0xb8}},
	"cemptyv":                         {Name: "cemptyv", CodePoints: []int{10674}, Characters: []byte{0xe2, 0xa6, 0xb2}},
	"cent":                            {Name: "cent", CodePoints: []int{162}, Characters: []byte{0xc2, 0xa2}},
	"centerdot":                       {Name: "centerdot", CodePoints: []int{183}, Characters: []byte{0xc2, 0xb7}},
	"cfr":                             {Name: "cfr", CodePoints: []int{120096}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa0}},
	"chcy":                            {Name: "chcy", CodePoints: []int{1095}, Characters: []byte{0xd1, 0x87}},
	"check":                           {Name: "check", CodePoints: []int{10003}, Characters: []byte{0xe2, 0x9c, 0x93}},
	"checkmark":                       {Name: "checkmark", CodePoints: []int{10003}, Characters: []byte{0xe2, 0x9c, 0x93}},
	"chi":                             {Name: "chi", CodePoints: []int{967}, Characters: []byte{0xcf, 0x87}},
	"cir":                             {Name: "cir", CodePoints: []int{9675}, Characters: []byte{0xe2, 0x97, 0x8b}},
	"cirE":                            {Name: "cirE", CodePoints: []int{10691}, Characters: []byte{0xe2, 0xa7, 0x83}},
	"circ":                            {Name: "circ", CodePoints: []int{710}, Characters: []byte{0xcb, 0x86}},
	"circeq":                          {Name: "circeq", CodePoints: []int{8791}, Characters: []byte{0xe2, 0x89, 0x97}},
	"circlearrowleft":                 {Name: "circlearrowleft", CodePoints: []int{8634}, Characters: []byte{0xe2, 0x86, 0xba}},
	"circlearrowright":                {Name: "circlearrowright", CodePoints: []int{8635}, Characters: []byte{0xe2, 0x86, 0xbb}},
	"circledR":                        {Name: "circledR", CodePoints: []int{174}, Characters: []byte{0xc2, 0xae}},
	"circledS":                        {Name: "circledS", CodePoints: []int{9416}, Characters: []byte{0xe2, 0x93, 0x88}},
	"circledast":                      {Name: "circledast", CodePoints: []int{8859}, Characters: []byte{0xe2, 0x8a, 0x9b}},
	"circledcirc":                     {Name: "circledcirc", CodePoints: []int{8858}, Characters: []byte{0xe2, 0x8a, 0x9a}},
	"circleddash":                     {Name: "circleddash", CodePoints: []int{8861}, Characters: []byte{0xe2, 0x8a, 0x9d}},
	"cire":                            {Name: "cire", CodePoints: []int{8791}, Characters: []byte{0xe2, 0x89, 0x97}},
	"cirfnint":                        {Name: "cirfnint", CodePoints: []int{10768}, Characters: []byte{0xe2, 0xa8, 0x90}},
	"cirmid":                          {Name: "cirmid", CodePoints: []int{10991}, Characters: []byte{0xe2, 0xab, 0xaf}},
	"cirscir":                         {Name: "cirscir", CodePoints: []int{10690}, Characters: []byte{0xe2, 0xa7, 0x82}},
	"clubs":                           {Name: "clubs", CodePoints: []int{9827}, Characters: []byte{0xe2, 0x99, 0xa3}},
	"clubsuit":                        {Name: "clubsuit", CodePoints: []int{9827}, Characters: []byte{0xe2, 0x99, 0xa3}},
	"colon":                           {Name: "colon", CodePoints: []int{58}, Characters: []byte{0x3a}},
	"colone":                          {Name: "colone", CodePoints: []int{8788}, Characters: []byte{0xe2, 0x89, 0x94}},
	"coloneq":                         {Name: "coloneq", CodePoints: []int{8788}, Characters: []byte{0xe2, 0x89, 0x94}},
	"comma":                           {Name: "comma", CodePoints: []int{44}, Characters: []byte{0x2c}},
	"commat":                          {Name: "commat", CodePoints: []int{64}, Characters: []byte{0x40}},
	"comp":                            {Name: "comp", CodePoints: []int{8705}, Characters: []byte{0xe2, 0x88, 0x81}},
	"compfn":                          {Name: "compfn", CodePoints: []int{8728}, Characters: []byte{0xe2, 0x88, 0x98}},
	"complement":                      {Name: "complement", CodePoints: []int{8705}, Characters: []byte{0xe2, 0x88, 0x81}},
	"complexes":                       {Name: "complexes", CodePoints: []int{8450}, Characters: []byte{0xe2, 0x84, 0x82}},
	"cong":                            {Name: "cong", CodePoints: []int{8773}, Characters: []byte{0xe2, 0x89, 0x85}},
	"congdot":                         {Name: "congdot", CodePoints: []int{10861}, Characters: []byte{0xe2, 0xa9, 0xad}},
	"conint":                          {Name: "conint", CodePoints: []int{8750}, Characters: []byte{0xe2, 0x88, 0xae}},
	"copf":                            {Name: "copf", CodePoints: []int{120148}, Characters: []byte{0xf0, 0x9d, 0x95, 0x94}},
	"coprod":                          {Name: "coprod", CodePoints: []int{8720}, Characters: []byte{0xe2, 0x88, 0x90}},
	"copy":                            {Name: "copy", CodePoints: []int{169}, Characters: []byte{0xc2, 0xa9}},
	"copysr":                          {Name: "copysr", CodePoints: []int{8471}, Characters: []byte{0xe2, 0x84, 0x97}},
	"crarr":                           {Name: "crarr", CodePoints: []int{8629}, Characters: []byte{0xe2, 0x86, 0xb5}},
	"cross":                           {Name: "cross", CodePoints: []int{10007}, Characters: []byte{0xe2, 0x9c, 0x97}},
	"cscr":                            {Name: "cscr", CodePoints: []int{119992}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb8}},
	"csub":                            {Name: "csub", CodePoints: []int{10959}, Characters: []byte{0xe2, 0xab, 0x8f}},
	"csube":                           {Name: "csube", CodePoints: []int{10961}, Characters: []byte{0xe2, 0xab, 0x91}},
	"csup":                            {Name: "csup", CodePoints: []int{10960}, Characters: []byte{0xe2, 0xab, 0x90}},
	"csupe":                           {Name: "csupe", CodePoints: []int{10962}, Characters: []byte{0xe2, 0xab, 0x92}},
	"ctdot":                           {Name: "ctdot", CodePoints: []int{8943}, Characters: []byte{0xe2, 0x8b, 0xaf}},
	"cudarrl":                         {Name: "cudarrl", CodePoints: []int{10552}, Characters: []byte{0xe2, 0xa4, 0xb8}},
	"cudarrr":                         {Name: "cudarrr", CodePoints: []int{10549}, Characters: []byte{0xe2, 0xa4, 0xb5}},
	"cuepr":                           {Name: "cuepr", CodePoints: []int{8926}, Characters: []byte{0xe2, 0x8b, 0x9e}},
	"cuesc":                           {Name: "cuesc", CodePoints: []int{8927}, Characters: []byte{0xe2, 0x8b, 0x9f}},
	"cularr":                          {Name: "cularr", CodePoints: []int{8630}, Characters: []byte{0xe2, 0x86, 0xb6}},
	"cularrp":                         {Name: "cularrp", CodePoints: []int{10557}, Characters: []byte{0xe2, 0xa4, 0xbd}},
	"cup":                             {Name: "cup", CodePoints: []int{8746}, Characters: []byte{0xe2, 0x88, 0xaa}},
	"cupbrcap":                        {Name: "cupbrcap", CodePoints: []int{10824}, Characters: []byte{0xe2, 0xa9, 0x88}},
	"cupcap":                          {Name: "cupcap", CodePoints: []int{10822}, Characters: []byte{0xe2, 0xa9, 0x86}},
	"cupcup":                          {Name: "cupcup", CodePoints: []int{10826}, Characters: []byte{0xe2, 0xa9, 0x8a}},
	"cupdot":                          {Name: "cupdot", CodePoints: []int{8845}, Characters: []byte{0xe2, 0x8a, 0x8d}},
	"cupor":                           {Name: "cupor", CodePoints: []int{10821}, Characters: []byte{0xe2, 0xa9, 0x85}},
	"cups":                            {Name: "cups", CodePoints: []int{8746, 65024}, Characters: []byte{0xe2, 0x88, 0xaa, 0xef, 0xb8, 0x80}},
	"curarr":                          {Name: "curarr", CodePoints: []int{8631}, Characters: []byte{0xe2, 0x86, 0xb7}},
	"curarrm":                         {Name: "curarrm", CodePoints: []int{10556}, Characters: []byte{0xe2, 0xa4, 0xbc}},
	"curlyeqprec":                     {Name: "curlyeqprec", CodePoints: []int{8926}, Characters: []byte{0xe2, 0x8b, 0x9e}},
	"curlyeqsucc":                     {Name: "curlyeqsucc", CodePoints: []int{8927}, Characters: []byte{0xe2, 0x8b, 0x9f}},
	"curlyvee":                        {Name: "curlyvee", CodePoints: []int{8910}, Characters: []byte{0xe2, 0x8b, 0x8e}},
	"curlywedge":                      {Name: "curlywedge", CodePoints: []int{8911}, Characters: []byte{0xe2, 0x8b, 0x8f}},
	"curren":                          {Name: "curren", CodePoints: []int{164}, Characters: []byte{0xc2, 0xa4}},
	"curvearrowleft":                  {Name: "curvearrowleft", CodePoints: []int{8630}, Characters: []byte{0xe2, 0x86, 0xb6}},
	"curvearrowright":                 {Name: "curvearrowright", CodePoints: []int{8631}, Characters: []byte{0xe2, 0x86, 0xb7}},
	"cuvee":                           {Name: "cuvee", CodePoints: []int{8910}, Characters: []byte{0xe2, 0x8b, 0x8e}},
	"cuwed":                           {Name: "cuwed", CodePoints: []int{8911}, Characters: []byte{0xe2, 0x8b, 0x8f}},
	"cwconint":                        {Name: "cwconint", CodePoints: []int{8754}, Characters: []byte{0xe2, 0x88, 0xb2}},
	"cwint":                           {Name: "cwint", CodePoints: []int{8753}, Characters: []byte{0xe2, 0x88, 0xb1}},
	"cylcty":                          {Name: "cylcty", CodePoints: []int{9005}, Characters: []byte{0xe2, 0x8c, 0xad}},
	"dArr":                            {Name: "dArr", CodePoints: []int{8659}, Characters: []byte{0xe2, 0x87, 0x93}},
	"dHar":                            {Name: "dHar", CodePoints: []int{10597}, Characters: []byte{0xe2, 0xa5, 0xa5}},
	"dagger":                          {Name: "dagger", CodePoints: []int{8224}, Characters: []byte{0xe2, 0x80, 0xa0}},
	"daleth":                          {Name: "daleth", CodePoints: []int{8504}, Characters: []byte{0xe2, 0x84, 0xb8}},
	"darr":                            {Name: "darr", CodePoints: []int{8595}, Characters: []byte{0xe2, 0x86, 0x93}},
	"dash":                            {Name: "dash", CodePoints: []int{8208}, Characters: []byte{0xe2, 0x80, 0x90}},
	"dashv":                           {Name: "dashv", CodePoints: []int{8867}, Characters: []byte{0xe2, 0x8a, 0xa3}},
	"dbkarow":                         {Name: "dbkarow", CodePoints: []int{10511}, Characters: []byte{0xe2, 0xa4, 0x8f}},
	"dblac":                           {Name: "dblac", CodePoints: []int{733}, Characters: []byte{0xcb, 0x9d}},
	"dcaron":                          {Name: "dcaron", CodePoints: []int{271}, Characters: []byte{0xc4, 0x8f}},
	"dcy":                             {Name: "dcy", CodePoints: []int{1076}, Characters: []byte{0xd0, 0xb4}},
	"dd":                              {Name: "dd", CodePoints: []int{8518}, Characters: []byte{0xe2, 0x85, 0x86}},
	"ddagger":                         {Name: "ddagger", CodePoints: []int{8225}, Characters: []byte{0xe2, 0x80, 0xa1}},
	"ddarr":                           {Name: "ddarr", CodePoints: []int{8650}, Characters: []byte{0xe2, 0x87, 0x8a}},
	"ddotseq":                         {Name: "ddotseq", CodePoints: []int{10871}, Characters: []byte{0xe2, 0xa9, 0xb7}},
	"deg":                             {Name: "deg", CodePoints: []int{176}, Characters: []byte{0xc2, 0xb0}},
	"delta":                           {Name: "delta", CodePoints: []int{948}, Characters: []byte{0xce, 0xb4}},
	"demptyv":                         {Name: "demptyv", CodePoints: []int{10673}, Characters: []byte{0xe2, 0xa6, 0xb1}},
	"dfisht":                          {Name: "dfisht", CodePoints: []int{10623}, Characters: []byte{0xe2, 0xa5, 0xbf}},
	"dfr":                             {Name: "dfr", CodePoints: []int{120097}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa1}},
	"dharl":                           {Name: "dharl", CodePoints: []int{8643}, Characters: []byte{0xe2, 0x87, 0x83}},
	"dharr":                           {Name: "dharr", CodePoints: []int{8642}, Characters: []byte{0xe2, 0x87, 0x82}},
	"diam":                            {Name: "diam", CodePoints: []int{8900}, Characters: []byte{0xe2, 0x8b, 0x84}},
	"diamond":                         {Name: "diamond", CodePoints: []int{8900}, Characters: []byte{0xe2, 0x8b, 0x84}},
	"diamondsuit":                     {Name: "diamondsuit", CodePoints: []int{9830}, Characters: []byte{0xe2, 0x99, 0xa6}},
	"diams":                           {Name: "diams", CodePoints: []int{9830}, Characters: []byte{0xe2, 0x99, 0xa6}},
	"die":                             {Name: "die", CodePoints: []int{168}, Characters: []byte{0xc2, 0xa8}},
	"digamma":                         {Name: "digamma", CodePoints: []int{989}, Characters: []byte{0xcf, 0x9d}},
	"disin":                           {Name: "disin", CodePoints: []int{8946}, Characters: []byte{0xe2, 0x8b, 0xb2}},
	"div":                             {Name: "div", CodePoints: []int{247}, Characters: []byte{0xc3, 0xb7}},
	"divide":                          {Name: "divide", CodePoints: []int{247}, Characters: []byte{0xc3, 0xb7}},
	"divideontimes":                   {Name: "divideontimes", CodePoints: []int{8903}, Characters: []byte{0xe2, 0x8b, 0x87}},
	"divonx":                          {Name: "divonx", CodePoints: []int{8903}, Characters: []byte{0xe2, 0x8b, 0x87}},
	"djcy":                            {Name: "djcy", CodePoints: []int{1106}, Characters: []byte{0xd1, 0x92}},
	"dlcorn":                          {Name: "dlcorn", CodePoints: []int{8990}, Characters: []byte{0xe2, 0x8c, 0x9e}},
	"dlcrop":                          {Name: "dlcrop", CodePoints: []int{8973}, Characters: []byte{0xe2, 0x8c, 0x8d}},
	"dollar":                          {Name: "dollar", CodePoints: []int{36}, Characters: []byte{0x24}},
	"dopf":                            {Name: "dopf", CodePoints: []int{120149}, Characters: []byte{0xf0, 0x9d, 0x95, 0x95}},
	"dot":                             {Name: "dot", CodePoints: []int{729}, Characters: []byte{0xcb, 0x99}},
	"doteq":                           {Name: "doteq", CodePoints: []int{8784}, Characters: []byte{0xe2, 0x89, 0x90}},
	"doteqdot":                        {Name: "doteqdot", CodePoints: []int{8785}, Characters: []byte{0xe2, 0x89, 0x91}},
	"dotminus":                        {Name: "dotminus", CodePoints: []int{8760}, Characters: []byte{0xe2, 0x88, 0xb8}},
	"dotplus":                         {Name: "dotplus", CodePoints: []int{8724}, Characters: []byte{0xe2, 0x88, 0x94}},
	"dotsquare":                       {Name: "dotsquare", CodePoints: []int{8865}, Characters: []byte{0xe2, 0x8a, 0xa1}},
	"doublebarwedge":                  {Name: "doublebarwedge", CodePoints: []int{8966}, Characters: []byte{0xe2, 0x8c, 0x86}},
	"downarrow":                       {Name: "downarrow", CodePoints: []int{8595}, Characters: []byte{0xe2, 0x86, 0x93}},
	"downdownarrows":                  {Name: "downdownarrows", CodePoints: []int{8650}, Characters: []byte{0xe2, 0x87, 0x8a}},
	"downharpoonleft":                 {Name: "downharpoonleft", CodePoints: []int{8643}, Characters: []byte{0xe2, 0x87, 0x83}},
	"downharpoonright":                {Name: "downharpoonright", CodePoints: []int{8642}, Characters: []byte{0xe2, 0x87, 0x82}},
	"drbkarow":                        {Name: "drbkarow", CodePoints: []int{10512}, Characters: []byte{0xe2, 0xa4, 0x90}},
	"drcorn":                          {Name: "drcorn", CodePoints: []int{8991}, Characters: []byte{0xe2, 0x8c, 0x9f}},
	"drcrop":                          {Name: "drcrop", CodePoints: []int{8972}, Characters: []byte{0xe2, 0x8c, 0x8c}},
	"dscr":                            {Name: "dscr", CodePoints: []int{119993}, Characters: []byte{0xf0, 0x9d, 0x92, 0xb9}},
	"dscy":                            {Name: "dscy", CodePoints: []int{1109}, Characters: []byte{0xd1, 0x95}},
	"dsol":                            {Name: "dsol", CodePoints: []int{10742}, Characters: []byte{0xe2, 0xa7, 0xb6}},
	"dstrok":                          {Name: "dstrok", CodePoints: []int{273}, Characters: []byte{0xc4, 0x91}},
	"dtdot":                           {Name: "dtdot", CodePoints: []int{8945}, Characters: []byte{0xe2, 0x8b, 0xb1}},
	"dtri":                            {Name: "dtri", CodePoints: []int{9663}, Characters: []byte{0xe2, 0x96, 0xbf}},
	"dtrif":                           {Name: "dtrif", CodePoints: []int{9662}, Characters: []byte{0xe2, 0x96, 0xbe}},
	"duarr":                           {Name: "duarr", CodePoints: []int{8693}, Characters: []byte{0xe2, 0x87, 0xb5}},
	"duhar":                           {Name: "duhar", CodePoints: []int{10607}, Characters: []byte{0xe2, 0xa5, 0xaf}},
	"dwangle":                         {Name: "dwangle", CodePoints: []int{10662}, Characters: []byte{0xe2, 0xa6, 0xa6}},
	"dzcy":                            {Name: "dzcy", CodePoints: []int{1119}, Characters: []byte{0xd1, 0x9f}},
	"dzigrarr":                        {Name: "dzigrarr", CodePoints: []int{10239}, Characters: []byte{0xe2, 0x9f, 0xbf}},
	"eDDot":                           {Name: "eDDot", CodePoints: []int{10871}, Characters: []byte{0xe2, 0xa9, 0xb7}},
	"eDot":                            {Name: "eDot", CodePoints: []int{8785}, Characters: []byte{0xe2, 0x89, 0x91}},
	"eacute":                          {Name: "eacute", CodePoints: []int{233}, Characters: []byte{0xc3, 0xa9}},
	"easter":                          {Name: "easter", CodePoints: []int{10862}, Characters: []byte{0xe2, 0xa9, 0xae}},
	"ecaron":                          {Name: "ecaron", CodePoints: []int{283}, Characters: []byte{0xc4, 0x9b}},
	"ecir":                            {Name: "ecir", CodePoints: []int{8790}, Characters: []byte{0xe2, 0x89, 0x96}},
	"ecirc":                           {Name: "ecirc", CodePoints: []int{234}, Characters: []byte{0xc3, 0xaa}},
	"ecolon":                          {Name: "ecolon", CodePoints: []int{8789}, Characters: []byte{0xe2, 0x89, 0x95}},
	"ecy":                             {Name: "ecy", CodePoints: []int{1101}, Characters: []byte{0xd1, 0x8d}},
	"edot":                            {Name: "edot", CodePoints: []int{279}, Characters: []byte{0xc4, 0x97}},
	"ee":                              {Name: "ee", CodePoints: []int{8519}, Characters: []byte{0xe2, 0x85, 0x87}},
	"efDot":                           {Name: "efDot", CodePoints: []int{8786}, Characters: []byte{0xe2, 0x89, 0x92}},
	"efr":                             {Name: "efr", CodePoints: []int{120098}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa2}},
	"eg":                              {Name: "eg", CodePoints: []int{10906}, Characters: []byte{0xe2, 0xaa, 0x9a}},
	"egrave":                          {Name: "egrave", CodePoints: []int{232}, Characters: []byte{0xc3, 0xa8}},
	"egs":                             {Name: "egs", CodePoints: []int{10902}, Characters: []byte{0xe2, 0xaa, 0x96}},
	"egsdot":                          {Name: "egsdot", CodePoints: []int{10904}, Characters: []byte{0xe2, 0xaa, 0x98}},
	"el":                              {Name: "el", CodePoints: []int{10905}, Characters: []byte{0xe2, 0xaa, 0x99}},
	"elinters":                        {Name: "elinters", CodePoints: []int{9191}, Characters: []byte{0xe2, 0x8f, 0xa7}},
	"ell":                             {Name: "ell", CodePoints: []int{8467}, Characters: []byte{0xe2, 0x84, 0x93}},
	"els":                             {Name: "els", CodePoints: []int{10901}, Characters: []byte{0xe2, 0xaa, 0x95}},
	"elsdot":                          {Name: "elsdot", CodePoints: []int{10903}, Characters: []byte{0xe2, 0xaa, 0x97}},
	"emacr":                           {Name: "emacr", CodePoints: []int{275}, Characters: []byte{0xc4, 0x93}},
	"empty":                           {Name: "empty", CodePoints: []int{8709}, Characters: []byte{0xe2, 0x88, 0x85}},
	"emptyset":                        {Name: "emptyset", CodePoints: []int{8709}, Characters: []byte{0xe2, 0x88, 0x85}},
	"emptyv":                          {Name: "emptyv", CodePoints: []int{8709}, Characters: []byte{0xe2, 0x88, 0x85}},
	"emsp":                            {Name: "emsp", CodePoints: []int{8195}, Characters: []byte{0xe2, 0x80, 0x83}},
	"emsp13":                          {Name: "emsp13", CodePoints: []int{8196}, Characters: []byte{0xe2, 0x80, 0x84}},
	"emsp14":                          {Name: "emsp14", CodePoints: []int{8197}, Characters: []byte{0xe2, 0x80, 0x85}},
	"eng":                             {Name: "eng", CodePoints: []int{331}, Characters: []byte{0xc5, 0x8b}},
	"ensp":                            {Name: "ensp", CodePoints: []int{8194}, Characters: []byte{0xe2, 0x80, 0x82}},
	"eogon":                           {Name: "eogon", CodePoints: []int{281}, Characters: []byte{0xc4, 0x99}},
	"eopf":                            {Name: "eopf", CodePoints: []int{120150}, Characters: []byte{0xf0, 0x9d, 0x95, 0x96}},
	"epar":                            {Name: "epar", CodePoints: []int{8917}, Characters: []byte{0xe2, 0x8b, 0x95}},
	"eparsl":                          {Name: "eparsl", CodePoints: []int{10723}, Characters: []byte{0xe2, 0xa7, 0xa3}},
	"eplus":                           {Name: "eplus", CodePoints: []int{10865}, Characters: []byte{0xe2, 0xa9, 0xb1}},
	"epsi":                            {Name: "epsi", CodePoints: []int{949}, Characters: []byte{0xce, 0xb5}},
	"epsilon":                         {Name: "epsilon", CodePoints: []int{949}, Characters: []byte{0xce, 0xb5}},
	"epsiv":                           {Name: "epsiv", CodePoints: []int{1013}, Characters: []byte{0xcf, 0xb5}},
	"eqcirc":                          {Name: "eqcirc", CodePoints: []int{8790}, Characters: []byte{0xe2, 0x89, 0x96}},
	"eqcolon":                         {Name: "eqcolon", CodePoints: []int{8789}, Characters: []byte{0xe2, 0x89, 0x95}},
	"eqsim":                           {Name: "eqsim", CodePoints: []int{8770}, Characters: []byte{0xe2, 0x89, 0x82}},
	"eqslantgtr":                      {Name: "eqslantgtr", CodePoints: []int{10902}, Characters: []byte{0xe2, 0xaa, 0x96}},
	"eqslantless":                     {Name: "eqslantless", CodePoints: []int{10901}, Characters: []byte{0xe2, 0xaa, 0x95}},
	"equals":                          {Name: "equals", CodePoints: []int{61}, Characters: []byte{0x3d}},
	"equest":                          {Name: "equest", CodePoints: []int{8799}, Characters: []byte{0xe2, 0x89, 0x9f}},
	"equiv":                           {Name: "equiv", CodePoints: []int{8801}, Characters: []byte{0xe2, 0x89, 0xa1}},
	"equivDD":                         {Name: "equivDD", CodePoints: []int{10872}, Characters: []byte{0xe2, 0xa9, 0xb8}},
	"eqvparsl":                        {Name: "eqvparsl", CodePoints: []int{10725}, Characters: []byte{0xe2, 0xa7, 0xa5}},
	"erDot":                           {Name: "erDot", CodePoints: []int{8787}, Characters: []byte{0xe2, 0x89, 0x93}},
	"erarr":                           {Name: "erarr", CodePoints: []int{10609}, Characters: []byte{0xe2, 0xa5, 0xb1}},
	"escr":                            {Name: "escr", CodePoints: []int{8495}, Characters: []byte{0xe2, 0x84, 0xaf}},
	"esdot":                           {Name: "esdot", CodePoints: []int{8784}, Characters: []byte{0xe2, 0x89, 0x90}},
	"esim":                            {Name: "esim", CodePoints: []int{8770}, Characters: []byte{0xe2, 0x89, 0x82}},
	"eta":                             {Name: "eta", CodePoints: []int{951}, Characters: []byte{0xce, 0xb7}},
	"eth":                             {Name: "eth", CodePoints: []int{240}, Characters: []byte{0xc3, 0xb0}},
	"euml":                            {Name: "euml", CodePoints: []int{235}, Characters: []byte{0xc3, 0xab}},
	"euro":                            {Name: "euro", CodePoints: []int{8364}, Characters: []byte{0xe2, 0x82, 0xac}},
	"excl":                            {Name: "excl", CodePoints: []int{33}, Characters: []byte{0x21}},
	"exist":                           {Name: "exist", CodePoints: []int{8707}, Characters: []byte{0xe2, 0x88, 0x83}},
	"expectation":                     {Name: "expectation", CodePoints: []int{8496}, Characters: []byte{0xe2, 0x84, 0xb0}},
	"exponentiale":                    {Name: "exponentiale", CodePoints: []int{8519}, Characters: []byte{0xe2, 0x85, 0x87}},
	"fallingdotseq":                   {Name: "fallingdotseq", CodePoints: []int{8786}, Characters: []byte{0xe2, 0x89, 0x92}},
	"fcy":                             {Name: "fcy", CodePoints: []int{1092}, Characters: []byte{0xd1, 0x84}},
	"female":                          {Name: "female", CodePoints: []int{9792}, Characters: []byte{0xe2, 0x99, 0x80}},
	"ffilig":                          {Name: "ffilig", CodePoints: []int{64259}, Characters: []byte{0xef, 0xac, 0x83}},
	"fflig":                           {Name: "fflig", CodePoints: []int{64256}, Characters: []byte{0xef, 0xac, 0x80}},
	"ffllig":                          {Name: "ffllig", CodePoints: []int{64260}, Characters: []byte{0xef, 0xac, 0x84}},
	"ffr":                             {Name: "ffr", CodePoints: []int{120099}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa3}},
	"filig":                           {Name: "filig", CodePoints: []int{64257}, Characters: []byte{0xef, 0xac, 0x81}},
	"fjlig":                           {Name: "fjlig", CodePoints: []int{102, 106}, Characters: []byte{0x66, 0x6a}},
	"flat":                            {Name: "flat", CodePoints: []int{9837}, Characters: []byte{0xe2, 0x99, 0xad}},
	"fllig":                           {Name: "fllig", CodePoints: []int{64258}, Characters: []byte{0xef, 0xac, 0x82}},
	"fltns":                           {Name: "fltns", CodePoints: []int{9649}, Characters: []byte{0xe2, 0x96, 0xb1}},
	"fnof":                            {Name: "fnof", CodePoints: []int{402}, Characters: []byte{0xc6, 0x92}},
	"fopf":                            {Name: "fopf", CodePoints: []int{120151}, Characters: []byte{0xf0, 0x9d, 0x95, 0x97}},
	"forall":                          {Name: "forall", CodePoints: []int{8704}, Characters: []byte{0xe2, 0x88, 0x80}},
	"fork":                            {Name: "fork", CodePoints: []int{8916}, Characters: []byte{0xe2, 0x8b, 0x94}},
	"forkv":                           {Name: "forkv", CodePoints: []int{10969}, Characters: []byte{0xe2, 0xab, 0x99}},
	"fpartint":                        {Name: "fpartint", CodePoints: []int{10765}, Characters: []byte{0xe2, 0xa8, 0x8d}},
	"frac12":                          {Name: "frac12", CodePoints: []int{189}, Characters: []byte{0xc2, 0xbd}},
	"frac13":                          {Name: "frac13", CodePoints: []int{8531}, Characters: []byte{0xe2, 0x85, 0x93}},
	"frac14":                          {Name: "frac14", CodePoints: []int{188}, Characters: []byte{0xc2, 0xbc}},
	"frac15":                          {Name: "frac15", CodePoints: []int{8533}, Characters: []byte{0xe2, 0x85, 0x95}},
	"frac16":                          {Name: "frac16", CodePoints: []int{8537}, Characters: []byte{0xe2, 0x85, 0x99}},
	"frac18":                          {Name: "frac18", CodePoints: []int{8539}, Characters: []byte{0xe2, 0x85, 0x9b}},
	"frac23":                          {Name: "frac23", CodePoints: []int{8532}, Characters: []byte{0xe2, 0x85, 0x94}},
	"frac25":                          {Name: "frac25", CodePoints: []int{8534}, Characters: []byte{0xe2, 0x85, 0x96}},
	"frac34":                          {Name: "frac34", CodePoints: []int{190}, Characters: []byte{0xc2, 0xbe}},
	"frac35":                          {Name: "frac35", CodePoints: []int{8535}, Characters: []byte{0xe2, 0x85, 0x97}},
	"frac38":                          {Name: "frac38", CodePoints: []int{8540}, Characters: []byte{0xe2, 0x85, 0x9c}},
	"frac45":                          {Name: "frac45", CodePoints: []int{8536}, Characters: []byte{0xe2, 0x85, 0x98}},
	"frac56":                          {Name: "frac56", CodePoints: []int{8538}, Characters: []byte{0xe2, 0x85, 0x9a}},
	"frac58":                          {Name: "frac58", CodePoints: []int{8541}, Characters: []byte{0xe2, 0x85, 0x9d}},
	"frac78":                          {Name: "frac78", CodePoints: []int{8542}, Characters: []byte{0xe2, 0x85, 0x9e}},
	"frasl":                           {Name: "frasl", CodePoints: []int{8260}, Characters: []byte{0xe2, 0x81, 0x84}},
	"frown":                           {Name: "frown", CodePoints: []int{8994}, Characters: []byte{0xe2, 0x8c, 0xa2}},
	"fscr":                            {Name: "fscr", CodePoints: []int{119995}, Characters: []byte{0xf0, 0x9d, 0x92, 0xbb}},
	"gE":                              {Name: "gE", CodePoints: []int{8807}, Characters: []byte{0xe2, 0x89, 0xa7}},
	"gEl":                             {Name: "gEl", CodePoints: []int{10892}, Characters: []byte{0xe2, 0xaa, 0x8c}},
	"gacute":                          {Name: "gacute", CodePoints: []int{501}, Characters: []byte{0xc7, 0xb5}},
	"gamma":                           {Name: "gamma", CodePoints: []int{947}, Characters: []byte{0xce, 0xb3}},
	"gammad":                          {Name: "gammad", CodePoints: []int{989}, Characters: []byte{0xcf, 0x9d}},
	"gap":                             {Name: "gap", CodePoints: []int{10886}, Characters: []byte{0xe2, 0xaa, 0x86}},
	"gbreve":                          {Name: "gbreve", CodePoints: []int{287}, Characters: []byte{0xc4, 0x9f}},
	"gcirc":                           {Name: "gcirc", CodePoints: []int{285}, Characters: []byte{0xc4, 0x9d}},
	"gcy":                             {Name: "gcy", CodePoints: []int{1075}, Characters: []byte{0xd0, 0xb3}},
	"gdot":                            {Name: "gdot", CodePoints: []int{289}, Characters: []byte{0xc4, 0xa1}},
	"ge":                              {Name: "ge", CodePoints: []int{8805}, Characters: []byte{0xe2, 0x89, 0xa5}},
	"gel":                             {Name: "gel", CodePoints: []int{8923}, Characters: []byte{0xe2, 0x8b, 0x9b}},
	"geq":                             {Name: "geq", CodePoints: []int{8805}, Characters: []byte{0xe2, 0x89, 0xa5}},
	"geqq":                            {Name: "geqq", CodePoints: []int{8807}, Characters: []byte{0xe2, 0x89, 0xa7}},
	"geqslant":                        {Name: "geqslant", CodePoints: []int{10878}, Characters: []byte{0xe2, 0xa9, 0xbe}},
	"ges":                             {Name: "ges", CodePoints: []int{10878}, Characters: []byte{0xe2, 0xa9, 0xbe}},
	"gescc":                           {Name: "gescc", CodePoints: []int{10921}, Characters: []byte{0xe2, 0xaa, 0xa9}},
	"gesdot":                          {Name: "gesdot", CodePoints: []int{10880}, Characters: []byte{0xe2, 0xaa, 0x80}},
	"gesdoto":                         {Name: "gesdoto", CodePoints: []int{10882}, Characters: []byte{0xe2, 0xaa, 0x82}},
	"gesdotol":                        {Name: "gesdotol", CodePoints: []int{10884}, Characters: []byte{0xe2, 0xaa, 0x84}},
	"gesl":                            {Name: "gesl", CodePoints: []int{8923, 65024}, Characters: []byte{0xe2, 0x8b, 0x9b, 0xef, 0xb8, 0x80}},
	"gesles":                          {Name: "gesles", CodePoints: []int{10900}, Characters: []byte{0xe2, 0xaa, 0x94}},
	"gfr":                             {Name: "gfr", CodePoints: []int{120100}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa4}},
	"gg":                              {Name: "gg", CodePoints: []int{8811}, Characters: []byte{0xe2, 0x89, 0xab}},
	"ggg":                             {Name: "ggg", CodePoints: []int{8921}, Characters: []byte{0xe2, 0x8b, 0x99}},
	"gimel":                           {Name: "gimel", CodePoints: []int{8503}, Characters: []byte{0xe2, 0x84, 0xb7}},
	"gjcy":                            {Name: "gjcy", CodePoints: []int{1107}, Characters: []byte{0xd1, 0x93}},
	"gl":                              {Name: "gl", CodePoints: []int{8823}, Characters: []byte{0xe2, 0x89, 0xb7}},
	"glE":                             {Name: "glE", CodePoints: []int{10898}, Characters: []byte{0xe2, 0xaa, 0x92}},
	"gla":                             {Name: "gla", CodePoints: []int{10917}, Characters: []byte{0xe2, 0xaa, 0xa5}},
	"glj":                             {Name: "glj", CodePoints: []int{10916}, Characters: []byte{0xe2, 0xaa, 0xa4}},
	"gnE":                             {Name: "gnE", CodePoints: []int{8809}, Characters: []byte{0xe2, 0x89, 0xa9}},
	"gnap":                            {Name: "gnap", CodePoints: []int{10890}, Characters: []byte{0xe2, 0xaa, 0x8a}},
	"gnapprox":                        {Name: "gnapprox", CodePoints: []int{10890}, Characters: []byte{0xe2, 0xaa, 0x8a}},
	"gne":                             {Name: "gne", CodePoints: []int{10888}, Characters: []byte{0xe2, 0xaa, 0x88}},
	"gneq":                            {Name: "gneq", CodePoints: []int{10888}, Characters: []byte{0xe2, 0xaa, 0x88}},
	"gneqq":                           {Name: "gneqq", CodePoints: []int{8809}, Characters: []byte{0xe2, 0x89, 0xa9}},
	"gnsim":                           {Name: "gnsim", CodePoints: []int{8935}, Characters: []byte{0xe2, 0x8b, 0xa7}},
	"gopf":                            {Name: "gopf", CodePoints: []int{120152}, Characters: []byte{0xf0, 0x9d, 0x95, 0x98}},
	"grave":                           {Name: "grave", CodePoints: []int{96}, Characters: []byte{0x60}},
	"gscr":                            {Name: "gscr", CodePoints: []int{8458}, Characters: []byte{0xe2, 0x84, 0x8a}},
	"gsim":                            {Name: "gsim", CodePoints: []int{8819}, Characters: []byte{0xe2, 0x89, 0xb3}},
	"gsime":                           {Name: "gsime", CodePoints: []int{10894}, Characters: []byte{0xe2, 0xaa, 0x8e}},
	"gsiml":                           {Name: "gsiml", CodePoints: []int{10896}, Characters: []byte{0xe2, 0xaa, 0x90}},
	"gt":                              {Name: "gt", CodePoints: []int{62}, Characters: []byte{0x3e}},
	"gtcc":                            {Name: "gtcc", CodePoints: []int{10919}, Characters: []byte{0xe2, 0xaa, 0xa7}},
	"gtcir":                           {Name: "gtcir", CodePoints: []int{10874}, Characters: []byte{0xe2, 0xa9, 0xba}},
	"gtdot":                           {Name: "gtdot", CodePoints: []int{8919}, Characters: []byte{0xe2, 0x8b, 0x97}},
	"gtlPar":                          {Name: "gtlPar", CodePoints: []int{10645}, Characters: []byte{0xe2, 0xa6, 0x95}},
	"gtquest":                         {Name: "gtquest", CodePoints: []int{10876}, Characters: []byte{0xe2, 0xa9, 0xbc}},
	"gtrapprox":                       {Name: "gtrapprox", CodePoints: []int{10886}, Characters: []byte{0xe2, 0xaa, 0x86}},
	"gtrarr":                          {Name: "gtrarr", CodePoints: []int{10616}, Characters: []byte{0xe2, 0xa5, 0xb8}},
	"gtrdot":                          {Name: "gtrdot", CodePoints: []int{8919}, Characters: []byte{0xe2, 0x8b, 0x97}},
	"gtreqless":                       {Name: "gtreqless", CodePoints: []int{8923}, Characters: []byte{0xe2, 0x8b, 0x9b}},
	"gtreqqless":                      {Name: "gtreqqless", CodePoints: []int{10892}, Characters: []byte{0xe2, 0xaa, 0x8c}},
	"gtrless":                         {Name: "gtrless", CodePoints: []int{8823}, Characters: []byte{0xe2, 0x89, 0xb7}},
	"gtrsim":                          {Name: "gtrsim", CodePoints: []int{8819}, Characters: []byte{0xe2, 0x89, 0xb3}},
	"gvertneqq":                       {Name: "gvertneqq", CodePoints: []int{8809, 65024}, Characters: []byte{0xe2, 0x89, 0xa9, 0xef, 0xb8, 0x80}},
	"gvnE":                            {Name: "gvnE", CodePoints: []int{8809, 65024}, Characters: []byte{0xe2, 0x89, 0xa9, 0xef, 0xb8, 0x80}},
	"hArr":                            {Name: "hArr", CodePoints: []int{8660}, Characters: []byte{0xe2, 0x87, 0x94}},
	"hairsp":                          {Name: "hairsp", CodePoints: []int{8202}, Characters: []byte{0xe2, 0x80, 0x8a}},
	"half":                            {Name: "half", CodePoints: []int{189}, Characters: []byte{0xc2, 0xbd}},
	"hamilt":                          {Name: "hamilt", CodePoints: []int{8459}, Characters: []byte{0xe2, 0x84, 0x8b}},
	"hardcy":                          {Name: "hardcy", CodePoints: []int{1098}, Characters: []byte{0xd1, 0x8a}},
	"harr":                            {Name: "harr", CodePoints: []int{8596}, Characters: []byte{0xe2, 0x86, 0x94}},
	"harrcir":                         {Name: "harrcir", CodePoints: []int{10568}, Characters: []byte{0xe2, 0xa5, 0x88}},
	"harrw":                           {Name: "harrw", CodePoints: []int{8621}, Characters: []byte{0xe2, 0x86, 0xad}},
	"hbar":                            {Name: "hbar", CodePoints: []int{8463}, Characters: []byte{0xe2, 0x84, 0x8f}},
	"hcirc":                           {Name: "hcirc", CodePoints: []int{293}, Characters: []byte{0xc4, 0xa5}},
	"hearts":                          {Name: "hearts", CodePoints: []int{9829}, Characters: []byte{0xe2, 0x99, 0xa5}},
	"heartsuit":                       {Name: "heartsuit", CodePoints: []int{9829}, Characters: []byte{0xe2, 0x99, 0xa5}},
	"hellip":                          {Name: "hellip", CodePoints: []int{8230}, Characters: []byte{0xe2, 0x80, 0xa6}},
	"hercon":                          {Name: "hercon", CodePoints: []int{8889}, Characters: []byte{0xe2, 0x8a, 0xb9}},
	"hfr":                             {Name: "hfr", CodePoints: []int{120101}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa5}},
	"hksearow":                        {Name: "hksearow", CodePoints: []int{10533}, Characters: []byte{0xe2, 0xa4, 0xa5}},
	"hkswarow":                        {Name: "hkswarow", CodePoints: []int{10534}, Characters: []byte{0xe2, 0xa4, 0xa6}},
	"hoarr":                           {Name: "hoarr", CodePoints: []int{8703}, Characters: []byte{0xe2, 0x87, 0xbf}},
	"homtht":                          {Name: "homtht", CodePoints: []int{8763}, Characters: []byte{0xe2, 0x88, 0xbb}},
	"hookleftarrow":                   {Name: "hookleftarrow", CodePoints: []int{8617}, Characters: []byte{0xe2, 0x86, 0xa9}},
	"hookrightarrow":                  {Name: "hookrightarrow", CodePoints: []int{8618}, Characters: []byte{0xe2, 0x86, 0xaa}},
	"hopf":                            {Name: "hopf", CodePoints: []int{120153}, Characters: []byte{0xf0, 0x9d, 0x95, 0x99}},
	"horbar":                          {Name: "horbar", CodePoints: []int{8213}, Characters: []byte{0xe2, 0x80, 0x95}},
	"hscr":                            {Name: "hscr", CodePoints: []int{119997}, Characters: []byte{0xf0, 0x9d, 0x92, 0xbd}},
	"hslash":                          {Name: "hslash", CodePoints: []int{8463}, Characters: []byte{0xe2, 0x84, 0x8f}},
	"hstrok":                          {Name: "hstrok", CodePoints: []int{295}, Characters: []byte{0xc4, 0xa7}},
	"hybull":                          {Name: "hybull", CodePoints: []int{8259}, Characters: []byte{0xe2, 0x81, 0x83}},
	"hyphen":                          {Name: "hyphen", CodePoints: []int{8208}, Characters: []byte{0xe2, 0x80, 0x90}},
	"iacute":                          {Name: "iacute", CodePoints: []int{237}, Characters: []byte{0xc3, 0xad}},
	"ic":                              {Name: "ic", CodePoints: []int{8291}, Characters: []byte{0xe2, 0x81, 0xa3}},
	"icirc":                           {Name: "icirc", CodePoints: []int{238}, Characters: []byte{0xc3, 0xae}},
	"icy":                             {Name: "icy", CodePoints: []int{1080}, Characters: []byte{0xd0, 0xb8}},
	"iecy":                            {Name: "iecy", CodePoints: []int{1077}, Characters: []byte{0xd0, 0xb5}},
	"iexcl":                           {Name: "iexcl", CodePoints: []int{161}, Characters: []byte{0xc2, 0xa1}},
	"iff":                             {Name: "iff", CodePoints: []int{8660}, Characters: []byte{0xe2, 0x87, 0x94}},
	"ifr":                             {Name: "ifr", CodePoints: []int{120102}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa6}},
	"igrave":                          {Name: "igrave", CodePoints: []int{236}, Characters: []byte{0xc3, 0xac}},
	"ii":                              {Name: "ii", CodePoints: []int{8520}, Characters: []byte{0xe2, 0x85, 0x88}},
	"iiiint":                          {Name: "iiiint", CodePoints: []int{10764}, Characters: []byte{0xe2, 0xa8, 0x8c}},
	"iiint":                           {Name: "iiint", CodePoints: []int{8749}, Characters: []byte{0xe2, 0x88, 0xad}},
	"iinfin":                          {Name: "iinfin", CodePoints: []int{10716}, Characters: []byte{0xe2, 0xa7, 0x9c}},
	"iiota":                           {Name: "iiota", CodePoints: []int{8489}, Characters: []byte{0xe2, 0x84, 0xa9}},
	"ijlig":                           {Name: "ijlig", CodePoints: []int{307}, Characters: []byte{0xc4, 0xb3}},
	"imacr":                           {Name: "imacr", CodePoints: []int{299}, Characters: []byte{0xc4, 0xab}},
	"image":                           {Name: "image", CodePoints: []int{8465}, Characters: []byte{0xe2, 0x84, 0x91}},
	"imagline":                        {Name: "imagline", CodePoints: []int{8464}, Characters: []byte{0xe2, 0x84, 0x90}},
	"imagpart":                        {Name: "imagpart", CodePoints: []int{8465}, Characters: []byte{0xe2, 0x84, 0x91}},
	"imath":                           {Name: "imath", CodePoints: []int{305}, Characters: []byte{0xc4, 0xb1}},
	"imof":                            {Name: "imof", CodePoints: []int{8887}, Characters: []byte{0xe2, 0x8a, 0xb7}},
	"imped":                           {Name: "imped", CodePoints: []int{437}, Characters: []byte{0xc6, 0xb5}},
	"in":                              {Name: "in", CodePoints: []int{8712}, Characters: []byte{0xe2, 0x88, 0x88}},
	"incare":                          {Name: "incare", CodePoints: []int{8453}, Characters: []byte{0xe2, 0x84, 0x85}},
	"infin":                           {Name: "infin", CodePoints: []int{8734}, Characters: []byte{0xe2, 0x88, 0x9e}},
	"infintie":                        {Name: "infintie", CodePoints: []int{10717}, Characters: []byte{0xe2, 0xa7, 0x9d}},
	"inodot":                          {Name: "inodot", CodePoints: []int{305}, Characters: []byte{0xc4, 0xb1}},
	"int":                             {Name: "int", CodePoints: []int{8747}, Characters: []byte{0xe2, 0x88, 0xab}},
	"intcal":                          {Name: "intcal", CodePoints: []int{8890}, Characters: []byte{0xe2, 0x8a, 0xba}},
	"integers":                        {Name: "integers", CodePoints: []int{8484}, Characters: []byte{0xe2, 0x84, 0xa4}},
	"intercal":                        {Name: "intercal", CodePoints: []int{8890}, Characters: []byte{0xe2, 0x8a, 0xba}},
	"intlarhk":                        {Name: "intlarhk", CodePoints: []int{10775}, Characters: []byte{0xe2, 0xa8, 0x97}},
	"intprod":                         {Name: "intprod", CodePoints: []int{10812}, Characters: []byte{0xe2, 0xa8, 0xbc}},
	"iocy":                            {Name: "iocy", CodePoints: []int{1105}, Characters: []byte{0xd1, 0x91}},
	"iogon":                           {Name: "iogon", CodePoints: []int{303}, Characters: []byte{0xc4, 0xaf}},
	"iopf":                            {Name: "iopf", CodePoints: []int{120154}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9a}},
	"iota":                            {Name: "iota", CodePoints: []int{953}, Characters: []byte{0xce, 0xb9}},
	"iprod":                           {Name: "iprod", CodePoints: []int{10812}, Characters: []byte{0xe2, 0xa8, 0xbc}},
	"iquest":                          {Name: "iquest", CodePoints: []int{191}, Characters: []byte{0xc2, 0xbf}},
	"iscr":                            {Name: "iscr", CodePoints: []int{119998}, Characters: []byte{0xf0, 0x9d, 0x92, 0xbe}},
	"isin":                            {Name: "isin", CodePoints: []int{8712}, Characters: []byte{0xe2, 0x88, 0x88}},
	"isinE":                           {Name: "isinE", CodePoints: []int{8953}, Characters: []byte{0xe2, 0x8b, 0xb9}},
	"isindot":                         {Name: "isindot", CodePoints: []int{8949}, Characters: []byte{0xe2, 0x8b, 0xb5}},
	"isins":                           {Name: "isins", CodePoints: []int{8948}, Characters: []byte{0xe2, 0x8b, 0xb4}},
	"isinsv":                          {Name: "isinsv", CodePoints: []int{8947}, Characters: []byte{0xe2, 0x8b, 0xb3}},
	"isinv":                           {Name: "isinv", CodePoints: []int{8712}, Characters: []byte{0xe2, 0x88, 0x88}},
	"it":                              {Name: "it", CodePoints: []int{8290}, Characters: []byte{0xe2, 0x81, 0xa2}},
	"itilde":                          {Name: "itilde", CodePoints: []int{297}, Characters: []byte{0xc4, 0xa9}},
	"iukcy":                           {Name: "iukcy", CodePoints: []int{1110}, Characters: []byte{0xd1, 0x96}},
	"iuml":                            {Name: "iuml", CodePoints: []int{239}, Characters: []byte{0xc3, 0xaf}},
	"jcirc":                           {Name: "jcirc", CodePoints: []int{309}, Characters: []byte{0xc4, 0xb5}},
	"jcy":                             {Name: "jcy", CodePoints: []int{1081}, Characters: []byte{0xd0, 0xb9}},
	"jfr":                             {Name: "jfr", CodePoints: []int{120103}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa7}},
	"jmath":                           {Name: "jmath", CodePoints: []int{567}, Characters: []byte{0xc8, 0xb7}},
	"jopf":                            {Name: "jopf", CodePoints: []int{120155}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9b}},
	"jscr":                            {Name: "jscr", CodePoints: []int{119999}, Characters: []byte{0xf0, 0x9d, 0x92, 0xbf}},
	"jsercy":                          {Name: "jsercy", CodePoints: []int{1112}, Characters: []byte{0xd1, 0x98}},
	"jukcy":                           {Name: "jukcy", CodePoints: []int{1108}, Characters: []byte{0xd1, 0x94}},
	"kappa":                           {Name: "kappa", CodePoints: []int{954}, Characters: []byte{0xce, 0xba}},
	"kappav":                          {Name: "kappav", CodePoints: []int{1008}, Characters: []byte{0xcf, 0xb0}},
	"kcedil":                          {Name: "kcedil", CodePoints: []int{311}, Characters: []byte{0xc4, 0xb7}},
	"kcy":                             {Name: "kcy", CodePoints: []int{1082}, Characters: []byte{0xd0, 0xba}},
	"kfr":                             {Name: "kfr", CodePoints: []int{120104}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa8}},
	"kgreen":                          {Name: "kgreen", CodePoints: []int{312}, Characters: []byte{0xc4, 0xb8}},
	"khcy":                            {Name: "khcy", CodePoints: []int{1093}, Characters: []byte{0xd1, 0x85}},
	"kjcy":                            {Name: "kjcy", CodePoints: []int{1116}, Characters: []byte{0xd1, 0x9c}},
	"kopf":                            {Name: "kopf", CodePoints: []int{120156}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9c}},
	"kscr":                            {Name: "kscr", CodePoints: []int{120000}, Characters: []byte{0xf0, 0x9d, 0x93, 0x80}},
	"lAarr":                           {Name: "lAarr", CodePoints: []int{8666}, Characters: []byte{0xe2, 0x87, 0x9a}},
	"lArr":                            {Name: "lArr", CodePoints: []int{8656}, Characters: []byte{0xe2, 0x87, 0x90}},
	"lAtail":                          {Name: "lAtail", CodePoints: []int{10523}, Characters: []byte{0xe2, 0xa4, 0x9b}},
	"lBarr":                           {Name: "lBarr", CodePoints: []int{10510}, Characters: []byte{0xe2, 0xa4, 0x8e}},
	"lE":                              {Name: "lE", CodePoints: []int{8806}, Characters: []byte{0xe2, 0x89, 0xa6}},
	"lEg":                             {Name: "lEg", CodePoints: []int{10891}, Characters: []byte{0xe2, 0xaa, 0x8b}},
	"lHar":                            {Name: "lHar", CodePoints: []int{10594}, Characters: []byte{0xe2, 0xa5, 0xa2}},
	"lacute":                          {Name: "lacute", CodePoints: []int{314}, Characters: []byte{0xc4, 0xba}},
	"laemptyv":                        {Name: "laemptyv", CodePoints: []int{10676}, Characters: []byte{0xe2, 0xa6, 0xb4}},
	"lagran":                          {Name: "lagran", CodePoints: []int{8466}, Characters: []byte{0xe2, 0x84, 0x92}},
	"lambda":                          {Name: "lambda", CodePoints: []int{955}, Characters: []byte{0xce, 0xbb}},
	"lang":                            {Name: "lang", CodePoints: []int{10216}, Characters: []byte{0xe2, 0x9f, 0xa8}},
	"langd":                           {Name: "langd", CodePoints: []int{10641}, Characters: []byte{0xe2, 0xa6, 0x91}},
	"langle":                          {Name: "langle", CodePoints: []int{10216}, Characters: []byte{0xe2, 0x9f, 0xa8}},
	"lap":                             {Name: "lap", CodePoints: []int{10885}, Characters: []byte{0xe2, 0xaa, 0x85}},
	"laquo":                           {Name: "laquo", CodePoints: []int{171}, Characters: []byte{0xc2, 0xab}},
	"larr":                            {Name: "larr", CodePoints: []int{8592}, Characters: []byte{0xe2, 0x86, 0x90}},
	"larrb":                           {Name: "larrb", CodePoints: []int{8676}, Characters: []byte{0xe2, 0x87, 0xa4}},
	"larrbfs":                         {Name: "larrbfs", CodePoints: []int{10527}, Characters: []byte{0xe2, 0xa4, 0x9f}},
	"larrfs":                          {Name: "larrfs", CodePoints: []int{10525}, Characters: []byte{0xe2, 0xa4, 0x9d}},
	"larrhk":                          {Name: "larrhk", CodePoints: []int{8617}, Characters: []byte{0xe2, 0x86, 0xa9}},
	"larrlp":                          {Name: "larrlp", CodePoints: []int{8619}, Characters: []byte{0xe2, 0x86, 0xab}},
	"larrpl":                          {Name: "larrpl", CodePoints: []int{10553}, Characters: []byte{0xe2, 0xa4, 0xb9}},
	"larrsim":                         {Name: "larrsim", CodePoints: []int{10611}, Characters: []byte{0xe2, 0xa5, 0xb3}},
	"larrtl":                          {Name: "larrtl", CodePoints: []int{8610}, Characters: []byte{0xe2, 0x86, 0xa2}},
	"lat":                             {Name: "lat", CodePoints: []int{10923}, Characters: []byte{0xe2, 0xaa, 0xab}},
	"latail":                          {Name: "latail", CodePoints: []int{10521}, Characters: []byte{0xe2, 0xa4, 0x99}},
	"late":                            {Name: "late", CodePoints: []int{10925}, Characters: []byte{0xe2, 0xaa, 0xad}},
	"lates":                           {Name: "lates", CodePoints: []int{10925, 65024}, Characters: []byte{0xe2, 0xaa, 0xad, 0xef, 0xb8, 0x80}},
	"lbarr":                           {Name: "lbarr", CodePoints: []int{10508}, Characters: []byte{0xe2, 0xa4, 0x8c}},
	"lbbrk":                           {Name: "lbbrk", CodePoints: []int{10098}, Characters: []byte{0xe2, 0x9d, 0xb2}},
	"lbrace":                          {Name: "lbrace", CodePoints: []int{123}, Characters: []byte{0x7b}},
	"lbrack":                          {Name: "lbrack", CodePoints: []int{91}, Characters: []byte{0x5b}},
	"lbrke":                           {Name: "lbrke", CodePoints: []int{10635}, Characters: []byte{0xe2, 0xa6, 0x8b}},
	"lbrksld":                         {Name: "lbrksld", CodePoints: []int{10639}, Characters: []byte{0xe2, 0xa6, 0x8f}},
	"lbrkslu":                         {Name: "lbrkslu", CodePoints: []int{10637}, Characters: []byte{0xe2, 0xa6, 0x8d}},
	"lcaron":                          {Name: "lcaron", CodePoints: []int{318}, Characters: []byte{0xc4, 0xbe}},
	"lcedil":                          {Name: "lcedil", CodePoints: []int{316}, Characters: []byte{0xc4, 0xbc}},
	"lceil":                           {Name: "lceil", CodePoints: []int{8968}, Characters: []byte{0xe2, 0x8c, 0x88}},
	"lcub":                            {Name: "lcub", CodePoints: []int{123}, Characters: []byte{0x7b}},
	"lcy":                             {Name: "lcy", CodePoints: []int{1083}, Characters: []byte{0xd0, 0xbb}},
	"ldca":                            {Name: "ldca", CodePoints: []int{10550}, Characters: []byte{0xe2, 0xa4, 0xb6}},
	"ldquo":                           {Name: "ldquo", CodePoints: []int{8220}, Characters: []byte{0xe2, 0x80, 0x9c}},
	"ldquor":                          {Name: "ldquor", CodePoints: []int{8222}, Characters: []byte{0xe2, 0x80, 0x9e}},
	"ldrdhar":                         {Name: "ldrdhar", CodePoints: []int{10599}, Characters: []byte{0xe2, 0xa5, 0xa7}},
	"ldrushar":                        {Name: "ldrushar", CodePoints: []int{10571}, Characters: []byte{0xe2, 0xa5, 0x8b}},
	"ldsh":                            {Name: "ldsh", CodePoints: []int{8626}, Characters: []byte{0xe2, 0x86, 0xb2}},
	"le":                              {Name: "le", CodePoints: []int{8804}, Characters: []byte{0xe2, 0x89, 0xa4}},
	"leftarrow":                       {Name: "leftarrow", CodePoints: []int{8592}, Characters: []byte{0xe2, 0x86, 0x90}},
	"leftarrowtail":                   {Name: "leftarrowtail", CodePoints: []int{8610}, Characters: []byte{0xe2, 0x86, 0xa2}},
	"leftharpoondown":                 {Name: "leftharpoondown", CodePoints: []int{8637}, Characters: []byte{0xe2, 0x86, 0xbd}},
	"leftharpoonup":                   {Name: "leftharpoonup", CodePoints: []int{8636}, Characters: []byte{0xe2, 0x86, 0xbc}},
	"leftleftarrows":                  {Name: "leftleftarrows", CodePoints: []int{8647}, Characters: []byte{0xe2, 0x87, 0x87}},
	"leftrightarrow":                  {Name: "leftrightarrow", CodePoints: []int{8596}, Characters: []byte{0xe2, 0x86, 0x94}},
	"leftrightarrows":                 {Name: "leftrightarrows", CodePoints: []int{8646}, Characters: []byte{0xe2, 0x87, 0x86}},
	"leftrightharpoons":               {Name: "leftrightharpoons", CodePoints: []int{8651}, Characters: []byte{0xe2, 0x87, 0x8b}},
	"leftrightsquigarrow":             {Name: "leftrightsquigarrow", CodePoints: []int{8621}, Characters: []byte{0xe2, 0x86, 0xad}},
	"leftthreetimes":                  {Name: "leftthreetimes", CodePoints: []int{8907}, Characters: []byte{0xe2, 0x8b, 0x8b}},
	"leg":                             {Name: "leg", CodePoints: []int{8922}, Characters: []byte{0xe2, 0x8b, 0x9a}},
	"leq":                             {Name: "leq", CodePoints: []int{8804}, Characters: []byte{0xe2, 0x89, 0xa4}},
	"leqq":                            {Name: "leqq", CodePoints: []int{8806}, Characters: []byte{0xe2, 0x89, 0xa6}},
	"leqslant":                        {Name: "leqslant", CodePoints: []int{10877}, Characters: []byte{0xe2, 0xa9, 0xbd}},
	"les":                             {Name: "les", CodePoints: []int{10877}, Characters: []byte{0xe2, 0xa9, 0xbd}},
	"lescc":                           {Name: "lescc", CodePoints: []int{10920}, Characters: []byte{0xe2, 0xaa, 0xa8}},
	"lesdot":                          {Name: "lesdot", CodePoints: []int{10879}, Characters: []byte{0xe2, 0xa9, 0xbf}},
	"lesdoto":                         {Name: "lesdoto", CodePoints: []int{10881}, Characters: []byte{0xe2, 0xaa, 0x81}},
	"lesdotor":                        {Name: "lesdotor", CodePoints: []int{10883}, Characters: []byte{0xe2, 0xaa, 0x83}},
	"lesg":                            {Name: "lesg", CodePoints: []int{8922, 65024}, Characters: []byte{0xe2, 0x8b, 0x9a, 0xef, 0xb8, 0x80}},
	"lesges":                          {Name: "lesges", CodePoints: []int{10899}, Characters: []byte{0xe2, 0xaa, 0x93}},
	"lessapprox":                      {Name: "lessapprox", CodePoints: []int{10885}, Characters: []byte{0xe2, 0xaa, 0x85}},
	"lessdot":                         {Name: "lessdot", CodePoints: []int{8918}, Characters: []byte{0xe2, 0x8b, 0x96}},
	"lesseqgtr":                       {Name: "lesseqgtr", CodePoints: []int{8922}, Characters: []byte{0xe2, 0x8b, 0x9a}},
	"lesseqqgtr":                      {Name: "lesseqqgtr", CodePoints: []int{10891}, Characters: []byte{0xe2, 0xaa, 0x8b}},
	"lessgtr":                         {Name: "lessgtr", CodePoints: []int{8822}, Characters: []byte{0xe2, 0x89, 0xb6}},
	"lesssim":                         {Name: "lesssim", CodePoints: []int{8818}, Characters: []byte{0xe2, 0x89, 0xb2}},
	"lfisht":                          {Name: "lfisht", CodePoints: []int{10620}, Characters: []byte{0xe2, 0xa5, 0xbc}},
	"lfloor":                          {Name: "lfloor", CodePoints: []int{8970}, Characters: []byte{0xe2, 0x8c, 0x8a}},
	"lfr":                             {Name: "lfr", CodePoints: []int{120105}, Characters: []byte{0xf0, 0x9d, 0x94, 0xa9}},
	"lg":                              {Name: "lg", CodePoints: []int{8822}, Characters: []byte{0xe2, 0x89, 0xb6}},
	"lgE":                             {Name: "lgE", CodePoints: []int{10897}, Characters: []byte{0xe2, 0xaa, 0x91}},
	"lhard":                           {Name: "lhard", CodePoints: []int{8637}, Characters: []byte{0xe2, 0x86, 0xbd}},
	"lharu":                           {Name: "lharu", CodePoints: []int{8636}, Characters: []byte{0xe2, 0x86, 0xbc}},
	"lharul":                          {Name: "lharul", CodePoints: []int{10602}, Characters: []byte{0xe2, 0xa5, 0xaa}},
	"lhblk":                           {Name: "lhblk", CodePoints: []int{9604}, Characters: []byte{0xe2, 0x96, 0x84}},
	"ljcy":                            {Name: "ljcy", CodePoints: []int{1113}, Characters: []byte{0xd1, 0x99}},
	"ll":                              {Name: "ll", CodePoints: []int{8810}, Characters: []byte{0xe2, 0x89, 0xaa}},
	"llarr":                           {Name: "llarr", CodePoints: []int{8647}, Characters: []byte{0xe2, 0x87, 0x87}},
	"llcorner":                        {Name: "llcorner", CodePoints: []int{8990}, Characters: []byte{0xe2, 0x8c, 0x9e}},
	"llhard":                          {Name: "llhard", CodePoints: []int{10603}, Characters: []byte{0xe2, 0xa5, 0xab}},
	"lltri":                           {Name: "lltri", CodePoints: []int{9722}, Characters: []byte{0xe2, 0x97, 0xba}},
	"lmidot":                          {Name: "lmidot", CodePoints: []int{320}, Characters: []byte{0xc5, 0x80}},
	"lmoust":                          {Name: "lmoust", CodePoints: []int{9136}, Characters: []byte{0xe2, 0x8e, 0xb0}},
	"lmoustache":                      {Name: "lmoustache", CodePoints: []int{9136}, Characters: []byte{0xe2, 0x8e, 0xb0}},
	"lnE":                             {Name: "lnE", CodePoints: []int{8808}, Characters: []byte{0xe2, 0x89, 0xa8}},
	"lnap":                            {Name: "lnap", CodePoints: []int{10889}, Characters: []byte{0xe2, 0xaa, 0x89}},
	"lnapprox":                        {Name: "lnapprox", CodePoints: []int{10889}, Characters: []byte{0xe2, 0xaa, 0x89}},
	"lne":                             {Name: "lne", CodePoints: []int{10887}, Characters: []byte{0xe2, 0xaa, 0x87}},
	"lneq":                            {Name: "lneq", CodePoints: []int{10887}, Characters: []byte{0xe2, 0xaa, 0x87}},
	"lneqq":                           {Name: "lneqq", CodePoints: []int{8808}, Characters: []byte{0xe2, 0x89, 0xa8}},
	"lnsim":                           {Name: "lnsim", CodePoints: []int{8934}, Characters: []byte{0xe2, 0x8b, 0xa6}},
	"loang":                           {Name: "loang", CodePoints: []int{10220}, Characters: []byte{0xe2, 0x9f, 0xac}},
	"loarr":                           {Name: "loarr", CodePoints: []int{8701}, Characters: []byte{0xe2, 0x87, 0xbd}},
	"lobrk":                           {Name: "lobrk", CodePoints: []int{10214}, Characters: []byte{0xe2, 0x9f, 0xa6}},
	"longleftarrow":                   {Name: "longleftarrow", CodePoints: []int{10229}, Characters: []byte{0xe2, 0x9f, 0xb5}},
	"longleftrightarrow":              {Name: "longleftrightarrow", CodePoints: []int{10231}, Characters: []byte{0xe2, 0x9f, 0xb7}},
	"longmapsto":                      {Name: "longmapsto", CodePoints: []int{10236}, Characters: []byte{0xe2, 0x9f, 0xbc}},
	"longrightarrow":                  {Name: "longrightarrow", CodePoints: []int{10230}, Characters: []byte{0xe2, 0x9f, 0xb6}},
	"looparrowleft":                   {Name: "looparrowleft", CodePoints: []int{8619}, Characters: []byte{0xe2, 0x86, 0xab}},
	"looparrowright":                  {Name: "looparrowright", CodePoints: []int{8620}, Characters: []byte{0xe2, 0x86, 0xac}},
	"lopar":                           {Name: "lopar", CodePoints: []int{10629}, Characters: []byte{0xe2, 0xa6, 0x85}},
	"lopf":                            {Name: "lopf", CodePoints: []int{120157}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9d}},
	"loplus":                          {Name: "loplus", CodePoints: []int{10797}, Characters: []byte{0xe2, 0xa8, 0xad}},
	"lotimes":                         {Name: "lotimes", CodePoints: []int{10804}, Characters: []byte{0xe2, 0xa8, 0xb4}},
	"lowast":                          {Name: "lowast", CodePoints: []int{8727}, Characters: []byte{0xe2, 0x88, 0x97}},
	"lowbar":                          {Name: "lowbar", CodePoints: []int{95}, Characters: []byte{0x5f}},
	"loz":                             {Name: "loz", CodePoints: []int{9674}, Characters: []byte{0xe2, 0x97, 0x8a}},
	"lozenge":                         {Name: "lozenge", CodePoints: []int{9674}, Characters: []byte{0xe2, 0x97, 0x8a}},
	"lozf":                            {Name: "lozf", CodePoints: []int{10731}, Characters: []byte{0xe2, 0xa7, 0xab}},
	"lpar":                            {Name: "lpar", CodePoints: []int{40}, Characters: []byte{0x28}},
	"lparlt":                          {Name: "lparlt", CodePoints: []int{10643}, Characters: []byte{0xe2, 0xa6, 0x93}},
	"lrarr":                           {Name: "lrarr", CodePoints: []int{8646}, Characters: []byte{0xe2, 0x87, 0x86}},
	"lrcorner":                        {Name: "lrcorner", CodePoints: []int{8991}, Characters: []byte{0xe2, 0x8c, 0x9f}},
	"lrhar":                           {Name: "lrhar", CodePoints: []int{8651}, Characters: []byte{0xe2, 0x87, 0x8b}},
	"lrhard":                          {Name: "lrhard", CodePoints: []int{10605}, Characters: []byte{0xe2, 0xa5, 0xad}},
	"lrm":                             {Name: "lrm", CodePoints: []int{8206}, Characters: []byte{0xe2, 0x80, 0x8e}},
	"lrtri":                           {Name: "lrtri", CodePoints: []int{8895}, Characters: []byte{0xe2, 0x8a, 0xbf}},
	"lsaquo":                          {Name: "lsaquo", CodePoints: []int{8249}, Characters: []byte{0xe2, 0x80, 0xb9}},
	"lscr":                            {Name: "lscr", CodePoints: []int{120001}, Characters: []byte{0xf0, 0x9d, 0x93, 0x81}},
	"lsh":                             {Name: "lsh", CodePoints: []int{8624}, Characters: []byte{0xe2, 0x86, 0xb0}},
	"lsim":                            {Name: "lsim", CodePoints: []int{8818}, Characters: []byte{0xe2, 0x89, 0xb2}},
	"lsime":                           {Name: "lsime", CodePoints: []int{10893}, Characters: []byte{0xe2, 0xaa, 0x8d}},
	"lsimg":                           {Name: "lsimg", CodePoints: []int{10895}, Characters: []byte{0xe2, 0xaa, 0x8f}},
	"lsqb":                            {Name: "lsqb", CodePoints: []int{91}, Characters: []byte{0x5b}},
	"lsquo":                           {Name: "lsquo", CodePoints: []int{8216}, Characters: []byte{0xe2, 0x80, 0x98}},
	"lsquor":                          {Name: "lsquor", CodePoints: []int{8218}, Characters: []byte{0xe2, 0x80, 0x9a}},
	"lstrok":                          {Name: "lstrok", CodePoints: []int{322}, Characters: []byte{0xc5, 0x82}},
	"lt":                              {Name: "lt", CodePoints: []int{60}, Characters: []byte{0x3c}},
	"ltcc":                            {Name: "ltcc", CodePoints: []int{10918}, Characters: []byte{0xe2, 0xaa, 0xa6}},
	"ltcir":                           {Name: "ltcir", CodePoints: []int{10873}, Characters: []byte{0xe2, 0xa9, 0xb9}},
	"ltdot":                           {Name: "ltdot", CodePoints: []int{8918}, Characters: []byte{0xe2, 0x8b, 0x96}},
	"lthree":                          {Name: "lthree", CodePoints: []int{8907}, Characters: []byte{0xe2, 0x8b, 0x8b}},
	"ltimes":                          {Name: "ltimes", CodePoints: []int{8905}, Characters: []byte{0xe2, 0x8b, 0x89}},
	"ltlarr":                          {Name: "ltlarr", CodePoints: []int{10614}, Characters: []byte{0xe2, 0xa5, 0xb6}},
	"ltquest":                         {Name: "ltquest", CodePoints: []int{10875}, Characters: []byte{0xe2, 0xa9, 0xbb}},
	"ltrPar":                          {Name: "ltrPar", CodePoints: []int{10646}, Characters: []byte{0xe2, 0xa6, 0x96}},
	"ltri":                            {Name: "ltri", CodePoints: []int{9667}, Characters: []byte{0xe2, 0x97, 0x83}},
	"ltrie":                           {Name: "ltrie", CodePoints: []int{8884}, Characters: []byte{0xe2, 0x8a, 0xb4}},
	"ltrif":                           {Name: "ltrif", CodePoints: []int{9666}, Characters: []byte{0xe2, 0x97, 0x82}},
	"lurdshar":                        {Name: "lurdshar", CodePoints: []int{10570}, Characters: []byte{0xe2, 0xa5, 0x8a}},
	"luruhar":                         {Name: "luruhar", CodePoints: []int{10598}, Characters: []byte{0xe2, 0xa5, 0xa6}},
	"lvertneqq":                       {Name: "lvertneqq", CodePoints: []int{8808, 65024}, Characters: []byte{0xe2, 0x89, 0xa8, 0xef, 0xb8, 0x80}},
	"lvnE":                            {Name: "lvnE", CodePoints: []int{8808, 65024}, Characters: []byte{0xe2, 0x89, 0xa8, 0xef, 0xb8, 0x80}},
	"mDDot":                           {Name: "mDDot", CodePoints: []int{8762}, Characters: []byte{0xe2, 0x88, 0xba}},
	"macr":                            {Name: "macr", CodePoints: []int{175}, Characters: []byte{0xc2, 0xaf}},
	"male":                            {Name: "male", CodePoints: []int{9794}, Characters: []byte{0xe2, 0x99, 0x82}},
	"malt":                            {Name: "malt", CodePoints: []int{10016}, Characters: []byte{0xe2, 0x9c, 0xa0}},
	"maltese":                         {Name: "maltese", CodePoints: []int{10016}, Characters: []byte{0xe2, 0x9c, 0xa0}},
	"map":                             {Name: "map", CodePoints: []int{8614}, Characters: []byte{0xe2, 0x86, 0xa6}},
	"mapsto":                          {Name: "mapsto", CodePoints: []int{8614}, Characters: []byte{0xe2, 0x86, 0xa6}},
	"mapstodown":                      {Name: "mapstodown", CodePoints: []int{8615}, Characters: []byte{0xe2, 0x86, 0xa7}},
	"mapstoleft":                      {Name: "mapstoleft", CodePoints: []int{8612}, Characters: []byte{0xe2, 0x86, 0xa4}},
	"mapstoup":                        {Name: "mapstoup", CodePoints: []int{8613}, Characters: []byte{0xe2, 0x86, 0xa5}},
	"marker":                          {Name: "marker", CodePoints: []int{9646}, Characters: []byte{0xe2, 0x96, 0xae}},
	"mcomma":                          {Name: "mcomma", CodePoints: []int{10793}, Characters: []byte{0xe2, 0xa8, 0xa9}},
	"mcy":                             {Name: "mcy", CodePoints: []int{1084}, Characters: []byte{0xd0, 0xbc}},
	"mdash":                           {Name: "mdash", CodePoints: []int{8212}, Characters: []byte{0xe2, 0x80, 0x94}},
	"measuredangle":                   {Name: "measuredangle", CodePoints: []int{8737}, Characters: []byte{0xe2, 0x88, 0xa1}},
	"mfr":                             {Name: "mfr", CodePoints: []int{120106}, Characters: []byte{0xf0, 0x9d, 0x94, 0xaa}},
	"mho":                             {Name: "mho", CodePoints: []int{8487}, Characters: []byte{0xe2, 0x84, 0xa7}},
	"micro":                           {Name: "micro", CodePoints: []int{181}, Characters: []byte{0xc2, 0xb5}},
	"mid":                             {Name: "mid", CodePoints: []int{8739}, Characters: []byte{0xe2, 0x88, 0xa3}},
	"midast":                          {Name: "midast", CodePoints: []int{42}, Characters: []byte{0x2a}},
	"midcir":                          {Name: "midcir", CodePoints: []int{10992}, Characters: []byte{0xe2, 0xab, 0xb0}},
	"middot":                          {Name: "middot", CodePoints: []int{183}, Characters: []byte{0xc2, 0xb7}},
	"minus":                           {Name: "minus", CodePoints: []int{8722}, Characters: []byte{0xe2, 0x88, 0x92}},
	"minusb":                          {Name: "minusb", CodePoints: []int{8863}, Characters: []byte{0xe2, 0x8a, 0x9f}},
	"minusd":                          {Name: "minusd", CodePoints: []int{8760}, Characters: []byte{0xe2, 0x88, 0xb8}},
	"minusdu":                         {Name: "minusdu", CodePoints: []int{10794}, Characters: []byte{0xe2, 0xa8, 0xaa}},
	"mlcp":                            {Name: "mlcp", CodePoints: []int{10971}, Characters: []byte{0xe2, 0xab, 0x9b}},
	"mldr":                            {Name: "mldr", CodePoints: []int{8230}, Characters: []byte{0xe2, 0x80, 0xa6}},
	"mnplus":                          {Name: "mnplus", CodePoints: []int{8723}, Characters: []byte{0xe2, 0x88, 0x93}},
	"models":                          {Name: "models", CodePoints: []int{8871}, Characters: []byte{0xe2, 0x8a, 0xa7}},
	"mopf":                            {Name: "mopf", CodePoints: []int{120158}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9e}},
	"mp":                              {Name: "mp", CodePoints: []int{8723}, Characters: []byte{0xe2, 0x88, 0x93}},
	"mscr":                            {Name: "mscr", CodePoints: []int{120002}, Characters: []byte{0xf0, 0x9d, 0x93, 0x82}},
	"mstpos":                          {Name: "mstpos", CodePoints: []int{8766}, Characters: []byte{0xe2, 0x88, 0xbe}},
	"mu":                              {Name: "mu", CodePoints: []int{956}, Characters: []byte{0xce, 0xbc}},
	"multimap":                        {Name: "multimap", CodePoints: []int{8888}, Characters: []byte{0xe2, 0x8a, 0xb8}},
	"mumap":                           {Name: "mumap", CodePoints: []int{8888}, Characters: []byte{0xe2, 0x8a, 0xb8}},
	"nGg":                             {Name: "nGg", CodePoints: []int{8921, 824}, Characters: []byte{0xe2, 0x8b, 0x99, 0xcc, 0xb8}},
	"nGt":                             {Name: "nGt", CodePoints: []int{8811, 8402}, Characters: []byte{0xe2, 0x89, 0xab, 0xe2, 0x83, 0x92}},
	"nGtv":                            {Name: "nGtv", CodePoints: []int{8811, 824}, Characters: []byte{0xe2, 0x89, 0xab, 0xcc, 0xb8}},
	"nLeftarrow":                      {Name: "nLeftarrow", CodePoints: []int{8653}, Characters: []byte{0xe2, 0x87, 0x8d}},
	"nLeftrightarrow":                 {Name: "nLeftrightarrow", CodePoints: []int{8654}, Characters: []byte{0xe2, 0x87, 0x8e}},
	"nLl":                             {Name: "nLl", CodePoints: []int{8920, 824}, Characters: []byte{0xe2, 0x8b, 0x98, 0xcc, 0xb8}},
	"nLt":                             {Name: "nLt", CodePoints: []int{8810, 8402}, Characters: []byte{0xe2, 0x89, 0xaa, 0xe2, 0x83, 0x92}},
	"nLtv":                            {Name: "nLtv", CodePoints: []int{8810, 824}, Characters: []byte{0xe2, 0x89, 0xaa, 0xcc, 0xb8}},
	"nRightarrow":                     {Name: "nRightarrow", CodePoints: []int{8655}, Characters: []byte{0xe2, 0x87, 0x8f}},
	"nVDash":                          {Name: "nVDash", CodePoints: []int{8879}, Characters: []byte{0xe2, 0x8a, 0xaf}},
	"nVdash":                          {Name: "nVdash", CodePoints: []int{8878}, Characters: []byte{0xe2, 0x8a, 0xae}},
	"nabla":                           {Name: "nabla", CodePoints: []int{8711}, Characters: []byte{0xe2, 0x88, 0x87}},
	"nacute":                          {Name: "nacute", CodePoints: []int{324}, Characters: []byte{0xc5, 0x84}},
	"nang":                            {Name: "nang", CodePoints: []int{8736, 8402}, Characters: []byte{0xe2, 0x88, 0xa0, 0xe2, 0x83, 0x92}},
	"nap":                             {Name: "nap", CodePoints: []int{8777}, Characters: []byte{0xe2, 0x89, 0x89}},
	"napE":                            {Name: "napE", CodePoints: []int{10864, 824}, Characters: []byte{0xe2, 0xa9, 0xb0, 0xcc, 0xb8}},
	"napid":                           {Name: "napid", CodePoints: []int{8779, 824}, Characters: []byte{0xe2, 0x89, 0x8b, 0xcc, 0xb8}},
	"napos":                           {Name: "napos", CodePoints: []int{329}, Characters: []byte{0xc5, 0x89}},
	"napprox":                         {Name: "napprox", CodePoints: []int{8777}, Characters: []byte{0xe2, 0x89, 0x89}},
	"natur":                           {Name: "natur", CodePoints: []int{9838}, Characters: []byte{0xe2, 0x99, 0xae}},
	"natural":                         {Name: "natural", CodePoints: []int{9838}, Characters: []byte{0xe2, 0x99, 0xae}},
	"naturals":                        {Name: "naturals", CodePoints: []int{8469}, Characters: []byte{0xe2, 0x84, 0x95}},
	"nbsp":                            {Name: "nbsp", CodePoints: []int{160}, Characters: []byte{0xc2, 0xa0}},
	"nbump":                           {Name: "nbump", CodePoints: []int{8782, 824}, Characters: []byte{0xe2, 0x89, 0x8e, 0xcc, 0xb8}},
	"nbumpe":                          {Name: "nbumpe", CodePoints: []int{8783, 824}, Characters: []byte{0xe2, 0x89, 0x8f, 0xcc, 0xb8}},
	"ncap":                            {Name: "ncap", CodePoints: []int{10819}, Characters: []byte{0xe2, 0xa9, 0x83}},
	"ncaron":                          {Name: "ncaron", CodePoints: []int{328}, Characters: []byte{0xc5, 0x88}},
	"ncedil":                          {Name: "ncedil", CodePoints: []int{326}, Characters: []byte{0xc5, 0x86}},
	"ncong":                           {Name: "ncong", CodePoints: []int{8775}, Characters: []byte{0xe2, 0x89, 0x87}},
	"ncongdot":                        {Name: "ncongdot", CodePoints: []int{10861, 824}, Characters: []byte{0xe2, 0xa9, 0xad, 0xcc, 0xb8}},
	"ncup":                            {Name: "ncup", CodePoints: []int{10818}, Characters: []byte{0xe2, 0xa9, 0x82}},
	"ncy":                             {Name: "ncy", CodePoints: []int{1085}, Characters: []byte{0xd0, 0xbd}},
	"ndash":                           {Name: "ndash", CodePoints: []int{8211}, Characters: []byte{0xe2, 0x80, 0x93}},
	"ne":                              {Name: "ne", CodePoints: []int{8800}, Characters: []byte{0xe2, 0x89, 0xa0}},
	"neArr":                           {Name: "neArr", CodePoints: []int{8663}, Characters: []byte{0xe2, 0x87, 0x97}},
	"nearhk":                          {Name: "nearhk", CodePoints: []int{10532}, Characters: []byte{0xe2, 0xa4, 0xa4}},
	"nearr":                           {Name: "nearr", CodePoints: []int{8599}, Characters: []byte{0xe2, 0x86, 0x97}},
	"nearrow":                         {Name: "nearrow", CodePoints: []int{8599}, Characters: []byte{0xe2, 0x86, 0x97}},
	"nedot":                           {Name: "nedot", CodePoints: []int{8784, 824}, Characters: []byte{0xe2, 0x89, 0x90, 0xcc, 0xb8}},
	"nequiv":                          {Name: "nequiv", CodePoints: []int{8802}, Characters: []byte{0xe2, 0x89, 0xa2}},
	"nesear":                          {Name: "nesear", CodePoints: []int{10536}, Characters: []byte{0xe2, 0xa4, 0xa8}},
	"nesim":                           {Name: "nesim", CodePoints: []int{8770, 824}, Characters: []byte{0xe2, 0x89, 0x82, 0xcc, 0xb8}},
	"nexist":                          {Name: "nexist", CodePoints: []int{8708}, Characters: []byte{0xe2, 0x88, 0x84}},
	"nexists":                         {Name: "nexists", CodePoints: []int{8708}, Characters: []byte{0xe2, 0x88, 0x84}},
	"nfr":                             {Name: "nfr", CodePoints: []int{120107}, Characters: []byte{0xf0, 0x9d, 0x94, 0xab}},
	"ngE":                             {Name: "ngE", CodePoints: []int{8807, 824}, Characters: []byte{0xe2, 0x89, 0xa7, 0xcc, 0xb8}},
	"nge":                             {Name: "nge", CodePoints: []int{8817}, Characters: []byte{0xe2, 0x89, 0xb1}},
	"ngeq":                            {Name: "ngeq", CodePoints: []int{8817}, Characters: []byte{0xe2, 0x89, 0xb1}},
	"ngeqq":                           {Name: "ngeqq", CodePoints: []int{8807, 824}, Characters: []byte{0xe2, 0x89, 0xa7, 0xcc, 0xb8}},
	"ngeqslant":                       {Name: "ngeqslant", CodePoints: []int{10878, 824}, Characters: []byte{0xe2, 0xa9, 0xbe, 0xcc, 0xb8}},
	"nges":                            {Name: "nges", CodePoints: []int{10878, 824}, Characters: []byte{0xe2, 0xa9, 0xbe, 0xcc, 0xb8}},
	"ngsim":                           {Name: "ngsim", CodePoints: []int{8821}, Characters: []byte{0xe2, 0x89, 0xb5}},
	"ngt":                             {Name: "ngt", CodePoints: []int{8815}, Characters: []byte{0xe2, 0x89, 0xaf}},
	"ngtr":                            {Name: "ngtr", CodePoints: []int{8815}, Characters: []byte{0xe2, 0x89, 0xaf}},
	"nhArr":                           {Name: "nhArr", CodePoints: []int{8654}, Characters: []byte{0xe2, 0x87, 0x8e}},
	"nharr":                           {Name: "nharr", CodePoints: []int{8622}, Characters: []byte{0xe2, 0x86, 0xae}},
	"nhpar":                           {Name: "nhpar", CodePoints: []int{10994}, Characters: []byte{0xe2, 0xab, 0xb2}},
	"ni":                              {Name: "ni", CodePoints: []int{8715}, Characters: []byte{0xe2, 0x88, 0x8b}},
	"nis":                             {Name: "nis", CodePoints: []int{8956}, Characters: []byte{0xe2, 0x8b, 0xbc}},
	"nisd":                            {Name: "nisd", CodePoints: []int{8954}, Characters: []byte{0xe2, 0x8b, 0xba}},
	"niv":                             {Name: "niv", CodePoints: []int{8715}, Characters: []byte{0xe2, 0x88, 0x8b}},
	"njcy":                            {Name: "njcy", CodePoints: []int{1114}, Characters: []byte{0xd1, 0x9a}},
	"nlArr":                           {Name: "nlArr", CodePoints: []int{8653}, Characters: []byte{0xe2, 0x87, 0x8d}},
	"nlE":                             {Name: "nlE", CodePoints: []int{8806, 824}, Characters: []byte{0xe2, 0x89, 0xa6, 0xcc, 0xb8}},
	"nlarr":                           {Name: "nlarr", CodePoints: []int{8602}, Characters: []byte{0xe2, 0x86, 0x9a}},
	"nldr":                            {Name: "nldr", CodePoints: []int{8229}, Characters: []byte{0xe2, 0x80, 0xa5}},
	"nle":                             {Name: "nle", CodePoints: []int{8816}, Characters: []byte{0xe2, 0x89, 0xb0}},
	"nleftarrow":                      {Name: "nleftarrow", CodePoints: []int{8602}, Characters: []byte{0xe2, 0x86, 0x9a}},
	"nleftrightarrow":                 {Name: "nleftrightarrow", CodePoints: []int{8622}, Characters: []byte{0xe2, 0x86, 0xae}},
	"nleq":                            {Name: "nleq", CodePoints: []int{8816}, Characters: []byte{0xe2, 0x89, 0xb0}},
	"nleqq":                           {Name: "nleqq", CodePoints: []int{8806, 824}, Characters: []byte{0xe2, 0x89, 0xa6, 0xcc, 0xb8}},
	"nleqslant":                       {Name: "nleqslant", CodePoints: []int{10877, 824}, Characters: []byte{0xe2, 0xa9, 0xbd, 0xcc, 0xb8}},
	"nles":                            {Name: "nles", CodePoints: []int{10877, 824}, Characters: []byte{0xe2, 0xa9, 0xbd, 0xcc, 0xb8}},
	"nless":                           {Name: "nless", CodePoints: []int{8814}, Characters: []byte{0xe2, 0x89, 0xae}},
	"nlsim":                           {Name: "nlsim", CodePoints: []int{8820}, Characters: []byte{0xe2, 0x89, 0xb4}},
	"nlt":                             {Name: "nlt", CodePoints: []int{8814}, Characters: []byte{0xe2, 0x89, 0xae}},
	"nltri":                           {Name: "nltri", CodePoints: []int{8938}, Characters: []byte{0xe2, 0x8b, 0xaa}},
	"nltrie":                          {Name: "nltrie", CodePoints: []int{8940}, Characters: []byte{0xe2, 0x8b, 0xac}},
	"nmid":                            {Name: "nmid", CodePoints: []int{8740}, Characters: []byte{0xe2, 0x88, 0xa4}},
	"nopf":                            {Name: "nopf", CodePoints: []int{120159}, Characters: []byte{0xf0, 0x9d, 0x95, 0x9f}},
	"not":                             {Name: "not", CodePoints: []int{172}, Characters: []byte{0xc2, 0xac}},
	"notin":                           {Name: "notin", CodePoints: []int{8713}, Characters: []byte{0xe2, 0x88, 0x89}},
	"notinE":                          {Name: "notinE", CodePoints: []int{8953, 824}, Characters: []byte{0xe2, 0x8b, 0xb9, 0xcc, 0xb8}},
	"notindot":                        {Name: "notindot", CodePoints: []int{8949, 824}, Characters: []byte{0xe2, 0x8b, 0xb5, 0xcc, 0xb8}},
	"notinva":                         {Name: "notinva", CodePoints: []int{8713}, Characters: []byte{0xe2, 0x88, 0x89}},
	"notinvb":                         {Name: "notinvb", CodePoints: []int{8951}, Characters: []byte{0xe2, 0x8b, 0xb7}},
	"notinvc":                         {Name: "notinvc", CodePoints: []int{8950}, Characters: []byte{0xe2, 0x8b, 0xb6}},
	"notni":                           {Name: "notni", CodePoints: []int{8716}, Characters: []byte{0xe2, 0x88, 0x8c}},
	"notniva":                         {Name: "notniva", CodePoints: []int{8716}, Characters: []byte{0xe2, 0x88, 0x8c}},
	"notnivb":                         {Name: "notnivb", CodePoints: []int{8958}, Characters: []byte{0xe2, 0x8b, 0xbe}},
	"notnivc":                         {Name: "notnivc", CodePoints: []int{8957}, Characters: []byte{0xe2, 0x8b, 0xbd}},
	"npar":                            {Name: "npar", CodePoints: []int{8742}, Characters: []byte{0xe2, 0x88, 0xa6}},
	"nparallel":                       {Name: "nparallel", CodePoints: []int{8742}, Characters: []byte{0xe2, 0x88, 0xa6}},
	"nparsl":                          {Name: "nparsl", CodePoints: []int{11005, 8421}, Characters: []byte{0xe2, 0xab, 0xbd, 0xe2, 0x83, 0xa5}},
	"npart":                           {Name: "npart", CodePoints: []int{8706, 824}, Characters: []byte{0xe2, 0x88, 0x82, 0xcc, 0xb8}},
	"npolint":                         {Name: "npolint", CodePoints: []int{10772}, Characters: []byte{0xe2, 0xa8, 0x94}},
	"npr":                             {Name: "npr", CodePoints: []int{8832}, Characters: []byte{0xe2, 0x8a, 0x80}},
	"nprcue":                          {Name: "nprcue", CodePoints: []int{8928}, Characters: []byte{0xe2, 0x8b, 0xa0}},
	"npre":                            {Name: "npre", CodePoints: []int{10927, 824}, Characters: []byte{0xe2, 0xaa, 0xaf, 0xcc, 0xb8}},
	"nprec":                           {Name: "nprec", CodePoints: []int{8832}, Characters: []byte{0xe2, 0x8a, 0x80}},
	"npreceq":                         {Name: "npreceq", CodePoints: []int{10927, 824}, Characters: []byte{0xe2, 0xaa, 0xaf, 0xcc, 0xb8}},
	"nrArr":                           {Name: "nrArr", CodePoints: []int{8655}, Characters: []byte{0xe2, 0x87, 0x8f}},
	"nrarr":                           {Name: "nrarr", CodePoints: []int{8603}, Characters: []byte{0xe2, 0x86, 0x9b}},
	"nrarrc":                          {Name: "nrarrc", CodePoints: []int{10547, 824}, Characters: []byte{0xe2, 0xa4, 0xb3, 0xcc, 0xb8}},
	"nrarrw":                          {Name: "nrarrw", CodePoints: []int{8605, 824}, Characters: []byte{0xe2, 0x86, 0x9d, 0xcc, 0xb8}},
	"nrightarrow":                     {Name: "nrightarrow", CodePoints: []int{8603}, Characters: []byte{0xe2, 0x86, 0x9b}},
	"nrtri":                           {Name: "nrtri", CodePoints: []int{8939}, Characters: []byte{0xe2, 0x8b, 0xab}},
	"nrtrie":                          {Name: "nrtrie", CodePoints: []int{8941}, Characters: []byte{0xe2, 0x8b, 0xad}},
	"nsc":                             {Name: "nsc", CodePoints: []int{8833}, Characters: []byte{0xe2, 0x8a, 0x81}},
	"nsccue":                          {Name: "nsccue", CodePoints: []int{8929}, Characters: []byte{0xe2, 0x8b, 0xa1}},
	"nsce":                            {Name: "nsce", CodePoints: []int{10928, 824}, Characters: []byte{0xe2, 0xaa, 0xb0, 0xcc, 0xb8}},
	"nscr":                            {Name: "nscr", CodePoints: []int{120003}, Characters: []byte{0xf0, 0x9d, 0x93, 0x83}},
	"nshortmid":                       {Name: "nshortmid", CodePoints: []int{8740}, Characters: []byte{0xe2, 0x88, 0xa4}},
	"nshortparallel":                  {Name: "nshortparallel", CodePoints: []int{8742}, Characters: []byte{0xe2, 0x88, 0xa6}},
	"nsim":                            {Name: "nsim", CodePoints: []int{8769}, Characters: []byte{0xe2, 0x89, 0x81}},
	"nsime":                           {Name: "nsime", CodePoints: []int{8772}, Characters: []byte{0xe2, 0x89, 0x84}},
	"nsimeq":                          {Name: "nsimeq", CodePoints: []int{8772}, Characters: []byte{0xe2, 0x89, 0x84}},
	"nsmid":                           {Name: "nsmid", CodePoints: []int{8740}, Characters: []byte{0xe2, 0x88, 0xa4}},
	"nspar":                           {Name: "nspar", CodePoints: []int{8742}, Characters: []byte{0xe2, 0x88, 0xa6}},
	"nsqsube":                         {Name: "nsqsube", CodePoints: []int{8930}, Characters: []byte{0xe2, 0x8b, 0xa2}},
	"nsqsupe":                         {Name: "nsqsupe", CodePoints: []int{8931}, Characters: []byte{0xe2, 0x8b, 0xa3}},
	"nsub":                            {Name: "nsub", CodePoints: []int{8836}, Characters: []byte{0xe2, 0x8a, 0x84}},
	"nsubE":                           {Name: "nsubE", CodePoints: []int{10949, 824}, Characters: []byte{0xe2, 0xab, 0x85, 0xcc, 0xb8}},
	"nsube":                           {Name: "nsube", CodePoints: []int{8840}, Characters: []byte{0xe2, 0x8a, 0x88}},
	"nsubset":                         {Name: "nsubset", CodePoints: []int{8834, 8402}, Characters: []byte{0xe2, 0x8a, 0x82, 0xe2, 0x83, 0x92}},
	"nsubseteq":                       {Name: "nsubseteq", CodePoints: []int{8840}, Characters: []byte{0xe2, 0x8a, 0x88}},
	"nsubseteqq":                      {Name: "nsubseteqq", CodePoints: []int{10949, 824}, Characters: []byte{0xe2, 0xab, 0x85, 0xcc, 0xb8}},
	"nsucc":                           {Name: "nsucc", CodePoints: []int{8833}, Characters: []byte{0xe2, 0x8a, 0x81}},
	"nsucceq":                         {Name: "nsucceq", CodePoints: []int{10928, 824}, Characters: []byte{0xe2, 0xaa, 0xb0, 0xcc, 0xb8}},
	"nsup":                            {Name: "nsup", CodePoints: []int{8837}, Characters: []byte{0xe2, 0x8a, 0x85}},
	"nsupE":                           {Name: "nsupE", CodePoints: []int{10950, 824}, Characters: []byte{0xe2, 0xab, 0x86, 0xcc, 0xb8}},
	"nsupe":                           {Name: "nsupe", CodePoints: []int{8841}, Characters: []byte{0xe2, 0x8a, 0x89}},
	"nsupset":                         {Name: "nsupset", CodePoints: []int{8835, 8402}, Characters: []byte{0xe2, 0x8a, 0x83, 0xe2, 0x83, 0x92}},
	"nsupseteq":                       {Name: "nsupseteq", CodePoints: []int{8841}, Characters: []byte{0xe2, 0x8a, 0x89}},
	"nsupseteqq":                      {Name: "nsupseteqq", CodePoints: []int{10950, 824}, Characters: []byte{0xe2, 0xab, 0x86, 0xcc, 0xb8}},
	"ntgl":                            {Name: "ntgl", CodePoints: []int{8825}, Characters: []byte{0xe2, 0x89, 0xb9}},
	"ntilde":                          {Name: "ntilde", CodePoints: []int{241}, Characters: []byte{0xc3, 0xb1}},
	"ntlg":                            {Name: "ntlg", CodePoints: []int{8824}, Characters: []byte{0xe2, 0x89, 0xb8}},
	"ntriangleleft":                   {Name: "ntriangleleft", CodePoints: []int{8938}, Characters: []byte{0xe2, 0x8b, 0xaa}},
	"ntrianglelefteq":                 {Name: "ntrianglelefteq", CodePoints: []int{8940}, Characters: []byte{0xe2, 0x8b, 0xac}},
	"ntriangleright":                  {Name: "ntriangleright", CodePoints: []int{8939}, Characters: []byte{0xe2, 0x8b, 0xab}},
	"ntrianglerighteq":                {Name: "ntrianglerighteq", CodePoints: []int{8941}, Characters: []byte{0xe2, 0x8b, 0xad}},
	"nu":                              {Name: "nu", CodePoints: []int{957}, Characters: []byte{0xce, 0xbd}},
	"num":                             {Name: "num", CodePoints: []int{35}, Characters: []byte{0x23}},
	"numero":                          {Name: "numero", CodePoints: []int{8470}, Characters: []byte{0xe2, 0x84, 0x96}},
	"numsp":                           {Name: "numsp", CodePoints: []int{8199}, Characters: []byte{0xe2, 0x80, 0x87}},
	"nvDash":                          {Name: "nvDash", CodePoints: []int{8877}, Characters: []byte{0xe2, 0x8a, 0xad}},
	"nvHarr":                          {Name: "nvHarr", CodePoints: []int{10500}, Characters: []byte{0xe2, 0xa4, 0x84}},
	"nvap":                            {Name: "nvap", CodePoints: []int{8781, 8402}, Characters: []byte{0xe2, 0x89, 0x8d, 0xe2, 0x83, 0x92}},
	"nvdash":                          {Name: "nvdash", CodePoints: []int{8876}, Characters: []byte{0xe2, 0x8a, 0xac}},
	"nvge":                            {Name: "nvge", CodePoints: []int{8805, 8402}, Characters: []byte{0xe2, 0x89, 0xa5, 0xe2, 0x83, 0x92}},
	"nvgt":                            {Name: "nvgt", CodePoints: []int{62, 8402}, Characters: []byte{0x3e, 0xe2, 0x83, 0x92}},
	"nvinfin":                         {Name: "nvinfin", CodePoints: []int{10718}, Characters: []byte{0xe2, 0xa7, 0x9e}},
	"nvlArr":                          {Name: "nvlArr", CodePoints: []int{10498}, Characters: []byte{0xe2, 0xa4, 0x82}},
	"nvle":                            {Name: "nvle", CodePoints: []int{8804, 8402}, Characters: []byte{0xe2, 0x89, 0xa4, 0xe2, 0x83, 0x92}},
	"nvlt":                            {Name: "nvlt", CodePoints: []int{60, 8402}, Characters: []byte{0x3c, 0xe2, 0x83, 0x92}},
	"nvltrie":                         {Name: "nvltrie", CodePoints: []int{8884, 8402}, Characters: []byte{0xe2, 0x8a, 0xb4, 0xe2, 0x83, 0x92}},
	"nvrArr":                          {Name: "nvrArr", CodePoints: []int{10499}, Characters: []byte{0xe2, 0xa4, 0x83}},
	"nvrtrie":                         {Name: "nvrtrie", CodePoints: []int{8885, 8402}, Characters: []byte{0xe2, 0x8a, 0xb5, 0xe2, 0x83, 0x92}},
	"nvsim":                           {Name: "nvsim", CodePoints: []int{8764, 8402}, Characters: []byte{0xe2, 0x88, 0xbc, 0xe2, 0x83, 0x92}},
	"nwArr":                           {Name: "nwArr", CodePoints: []int{8662}, Characters: []byte{0xe2, 0x87, 0x96}},
	"nwarhk":                          {Name: "nwarhk", CodePoints: []int{10531}, Characters: []byte{0xe2, 0xa4, 0xa3}},
	"nwarr":                           {Name: "nwarr", CodePoints: []int{8598}, Characters: []byte{0xe2, 0x86, 0x96}},
	"nwarrow":                         {Name: "nwarrow", CodePoints: []int{8598}, Characters: []byte{0xe2, 0x86, 0x96}},
	"nwnear":                          {Name: "nwnear", CodePoints: []int{10535}, Characters: []byte{0xe2, 0xa4, 0xa7}},
	"oS":                              {Name: "oS", CodePoints: []int{9416}, Characters: []byte{0xe2, 0x93, 0x88}},
	"oacute":                          {Name: "oacute", CodePoints: []int{243}, Characters: []byte{0xc3, 0xb3}},
	"oast":                            {Name: "oast", CodePoints: []int{8859}, Characters: []byte{0xe2, 0x8a, 0x9b}},
	"ocir":                            {Name: "ocir", CodePoints: []int{8858}, Characters: []byte{0xe2, 0x8a, 0x9a}},
	"ocirc":                           {Name: "ocirc", CodePoints: []int{244}, Characters: []byte{0xc3, 0xb4}},
	"ocy":                             {Name: "ocy", CodePoints: []int{1086}, Characters: []byte{0xd0, 0xbe}},
	"odash":                           {Name: "odash", CodePoints: []int{8861}, Characters: []byte{0xe2, 0x8a, 0x9d}},
	"odblac":                          {Name: "odblac", CodePoints: []int{337}, Characters: []byte{0xc5, 0x91}},
	"odiv":                            {Name: "odiv", CodePoints: []int{10808}, Characters: []byte{0xe2, 0xa8, 0xb8}},
	"odot":                            {Name: "odot", CodePoints: []int{8857}, Characters: []byte{0xe2, 0x8a, 0x99}},
	"odsold":                          {Name: "odsold", CodePoints: []int{10684}, Characters: []byte{0xe2, 0xa6, 0xbc}},
	"oelig":                           {Name: "oelig", CodePoints: []int{339}, Characters: []byte{0xc5, 0x93}},
	"ofcir":                           {Name: "ofcir", CodePoints: []int{10687}, Characters: []byte{0xe2, 0xa6, 0xbf}},
	"ofr":                             {Name: "ofr", CodePoints: []int{120108}, Characters: []byte{0xf0, 0x9d, 0x94, 0xac}},
	"ogon":                            {Name: "ogon", CodePoints: []int{731}, Characters: []byte{0xcb, 0x9b}},
	"ograve":                          {Name: "ograve", CodePoints: []int{242}, Characters: []byte{0xc3, 0xb2}},
	"ogt":                             {Name: "ogt", CodePoints: []int{10689}, Characters: []byte{0xe2, 0xa7, 0x81}},
	"ohbar":                           {Name: "ohbar", CodePoints: []int{10677}, Characters: []byte{0xe2, 0xa6, 0xb5}},
	"ohm":                             {Name: "ohm", CodePoints: []int{937}, Characters: []byte{0xce, 0xa9}},
	"oint":                            {Name: "oint", CodePoints: []int{8750}, Characters: []byte{0xe2, 0x88, 0xae}},
	"olarr":                           {Name: "olarr", CodePoints: []int{8634}, Characters: []byte{0xe2, 0x86, 0xba}},
	"olcir":                           {Name: "olcir", CodePoints: []int{10686}, Characters: []byte{0xe2, 0xa6, 0xbe}},
	"olcross":                         {Name: "olcross", CodePoints: []int{10683}, Characters: []byte{0xe2, 0xa6, 0xbb}},
	"oline":                           {Name: "oline", CodePoints: []int{8254}, Characters: []byte{0xe2, 0x80, 0xbe}},
	"olt":                             {Name: "olt", CodePoints: []int{10688}, Characters: []byte{0xe2, 0xa7, 0x80}},
	"omacr":                           {Name: "omacr", CodePoints: []int{333}, Characters: []byte{0xc5, 0x8d}},
	"omega":                           {Name: "omega", CodePoints: []int{969}, Characters: []byte{0xcf, 0x89}},
	"omicron":                         {Name: "omicron", CodePoints: []int{959}, Characters: []byte{0xce, 0xbf}},
	"omid":                            {Name: "omid", CodePoints: []int{10678}, Characters: []byte{0xe2, 0xa6, 0xb6}},
	"ominus":                          {Name: "ominus", CodePoints: []int{8854}, Characters: []byte{0xe2, 0x8a, 0x96}},
	"oopf":                            {Name: "oopf", CodePoints: []int{120160}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa0}},
	"opar":                            {Name: "opar", CodePoints: []int{10679}, Characters: []byte{0xe2, 0xa6, 0xb7}},
	"operp":                           {Name: "operp", CodePoints: []int{10681}, Characters: []byte{0xe2, 0xa6, 0xb9}},
	"oplus":                           {Name: "oplus", CodePoints: []int{8853}, Characters: []byte{0xe2, 0x8a, 0x95}},
	"or":                              {Name: "or", CodePoints: []int{8744}, Characters: []byte{0xe2, 0x88, 0xa8}},
	"orarr":                           {Name: "orarr", CodePoints: []int{8635}, Characters: []byte{0xe2, 0x86, 0xbb}},
	"ord":                             {Name: "ord", CodePoints: []int{10845}, Characters: []byte{0xe2, 0xa9, 0x9d}},
	"order":                           {Name: "order", CodePoints: []int{8500}, Characters: []byte{0xe2, 0x84, 0xb4}},
	"orderof":                         {Name: "orderof", CodePoints: []int{8500}, Characters: []byte{0xe2, 0x84, 0xb4}},
	"ordf":                            {Name: "ordf", CodePoints: []int{170}, Characters: []byte{0xc2, 0xaa}},
	"ordm":                            {Name: "ordm", CodePoints: []int{186}, Characters: []byte{0xc2, 0xba}},
	"origof":                          {Name: "origof", CodePoints: []int{8886}, Characters: []byte{0xe2, 0x8a, 0xb6}},
	"oror":                            {Name: "oror", CodePoints: []int{10838}, Characters: []byte{0xe2, 0xa9, 0x96}},
	"orslope":                         {Name: "orslope", CodePoints: []int{10839}, Characters: []byte{0xe2, 0xa9, 0x97}},
	"orv":                             {Name: "orv", CodePoints: []int{10843}, Characters: []byte{0xe2, 0xa9, 0x9b}},
	"oscr":                            {Name: "oscr", CodePoints: []int{8500}, Characters: []byte{0xe2, 0x84, 0xb4}},
	"oslash":                          {Name: "oslash", CodePoints: []int{248}, Characters: []byte{0xc3, 0xb8}},
	"osol":                            {Name: "osol", CodePoints: []int{8856}, Characters: []byte{0xe2, 0x8a, 0x98}},
	"otilde":                          {Name: "otilde", CodePoints: []int{245}, Characters: []byte{0xc3, 0xb5}},
	"otimes":                          {Name: "otimes", CodePoints: []int{8855}, Characters: []byte{0xe2, 0x8a, 0x97}},
	"otimesas":                        {Name: "otimesas", CodePoints: []int{10806}, Characters: []byte{0xe2, 0xa8, 0xb6}},
	"ouml":                            {Name: "ouml", CodePoints: []int{246}, Characters: []byte{0xc3, 0xb6}},
	"ovbar":                           {Name: "ovbar", CodePoints: []int{9021}, Characters: []byte{0xe2, 0x8c, 0xbd}},
	"par":                             {Name: "par", CodePoints: []int{8741}, Characters: []byte{0xe2, 0x88, 0xa5}},
	"para":                            {Name: "para", CodePoints: []int{182}, Characters: []byte{0xc2, 0xb6}},
	"parallel":                        {Name: "parallel", CodePoints: []int{8741}, Characters: []byte{0xe2, 0x88, 0xa5}},
	"parsim":                          {Name: "parsim", CodePoints: []int{10995}, Characters: []byte{0xe2, 0xab, 0xb3}},
	"parsl":                           {Name: "parsl", CodePoints: []int{11005}, Characters: []byte{0xe2, 0xab, 0xbd}},
	"part":                            {Name: "part", CodePoints: []int{8706}, Characters: []byte{0xe2, 0x88, 0x82}},
	"pcy":                             {Name: "pcy", CodePoints: []int{1087}, Characters: []byte{0xd0, 0xbf}},
	"percnt":                          {Name: "percnt", CodePoints: []int{37}, Characters: []byte{0x25}},
	"period":                          {Name: "period", CodePoints: []int{46}, Characters: []byte{0x2e}},
	"permil":                          {Name: "permil", CodePoints: []int{8240}, Characters: []byte{0xe2, 0x80, 0xb0}},
	"perp":                            {Name: "perp", CodePoints: []int{8869}, Characters: []byte{0xe2, 0x8a, 0xa5}},
	"pertenk":                         {Name: "pertenk", CodePoints: []int{8241}, Characters: []byte{0xe2, 0x80, 0xb1}},
	"pfr":                             {Name: "pfr", CodePoints: []int{120109}, Characters: []byte{0xf0, 0x9d, 0x94, 0xad}},
	"phi":                             {Name: "phi", CodePoints: []int{966}, Characters: []byte{0xcf, 0x86}},
	"phiv":                            {Name: "phiv", CodePoints: []int{981}, Characters: []byte{0xcf, 0x95}},
	"phmmat":                          {Name: "phmmat", CodePoints: []int{8499}, Characters: []byte{0xe2, 0x84, 0xb3}},
	"phone":                           {Name: "phone", CodePoints: []int{9742}, Characters: []byte{0xe2, 0x98, 0x8e}},
	"pi":                              {Name: "pi", CodePoints: []int{960}, Characters: []byte{0xcf, 0x80}},
	"pitchfork":                       {Name: "pitchfork", CodePoints: []int{8916}, Characters: []byte{0xe2, 0x8b, 0x94}},
	"piv":                             {Name: "piv", CodePoints: []int{982}, Characters: []byte{0xcf, 0x96}},
	"planck":                          {Name: "planck", CodePoints: []int{8463}, Characters: []byte{0xe2, 0x84, 0x8f}},
	"planckh":                         {Name: "planckh", CodePoints: []int{8462}, Characters: []byte{0xe2, 0x84, 0x8e}},
	"plankv":                          {Name: "plankv", CodePoints: []int{8463}, Characters: []byte{0xe2, 0x84, 0x8f}},
	"plus":                            {Name: "plus", CodePoints: []int{43}, Characters: []byte{0x2b}},
	"plusacir":                        {Name: "plusacir", CodePoints: []int{10787}, Characters: []byte{0xe2, 0xa8, 0xa3}},
	"plusb":                           {Name: "plusb", CodePoints: []int{8862}, Characters: []byte{0xe2, 0x8a, 0x9e}},
	"pluscir":                         {Name: "pluscir", CodePoints: []int{10786}, Characters: []byte{0xe2, 0xa8, 0xa2}},
	"plusdo":                          {Name: "plusdo", CodePoints: []int{8724}, Characters: []byte{0xe2, 0x88, 0x94}},
	"plusdu":                          {Name: "plusdu", CodePoints: []int{10789}, Characters: []byte{0xe2, 0xa8, 0xa5}},
	"pluse":                           {Name: "pluse", CodePoints: []int{10866}, Characters: []byte{0xe2, 0xa9, 0xb2}},
	"plusmn":                          {Name: "plusmn", CodePoints: []int{177}, Characters: []byte{0xc2, 0xb1}},
	"plussim":                         {Name: "plussim", CodePoints: []int{10790}, Characters: []byte{0xe2, 0xa8, 0xa6}},
	"plustwo":                         {Name: "plustwo", CodePoints: []int{10791}, Characters: []byte{0xe2, 0xa8, 0xa7}},
	"pm":                              {Name: "pm", CodePoints: []int{177}, Characters: []byte{0xc2, 0xb1}},
	"pointint":                        {Name: "pointint", CodePoints: []int{10773}, Characters: []byte{0xe2, 0xa8, 0x95}},
	"popf":                            {Name: "popf", CodePoints: []int{120161}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa1}},
	"pound":                           {Name: "pound", CodePoints: []int{163}, Characters: []byte{0xc2, 0xa3}},
	"pr":                              {Name: "pr", CodePoints: []int{8826}, Characters: []byte{0xe2, 0x89, 0xba}},
	"prE":                             {Name: "prE", CodePoints: []int{10931}, Characters: []byte{0xe2, 0xaa, 0xb3}},
	"prap":                            {Name: "prap", CodePoints: []int{10935}, Characters: []byte{0xe2, 0xaa, 0xb7}},
	"prcue":                           {Name: "prcue", CodePoints: []int{8828}, Characters: []byte{0xe2, 0x89, 0xbc}},
	"pre":                             {Name: "pre", CodePoints: []int{10927}, Characters: []byte{0xe2, 0xaa, 0xaf}},
	"prec":                            {Name: "prec", CodePoints: []int{8826}, Characters: []byte{0xe2, 0x89, 0xba}},
	"precapprox":                      {Name: "precapprox", CodePoints: []int{10935}, Characters: []byte{0xe2, 0xaa, 0xb7}},
	"preccurlyeq":                     {Name: "preccurlyeq", CodePoints: []int{8828}, Characters: []byte{0xe2, 0x89, 0xbc}},
	"preceq":                          {Name: "preceq", CodePoints: []int{10927}, Characters: []byte{0xe2, 0xaa, 0xaf}},
	"precnapprox":                     {Name: "precnapprox", CodePoints: []int{10937}, Characters: []byte{0xe2, 0xaa, 0xb9}},
	"precneqq":                        {Name: "precneqq", CodePoints: []int{10933}, Characters: []byte{0xe2, 0xaa, 0xb5}},
	"precnsim":                        {Name: "precnsim", CodePoints: []int{8936}, Characters: []byte{0xe2, 0x8b, 0xa8}},
	"precsim":                         {Name: "precsim", CodePoints: []int{8830}, Characters: []byte{0xe2, 0x89, 0xbe}},
	"prime":                           {Name: "prime", CodePoints: []int{8242}, Characters: []byte{0xe2, 0x80, 0xb2}},
	"primes":                          {Name: "primes", CodePoints: []int{8473}, Characters: []byte{0xe2, 0x84, 0x99}},
	"prnE":                            {Name: "prnE", CodePoints: []int{10933}, Characters: []byte{0xe2, 0xaa, 0xb5}},
	"prnap":                           {Name: "prnap", CodePoints: []int{10937}, Characters: []byte{0xe2, 0xaa, 0xb9}},
	"prnsim":                          {Name: "prnsim", CodePoints: []int{8936}, Characters: []byte{0xe2, 0x8b, 0xa8}},
	"prod":                            {Name: "prod", CodePoints: []int{8719}, Characters: []byte{0xe2, 0x88, 0x8f}},
	"profalar":                        {Name: "profalar", CodePoints: []int{9006}, Characters: []byte{0xe2, 0x8c, 0xae}},
	"profline":                        {Name: "profline", CodePoints: []int{8978}, Characters: []byte{0xe2, 0x8c, 0x92}},
	"profsurf":                        {Name: "profsurf", CodePoints: []int{8979}, Characters: []byte{0xe2, 0x8c, 0x93}},
	"prop":                            {Name: "prop", CodePoints: []int{8733}, Characters: []byte{0xe2, 0x88, 0x9d}},
	"propto":                          {Name: "propto", CodePoints: []int{8733}, Characters: []byte{0xe2, 0x88, 0x9d}},
	"prsim":                           {Name: "prsim", CodePoints: []int{8830}, Characters: []byte{0xe2, 0x89, 0xbe}},
	"prurel":                          {Name: "prurel", CodePoints: []int{8880}, Characters: []byte{0xe2, 0x8a, 0xb0}},
	"pscr":                            {Name: "pscr", CodePoints: []int{120005}, Characters: []byte{0xf0, 0x9d, 0x93, 0x85}},
	"psi":                             {Name: "psi", CodePoints: []int{968}, Characters: []byte{0xcf, 0x88}},
	"puncsp":                          {Name: "puncsp", CodePoints: []int{8200}, Characters: []byte{0xe2, 0x80, 0x88}},
	"qfr":                             {Name: "qfr", CodePoints: []int{120110}, Characters: []byte{0xf0, 0x9d, 0x94, 0xae}},
	"qint":                            {Name: "qint", CodePoints: []int{10764}, Characters: []byte{0xe2, 0xa8, 0x8c}},
	"qopf":                            {Name: "qopf", CodePoints: []int{120162}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa2}},
	"qprime":                          {Name: "qprime", CodePoints: []int{8279}, Characters: []byte{0xe2, 0x81, 0x97}},
	"qscr":                            {Name: "qscr", CodePoints: []int{120006}, Characters: []byte{0xf0, 0x9d, 0x93, 0x86}},
	"quaternions":                     {Name: "quaternions", CodePoints: []int{8461}, Characters: []byte{0xe2, 0x84, 0x8d}},
	"quatint":                         {Name: "quatint", CodePoints: []int{10774}, Characters: []byte{0xe2, 0xa8, 0x96}},
	"quest":                           {Name: "quest", CodePoints: []int{63}, Characters: []byte{0x3f}},
	"questeq":                         {Name: "questeq", CodePoints: []int{8799}, Characters: []byte{0xe2, 0x89, 0x9f}},
	"quot":                            {Name: "quot", CodePoints: []int{34}, Characters: []byte{0x22}},
	"rAarr":                           {Name: "rAarr", CodePoints: []int{8667}, Characters: []byte{0xe2, 0x87, 0x9b}},
	"rArr":                            {Name: "rArr", CodePoints: []int{8658}, Characters: []byte{0xe2, 0x87, 0x92}},
	"rAtail":                          {Name: "rAtail", CodePoints: []int{10524}, Characters: []byte{0xe2, 0xa4, 0x9c}},
	"rBarr":                           {Name: "rBarr", CodePoints: []int{10511}, Characters: []byte{0xe2, 0xa4, 0x8f}},
	"rHar":                            {Name: "rHar", CodePoints: []int{10596}, Characters: []byte{0xe2, 0xa5, 0xa4}},
	"race":                            {Name: "race", CodePoints: []int{8765, 817}, Characters: []byte{0xe2, 0x88, 0xbd, 0xcc, 0xb1}},
	"racute":                          {Name: "racute", CodePoints: []int{341}, Characters: []byte{0xc5, 0x95}},
	"radic":                           {Name: "radic", CodePoints: []int{8730}, Characters: []byte{0xe2, 0x88, 0x9a}},
	"raemptyv":                        {Name: "raemptyv", CodePoints: []int{10675}, Characters: []byte{0xe2, 0xa6, 0xb3}},
	"rang":                            {Name: "rang", CodePoints: []int{10217}, Characters: []byte{0xe2, 0x9f, 0xa9}},
	"rangd":                           {Name: "rangd", CodePoints: []int{10642}, Characters: []byte{0xe2, 0xa6, 0x92}},
	"range":                           {Name: "range", CodePoints: []int{10661}, Characters: []byte{0xe2, 0xa6, 0xa5}},
	"rangle":                          {Name: "rangle", CodePoints: []int{10217}, Characters: []byte{0xe2, 0x9f, 0xa9}},
	"raquo":                           {Name: "raquo", CodePoints: []int{187}, Characters: []byte{0xc2, 0xbb}},
	"rarr":                            {Name: "rarr", CodePoints: []int{8594}, Characters: []byte{0xe2, 0x86, 0x92}},
	"rarrap":                          {Name: "rarrap", CodePoints: []int{10613}, Characters: []byte{0xe2, 0xa5, 0xb5}},
	"rarrb":                           {Name: "rarrb", CodePoints: []int{8677}, Characters: []byte{0xe2, 0x87, 0xa5}},
	"rarrbfs":                         {Name: "rarrbfs", CodePoints: []int{10528}, Characters: []byte{0xe2, 0xa4, 0xa0}},
	"rarrc":                           {Name: "rarrc", CodePoints: []int{10547}, Characters: []byte{0xe2, 0xa4, 0xb3}},
	"rarrfs":                          {Name: "rarrfs", CodePoints: []int{10526}, Characters: []byte{0xe2, 0xa4, 0x9e}},
	"rarrhk":                          {Name: "rarrhk", CodePoints: []int{8618}, Characters: []byte{0xe2, 0x86, 0xaa}},
	"rarrlp":                          {Name: "rarrlp", CodePoints: []int{8620}, Characters: []byte{0xe2, 0x86, 0xac}},
	"rarrpl":                          {Name: "rarrpl", CodePoints: []int{10565}, Characters: []byte{0xe2, 0xa5, 0x85}},
	"rarrsim":                         {Name: "rarrsim", CodePoints: []int{10612}, Characters: []byte{0xe2, 0xa5, 0xb4}},
	"rarrtl":                          {Name: "rarrtl", CodePoints: []int{8611}, Characters: []byte{0xe2, 0x86, 0xa3}},
	"rarrw":                           {Name: "rarrw", CodePoints: []int{8605}, Characters: []byte{0xe2, 0x86, 0x9d}},
	"ratail":                          {Name: "ratail", CodePoints: []int{10522}, Characters: []byte{0xe2, 0xa4, 0x9a}},
	"ratio":                           {Name: "ratio", CodePoints: []int{8758}, Characters: []byte{0xe2, 0x88, 0xb6}},
	"rationals":                       {Name: "rationals", CodePoints: []int{8474}, Characters: []byte{0xe2, 0x84, 0x9a}},
	"rbarr":                           {Name: "rbarr", CodePoints: []int{10509}, Characters: []byte{0xe2, 0xa4, 0x8d}},
	"rbbrk":                           {Name: "rbbrk", CodePoints: []int{10099}, Characters: []byte{0xe2, 0x9d, 0xb3}},
	"rbrace":                          {Name: "rbrace", CodePoints: []int{125}, Characters: []byte{0x7d}},
	"rbrack":                          {Name: "rbrack", CodePoints: []int{93}, Characters: []byte{0x5d}},
	"rbrke":                           {Name: "rbrke", CodePoints: []int{10636}, Characters: []byte{0xe2, 0xa6, 0x8c}},
	"rbrksld":                         {Name: "rbrksld", CodePoints: []int{10638}, Characters: []byte{0xe2, 0xa6, 0x8e}},
	"rbrkslu":                         {Name: "rbrkslu", CodePoints: []int{10640}, Characters: []byte{0xe2, 0xa6, 0x90}},
	"rcaron":                          {Name: "rcaron", CodePoints: []int{345}, Characters: []byte{0xc5, 0x99}},
	"rcedil":                          {Name: "rcedil", CodePoints: []int{343}, Characters: []byte{0xc5, 0x97}},
	"rceil":                           {Name: "rceil", CodePoints: []int{8969}, Characters: []byte{0xe2, 0x8c, 0x89}},
	"rcub":                            {Name: "rcub", CodePoints: []int{125}, Characters: []byte{0x7d}},
	"rcy":                             {Name: "rcy", CodePoints: []int{1088}, Characters: []byte{0xd1, 0x80}},
	"rdca":                            {Name: "rdca", CodePoints: []int{10551}, Characters: []byte{0xe2, 0xa4, 0xb7}},
	"rdldhar":                         {Name: "rdldhar", CodePoints: []int{10601}, Characters: []byte{0xe2, 0xa5, 0xa9}},
	"rdquo":                           {Name: "rdquo", CodePoints: []int{8221}, Characters: []byte{0xe2, 0x80, 0x9d}},
	"rdquor":                          {Name: "rdquor", CodePoints: []int{8221}, Characters: []byte{0xe2, 0x80, 0x9d}},
	"rdsh":                            {Name: "rdsh", CodePoints: []int{8627}, Characters: []byte{0xe2, 0x86, 0xb3}},
	"real":                            {Name: "real", CodePoints: []int{8476}, Characters: []byte{0xe2, 0x84, 0x9c}},
	"realine":                         {Name: "realine", CodePoints: []int{8475}, Characters: []byte{0xe2, 0x84, 0x9b}},
	"realpart":                        {Name: "realpart", CodePoints: []int{8476}, Characters: []byte{0xe2, 0x84, 0x9c}},
	"reals":                           {Name: "reals", CodePoints: []int{8477}, Characters: []byte{0xe2, 0x84, 0x9d}},
	"rect":                            {Name: "rect", CodePoints: []int{9645}, Characters: []byte{0xe2, 0x96, 0xad}},
	"reg":                             {Name: "reg", CodePoints: []int{174}, Characters: []byte{0xc2, 0xae}},
	"rfisht":                          {Name: "rfisht", CodePoints: []int{10621}, Characters: []byte{0xe2, 0xa5, 0xbd}},
	"rfloor":                          {Name: "rfloor", CodePoints: []int{8971}, Characters: []byte{0xe2, 0x8c, 0x8b}},
	"rfr":                             {Name: "rfr", CodePoints: []int{120111}, Characters: []byte{0xf0, 0x9d, 0x94, 0xaf}},
	"rhard":                           {Name: "rhard", CodePoints: []int{8641}, Characters: []byte{0xe2, 0x87, 0x81}},
	"rharu":                           {Name: "rharu", CodePoints: []int{8640}, Characters: []byte{0xe2, 0x87, 0x80}},
	"rharul":                          {Name: "rharul", CodePoints: []int{10604}, Characters: []byte{0xe2, 0xa5, 0xac}},
	"rho":                             {Name: "rho", CodePoints: []int{961}, Characters: []byte{0xcf, 0x81}},
	"rhov":                            {Name: "rhov", CodePoints: []int{1009}, Characters: []byte{0xcf, 0xb1}},
	"rightarrow":                      {Name: "rightarrow", CodePoints: []int{8594}, Characters: []byte{0xe2, 0x86, 0x92}},
	"rightarrowtail":                  {Name: "rightarrowtail", CodePoints: []int{8611}, Characters: []byte{0xe2, 0x86, 0xa3}},
	"rightharpoondown":                {Name: "rightharpoondown", CodePoints: []int{8641}, Characters: []byte{0xe2, 0x87, 0x81}},
	"rightharpoonup":                  {Name: "rightharpoonup", CodePoints: []int{8640}, Characters: []byte{0xe2, 0x87, 0x80}},
	"rightleftarrows":                 {Name: "rightleftarrows", CodePoints: []int{8644}, Characters: []byte{0xe2, 0x87, 0x84}},
	"rightleftharpoons":               {Name: "rightleftharpoons", CodePoints: []int{8652}, Characters: []byte{0xe2, 0x87, 0x8c}},
	"rightrightarrows":                {Name: "rightrightarrows", CodePoints: []int{8649}, Characters: []byte{0xe2, 0x87, 0x89}},
	"rightsquigarrow":                 {Name: "rightsquigarrow", CodePoints: []int{8605}, Characters: []byte{0xe2, 0x86, 0x9d}},
	"rightthreetimes":                 {Name: "rightthreetimes", CodePoints: []int{8908}, Characters: []byte{0xe2, 0x8b, 0x8c}},
	"ring":                            {Name: "ring", CodePoints: []int{730}, Characters: []byte{0xcb, 0x9a}},
	"risingdotseq":                    {Name: "risingdotseq", CodePoints: []int{8787}, Characters: []byte{0xe2, 0x89, 0x93}},
	"rlarr":                           {Name: "rlarr", CodePoints: []int{8644}, Characters: []byte{0xe2, 0x87, 0x84}},
	"rlhar":                           {Name: "rlhar", CodePoints: []int{8652}, Characters: []byte{0xe2, 0x87, 0x8c}},
	"rlm":                             {Name: "rlm", CodePoints: []int{8207}, Characters: []byte{0xe2, 0x80, 0x8f}},
	"rmoust":                          {Name: "rmoust", CodePoints: []int{9137}, Characters: []byte{0xe2, 0x8e, 0xb1}},
	"rmoustache":                      {Name: "rmoustache", CodePoints: []int{9137}, Characters: []byte{0xe2, 0x8e, 0xb1}},
	"rnmid":                           {Name: "rnmid", CodePoints: []int{10990}, Characters: []byte{0xe2, 0xab, 0xae}},
	"roang":                           {Name: "roang", CodePoints: []int{10221}, Characters: []byte{0xe2, 0x9f, 0xad}},
	"roarr":                           {Name: "roarr", CodePoints: []int{8702}, Characters: []byte{0xe2, 0x87, 0xbe}},
	"robrk":                           {Name: "robrk", CodePoints: []int{10215}, Characters: []byte{0xe2, 0x9f, 0xa7}},
	"ropar":                           {Name: "ropar", CodePoints: []int{10630}, Characters: []byte{0xe2, 0xa6, 0x86}},
	"ropf":                            {Name: "ropf", CodePoints: []int{120163}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa3}},
	"roplus":                          {Name: "roplus", CodePoints: []int{10798}, Characters: []byte{0xe2, 0xa8, 0xae}},
	"rotimes":                         {Name: "rotimes", CodePoints: []int{10805}, Characters: []byte{0xe2, 0xa8, 0xb5}},
	"rpar":                            {Name: "rpar", CodePoints: []int{41}, Characters: []byte{0x29}},
	"rpargt":                          {Name: "rpargt", CodePoints: []int{10644}, Characters: []byte{0xe2, 0xa6, 0x94}},
	"rppolint":                        {Name: "rppolint", CodePoints: []int{10770}, Characters: []byte{0xe2, 0xa8, 0x92}},
	"rrarr":                           {Name: "rrarr", CodePoints: []int{8649}, Characters: []byte{0xe2, 0x87, 0x89}},
	"rsaquo":                          {Name: "rsaquo", CodePoints: []int{8250}, Characters: []byte{0xe2, 0x80, 0xba}},
	"rscr":                            {Name: "rscr", CodePoints: []int{120007}, Characters: []byte{0xf0, 0x9d, 0x93, 0x87}},
	"rsh":                             {Name: "rsh", CodePoints: []int{8625}, Characters: []byte{0xe2, 0x86, 0xb1}},
	"rsqb":                            {Name: "rsqb", CodePoints: []int{93}, Characters: []byte{0x5d}},
	"rsquo":                           {Name: "rsquo", CodePoints: []int{8217}, Characters: []byte{0xe2, 0x80, 0x99}},
	"rsquor":                          {Name: "rsquor", CodePoints: []int{8217}, Characters: []byte{0xe2, 0x80, 0x99}},
	"rthree":                          {Name: "rthree", CodePoints: []int{8908}, Characters: []byte{0xe2, 0x8b, 0x8c}},
	"rtimes":                          {Name: "rtimes", CodePoints: []int{8906}, Characters: []byte{0xe2, 0x8b, 0x8a}},
	"rtri":                            {Name: "rtri", CodePoints: []int{9657}, Characters: []byte{0xe2, 0x96, 0xb9}},
	"rtrie":                           {Name: "rtrie", CodePoints: []int{8885}, Characters: []byte{0xe2, 0x8a, 0xb5}},
	"rtrif":                           {Name: "rtrif", CodePoints: []int{9656}, Characters: []byte{0xe2, 0x96, 0xb8}},
	"rtriltri":                        {Name: "rtriltri", CodePoints: []int{10702}, Characters: []byte{0xe2, 0xa7, 0x8e}},
	"ruluhar":                         {Name: "ruluhar", CodePoints: []int{10600}, Characters: []byte{0xe2, 0xa5, 0xa8}},
	"rx":                              {Name: "rx", CodePoints: []int{8478}, Characters: []byte{0xe2, 0x84, 0x9e}},
	"sacute":                          {Name: "sacute", CodePoints: []int{347}, Characters: []byte{0xc5, 0x9b}},
	"sbquo":                           {Name: "sbquo", CodePoints: []int{8218}, Characters: []byte{0xe2, 0x80, 0x9a}},
	"sc":                              {Name: "sc", CodePoints: []int{8827}, Characters: []byte{0xe2, 0x89, 0xbb}},
	"scE":                             {Name: "scE", CodePoints: []int{10932}, Characters: []byte{0xe2, 0xaa, 0xb4}},
	"scap":                            {Name: "scap", CodePoints: []int{10936}, Characters: []byte{0xe2, 0xaa, 0xb8}},
	"scaron":                          {Name: "scaron", CodePoints: []int{353}, Characters: []byte{0xc5, 0xa1}},
	"sccue":                           {Name: "sccue", CodePoints: []int{8829}, Characters: []byte{0xe2, 0x89, 0xbd}},
	"sce":                             {Name: "sce", CodePoints: []int{10928}, Characters: []byte{0xe2, 0xaa, 0xb0}},
	"scedil":                          {Name: "scedil", CodePoints: []int{351}, Characters: []byte{0xc5, 0x9f}},
	"scirc":                           {Name: "scirc", CodePoints: []int{349}, Characters: []byte{0xc5, 0x9d}},
	"scnE":                            {Name: "scnE", CodePoints: []int{10934}, Characters: []byte{0xe2, 0xaa, 0xb6}},
	"scnap":                           {Name: "scnap", CodePoints: []int{10938}, Characters: []byte{0xe2, 0xaa, 0xba}},
	"scnsim":                          {Name: "scnsim", CodePoints: []int{8937}, Characters: []byte{0xe2, 0x8b, 0xa9}},
	"scpolint":                        {Name: "scpolint", CodePoints: []int{10771}, Characters: []byte{0xe2, 0xa8, 0x93}},
	"scsim":                           {Name: "scsim", CodePoints: []int{8831}, Characters: []byte{0xe2, 0x89, 0xbf}},
	"scy":                             {Name: "scy", CodePoints: []int{1089}, Characters: []byte{0xd1, 0x81}},
	"sdot":                            {Name: "sdot", CodePoints: []int{8901}, Characters: []byte{0xe2, 0x8b, 0x85}},
	"sdotb":                           {Name: "sdotb", CodePoints: []int{8865}, Characters: []byte{0xe2, 0x8a, 0xa1}},
	"sdote":                           {Name: "sdote", CodePoints: []int{10854}, Characters: []byte{0xe2, 0xa9, 0xa6}},
	"seArr":                           {Name: "seArr", CodePoints: []int{8664}, Characters: []byte{0xe2, 0x87, 0x98}},
	"searhk":                          {Name: "searhk", CodePoints: []int{10533}, Characters: []byte{0xe2, 0xa4, 0xa5}},
	"searr":                           {Name: "searr", CodePoints: []int{8600}, Characters: []byte{0xe2, 0x86, 0x98}},
	"searrow":                         {Name: "searrow", CodePoints: []int{8600}, Characters: []byte{0xe2, 0x86, 0x98}},
	"sect":                            {Name: "sect", CodePoints: []int{167}, Characters: []byte{0xc2, 0xa7}},
	"semi":                            {Name: "semi", CodePoints: []int{59}, Characters: []byte{0x3b}},
	"seswar":                          {Name: "seswar", CodePoints: []int{10537}, Characters: []byte{0xe2, 0xa4, 0xa9}},
	"setminus":                        {Name: "setminus", CodePoints: []int{8726}, Characters: []byte{0xe2, 0x88, 0x96}},
	"setmn":                           {Name: "setmn", CodePoints: []int{8726}, Characters: []byte{0xe2, 0x88, 0x96}},
	"sext":                            {Name: "sext", CodePoints: []int{10038}, Characters: []byte{0xe2, 0x9c, 0xb6}},
	"sfr":                             {Name: "sfr", CodePoints: []int{120112}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb0}},
	"sfrown":                          {Name: "sfrown", CodePoints: []int{8994}, Characters: []byte{0xe2, 0x8c, 0xa2}},
	"sharp":                           {Name: "sharp", CodePoints: []int{9839}, Characters: []byte{0xe2, 0x99, 0xaf}},
	"shchcy":                          {Name: "shchcy", CodePoints: []int{1097}, Characters: []byte{0xd1, 0x89}},
	"shcy":                            {Name: "shcy", CodePoints: []int{1096}, Characters: []byte{0xd1, 0x88}},
	"shortmid":                        {Name: "shortmid", CodePoints: []int{8739}, Characters: []byte{0xe2, 0x88, 0xa3}},
	"shortparallel":                   {Name: "shortparallel", CodePoints: []int{8741}, Characters: []byte{0xe2, 0x88, 0xa5}},
	"shy":                             {Name: "shy", CodePoints: []int{173}, Characters: []byte{0xc2, 0xad}},
	"sigma":                           {Name: "sigma", CodePoints: []int{963}, Characters: []byte{0xcf, 0x83}},
	"sigmaf":                          {Name: "sigmaf", CodePoints: []int{962}, Characters: []byte{0xcf, 0x82}},
	"sigmav":                          {Name: "sigmav", CodePoints: []int{962}, Characters: []byte{0xcf, 0x82}},
	"sim":                             {Name: "sim", CodePoints: []int{8764}, Characters: []byte{0xe2, 0x88, 0xbc}},
	"simdot":                          {Name: "simdot", CodePoints: []int{10858}, Characters: []byte{0xe2, 0xa9, 0xaa}},
	"sime":                            {Name: "sime", CodePoints: []int{8771}, Characters: []byte{0xe2, 0x89, 0x83}},
	"simeq":                           {Name: "simeq", CodePoints: []int{8771}, Characters: []byte{0xe2, 0x89, 0x83}},
	"simg":                            {Name: "simg", CodePoints: []int{10910}, Characters: []byte{0xe2, 0xaa, 0x9e}},
	"simgE":                           {Name: "simgE", CodePoints: []int{10912}, Characters: []byte{0xe2, 0xaa, 0xa0}},
	"siml":                            {Name: "siml", CodePoints: []int{10909}, Characters: []byte{0xe2, 0xaa, 0x9d}},
	"simlE":                           {Name: "simlE", CodePoints: []int{10911}, Characters: []byte{0xe2, 0xaa, 0x9f}},
	"simne":                           {Name: "simne", CodePoints: []int{8774}, Characters: []byte{0xe2, 0x89, 0x86}},
	"simplus":                         {Name: "simplus", CodePoints: []int{10788}, Characters: []byte{0xe2, 0xa8, 0xa4}},
	"simrarr":                         {Name: "simrarr", CodePoints: []int{10610}, Characters: []byte{0xe2, 0xa5, 0xb2}},
	"slarr":                           {Name: "slarr", CodePoints: []int{8592}, Characters: []byte{0xe2, 0x86, 0x90}},
	"smallsetminus":                   {Name: "smallsetminus", CodePoints: []int{8726}, Characters: []byte{0xe2, 0x88, 0x96}},
	"smashp":                          {Name: "smashp", CodePoints: []int{10803}, Characters: []byte{0xe2, 0xa8, 0xb3}},
	"smeparsl":                        {Name: "smeparsl", CodePoints: []int{10724}, Characters: []byte{0xe2, 0xa7, 0xa4}},
	"smid":                            {Name: "smid", CodePoints: []int{8739}, Characters: []byte{0xe2, 0x88, 0xa3}},
	"smile":                           {Name: "smile", CodePoints: []int{8995}, Characters: []byte{0xe2, 0x8c, 0xa3}},
	"smt":                             {Name: "smt", CodePoints: []int{10922}, Characters: []byte{0xe2, 0xaa, 0xaa}},
	"smte":                            {Name: "smte", CodePoints: []int{10924}, Characters: []byte{0xe2, 0xaa, 0xac}},
	"smtes":                           {Name: "smtes", CodePoints: []int{10924, 65024}, Characters: []byte{0xe2, 0xaa, 0xac, 0xef, 0xb8, 0x80}},
	"softcy":                          {Name: "softcy", CodePoints: []int{1100}, Characters: []byte{0xd1, 0x8c}},
	"sol":                             {Name: "sol", CodePoints: []int{47}, Characters: []byte{0x2f}},
	"solb":                            {Name: "solb", CodePoints: []int{10692}, Characters: []byte{0xe2, 0xa7, 0x84}},
	"solbar":                          {Name: "solbar", CodePoints: []int{9023}, Characters: []byte{0xe2, 0x8c, 0xbf}},
	"sopf":                            {Name: "sopf", CodePoints: []int{120164}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa4}},
	"spades":                          {Name: "spades", CodePoints: []int{9824}, Characters: []byte{0xe2, 0x99, 0xa0}},
	"spadesuit":                       {Name: "spadesuit", CodePoints: []int{9824}, Characters: []byte{0xe2, 0x99, 0xa0}},
	"spar":                            {Name: "spar", CodePoints: []int{8741}, Characters: []byte{0xe2, 0x88, 0xa5}},
	"sqcap":                           {Name: "sqcap", CodePoints: []int{8851}, Characters: []byte{0xe2, 0x8a, 0x93}},
	"sqcaps":                          {Name: "sqcaps", CodePoints: []int{8851, 65024}, Characters: []byte{0xe2, 0x8a, 0x93, 0xef, 0xb8, 0x80}},
	"sqcup":                           {Name: "sqcup", CodePoints: []int{8852}, Characters: []byte{0xe2, 0x8a, 0x94}},
	"sqcups":                          {Name: "sqcups", CodePoints: []int{8852, 65024}, Characters: []byte{0xe2, 0x8a, 0x94, 0xef, 0xb8, 0x80}},
	"sqsub":                           {Name: "sqsub", CodePoints: []int{8847}, Characters: []byte{0xe2, 0x8a, 0x8f}},
	"sqsube":                          {Name: "sqsube", CodePoints: []int{8849}, Characters: []byte{0xe2, 0x8a, 0x91}},
	"sqsubset":                        {Name: "sqsubset", CodePoints: []int{8847}, Characters: []byte{0xe2, 0x8a, 0x8f}},
	"sqsubseteq":                      {Name: "sqsubseteq", CodePoints: []int{8849}, Characters: []byte{0xe2, 0x8a, 0x91}},
	"sqsup":                           {Name: "sqsup", CodePoints: []int{8848}, Characters: []byte{0xe2, 0x8a, 0x90}},
	"sqsupe":                          {Name: "sqsupe", CodePoints: []int{8850}, Characters: []byte{0xe2, 0x8a, 0x92}},
	"sqsupset":                        {Name: "sqsupset", CodePoints: []int{8848}, Characters: []byte{0xe2, 0x8a, 0x90}},
	"sqsupseteq":                      {Name: "sqsupseteq", CodePoints: []int{8850}, Characters: []byte{0xe2, 0x8a, 0x92}},
	"squ":                             {Name: "squ", CodePoints: []int{9633}, Characters: []byte{0xe2, 0x96, 0xa1}},
	"square":                          {Name: "square", CodePoints: []int{9633}, Characters: []byte{0xe2, 0x96, 0xa1}},
	"squarf":                          {Name: "squarf", CodePoints: []int{9642}, Characters: []byte{0xe2, 0x96, 0xaa}},
	"squf":                            {Name: "squf", CodePoints: []int{9642}, Characters: []byte{0xe2, 0x96, 0xaa}},
	"srarr":                           {Name: "srarr", CodePoints: []int{8594}, Characters: []byte{0xe2, 0x86, 0x92}},
	"sscr":                            {Name: "sscr", CodePoints: []int{120008}, Characters: []byte{0xf0, 0x9d, 0x93, 0x88}},
	"ssetmn":                          {Name: "ssetmn", CodePoints: []int{8726}, Characters: []byte{0xe2, 0x88, 0x96}},
	"ssmile":                          {Name: "ssmile", CodePoints: []int{8995}, Characters: []byte{0xe2, 0x8c, 0xa3}},
	"sstarf":                          {Name: "sstarf", CodePoints: []int{8902}, Characters: []byte{0xe2, 0x8b, 0x86}},
	"star":                            {Name: "star", CodePoints: []int{9734}, Characters: []byte{0xe2, 0x98, 0x86}},
	"starf":                           {Name: "starf", CodePoints: []int{9733}, Characters: []byte{0xe2, 0x98, 0x85}},
	"straightepsilon":                 {Name: "straightepsilon", CodePoints: []int{1013}, Characters: []byte{0xcf, 0xb5}},
	"straightphi":                     {Name: "straightphi", CodePoints: []int{981}, Characters: []byte{0xcf, 0x95}},
	"strns":                           {Name: "strns", CodePoints: []int{175}, Characters: []byte{0xc2, 0xaf}},
	"sub":                             {Name: "sub", CodePoints: []int{8834}, Characters: []byte{0xe2, 0x8a, 0x82}},
	"subE":                            {Name: "subE", CodePoints: []int{10949}, Characters: []byte{0xe2, 0xab, 0x85}},
	"subdot":                          {Name: "subdot", CodePoints: []int{10941}, Characters: []byte{0xe2, 0xaa, 0xbd}},
	"sube":                            {Name: "sube", CodePoints: []int{8838}, Characters: []byte{0xe2, 0x8a, 0x86}},
	"subedot":                         {Name: "subedot", CodePoints: []int{10947}, Characters: []byte{0xe2, 0xab, 0x83}},
	"submult":                         {Name: "submult", CodePoints: []int{10945}, Characters: []byte{0xe2, 0xab, 0x81}},
	"subnE":                           {Name: "subnE", CodePoints: []int{10955}, Characters: []byte{0xe2, 0xab, 0x8b}},
	"subne":                           {Name: "subne", CodePoints: []int{8842}, Characters: []byte{0xe2, 0x8a, 0x8a}},
	"subplus":                         {Name: "subplus", CodePoints: []int{10943}, Characters: []byte{0xe2, 0xaa, 0xbf}},
	"subrarr":                         {Name: "subrarr", CodePoints: []int{10617}, Characters: []byte{0xe2, 0xa5, 0xb9}},
	"subset":                          {Name: "subset", CodePoints: []int{8834}, Characters: []byte{0xe2, 0x8a, 0x82}},
	"subseteq":                        {Name: "subseteq", CodePoints: []int{8838}, Characters: []byte{0xe2, 0x8a, 0x86}},
	"subseteqq":                       {Name: "subseteqq", CodePoints: []int{10949}, Characters: []byte{0xe2, 0xab, 0x85}},
	"subsetneq":                       {Name: "subsetneq", CodePoints: []int{8842}, Characters: []byte{0xe2, 0x8a, 0x8a}},
	"subsetneqq":                      {Name: "subsetneqq", CodePoints: []int{10955}, Characters: []byte{0xe2, 0xab, 0x8b}},
	"subsim":                          {Name: "subsim", CodePoints: []int{10951}, Characters: []byte{0xe2, 0xab, 0x87}},
	"subsub":                          {Name: "subsub", CodePoints: []int{10965}, Characters: []byte{0xe2, 0xab, 0x95}},
	"subsup":                          {Name: "subsup", CodePoints: []int{10963}, Characters: []byte{0xe2, 0xab, 0x93}},
	"succ":                            {Name: "succ", CodePoints: []int{8827}, Characters: []byte{0xe2, 0x89, 0xbb}},
	"succapprox":                      {Name: "succapprox", CodePoints: []int{10936}, Characters: []byte{0xe2, 0xaa, 0xb8}},
	"succcurlyeq":                     {Name: "succcurlyeq", CodePoints: []int{8829}, Characters: []byte{0xe2, 0x89, 0xbd}},
	"succeq":                          {Name: "succeq", CodePoints: []int{10928}, Characters: []byte{0xe2, 0xaa, 0xb0}},
	"succnapprox":                     {Name: "succnapprox", CodePoints: []int{10938}, Characters: []byte{0xe2, 0xaa, 0xba}},
	"succneqq":                        {Name: "succneqq", CodePoints: []int{10934}, Characters: []byte{0xe2, 0xaa, 0xb6}},
	"succnsim":                        {Name: "succnsim", CodePoints: []int{8937}, Characters: []byte{0xe2, 0x8b, 0xa9}},
	"succsim":                         {Name: "succsim", CodePoints: []int{8831}, Characters: []byte{0xe2, 0x89, 0xbf}},
	"sum":                             {Name: "sum", CodePoints: []int{8721}, Characters: []byte{0xe2, 0x88, 0x91}},
	"sung":                            {Name: "sung", CodePoints: []int{9834}, Characters: []byte{0xe2, 0x99, 0xaa}},
	"sup":                             {Name: "sup", CodePoints: []int{8835}, Characters: []byte{0xe2, 0x8a, 0x83}},
	"sup1":                            {Name: "sup1", CodePoints: []int{185}, Characters: []byte{0xc2, 0xb9}},
	"sup2":                            {Name: "sup2", CodePoints: []int{178}, Characters: []byte{0xc2, 0xb2}},
	"sup3":                            {Name: "sup3", CodePoints: []int{179}, Characters: []byte{0xc2, 0xb3}},
	"supE":                            {Name: "supE", CodePoints: []int{10950}, Characters: []byte{0xe2, 0xab, 0x86}},
	"supdot":                          {Name: "supdot", CodePoints: []int{10942}, Characters: []byte{0xe2, 0xaa, 0xbe}},
	"supdsub":                         {Name: "supdsub", CodePoints: []int{10968}, Characters: []byte{0xe2, 0xab, 0x98}},
	"supe":                            {Name: "supe", CodePoints: []int{8839}, Characters: []byte{0xe2, 0x8a, 0x87}},
	"supedot":                         {Name: "supedot", CodePoints: []int{10948}, Characters: []byte{0xe2, 0xab, 0x84}},
	"suphsol":                         {Name: "suphsol", CodePoints: []int{10185}, Characters: []byte{0xe2, 0x9f, 0x89}},
	"suphsub":                         {Name: "suphsub", CodePoints: []int{10967}, Characters: []byte{0xe2, 0xab, 0x97}},
	"suplarr":                         {Name: "suplarr", CodePoints: []int{10619}, Characters: []byte{0xe2, 0xa5, 0xbb}},
	"supmult":                         {Name: "supmult", CodePoints: []int{10946}, Characters: []byte{0xe2, 0xab, 0x82}},
	"supnE":                           {Name: "supnE", CodePoints: []int{10956}, Characters: []byte{0xe2, 0xab, 0x8c}},
	"supne":                           {Name: "supne", CodePoints: []int{8843}, Characters: []byte{0xe2, 0x8a, 0x8b}},
	"supplus":                         {Name: "supplus", CodePoints: []int{10944}, Characters: []byte{0xe2, 0xab, 0x80}},
	"supset":                          {Name: "supset", CodePoints: []int{8835}, Characters: []byte{0xe2, 0x8a, 0x83}},
	"supseteq":                        {Name: "supseteq", CodePoints: []int{8839}, Characters: []byte{0xe2, 0x8a, 0x87}},
	"supseteqq":                       {Name: "supseteqq", CodePoints: []int{10950}, Characters: []byte{0xe2, 0xab, 0x86}},
	"supsetneq":                       {Name: "supsetneq", CodePoints: []int{8843}, Characters: []byte{0xe2, 0x8a, 0x8b}},
	"supsetneqq":                      {Name: "supsetneqq", CodePoints: []int{10956}, Characters: []byte{0xe2, 0xab, 0x8c}},
	"supsim":                          {Name: "supsim", CodePoints: []int{10952}, Characters: []byte{0xe2, 0xab, 0x88}},
	"supsub":                          {Name: "supsub", CodePoints: []int{10964}, Characters: []byte{0xe2, 0xab, 0x94}},
	"supsup":                          {Name: "supsup", CodePoints: []int{10966}, Characters: []byte{0xe2, 0xab, 0x96}},
	"swArr":                           {Name: "swArr", CodePoints: []int{8665}, Characters: []byte{0xe2, 0x87, 0x99}},
	"swarhk":                          {Name: "swarhk", CodePoints: []int{10534}, Characters: []byte{0xe2, 0xa4, 0xa6}},
	"swarr":                           {Name: "swarr", CodePoints: []int{8601}, Characters: []byte{0xe2, 0x86, 0x99}},
	"swarrow":                         {Name: "swarrow", CodePoints: []int{8601}, Characters: []byte{0xe2, 0x86, 0x99}},
	"swnwar":                          {Name: "swnwar", CodePoints: []int{10538}, Characters: []byte{0xe2, 0xa4, 0xaa}},
	"szlig":                           {Name: "szlig", CodePoints: []int{223}, Characters: []byte{0xc3, 0x9f}},
	"target":                          {Name: "target", CodePoints: []int{8982}, Characters: []byte{0xe2, 0x8c, 0x96}},
	"tau":                             {Name: "tau", CodePoints: []int{964}, Characters: []byte{0xcf, 0x84}},
	"tbrk":                            {Name: "tbrk", CodePoints: []int{9140}, Characters: []byte{0xe2, 0x8e, 0xb4}},
	"tcaron":                          {Name: "tcaron", CodePoints: []int{357}, Characters: []byte{0xc5, 0xa5}},
	"tcedil":                          {Name: "tcedil", CodePoints: []int{355}, Characters: []byte{0xc5, 0xa3}},
	"tcy":                             {Name: "tcy", CodePoints: []int{1090}, Characters: []byte{0xd1, 0x82}},
	"tdot":                            {Name: "tdot", CodePoints: []int{8411}, Characters: []byte{0xe2, 0x83, 0x9b}},
	"telrec":                          {Name: "telrec", CodePoints: []int{8981}, Characters: []byte{0xe2, 0x8c, 0x95}},
	"tfr":                             {Name: "tfr", CodePoints: []int{120113}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb1}},
	"there4":                          {Name: "there4", CodePoints: []int{8756}, Characters: []byte{0xe2, 0x88, 0xb4}},
	"therefore":                       {Name: "therefore", CodePoints: []int{8756}, Characters: []byte{0xe2, 0x88, 0xb4}},
	"theta":                           {Name: "theta", CodePoints: []int{952}, Characters: []byte{0xce, 0xb8}},
	"thetasym":                        {Name: "thetasym", CodePoints: []int{977}, Characters: []byte{0xcf, 0x91}},
	"thetav":                          {Name: "thetav", CodePoints: []int{977}, Characters: []byte{0xcf, 0x91}},
	"thickapprox":                     {Name: "thickapprox", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"thicksim":                        {Name: "thicksim", CodePoints: []int{8764}, Characters: []byte{0xe2, 0x88, 0xbc}},
	"thinsp":                          {Name: "thinsp", CodePoints: []int{8201}, Characters: []byte{0xe2, 0x80, 0x89}},
	"thkap":                           {Name: "thkap", CodePoints: []int{8776}, Characters: []byte{0xe2, 0x89, 0x88}},
	"thksim":                          {Name: "thksim", CodePoints: []int{8764}, Characters: []byte{0xe2, 0x88, 0xbc}},
	"thorn":                           {Name: "thorn", CodePoints: []int{254}, Characters: []byte{0xc3, 0xbe}},
	"tilde":                           {Name: "tilde", CodePoints: []int{732}, Characters: []byte{0xcb, 0x9c}},
	"times":                           {Name: "times", CodePoints: []int{215}, Characters: []byte{0xc3, 0x97}},
	"timesb":                          {Name: "timesb", CodePoints: []int{8864}, Characters: []byte{0xe2, 0x8a, 0xa0}},
	"timesbar":                        {Name: "timesbar", CodePoints: []int{10801}, Characters: []byte{0xe2, 0xa8, 0xb1}},
	"timesd":                          {Name: "timesd", CodePoints: []int{10800}, Characters: []byte{0xe2, 0xa8, 0xb0}},
	"tint":                            {Name: "tint", CodePoints: []int{8749}, Characters: []byte{0xe2, 0x88, 0xad}},
	"toea":                            {Name: "toea", CodePoints: []int{10536}, Characters: []byte{0xe2, 0xa4, 0xa8}},
	"top":                             {Name: "top", CodePoints: []int{8868}, Characters: []byte{0xe2, 0x8a, 0xa4}},
	"topbot":                          {Name: "topbot", CodePoints: []int{9014}, Characters: []byte{0xe2, 0x8c, 0xb6}},
	"topcir":                          {Name: "topcir", CodePoints: []int{10993}, Characters: []byte{0xe2, 0xab, 0xb1}},
	"topf":                            {Name: "topf", CodePoints: []int{120165}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa5}},
	"topfork":                         {Name: "topfork", CodePoints: []int{10970}, Characters: []byte{0xe2, 0xab, 0x9a}},
	"tosa":                            {Name: "tosa", CodePoints: []int{10537}, Characters: []byte{0xe2, 0xa4, 0xa9}},
	"tprime":                          {Name: "tprime", CodePoints: []int{8244}, Characters: []byte{0xe2, 0x80, 0xb4}},
	"trade":                           {Name: "trade", CodePoints: []int{8482}, Characters: []byte{0xe2, 0x84, 0xa2}},
	"triangle":                        {Name: "triangle", CodePoints: []int{9653}, Characters: []byte{0xe2, 0x96, 0xb5}},
	"triangledown":                    {Name: "triangledown", CodePoints: []int{9663}, Characters: []byte{0xe2, 0x96, 0xbf}},
	"triangleleft":                    {Name: "triangleleft", CodePoints: []int{9667}, Characters: []byte{0xe2, 0x97, 0x83}},
	"trianglelefteq":                  {Name: "trianglelefteq", CodePoints: []int{8884}, Characters: []byte{0xe2, 0x8a, 0xb4}},
	"triangleq":                       {Name: "triangleq", CodePoints: []int{8796}, Characters: []byte{0xe2, 0x89, 0x9c}},
	"triangleright":                   {Name: "triangleright", CodePoints: []int{9657}, Characters: []byte{0xe2, 0x96, 0xb9}},
	"trianglerighteq":                 {Name: "trianglerighteq", CodePoints: []int{8885}, Characters: []byte{0xe2, 0x8a, 0xb5}},
	"tridot":                          {Name: "tridot", CodePoints: []int{9708}, Characters: []byte{0xe2, 0x97, 0xac}},
	"trie":                            {Name: "trie", CodePoints: []int{8796}, Characters: []byte{0xe2, 0x89, 0x9c}},
	"triminus":                        {Name: "triminus", CodePoints: []int{10810}, Characters: []byte{0xe2, 0xa8, 0xba}},
	"triplus":                         {Name: "triplus", CodePoints: []int{10809}, Characters: []byte{0xe2, 0xa8, 0xb9}},
	"trisb":                           {Name: "trisb", CodePoints: []int{10701}, Characters: []byte{0xe2, 0xa7, 0x8d}},
	"tritime":                         {Name: "tritime", CodePoints: []int{10811}, Characters: []byte{0xe2, 0xa8, 0xbb}},
	"trpezium":                        {Name: "trpezium", CodePoints: []int{9186}, Characters: []byte{0xe2, 0x8f, 0xa2}},
	"tscr":                            {Name: "tscr", CodePoints: []int{120009}, Characters: []byte{0xf0, 0x9d, 0x93, 0x89}},
	"tscy":                            {Name: "tscy", CodePoints: []int{1094}, Characters: []byte{0xd1, 0x86}},
	"tshcy":                           {Name: "tshcy", CodePoints: []int{1115}, Characters: []byte{0xd1, 0x9b}},
	"tstrok":                          {Name: "tstrok", CodePoints: []int{359}, Characters: []byte{0xc5, 0xa7}},
	"twixt":                           {Name: "twixt", CodePoints: []int{8812}, Characters: []byte{0xe2, 0x89, 0xac}},
	"twoheadleftarrow":                {Name: "twoheadleftarrow", CodePoints: []int{8606}, Characters: []byte{0xe2, 0x86, 0x9e}},
	"twoheadrightarrow":               {Name: "twoheadrightarrow", CodePoints: []int{8608}, Characters: []byte{0xe2, 0x86, 0xa0}},
	"uArr":                            {Name: "uArr", CodePoints: []int{8657}, Characters: []byte{0xe2, 0x87, 0x91}},
	"uHar":                            {Name: "uHar", CodePoints: []int{10595}, Characters: []byte{0xe2, 0xa5, 0xa3}},
	"uacute":                          {Name: "uacute", CodePoints: []int{250}, Characters: []byte{0xc3, 0xba}},
	"uarr":                            {Name: "uarr", CodePoints: []int{8593}, Characters: []byte{0xe2, 0x86, 0x91}},
	"ubrcy":                           {Name: "ubrcy", CodePoints: []int{1118}, Characters: []byte{0xd1, 0x9e}},
	"ubreve":                          {Name: "ubreve", CodePoints: []int{365}, Characters: []byte{0xc5, 0xad}},
	"ucirc":                           {Name: "ucirc", CodePoints: []int{251}, Characters: []byte{0xc3, 0xbb}},
	"ucy":                             {Name: "ucy", CodePoints: []int{1091}, Characters: []byte{0xd1, 0x83}},
	"udarr":                           {Name: "udarr", CodePoints: []int{8645}, Characters: []byte{0xe2, 0x87, 0x85}},
	"udblac":                          {Name: "udblac", CodePoints: []int{369}, Characters: []byte{0xc5, 0xb1}},
	"udhar":                           {Name: "udhar", CodePoints: []int{10606}, Characters: []byte{0xe2, 0xa5, 0xae}},
	"ufisht":                          {Name: "ufisht", CodePoints: []int{10622}, Characters: []byte{0xe2, 0xa5, 0xbe}},
	"ufr":                             {Name: "ufr", CodePoints: []int{120114}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb2}},
	"ugrave":                          {Name: "ugrave", CodePoints: []int{249}, Characters: []byte{0xc3, 0xb9}},
	"uharl":                           {Name: "uharl", CodePoints: []int{8639}, Characters: []byte{0xe2, 0x86, 0xbf}},
	"uharr":                           {Name: "uharr", CodePoints: []int{8638}, Characters: []byte{0xe2, 0x86, 0xbe}},
	"uhblk":                           {Name: "uhblk", CodePoints: []int{9600}, Characters: []byte{0xe2, 0x96, 0x80}},
	"ulcorn":                          {Name: "ulcorn", CodePoints: []int{8988}, Characters: []byte{0xe2, 0x8c, 0x9c}},
	"ulcorner":                        {Name: "ulcorner", CodePoints: []int{8988}, Characters: []byte{0xe2, 0x8c, 0x9c}},
	"ulcrop":                          {Name: "ulcrop", CodePoints: []int{8975}, Characters: []byte{0xe2, 0x8c, 0x8f}},
	"ultri":                           {Name: "ultri", CodePoints: []int{9720}, Characters: []byte{0xe2, 0x97, 0xb8}},
	"umacr":                           {Name: "umacr", CodePoints: []int{363}, Characters: []byte{0xc5, 0xab}},
	"uml":                             {Name: "uml", CodePoints: []int{168}, Characters: []byte{0xc2, 0xa8}},
	"uogon":                           {Name: "uogon", CodePoints: []int{371}, Characters: []byte{0xc5, 0xb3}},
	"uopf":                            {Name: "uopf", CodePoints: []int{120166}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa6}},
	"uparrow":                         {Name: "uparrow", CodePoints: []int{8593}, Characters: []byte{0xe2, 0x86, 0x91}},
	"updownarrow":                     {Name: "updownarrow", CodePoints: []int{8597}, Characters: []byte{0xe2, 0x86, 0x95}},
	"upharpoonleft":                   {Name: "upharpoonleft", CodePoints: []int{8639}, Characters: []byte{0xe2, 0x86, 0xbf}},
	"upharpoonright":                  {Name: "upharpoonright", CodePoints: []int{8638}, Characters: []byte{0xe2, 0x86, 0xbe}},
	"uplus":                           {Name: "uplus", CodePoints: []int{8846}, Characters: []byte{0xe2, 0x8a, 0x8e}},
	"upsi":                            {Name: "upsi", CodePoints: []int{965}, Characters: []byte{0xcf, 0x85}},
	"upsih":                           {Name: "upsih", CodePoints: []int{978}, Characters: []byte{0xcf, 0x92}},
	"upsilon":                         {Name: "upsilon", CodePoints: []int{965}, Characters: []byte{0xcf, 0x85}},
	"upuparrows":                      {Name: "upuparrows", CodePoints: []int{8648}, Characters: []byte{0xe2, 0x87, 0x88}},
	"urcorn":                          {Name: "urcorn", CodePoints: []int{8989}, Characters: []byte{0xe2, 0x8c, 0x9d}},
	"urcorner":                        {Name: "urcorner", CodePoints: []int{8989}, Characters: []byte{0xe2, 0x8c, 0x9d}},
	"urcrop":                          {Name: "urcrop", CodePoints: []int{8974}, Characters: []byte{0xe2, 0x8c, 0x8e}},
	"uring":                           {Name: "uring", CodePoints: []int{367}, Characters: []byte{0xc5, 0xaf}},
	"urtri":                           {Name: "urtri", CodePoints: []int{9721}, Characters: []byte{0xe2, 0x97, 0xb9}},
	"uscr":                            {Name: "uscr", CodePoints: []int{120010}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8a}},
	"utdot":                           {Name: "utdot", CodePoints: []int{8944}, Characters: []byte{0xe2, 0x8b, 0xb0}},
	"utilde":                          {Name: "utilde", CodePoints: []int{361}, Characters: []byte{0xc5, 0xa9}},
	"utri":                            {Name: "utri", CodePoints: []int{9653}, Characters: []byte{0xe2, 0x96, 0xb5}},
	"utrif":                           {Name: "utrif", CodePoints: []int{9652}, Characters: []byte{0xe2, 0x96, 0xb4}},
	"uuarr":                           {Name: "uuarr", CodePoints: []int{8648}, Characters: []byte{0xe2, 0x87, 0x88}},
	"uuml":                            {Name: "uuml", CodePoints: []int{252}, Characters: []byte{0xc3, 0xbc}},
	"uwangle":                         {Name: "uwangle", CodePoints: []int{10663}, Characters: []byte{0xe2, 0xa6, 0xa7}},
	"vArr":                            {Name: "vArr", CodePoints: []int{8661}, Characters: []byte{0xe2, 0x87, 0x95}},
	"vBar":                            {Name: "vBar", CodePoints: []int{10984}, Characters: []byte{0xe2, 0xab, 0xa8}},
	"vBarv":                           {Name: "vBarv", CodePoints: []int{10985}, Characters: []byte{0xe2, 0xab, 0xa9}},
	"vDash":                           {Name: "vDash", CodePoints: []int{8872}, Characters: []byte{0xe2, 0x8a, 0xa8}},
	"vangrt":                          {Name: "vangrt", CodePoints: []int{10652}, Characters: []byte{0xe2, 0xa6, 0x9c}},
	"varepsilon":                      {Name: "varepsilon", CodePoints: []int{1013}, Characters: []byte{0xcf, 0xb5}},
	"varkappa":                        {Name: "varkappa", CodePoints: []int{1008}, Characters: []byte{0xcf, 0xb0}},
	"varnothing":                      {Name: "varnothing", CodePoints: []int{8709}, Characters: []byte{0xe2, 0x88, 0x85}},
	"varphi":                          {Name: "varphi", CodePoints: []int{981}, Characters: []byte{0xcf, 0x95}},
	"varpi":                           {Name: "varpi", CodePoints: []int{982}, Characters: []byte{0xcf, 0x96}},
	"varpropto":                       {Name: "varpropto", CodePoints: []int{8733}, Characters: []byte{0xe2, 0x88, 0x9d}},
	"varr":                            {Name: "varr", CodePoints: []int{8597}, Characters: []byte{0xe2, 0x86, 0x95}},
	"varrho":                          {Name: "varrho", CodePoints: []int{1009}, Characters: []byte{0xcf, 0xb1}},
	"varsigma":                        {Name: "varsigma", CodePoints: []int{962}, Characters: []byte{0xcf, 0x82}},
	"varsubsetneq":                    {Name: "varsubsetneq", CodePoints: []int{8842, 65024}, Characters: []byte{0xe2, 0x8a, 0x8a, 0xef, 0xb8, 0x80}},
	"varsubsetneqq":                   {Name: "varsubsetneqq", CodePoints: []int{10955, 65024}, Characters: []byte{0xe2, 0xab, 0x8b, 0xef, 0xb8, 0x80}},
	"varsupsetneq":                    {Name: "varsupsetneq", CodePoints: []int{8843, 65024}, Characters: []byte{0xe2, 0x8a, 0x8b, 0xef, 0xb8, 0x80}},
	"varsupsetneqq":                   {Name: "varsupsetneqq", CodePoints: []int{10956, 65024}, Characters: []byte{0xe2, 0xab, 0x8c, 0xef, 0xb8, 0x80}},
	"vartheta":                        {Name: "vartheta", CodePoints: []int{977}, Characters: []byte{0xcf, 0x91}},
	"vartriangleleft":                 {Name: "vartriangleleft", CodePoints: []int{8882}, Characters: []byte{0xe2, 0x8a, 0xb2}},
	"vartriangleright":                {Name: "vartriangleright", CodePoints: []int{8883}, Characters: []byte{0xe2, 0x8a, 0xb3}},
	"vcy":                             {Name: "vcy", CodePoints: []int{1074}, Characters: []byte{0xd0, 0xb2}},
	"vdash":                           {Name: "vdash", CodePoints: []int{8866}, Characters: []byte{0xe2, 0x8a, 0xa2}},
	"vee":                             {Name: "vee", CodePoints: []int{8744}, Characters: []byte{0xe2, 0x88, 0xa8}},
	"veebar":                          {Name: "veebar", CodePoints: []int{8891}, Characters: []byte{0xe2, 0x8a, 0xbb}},
	"veeeq":                           {Name: "veeeq", CodePoints: []int{8794}, Characters: []byte{0xe2, 0x89, 0x9a}},
	"vellip":                          {Name: "vellip", CodePoints: []int{8942}, Characters: []byte{0xe2, 0x8b, 0xae}},
	"verbar":                          {Name: "verbar", CodePoints: []int{124}, Characters: []byte{0x7c}},
	"vert":                            {Name: "vert", CodePoints: []int{124}, Characters: []byte{0x7c}},
	"vfr":                             {Name: "vfr", CodePoints: []int{120115}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb3}},
	"vltri":                           {Name: "vltri", CodePoints: []int{8882}, Characters: []byte{0xe2, 0x8a, 0xb2}},
	"vnsub":                           {Name: "vnsub", CodePoints: []int{8834, 8402}, Characters: []byte{0xe2, 0x8a, 0x82, 0xe2, 0x83, 0x92}},
	"vnsup":                           {Name: "vnsup", CodePoints: []int{8835, 8402}, Characters: []byte{0xe2, 0x8a, 0x83, 0xe2, 0x83, 0x92}},
	"vopf":                            {Name: "vopf", CodePoints: []int{120167}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa7}},
	"vprop":                           {Name: "vprop", CodePoints: []int{8733}, Characters: []byte{0xe2, 0x88, 0x9d}},
	"vrtri":                           {Name: "vrtri", CodePoints: []int{8883}, Characters: []byte{0xe2, 0x8a, 0xb3}},
	"vscr":                            {Name: "vscr", CodePoints: []int{120011}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8b}},
	"vsubnE":                          {Name: "vsubnE", CodePoints: []int{10955, 65024}, Characters: []byte{0xe2, 0xab, 0x8b, 0xef, 0xb8, 0x80}},
	"vsubne":                          {Name: "vsubne", CodePoints: []int{8842, 65024}, Characters: []byte{0xe2, 0x8a, 0x8a, 0xef, 0xb8, 0x80}},
	"vsupnE":                          {Name: "vsupnE", CodePoints: []int{10956, 65024}, Characters: []byte{0xe2, 0xab, 0x8c, 0xef, 0xb8, 0x80}},
	"vsupne":                          {Name: "vsupne", CodePoints: []int{8843, 65024}, Characters: []byte{0xe2, 0x8a, 0x8b, 0xef, 0xb8, 0x80}},
	"vzigzag":                         {Name: "vzigzag", CodePoints: []int{10650}, Characters: []byte{0xe2, 0xa6, 0x9a}},
	"wcirc":                           {Name: "wcirc", CodePoints: []int{373}, Characters: []byte{0xc5, 0xb5}},
	"wedbar":                          {Name: "wedbar", CodePoints: []int{10847}, Characters: []byte{0xe2, 0xa9, 0x9f}},
	"wedge":                           {Name: "wedge", CodePoints: []int{8743}, Characters: []byte{0xe2, 0x88, 0xa7}},
	"wedgeq":                          {Name: "wedgeq", CodePoints: []int{8793}, Characters: []byte{0xe2, 0x89, 0x99}},
	"weierp":                          {Name: "weierp", CodePoints: []int{8472}, Characters: []byte{0xe2, 0x84, 0x98}},
	"wfr":                             {Name: "wfr", CodePoints: []int{120116}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb4}},
	"wopf":                            {Name: "wopf", CodePoints: []int{120168}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa8}},
	"wp":                              {Name: "wp", CodePoints: []int{8472}, Characters: []byte{0xe2, 0x84, 0x98}},
	"wr":                              {Name: "wr", CodePoints: []int{8768}, Characters: []byte{0xe2, 0x89, 0x80}},
	"wreath":                          {Name: "wreath", CodePoints: []int{8768}, Characters: []byte{0xe2, 0x89, 0x80}},
	"wscr":                            {Name: "wscr", CodePoints: []int{120012}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8c}},
	"xcap":                            {Name: "xcap", CodePoints: []int{8898}, Characters: []byte{0xe2, 0x8b, 0x82}},
	"xcirc":                           {Name: "xcirc", CodePoints: []int{9711}, Characters: []byte{0xe2, 0x97, 0xaf}},
	"xcup":                            {Name: "xcup", CodePoints: []int{8899}, Characters: []byte{0xe2, 0x8b, 0x83}},
	"xdtri":                           {Name: "xdtri", CodePoints: []int{9661}, Characters: []byte{0xe2, 0x96, 0xbd}},
	"xfr":                             {Name: "xfr", CodePoints: []int{120117}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb5}},
	"xhArr":                           {Name: "xhArr", CodePoints: []int{10234}, Characters: []byte{0xe2, 0x9f, 0xba}},
	"xharr":                           {Name: "xharr", CodePoints: []int{10231}, Characters: []byte{0xe2, 0x9f, 0xb7}},
	"xi":                              {Name: "xi", CodePoints: []int{958}, Characters: []byte{0xce, 0xbe}},
	"xlArr":                           {Name: "xlArr", CodePoints: []int{10232}, Characters: []byte{0xe2, 0x9f, 0xb8}},
	"xlarr":                           {Name: "xlarr", CodePoints: []int{10229}, Characters: []byte{0xe2, 0x9f, 0xb5}},
	"xmap":                            {Name: "xmap", CodePoints: []int{10236}, Characters: []byte{0xe2, 0x9f, 0xbc}},
	"xnis":                            {Name: "xnis", CodePoints: []int{8955}, Characters: []byte{0xe2, 0x8b, 0xbb}},
	"xodot":                           {Name: "xodot", CodePoints: []int{10752}, Characters: []byte{0xe2, 0xa8, 0x80}},
	"xopf":                            {Name: "xopf", CodePoints: []int{120169}, Characters: []byte{0xf0, 0x9d, 0x95, 0xa9}},
	"xoplus":                          {Name: "xoplus", CodePoints: []int{10753}, Characters: []byte{0xe2, 0xa8, 0x81}},
	"xotime":                          {Name: "xotime", CodePoints: []int{10754}, Characters: []byte{0xe2, 0xa8, 0x82}},
	"xrArr":                           {Name: "xrArr", CodePoints: []int{10233}, Characters: []byte{0xe2, 0x9f, 0xb9}},
	"xrarr":                           {Name: "xrarr", CodePoints: []int{10230}, Characters: []byte{0xe2, 0x9f, 0xb6}},
	"xscr":                            {Name: "xscr", CodePoints: []int{120013}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8d}},
	"xsqcup":                          {Name: "xsqcup", CodePoints: []int{10758}, Characters: []byte{0xe2, 0xa8, 0x86}},
	"xuplus":                          {Name: "xuplus", CodePoints: []int{10756}, Characters: []byte{0xe2, 0xa8, 0x84}},
	"xutri":                           {Name: "xutri", CodePoints: []int{9651}, Characters: []byte{0xe2, 0x96, 0xb3}},
	"xvee":                            {Name: "xvee", CodePoints: []int{8897}, Characters: []byte{0xe2, 0x8b, 0x81}},
	"xwedge":                          {Name: "xwedge", CodePoints: []int{8896}, Characters: []byte{0xe2, 0x8b, 0x80}},
	"yacute":                          {Name: "yacute", CodePoints: []int{253}, Characters: []byte{0xc3, 0xbd}},
	"yacy":                            {Name: "yacy", CodePoints: []int{1103}, Characters: []byte{0xd1, 0x8f}},
	"ycirc":                           {Name: "ycirc", CodePoints: []int{375}, Characters: []byte{0xc5, 0xb7}},
	"ycy":                             {Name: "ycy", CodePoints: []int{1099}, Characters: []byte{0xd1, 0x8b}},
	"yen":                             {Name: "yen", CodePoints: []int{165}, Characters: []byte{0xc2, 0xa5}},
	"yfr":                             {Name: "yfr", CodePoints: []int{120118}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb6}},
	"yicy":                            {Name: "yicy", CodePoints: []int{1111}, Characters: []byte{0xd1, 0x97}},
	"yopf":                            {Name: "yopf", CodePoints: []int{120170}, Characters: []byte{0xf0, 0x9d, 0x95, 0xaa}},
	"yscr":                            {Name: "yscr", CodePoints: []int{120014}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8e}},
	"yucy":                            {Name: "yucy", CodePoints: []int{1102}, Characters: []byte{0xd1, 0x8e}},
	"yuml":                            {Name: "yuml", CodePoints: []int{255}, Characters: []byte{0xc3, 0xbf}},
	"zacute":                          {Name: "zacute", CodePoints: []int{378}, Characters: []byte{0xc5, 0xba}},
	"zcaron":                          {Name: "zcaron", CodePoints: []int{382}, Characters: []byte{0xc5, 0xbe}},
	"zcy":                             {Name: "zcy", CodePoints: []int{1079}, Characters: []byte{0xd0, 0xb7}},
	"zdot":                            {Name: "zdot", CodePoints: []int{380}, Characters: []byte{0xc5, 0xbc}},
	"zeetrf":                          {Name: "zeetrf", CodePoints: []int{8488}, Characters: []byte{0xe2, 0x84, 0xa8}},
	"zeta":                            {Name: "zeta", CodePoints: []int{950}, Characters: []byte{0xce, 0xb6}},
	"zfr":                             {Name: "zfr", CodePoints: []int{120119}, Characters: []byte{0xf0, 0x9d, 0x94, 0xb7}},
	"zhcy":                            {Name: "zhcy", CodePoints: []int{1078}, Characters: []byte{0xd0, 0xb6}},
	"zigrarr":                         {Name: "zigrarr", CodePoints: []int{8669}, Characters: []byte{0xe2, 0x87, 0x9d}},
	"zopf":                            {Name: "zopf", CodePoints: []int{120171}, Characters: []byte{0xf0, 0x9d, 0x95, 0xab}},
	"zscr":                            {Name: "zscr", CodePoints: []int{120015}, Characters: []byte{0xf0, 0x9d, 0x93, 0x8f}},
	"zwj":                             {Name: "zwj", CodePoints: []int{8205}, Characters: []byte{0xe2, 0x80, 0x8d}},
	"zwnj":                            {Name: "zwnj", CodePoints: []int{8204}, Characters: []byte{0xe2, 0x80, 0x8c}},
}
