/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.time.Clock;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefBuilder;
import org.apache.lucene.util.UnicodeUtil;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.bulk.BackoffPolicy;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexAction;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.support.master.AcknowledgedRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateRequestBuilder;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.InputStreamStreamInput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.util.iterable.Iterables;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.core.internal.io.Streams;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.IndexClosedException;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.SearchService;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.security.ScrollHelper;
import org.elasticsearch.xpack.core.security.SecurityContext;
import org.elasticsearch.xpack.core.security.authc.Authentication;
import org.elasticsearch.xpack.core.security.authc.KeyAndTimestamp;
import org.elasticsearch.xpack.core.security.authc.TokenMetadata;
import org.elasticsearch.xpack.core.security.authc.support.Hasher;
import org.elasticsearch.xpack.core.security.authc.support.TokensInvalidationResult;
import org.elasticsearch.xpack.security.Security;
import org.elasticsearch.xpack.security.authc.BytesKey;
import org.elasticsearch.xpack.security.authc.ExpiredTokenRemover;
import org.elasticsearch.xpack.security.authc.UserToken;
import org.elasticsearch.xpack.security.support.FeatureNotEnabledException;
import org.elasticsearch.xpack.security.support.SecurityIndexManager;

public final class TokenService {
    static final int TOKEN_SERVICE_KEY_ITERATIONS = 100000;
    static final int TOKENS_ENCRYPTION_KEY_ITERATIONS = 1024;
    private static final String KDF_ALGORITHM = "PBKDF2withHMACSHA512";
    static final int SALT_BYTES = 32;
    private static final int KEY_BYTES = 64;
    static final int IV_BYTES = 12;
    private static final int VERSION_BYTES = 4;
    private static final String ENCRYPTION_CIPHER = "AES/GCM/NoPadding";
    private static final String EXPIRED_TOKEN_WWW_AUTH_VALUE = "Bearer realm=\"security\", error=\"invalid_token\", error_description=\"The access token expired\"";
    private static final String MALFORMED_TOKEN_WWW_AUTH_VALUE = "Bearer realm=\"security\", error=\"invalid_token\", error_description=\"The access token is malformed\"";
    private static final BackoffPolicy DEFAULT_BACKOFF = BackoffPolicy.exponentialBackoff();
    public static final String THREAD_POOL_NAME = "security-token-key";
    public static final Setting<TimeValue> TOKEN_EXPIRATION = Setting.timeSetting((String)"xpack.security.authc.token.timeout", (TimeValue)TimeValue.timeValueMinutes((long)20L), (TimeValue)TimeValue.timeValueSeconds((long)1L), (TimeValue)TimeValue.timeValueHours((long)1L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_INTERVAL = Setting.timeSetting((String)"xpack.security.authc.token.delete.interval", (TimeValue)TimeValue.timeValueMinutes((long)30L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_TIMEOUT = Setting.timeSetting((String)"xpack.security.authc.token.delete.timeout", (TimeValue)TimeValue.MINUS_ONE, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    static final String TOKEN_DOC_TYPE = "token";
    private static final int HASHED_TOKEN_LENGTH = 43;
    private static final int TOKEN_LENGTH = 22;
    private static final String TOKEN_DOC_ID_PREFIX = "token_";
    static final int LEGACY_MINIMUM_BYTES = 49;
    static final int MINIMUM_BYTES = 27;
    static final int LEGACY_MINIMUM_BASE64_BYTES = Double.valueOf(Math.ceil(65.0)).intValue();
    public static final int MINIMUM_BASE64_BYTES = Double.valueOf(Math.ceil(36.0)).intValue();
    static final Version VERSION_HASHED_TOKENS = Version.V_7_2_0;
    static final Version VERSION_TOKENS_INDEX_INTRODUCED = Version.V_7_2_0;
    static final Version VERSION_ACCESS_TOKENS_AS_UUIDS = Version.V_7_2_0;
    static final Version VERSION_MULTIPLE_CONCURRENT_REFRESHES = Version.V_7_2_0;
    private static final Logger logger = LogManager.getLogger(TokenService.class);
    private final SecureRandom secureRandom = new SecureRandom();
    private final Settings settings;
    private final ClusterService clusterService;
    private final Clock clock;
    private final TimeValue expirationDelay;
    private final TimeValue deleteInterval;
    private final Client client;
    private final SecurityIndexManager securityMainIndex;
    private final SecurityIndexManager securityTokensIndex;
    private final ExpiredTokenRemover expiredTokenRemover;
    private final boolean enabled;
    private final XPackLicenseState licenseState;
    private final SecurityContext securityContext;
    private volatile TokenKeys keyCache;
    private volatile long lastExpirationRunMs;
    private final AtomicLong createdTimeStamps = new AtomicLong(-1L);
    private final AtomicBoolean installTokenMetadataInProgress = new AtomicBoolean(false);

    public TokenService(Settings settings, Clock clock, Client client, XPackLicenseState licenseState, SecurityContext securityContext, SecurityIndexManager securityMainIndex, SecurityIndexManager securityTokensIndex, ClusterService clusterService) throws GeneralSecurityException {
        byte[] saltArr = new byte[32];
        this.secureRandom.nextBytes(saltArr);
        SecureString tokenPassphrase = this.generateTokenKey();
        this.settings = settings;
        this.clock = clock.withZone(ZoneOffset.UTC);
        this.expirationDelay = (TimeValue)TOKEN_EXPIRATION.get(settings);
        this.client = client;
        this.licenseState = licenseState;
        this.securityContext = securityContext;
        this.securityMainIndex = securityMainIndex;
        this.securityTokensIndex = securityTokensIndex;
        this.lastExpirationRunMs = client.threadPool().relativeTimeInMillis();
        this.deleteInterval = (TimeValue)DELETE_INTERVAL.get(settings);
        this.enabled = TokenService.isTokenServiceEnabled(settings);
        this.expiredTokenRemover = new ExpiredTokenRemover(settings, client, securityMainIndex, securityTokensIndex);
        this.ensureEncryptionCiphersSupported();
        KeyAndCache keyAndCache = new KeyAndCache(new KeyAndTimestamp(tokenPassphrase, this.createdTimeStamps.incrementAndGet()), new BytesKey(saltArr));
        this.keyCache = new TokenKeys(Collections.singletonMap(keyAndCache.getKeyHash(), keyAndCache), keyAndCache.getKeyHash());
        this.clusterService = clusterService;
        this.initialize(clusterService);
        this.getTokenMetadata();
    }

    public void createOAuth2Tokens(Authentication authentication, Authentication originatingClientAuth, Map<String, Object> metadata, boolean includeRefreshToken, ActionListener<CreateTokenResult> listener) {
        Version tokenVersion = this.getTokenVersionCompatibility();
        SecurityIndexManager tokensIndex = this.getTokensIndexForVersion(tokenVersion);
        String accessToken = UUIDs.randomBase64UUID();
        String refreshToken = includeRefreshToken ? UUIDs.randomBase64UUID() : null;
        this.createOAuth2Tokens(accessToken, refreshToken, tokenVersion, tokensIndex, authentication, originatingClientAuth, metadata, listener);
    }

    public void createOAuth2Tokens(String accessToken, String refreshToken, Authentication authentication, Authentication originatingClientAuth, Map<String, Object> metadata, ActionListener<CreateTokenResult> listener) {
        Version tokenVersion = this.getTokenVersionCompatibility();
        SecurityIndexManager tokensIndex = this.getTokensIndexForVersion(tokenVersion);
        this.createOAuth2Tokens(accessToken, refreshToken, tokenVersion, tokensIndex, authentication, originatingClientAuth, metadata, listener);
    }

    private void createOAuth2Tokens(String accessToken, String refreshToken, Version tokenVersion, SecurityIndexManager tokensIndex, Authentication authentication, Authentication originatingClientAuth, Map<String, Object> metadata, ActionListener<CreateTokenResult> listener) {
        assert (accessToken.length() == 22) : "We assume token ids have a fixed length for nodes of a certain version. When changing the token length, be careful that the inferences about its length still hold.";
        this.ensureEnabled();
        if (authentication == null) {
            listener.onFailure((Exception)this.traceLog("create token", new IllegalArgumentException("authentication must be provided")));
        } else if (originatingClientAuth == null) {
            listener.onFailure((Exception)this.traceLog("create token", new IllegalArgumentException("originating client authentication must be provided")));
        } else {
            String storedRefreshToken;
            String storedAccessToken;
            Authentication tokenAuth = new Authentication(authentication.getUser(), authentication.getAuthenticatedBy(), authentication.getLookedUpBy(), tokenVersion, Authentication.AuthenticationType.TOKEN, authentication.getMetadata());
            if (tokenVersion.onOrAfter(VERSION_HASHED_TOKENS)) {
                storedAccessToken = TokenService.hashTokenString(accessToken);
                storedRefreshToken = null == refreshToken ? null : TokenService.hashTokenString(refreshToken);
            } else {
                storedAccessToken = accessToken;
                storedRefreshToken = refreshToken;
            }
            UserToken userToken = new UserToken(storedAccessToken, tokenVersion, tokenAuth, this.getExpirationTime(), metadata);
            BytesReference tokenDocument = this.createTokenDocument(userToken, storedRefreshToken, originatingClientAuth);
            String documentId = TokenService.getTokenDocumentId(storedAccessToken);
            IndexRequest indexTokenRequest = (IndexRequest)((IndexRequestBuilder)this.client.prepareIndex(tokensIndex.aliasName(), "_doc", documentId).setOpType(DocWriteRequest.OpType.CREATE).setSource(tokenDocument, XContentType.JSON).setRefreshPolicy(WriteRequest.RefreshPolicy.WAIT_UNTIL)).request();
            tokensIndex.prepareIndexIfNeededThenExecute(ex -> listener.onFailure(this.traceLog("prepare tokens index [" + tokensIndex.aliasName() + "]", documentId, ex)), () -> ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"security", (ActionType)IndexAction.INSTANCE, (ActionRequest)indexTokenRequest, (ActionListener)ActionListener.wrap(indexResponse -> {
                if (indexResponse.getResult() == DocWriteResponse.Result.CREATED) {
                    String versionedAccessToken = this.prependVersionAndEncodeAccessToken(tokenVersion, accessToken);
                    if (tokenVersion.onOrAfter(VERSION_TOKENS_INDEX_INTRODUCED)) {
                        String versionedRefreshToken = refreshToken != null ? TokenService.prependVersionAndEncodeRefreshToken(tokenVersion, refreshToken) : null;
                        listener.onResponse((Object)new CreateTokenResult(versionedAccessToken, versionedRefreshToken, authentication));
                    } else {
                        listener.onResponse((Object)new CreateTokenResult(versionedAccessToken, refreshToken, authentication));
                    }
                } else {
                    listener.onFailure((Exception)((Object)this.traceLog("create token", new ElasticsearchException("failed to create token document [{}]", new Object[]{indexResponse}))));
                }
            }, arg_0 -> ((ActionListener)listener).onFailure(arg_0))));
        }
    }

    public static String hashTokenString(String accessTokenString) {
        return new String(Hasher.SHA256.hash(new SecureString(accessTokenString.toCharArray())));
    }

    void tryAuthenticateToken(SecureString token, ActionListener<UserToken> listener) {
        if (this.isEnabled() && token != null) {
            this.decodeAndValidateToken(token, listener);
        } else {
            listener.onResponse(null);
        }
    }

    public void authenticateToken(SecureString tokenString, ActionListener<Authentication> listener) {
        this.decodeAndValidateToken(tokenString, (ActionListener<UserToken>)listener.map(token -> {
            if (token == null) {
                throw new IllegalArgumentException("Cannot validate access token");
            }
            return token.getAuthentication();
        }));
    }

    public void getAuthenticationAndMetadata(String token, ActionListener<Tuple<Authentication, Map<String, Object>>> listener) {
        this.decodeToken(token, (ActionListener<UserToken>)ActionListener.wrap(userToken -> {
            if (userToken == null) {
                listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("supplied token is not valid", new Object[0])));
            } else {
                listener.onResponse((Object)new Tuple((Object)userToken.getAuthentication(), userToken.getMetadata()));
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    private void getUserTokenFromId(String userTokenId, Version tokenVersion, ActionListener<UserToken> listener) {
        SecurityIndexManager tokensIndex = this.getTokensIndexForVersion(tokenVersion);
        SecurityIndexManager frozenTokensIndex = tokensIndex.freeze();
        if (!frozenTokensIndex.isAvailable()) {
            logger.warn("failed to get access token [{}] because index [{}] is not available", (Object)userTokenId, (Object)tokensIndex.aliasName());
            listener.onFailure((Exception)((Object)frozenTokensIndex.getUnavailableReason()));
        } else {
            GetRequest getRequest = (GetRequest)this.client.prepareGet(tokensIndex.aliasName(), "_doc", TokenService.getTokenDocumentId(userTokenId)).request();
            Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("get token from id", userTokenId, ex));
            tokensIndex.checkIndexVersionThenExecute(ex -> listener.onFailure(this.traceLog("prepare tokens index [" + tokensIndex.aliasName() + "]", userTokenId, ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)getRequest, (ActionListener)ActionListener.wrap(response -> {
                if (response.isExists()) {
                    Map accessTokenSource = (Map)response.getSource().get("access_token");
                    if (accessTokenSource == null) {
                        onFailure.accept(new IllegalStateException("token document is missing the access_token field"));
                    } else if (!accessTokenSource.containsKey("user_token")) {
                        onFailure.accept(new IllegalStateException("token document is missing the user_token field"));
                    } else {
                        Map userTokenSource = (Map)accessTokenSource.get("user_token");
                        listener.onResponse((Object)UserToken.fromSourceMap(userTokenSource));
                    }
                } else {
                    logger.trace("The access token [{}] is expired and already deleted", (Object)userTokenId);
                    listener.onResponse(null);
                }
            }, e -> {
                if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                    logger.warn("failed to get access token [{}] because index [{}] is not available", (Object)userTokenId, (Object)tokensIndex.aliasName());
                } else {
                    logger.error((Message)new ParameterizedMessage("failed to get access token [{}]", (Object)userTokenId), (Throwable)e);
                }
                listener.onFailure(e);
            }), (arg_0, arg_1) -> ((Client)this.client).get(arg_0, arg_1)));
        }
    }

    private void decodeAndValidateToken(SecureString tokenString, ActionListener<UserToken> listener) {
        this.ensureEnabled();
        this.decodeToken(tokenString.toString(), (ActionListener<UserToken>)ActionListener.wrap(userToken -> {
            if (userToken != null) {
                this.checkIfTokenIsValid((UserToken)userToken, listener);
            } else {
                listener.onResponse(null);
            }
        }, e -> {
            if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                listener.onResponse(null);
            } else {
                listener.onFailure(e);
            }
        }));
    }

    void decodeToken(String token, ActionListener<UserToken> listener) {
        byte[] bytes = token.getBytes(StandardCharsets.UTF_8);
        try (InputStreamStreamInput in = new InputStreamStreamInput(Base64.getDecoder().wrap(new ByteArrayInputStream(bytes)), (long)bytes.length);){
            Version version = Version.readVersion((StreamInput)in);
            in.setVersion(version);
            if (version.onOrAfter(VERSION_ACCESS_TOKENS_AS_UUIDS)) {
                if (in.available() < 27) {
                    logger.debug("invalid token, smaller than [{}] bytes", (Object)27);
                    listener.onResponse(null);
                    return;
                }
                String accessToken = in.readString();
                if (version.onOrAfter(VERSION_HASHED_TOKENS)) {
                    String userTokenId = TokenService.hashTokenString(accessToken);
                    this.getUserTokenFromId(userTokenId, version, listener);
                } else {
                    this.getUserTokenFromId(accessToken, version, listener);
                }
            } else {
                if (in.available() < 49) {
                    logger.debug("invalid token, smaller than [{}] bytes", (Object)49);
                    listener.onResponse(null);
                    return;
                }
                BytesKey decodedSalt = new BytesKey(in.readByteArray());
                BytesKey passphraseHash = new BytesKey(in.readByteArray());
                byte[] iv = in.readByteArray();
                BytesStreamOutput out = new BytesStreamOutput();
                Streams.copy((InputStream)in, (OutputStream)out);
                byte[] encryptedTokenId = BytesReference.toBytes((BytesReference)out.bytes());
                KeyAndCache keyAndCache = this.keyCache.get(passphraseHash);
                if (keyAndCache != null) {
                    this.getKeyAsync(decodedSalt, keyAndCache, (ActionListener<SecretKey>)ActionListener.wrap(decodeKey -> {
                        if (decodeKey != null) {
                            try {
                                Cipher cipher = this.getDecryptionCipher(iv, (SecretKey)decodeKey, version, decodedSalt);
                                String tokenId = TokenService.decryptTokenId(encryptedTokenId, cipher, version);
                                this.getUserTokenFromId(tokenId, version, listener);
                            }
                            catch (IOException | GeneralSecurityException e) {
                                logger.warn("invalid token", (Throwable)e);
                                listener.onResponse(null);
                            }
                        } else {
                            listener.onResponse(null);
                            return;
                        }
                    }, arg_0 -> listener.onFailure(arg_0)));
                } else {
                    logger.debug(() -> new ParameterizedMessage("invalid key {} key: {}", (Object)passphraseHash, this.keyCache.cache.keySet()));
                    listener.onResponse(null);
                }
            }
        }
        catch (Exception e) {
            if (logger.isDebugEnabled()) {
                logger.debug("built in token service unable to decode token", (Throwable)e);
            } else {
                logger.warn("built in token service unable to decode token");
            }
            listener.onResponse(null);
        }
    }

    public void invalidateAccessToken(String accessToken, ActionListener<TokensInvalidationResult> listener) {
        this.ensureEnabled();
        if (Strings.isNullOrEmpty((String)accessToken)) {
            listener.onFailure((Exception)this.traceLog("invalidate access token", new IllegalArgumentException("access token must be provided")));
        } else {
            this.maybeStartTokenRemover();
            Iterator backoff = DEFAULT_BACKOFF.iterator();
            this.decodeToken(accessToken, (ActionListener<UserToken>)ActionListener.wrap(userToken -> {
                if (userToken == null) {
                    logger.trace("The access token [{}] is expired and already deleted", (Object)accessToken);
                    listener.onResponse((Object)TokensInvalidationResult.emptyResult((RestStatus)RestStatus.NOT_FOUND));
                } else {
                    this.indexInvalidation(Collections.singleton(userToken), backoff, "access_token", null, listener);
                }
            }, e -> {
                if (e instanceof IndexNotFoundException || e instanceof IndexClosedException) {
                    listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("failed to invalidate token", RestStatus.BAD_REQUEST, new Object[0])));
                } else {
                    listener.onFailure((Exception)((Object)TokenService.unableToPerformAction(e)));
                }
            }));
        }
    }

    public void invalidateAccessToken(UserToken userToken, ActionListener<TokensInvalidationResult> listener) {
        this.ensureEnabled();
        if (userToken == null) {
            logger.trace("No access token provided");
            listener.onFailure((Exception)new IllegalArgumentException("access token must be provided"));
        } else {
            this.maybeStartTokenRemover();
            Iterator backoff = DEFAULT_BACKOFF.iterator();
            this.indexInvalidation(Collections.singleton(userToken), backoff, "access_token", null, listener);
        }
    }

    public void invalidateRefreshToken(String refreshToken, ActionListener<TokensInvalidationResult> listener) {
        this.ensureEnabled();
        if (Strings.isNullOrEmpty((String)refreshToken)) {
            logger.trace("No refresh token provided");
            listener.onFailure((Exception)new IllegalArgumentException("refresh token must be provided"));
        } else {
            this.maybeStartTokenRemover();
            Iterator backoff = DEFAULT_BACKOFF.iterator();
            this.findTokenFromRefreshToken(refreshToken, backoff, (ActionListener<SearchHits>)ActionListener.wrap(searchHits -> {
                if (searchHits.getHits().length < 1) {
                    logger.debug("could not find token document for refresh token");
                    listener.onResponse((Object)TokensInvalidationResult.emptyResult((RestStatus)RestStatus.NOT_FOUND));
                } else if (searchHits.getHits().length > 1) {
                    listener.onFailure((Exception)new IllegalStateException("multiple tokens share the same refresh token"));
                } else {
                    Tuple<UserToken, RefreshTokenStatus> parsedTokens = TokenService.parseTokenAndRefreshStatus(searchHits.getAt(0).getSourceAsMap());
                    UserToken userToken = (UserToken)parsedTokens.v1();
                    RefreshTokenStatus refresh = (RefreshTokenStatus)parsedTokens.v2();
                    if (refresh.isInvalidated()) {
                        listener.onResponse((Object)new TokensInvalidationResult(Collections.emptyList(), Collections.singletonList(userToken.getId()), null, RestStatus.OK));
                    } else {
                        this.indexInvalidation(Collections.singletonList(userToken), backoff, "refresh_token", null, listener);
                    }
                }
            }, e -> {
                if (e instanceof IndexNotFoundException || e instanceof IndexClosedException) {
                    listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("failed to invalidate token", RestStatus.BAD_REQUEST, new Object[0])));
                } else {
                    listener.onFailure((Exception)((Object)TokenService.unableToPerformAction(e)));
                }
            }));
        }
    }

    public void invalidateActiveTokensForRealmAndUser(@Nullable String realmName, @Nullable String username, ActionListener<TokensInvalidationResult> listener) {
        this.ensureEnabled();
        if (Strings.isNullOrEmpty((String)realmName) && Strings.isNullOrEmpty((String)username)) {
            logger.trace("No realm name or username provided");
            listener.onFailure((Exception)new IllegalArgumentException("realm name or username must be provided"));
        } else if (Strings.isNullOrEmpty((String)realmName)) {
            this.findActiveTokensForUser(username, (ActionListener<Collection<Tuple<UserToken, String>>>)ActionListener.wrap(tokenTuples -> {
                if (tokenTuples.isEmpty()) {
                    logger.warn("No tokens to invalidate for realm [{}] and username [{}]", (Object)realmName, (Object)username);
                    listener.onResponse((Object)TokensInvalidationResult.emptyResult((RestStatus)RestStatus.OK));
                } else {
                    this.invalidateAllTokens(tokenTuples.stream().map(t -> (UserToken)t.v1()).collect(Collectors.toList()), listener);
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        } else {
            Predicate<Map<String, Object>> filter = null;
            if (Strings.hasText((String)username)) {
                filter = TokenService.isOfUser(username);
            }
            this.findActiveTokensForRealm(realmName, filter, (ActionListener<Collection<Tuple<UserToken, String>>>)ActionListener.wrap(tokenTuples -> {
                if (tokenTuples.isEmpty()) {
                    logger.warn("No tokens to invalidate for realm [{}] and username [{}]", (Object)realmName, (Object)username);
                    listener.onResponse((Object)TokensInvalidationResult.emptyResult((RestStatus)RestStatus.OK));
                } else {
                    this.invalidateAllTokens(tokenTuples.stream().map(t -> (UserToken)t.v1()).collect(Collectors.toList()), listener);
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        }
    }

    private void invalidateAllTokens(Collection<UserToken> userTokens, ActionListener<TokensInvalidationResult> listener) {
        this.maybeStartTokenRemover();
        Iterator backoff = DEFAULT_BACKOFF.iterator();
        this.indexInvalidation(userTokens, backoff, "refresh_token", null, (ActionListener<TokensInvalidationResult>)ActionListener.wrap(result -> this.indexInvalidation(userTokens, backoff, "access_token", (TokensInvalidationResult)result, listener), arg_0 -> listener.onFailure(arg_0)));
    }

    private void indexInvalidation(Collection<UserToken> userTokens, Iterator<TimeValue> backoff, String srcPrefix, @Nullable TokensInvalidationResult previousResult, ActionListener<TokensInvalidationResult> listener) {
        HashSet<String> idsOfRecentTokens = new HashSet<String>();
        HashSet<String> idsOfOlderTokens = new HashSet<String>();
        for (UserToken userToken : userTokens) {
            if (userToken.getVersion().onOrAfter(VERSION_TOKENS_INDEX_INTRODUCED)) {
                idsOfRecentTokens.add(userToken.getId());
                continue;
            }
            idsOfOlderTokens.add(userToken.getId());
        }
        if (!idsOfOlderTokens.isEmpty()) {
            this.indexInvalidation(idsOfOlderTokens, this.securityMainIndex, backoff, srcPrefix, previousResult, (ActionListener<TokensInvalidationResult>)ActionListener.wrap(newResult -> {
                if (!idsOfRecentTokens.isEmpty()) {
                    this.indexInvalidation((Collection<String>)idsOfRecentTokens, this.securityTokensIndex, backoff, srcPrefix, (TokensInvalidationResult)newResult, listener);
                } else {
                    listener.onResponse(newResult);
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        } else {
            this.indexInvalidation(idsOfRecentTokens, this.securityTokensIndex, backoff, srcPrefix, previousResult, listener);
        }
    }

    private void indexInvalidation(Collection<String> tokenIds, SecurityIndexManager tokensIndexManager, Iterator<TimeValue> backoff, String srcPrefix, @Nullable TokensInvalidationResult previousResult, ActionListener<TokensInvalidationResult> listener) {
        if (tokenIds.isEmpty()) {
            logger.warn("No [{}] tokens provided for invalidation", (Object)srcPrefix);
            listener.onFailure((Exception)((Object)TokenService.invalidGrantException("No tokens provided for invalidation")));
        } else {
            BulkRequestBuilder bulkRequestBuilder = this.client.prepareBulk();
            for (String tokenId : tokenIds) {
                UpdateRequest request = (UpdateRequest)this.client.prepareUpdate(tokensIndexManager.aliasName(), "_doc", TokenService.getTokenDocumentId(tokenId)).setDoc(new Object[]{srcPrefix, Collections.singletonMap("invalidated", true)}).setFetchSource(srcPrefix, null).request();
                bulkRequestBuilder.add(request);
            }
            bulkRequestBuilder.setRefreshPolicy(WriteRequest.RefreshPolicy.WAIT_UNTIL);
            tokensIndexManager.prepareIndexIfNeededThenExecute(ex -> listener.onFailure(this.traceLog("prepare index [" + tokensIndexManager.aliasName() + "]", ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)((BulkRequest)bulkRequestBuilder.request()), (ActionListener)ActionListener.wrap(bulkResponse -> {
                ArrayList<String> retryTokenDocIds = new ArrayList<String>();
                ArrayList<ElasticsearchException> failedRequestResponses = new ArrayList<ElasticsearchException>();
                ArrayList<String> previouslyInvalidated = new ArrayList<String>();
                ArrayList<String> invalidated = new ArrayList<String>();
                if (null != previousResult) {
                    failedRequestResponses.addAll(previousResult.getErrors());
                    previouslyInvalidated.addAll(previousResult.getPreviouslyInvalidatedTokens());
                    invalidated.addAll(previousResult.getInvalidatedTokens());
                }
                for (BulkItemResponse bulkItemResponse : bulkResponse.getItems()) {
                    if (bulkItemResponse.isFailed()) {
                        Exception cause = bulkItemResponse.getFailure().getCause();
                        String failedTokenDocId = TokenService.getTokenIdFromDocumentId(bulkItemResponse.getFailure().getId());
                        if (TransportActions.isShardNotAvailableException((Throwable)cause)) {
                            retryTokenDocIds.add(failedTokenDocId);
                            continue;
                        }
                        this.traceLog("invalidate access token", failedTokenDocId, cause);
                        failedRequestResponses.add(new ElasticsearchException("Error invalidating " + srcPrefix + ": ", (Throwable)cause, new Object[0]));
                        continue;
                    }
                    UpdateResponse updateResponse = (UpdateResponse)bulkItemResponse.getResponse();
                    if (updateResponse.getResult() == DocWriteResponse.Result.UPDATED) {
                        logger.debug(() -> new ParameterizedMessage("Invalidated [{}] for doc [{}]", (Object)srcPrefix, (Object)updateResponse.getGetResult().getId()));
                        invalidated.add(updateResponse.getGetResult().getId());
                        continue;
                    }
                    if (updateResponse.getResult() != DocWriteResponse.Result.NOOP) continue;
                    previouslyInvalidated.add(updateResponse.getGetResult().getId());
                }
                if (!retryTokenDocIds.isEmpty() && backoff.hasNext()) {
                    logger.debug("failed to invalidate [{}] tokens out of [{}], retrying to invalidate these too", (Object)retryTokenDocIds.size(), (Object)tokenIds.size());
                    TokensInvalidationResult incompleteResult = new TokensInvalidationResult(invalidated, previouslyInvalidated, failedRequestResponses, RestStatus.OK);
                    this.client.threadPool().schedule(() -> this.indexInvalidation(retryTokenDocIds, tokensIndexManager, backoff, srcPrefix, incompleteResult, listener), (TimeValue)backoff.next(), "generic");
                } else {
                    if (!retryTokenDocIds.isEmpty()) {
                        logger.warn("failed to invalidate [{}] tokens out of [{}] after all retries", (Object)retryTokenDocIds.size(), (Object)tokenIds.size());
                        for (String retryTokenDocId : retryTokenDocIds) {
                            failedRequestResponses.add(new ElasticsearchException("Error invalidating [{}] with doc id [{}] after retries exhausted", new Object[]{srcPrefix, retryTokenDocId}));
                        }
                    }
                    TokensInvalidationResult result = new TokensInvalidationResult(invalidated, previouslyInvalidated, failedRequestResponses, RestStatus.OK);
                    listener.onResponse((Object)result);
                }
            }, e -> {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                this.traceLog("invalidate tokens", cause);
                if (TransportActions.isShardNotAvailableException((Throwable)cause) && backoff.hasNext()) {
                    logger.debug("failed to invalidate tokens, retrying ");
                    this.client.threadPool().schedule(() -> this.indexInvalidation(tokenIds, tokensIndexManager, backoff, srcPrefix, previousResult, listener), (TimeValue)backoff.next(), "generic");
                } else {
                    listener.onFailure(e);
                }
            }), (arg_0, arg_1) -> ((Client)this.client).bulk(arg_0, arg_1)));
        }
    }

    public void refreshToken(String refreshToken, ActionListener<CreateTokenResult> listener) {
        this.ensureEnabled();
        Instant refreshRequested = this.clock.instant();
        Iterator backoff = DEFAULT_BACKOFF.iterator();
        Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("find token by refresh token", refreshToken, ex));
        this.findTokenFromRefreshToken(refreshToken, backoff, (ActionListener<SearchHits>)ActionListener.wrap(searchHits -> {
            if (searchHits.getHits().length < 1) {
                logger.warn("could not find token document for refresh token");
                onFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
            } else if (searchHits.getHits().length > 1) {
                onFailure.accept(new IllegalStateException("multiple tokens share the same refresh token"));
            } else {
                SearchHit tokenDocHit = searchHits.getAt(0);
                Authentication clientAuth = this.securityContext.getAuthentication();
                this.innerRefresh(refreshToken, tokenDocHit.getId(), tokenDocHit.getSourceAsMap(), tokenDocHit.getSeqNo(), tokenDocHit.getPrimaryTerm(), clientAuth, backoff, refreshRequested, listener);
            }
        }, e -> listener.onFailure((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")))));
    }

    private void findTokenFromRefreshToken(String refreshToken, Iterator<TimeValue> backoff, ActionListener<SearchHits> listener) {
        if (refreshToken.length() == 22) {
            logger.debug("Assuming an unversioned refresh token [{}], generated for node versions prior to the introduction of the version-header format.", (Object)refreshToken);
            this.findTokenFromRefreshToken(refreshToken, this.securityMainIndex, backoff, listener);
        } else if (refreshToken.length() == 43) {
            logger.debug("Assuming a hashed refresh token [{}] retrieved from the tokens index", (Object)refreshToken);
            this.findTokenFromRefreshToken(refreshToken, this.securityTokensIndex, backoff, listener);
        } else {
            String unencodedRefreshToken;
            Version refreshTokenVersion;
            logger.debug("Assuming a refresh token [{}] provided from a client", (Object)refreshToken);
            try {
                Tuple<Version, String> versionAndRefreshTokenTuple = TokenService.unpackVersionAndPayload(refreshToken);
                refreshTokenVersion = (Version)versionAndRefreshTokenTuple.v1();
                unencodedRefreshToken = (String)versionAndRefreshTokenTuple.v2();
            }
            catch (IOException e) {
                logger.debug(() -> new ParameterizedMessage("Could not decode refresh token [{}].", (Object)refreshToken), (Throwable)e);
                listener.onResponse((Object)SearchHits.empty());
                return;
            }
            if (refreshTokenVersion.before(VERSION_TOKENS_INDEX_INTRODUCED) || unencodedRefreshToken.length() != 22) {
                logger.debug("Decoded refresh token [{}] with version [{}] is invalid.", (Object)unencodedRefreshToken, (Object)refreshTokenVersion);
                listener.onResponse((Object)SearchHits.empty());
            } else if (refreshTokenVersion.onOrAfter(VERSION_HASHED_TOKENS)) {
                String hashedRefreshToken = TokenService.hashTokenString(unencodedRefreshToken);
                this.findTokenFromRefreshToken(hashedRefreshToken, this.securityTokensIndex, backoff, listener);
            } else {
                this.findTokenFromRefreshToken(unencodedRefreshToken, this.securityTokensIndex, backoff, listener);
            }
        }
    }

    private void findTokenFromRefreshToken(String refreshToken, SecurityIndexManager tokensIndexManager, Iterator<TimeValue> backoff, ActionListener<SearchHits> listener) {
        Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("find token by refresh token", refreshToken, ex));
        Consumer<Exception> maybeRetryOnFailure = ex -> {
            if (backoff.hasNext()) {
                TimeValue backofTimeValue = (TimeValue)backoff.next();
                logger.debug("retrying after [{}] back off", (Object)backofTimeValue);
                this.client.threadPool().schedule(() -> this.findTokenFromRefreshToken(refreshToken, tokensIndexManager, backoff, listener), backofTimeValue, "generic");
            } else {
                logger.warn("failed to find token from refresh token after all retries");
                onFailure.accept((Exception)ex);
            }
        };
        SecurityIndexManager frozenTokensIndex = tokensIndexManager.freeze();
        if (!frozenTokensIndex.indexExists()) {
            logger.warn("index [{}] does not exist so we can't find token from refresh token", (Object)frozenTokensIndex.aliasName());
            listener.onFailure((Exception)((Object)frozenTokensIndex.getUnavailableReason()));
        } else if (!frozenTokensIndex.isAvailable()) {
            logger.debug("index [{}] is not available to find token from refresh token, retrying", (Object)frozenTokensIndex.aliasName());
            maybeRetryOnFailure.accept((Exception)((Object)frozenTokensIndex.getUnavailableReason()));
        } else {
            SearchRequest request = (SearchRequest)this.client.prepareSearch(new String[]{tokensIndexManager.aliasName()}).setQuery((QueryBuilder)QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)TOKEN_DOC_TYPE)).filter((QueryBuilder)QueryBuilders.termQuery((String)"refresh_token.token", (String)refreshToken))).seqNoAndPrimaryTerm(true).request();
            tokensIndexManager.checkIndexVersionThenExecute(arg_0 -> listener.onFailure(arg_0), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)request, (ActionListener)ActionListener.wrap(searchResponse -> {
                if (searchResponse.isTimedOut()) {
                    logger.debug("find token from refresh token response timed out, retrying");
                    maybeRetryOnFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                } else {
                    listener.onResponse((Object)searchResponse.getHits());
                }
            }, e -> {
                if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                    logger.debug("find token from refresh token request failed because of unavailable shards, retrying");
                    maybeRetryOnFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                } else {
                    onFailure.accept((Exception)e);
                }
            }), (arg_0, arg_1) -> ((Client)this.client).search(arg_0, arg_1)));
        }
    }

    private void innerRefresh(final String refreshToken, final String tokenDocId, Map<String, Object> source, long seqNo, long primaryTerm, final Authentication clientAuth, final Iterator<TimeValue> backoff, final Instant refreshRequested, final ActionListener<CreateTokenResult> listener) {
        Tuple<RefreshTokenStatus, Optional<ElasticsearchSecurityException>> checkRefreshResult;
        logger.debug("Attempting to refresh token stored in token document [{}]", (Object)tokenDocId);
        final Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("refresh token", tokenDocId, ex));
        try {
            checkRefreshResult = TokenService.checkTokenDocumentForRefresh(refreshRequested, clientAuth, source);
        }
        catch (IllegalStateException | DateTimeException e) {
            onFailure.accept((Exception)((Object)new ElasticsearchSecurityException("invalid token document", (Exception)e, new Object[0])));
            return;
        }
        if (((Optional)checkRefreshResult.v2()).isPresent()) {
            onFailure.accept((Exception)((Optional)checkRefreshResult.v2()).get());
            return;
        }
        RefreshTokenStatus refreshTokenStatus = (RefreshTokenStatus)checkRefreshResult.v1();
        final SecurityIndexManager refreshedTokenIndex = this.getTokensIndexForVersion(refreshTokenStatus.getVersion());
        if (refreshTokenStatus.isRefreshed()) {
            logger.debug("Token document [{}] was recently refreshed, when a new token document was generated. Reusing that result.", (Object)tokenDocId);
            Tuple<UserToken, String> parsedTokens = this.parseTokensFromDocument(source, null);
            Authentication authentication = ((UserToken)parsedTokens.v1()).getAuthentication();
            this.decryptAndReturnSupersedingTokens(refreshToken, refreshTokenStatus, refreshedTokenIndex, authentication, listener);
        } else {
            String newAccessTokenString = UUIDs.randomBase64UUID();
            String newRefreshTokenString = UUIDs.randomBase64UUID();
            Version newTokenVersion = this.getTokenVersionCompatibility();
            HashMap<String, Object> updateMap = new HashMap<String, Object>();
            updateMap.put("refreshed", true);
            if (newTokenVersion.onOrAfter(VERSION_MULTIPLE_CONCURRENT_REFRESHES)) {
                updateMap.put("refresh_time", this.clock.instant().toEpochMilli());
                try {
                    byte[] iv = this.getRandomBytes(12);
                    byte[] salt = this.getRandomBytes(32);
                    String encryptedAccessAndRefreshToken = this.encryptSupersedingTokens(newAccessTokenString, newRefreshTokenString, refreshToken, iv, salt);
                    updateMap.put("superseding.encrypted_tokens", encryptedAccessAndRefreshToken);
                    updateMap.put("superseding.encryption_iv", Base64.getEncoder().encodeToString(iv));
                    updateMap.put("superseding.encryption_salt", Base64.getEncoder().encodeToString(salt));
                }
                catch (GeneralSecurityException e) {
                    logger.warn("could not encrypt access token and refresh token string", (Throwable)e);
                    onFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                }
            }
            assert (seqNo != -2L) : "expected an assigned sequence number";
            assert (primaryTerm != 0L) : "expected an assigned primary term";
            UpdateRequestBuilder updateRequest = ((UpdateRequestBuilder)this.client.prepareUpdate(refreshedTokenIndex.aliasName(), "_doc", tokenDocId).setDoc(new Object[]{"refresh_token", updateMap}).setFetchSource(true).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE)).setIfSeqNo(seqNo).setIfPrimaryTerm(primaryTerm);
            refreshedTokenIndex.prepareIndexIfNeededThenExecute(ex -> listener.onFailure(this.traceLog("prepare index [" + refreshedTokenIndex.aliasName() + "]", ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)((UpdateRequest)updateRequest.request()), (ActionListener)ActionListener.wrap(updateResponse -> {
                if (updateResponse.getResult() == DocWriteResponse.Result.UPDATED) {
                    logger.debug(() -> new ParameterizedMessage("updated the original token document to {}", (Object)updateResponse.getGetResult().sourceAsMap()));
                    Tuple<UserToken, String> parsedTokens = this.parseTokensFromDocument(source, null);
                    UserToken toRefreshUserToken = (UserToken)parsedTokens.v1();
                    this.createOAuth2Tokens(newAccessTokenString, newRefreshTokenString, newTokenVersion, this.getTokensIndexForVersion(newTokenVersion), toRefreshUserToken.getAuthentication(), clientAuth, toRefreshUserToken.getMetadata(), listener);
                } else if (backoff.hasNext()) {
                    logger.info("failed to update the original token document [{}], the update result was [{}]. Retrying", (Object)tokenDocId, (Object)updateResponse.getResult());
                    this.client.threadPool().schedule(() -> this.innerRefresh(refreshToken, tokenDocId, source, seqNo, primaryTerm, clientAuth, backoff, refreshRequested, listener), (TimeValue)backoff.next(), "generic");
                } else {
                    logger.info("failed to update the original token document [{}] after all retries, the update result was [{}]. ", (Object)tokenDocId, (Object)updateResponse.getResult());
                    listener.onFailure((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                }
            }, e -> {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                if (cause instanceof VersionConflictEngineException) {
                    logger.debug("version conflict while updating document [{}], attempting to get it again", (Object)tokenDocId);
                    this.getTokenDocAsync(tokenDocId, refreshedTokenIndex, true, new ActionListener<GetResponse>(){

                        public void onResponse(GetResponse response) {
                            if (response.isExists()) {
                                TokenService.this.innerRefresh(refreshToken, tokenDocId, response.getSource(), response.getSeqNo(), response.getPrimaryTerm(), clientAuth, backoff, refreshRequested, (ActionListener<CreateTokenResult>)listener);
                            } else {
                                logger.warn("could not find token document [{}] for refresh", (Object)tokenDocId);
                                onFailure.accept(TokenService.invalidGrantException("could not refresh the requested token"));
                            }
                        }

                        public void onFailure(Exception e) {
                            if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                                if (backoff.hasNext()) {
                                    logger.info("could not get token document [{}] for refresh, retrying", (Object)tokenDocId);
                                    TokenService.this.client.threadPool().schedule(() -> TokenService.this.getTokenDocAsync(tokenDocId, refreshedTokenIndex, true, (ActionListener<GetResponse>)this), (TimeValue)backoff.next(), "generic");
                                } else {
                                    logger.warn("could not get token document [{}] for refresh after all retries", (Object)tokenDocId);
                                    onFailure.accept(TokenService.invalidGrantException("could not refresh the requested token"));
                                }
                            } else {
                                onFailure.accept(e);
                            }
                        }
                    });
                } else if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                    if (backoff.hasNext()) {
                        logger.debug("failed to update the original token document [{}], retrying", (Object)tokenDocId);
                        this.client.threadPool().schedule(() -> this.innerRefresh(refreshToken, tokenDocId, source, seqNo, primaryTerm, clientAuth, backoff, refreshRequested, listener), (TimeValue)backoff.next(), "generic");
                    } else {
                        logger.warn("failed to update the original token document [{}], after all retries", (Object)tokenDocId);
                        onFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                    }
                } else {
                    onFailure.accept((Exception)e);
                }
            }), (arg_0, arg_1) -> ((Client)this.client).update(arg_0, arg_1)));
        }
    }

    void decryptAndReturnSupersedingTokens(String refreshToken, final RefreshTokenStatus refreshTokenStatus, SecurityIndexManager tokensIndex, final Authentication authentication, final ActionListener<CreateTokenResult> listener) {
        byte[] iv = Base64.getDecoder().decode(refreshTokenStatus.getIv());
        byte[] salt = Base64.getDecoder().decode(refreshTokenStatus.getSalt());
        byte[] encryptedSupersedingTokens = Base64.getDecoder().decode(refreshTokenStatus.getSupersedingTokens());
        try {
            Cipher cipher = this.getDecryptionCipher(iv, refreshToken, salt);
            String supersedingTokens = new String(cipher.doFinal(encryptedSupersedingTokens), StandardCharsets.UTF_8);
            final String[] decryptedTokens = supersedingTokens.split("\\|");
            if (decryptedTokens.length != 2) {
                logger.warn("Decrypted tokens string is not correctly formatted");
                listener.onFailure((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
            } else {
                Iterator backoff = BackoffPolicy.exponentialBackoff((TimeValue)TimeValue.timeValueMillis((long)10L), (int)8).iterator();
                String tokenDocId = TokenService.getTokenDocumentId(TokenService.hashTokenString(decryptedTokens[0]));
                final Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("decrypt and get superseding token", tokenDocId, ex));
                final Consumer<ActionListener> maybeRetryGet = actionListener -> {
                    if (backoff.hasNext()) {
                        logger.info("could not get token document [{}] that should have been created, retrying", (Object)tokenDocId);
                        this.client.threadPool().schedule(() -> this.getTokenDocAsync(tokenDocId, tokensIndex, false, (ActionListener<GetResponse>)actionListener), (TimeValue)backoff.next(), "generic");
                    } else {
                        logger.warn("could not get token document [{}] that should have been created after all retries", (Object)tokenDocId);
                        onFailure.accept((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
                    }
                };
                this.getTokenDocAsync(tokenDocId, tokensIndex, false, new ActionListener<GetResponse>(){

                    public void onResponse(GetResponse response) {
                        if (response.isExists()) {
                            try {
                                logger.debug("Found superseding document: index=[{}] id=[{}] primTerm=[{}] seqNo=[{}]", (Object)response.getIndex(), (Object)response.getId(), (Object)response.getPrimaryTerm(), (Object)response.getSeqNo());
                                listener.onResponse((Object)new CreateTokenResult(TokenService.this.prependVersionAndEncodeAccessToken(refreshTokenStatus.getVersion(), decryptedTokens[0]), TokenService.prependVersionAndEncodeRefreshToken(refreshTokenStatus.getVersion(), decryptedTokens[1]), authentication));
                            }
                            catch (IOException | GeneralSecurityException e) {
                                logger.warn("Could not format stored superseding token values", (Throwable)e);
                                onFailure.accept(TokenService.invalidGrantException("could not refresh the requested token"));
                            }
                        } else {
                            maybeRetryGet.accept(this);
                        }
                    }

                    public void onFailure(Exception e) {
                        if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                            maybeRetryGet.accept(this);
                        } else {
                            onFailure.accept(e);
                        }
                    }
                });
            }
        }
        catch (GeneralSecurityException e) {
            logger.warn("Could not get stored superseding token values", (Throwable)e);
            listener.onFailure((Exception)((Object)TokenService.invalidGrantException("could not refresh the requested token")));
        }
    }

    String encryptSupersedingTokens(String supersedingAccessToken, String supersedingRefreshToken, String refreshToken, byte[] iv, byte[] salt) throws GeneralSecurityException {
        Cipher cipher = this.getEncryptionCipher(iv, refreshToken, salt);
        String supersedingTokens = supersedingAccessToken + "|" + supersedingRefreshToken;
        return Base64.getEncoder().encodeToString(cipher.doFinal(supersedingTokens.getBytes(StandardCharsets.UTF_8)));
    }

    private void getTokenDocAsync(String tokenDocId, SecurityIndexManager tokensIndex, boolean fetchSource, ActionListener<GetResponse> listener) {
        GetRequest getRequest = (GetRequest)this.client.prepareGet(tokensIndex.aliasName(), "_doc", tokenDocId).setFetchSource(fetchSource).request();
        tokensIndex.checkIndexVersionThenExecute(ex -> listener.onFailure(this.traceLog("prepare tokens index [" + tokensIndex.aliasName() + "]", tokenDocId, ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)getRequest, (ActionListener)listener, (arg_0, arg_1) -> ((Client)this.client).get(arg_0, arg_1)));
    }

    Version getTokenVersionCompatibility() {
        return this.clusterService.state().nodes().getMinNodeVersion();
    }

    public static Boolean isTokenServiceEnabled(Settings settings) {
        return (Boolean)XPackSettings.TOKEN_SERVICE_ENABLED_SETTING.get(settings);
    }

    private static Optional<ElasticsearchSecurityException> checkTokenDocumentExpired(Instant refreshRequested, Map<String, Object> src) {
        Long creationEpochMilli = (Long)src.get("creation_time");
        if (creationEpochMilli == null) {
            throw new IllegalStateException("token document is missing creation time value");
        }
        Instant creationTime = Instant.ofEpochMilli(creationEpochMilli);
        if (refreshRequested.isAfter(creationTime.plus(24L, ChronoUnit.HOURS))) {
            return Optional.of(TokenService.invalidGrantException("token document has expired"));
        }
        return Optional.empty();
    }

    private static Tuple<RefreshTokenStatus, Optional<ElasticsearchSecurityException>> checkTokenDocumentForRefresh(Instant refreshRequested, Authentication clientAuth, Map<String, Object> source) throws IllegalStateException, DateTimeException {
        RefreshTokenStatus refreshTokenStatus = (RefreshTokenStatus)TokenService.parseTokenAndRefreshStatus(source).v2();
        ElasticsearchSecurityException validationException = TokenService.checkTokenDocumentExpired(refreshRequested, source).orElseGet(() -> {
            if (refreshTokenStatus.isInvalidated()) {
                return TokenService.invalidGrantException("token has been invalidated");
            }
            return TokenService.checkClientCanRefresh(refreshTokenStatus, clientAuth).orElse(TokenService.checkMultipleRefreshes(refreshRequested, refreshTokenStatus).orElse(null));
        });
        return new Tuple((Object)refreshTokenStatus, Optional.ofNullable(validationException));
    }

    private static Tuple<UserToken, RefreshTokenStatus> parseTokenAndRefreshStatus(Map<String, Object> source) {
        RefreshTokenStatus refreshTokenStatus = RefreshTokenStatus.fromSourceMap(TokenService.getRefreshTokenSourceMap(source));
        UserToken userToken = UserToken.fromSourceMap(TokenService.getUserTokenSourceMap(source));
        refreshTokenStatus.setVersion(userToken.getVersion());
        return new Tuple((Object)userToken, (Object)refreshTokenStatus);
    }

    private static Optional<ElasticsearchSecurityException> checkClientCanRefresh(RefreshTokenStatus refreshToken, Authentication clientAuthentication) {
        if (!clientAuthentication.getUser().principal().equals(refreshToken.getAssociatedUser())) {
            logger.warn("Token was originally created by [{}] but [{}] attempted to refresh it", (Object)refreshToken.getAssociatedUser(), (Object)clientAuthentication.getUser().principal());
            return Optional.of(TokenService.invalidGrantException("tokens must be refreshed by the creating client"));
        }
        if (!clientAuthentication.getAuthenticatedBy().getName().equals(refreshToken.getAssociatedRealm())) {
            logger.warn("[{}] created the refresh token while authenticated by [{}] but is now authenticated by [{}]", (Object)refreshToken.getAssociatedUser(), (Object)refreshToken.getAssociatedRealm(), (Object)clientAuthentication.getAuthenticatedBy().getName());
            return Optional.of(TokenService.invalidGrantException("tokens must be refreshed by the creating client"));
        }
        return Optional.empty();
    }

    private static Map<String, Object> getRefreshTokenSourceMap(Map<String, Object> source) {
        Map refreshTokenSource = (Map)source.get("refresh_token");
        if (refreshTokenSource == null || refreshTokenSource.isEmpty()) {
            throw new IllegalStateException("token document is missing the refresh_token object");
        }
        return refreshTokenSource;
    }

    private static Map<String, Object> getUserTokenSourceMap(Map<String, Object> source) {
        Map accessTokenSource = (Map)source.get("access_token");
        if (accessTokenSource == null || accessTokenSource.isEmpty()) {
            throw new IllegalStateException("token document is missing the access_token object");
        }
        Map userTokenSource = (Map)accessTokenSource.get("user_token");
        if (userTokenSource == null || userTokenSource.isEmpty()) {
            throw new IllegalStateException("token document is missing the user token info");
        }
        return userTokenSource;
    }

    private static Optional<ElasticsearchSecurityException> checkMultipleRefreshes(Instant refreshRequested, RefreshTokenStatus refreshTokenStatus) {
        if (refreshTokenStatus.isRefreshed()) {
            if (refreshTokenStatus.getVersion().onOrAfter(VERSION_MULTIPLE_CONCURRENT_REFRESHES)) {
                if (refreshRequested.isAfter(refreshTokenStatus.getRefreshInstant().plus(30L, ChronoUnit.SECONDS))) {
                    return Optional.of(TokenService.invalidGrantException("token has already been refreshed more than 30 seconds in the past"));
                }
                if (refreshRequested.isBefore(refreshTokenStatus.getRefreshInstant().minus(30L, ChronoUnit.SECONDS))) {
                    return Optional.of(TokenService.invalidGrantException("token has been refreshed more than 30 seconds in the future, clock skew too great"));
                }
            } else {
                return Optional.of(TokenService.invalidGrantException("token has already been refreshed"));
            }
        }
        return Optional.empty();
    }

    public void findActiveTokensForRealm(String realmName, @Nullable Predicate<Map<String, Object>> filter, ActionListener<Collection<Tuple<UserToken, String>>> listener) {
        this.ensureEnabled();
        if (Strings.isNullOrEmpty((String)realmName)) {
            listener.onFailure((Exception)new IllegalArgumentException("realm name is required"));
            return;
        }
        this.sourceIndicesWithTokensAndRun((ActionListener<List<String>>)ActionListener.wrap(indicesWithTokens -> {
            if (indicesWithTokens.isEmpty()) {
                listener.onResponse(Collections.emptyList());
            } else {
                Instant now = this.clock.instant();
                BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)TOKEN_DOC_TYPE)).filter((QueryBuilder)QueryBuilders.termQuery((String)"access_token.realm", (String)realmName)).filter((QueryBuilder)QueryBuilders.boolQuery().should((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.termQuery((String)"access_token.invalidated", (boolean)false)).must((QueryBuilder)QueryBuilders.rangeQuery((String)"access_token.user_token.expiration_time").gte((Object)now.toEpochMilli()))).should((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.termQuery((String)"refresh_token.invalidated", (boolean)false)).must((QueryBuilder)QueryBuilders.rangeQuery((String)"creation_time").gte((Object)(now.toEpochMilli() - TimeValue.timeValueHours((long)24L).millis())))));
                Supplier supplier = this.client.threadPool().getThreadContext().newRestorableContext(false);
                try (ThreadContext.StoredContext ignore = this.client.threadPool().getThreadContext().stashWithOrigin("security");){
                    SearchRequest request = (SearchRequest)this.client.prepareSearch(indicesWithTokens.toArray(new String[0])).setScroll((TimeValue)SearchService.DEFAULT_KEEPALIVE_SETTING.get(this.settings)).setQuery((QueryBuilder)boolQuery).setVersion(false).setSize(1000).setFetchSource(true).request();
                    ScrollHelper.fetchAllByEntity((Client)this.client, (SearchRequest)request, (ActionListener)new ContextPreservingActionListener(supplier, listener), hit -> this.filterAndParseHit((SearchHit)hit, filter));
                }
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    public void findActiveTokensForUser(String username, ActionListener<Collection<Tuple<UserToken, String>>> listener) {
        this.ensureEnabled();
        if (Strings.isNullOrEmpty((String)username)) {
            listener.onFailure((Exception)new IllegalArgumentException("username is required"));
            return;
        }
        this.sourceIndicesWithTokensAndRun((ActionListener<List<String>>)ActionListener.wrap(indicesWithTokens -> {
            if (indicesWithTokens.isEmpty()) {
                listener.onResponse(Collections.emptyList());
            } else {
                Instant now = this.clock.instant();
                BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)TOKEN_DOC_TYPE)).filter((QueryBuilder)QueryBuilders.boolQuery().should((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.termQuery((String)"access_token.invalidated", (boolean)false)).must((QueryBuilder)QueryBuilders.rangeQuery((String)"access_token.user_token.expiration_time").gte((Object)now.toEpochMilli()))).should((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.termQuery((String)"refresh_token.invalidated", (boolean)false)).must((QueryBuilder)QueryBuilders.rangeQuery((String)"creation_time").gte((Object)(now.toEpochMilli() - TimeValue.timeValueHours((long)24L).millis())))));
                Supplier supplier = this.client.threadPool().getThreadContext().newRestorableContext(false);
                try (ThreadContext.StoredContext ignore = this.client.threadPool().getThreadContext().stashWithOrigin("security");){
                    SearchRequest request = (SearchRequest)this.client.prepareSearch(indicesWithTokens.toArray(new String[0])).setScroll((TimeValue)SearchService.DEFAULT_KEEPALIVE_SETTING.get(this.settings)).setQuery((QueryBuilder)boolQuery).setVersion(false).setSize(1000).setFetchSource(true).request();
                    ScrollHelper.fetchAllByEntity((Client)this.client, (SearchRequest)request, (ActionListener)new ContextPreservingActionListener(supplier, listener), hit -> this.filterAndParseHit((SearchHit)hit, TokenService.isOfUser(username)));
                }
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    private void sourceIndicesWithTokensAndRun(ActionListener<List<String>> listener) {
        SecurityIndexManager frozenMainIndex;
        ArrayList<String> indicesWithTokens = new ArrayList<String>(2);
        SecurityIndexManager frozenTokensIndex = this.securityTokensIndex.freeze();
        if (frozenTokensIndex.indexExists()) {
            if (!frozenTokensIndex.isAvailable()) {
                listener.onFailure((Exception)((Object)frozenTokensIndex.getUnavailableReason()));
                return;
            }
            if (!frozenTokensIndex.isIndexUpToDate()) {
                listener.onFailure((Exception)new IllegalStateException("Index [" + frozenTokensIndex.aliasName() + "] is not on the current version. Features relying on the index will not be available until the upgrade API is run on the index"));
                return;
            }
            indicesWithTokens.add(frozenTokensIndex.aliasName());
        }
        if ((frozenMainIndex = this.securityMainIndex.freeze()).indexExists() && (!frozenTokensIndex.indexExists() || frozenTokensIndex.getCreationTime().isAfter(this.clock.instant().minus(24L, ChronoUnit.HOURS)))) {
            if (!frozenMainIndex.isAvailable()) {
                listener.onFailure((Exception)((Object)frozenMainIndex.getUnavailableReason()));
                return;
            }
            if (!frozenMainIndex.isIndexUpToDate()) {
                listener.onFailure((Exception)new IllegalStateException("Index [" + frozenMainIndex.aliasName() + "] is not on the current version. Features relying on the index will not be available until the upgrade API is run on the index"));
                return;
            }
            indicesWithTokens.add(frozenMainIndex.aliasName());
        }
        listener.onResponse(indicesWithTokens);
    }

    private BytesReference createTokenDocument(UserToken userToken, @Nullable String refreshToken, @Nullable Authentication originatingClientAuth) {
        Instant creationTime = this.getCreationTime(userToken.getExpirationTime());
        return TokenService.createTokenDocument(userToken, refreshToken, originatingClientAuth, creationTime);
    }

    static BytesReference createTokenDocument(UserToken userToken, String refreshToken, Authentication originatingClientAuth, Instant creationTime) {
        BytesReference bytesReference;
        block10: {
            assert (refreshToken == null || originatingClientAuth != null) : "non-null refresh token " + refreshToken + " requires non-null client authn " + originatingClientAuth;
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                builder.startObject();
                builder.field("doc_type", TOKEN_DOC_TYPE);
                builder.field("creation_time", creationTime.toEpochMilli());
                if (refreshToken != null) {
                    builder.startObject("refresh_token").field(TOKEN_DOC_TYPE, refreshToken).field("invalidated", false).field("refreshed", false).startObject("client").field("type", "unassociated_client").field("user", originatingClientAuth.getUser().principal()).field("realm", originatingClientAuth.getAuthenticatedBy().getName()).endObject().endObject();
                }
                builder.startObject("access_token").field("invalidated", false).field("user_token", (ToXContent)userToken).field("realm", userToken.getAuthentication().getAuthenticatedBy().getName()).endObject();
                builder.endObject();
                bytesReference = BytesReference.bytes((XContentBuilder)builder);
                if (builder == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new RuntimeException("Unexpected exception when constructing a JSON document.", e);
                }
            }
            builder.close();
        }
        return bytesReference;
    }

    private static Predicate<Map<String, Object>> isOfUser(String username) {
        return source -> {
            boolean bl;
            block8: {
                String auth = (String)source.get("authentication");
                Integer version = (Integer)source.get("version");
                Version authVersion = Version.fromId((int)version);
                StreamInput in = StreamInput.wrap((byte[])Base64.getDecoder().decode(auth));
                try {
                    in.setVersion(authVersion);
                    Authentication authentication = new Authentication(in);
                    bl = authentication.getUser().principal().equals(username);
                    if (in == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (in != null) {
                            try {
                                in.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                }
                in.close();
            }
            return bl;
        };
    }

    private Tuple<UserToken, String> filterAndParseHit(SearchHit hit, @Nullable Predicate<Map<String, Object>> filter) throws IllegalStateException, DateTimeException {
        Map source = hit.getSourceAsMap();
        if (source == null) {
            throw new IllegalStateException("token document did not have source but source should have been fetched");
        }
        return this.parseTokensFromDocument(source, filter);
    }

    private Tuple<UserToken, String> parseTokensFromDocument(Map<String, Object> source, @Nullable Predicate<Map<String, Object>> filter) throws IllegalStateException, DateTimeException {
        String hashedRefreshToken = (String)((Map)source.get("refresh_token")).get(TOKEN_DOC_TYPE);
        Map userTokenSource = (Map)((Map)source.get("access_token")).get("user_token");
        if (null != filter && !filter.test(userTokenSource)) {
            return null;
        }
        UserToken userToken = UserToken.fromSourceMap(userTokenSource);
        return new Tuple((Object)userToken, (Object)hashedRefreshToken);
    }

    private static String getTokenDocumentId(UserToken userToken) {
        return TokenService.getTokenDocumentId(userToken.getId());
    }

    private static String getTokenDocumentId(String id) {
        return TOKEN_DOC_ID_PREFIX + id;
    }

    private static String getTokenIdFromDocumentId(String docId) {
        if (!docId.startsWith(TOKEN_DOC_ID_PREFIX)) {
            throw new IllegalStateException("TokenDocument ID [" + docId + "] has unexpected value");
        }
        return docId.substring(TOKEN_DOC_ID_PREFIX.length());
    }

    private boolean isEnabled() {
        return this.enabled && this.licenseState.isSecurityEnabled() && Security.TOKEN_SERVICE_FEATURE.check(this.licenseState);
    }

    private void ensureEnabled() {
        if (!this.licenseState.isSecurityEnabled() || !Security.TOKEN_SERVICE_FEATURE.check(this.licenseState)) {
            throw LicenseUtils.newComplianceException((String)"security tokens");
        }
        if (!this.enabled) {
            throw new FeatureNotEnabledException(FeatureNotEnabledException.Feature.TOKEN_SERVICE, "security tokens are not enabled", new Object[0]);
        }
    }

    private SecurityIndexManager getTokensIndexForVersion(Version version) {
        if (version.onOrAfter(VERSION_TOKENS_INDEX_INTRODUCED)) {
            return this.securityTokensIndex;
        }
        return this.securityMainIndex;
    }

    private void checkIfTokenIsValid(UserToken userToken, ActionListener<UserToken> listener) {
        if (this.clock.instant().isAfter(userToken.getExpirationTime())) {
            listener.onFailure((Exception)((Object)this.traceLog("validate token", userToken.getId(), TokenService.expiredTokenException())));
            return;
        }
        SecurityIndexManager tokensIndex = this.getTokensIndexForVersion(userToken.getVersion());
        if (!tokensIndex.indexExists()) {
            logger.warn("failed to validate access token because the index [" + tokensIndex.aliasName() + "] doesn't exist");
            listener.onResponse(null);
        } else {
            GetRequest getRequest = (GetRequest)this.client.prepareGet(tokensIndex.aliasName(), "_doc", TokenService.getTokenDocumentId(userToken)).request();
            Consumer<Exception> onFailure = ex -> listener.onFailure(this.traceLog("check token state", userToken.getId(), ex));
            tokensIndex.checkIndexVersionThenExecute(arg_0 -> listener.onFailure(arg_0), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)getRequest, (ActionListener)ActionListener.wrap(response -> {
                if (response.isExists()) {
                    Map source = response.getSource();
                    Map accessTokenSource = (Map)source.get("access_token");
                    if (accessTokenSource == null) {
                        onFailure.accept(new IllegalStateException("token document is missing access_token field"));
                    } else {
                        Boolean invalidated = (Boolean)accessTokenSource.get("invalidated");
                        if (invalidated == null) {
                            onFailure.accept(new IllegalStateException("token document is missing invalidated field"));
                        } else if (invalidated.booleanValue()) {
                            onFailure.accept((Exception)((Object)TokenService.expiredTokenException()));
                        } else {
                            listener.onResponse((Object)userToken);
                        }
                    }
                } else {
                    logger.warn("Could not find token document (index=[{}] id=[{}]) in order to validate user token [{}] for [{}]", (Object)response.getIndex(), (Object)response.getId(), (Object)userToken.getId(), (Object)userToken.getAuthentication().getUser().principal());
                    onFailure.accept(this.traceLog("validate token", userToken.getId(), new IllegalStateException("token document is missing and must be present")));
                }
            }, e -> {
                if (TransportActions.isShardNotAvailableException((Throwable)e)) {
                    logger.warn("failed to get access token because index is not available");
                    listener.onResponse(null);
                } else {
                    logger.error((Message)new ParameterizedMessage("failed to get token [{}]", (Object)userToken.getId()), (Throwable)e);
                    listener.onFailure(e);
                }
            }), (arg_0, arg_1) -> ((Client)this.client).get(arg_0, arg_1)));
        }
    }

    public TimeValue getExpirationDelay() {
        return this.expirationDelay;
    }

    Instant getExpirationTime() {
        return this.clock.instant().plusSeconds(this.expirationDelay.getSeconds());
    }

    private Instant getCreationTime(Instant expire) {
        return expire.minusSeconds(this.expirationDelay.getSeconds());
    }

    private void maybeStartTokenRemover() {
        if (this.client.threadPool().relativeTimeInMillis() - this.lastExpirationRunMs > this.deleteInterval.getMillis()) {
            this.expiredTokenRemover.submit(this.client.threadPool());
            this.lastExpirationRunMs = this.client.threadPool().relativeTimeInMillis();
        }
    }

    /*
     * Exception decompiling
     */
    String prependVersionAndEncodeAccessToken(Version version, String accessToken) throws IOException, GeneralSecurityException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 4 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    public static String prependVersionAndEncodeRefreshToken(Version version, String payload) {
        String string;
        BytesStreamOutput out = new BytesStreamOutput();
        try {
            out.setVersion(version);
            Version.writeVersion((Version)version, (StreamOutput)out);
            out.writeString(payload);
            string = Base64.getEncoder().encodeToString(out.bytes().toBytesRef().bytes);
        }
        catch (Throwable throwable) {
            try {
                try {
                    out.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new RuntimeException("Unexpected exception when working with small in-memory streams", e);
            }
        }
        out.close();
        return string;
    }

    public static Tuple<Version, String> unpackVersionAndPayload(String encodedPack) throws IOException {
        byte[] bytes = encodedPack.getBytes(StandardCharsets.UTF_8);
        try (InputStreamStreamInput in = new InputStreamStreamInput(Base64.getDecoder().wrap(new ByteArrayInputStream(bytes)), (long)bytes.length);){
            Version version = Version.readVersion((StreamInput)in);
            in.setVersion(version);
            String payload = in.readString();
            Tuple tuple = new Tuple((Object)version, (Object)payload);
            return tuple;
        }
    }

    private void ensureEncryptionCiphersSupported() throws NoSuchPaddingException, NoSuchAlgorithmException {
        Cipher.getInstance(ENCRYPTION_CIPHER);
        SecretKeyFactory.getInstance(KDF_ALGORITHM);
    }

    Cipher getEncryptionCipher(byte[] iv, KeyAndCache keyAndCache, Version version) throws GeneralSecurityException {
        Cipher cipher = Cipher.getInstance(ENCRYPTION_CIPHER);
        BytesKey salt = keyAndCache.getSalt();
        try {
            cipher.init(1, (Key)keyAndCache.getOrComputeKey(salt), new GCMParameterSpec(128, iv), this.secureRandom);
        }
        catch (ExecutionException e) {
            throw new ElasticsearchSecurityException("Failed to compute secret key for active salt", (Exception)e, new Object[0]);
        }
        cipher.updateAAD(ByteBuffer.allocate(4).putInt(version.id).array());
        cipher.updateAAD(salt.bytes);
        return cipher;
    }

    Cipher getEncryptionCipher(byte[] iv, String password, byte[] salt) throws GeneralSecurityException {
        SecretKey key = TokenService.computeSecretKey(password.toCharArray(), salt, 1024);
        Cipher cipher = Cipher.getInstance(ENCRYPTION_CIPHER);
        cipher.init(1, (Key)key, new GCMParameterSpec(128, iv), this.secureRandom);
        cipher.updateAAD(salt);
        return cipher;
    }

    private void getKeyAsync(BytesKey decodedSalt, KeyAndCache keyAndCache, ActionListener<SecretKey> listener) {
        SecretKey decodeKey = keyAndCache.getKey(decodedSalt);
        if (decodeKey != null) {
            listener.onResponse((Object)decodeKey);
        } else {
            this.client.threadPool().executor(THREAD_POOL_NAME).submit((Runnable)((Object)new KeyComputingRunnable(decodedSalt, keyAndCache, listener)));
        }
    }

    /*
     * Exception decompiling
     */
    private static String decryptTokenId(byte[] encryptedTokenId, Cipher cipher, Version version) throws IOException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    private Cipher getDecryptionCipher(byte[] iv, SecretKey key, Version version, BytesKey salt) throws GeneralSecurityException {
        Cipher cipher = Cipher.getInstance(ENCRYPTION_CIPHER);
        cipher.init(2, (Key)key, new GCMParameterSpec(128, iv), this.secureRandom);
        cipher.updateAAD(ByteBuffer.allocate(4).putInt(version.id).array());
        cipher.updateAAD(salt.bytes);
        return cipher;
    }

    private Cipher getDecryptionCipher(byte[] iv, String password, byte[] salt) throws GeneralSecurityException {
        SecretKey key = TokenService.computeSecretKey(password.toCharArray(), salt, 1024);
        Cipher cipher = Cipher.getInstance(ENCRYPTION_CIPHER);
        cipher.init(2, (Key)key, new GCMParameterSpec(128, iv), this.secureRandom);
        cipher.updateAAD(salt);
        return cipher;
    }

    byte[] getRandomBytes(int length) {
        byte[] bytes = new byte[length];
        this.secureRandom.nextBytes(bytes);
        return bytes;
    }

    static SecretKey computeSecretKey(char[] rawPassword, byte[] salt, int iterations) throws NoSuchAlgorithmException, InvalidKeySpecException {
        SecretKeyFactory secretKeyFactory = SecretKeyFactory.getInstance(KDF_ALGORITHM);
        PBEKeySpec keySpec = new PBEKeySpec(rawPassword, salt, iterations, 128);
        SecretKey tmp = secretKeyFactory.generateSecret(keySpec);
        return new SecretKeySpec(tmp.getEncoded(), "AES");
    }

    private static ElasticsearchSecurityException expiredTokenException() {
        ElasticsearchSecurityException e = new ElasticsearchSecurityException("token expired", RestStatus.UNAUTHORIZED, new Object[0]);
        e.addHeader("WWW-Authenticate", new String[]{EXPIRED_TOKEN_WWW_AUTH_VALUE});
        return e;
    }

    private static ElasticsearchSecurityException invalidGrantException(String detail) {
        ElasticsearchSecurityException e = new ElasticsearchSecurityException("invalid_grant", RestStatus.BAD_REQUEST, new Object[0]);
        e.addHeader("error_description", new String[]{detail});
        return e;
    }

    private static ElasticsearchSecurityException unableToPerformAction(@Nullable Throwable cause) {
        return new ElasticsearchSecurityException("unable to perform requested action", RestStatus.SERVICE_UNAVAILABLE, cause, new Object[0]);
    }

    private <E extends Throwable> E traceLog(String action, String identifier, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}] - [{}]", new Object[]{action, identifier, detail}), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), exception);
            }
        }
        return exception;
    }

    private <E extends Throwable> E traceLog(String action, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] - [{}]", (Object)action, detail), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), exception);
            }
        }
        return exception;
    }

    boolean isExpiredTokenException(ElasticsearchSecurityException e) {
        List headers = e.getHeader("WWW-Authenticate");
        return headers != null && headers.stream().anyMatch(EXPIRED_TOKEN_WWW_AUTH_VALUE::equals);
    }

    boolean isExpirationInProgress() {
        return this.expiredTokenRemover.isExpirationInProgress();
    }

    synchronized TokenMetadata generateSpareKey() {
        KeyAndCache currentKey = this.keyCache.activeKeyCache;
        KeyAndCache maxKey = this.keyCache.cache.values().stream().max(Comparator.comparingLong(v -> ((KeyAndCache)v).keyAndTimestamp.getTimestamp())).get();
        if (currentKey == maxKey) {
            KeyAndCache keyAndCache;
            long timestamp = this.createdTimeStamps.incrementAndGet();
            do {
                byte[] saltArr = new byte[32];
                this.secureRandom.nextBytes(saltArr);
                SecureString tokenKey = this.generateTokenKey();
                keyAndCache = new KeyAndCache(new KeyAndTimestamp(tokenKey, timestamp), new BytesKey(saltArr));
            } while (this.keyCache.cache.containsKey(keyAndCache.getKeyHash()));
            return this.newTokenMetadata(this.keyCache.currentTokenKeyHash, Iterables.concat((Iterable[])new Iterable[]{this.keyCache.cache.values(), Collections.singletonList(keyAndCache)}));
        }
        return this.newTokenMetadata(this.keyCache.currentTokenKeyHash, this.keyCache.cache.values());
    }

    synchronized TokenMetadata rotateToSpareKey() {
        KeyAndCache maxKey = this.keyCache.cache.values().stream().max(Comparator.comparingLong(v -> ((KeyAndCache)v).keyAndTimestamp.getTimestamp())).get();
        if (maxKey == this.keyCache.activeKeyCache) {
            throw new IllegalStateException("call generateSpareKey first");
        }
        return this.newTokenMetadata(maxKey.getKeyHash(), this.keyCache.cache.values());
    }

    synchronized TokenMetadata pruneKeys(int numKeysToKeep) {
        if (this.keyCache.cache.size() <= numKeysToKeep) {
            return this.getTokenMetadata();
        }
        HashMap<BytesKey, KeyAndCache> map = new HashMap<BytesKey, KeyAndCache>(this.keyCache.cache.size() + 1);
        KeyAndCache currentKey = this.keyCache.get(this.keyCache.currentTokenKeyHash);
        ArrayList<KeyAndCache> entries = new ArrayList<KeyAndCache>(this.keyCache.cache.values());
        Collections.sort(entries, (left, right) -> Long.compare(((KeyAndCache)right).keyAndTimestamp.getTimestamp(), ((KeyAndCache)left).keyAndTimestamp.getTimestamp()));
        for (KeyAndCache value : entries) {
            if (map.size() < numKeysToKeep || value.keyAndTimestamp.getTimestamp() >= currentKey.keyAndTimestamp.getTimestamp()) {
                logger.debug("keeping key {} ", (Object)value.getKeyHash());
                map.put(value.getKeyHash(), value);
                continue;
            }
            logger.debug("prune key {} ", (Object)value.getKeyHash());
        }
        assert (!map.isEmpty());
        assert (map.containsKey(this.keyCache.currentTokenKeyHash));
        return this.newTokenMetadata(this.keyCache.currentTokenKeyHash, map.values());
    }

    public synchronized TokenMetadata getTokenMetadata() {
        return this.newTokenMetadata(this.keyCache.currentTokenKeyHash, this.keyCache.cache.values());
    }

    private TokenMetadata newTokenMetadata(BytesKey activeTokenKey, Iterable<KeyAndCache> iterable) {
        ArrayList<KeyAndTimestamp> list = new ArrayList<KeyAndTimestamp>();
        for (KeyAndCache v : iterable) {
            list.add(v.keyAndTimestamp);
        }
        return new TokenMetadata(list, activeTokenKey.bytes);
    }

    synchronized void refreshMetadata(TokenMetadata metadata) {
        BytesKey currentUsedKeyHash = new BytesKey(metadata.getCurrentKeyHash());
        byte[] saltArr = new byte[32];
        HashMap<BytesKey, KeyAndCache> map = new HashMap<BytesKey, KeyAndCache>(metadata.getKeys().size());
        long maxTimestamp = this.createdTimeStamps.get();
        for (KeyAndTimestamp key : metadata.getKeys()) {
            this.secureRandom.nextBytes(saltArr);
            KeyAndCache keyAndCache = new KeyAndCache(key, new BytesKey(saltArr));
            maxTimestamp = Math.max(keyAndCache.keyAndTimestamp.getTimestamp(), maxTimestamp);
            if (!this.keyCache.cache.containsKey(keyAndCache.getKeyHash())) {
                map.put(keyAndCache.getKeyHash(), keyAndCache);
                continue;
            }
            map.put(keyAndCache.getKeyHash(), this.keyCache.get(keyAndCache.getKeyHash()));
        }
        if (!map.containsKey(currentUsedKeyHash)) {
            throw new IllegalStateException("Current key is not in the map: " + map.keySet() + " key: " + currentUsedKeyHash);
        }
        this.createdTimeStamps.set(maxTimestamp);
        this.keyCache = new TokenKeys(Collections.unmodifiableMap(map), currentUsedKeyHash);
        logger.debug(() -> new ParameterizedMessage("refreshed keys current: {}, keys: {}", (Object)currentUsedKeyHash, this.keyCache.cache.keySet()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SecureString generateTokenKey() {
        byte[] keyBytes = new byte[64];
        byte[] encode = new byte[]{};
        char[] ref = new char[]{};
        try {
            this.secureRandom.nextBytes(keyBytes);
            encode = Base64.getUrlEncoder().withoutPadding().encode(keyBytes);
            ref = new char[encode.length];
            int len = UnicodeUtil.UTF8toUTF16((byte[])encode, (int)0, (int)encode.length, (char[])ref);
            SecureString secureString = new SecureString(Arrays.copyOfRange(ref, 0, len));
            return secureString;
        }
        finally {
            Arrays.fill(keyBytes, (byte)0);
            Arrays.fill(encode, (byte)0);
            Arrays.fill(ref, '\u0000');
        }
    }

    synchronized String getActiveKeyHash() {
        return new BytesRef(Base64.getUrlEncoder().withoutPadding().encode(this.keyCache.currentTokenKeyHash.bytes)).utf8ToString();
    }

    void rotateKeysOnMaster(ActionListener<AcknowledgedResponse> listener) {
        logger.info("rotate keys on master");
        TokenMetadata tokenMetadata = this.generateSpareKey();
        this.clusterService.submitStateUpdateTask("publish next key to prepare key rotation", (ClusterStateTaskConfig)new TokenMetadataPublishAction(tokenMetadata, (ActionListener<AcknowledgedResponse>)ActionListener.wrap(res -> {
            if (res.isAcknowledged()) {
                TokenMetadata metadata = this.rotateToSpareKey();
                this.clusterService.submitStateUpdateTask("publish next key to prepare key rotation", (ClusterStateTaskConfig)new TokenMetadataPublishAction(metadata, listener));
            } else {
                listener.onFailure((Exception)new IllegalStateException("not acked"));
            }
        }, arg_0 -> listener.onFailure(arg_0))));
    }

    private void initialize(ClusterService clusterService) {
        clusterService.addListener(event -> {
            TokenMetadata custom;
            ClusterState state = event.state();
            if (state.getBlocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
                return;
            }
            if (state.nodes().isLocalNodeElectedMaster()) {
                if (XPackPlugin.isReadyForXPackCustomMetadata((ClusterState)state)) {
                    this.installTokenMetadata(state);
                } else {
                    logger.debug("cannot add token metadata to cluster as the following nodes might not understand the metadata: {}", new org.apache.logging.log4j.util.Supplier[]{() -> XPackPlugin.nodesNotReadyForXPackCustomMetadata((ClusterState)state)});
                }
            }
            if ((custom = (TokenMetadata)event.state().custom("security_tokens")) != null && !custom.equals((Object)this.getTokenMetadata())) {
                logger.info("refresh keys");
                try {
                    this.refreshMetadata(custom);
                }
                catch (Exception e) {
                    logger.warn("refreshing metadata failed", (Throwable)e);
                }
                logger.info("refreshed keys");
            }
        });
    }

    private void installTokenMetadata(ClusterState state) {
        if (state.custom("security_tokens") == null && this.installTokenMetadataInProgress.compareAndSet(false, true)) {
            this.clusterService.submitStateUpdateTask("install-token-metadata", (ClusterStateTaskConfig)new ClusterStateUpdateTask(Priority.URGENT){

                public ClusterState execute(ClusterState currentState) {
                    XPackPlugin.checkReadyForXPackCustomMetadata((ClusterState)currentState);
                    if (currentState.custom("security_tokens") == null) {
                        return ClusterState.builder((ClusterState)currentState).putCustom("security_tokens", (ClusterState.Custom)TokenService.this.getTokenMetadata()).build();
                    }
                    return currentState;
                }

                public void onFailure(String source, Exception e) {
                    TokenService.this.installTokenMetadataInProgress.set(false);
                    logger.error("unable to install token metadata", (Throwable)e);
                }

                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    TokenService.this.installTokenMetadataInProgress.set(false);
                }
            });
        }
    }

    void clearActiveKeyCache() {
        this.keyCache.activeKeyCache.keyCache.invalidateAll();
    }

    KeyAndCache getActiveKeyCache() {
        return this.keyCache.activeKeyCache;
    }

    static final class KeyAndCache
    implements Closeable {
        private final KeyAndTimestamp keyAndTimestamp;
        private final Cache<BytesKey, SecretKey> keyCache;
        private final BytesKey salt;
        private final BytesKey keyHash;

        private KeyAndCache(KeyAndTimestamp keyAndTimestamp, BytesKey salt) {
            this.keyAndTimestamp = keyAndTimestamp;
            this.keyCache = CacheBuilder.builder().setExpireAfterAccess(TimeValue.timeValueMinutes((long)60L)).setMaximumWeight(500L).build();
            try {
                SecretKey secretKey = TokenService.computeSecretKey(keyAndTimestamp.getKey().getChars(), salt.bytes, 100000);
                this.keyCache.put((Object)salt, (Object)secretKey);
            }
            catch (Exception e) {
                throw new IllegalStateException(e);
            }
            this.salt = salt;
            this.keyHash = KeyAndCache.calculateKeyHash(keyAndTimestamp.getKey());
        }

        private SecretKey getKey(BytesKey salt) {
            return (SecretKey)this.keyCache.get((Object)salt);
        }

        public SecretKey getOrComputeKey(BytesKey decodedSalt) throws ExecutionException {
            return (SecretKey)this.keyCache.computeIfAbsent((Object)decodedSalt, salt -> {
                try (SecureString closeableChars = this.keyAndTimestamp.getKey().clone();){
                    SecretKey secretKey = TokenService.computeSecretKey(closeableChars.getChars(), salt.bytes, 100000);
                    return secretKey;
                }
            });
        }

        @Override
        public void close() throws IOException {
            this.keyAndTimestamp.getKey().close();
        }

        BytesKey getKeyHash() {
            return this.keyHash;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private static BytesKey calculateKeyHash(SecureString key) {
            MessageDigest messageDigest = MessageDigests.sha256();
            BytesRefBuilder b = new BytesRefBuilder();
            try {
                BytesKey bytesKey;
                b.copyChars((CharSequence)key);
                BytesRef bytesRef = b.toBytesRef();
                try {
                    messageDigest.update(bytesRef.bytes, bytesRef.offset, bytesRef.length);
                    bytesKey = new BytesKey(Arrays.copyOfRange(messageDigest.digest(), 0, 8));
                }
                catch (Throwable throwable) {
                    Arrays.fill(bytesRef.bytes, (byte)0);
                    throw throwable;
                }
                Arrays.fill(bytesRef.bytes, (byte)0);
                return bytesKey;
            }
            finally {
                Arrays.fill(b.bytes(), (byte)0);
            }
        }

        BytesKey getSalt() {
            return this.salt;
        }
    }

    private static final class TokenKeys {
        final Map<BytesKey, KeyAndCache> cache;
        final BytesKey currentTokenKeyHash;
        final KeyAndCache activeKeyCache;

        private TokenKeys(Map<BytesKey, KeyAndCache> cache, BytesKey currentTokenKeyHash) {
            this.cache = cache;
            this.currentTokenKeyHash = currentTokenKeyHash;
            this.activeKeyCache = cache.get(currentTokenKeyHash);
        }

        KeyAndCache get(BytesKey passphraseHash) {
            return this.cache.get(passphraseHash);
        }
    }

    static final class RefreshTokenStatus {
        private final boolean invalidated;
        private final String associatedUser;
        private final String associatedRealm;
        private final boolean refreshed;
        @Nullable
        private final Instant refreshInstant;
        @Nullable
        private final String supersedingTokens;
        @Nullable
        private final String iv;
        @Nullable
        private final String salt;
        private Version version;

        RefreshTokenStatus(boolean invalidated, String associatedUser, String associatedRealm, boolean refreshed, Instant refreshInstant, String supersedingTokens, String iv, String salt) {
            this.invalidated = invalidated;
            this.associatedUser = associatedUser;
            this.associatedRealm = associatedRealm;
            this.refreshed = refreshed;
            this.refreshInstant = refreshInstant;
            this.supersedingTokens = supersedingTokens;
            this.iv = iv;
            this.salt = salt;
        }

        boolean isInvalidated() {
            return this.invalidated;
        }

        String getAssociatedUser() {
            return this.associatedUser;
        }

        String getAssociatedRealm() {
            return this.associatedRealm;
        }

        boolean isRefreshed() {
            return this.refreshed;
        }

        @Nullable
        Instant getRefreshInstant() {
            return this.refreshInstant;
        }

        @Nullable
        String getSupersedingTokens() {
            return this.supersedingTokens;
        }

        @Nullable
        String getIv() {
            return this.iv;
        }

        @Nullable
        String getSalt() {
            return this.salt;
        }

        Version getVersion() {
            return this.version;
        }

        void setVersion(Version version) {
            this.version = version;
        }

        static RefreshTokenStatus fromSourceMap(Map<String, Object> refreshTokenSource) {
            Boolean invalidated = (Boolean)refreshTokenSource.get("invalidated");
            if (invalidated == null) {
                throw new IllegalStateException("token document is missing the \"invalidated\" field");
            }
            Map clientInfo = (Map)refreshTokenSource.get("client");
            if (clientInfo == null) {
                throw new IllegalStateException("token document is missing the \"client\" field");
            }
            if (!clientInfo.containsKey("user")) {
                throw new IllegalStateException("token document is missing the \"client.user\" field");
            }
            String associatedUser = (String)clientInfo.get("user");
            if (!clientInfo.containsKey("realm")) {
                throw new IllegalStateException("token document is missing the \"client.realm\" field");
            }
            String associatedRealm = (String)clientInfo.get("realm");
            Boolean refreshed = (Boolean)refreshTokenSource.get("refreshed");
            if (refreshed == null) {
                throw new IllegalStateException("token document is missing the \"refreshed\" field");
            }
            Long refreshEpochMilli = (Long)refreshTokenSource.get("refresh_time");
            Instant refreshInstant = refreshEpochMilli == null ? null : Instant.ofEpochMilli(refreshEpochMilli);
            String supersedingTokens = (String)refreshTokenSource.get("superseding.encrypted_tokens");
            String iv = (String)refreshTokenSource.get("superseding.encryption_iv");
            String salt = (String)refreshTokenSource.get("superseding.encryption_salt");
            return new RefreshTokenStatus(invalidated, associatedUser, associatedRealm, refreshed, refreshInstant, supersedingTokens, iv, salt);
        }
    }

    private class KeyComputingRunnable
    extends AbstractRunnable {
        private final BytesKey decodedSalt;
        private final KeyAndCache keyAndCache;
        private final ActionListener<SecretKey> listener;

        KeyComputingRunnable(BytesKey decodedSalt, KeyAndCache keyAndCache, ActionListener<SecretKey> listener) {
            this.decodedSalt = decodedSalt;
            this.keyAndCache = keyAndCache;
            this.listener = listener;
        }

        protected void doRun() {
            try {
                SecretKey computedKey = this.keyAndCache.getOrComputeKey(this.decodedSalt);
                this.listener.onResponse((Object)computedKey);
            }
            catch (ExecutionException e) {
                if (e.getCause() != null && (e.getCause() instanceof GeneralSecurityException || e.getCause() instanceof IOException || e.getCause() instanceof IllegalArgumentException)) {
                    logger.debug("unable to decode bearer token", (Throwable)e);
                    this.listener.onResponse(null);
                }
                this.listener.onFailure((Exception)e);
            }
        }

        public void onFailure(Exception e) {
            this.listener.onFailure(e);
        }
    }

    private static final class TokenMetadataPublishAction
    extends AckedClusterStateUpdateTask {
        private final TokenMetadata tokenMetadata;

        protected TokenMetadataPublishAction(TokenMetadata tokenMetadata, ActionListener<AcknowledgedResponse> listener) {
            super(new AckedRequest(){

                public TimeValue ackTimeout() {
                    return AcknowledgedRequest.DEFAULT_ACK_TIMEOUT;
                }

                public TimeValue masterNodeTimeout() {
                    return AcknowledgedRequest.DEFAULT_MASTER_NODE_TIMEOUT;
                }
            }, listener);
            this.tokenMetadata = tokenMetadata;
        }

        public ClusterState execute(ClusterState currentState) throws Exception {
            XPackPlugin.checkReadyForXPackCustomMetadata((ClusterState)currentState);
            if (this.tokenMetadata.equals((Object)currentState.custom("security_tokens"))) {
                return currentState;
            }
            return ClusterState.builder((ClusterState)currentState).putCustom("security_tokens", (ClusterState.Custom)this.tokenMetadata).build();
        }
    }

    public static final class CreateTokenResult {
        private final String accessToken;
        private final String refreshToken;
        private final Authentication authentication;

        public CreateTokenResult(String accessToken, String refreshToken, Authentication authentication) {
            this.accessToken = accessToken;
            this.refreshToken = refreshToken;
            this.authentication = authentication;
        }

        public String getAccessToken() {
            return this.accessToken;
        }

        public String getRefreshToken() {
            return this.refreshToken;
        }

        public Authentication getAuthentication() {
            return this.authentication;
        }
    }
}

