# tmux

[![Go Reference](https://pkg.go.dev/badge/github.com/owenthereal/tmux.svg)](https://pkg.go.dev/github.com/owenthereal/tmux)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

A comprehensive Go library for programmatically interacting with tmux sessions, windows, and panes. This library provides a clean, type-safe interface to manage tmux through Go, making it easy to automate terminal multiplexing tasks and build tmux-based applications.

## Features

- **Complete tmux Integration**: Full access to tmux sessions, windows, and panes
- **Type-Safe API**: Strongly typed structures for all tmux entities
- **Context Support**: All operations accept `context.Context` for cancellation and timeouts
- **Comprehensive Data Access**: Retrieve all tmux fields and properties
- **Session Management**: Create, rename, delete, and manage sessions
- **Window Operations**: Handle windows, layouts, and navigation
- **Pane Control**: Split, resize, and manage panes programmatically
- **Server Information**: Query tmux server status and client connections
- **Error Handling**: Robust error handling throughout the API

## Requirements

- tmux installed on your system

## Installation

```bash
go get github.com/owenthereal/tmux
```

## Quick Start

```go
package main

import (
    "context"
    "log"

    "github.com/owenthereal/tmux"
)

func main() {
    ctx := context.Background()

    // Initialize tmux client
    t, err := tmux.Default()
    if err != nil {
        log.Fatal(err)
    }

    // Create a new session
    session, err := t.New(ctx)
    if err != nil {
        log.Fatal(err)
    }

    // Attach to the session
    err = session.Attach(ctx)
    if err != nil {
        log.Fatal(err)
    }
}
```

## API Overview

### Session Management

Create and manage tmux sessions with full control over their properties:

```go
ctx := context.Background()

// Create a new session with custom options
session, err := t.NewSession(ctx, &tmux.SessionOptions{
    StartDirectory: "/home/user",
    Name:           "my-session",
})

// List all sessions
sessions, err := t.ListSessions(ctx)

// Attach to a session
err = session.Attach(ctx)

// Rename a session
err = session.Rename(ctx, "new-name")
```

### Window Operations

Manage windows within sessions:

```go
// Create a new window
window, err := session.New(ctx)

// Get window by index
window, err := session.GetWindowByIndex(ctx, 0)

// Select a specific layout
err = window.SelectLayout(ctx, tmux.WindowLayoutEvenVertical)

// Move window to another session
err = window.Move(ctx, targetSession, 0)
```

### Pane Management

Control individual panes within windows:

```go
// Get first pane
pane, err := window.FirstPane(ctx)

// Split pane (returns the new pane)
newPane, err := pane.Split(ctx)

// Split with options (horizontal/vertical)
newPane, err = pane.SplitWindow(ctx, &tmux.SplitWindowOptions{
    SplitDirection: tmux.PaneSplitDirectionHorizontal,
})

// Execute command in pane
err = pane.SendKeys(ctx, "ls -la")

// Execute command and press Enter
err = pane.SendLine(ctx, "ls -la")
```

### Server and Client Information

Query tmux server status and connected clients:

```go
// Get server information
server, err := t.GetServerInformation(ctx)
fmt.Printf("tmux version: %s\n", server.Version)

// List connected clients
clients, err := t.ListClients(ctx)
for _, client := range clients {
    fmt.Printf("Client: %s, Session: %s\n", client.Tty, client.Session)
}
```

## Data Structures

This library provides comprehensive data structures that capture all tmux fields. For example, the Session struct includes:

```go
type Session struct {
    Activity          string   // Last activity time
    Alerts            string   // Alert flags
    Attached          int      // Number of attached clients
    AttachedList      []string // List of attached clients
    Created           string   // Creation time
    Format            bool     // Format flag
    Group             string   // Session group
    GroupAttached     int      // Number of attached clients in group
    GroupAttachedList []string // List of attached clients in group
    GroupList         []string // List of sessions in group
    GroupManyAttached bool     // Multiple clients attached to group
    GroupSize         int      // Number of sessions in group
    Grouped           bool     // Whether session is grouped
    Id                string   // Session ID
    LastAttached      string   // Last attachment time
    ManyAttached      bool     // Multiple clients attached
    Marked            bool     // Marked flag
    Name              string   // Session name
    Path              string   // Working directory
    Stack             string   // Stack information
    Windows           int      // Number of windows
}
```

## Examples

The project includes comprehensive examples in the `examples/` directory:

- **create/**: Basic session, window, and pane creation
- **sessions/**: Session management operations
- **windows_panes/**: Window and pane manipulation
- **listing/**: Querying and listing tmux entities
- **get_server_info/**: Server information retrieval
- **list_clients/**: Client connection management

### Example: Advanced Session Management

```go
func main() {
    ctx := context.Background()

    t, err := tmux.Default()
    if err != nil {
        log.Fatal(err)
    }

    // Create session with specific options
    session, err := t.NewSession(ctx, &tmux.SessionOptions{
        StartDirectory: "/home/user/projects",
        Name:           "development",
    })
    if err != nil {
        log.Fatal(err)
    }

    // Create a window and split it
    window, err := session.New(ctx)
    if err != nil {
        log.Fatal(err)
    }

    pane, err := window.FirstPane(ctx)
    if err != nil {
        log.Fatal(err)
    }

    // Split the pane and execute commands
    newPane, err := pane.Split(ctx)
    if err != nil {
        log.Fatal(err)
    }

    err = newPane.SendLine(ctx, "cd /home/user/projects && ls -la")
    if err != nil {
        log.Fatal(err)
    }
}
```

## Documentation

- **API Reference**: [pkg.go.dev](https://pkg.go.dev/github.com/owenthereal/tmux)
- **Examples**: [examples directory](https://github.com/owenthereal/tmux/tree/main/examples)
- **Source Code**: [GitHub repository](https://github.com/owenthereal/tmux)

## Contributing

Contributions are welcome! The project aims to provide complete tmux functionality through Go. Please feel free to:

1. Report bugs or request features
2. Submit pull requests
3. Improve documentation
4. Add new examples

## Development Status

This library is actively developed and aims to provide complete tmux functionality. While not all tmux features are implemented yet, the core functionality is stable and ready for production use.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.
