/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package tlsconfig

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestNewDetails(t *testing.T) {
	t.Parallel()

	type args struct {
		session string
		uri     string
		opts    []Option
	}

	tests := []struct {
		name string
		args args
		want Details
	}{
		{
			name: "+noOptions",
			args: args{
				session: "test_session",
				uri:     "http://localhost",
			},
			want: Details{
				SessionName:        "test_session",
				RawURI:             "http://localhost",
				AllowedConnections: map[string]bool{},
			},
		},
		{
			name: "+withOptions",
			args: args{
				session: "test_session",
				uri:     "http://localhost",
				opts: []Option{
					WithTLSConnect("verify_ca"),
					WithTLSCaFile("/tmp/ca.pem"),
					WithAllowedConnections("verify_ca", "insecure"),
				},
			},
			want: Details{
				SessionName: "test_session",
				RawURI:      "http://localhost",
				TLSConnect:  "verify_ca",
				TLSCaFile:   "/tmp/ca.pem",
				AllowedConnections: map[string]bool{
					"verify_ca": true,
					"insecure":  true,
				},
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := NewDetails(tt.args.session, tt.args.uri, tt.args.opts...)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("NewDetails() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestDetails_Apply(t *testing.T) {
	t.Parallel()

	type fields struct {
		SessionName string
		RawURI      string
	}

	type args struct {
		opts []Option
	}

	tests := []struct {
		name   string
		fields fields
		args   args
		want   Details
	}{
		{
			name: "+applySingleOption",
			fields: fields{
				SessionName: "initial_session",
				RawURI:      "http://initial",
			},
			args: args{
				opts: []Option{WithTLSConnect("verify_full")},
			},
			want: Details{
				SessionName:        "initial_session",
				RawURI:             "http://initial",
				TLSConnect:         "verify_full",
				AllowedConnections: map[string]bool{},
			},
		},
		{
			name: "+applyMultipleOptions",
			fields: fields{
				SessionName: "initial_session",
				RawURI:      "http://initial",
			},
			args: args{
				opts: []Option{
					WithTLSConnect("verify_ca"),
					WithTLSCertFile("cert.pem"),
				},
			},
			want: Details{
				SessionName:        "initial_session",
				RawURI:             "http://initial",
				TLSConnect:         "verify_ca",
				TLSCertFile:        "cert.pem",
				AllowedConnections: map[string]bool{},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			d := NewDetails(tt.fields.SessionName, tt.fields.RawURI)
			d.Apply(tt.args.opts...)

			if diff := cmp.Diff(tt.want, d); diff != "" {
				t.Fatalf("Details.Apply() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestDetails_Validate(t *testing.T) {
	t.Parallel()

	type fields struct {
		TLSConnect         TLSConnectionType
		TLSCaFile          string
		TLSCertFile        string
		TLSKeyFile         string
		AllowedConnections map[string]bool
	}

	type args struct {
		checkCA       bool
		checkCertFile bool
		checkKeyFile  bool
	}

	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{
			name: "+fullValidation",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_full": true},
				TLSConnect:         "verify_full",
				TLSCaFile:          "ca.pem",
				TLSCertFile:        "cert.pem",
				TLSKeyFile:         "key.pem",
			},
			args: args{
				checkCA:       true,
				checkCertFile: true,
				checkKeyFile:  true,
			},
			wantErr: false,
		},
		{
			name:    "-noAllowedConnections",
			fields:  fields{AllowedConnections: map[string]bool{}},
			wantErr: true,
		},
		{
			name: "-connectionTypeNotSet",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_ca": true},
			},
			wantErr: true,
		},
		{
			name: "-connectionTypeNotAllowed",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_ca": true},
				TLSConnect:         "insecure",
			},
			wantErr: true,
		},
		{
			name: "-caFileRequiredButNotSet",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_ca": true},
				TLSConnect:         "verify_ca",
			},
			args:    args{checkCA: true},
			wantErr: true,
		},
		{
			name: "-certFileRequiredButNotSet",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_full": true},
				TLSConnect:         "verify_full",
			},
			args:    args{checkCertFile: true},
			wantErr: true,
		},
		{
			name: "-keyFileRequiredButNotSet",
			fields: fields{
				AllowedConnections: map[string]bool{"verify_full": true},
				TLSConnect:         "verify_full",
			},
			args:    args{checkKeyFile: true},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			d := &Details{
				TLSConnect:         tt.fields.TLSConnect,
				TLSCaFile:          tt.fields.TLSCaFile,
				TLSCertFile:        tt.fields.TLSCertFile,
				TLSKeyFile:         tt.fields.TLSKeyFile,
				AllowedConnections: tt.fields.AllowedConnections,
			}

			err := d.Validate(tt.args.checkCA, tt.args.checkCertFile, tt.args.checkKeyFile)
			if (err != nil) != tt.wantErr {
				t.Fatalf("Details.Validate() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}
