/* SPDX-License-Identifier: MIT */
/*
 * Copyright © 2023 Intel Corporation
 */

#ifndef __INTEL_PSR_REGS_H__
#define __INTEL_PSR_REGS_H__

#include "intel_display_reg_defs.h"
#include "intel_dp_aux_regs.h"

#define TRANS_EXITLINE(trans)	_MMIO_TRANS2((trans), _TRANS_EXITLINE_A)
#define   EXITLINE_ENABLE	REG_BIT(31)
#define   EXITLINE_MASK		REG_GENMASK(12, 0)
#define   EXITLINE_SHIFT	0

/*
 * HSW+ eDP PSR registers
 *
 * HSW PSR registers are relative to DDIA(_DDI_BUF_CTL_A + 0x800) with just one
 * instance of it
 */
#define HSW_SRD_CTL				_MMIO(0x64800)
#define _SRD_CTL_A				0x60800
#define _SRD_CTL_EDP				0x6f800
#define EDP_PSR_CTL(tran)			_MMIO_TRANS2(tran, _SRD_CTL_A)
#define   EDP_PSR_ENABLE			REG_BIT(31)
#define   BDW_PSR_SINGLE_FRAME			REG_BIT(30)
#define   EDP_PSR_RESTORE_PSR_ACTIVE_CTX_MASK	REG_BIT(29) /* SW can't modify */
#define   EDP_PSR_LINK_STANDBY			REG_BIT(27)
#define   EDP_PSR_MIN_LINK_ENTRY_TIME_MASK	REG_GENMASK(26, 25)
#define   EDP_PSR_MIN_LINK_ENTRY_TIME_8_LINES	REG_FIELD_PREP(EDP_PSR_MIN_LINK_ENTRY_TIME_MASK, 0)
#define   EDP_PSR_MIN_LINK_ENTRY_TIME_4_LINES	REG_FIELD_PREP(EDP_PSR_MIN_LINK_ENTRY_TIME_MASK, 1)
#define   EDP_PSR_MIN_LINK_ENTRY_TIME_2_LINES	REG_FIELD_PREP(EDP_PSR_MIN_LINK_ENTRY_TIME_MASK, 2)
#define   EDP_PSR_MIN_LINK_ENTRY_TIME_0_LINES	REG_FIELD_PREP(EDP_PSR_MIN_LINK_ENTRY_TIME_MASK, 3)
#define   EDP_PSR_MAX_SLEEP_TIME_MASK		REG_GENMASK(24, 20)
#define   EDP_PSR_MAX_SLEEP_TIME(x)		REG_FIELD_PREP(EDP_PSR_MAX_SLEEP_TIME_MASK, (x))
#define   LNL_EDP_PSR_ENTRY_SETUP_FRAMES_MASK	REG_GENMASK(17, 16)
#define   LNL_EDP_PSR_ENTRY_SETUP_FRAMES(x)	REG_FIELD_PREP(LNL_EDP_PSR_ENTRY_SETUP_FRAMES_MASK, (x))
#define   EDP_PSR_SKIP_AUX_EXIT			REG_BIT(12)
#define   EDP_PSR_TP_MASK			REG_BIT(11)
#define   EDP_PSR_TP_TP1_TP2			REG_FIELD_PREP(EDP_PSR_TP_MASK, 0)
#define   EDP_PSR_TP_TP1_TP3			REG_FIELD_PREP(EDP_PSR_TP_MASK, 1)
#define   EDP_PSR_CRC_ENABLE			REG_BIT(10) /* BDW+ */
#define   EDP_PSR_TP2_TP3_TIME_MASK		REG_GENMASK(9, 8)
#define   EDP_PSR_TP2_TP3_TIME_500us		REG_FIELD_PREP(EDP_PSR_TP2_TP3_TIME_MASK, 0)
#define   EDP_PSR_TP2_TP3_TIME_100us		REG_FIELD_PREP(EDP_PSR_TP2_TP3_TIME_MASK, 1)
#define   EDP_PSR_TP2_TP3_TIME_2500us		REG_FIELD_PREP(EDP_PSR_TP2_TP3_TIME_MASK, 2)
#define   EDP_PSR_TP2_TP3_TIME_0us		REG_FIELD_PREP(EDP_PSR_TP2_TP3_TIME_MASK, 3)
#define   EDP_PSR_TP4_TIME_MASK			REG_GENMASK(7, 6)
#define   EDP_PSR_TP4_TIME_0us			REG_FIELD_PREP(EDP_PSR_TP4_TIME_MASK, 3) /* ICL+ */
#define   EDP_PSR_TP1_TIME_MASK			REG_GENMASK(5, 4)
#define   EDP_PSR_TP1_TIME_500us		REG_FIELD_PREP(EDP_PSR_TP1_TIME_MASK, 0)
#define   EDP_PSR_TP1_TIME_100us		REG_FIELD_PREP(EDP_PSR_TP1_TIME_MASK, 1)
#define   EDP_PSR_TP1_TIME_2500us		REG_FIELD_PREP(EDP_PSR_TP1_TIME_MASK, 2)
#define   EDP_PSR_TP1_TIME_0us			REG_FIELD_PREP(EDP_PSR_TP1_TIME_MASK, 3)
#define   EDP_PSR_IDLE_FRAMES_MASK		REG_GENMASK(3, 0)
#define   EDP_PSR_IDLE_FRAMES(x)		REG_FIELD_PREP(EDP_PSR_IDLE_FRAMES_MASK, (x))

/*
 * Until TGL, IMR/IIR are fixed at 0x648xx. On TGL+ those registers are relative
 * to transcoder and bits defined for each one as if using no shift (i.e. as if
 * it was for TRANSCODER_EDP)
 */
#define EDP_PSR_IMR				_MMIO(0x64834)
#define EDP_PSR_IIR				_MMIO(0x64838)
#define _PSR_IMR_A				0x60814
#define _PSR_IIR_A				0x60818
#define TRANS_PSR_IMR(tran)			_MMIO_TRANS2(tran, _PSR_IMR_A)
#define TRANS_PSR_IIR(tran)			_MMIO_TRANS2(tran, _PSR_IIR_A)
#define   _EDP_PSR_TRANS_SHIFT(trans)		((trans) == TRANSCODER_EDP ? \
						 0 : ((trans) - TRANSCODER_A + 1) * 8)
#define   TGL_PSR_MASK			REG_GENMASK(2, 0)
#define   TGL_PSR_ERROR			REG_BIT(2)
#define   TGL_PSR_POST_EXIT		REG_BIT(1)
#define   TGL_PSR_PRE_ENTRY		REG_BIT(0)
#define   EDP_PSR_MASK(trans)		(TGL_PSR_MASK <<		\
					 _EDP_PSR_TRANS_SHIFT(trans))
#define   EDP_PSR_ERROR(trans)		(TGL_PSR_ERROR <<		\
					 _EDP_PSR_TRANS_SHIFT(trans))
#define   EDP_PSR_POST_EXIT(trans)	(TGL_PSR_POST_EXIT <<		\
					 _EDP_PSR_TRANS_SHIFT(trans))
#define   EDP_PSR_PRE_ENTRY(trans)	(TGL_PSR_PRE_ENTRY <<		\
					 _EDP_PSR_TRANS_SHIFT(trans))

#define HSW_SRD_AUX_CTL				_MMIO(0x64810)
#define _SRD_AUX_CTL_A				0x60810
#define _SRD_AUX_CTL_EDP			0x6f810
#define EDP_PSR_AUX_CTL(tran)			_MMIO_TRANS2(tran, _SRD_AUX_CTL_A)
#define   EDP_PSR_AUX_CTL_TIME_OUT_MASK		DP_AUX_CH_CTL_TIME_OUT_MASK
#define   EDP_PSR_AUX_CTL_MESSAGE_SIZE_MASK	DP_AUX_CH_CTL_MESSAGE_SIZE_MASK
#define   EDP_PSR_AUX_CTL_PRECHARGE_2US_MASK	DP_AUX_CH_CTL_PRECHARGE_2US_MASK
#define   EDP_PSR_AUX_CTL_ERROR_INTERRUPT	REG_BIT(11)
#define   EDP_PSR_AUX_CTL_BIT_CLOCK_2X_MASK	DP_AUX_CH_CTL_BIT_CLOCK_2X_MASK

#define HSW_SRD_AUX_DATA(i)			_MMIO(0x64814 + (i) * 4) /* 5 registers */
#define _SRD_AUX_DATA_A				0x60814
#define _SRD_AUX_DATA_EDP			0x6f814
#define EDP_PSR_AUX_DATA(tran, i)		_MMIO_TRANS2(tran, _SRD_AUX_DATA_A + (i) * 4) /* 5 registers */

#define HSW_SRD_STATUS				_MMIO(0x64840)
#define _SRD_STATUS_A				0x60840
#define _SRD_STATUS_EDP				0x6f840
#define EDP_PSR_STATUS(tran)			_MMIO_TRANS2(tran, _SRD_STATUS_A)
#define   EDP_PSR_STATUS_STATE_MASK		REG_GENMASK(31, 29)
#define   EDP_PSR_STATUS_STATE_IDLE		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 0)
#define   EDP_PSR_STATUS_STATE_SRDONACK		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 1)
#define   EDP_PSR_STATUS_STATE_SRDENT		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 2)
#define   EDP_PSR_STATUS_STATE_BUFOFF		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 3)
#define   EDP_PSR_STATUS_STATE_BUFON		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 4)
#define   EDP_PSR_STATUS_STATE_AUXACK		REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 5)
#define   EDP_PSR_STATUS_STATE_SRDOFFACK	REG_FIELD_PREP(EDP_PSR_STATUS_STATE_MASK, 6)
#define   EDP_PSR_STATUS_LINK_MASK		REG_GENMASK(27, 26)
#define   EDP_PSR_STATUS_LINK_FULL_OFF		REG_FIELD_PREP(EDP_PSR_STATUS_LINK_MASK, 0)
#define   EDP_PSR_STATUS_LINK_FULL_ON		REG_FIELD_PREP(EDP_PSR_STATUS_LINK_MASK, 1)
#define   EDP_PSR_STATUS_LINK_STANDBY		REG_FIELD_PREP(EDP_PSR_STATUS_LINK_MASK, 2)
#define   EDP_PSR_STATUS_MAX_SLEEP_TIMER_MASK	REG_GENMASK(24, 20)
#define   EDP_PSR_STATUS_COUNT_MASK		REG_GENMASK(19, 16)
#define   EDP_PSR_STATUS_AUX_ERROR		REG_BIT(15)
#define   EDP_PSR_STATUS_AUX_SENDING		REG_BIT(12)
#define   EDP_PSR_STATUS_SENDING_IDLE		REG_BIT(9)
#define   EDP_PSR_STATUS_SENDING_TP2_TP3	REG_BIT(8)
#define   EDP_PSR_STATUS_SENDING_TP1		REG_BIT(4)
#define   EDP_PSR_STATUS_IDLE_MASK		REG_GENMASK(3, 0)

#define HSW_SRD_PERF_CNT		_MMIO(0x64844)
#define _SRD_PERF_CNT_A			0x60844
#define _SRD_PERF_CNT_EDP		0x6f844
#define EDP_PSR_PERF_CNT(tran)		_MMIO_TRANS2(tran, _SRD_PERF_CNT_A)
#define   EDP_PSR_PERF_CNT_MASK		REG_GENMASK(23, 0)

/* PSR_MASK on SKL+ */
#define HSW_SRD_DEBUG				_MMIO(0x64860)
#define _SRD_DEBUG_A				0x60860
#define _SRD_DEBUG_EDP				0x6f860
#define EDP_PSR_DEBUG(tran)			_MMIO_TRANS2(tran, _SRD_DEBUG_A)
#define   EDP_PSR_DEBUG_MASK_MAX_SLEEP		REG_BIT(28)
#define   EDP_PSR_DEBUG_MASK_LPSP		REG_BIT(27)
#define   EDP_PSR_DEBUG_MASK_MEMUP		REG_BIT(26)
#define   EDP_PSR_DEBUG_MASK_HPD		REG_BIT(25)
#define   EDP_PSR_DEBUG_MASK_FBC_MODIFY		REG_BIT(24)
#define   EDP_PSR_DEBUG_MASK_PRIMARY_FLIP	REG_BIT(23)  /* hsw */
#define   EDP_PSR_DEBUG_MASK_HDCP_ENABLE	REG_BIT(22)  /* hsw/bdw */
#define   EDP_PSR_DEBUG_MASK_SPRITE_ENABLE	REG_BIT(21)  /* hsw */
#define   EDP_PSR_DEBUG_MASK_CURSOR_MOVE	REG_BIT(20)  /* hsw */
#define   EDP_PSR_DEBUG_MASK_VBLANK_VSYNC_INT	REG_BIT(19)  /* hsw */
#define   EDP_PSR_DEBUG_MASK_DPST_PHASE_IN	REG_BIT(18)  /* hsw */
#define   EDP_PSR_DEBUG_MASK_KVMR_SESSION_EN	REG_BIT(17)
#define   EDP_PSR_DEBUG_MASK_DISP_REG_WRITE	REG_BIT(16)  /* hsw-skl */
#define   EDP_PSR_DEBUG_EXIT_ON_PIXEL_UNDERRUN	REG_BIT(15)  /* skl+ */
#define   EDP_PSR_DEBUG_RFB_UPDATE_SENT		REG_BIT(2)  /* bdw */
#define   EDP_PSR_DEBUG_ENTRY_COMPLETION	REG_BIT(1)  /* hsw/bdw */

#define _PSR2_CTL_A				0x60900
#define _PSR2_CTL_EDP				0x6f900
#define EDP_PSR2_CTL(tran)			_MMIO_TRANS2(tran, _PSR2_CTL_A)
#define   EDP_PSR2_ENABLE			REG_BIT(31)
#define   EDP_SU_TRACK_ENABLE			REG_BIT(30) /* up to adl-p */
#define   TGL_EDP_PSR2_BLOCK_COUNT_MASK		REG_BIT(28)
#define   TGL_EDP_PSR2_BLOCK_COUNT_NUM_2	REG_FIELD_PREP(TGL_EDP_PSR2_BLOCK_COUNT_MASK, 0)
#define   TGL_EDP_PSR2_BLOCK_COUNT_NUM_3	REG_FIELD_PREP(TGL_EDP_PSR2_BLOCK_COUNT_MASK, 1)
#define   LNL_EDP_PSR2_SU_REGION_ET_ENABLE	REG_BIT(27)
#define   EDP_Y_COORDINATE_ENABLE		REG_BIT(25) /* display 10, 11 and 12 */
#define   EDP_PSR2_SU_SDP_SCANLINE		REG_BIT(25) /* display 13+ */
#define   EDP_MAX_SU_DISABLE_TIME_MASK		REG_GENMASK(24, 20)
#define   EDP_MAX_SU_DISABLE_TIME(t)		REG_FIELD_PREP(EDP_MAX_SU_DISABLE_TIME, (t))
#define   EDP_PSR2_IO_BUFFER_WAKE_MASK		REG_GENMASK(14, 13)
#define   EDP_PSR2_IO_BUFFER_WAKE_MAX_LINES	8
#define   EDP_PSR2_IO_BUFFER_WAKE(lines)	REG_FIELD_PREP(EDP_PSR2_IO_BUFFER_WAKE_MASK, \
							       EDP_PSR2_IO_BUFFER_WAKE_MAX_LINES - (lines))
#define   TGL_EDP_PSR2_IO_BUFFER_WAKE_MASK	REG_GENMASK(15, 13)
#define   TGL_EDP_PSR2_IO_BUFFER_WAKE_MIN_LINES	5
#define   TGL_EDP_PSR2_IO_BUFFER_WAKE(lines)	REG_FIELD_PREP(TGL_EDP_PSR2_IO_BUFFER_WAKE_MASK, \
							       (lines) - TGL_EDP_PSR2_IO_BUFFER_WAKE_MIN_LINES)
#define   EDP_PSR2_FAST_WAKE_MASK		REG_GENMASK(12, 11)
#define   EDP_PSR2_FAST_WAKE_MAX_LINES		8
#define   EDP_PSR2_FAST_WAKE(lines)		REG_FIELD_PREP(EDP_PSR2_FAST_WAKE_MASK, \
							       EDP_PSR2_FAST_WAKE_MAX_LINES - (lines))
#define   TGL_EDP_PSR2_FAST_WAKE_MASK		REG_GENMASK(12, 10)
#define   TGL_EDP_PSR2_FAST_WAKE_MIN_LINES	5
#define   TGL_EDP_PSR2_FAST_WAKE(lines)		REG_FIELD_PREP(TGL_EDP_PSR2_FAST_WAKE_MASK, \
							       (lines) - TGL_EDP_PSR2_FAST_WAKE_MIN_LINES)
#define   EDP_PSR2_TP2_TIME_MASK		REG_GENMASK(9, 8)
#define   EDP_PSR2_TP2_TIME_500us		REG_FIELD_PREP(EDP_PSR2_TP2_TIME_MASK, 0)
#define   EDP_PSR2_TP2_TIME_100us		REG_FIELD_PREP(EDP_PSR2_TP2_TIME_MASK, 1)
#define   EDP_PSR2_TP2_TIME_2500us		REG_FIELD_PREP(EDP_PSR2_TP2_TIME_MASK, 2)
#define   EDP_PSR2_TP2_TIME_50us		REG_FIELD_PREP(EDP_PSR2_TP2_TIME_MASK, 3)
#define   EDP_PSR2_FRAME_BEFORE_SU_MASK		REG_GENMASK(7, 4)
#define   EDP_PSR2_FRAME_BEFORE_SU(a)		REG_FIELD_PREP(EDP_PSR2_FRAME_BEFORE_SU_MASK, (a))
#define   EDP_PSR2_IDLE_FRAMES_MASK		REG_GENMASK(3, 0)
#define   EDP_PSR2_IDLE_FRAMES(x)		REG_FIELD_PREP(EDP_PSR2_IDLE_FRAMES_MASK, (x))

#define _PSR_EVENT_TRANS_A			0x60848
#define _PSR_EVENT_TRANS_B			0x61848
#define _PSR_EVENT_TRANS_C			0x62848
#define _PSR_EVENT_TRANS_D			0x63848
#define _PSR_EVENT_TRANS_EDP			0x6f848
#define PSR_EVENT(tran)				_MMIO_TRANS2(tran, _PSR_EVENT_TRANS_A)
#define  PSR_EVENT_PSR2_WD_TIMER_EXPIRE		REG_BIT(17)
#define  PSR_EVENT_PSR2_DISABLED		REG_BIT(16)
#define  PSR_EVENT_SU_DIRTY_FIFO_UNDERRUN	REG_BIT(15)
#define  PSR_EVENT_SU_CRC_FIFO_UNDERRUN		REG_BIT(14)
#define  PSR_EVENT_GRAPHICS_RESET		REG_BIT(12)
#define  PSR_EVENT_PCH_INTERRUPT		REG_BIT(11)
#define  PSR_EVENT_MEMORY_UP			REG_BIT(10)
#define  PSR_EVENT_FRONT_BUFFER_MODIFY		REG_BIT(9)
#define  PSR_EVENT_WD_TIMER_EXPIRE		REG_BIT(8)
#define  PSR_EVENT_PIPE_REGISTERS_UPDATE	REG_BIT(6)
#define  PSR_EVENT_REGISTER_UPDATE		REG_BIT(5) /* Reserved in ICL+ */
#define  PSR_EVENT_HDCP_ENABLE			REG_BIT(4)
#define  PSR_EVENT_KVMR_SESSION_ENABLE		REG_BIT(3)
#define  PSR_EVENT_VBI_ENABLE			REG_BIT(2)
#define  PSR_EVENT_LPSP_MODE_EXIT		REG_BIT(1)
#define  PSR_EVENT_PSR_DISABLE			REG_BIT(0)

#define _PSR2_STATUS_A				0x60940
#define _PSR2_STATUS_EDP			0x6f940
#define EDP_PSR2_STATUS(tran)			_MMIO_TRANS2(tran, _PSR2_STATUS_A)
#define EDP_PSR2_STATUS_STATE_MASK		REG_GENMASK(31, 28)
#define EDP_PSR2_STATUS_STATE_DEEP_SLEEP	REG_FIELD_PREP(EDP_PSR2_STATUS_STATE_MASK, 0x8)

#define _PSR2_SU_STATUS_A		0x60914
#define _PSR2_SU_STATUS_EDP		0x6f914
#define _PSR2_SU_STATUS(tran, index)	_MMIO_TRANS2(tran, _PSR2_SU_STATUS_A + (index) * 4)
#define PSR2_SU_STATUS(tran, frame)	(_PSR2_SU_STATUS(tran, (frame) / 3))
#define PSR2_SU_STATUS_SHIFT(frame)	(((frame) % 3) * 10)
#define PSR2_SU_STATUS_MASK(frame)	(0x3ff << PSR2_SU_STATUS_SHIFT(frame))
#define PSR2_SU_STATUS_FRAMES		8

#define _PSR2_MAN_TRK_CTL_A					0x60910
#define _PSR2_MAN_TRK_CTL_EDP					0x6f910
#define PSR2_MAN_TRK_CTL(tran)					_MMIO_TRANS2(tran, _PSR2_MAN_TRK_CTL_A)
#define  PSR2_MAN_TRK_CTL_ENABLE				REG_BIT(31)
#define  PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR_MASK		REG_GENMASK(30, 21)
#define  PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR(val)		REG_FIELD_PREP(PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR_MASK, val)
#define  PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR_MASK		REG_GENMASK(20, 11)
#define  PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR(val)		REG_FIELD_PREP(PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR_MASK, val)
#define  PSR2_MAN_TRK_CTL_SF_SINGLE_FULL_FRAME			REG_BIT(3)
#define  PSR2_MAN_TRK_CTL_SF_CONTINUOS_FULL_FRAME		REG_BIT(2)
#define  PSR2_MAN_TRK_CTL_SF_PARTIAL_FRAME_UPDATE		REG_BIT(1)
#define  ADLP_PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR_MASK	REG_GENMASK(28, 16)
#define  ADLP_PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR(val)	REG_FIELD_PREP(ADLP_PSR2_MAN_TRK_CTL_SU_REGION_START_ADDR_MASK, val)
#define  ADLP_PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR_MASK		REG_GENMASK(12, 0)
#define  ADLP_PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR(val)		REG_FIELD_PREP(ADLP_PSR2_MAN_TRK_CTL_SU_REGION_END_ADDR_MASK, val)
#define  ADLP_PSR2_MAN_TRK_CTL_SF_PARTIAL_FRAME_UPDATE		REG_BIT(31)
#define  ADLP_PSR2_MAN_TRK_CTL_SF_SINGLE_FULL_FRAME		REG_BIT(14)
#define  ADLP_PSR2_MAN_TRK_CTL_SF_CONTINUOS_FULL_FRAME		REG_BIT(13)

/* PSR2 Early transport */
#define _PIPE_SRCSZ_ERLY_TPT_A	0x70074

#define PIPE_SRCSZ_ERLY_TPT(trans)	_MMIO_TRANS2(trans, _PIPE_SRCSZ_ERLY_TPT_A)

#define _SEL_FETCH_PLANE_BASE_1_A		0x70890
#define _SEL_FETCH_PLANE_BASE_2_A		0x708B0
#define _SEL_FETCH_PLANE_BASE_3_A		0x708D0
#define _SEL_FETCH_PLANE_BASE_4_A		0x708F0
#define _SEL_FETCH_PLANE_BASE_5_A		0x70920
#define _SEL_FETCH_PLANE_BASE_6_A		0x70940
#define _SEL_FETCH_PLANE_BASE_7_A		0x70960
#define _SEL_FETCH_PLANE_BASE_CUR_A		0x70880
#define _SEL_FETCH_PLANE_BASE_1_B		0x71890

#define _SEL_FETCH_PLANE_BASE_A(plane) _PICK(plane, \
					     _SEL_FETCH_PLANE_BASE_1_A, \
					     _SEL_FETCH_PLANE_BASE_2_A, \
					     _SEL_FETCH_PLANE_BASE_3_A, \
					     _SEL_FETCH_PLANE_BASE_4_A, \
					     _SEL_FETCH_PLANE_BASE_5_A, \
					     _SEL_FETCH_PLANE_BASE_6_A, \
					     _SEL_FETCH_PLANE_BASE_7_A, \
					     _SEL_FETCH_PLANE_BASE_CUR_A)
#define _SEL_FETCH_PLANE_BASE_1(pipe) _PIPE(pipe, _SEL_FETCH_PLANE_BASE_1_A, _SEL_FETCH_PLANE_BASE_1_B)
#define _SEL_FETCH_PLANE_BASE(pipe, plane) (_SEL_FETCH_PLANE_BASE_1(pipe) - \
					    _SEL_FETCH_PLANE_BASE_1_A + \
					    _SEL_FETCH_PLANE_BASE_A(plane))

#define _SEL_FETCH_PLANE_CTL_1_A		0x70890
#define PLANE_SEL_FETCH_CTL(pipe, plane) _MMIO(_SEL_FETCH_PLANE_BASE(pipe, plane) + \
					       _SEL_FETCH_PLANE_CTL_1_A - \
					       _SEL_FETCH_PLANE_BASE_1_A)
#define PLANE_SEL_FETCH_CTL_ENABLE		REG_BIT(31)

#define _SEL_FETCH_PLANE_POS_1_A		0x70894
#define PLANE_SEL_FETCH_POS(pipe, plane) _MMIO(_SEL_FETCH_PLANE_BASE(pipe, plane) + \
					       _SEL_FETCH_PLANE_POS_1_A - \
					       _SEL_FETCH_PLANE_BASE_1_A)

#define _SEL_FETCH_PLANE_SIZE_1_A		0x70898
#define PLANE_SEL_FETCH_SIZE(pipe, plane) _MMIO(_SEL_FETCH_PLANE_BASE(pipe, plane) + \
						_SEL_FETCH_PLANE_SIZE_1_A - \
						_SEL_FETCH_PLANE_BASE_1_A)

#define _SEL_FETCH_PLANE_OFFSET_1_A		0x7089C
#define PLANE_SEL_FETCH_OFFSET(pipe, plane) _MMIO(_SEL_FETCH_PLANE_BASE(pipe, plane) + \
						  _SEL_FETCH_PLANE_OFFSET_1_A - \
						  _SEL_FETCH_PLANE_BASE_1_A)

#define _ALPM_CTL_A	0x60950
#define ALPM_CTL(tran)	_MMIO_TRANS2(tran, _ALPM_CTL_A)
#define  ALPM_CTL_ALPM_ENABLE				REG_BIT(31)
#define  ALPM_CTL_ALPM_AUX_LESS_ENABLE			REG_BIT(30)
#define  ALPM_CTL_LOBF_ENABLE				REG_BIT(29)
#define  ALPM_CTL_EXTENDED_FAST_WAKE_ENABLE		REG_BIT(28)
#define  ALPM_CTL_KEEP_FEC_ENABLE_FOR_AUX_WAKE_SLEEP	REG_BIT(27)
#define  ALPM_CTL_RESTORE_OCCURED			REG_BIT(26)
#define  ALPM_CTL_RESTORE_TO_SLEEP			REG_BIT(25)
#define  ALPM_CTL_RESTORE_TO_DEEP_SLEEP			REG_BIT(24)
#define  ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_MASK		REG_GENMASK(23, 21)
#define  ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_50_SYMBOLS	REG_FIELD_PREP(ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_MASK, 0)
#define  ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_128_SYMBOLS	REG_FIELD_PREP(ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_MASK, 1)
#define  ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_256_SYMBOLS	REG_FIELD_PREP(ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_MASK, 2)
#define  ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_512_SYMBOLS	REG_FIELD_PREP(ALPM_CTL_AUX_LESS_SLEEP_HOLD_TIME_MASK, 3)
#define  ALPM_CTL_AUX_WAKE_SLEEP_HOLD_ENABLE		REG_BIT(20)
#define  ALPM_CTL_ALPM_ENTRY_CHECK_MASK			REG_GENMASK(19, 16)
#define  ALPM_CTL_ALPM_ENTRY_CHECK(val)			REG_FIELD_PREP(ALPM_CTL_ALPM_ENTRY_CHECK_MASK, val)
#define  ALPM_CTL_EXTENDED_FAST_WAKE_TIME_MASK		REG_GENMASK(13, 8)
#define  ALPM_CTL_EXTENDED_FAST_WAKE_MIN_LINES		5
#define  ALPM_CTL_EXTENDED_FAST_WAKE_TIME(lines)	REG_FIELD_PREP(ALPM_CTL_EXTENDED_FAST_WAKE_TIME_MASK, (lines) - ALPM_CTL_EXTENDED_FAST_WAKE_MIN_LINES)
#define  ALPM_CTL_AUX_LESS_WAKE_TIME_MASK		REG_GENMASK(5, 0)
#define  ALPM_CTL_AUX_LESS_WAKE_TIME(val)		REG_FIELD_PREP(ALPM_CTL_AUX_LESS_WAKE_TIME_MASK, val)

#define _ALPM_CTL2_A	0x60954
#define ALPM_CTL2(tran)	_MMIO_TRANS2(tran, _ALPM_CTL2_A)
#define  ALPM_CTL2_SWITCH_TO_ACTIVE_LATENCY_MASK		REG_GENMASK(28, 24)
#define  ALPM_CTL2_SWITCH_TO_ACTIVE_LATENCY(val)		REG_FIELD_PREP(ALPM_CTL2_SWITCH_TO_ACTIVE_LATENCY_MASK, val)
#define  ALPM_CTL2_AUX_LESS_WAKE_TIME_EXTENSION_MASK		REG_GENMASK(19, 16)
#define  ALPM_CTL2_AUX_LESS_WAKE_TIME_EXTENSION(val)		REG_FIELD_PREP(ALPM_CTL2_AUX_LESS_WAKE_TIME_EXTENSION_MASK, val)
#define  ALPM_CTL2_NUMBER_OF_LTTPR_MASK				REG_GENMASK(15, 12)
#define  ALPM_CTL2_NUMBER_OF_LTTPR(val)				REG_FIELD_PREP(ALPM_CTL2_NUMBER_OF_LTTPR_MASK, val)
#define  ALPM_CTL2_LTTPR_AUX_LESS_SLEEP_HOLD_TIME_MASK		REG_GENMASK(10, 8)
#define  ALPM_CTL2_LTTPR_AUX_LESS_SLEEP_HOLD_TIME(val)		REG_FIELD_PREP(ALPM_CTL2_LTTPR_AUX_LESS_SLEEP_HOLD_TIME_MASK, val)
#define  ALPM_CTL2_FEC_DECODE_EN_POSITION_AFTER_WAKE_SR		REG_BIT(4)
#define  ALPM_CTL2_NUMBER_AUX_LESS_ML_PHY_SLEEP_SEQUENCES_MASK	REG_GENMASK(2, 0)
#define  ALPM_CTL2_NUMBER_AUX_LESS_ML_PHY_SLEEP_SEQUENCES(val)	REG_FIELD_PREP(ALPM_CTL2_NUMBER_AUX_LESS_ML_PHY_SLEEP_SEQUENCES_MASK, val)

#define _PORT_ALPM_CTL_A			0x16fa2c
#define PORT_ALPM_CTL(tran)			_MMIO_TRANS2(tran, _PORT_ALPM_CTL_A)
#define  PORT_ALPM_CTL_ALPM_AUX_LESS_ENABLE	REG_BIT(31)
#define  PORT_ALPM_CTL_MAX_PHY_SWING_SETUP_MASK	REG_GENMASK(23, 20)
#define  PORT_ALPM_CTL_MAX_PHY_SWING_SETUP(val)	REG_FIELD_PREP(PORT_ALPM_CTL_MAX_PHY_SWING_SETUP_MASK, val)
#define  PORT_ALPM_CTL_MAX_PHY_SWING_HOLD_MASK	REG_GENMASK(19, 16)
#define  PORT_ALPM_CTL_MAX_PHY_SWING_HOLD(val)	REG_FIELD_PREP(PORT_ALPM_CTL_MAX_PHY_SWING_HOLD_MASK, val)
#define  PORT_ALPM_CTL_SILENCE_PERIOD_MASK	REG_GENMASK(7, 0)
#define  PORT_ALPM_CTL_SILENCE_PERIOD(val)	REG_FIELD_PREP(PORT_ALPM_CTL_SILENCE_PERIOD_MASK, val)

#define _PORT_ALPM_LFPS_CTL_A					0x16fa30
#define PORT_ALPM_LFPS_CTL(tran)				_MMIO_TRANS2(tran, _PORT_ALPM_LFPS_CTL_A)
#define  PORT_ALPM_LFPS_CTL_LFPS_START_POLARITY			REG_BIT(31)
#define  PORT_ALPM_LFPS_CTL_LFPS_CYCLE_COUNT_MASK		REG_GENMASK(27, 24)
#define  ALPM_CTL_EXTENDED_FAST_WAKE_MIN_LINES		5
#define  ALPM_CTL_EXTENDED_FAST_WAKE_TIME(lines)	REG_FIELD_PREP(ALPM_CTL_EXTENDED_FAST_WAKE_TIME_MASK, (lines) - ALPM_CTL_EXTENDED_FAST_WAKE_MIN_LINES)
#define  ALPM_CTL_AUX_LESS_WAKE_TIME_MASK		REG_GENMASK(5, 0)
#define  ALPM_CTL_AUX_LESS_WAKE_TIME(val)		REG_FIELD_PREP(ALPM_CTL_AUX_LESS_WAKE_TIME_MASK, val)

#endif /* __INTEL_PSR_REGS_H__ */
